/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		plotPanel.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "plotPanel.h"

#include "palettePanel.h"
#include "OscilloscopeTools.h"
#include "FFT.h"
#ifndef WIN32
#include <unistd.h>
#endif

Qt::GlobalColor plotPanel::chooseDefaultColor(int i) {
	switch (i) {
	case 0:
		return Qt::blue;
	case 1:
		return Qt::red;
	case 2:
		return Qt::green;
	case 3:
		return Qt::yellow;
	case 4:
		return Qt::magenta;
	case 5:
		return Qt::black;
	case 6:
		return Qt::darkBlue;
	case 7:
		return Qt::gray;
	default:
		return Qt::cyan;
	}
}

plotPanel::plotPanel(QWidget *parent, int traces)
	: QMainWindow(parent)
	, mTrace(traces)
{

	mPlot = new QCustomPlot(this);
	QWidget* dummy = new QWidget();
	QHBoxLayout* l = new QHBoxLayout();
	l->setContentsMargins(5, 5, 5, 5);
	dummy->setStyleSheet("QWidget { border: 3px solid rgb(44, 188, 240); }");
	l->addWidget(mPlot);
	dummy->setLayout(l);

	setCentralWidget(dummy);

	mTracesVisible.resize(traces);
	this->mTraces = traces;
	for (int i = 0; i < traces; i++) {
		mTrace[i] = mPlot->addGraph();
		mTrace[i]->setPen(QPen(chooseDefaultColor(i)));
		mTrace[i]->rescaleKeyAxis();
		mTrace[i]->setLineStyle(QCPGraph::lsStepCenter);
		mTracesVisible[i] = true;
	}
	ActiveTrace = 0;
	mTracesBaselines.resize(mTraces);
	mTracesBaselines.fill(0);
	mTracesADCBits.resize(mTraces);
	mTracesADCBits.fill(0);
	mTracesSampleTomV.resize(mTraces);
	mTracesSampleTomV.fill(1);
	mTracesGains.resize(mTraces);
	mTracesGains.fill(1.0);
	
	mTools = new OscilloscopeTools(this);
	mToolsDock = new QDockWidget(tr("Tools"), this);
	mToolsDock->setAllowedAreas(Qt::LeftDockWidgetArea);
	mToolsDock->setFeatures(mToolsDock->features() & ~QDockWidget::DockWidgetClosable & ~QDockWidget::DockWidgetFloatable & ~QDockWidget::DockWidgetMovable);
	mToolsDock->setTitleBarWidget(new QWidget());
	mToolsDock->setWidget(mTools);
	addDockWidget(Qt::LeftDockWidgetArea, mToolsDock);
	mToolsDock->hide();

	mPalettePan = new palettePanel(this, NPLOTTRACES, mTrace.data());
	mTraceDataPan = new traceDataPanel(this);
	mPaletteDock = new QDockWidget("", this);
	mPaletteDock->setAllowedAreas(Qt::RightDockWidgetArea);
	mPaletteDock->setFeatures(mPaletteDock->features() & ~QDockWidget::DockWidgetClosable & ~QDockWidget::DockWidgetFloatable & ~QDockWidget::DockWidgetMovable);
	mPaletteDock->setMinimumWidth(220);
	mPaletteDock->setMaximumWidth(350);
	mTab = new QTabWidget(this);
	mTab->addTab(mPalettePan,"Palette");
	mPaletteDock->setWidget(mTab);
	mPaletteDock->setWidget(mTab);
	addDockWidget(Qt::RightDockWidgetArea, mPaletteDock);
	mPaletteDock->hide();

	//fill the traces map with default channels 0-7
	for (int t = 0; t < mTraces; t++) 
		mTracesChMap.insert(t, t);

	mToolBar = new QToolBar();
	this->addToolBar(mToolBar);

	popolateToolBar();
	mUomX = UOM_PHYS_UNIT;
	//mPlot->xAxis->setRange(0, 4000);
	QString mu = QChar(0xbc, 0x03);
	mPlot->xAxis->setLabel("Time [" + mu + "s]");
	fixedTickerX = QSharedPointer<QCPAxisTickerFixed>(new QCPAxisTickerFixed);
	fixedTickerX->setTickStep(400);
	mPlot->xAxis->setTicker(fixedTickerX);
	fixedTickerX->setScaleStrategy(QCPAxisTickerFixed::ssNone);

	mUomY = UOM_PHYS_UNIT;
	//mPlot->yAxis->setRange(0, 65535);
	mPlot->yAxis->setLabel("Voltage [mV]");
	fixedTickerY = QSharedPointer<QCPAxisTickerFixed>(new QCPAxisTickerFixed);
	fixedTickerY->setTickStep(400);
	mPlot->yAxis->setTicker(fixedTickerY);
	fixedTickerY->setScaleStrategy(QCPAxisTickerFixed::ssNone);

	mPlot->setInteraction(QCP::iRangeZoom);
	mPlot->xAxis2->setVisible(true);
	mPlot->xAxis2->setTickLabels(false);
	mPlot->yAxis2->setVisible(true);
	mPlot->yAxis2->setTickLabels(false);

	mPreTrgMarker = new CAENaxismarker(mPlot->xAxis);
	QPen MarkerPen(Qt::red, 2, Qt::SolidLine, Qt::SquareCap, Qt::RoundJoin);
	mPreTrgMarker->setPen(MarkerPen);
	mPreTrgMarker->setVisible(true);

	mZeroTracer.reserve(mTraces);
	mZeroMarker.reserve(mTraces);

	for (int t = 0; t < mTraces; t++) {

		QPixmap pix(30, 10);
        pix.fill(QColor("white"));
		QPainter painter(&pix);
		painter.setPen(mTrace[t]->pen());
		painter.setBrush(QBrush(mTrace[t]->pen().color(), Qt::SolidPattern));
		painter.drawPolygon(points, 5);

		QCPItemTracer *Tracer = new QCPItemTracer(mPlot);
		Tracer->setInterpolating(false);
		Tracer->setStyle(QCPItemTracer::tsNone);

		QCPItemPixmap *Marker = new QCPItemPixmap(mPlot);
		Marker->setLayer("legend");
		Marker->topLeft->setAxes(mPlot->xAxis->axisRect()->axis(QCPAxis::atBottom), mPlot->xAxis->axisRect()->axis(QCPAxis::atRight));
		Marker->bottomRight->setAxes(mPlot->xAxis->axisRect()->axis(QCPAxis::atBottom), mPlot->xAxis->axisRect()->axis(QCPAxis::atRight));
		Marker->setPixmap(pix);
		Marker->setScaled(false);
		Marker->topLeft->setParentAnchor(Tracer->position);
		Tracer->position->setAxes(mPlot->xAxis->axisRect()->axis(QCPAxis::atBottom), mPlot->yAxis->axisRect()->axis(QCPAxis::atLeft)); 
		Tracer->position->setTypeX(QCPItemPosition::ptAbsolute);
		Tracer->position->setTypeY(QCPItemPosition::ptPlotCoords);

		Marker->topLeft->setCoords(-15, -5); 
		Tracer->position->setCoords(mPlot->xAxis->axisRect()->left() +100, 0); 
		Marker->setClipToAxisRect(false);
		Marker->setSelectable(false);
		Marker->setVisible(true);

		mZeroTracer.push_back(Tracer);
		mZeroMarker.push_back(Marker);
	}

	mThrMarker = new CAENyaxismarker(mPlot->yAxis);
	QPen MarkerPen1(QColor(255, 239, 3), 2, Qt::SolidLine, Qt::FlatCap, Qt::MiterJoin);
	mThrMarker->setPen(MarkerPen1);
	mThrMarker->setVisible(false);
	mPlot->replot();

	mPositionMarker = new QCPItemLine(mPlot);
	QCPLineEnding *style = new QCPLineEnding(QCPLineEnding::esDiamond, 12, 12, false);
	mPositionMarker->setHead(*style);
	mPositionMarker->setSelectable(false);
	QPen MarkerPen2(QColor(65, 141, 255), 2, Qt::SolidLine, Qt::FlatCap, Qt::MiterJoin);
	mPositionMarker->setPen(MarkerPen2);
	mPositionMarker->setVisible(false);

	mPlot->addLayer("abovelegend", mPlot->layer("legend"), QCustomPlot::limAbove);
	mZeroMarker[ActiveTrace]->setLayer("abovelegend");
	mPreTrgMarker->setColor(mTrace[ActiveTrace]->pen().color());

	this->setDisabled(true);
	connect(mPalettePan, SIGNAL(paletteChanged(QVector<QString>, QVector<int>, int)), this, SLOT(paletteChanged(QVector<QString>, QVector<int>, int)));

	connect(mPlot, SIGNAL(mouseDoubleClick(QMouseEvent*)), this, SLOT(onMouseDoubleClick(QMouseEvent*)));

	connect(mPlot, SIGNAL(mouseMove(QMouseEvent*)), this, SLOT(onMouseMove(QMouseEvent*)));
	connect(mPlot, SIGNAL(mouseRelease(QMouseEvent*)), this, SLOT(onMouseRelease(QMouseEvent*)));
	connect(mPlot, SIGNAL(mousePress(QMouseEvent*)), this, SLOT(onMousePress(QMouseEvent*)));
	//limit axis range
	connect(mPlot->xAxis, SIGNAL(rangeChanged(QCPRange)), this, SLOT(onXRangeChanged(QCPRange)));
	connect(mPlot->yAxis, SIGNAL(rangeChanged(QCPRange)), this, SLOT(onYRangeChanged(QCPRange)));

	connect(this, SIGNAL(ThrChanged(double, bool)), parent, SLOT(ThrChanged(double, bool)));
	connect(this, SIGNAL(OffsetChanged(QString, int, double, bool)), parent, SLOT(DCOffsetChanged(QString, int, double, bool)));
	connect(this, SIGNAL(ActiveTraceChanged(QString, int)), parent, SLOT(PlotActiveTraceChanged(QString, int)));
	connect(this, SIGNAL(GenericTraceChanged(QString, int)), parent, SLOT(PlotGenericTraceChanged(QString, int)));
    connect(this, SIGNAL(ActiveTraceColorChanged()), parent, SLOT(PlotActiveTraceColorChanged()));
	connect(this, SIGNAL(ScaleChanged()), parent, SLOT(PlotScaleChanged()));
	connect(this, SIGNAL(TypeChanged()), parent, SLOT(PlotTypeChanged()));
	connect(this, SIGNAL(ToolsOpen(bool)), parent, SLOT(PlotToolsOpen(bool)));
	connect(this, SIGNAL(PlotFreeze(bool)), parent, SLOT(FreezePlot(bool)));
	connect(this, SIGNAL(PaletteOpen(bool)), parent, SLOT(OpenPalette(bool)));
	connect(this, SIGNAL(NWaves(int)), parent, SLOT(NWavesAvgChanged(int)));
	connect(this, SIGNAL(TraceEnabled(int,int)), parent, SLOT(TraceEnableChanged(int,int)));
	connect(this, SIGNAL(TraceChanged(int, QString, int)), parent, SLOT(TracePlotChanged(int, QString, int)));
	connect(this, SIGNAL(ClearPlotData()), parent, SLOT(ClearPlotData()));
	connect(this, SIGNAL(ChangePlotTime(double)), parent, SLOT(PlotTimeChanged(double)));
	connect(this, SIGNAL(NewEventToLoad(QString,int)), parent, SLOT(LoadNewEvent(QString, int)));
	
	

	//init variables
	mMaxX = 4096; 
	mMaxY = 65535;
	mMinY = 0;

	mElementSelected = false;
	mCursorsEnabled = false;

	mRubberBand = new QRubberBand(QRubberBand::Rectangle, mPlot);

	mC1Trace = 0;
	mC2Trace = 0;
}

plotPanel::~plotPanel()
{
}

void plotPanel::popolateToolBar()
{
	const QIcon paletteIcon = QIcon(":palette.png");
	const QIcon cleanIcon = QIcon(":clear.png");
	const QIcon shotIcon = QIcon(":screenshot.png");
	const QIcon freezeIcon = QIcon(":freeze.png");
	const QIcon toolsIcon = QIcon(":graph_tools.png");
	const QIcon scaleIcon = QIcon(":absolute_relative.png");
	const QIcon waveformsIcon = QIcon(":waveforms.png");
	const QIcon fftIcon = QIcon(":fft.png");
	const QIcon histoIcon = QIcon(":histo.png");

	mPaletteAct = new QAction(paletteIcon, nullptr, this);
	mPaletteAct->setStatusTip(tr("Open palette"));
	mPaletteAct->setToolTip(tr("Open palette"));
	mPaletteAct->setCheckable(true);

	QAction *cleanAct = new QAction(cleanIcon, nullptr, this);
	cleanAct->setStatusTip(tr("Clean"));
	cleanAct->setToolTip(tr("Clean"));

	mFreezeAct = new QAction(freezeIcon, nullptr, this);
	mFreezeAct->setStatusTip(tr("Freeze"));
	mFreezeAct->setToolTip(tr("Freeze"));
	mFreezeAct->setCheckable(true);

	mToolsAct = new QAction(toolsIcon, nullptr, this);
	mToolsAct->setStatusTip(tr("Open graph tools"));
	mToolsAct->setToolTip(tr("Open graph tools"));
	mToolsAct->setCheckable(true);

	mWaveformsAct = new QAction(waveformsIcon, nullptr, this);
	mWaveformsAct->setStatusTip(tr("WAVEFORMS"));
	mWaveformsAct->setToolTip(tr("WAVEFORMS"));
	mWaveformsAct->setCheckable(true);
	mWaveformsAct->setChecked(true);

	mFFTAct = new QAction(fftIcon, nullptr, this);
	mFFTAct->setStatusTip(tr("FFT"));
	mFFTAct->setToolTip(tr("FFT"));
	mFFTAct->setCheckable(true);

	mHistoAct = new QAction(histoIcon, nullptr, this);
	mHistoAct->setStatusTip(tr("SAMPLES HISTOGRAM"));
	mHistoAct->setToolTip(tr("SAMPLES HISTOGRAM"));
	mHistoAct->setCheckable(true);

	mScaleAct = new QAction(scaleIcon, nullptr, this);
	mScaleAct->setStatusTip(tr("Plot scale absolute/relative"));
	mScaleAct->setToolTip(tr("Plot scale absolute/relative"));
	mScaleAct->setCheckable(true);

	QLabel* label = new QLabel(this);
	label->setText("Plot Type: ");

	mLabel_avg = new QLabel(this);
	mLabel_avg->setText("N Waves to average: ");

	mSBox_avg = new QSpinBox(this);
	mSBox_avg->setMinimum(1);
	mSBox_avg->setMaximum(10);
	mSBox_avg->setValue(1);

	mLabel_rate = new QLabel(this);
	mLabel_rate->setText("Plot time interval [s]: ");

	mDBox_rate = new QDoubleSpinBox(this);
	mDBox_rate->setMinimum(0.1);
	mDBox_rate->setMaximum(10000);
	mDBox_rate->setValue(0.3);
	mDBox_rate->setLocale(QLocale::English);

	connect(mPaletteAct, SIGNAL(triggered()), this, SLOT(showPalette()));
	connect(mFreezeAct, SIGNAL(triggered()), this, SLOT(freezeToggle()));
	connect(mToolsAct, SIGNAL(triggered()), this, SLOT(showTools()));
	connect(mScaleAct, SIGNAL(triggered()), this, SLOT(switchPlotScale()));
	connect(mWaveformsAct, SIGNAL(triggered()), this, SLOT(plotTypeWaveforms()));
	connect(mFFTAct, SIGNAL(triggered()), this, SLOT(plotTypeFFT()));
	connect(mHistoAct, SIGNAL(triggered()), this, SLOT(plotTypeHisto()));
	connect(mSBox_avg, SIGNAL(valueChanged(int)), this, SLOT(NAvgChanged(int)));
	connect(mDBox_rate, SIGNAL(valueChanged(double)), this, SLOT(PlotTimeChanged(double)));
	connect(cleanAct, SIGNAL(triggered()), this, SLOT(ClearPlot()));

	mToolBar->addAction(mFreezeAct);
	mToolBar->addSeparator();
	mToolBar->addAction(cleanAct);
	mToolBar->addSeparator();
	mToolBar->addAction(mPaletteAct);
	mToolBar->addSeparator();
	mToolBar->addAction(mToolsAct);
	mToolBar->addSeparator();
	mToolBar->addWidget(label);
	mToolBar->addAction(mWaveformsAct);
	mToolBar->addAction(mFFTAct);
	mToolBar->addAction(mHistoAct);
	mToolBar->addSeparator();
	mToolBar->addWidget(mLabel_avg);
	mToolBar->addWidget(mSBox_avg);
	mToolBar->addSeparator();
	mlabel_rate_action = mToolBar->addWidget(mLabel_rate);
	mrate_action = mToolBar->addWidget(mDBox_rate);
	mToolBar->addSeparator();
	//mToolBar->addAction(mScaleAct);// add this button to allow switching to absolute scale
	mToolBar->setMovable(false);
}

void plotPanel::enableOfflineMode(bool enable) {
	if (enable) {
		mlabel_rate_action->setVisible(true);
		mrate_action->setVisible(true);
		mTools->enableOfflineMode(true);
		mTab->addTab(mTraceDataPan, "Event Data");
	}
	else {
		mlabel_rate_action->setVisible(false);
		mrate_action->setVisible(false);
		mTools->enableOfflineMode(false);
		mTraceDataPan->removeAllDevices();
		mTab->removeTab(1);
	}
}

void plotPanel::addOfflineDevice(QString devname) {
	mTraceDataPan->addDevice(devname);
}

void plotPanel::setMaxEvNumber(QString device, uint32_t N){
	mTraceDataPan->setMaxEv(device, N);
}

void plotPanel::removeOfflineDevice(QString devname) {
	mTraceDataPan->removeDevice(devname);
}

void plotPanel::showNewEvData(QString name, int ev, quint64 t) {
	mTraceDataPan->showEvData(name,ev,t);
}

void plotPanel::GoToNextEvent(QString devname, int ev) {
	emit NewEventToLoad(devname, ev);
}

void plotPanel::EnableEvDataTab(bool enable) {
	if (mTab->count() > 1) {
		mTab->widget(1)->setEnabled(enable);
		if (!enable)
			mTab->setCurrentIndex(0);
	}
}

void plotPanel::LockDataDevice(int t) {
	mTraceDataPan->DisableDev(t);
}

void plotPanel::UnLockDataDevice(int t) {
	mTraceDataPan->EnableDev(t);
}

void plotPanel::plotData(QVector<QVector<double>> arrayOfXVector, QVector< QVector< double > > arrayOfYVector, QVector<bool> showtrace, QVector<double> skew, double maxy) {
	if (mFreezePlot) 
		return;

	if (mNAvg == 1 || PlotType == PLOT_TYPE_SAMPLES_HISTO) {
		for (int i = 0; i < mTraces; i++) {
			if(mTracesVisible[i])
				mTrace[i]->setData(arrayOfXVector[i], arrayOfYVector[i]);
			mTrace[i]->setVisible(showtrace[i]);
		}

		if (skew[ActiveTrace] > 0) {
			double p = mPreTrgMarker->original_position();
			switch (mUomX) {
			case UOM_SAMPLE:
				mPreTrgMarker->updatePosition(p + skew[ActiveTrace]);
				break;
			case UOM_PHYS_UNIT:
				mPreTrgMarker->updatePosition(p + skew[ActiveTrace] / 1000.);
				break;
			}
		}


		if (PlotType == PLOT_TYPE_SAMPLES_HISTO && maxy > mMaxY) {
			mMaxY = maxy + 100;
			mPlot->yAxis->setRange(mMinY, mMaxY);
		}
		mPlot->replot();
		showCursorPositions();
	}
	else {
		double val, new_val;
		if (mSummed == 0) {
			for (int i = 0; i < mTraces; i++) {
				if(mTracesVisible[i])
					mTrace[i]->setData(arrayOfXVector[i], arrayOfYVector[i]);
			}
		}
		else {
			for (int i = 0; i < mTraces; i++) {
				if (!mTracesVisible[i])
					continue;
				for (int j = 0; j < mTrace[i]->data()->size(); j++)
					(mTrace[i]->data()->begin() + j)->value += arrayOfYVector[i][j];
			}
		}
		mSummed++;
		if (mSummed == mNAvg) {
			for (int i = 0; i < mTraces; i++) {
				if (!mTracesVisible[i])
					continue;
				for (int j = 0; j < mTrace[i]->data()->size(); j++) {
					val = (mTrace[i]->data()->begin() + j)->value;
					(mTrace[i]->data()->begin() + j)->value /= mNAvg;
					new_val = (mTrace[i]->data()->begin() + j)->value;
				}
			}

			mSummed = 0;
			mPlot->replot();
			showCursorPositions();
		}
	}
}

void plotPanel::ThrShow(int checked) {
	if (checked) {
		mThrMarker->setVisible(true);
		mThrVisible = true;
	}
	else {
		mThrMarker->setVisible(false);
		mThrVisible = false;
	}
	mPlot->replot();
}

void plotPanel::XPosShow(int checked) {
	if (checked) {
		mPositionMarker->setVisible(true);
		mPositionVisible = true;
	}
	else {
		mPositionMarker->setVisible(false);
		mPositionVisible = false;
	}
	mPlot->replot();
}

void plotPanel::showPalette() {
	if (mPaletteAct->isChecked()) {
		mPaletteDock->show();
		emit PaletteOpen(true);
	}
	else {
		mPaletteDock->hide();
		emit PaletteOpen(false);
	}
}

void plotPanel::ShowPaletteCmd(bool show) {
	mPaletteAct->setChecked(show);
	mPaletteAct->triggered();
}

void plotPanel::showTools() {
	if (mToolsAct->isChecked()) {
		mPlot->setInteraction(QCP::iRangeZoom, false);
		mPlot->replot();
		mToolsActive = true;
		mTools->UpdateGUI();
		mTools->updateXPos(mXPos);
		mTools->updateOffset(mOffset);
		mTools->updateThr(mThr);
		mTools->updateXUnit(mUomX);
		mTools->updateXMax();
		mTools->updateYUnit(mUomY);
		mTools->updateYMax();
		mToolsDock->show();
		emit ToolsOpen(true);
	}
	else {
		mToolsDock->hide();
		mPlot->setInteraction(QCP::iRangeZoom, true);
		mToolsActive = false;
		emit ToolsOpen(false);
	}
}

void plotPanel::ShowToolsCmd(bool show) {
	mToolsAct->setChecked(show);
	mToolsAct->triggered();
}

void plotPanel::updateBaselines(QString devname, int nbit, QVector<double> factor, QVector<double> baselines) {
	QHash<int, QString>::iterator i;
	for (i = mTracesDevMap.begin(); i != mTracesDevMap.end(); ++i) {
		if (i.value() == devname) {
			int k = i.key();
			int ch = mTracesChMap.value(k);
			if (ch == -1 || baselines.size()<(ch+1))
				continue;
			mTracesBaselines.replace(k, baselines[ch]);
			mTracesADCBits.replace(k, nbit);
			mTracesSampleTomV.replace(k, factor[ch]*1e3);
		}
	}
	UpdateZeroMarkers();
}

void plotPanel::updateGains(QString devname, QVector<double> gains){
	QHash<int, QString>::iterator i;
	for (i = mTracesDevMap.begin(); i != mTracesDevMap.end(); ++i) {
		if (i.value() == devname) {
			int k = i.key();
			int ch = mTracesChMap.value(k);
			if (ch == -1 || gains.size() < (ch + 1))
				return;
			mTracesGains.replace(k, gains[ch]);
		}
	}
	UpdateZeroMarkers();
}

void plotPanel::setPlotScale(int scale, bool enable) {
	mPlotScale = scale;
	/*if (scale == PLOT_SCALE_RELATIVE) {//add this if absolute scale should be used
		mScaleAct->setChecked(true);
	}
	else {
		mScaleAct->setChecked(false);
	}
	mScaleAct->setEnabled(enable);*/
	mTools->plotScaleChanged();
}

void plotPanel::switchPlotScale() {
	if (mScaleAct->isChecked()) {
		mPlotScale = PLOT_SCALE_RELATIVE;
	}
	else {
		mPlotScale = PLOT_SCALE_ABSOLUTE;
	}
	mTools->plotScaleChanged();
	emit ScaleChanged();
}

void plotPanel::PlotTypeChangeCmd(int type) {
	switch (type) {
		case PLOT_TYPE_WAVEFORMS:
			mWaveformsAct->setChecked(true);
			mWaveformsAct->triggered();
			break;
		case PLOT_TYPE_FFT:
			mFFTAct->setChecked(true);
			mFFTAct->triggered();
			break;
		case PLOT_TYPE_SAMPLES_HISTO:
			mHistoAct->setChecked(true);
			mHistoAct->triggered();
			break;
	}

}

void plotPanel::plotTypeWaveforms() {
	mHistoAct->blockSignals(true);
	mHistoAct->setChecked(false);
	mHistoAct->blockSignals(false);
	mFFTAct->blockSignals(true);
	mFFTAct->setChecked(false);
	mFFTAct->blockSignals(false);
	mSBox_avg->setEnabled(true);
	mLabel_avg->setText("N Waves to average");
	mSummed = 0;
	PlotType = PLOT_TYPE_WAVEFORMS;
	mPreTrgMarker->setVisible(true);
	for (int t = 0; t < mTraces; t++)
		mZeroMarker[t]->setVisible(mTracesVisible[t]);
	QString mu = QChar(0xbc, 0x03);
	switch (mUomX) {
	case UOM_SAMPLE:
		mPlot->xAxis->setLabel("samples");
		break;
	case UOM_PHYS_UNIT:
		mPlot->xAxis->setLabel("Time [" + mu + "s]");
		break;
	}
	switch (mUomY) {
	case UOM_SAMPLE:
		mPlot->yAxis->setLabel("ADC channels [LSB]");
		break;
	case UOM_PHYS_UNIT:
		mPlot->yAxis->setLabel("Voltage [mV]");
		break;
	}
	mplottypechanged = true;
	emit TypeChanged();

	mTools->plotTypeChanged(PlotType);
	
}

void plotPanel::plotTypeFFT() {
	mHistoAct->blockSignals(true);
	mHistoAct->setChecked(false);
	mHistoAct->blockSignals(false);
	mWaveformsAct->blockSignals(true);
	mWaveformsAct->setChecked(false);
	mWaveformsAct->blockSignals(false);
	mSBox_avg->setEnabled(true);
	mLabel_avg->setText("N FFT to average");
	mSummed = 0;
	PlotType = PLOT_TYPE_FFT;

	mPlot->xAxis->setLabel("MHz");
	mPlot->yAxis->setLabel("dB");
	mPreTrgMarker->setVisible(false);
	for (int t = 0; t < mTraces; t++)
		mZeroMarker[t]->setVisible(false);
	mplottypechanged = true;

	emit TypeChanged();

	mTools->plotTypeChanged(PlotType);
}

void plotPanel::plotTypeHisto() {
	mWaveformsAct->blockSignals(true);
	mWaveformsAct->setChecked(false);
	mWaveformsAct->blockSignals(false);
	mFFTAct->blockSignals(true);
	mFFTAct->setChecked(false);
	mFFTAct->blockSignals(false);
	mSBox_avg->setEnabled(false);
	mSummed = 0;
	PlotType = PLOT_TYPE_SAMPLES_HISTO;

	if(mUomY == UOM_PHYS_UNIT)
		mPlot->xAxis->setLabel("Voltage [mV]");
	else
		mPlot->xAxis->setLabel("ADC channels [LSB]");
	mPlot->yAxis->setLabel("Counts");
	mPreTrgMarker->setVisible(false);
	for (int t = 0; t < mTraces; t++)
		mZeroMarker[t]->setVisible(false);
	mplottypechanged = true;

	emit TypeChanged();
	
	mTools->plotTypeChanged(PlotType);

}

void plotPanel::NAvgChanged(int n){
	mNAvg = n;
	emit NWaves(n);
}

void plotPanel::NAvgChangeCmd(int n) {
	mSBox_avg->setValue(n);
}

void plotPanel::ClearPlot() {
	for (int i = 0; i < mTraces; i++)
			mTrace[i]->data()->clear();
	mPlot->replot();

	emit ClearPlotData();
}

void plotPanel::freezeToggle() {
	if (mFreezeAct->isChecked()) {
		mFreezePlot = true;
		emit PlotFreeze(true);
	}
	else {
		mFreezePlot = false;
		emit PlotFreeze(false);
	}
}

void plotPanel::FreezeCmd(bool freeze){
	mFreezeAct->setChecked(freeze);
	mFreezeAct->triggered();
}

void plotPanel::updateDevTrace(const QString& device, int t) {
	//update devtrace map
	mTracesDevMap.insert(t, device);
}

void plotPanel::updatePalette( int cmd, const QString& device, const QVector<QString>& names) {
	//fill the traces map with dev name
	if (mTracesDevMap.size() == 0 && cmd == 1) {
		for (int t = 0; t < mTraces; t++) {
			mTracesDevMap.insert(t, device);
		}
	}
	mPalettePan->updatePalette(cmd, device, names);
}

void plotPanel::RemoveDevFromPalette(QString devname) {
	int size = mTracesDevMap.size();
	for (int i = 0; i < size; i++) {
		if (mTracesDevMap.value(i) == devname)
			mTracesDevMap.remove(i);
	}
	//mPalettePan->cbDevChanged(0);
}

void plotPanel::paletteChanged(const QVector<QString>& d, const QVector<int>& c, int t) {
	if (mTracesDevMap.size() == 0)
		return;
	mUpdatingTraces = true;
	mTracesDevMap.insert(t, d.at(t)); //update trace dev map: trace t has new data to be updated
	mTracesChMap.insert(t, c.at(t)); //update trace ch map: trace t has new data to be updated
	emit tracesChanged(d, c);
	if (t == ActiveTrace) {
		mPreTrgMarker->setColor(mTrace[ActiveTrace]->pen().color());  //setColor(mZeroMarker[ActiveTrace]->pen().color());
		emit ActiveTraceChanged(mTracesDevMap.value(ActiveTrace), mTracesChMap.value(ActiveTrace));
	}
	else {
		//mZeroMarker[t]->setLayer("abovelegend");//move up the generic trace marker 
		emit GenericTraceChanged(mTracesDevMap.value(t), mTracesChMap.value(t));
	}
	emit TraceEnabled(t, mTracesVisible[t]);
	mUpdatingTraces = false;
	mPlot->replot();
}

int plotPanel::getTraces() {
	return this->mTraces;
}

void plotPanel::PlotTraceChanged(int t, QString dev_name, int ch) {
	emit TraceChanged(t, dev_name, ch);
}

const QVector<QString>& plotPanel::getDevToPlot() const {
	return mPalettePan->getDevToPlot();
}

const QVector<int>& plotPanel::getChanToPlot() const {
	return mPalettePan->getChanToPlot();
}

void plotPanel::setXAxis(double lower, double upper, int uom) {
	double use_size;
	double xmin = 0;
	switch (PlotType) {
		case PLOT_TYPE_WAVEFORMS:
			mXscale_f = 1.0;
			use_size = upper;
			if (uom != mUomX) {//update uom if necessary
				QString mu = QChar(0xbc, 0x03);
				switch (uom) {
				case UOM_SAMPLE:
					mPlot->xAxis->setLabel("samples");
					break;
				case UOM_PHYS_UNIT:
					mPlot->xAxis->setLabel("Time [" + mu + "s]");
					break;
				}
				mUomX = uom;
				if (mToolsActive)
					mTools->updateXUnit(mUomX);
			}
			break;
		case PLOT_TYPE_SAMPLES_HISTO:
			xmin = lower;
			use_size = upper;
			mXscale_f = 1.0;
			if (uom != mUomY) {//update uom if necessary
				QString mu = QChar(0xbc, 0x03);
				switch (uom) {
				case UOM_SAMPLE:
					mPlot->xAxis->setLabel("ADC channels[LSB]");
					break;
				case UOM_PHYS_UNIT:
					mPlot->xAxis->setLabel("Amplitude [mV]");
					break;
				}
				mUomY = uom;
				if (mToolsActive)
					mTools->updateYUnit(mUomY);
			}
			break;
		case PLOT_TYPE_FFT: {
			FFT f;
			int m;
			use_size = f.NSamplesFFT(upper, &m) -1.;
			mXscale_f = (1000. / mTSampl_ns) / (2. * use_size);			
			break;
		}
	}


	if (upper != mMaxX || mplottypechanged) {
		if (mToolsActive) {
			double xmax;
			if (PlotType != PLOT_TYPE_SAMPLES_HISTO)
				xmax = mPlot->xAxis->range().upper * (use_size / mMaxX) * mXscale_f;//a zoom can be active
			else
				xmax = use_size;
			mMaxX = use_size * mXscale_f;
			mMinX = lower * mXscale_f;
			mTools->updateXMax();
			mPlot->xAxis->setRange(xmin, xmax);
			if(uom == UOM_PHYS_UNIT)
				fixedTickerX->setTickStep(fabs(xmax-xmin) / X_TICKS);
			else
				fixedTickerX->setTickStep(qRound(fabs(xmax - xmin) / X_TICKS));
		}
		else {
			mMaxX = use_size * mXscale_f;
			mMinX = lower * mXscale_f;
			mPlot->xAxis->setRange(xmin, mMaxX);
			if (uom == UOM_PHYS_UNIT)
				fixedTickerX->setTickStep(fabs(mMaxX - xmin) / X_TICKS);
			else
				fixedTickerX->setTickStep(qRound(fabs(mMaxX - xmin) / X_TICKS));
		}	
		mplottypechanged = false;
		UpdateThrMarker();
	}
	
	for (int i = 0; i < mTraces; i++)
		mTrace[i]->data()->clear();
	mPlot->replot();
}

void plotPanel::setYAxis(double lower, double upper, int uom) {
	this->mMinY = lower;
	this->mMaxY = upper;
	mPlot->yAxis->setRange(lower, upper);
	fixedTickerY->setTickStep(fabs(upper - lower) / Y_TICKS);
	mPositionMarker->start->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().lower);//minY
	mPositionMarker->end->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().upper); //mMaxY
	if (uom != mUomY) {
		switch (uom) {
		case UOM_SAMPLE:
			mPlot->yAxis->setLabel("ADC channels [LSB]");
			break;
		case UOM_PHYS_UNIT:
			mPlot->yAxis->setLabel("Voltage [mV]");
			break;
		}
		mUomY = uom;
	}

	if (mToolsActive) {
		mTools->updateYUnit(mUomY);
		mTools->updateYMax();
	}

	UpdateZeroMarkers();
}

void plotPanel::UpdateZeroMarkers(){
	if (mPlotScale == PLOT_SCALE_RELATIVE) {
		for (int t = 0; t < mTraces; t++) {
			double zeroy=0;
			switch (mUomY) {
			case UOM_PHYS_UNIT:
				//zeroy = ((mTracesBaselines[t] - 50.) / 100. * qPow(2, mTracesADCBits[t]) * mTracesSampleTomV[t] * mTracesGains[t]) - ((mTracesBaselines[ActiveTrace] - 50.) / 100. * qPow(2, mTracesADCBits[ActiveTrace]) * mTracesSampleTomV[ActiveTrace] * mTracesGains[ActiveTrace]);
				zeroy = ((mTracesBaselines[t] - 50.) / 100. * qPow(2, mTracesADCBits[t]) * mTracesSampleTomV[t]) - ((mTracesBaselines[ActiveTrace] - 50.) / 100. * qPow(2, mTracesADCBits[ActiveTrace]) * mTracesSampleTomV[ActiveTrace]);
				break;
			case UOM_SAMPLE:
				zeroy = mTracesBaselines[t] / 100. * qPow(2, mTracesADCBits[t]);
				break;
			}
			mZeroTracer[t]->position->setCoords(mPlot->xAxis->axisRect()->left(), zeroy);
		}
	}
	else {
		for (int t = 0; t < mTraces; t++) {
			mZeroTracer[t]->position->setCoords(mPlot->xAxis->axisRect()->left(), 0);
		}
	}
	mPlot->replot();
}

void plotPanel::UpdateThrMarker(){
	if (mUomY == UOM_PHYS_UNIT)
		mThrMarker->updatePosition(mActiveTraceThr, mMaxX);
	else {
		if(mActiveTraceDevClass == DIG2)
			mThrMarker->updatePosition(mActiveTraceThr + (mTracesBaselines[ActiveTrace] / 100.) * qPow(2, mTracesADCBits[ActiveTrace]), mMaxX);
		else
			mThrMarker->updatePosition(mActiveTraceThr, mMaxX);
	}
}

void plotPanel::updateXPosInfo(double max, double min, double incr, double val, double factor) {
	mXPos.clear();
	mXPos.push_back(max);
	mXPos.push_back(min);
	mXPos.push_back(incr);
	mXPos.push_back(val);
	mXPos.push_back(factor);
	mPositionMarker->start->setCoords(mXZoomCenter * val * factor, mMinY);
	mPositionMarker->end->setCoords(mXZoomCenter * val * factor, mMaxY);
	if (mPositionVisible)
		mPlot->replot();
	if (mToolsActive) {
		mTools->updateXPos(mXPos);
	}
 }

void plotPanel::updatePreTrgInfo(double val) {
	mPreTrgMarker->updateOriginalPosition(val);
	mPreTrgMarker->updatePosition(val);
	mPlot->replot();
}

void plotPanel::updateOffsetInfo(double val) {
	mOffset = val;
	mTracesBaselines.replace(ActiveTrace, val);
	if (mPlotScale == PLOT_SCALE_RELATIVE) {
		UpdateThrMarker();
		mPlot->replot();
	}

	if (mToolsActive) {
		mTools->updateOffset(mOffset);
	}
}

void plotPanel::updateOffsetInfo1(QString name, int ch, double val) {
	int T = 0;
	for (T = 0; T < mTraces; T++) {
		if (mTracesDevMap.value(T) == name && mTracesChMap.value(T) == ch)
			break;
	}
	if(T < mTraces)
		mTracesBaselines.replace(T, val);
	UpdateZeroMarkers();
}

void plotPanel::updateThrInfo(double max, double min, double incr, double val, double f) {
	mThr.clear();	
	if (mActiveTraceDevClass == DIG2) {
		mThr.push_back(max);
		mThr.push_back(min);
		mThr.push_back(incr);
		mThr.push_back(val);
		mThr.push_back(f);
		mActiveTraceThr = val * f;
	}
	else {
		if (mUomY == UOM_PHYS_UNIT) {
			mThr.push_back(mMinY + (f / max * max));
			mThr.push_back(mMinY + (f / max * min));
			mThr.push_back(mMinY + (f / max * incr));
			mThr.push_back(mMinY + (f / max * val));//value in mV
			mThr.push_back(1.);
			mActiveTraceThr = mMinY + (f / max * val);
		}
		else {
			mThr.push_back(max);
			mThr.push_back(min);
			mThr.push_back(incr);
			mThr.push_back(val);
			mThr.push_back(1.);//f
			mActiveTraceThr = val;//val * f
		}
	}
	
	if (mToolsActive) {
		mTools->updateThr(mThr);
	}
	UpdateThrMarker();
	if (mThrVisible)
		mPlot->replot();
}

void plotPanel::setThr(double thr) {
	emit ThrChanged(thr, false);
}

void plotPanel::setXPos(double pos) {
	mXZoomCenter = pos;
	mPositionMarker->start->setCoords(mXZoomCenter * fabs(mMaxX-mMinX) + mMinX, mMinY);
	mPositionMarker->end->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().upper); //mMaxY
	if (mPositionVisible)
		mPlot->replot();
	mPlot->replot();
}

void plotPanel::setNewOffset(double value) {
	for (int i = 0; i < mTraces; i++) {
		mTrace[i]->data()->clear();
	}
	mPlot->replot();
	emit OffsetChanged(mTracesDevMap.value(ActiveTrace), mTracesChMap.value(ActiveTrace), value, false);
}

void plotPanel::OffsetToAll(double value) {
	for (int i = 0; i < mTraces; i++) {
		emit OffsetChanged(mTracesDevMap.value(i), mTracesChMap.value(i), value, true);
	}
#ifdef WIN32
                Sleep(300);
#else
                usleep(300000);
#endif
	UpdateZeroMarkers();
}


void plotPanel::ThresholdToAll(double value) {
	emit ThrChanged(value, true);	
}

void plotPanel::setActiveDevClass(int dclass) {
	mActiveTraceDevClass = dclass;
	mTools->setActiveDevClass(dclass);
}

void plotPanel::setActiveTrace(int index) {
	if (index < 0)
		return;
	
	for (int i = 0; i < mTraces; i++) {
		mTrace[i]->data()->clear();
	}
	mPlot->replot();
	mZeroMarker[ActiveTrace]->setLayer("legend");
	ActiveTrace = index;
	mZeroMarker[ActiveTrace]->setLayer("abovelegend");//move up active trace marker 
	mC1Trace = ActiveTrace;//cursors are associated to the active trace
	mC2Trace = ActiveTrace;
	emit ActiveTraceChanged(mTracesDevMap.value(ActiveTrace), mTracesChMap.value(ActiveTrace));
	if (mCursorsEnabled) {
		Cursor1->DeleteFromPlot(mPlot);
		Cursor2->DeleteFromPlot(mPlot);
		NewCursors(true);
	}
	mPreTrgMarker->setColor(mTrace[ActiveTrace]->pen().color());
	mPlot->replot();
}

void plotPanel::TraceColorChanged(int trace, QPen pen){
	QPixmap pix(30, 10);
	QPainter painter(&pix);
	painter.setPen(pen);
	painter.setBrush(QBrush(pen.color(), Qt::SolidPattern));
	painter.drawPolygon(points, 5);
	mZeroMarker[trace]->setPixmap(pix);
	mPlot->replot();
	if (mToolsActive)
		mTools->updateTraces();
	if (trace == ActiveTrace) {		
		mPreTrgMarker->setColor(mTrace[ActiveTrace]->pen().color());
		emit ActiveTraceColorChanged();
	}
	mPlot->replot();
}

void plotPanel::TraceEnableChanged(int trace, int enable) {
	if(PlotType == PLOT_TYPE_WAVEFORMS)
		mZeroMarker[trace]->setVisible((bool)(enable));
	mTrace[trace]->setVisible((bool)(enable));
	mTracesVisible[trace] = (bool)(enable);
	mPlot->replot();
	emit TraceEnabled(trace, enable);
}

void plotPanel::TraceEnableCmd(int trace, bool enable) {
	if (PlotType == PLOT_TYPE_WAVEFORMS)
		mZeroMarker[trace]->setVisible(enable);
	mTrace[trace]->setVisible(enable);
	mTracesVisible[trace] = enable;
	mPalettePan->TraceEnableCmd(trace, enable);
	mPlot->replot();
}

void plotPanel::TraceChangeCmd(int trace, QString dev, int ch) {
	mPalettePan->TraceChangeCmd(trace, dev, ch);
}


//if the user selects the point of the cursor the cursor is enabled
void plotPanel::ElementClicked(QCPAbstractItem *item, QMouseEvent *event) {
	Q_UNUSED(event);
	QCPItemLine* LineItem = dynamic_cast<QCPItemLine*> (item);

	//the cursor is selected by clicking on the point
	if (LineItem) {
		mElementSelected = item->selected();
		item->setSelected((mElementSelected) ? false : true);
		if (mCursorsEnabled) {
			Cursor1->HandleCursorSelect();
			Cursor2->HandleCursorSelect();
		}
	}
	else {
		if (mCursorsEnabled && Cursor1->isSelected()) {
			Cursor1->setSelected(false);
			Cursor1->HandleCursorSelect();
		}
		if (mCursorsEnabled && Cursor2->isSelected()) {
			Cursor2->setSelected(false);
			Cursor2->HandleCursorSelect();
		}
	}

	showCursorPositions();
	mPlot->replot();//TODO:check if this can e removed when autoreplot is active
}

//unzoom on double click
void plotPanel::onMouseDoubleClick(QMouseEvent *event) {
	Q_UNUSED(event);
	if (mToolsActive)
		return;
	mPlot->xAxis->setRange(mMinX, mMaxX);
	mPlot->yAxis->setRange(mMinY, mMaxY);
	mPositionMarker->start->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().lower);//minY
	mPositionMarker->end->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().upper); //mMaxY
	mPlot->replot();
}

//last draw of the cursor when mouse is released after move
void plotPanel::onMouseRelease(QMouseEvent *event) {
	//if (mToolsActive)
	//	return;
	double xc;
	xc = mPlot->xAxis->pixelToCoord(event->pos().x());
	if (mCursorsEnabled) {
		if (Cursor1->isSelected()) {
			Cursor1->DrawCursor(xc, mMinY, mPlot->yAxis->range().upper);
			ElementClicked(Cursor1->getLine(), event);
		}
		if (Cursor2->isSelected()) {
			Cursor2->DrawCursor(xc, mMinY, mPlot->yAxis->range().upper);
			ElementClicked(Cursor2->getLine(), event);
		}
		showCursorPositions();
	}
	if (mRubberBand->isVisible()) {
		auto Rect = mRubberBand->geometry();
		int x1, y1, x2, y2;
		Rect.getCoords(&x1, &y1, &x2, &y2);
		auto xP1 = mPlot->xAxis->pixelToCoord(x1);
		auto xP2 = mPlot->xAxis->pixelToCoord(x2);
		//set new x axis range
		mPlot->xAxis->setRange(xP1, xP2);
		fixedTickerX->setTickStep((xP2 - xP1) / X_TICKS);

		auto yP1 = mPlot->yAxis->pixelToCoord(y1);
		auto yP2 = mPlot->yAxis->pixelToCoord(y2);

		mPlot->yAxis->setRange(yP1, yP2);
		fixedTickerX->setTickStep((yP2 - yP1) / Y_TICKS);

		mRubberBand->hide();		
	}
	mPositionMarker->start->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().lower);//minY
	mPositionMarker->end->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().upper); //mMaxY
	mPlot->replot(); //TODO:check if this can e removed when autoreplot is active
	return;
}

//if the cursor is selected move it, else print only its coordinates on the lineedits
void plotPanel::onMouseMove(QMouseEvent *event)
{
	double xc = mPlot->xAxis->pixelToCoord(event->pos().x());
	//double yc = mPlot->yAxis->pixelToCoord(event->pos().y());

	if(mCursorsEnabled){
		bool replot = false;
		if (Cursor1->isSelected()) {
			Cursor1->DrawCursor(xc, mMinY, mPlot->yAxis->range().upper);
			replot = true;
		}
		if (Cursor2->isSelected()) {
			Cursor2->DrawCursor(xc, mMinY, mPlot->yAxis->range().upper);
			replot = true;
		}
		if (replot) {
			showCursorPositions();
			mPlot->replot();
		}
	}

	if (QApplication::keyboardModifiers().testFlag(Qt::ControlModifier) == true && !mToolsActive) {
		if (mRubberBand->isVisible())
			mRubberBand->setGeometry(QRect(mRectOrigin, event->pos()).normalized());
	}

	return;
}

//last draw of the cursor when mouse is released after move
void plotPanel::onMousePress(QMouseEvent *event)
{
	if (QApplication::keyboardModifiers().testFlag(Qt::ControlModifier) == true && !mToolsActive) {
		mRectOrigin = event->pos();
		mRubberBand->setGeometry(QRect(mRectOrigin, QSize()));
		mRubberBand->show();
	}
	else {
		if (mCursorsEnabled) {
			double x = mPlot->xAxis->pixelToCoord(event->pos().x());
			int ticks = mPlot->xAxis->ticker()->tickCount();
			double interval = 0.5*(mPlot->xAxis->range().upper / (double)ticks) / 5.0;
			if ((x >= Cursor1->CurrentPositionX() - interval) && (x <= Cursor1->CurrentPositionX() + interval) && !Cursor2->isSelected())
				ElementClicked(Cursor1->getLine(), event);
			if ((x >= Cursor2->CurrentPositionX() - interval) && (x <= Cursor2->CurrentPositionX() + interval) && !Cursor1->isSelected())
				ElementClicked(Cursor2->getLine(), event);
		}
	}
	return;
}

void plotPanel::onXRangeChanged(const QCPRange &range)
{
	QCPRange boundedRange = range;
	double lowerRangeBound = mMinX;
	// restrict max zoom in
	if (boundedRange.lower < lowerRangeBound) {
		boundedRange.lower = lowerRangeBound;
	}// restrict max zoom out
	if (boundedRange.upper > mMaxX) {
		boundedRange.upper = mMaxX;
	}
	mPlot->xAxis->setRange(boundedRange);
	fixedTickerX->setTickStep((boundedRange.upper - boundedRange.lower) / X_TICKS);
}

void plotPanel::onYRangeChanged(const QCPRange &range)
{
	QCPRange boundedRange = range;
	double lowerRangeBound = mMinY;
	// restrict max zoom in
	if (boundedRange.lower < lowerRangeBound) {
		boundedRange.lower = lowerRangeBound;
	}
	// restrict max zoom out
	if (boundedRange.upper > mMaxY) {
		boundedRange.upper = mMaxY;
	}
	mPlot->yAxis->setRange(boundedRange);
	fixedTickerY->setTickStep((boundedRange.upper - boundedRange.lower) / Y_TICKS);
	mPositionMarker->start->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().lower);//minY
	mPositionMarker->end->setCoords(mXZoomCenter * fabs(mMaxX - mMinX) + mMinX, mPlot->yAxis->range().upper); //mMaxY

	if (mCursorsEnabled) {
		double xc = Cursor1->CurrentPositionX();
		Cursor1->DrawCursor(xc, mMinY, boundedRange.upper);
		xc = Cursor2->CurrentPositionX();
		Cursor2->DrawCursor(xc, mMinY, boundedRange.upper);
	}
}

void plotPanel::NewCursors(bool enable) {
	if (enable) {
		double x_025_scale = mPlot->xAxis->range().lower + 0.25*(mPlot->xAxis->range().upper - mPlot->xAxis->range().lower);
		QCPGraph *cursor_trace = mTrace[mC1Trace];
		Cursor1 = new VCursor(mPlot, x_025_scale, cursor_trace);
		QPen pen;
		pen.setColor(Qt::black);
		QPen penSelect;
		penSelect.setColor(Qt::yellow);
		Cursor1->SetColors(pen, penSelect);
		Cursor1->DrawCursor(x_025_scale, mMinY, mPlot->yAxis->range().upper);		
		double test = mPlot->yAxis->range().upper;
		double test2 = mMaxY;

		cursor_trace = mTrace[mC2Trace];
		double x_075_scale = mPlot->xAxis->range().lower + 0.75*(mPlot->xAxis->range().upper - mPlot->xAxis->range().lower);
		Cursor2 = new VCursor(mPlot, x_075_scale, cursor_trace);
		Cursor2->SetColors(pen, penSelect);
		Cursor2->DrawCursor(x_075_scale, mMinY, mPlot->yAxis->range().upper);

		mCursorsEnabled = true;
		showCursorPositions();
	}
	else {
		Cursor1->DeleteFromPlot(mPlot);
		Cursor2->DeleteFromPlot(mPlot);
		delete Cursor1;
		delete Cursor2;
		mCursorsEnabled = false;
	}
	mPlot->replot();
}

double plotPanel::FindXWithNonZeroY(QCPGraph *trace, int start, int end) {
	double value_y = 0;
	for (int i = (start * 100); i < (end * 100); i++) {
		value_y = trace->data()->at(i)->value;
		if (value_y > 10)
			return (double)i / 100.0;
	}
	return ((double)end - (double)start) / 2.0;
}

void plotPanel::showCursorPositions() {
	if (mCursorsEnabled) {
		QVector<double> pos;
		pos.push_back(Cursor1->CurrentPositionX());
		pos.push_back(Cursor1->CurrentPositionY());
		pos.push_back(Cursor2->CurrentPositionX());
		pos.push_back(Cursor2->CurrentPositionY());
		emit NewCursorPositions(pos);
	}
}

void plotPanel::calcNewCursorPositions(double factor) {
	if (mCursorsEnabled) {
			Cursor1->DrawCursor(Cursor1->CurrentPositionX() * factor, mMinY, mPlot->yAxis->range().upper);
			Cursor2->DrawCursor(Cursor2->CurrentPositionX() * factor, mMinY, mPlot->yAxis->range().upper);
	}		
}

void plotPanel::PlotTimeChanged(double val) {
	emit ChangePlotTime(val);
}

