/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		palettePanel.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "palettePanel.h"

#include <QScrollArea>
#include <QLabel>
#include <QPushButton>
#include <QSpacerItem>
#include <QVBoxLayout>
#include <QTextEdit>
#include <QComboBox>
#include <QColorDialog>

#include "CAENPushButton.h"


palettePanel::palettePanel(QWidget *parent, int ntraces, QCPGraph **traces)
	: QWidget(parent)
	, mTraces(traces)
	, mNtraces(ntraces)
	, mCbDev(ntraces)
	, mCbChannels(ntraces)
	, mChTraces(ntraces)
{

	QVBoxLayout *paletteVerticalLayout = new QVBoxLayout();
	QVBoxLayout *scrolllLayout = new QVBoxLayout();
	this->setLayout(paletteVerticalLayout);
	QScrollArea *area = new QScrollArea(this);
	area->setFrameShape(QFrame::NoFrame);

	paletteVerticalLayout->addWidget(area);
	paletteVerticalLayout->setContentsMargins(0, 5, 0, 0);
	scrolllLayout->setContentsMargins(0, 0, 0, 0);

	QWidget *palette = new QWidget(this);

	palette->setLayout(scrolllLayout);
	

	for (int i = 0; i < ntraces; i++) {
		QWidget *panel = new QWidget(this);
		QVBoxLayout *panelLayout = new QVBoxLayout();
		panelLayout->setContentsMargins(3, 0, 0, 0);
		panelLayout->setSpacing(3);
		panel->setLayout(panelLayout);

		panel->setContentsMargins(0, 0, 0, 4);

		QHBoxLayout *row1Layout = new QHBoxLayout();
		QHBoxLayout *row2Layout = new QHBoxLayout();
		row1Layout->setContentsMargins(5, 0, 5, 0);
		row1Layout->setSpacing(10);
		row2Layout->setContentsMargins(5, 0, 5, 0);
		row2Layout->setSpacing(10);
		mChTraces[i] = new QCheckBox(this);
		mChTraces[i]->setProperty("index",i);
		mChTraces[i]->setChecked(true);
		QLabel *label = new QLabel("<b>Trace " + QString::number(i) + "</b>", this);
		CAENPushButton *pbColor = new CAENPushButton(this,i);
		pbColor->setText("");
		pbColor->setMaximumWidth(20);
		pbColor->setMaximumHeight(12);
		pbColor->setFlat(true);
		QPalette pal = pbColor->palette();
		pal.setColor(QPalette::Button, traces[i]->pen().color());
		pbColor->setAutoFillBackground(true);
		pbColor->setPalette(pal);
		pbColor->update();
		connect(pbColor, &CAENPushButton::clicked, [=] {openPalette(pbColor); });
		connect(mChTraces[i], SIGNAL(stateChanged(int)), this, SLOT(traceEnable(int)));
		QSpacerItem *s = new QSpacerItem(40, 5, QSizePolicy::Expanding, QSizePolicy::Minimum);
		row1Layout->addWidget(mChTraces[i]);
		row1Layout->addWidget(label);
		row1Layout->addWidget(pbColor);
		row1Layout->addItem(s);
		panelLayout->addLayout(row1Layout);

		mCbDev[i] = new QComboBox(this);
		mCbDev[i]->setProperty("index", i);
		mCbDev[i]->setContentsMargins(0, 0, 0, 0);
		mCbDev[i]->setMinimumWidth(150);
		connect(mCbDev[i], SIGNAL(currentIndexChanged(int)), this, SLOT(singlecbDevChanged(int)));
		mCbChannels[i] = new QComboBox(this);
		mCbChannels[i]->setProperty("index", i);
		mCbChannels[i]->setContentsMargins(0, 0, 0, 0);
		connect(mCbChannels[i], SIGNAL(currentIndexChanged(int)), this, SLOT(cbChannelChanged(int)));
		row2Layout->addWidget(mCbDev[i]);
		row2Layout->addWidget(mCbChannels[i]);
		panelLayout->addLayout(row2Layout);
		
		scrolllLayout->addWidget(panel);
	}
	scrolllLayout->addStretch(1);
	area->setWidget(palette);

	connect(this, SIGNAL(colorChanged(int,QPen)), parent, SLOT(TraceColorChanged(int,QPen)));
	connect(this, SIGNAL(traceEnableChanged(int, int)), parent, SLOT(TraceEnableChanged(int, int)));
	connect(this, SIGNAL(singleDevChanged(const QString &, int)), parent, SLOT(updateDevTrace(const QString &, int)));
	connect(this, SIGNAL(TraceToPlotChanged(int, QString, int)), parent, SLOT(PlotTraceChanged(int, QString, int)));
	
}

palettePanel::~palettePanel()
{
}


void palettePanel::traceEnable(int enable) {
	QCheckBox *cb = (QCheckBox *)QObject::sender();
	QVariant trace = cb->property("index");
	emit traceEnableChanged(trace.toInt(), enable);
}

void palettePanel::TraceEnableCmd(int trace, bool enable) {
	mChTraces[trace]->setChecked(enable);
	mChTraces[trace]->setCheckState(enable ? Qt::CheckState::Checked : Qt::CheckState::Unchecked);
	mChTraces[trace]->stateChanged(enable);	
}

void palettePanel::openPalette(CAENPushButton *b) {
	const QColor color = QColorDialog::getColor(b->palette().button().color(), this);
	if (color.isValid()) {
		QPalette pal = b->palette();
		pal.setColor(QPalette::Button, color);
		b->setAutoFillBackground(true);
		b->setPalette(pal);
		b->update();
		mTraces[b->getIndex()]->setPen(QPen(color));
		emit colorChanged(b->getIndex(), QPen(color));
	}
}

void palettePanel::updatePalette(int cmd, const QString& device, const QVector<QString>& names) {
	const bool wasEmpty = mDev.isEmpty();
	switch (cmd) {
	case 0: {// remove
		for (int idx = 0; idx < mDev.size(); idx++) {
			if (mDev.at(idx) == device) {
				mDev.removeAt(idx);
				mChannels.removeAt(idx);
				if (mDev.size() > 0) {
					for (int t = 0; t < mNtraces; t++) {
						if(mDevToPlot[t] == device)
							mDevToPlot[t] = mDev[0];
					}
				}

			}
		}

		for (int i = 0; i < mNtraces; i++) {
			if (mCbDev[i]->currentText() == device)
				if (mDev.size() > 0) {
					emit singleDevChanged(mDev[0], i);
					for (int tr = 0; tr < mCbDev.size(); tr++) {//check if switching to dev 0 the corresponding ch is already selected
						if (tr == i)
							continue;
						if ((mDevToPlot[tr] == mDev[0]) && (mChanToPlot[tr] == mChanToPlot[i])) {
							int ch = 0;
							for (ch = 0; ch < mChannels[0].size(); ch++) {
								if (!mChanToPlot.contains(ch)) {
									mChanToPlot[i] = ch;
									mCbChannels[i]->blockSignals(true);
									mCbChannels[i]->setCurrentIndex(mChanToPlot[i]);
									mCbChannels[i]->blockSignals(false);
									emit(paletteChanged(this->mDevToPlot, this->mChanToPlot, i));
									break;
								}
							}
						}
					}
				}
		}
		break;
	}
	case 1: // add
		mDev.append(device);
		mChannels.append(names);
		break;
	}


	for (int i = 0; i < mNtraces; i++) {
		mCbDev[i]->blockSignals(true);
		mCbDev[i]->clear();
		for (int j = 0; j < mDev.size(); j++) {
			mCbDev[i]->addItem(mDev[j]);
		}
		if (wasEmpty) {
			mCbChannels[i]->blockSignals(true);
			mCbChannels[i]->clear();
			mCbChannels[i]->blockSignals(false);
		}
		mCbDev[i]->blockSignals(false);
	}
	if (wasEmpty) {
		for (int i = 0; i < mNtraces; i++) {
			mCbDev[i]->setCurrentIndex(0);
		}
		cbDevChanged(0);
		this->mChanToPlot.clear();
		for (int i = 0; i < mNtraces; i++) {
			mCbChannels[i]->blockSignals(true);
			mCbChannels[i]->setCurrentIndex(i);
			this->mChanToPlot.append(mCbChannels[i]->currentIndex());
			mCbChannels[i]->blockSignals(false);
		}
	}
}

void palettePanel::cbDevChanged(int idx) {
	for (int i = 0; i < mNtraces; i++) {
		mCbDev[i]->blockSignals(true);
		mCbDev[i]->setCurrentIndex(idx);
		mCbChannels[i]->blockSignals(true);
		mCbChannels[i]->clear();

		if (mChannels.size() != 0) {
			for (int j = 0; j < mChannels[idx].size(); j++) {
				mCbChannels[i]->blockSignals(true);
				mCbChannels[i]->addItem(mChannels[idx][j]);
				mCbChannels[i]->blockSignals(false);
			}
		}
		mCbChannels[i]->blockSignals(false);
		mCbDev[i]->blockSignals(false);
	}
	this->mDevToPlot.clear();
	this->mChanToPlot.clear();
	for (int i = 0; i < mNtraces; i++) {
		this->mDevToPlot.append(mCbDev[i]->currentText());
		this->mChanToPlot.append((mCbChannels[i]->currentIndex() >= 0)? mCbChannels[i]->currentIndex() : 0);
	}
	emit(paletteChanged(this->mDevToPlot, this->mChanToPlot, idx));
}

void palettePanel::singlecbDevChanged(int idx){
	Q_UNUSED(idx);
	QComboBox *cbDev = (QComboBox *)QObject::sender();
	QVariant trace = cbDev->property("index");
	cbDev->blockSignals(true);
	for (int i = 0; i < mCbDev.size(); i++) {//channel already selected for plotting
		if ((mDevToPlot[trace.toInt()] == mCbDev[i]->currentText()) && (mChanToPlot[trace.toInt()] == mCbChannels[i]->currentIndex()) && (mChanToPlot[trace.toInt()] != -1)) {
			QString message = mCbDev[i]->currentText() + mCbChannels[i]->currentText() + " already selected for plot.";
			QMessageBox::critical(this, "Error", message, QMessageBox::Ok, 0);
			cbDev->setCurrentText(mDevToPlot[trace.toInt()]);
		}
	}
	this->mDevToPlot[trace.toInt()] = mCbDev[trace.toInt()]->currentText();
	cbDev->blockSignals(false);
	mCbChannels[trace.toInt()]->blockSignals(true);
	int count = mCbChannels[trace.toInt()]->count();
	if (mChannels[idx].size() > count) {
		for (int j = mCbChannels[trace.toInt()]->count(); j < mChannels[idx].size(); j++) {
			mCbChannels[trace.toInt()]->addItem(mChannels[idx][j]);
		}
	}
	else{
		for (int j = mChannels[idx].size(); j < count; j++) {
			mCbChannels[trace.toInt()]->removeItem(mCbChannels[trace.toInt()]->count()-1);
		}
		this->mChanToPlot[trace.toInt()] = 0;
		mCbChannels[trace.toInt()]->setCurrentIndex(0);
	}
	mCbChannels[trace.toInt()]->blockSignals(false);
	emit(paletteChanged(this->mDevToPlot, this->mChanToPlot, trace.toInt()));
	emit TraceToPlotChanged(trace.toInt(), this->mDevToPlot[trace.toInt()], this->mChanToPlot[trace.toInt()]);
}

void palettePanel::cbChannelChanged(int idx) {
	Q_UNUSED(idx);
	QComboBox *cbChan = (QComboBox *) QObject::sender();
	QVariant trace = cbChan->property("index");
	cbChan->blockSignals(true);

	for (int i = 0; i < mCbDev.size(); i++) {//channel already selected for plotting
		if ((mDevToPlot[trace.toInt()] == mCbDev[i]->currentText()) && (mChanToPlot[i] == cbChan->currentIndex())) {
			QString message = mCbDev[i]->currentText() + QString(" CH %1").arg(cbChan->currentIndex()) + " already selected for plot.";
			QMessageBox::critical(this,"Error", message , QMessageBox::Ok, 0);
			cbChan->setCurrentIndex(mChanToPlot[trace.toInt()]);
			break;
		}
	}

	this->mDevToPlot[trace.toInt()] = mCbDev[trace.toInt()]->currentText();
	this->mChanToPlot[trace.toInt()] = cbChan->currentIndex();

	cbChan->blockSignals(false);
	emit(paletteChanged(this->mDevToPlot, this->mChanToPlot, trace.toInt()));
	emit TraceToPlotChanged(trace.toInt(), this->mDevToPlot[trace.toInt()], this->mChanToPlot[trace.toInt()]);
}

void palettePanel::TraceChangeCmd(int trace, QString dev, int ch) {
	int index = mCbDev[trace]->findText(dev);
	this->mChanToPlot[trace] = ch;
	this->mDevToPlot[trace] = dev;
	mCbDev[trace]->setCurrentIndex(index);
	mCbChannels[trace]->blockSignals(true);
	mCbChannels[trace]->setCurrentIndex(ch);
	mCbChannels[trace]->blockSignals(false);
}


const QVector<QString>& palettePanel::getDevToPlot() const {
	return this->mDevToPlot;
}

const QVector<int>& palettePanel::getChanToPlot() const {
	return this->mChanToPlot;
}


