/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		infoPanel.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "infoPanel.h"
#include <QDebug>
#include "CAENFELibException.h"


infoPanel::infoPanel(QWidget* parent, QList<CAENDevice*>devices, QString active_dev)
	: QWidget(parent)
{
	ui.setupUi(this);
	setWindowIcon(QIcon(":WAVEDump2.png"));
	setWindowModality(Qt::ApplicationModal);
	setWindowTitle("Device Info");
	setAttribute(Qt::WA_DeleteOnClose);
	

	QString line_total = "";
	QString T_names[6] = { "Temp Sens Air In" ,"Temp Sens Air Out", "Temp Sens Core", "Temp Sens First ADC", "Temp Sens Last ADC", "Temp Sens DC DC" };
	
	QVector<QString> Infos;
	
	if (active_dev == "ALL") {
		for (int i = 0; i < devices.size(); i++) {
			mDevs.append(devices.at(i));
		}
	}
	else {
		for (int i = 0; i < devices.size(); i++) {
			if (devices.at(i)->getName() == active_dev) {
				mDevs.append(devices.at(i));
				break;
			}
		}
	}
	
	for (int i = 0; i < mDevs.size(); i++) {
		line_total = "";
		line_total = "Device name: " + mDevs.at(i)->getName() + "\n";
		line_total.append("Model: " + mDevs.at(i)->getModel());
		line_total.append("\n");
		line_total.append("Form factor: " + mDevs.at(i)->getFormFactor());
		line_total.append("\n");
		line_total.append("FW type: " + mDevs.at(i)->getFWType());
		try {
			line_total.append("  " + mDevs.at(i)->getFWGbE());
		}
		catch (CAENFELibException& exc) {
			line_total.append(" (error reading FW GbE) ");
		}
			line_total.append("\n");
			line_total.append("FW release: " + mDevs.at(i)->getFWRel());
			line_total.append("\n");
			line_total.append("PID: " + mDevs.at(i)->getSN());
			line_total.append("\n");
			line_total.append("Number of channels: " + QString("%1").arg(mDevs.at(i)->getNumCh()));
			line_total.append("\n");
			Infos.append(line_total);
			QTableWidget* t;
			if(mDevs.at(i)->DeviceClass == CAEN_DIG2)
				t = new QTableWidget(6, 2);
			else
				t = new QTableWidget(mDevs.at(i)->getNumCh(), 2);
			t->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
			t->horizontalHeader()->setVisible(false);
			t->verticalHeader()->setVisible(false);
			t->setEnabled(false);

			mTtables.append(t);
	}
	
	//create the items of column 1
	QString sDegree = u8"\u00B0";
	for (int i = 0; i < mTtables.size(); i++) {
		if (mDevs.at(i)->isVirtual())
			break;
		if (mDevs.at(i)->DeviceClass == CAEN_DIG2) {
			for (int r = 0; r < 6; r++) {
				QTableWidgetItem* newItem0_0 = new QTableWidgetItem(T_names[r]);
				newItem0_0->setTextAlignment(Qt::AlignHCenter);
				mTtables.at(i)->setItem(r, 0, newItem0_0);
			}
			CAENDig2Device* d2 = static_cast<CAENDig2Device*>(mDevs.at(i));
			QTableWidgetItem* newItem0_1 = new QTableWidgetItem(QString("%1").arg(d2->getTAirIn()) + sDegree + "C");
			newItem0_1->setTextAlignment(Qt::AlignHCenter);
			mTtables.at(i)->setItem(0, 1, newItem0_1);
			QTableWidgetItem* newItem1_1 = new QTableWidgetItem(QString("%1").arg(d2->getTAirOut()) + sDegree + "C");
			newItem1_1->setTextAlignment(Qt::AlignHCenter);
			mTtables.at(i)->setItem(1, 1, newItem1_1);
			QTableWidgetItem* newItem2_1 = new QTableWidgetItem(QString("%1").arg(d2->getTCore()) + sDegree + "C");
			newItem2_1->setTextAlignment(Qt::AlignHCenter);
			mTtables.at(i)->setItem(2, 1, newItem2_1);
			QTableWidgetItem* newItem3_1 = new QTableWidgetItem(QString("%1").arg(d2->getTFirstADC()) + sDegree + "C");
			newItem3_1->setTextAlignment(Qt::AlignHCenter);
			mTtables.at(i)->setItem(3, 1, newItem3_1);
			QTableWidgetItem* newItem4_1 = new QTableWidgetItem(QString("%1").arg(d2->getTLastADC()) + sDegree + "C");
			newItem4_1->setTextAlignment(Qt::AlignHCenter);
			mTtables.at(i)->setItem(4, 1, newItem4_1);
			QTableWidgetItem* newItem5_1 = new QTableWidgetItem(QString("%1").arg(d2->getTDCDC()) + sDegree + "C");
			newItem5_1->setTextAlignment(Qt::AlignHCenter);
			mTtables.at(i)->setItem(5, 1, newItem5_1);
		}
		else {
			if (mDevs.at(i)->getFamilyCode().contains("730") || mDevs.at(i)->getFamilyCode().contains("725") || mDevs.at(i)->getFamilyCode().contains("751")) {
				CAENDig1Device* d1 = static_cast<CAENDig1Device*>(mDevs.at(i));
				int Nch = mDevs.at(i)->getNumCh();
				
				for (int ch = 0; ch < Nch; ch++) {
					QTableWidgetItem* newItem0_0 = new QTableWidgetItem(QString("Ch %1 ADC Temp").arg(ch));
					newItem0_0->setTextAlignment(Qt::AlignHCenter);
					mTtables.at(i)->setItem(ch, 0, newItem0_0);
					QTableWidgetItem* newItem0_1 = new QTableWidgetItem(QString("%1").arg(d1->getChADCTemp(ch)) + sDegree + "C");
					newItem0_1->setTextAlignment(Qt::AlignHCenter);
					mTtables.at(i)->setItem(ch, 1, newItem0_1);
				}
			}
		}
	}

	QGridLayout* gr_hv = new QGridLayout;
	for (int N = 0; N < Infos.size(); N++) {
		QTextEdit* txt_info = new QTextEdit(this);
		txt_info->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
		txt_info->setStyleSheet("QTextEdit {background-color: rgb(215, 253, 255);}");
		txt_info->setAlignment(Qt::AlignCenter);
		txt_info->setEnabled(false);
		QFont font_text;
		font_text.setPointSize(13);
		txt_info->setFont(font_text);
		txt_info->setText(Infos.at(N));
		txt_info->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
		gr_hv->addWidget(txt_info, N, 0);		
	}
	for (int j = 0; j < mTtables.size(); j++) {
		gr_hv->addWidget(mTtables.at(j), j, 1);
	}
	this->setLayout(gr_hv);

	TTimer = new QTimer(this);
	QObject::connect(TTimer, SIGNAL(timeout()), this, SLOT(showTemperatures()));
	if(!mDevs.at(0)->isVirtual())
		TTimer->start(3000);
	
}

void infoPanel::Quit() {
	TTimer->stop();
	close();
}

void infoPanel::showTemperatures() {
	QString sDegree = u8"\u00B0";

	for (int i = 0; i < mTtables.size(); i++) {
		if (mDevs.at(i)->DeviceClass == CAEN_DIG2) {
			CAENDig2Device* d2 = static_cast<CAENDig2Device*>(mDevs.at(i));
			mTtables.at(i)->item(0, 1)->setText(QString("%1").arg(d2->getTAirIn()) + sDegree + "C");
			mTtables.at(i)->item(1, 1)->setText(QString("%1").arg(d2->getTAirOut()) + sDegree + "C");
			mTtables.at(i)->item(2, 1)->setText(QString("%1").arg(d2->getTCore()) + sDegree + "C");
			mTtables.at(i)->item(3, 1)->setText(QString("%1").arg(d2->getTFirstADC()) + sDegree + "C");
			mTtables.at(i)->item(4, 1)->setText(QString("%1").arg(d2->getTLastADC()) + sDegree + "C");
			mTtables.at(i)->item(5, 1)->setText(QString("%1").arg(d2->getTDCDC()) + sDegree + "C");
		}
		else {
			if (mDevs.at(i)->getFamilyCode().contains("730") || mDevs.at(i)->getFamilyCode().contains("725") || mDevs.at(i)->getFamilyCode().contains("751")) {
				CAENDig1Device* d1 = static_cast<CAENDig1Device*>(mDevs.at(i));
				for (int r = 0; r < mTtables.at(i)->rowCount(); r++)
					mTtables.at(i)->item(r, 1)->setText(QString("%1").arg(QString("%1").arg(d1->getChADCTemp(r)) + sDegree + "C"));
			}
		}
	}
}

infoPanel::~infoPanel()
{

}

void infoPanel::closeEvent(QCloseEvent* event)
{
	TTimer->stop();
	event->accept();
}
