/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		eventBuildThread.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "eventBuildThread.h"
#include "WaveDump2.h"
#include <iostream>
#include <fstream>
#include <algorithm>

 using namespace std;


eventBuildThread::eventBuildThread(WaveDump2 *w, QList<CAENDevice *> *devices)
{
	mWaveDump = w;
	mDevices = devices;
	mTotDevices = mDevices->size();

	int Nchannels = 0;
	for (int d = 0; d < mTotDevices; d++)
		Nchannels += mDevices->at(d)->getNumCh();

	//init the global event
	mGlobalEvent = new GlobalEvent();
	*mGlobalEvent->T() = 0;
	*mGlobalEvent->Fl() = 0;
	*mGlobalEvent->ID() = 0;
	mNSamples = mWaveDump->getALLMaxRecLenS();
	mGlobalEvent->init(MAX_NDEV, MAX_NCH, mNSamples);
	mCoincWindow = mWaveDump->getCoincWin();

	if (mWaveDump->OfflineMode)
		PLOT_TIME = mWaveDump->PlotInterval_ms;
	else {
		if (mNSamples < 1000) //8 us
			PLOT_TIME = 300;
		else if (mNSamples < 30000) //240 us
			PLOT_TIME = 500;
		else if (mNSamples < 50000) //400 us
			PLOT_TIME = 600;
		else if (mNSamples < 60000) // 480 us
			PLOT_TIME = 700;
		else if (mNSamples < 70000) //560 us
			PLOT_TIME = 800;
		else if (mNSamples < 88000) //704 us
			PLOT_TIME = 1000;
		else if (mNSamples < 120000) //960 us
			PLOT_TIME = 1500;
		else if (mNSamples < 250000) //2 ms
			PLOT_TIME = 2000;
		else if (mNSamples < 375000) //3 ms
			PLOT_TIME = 2500;
		else if (mNSamples < 625000) //5 ms
			PLOT_TIME = 3000;
		else
			PLOT_TIME = 4000;
	}

	//init fft
	mFFT = FFT();
	mFFT.Init(mNSamples);

	InitFiles();

	//init the plot vector
	mPlotVectorY.resize(PLOT_TRACES);
	for (int i = 0; i < PLOT_TRACES; i++) {
		mPlotVectorY[i].resize(mNSamples); //MAX_PLOT_VECTOR_SIZE
		mPlotVectorY[i].fill(0);
	}

	mPlotVectorX.resize(PLOT_TRACES);
	for (int i = 0; i < PLOT_TRACES; i++) {
		mPlotVectorX[i].resize(mNSamples); //MAX_PLOT_VECTOR_SIZE
		for (int j = 0; j < mNSamples; j++)
			mPlotVectorX[i][j] = j; // this is the x axe
	}
	mShowTrace.resize(PLOT_TRACES);
	mShowTrace.fill(true);

	mSample_to_V.resize(PLOT_TRACES);
	mSample_to_V.fill(1.0);

	mGain.resize(PLOT_TRACES);
	mGain.fill(1.0);

	mNSkew.resize(PLOT_TRACES);
	mNSkew.fill(0.);

	//clear statistics
	mStats = mWaveDump->Stats();
	mStats->clear();

	mGlobalStartMode = mWaveDump->getGlobalSMode();
	connect(this, SIGNAL(GenericError(QString)), mWaveDump, SLOT(ReadoutError1(QString)));
}


void eventBuildThread::updateXVector() {
	double sampl_to_us;
	switch (mWaveDump->getPlotType()) {
		case PLOT_TYPE_WAVEFORMS:
			for (int t = 0; t < PLOT_TRACES; t++) {
				mPlotVectorX[t].resize(mNSamples);
				switch (mUoM_x) {
					case UOM_SAMPLE:
						for (int i = 0; i < mNSamples; i++) {
							mPlotVectorX[t][i] = i; //samples
						}
						break;
					case UOM_PHYS_UNIT:
						sampl_to_us = mWaveDump->getDevSampleToS(mWaveDump->getPlotTraceDev(t)) * mWaveDump->getDevDecimation(mWaveDump->getPlotTraceDev(t)) * 1e+6; //micro-seconds
						for (int i = 0; i < mNSamples; i++) {
							mPlotVectorX[t][i] = i * sampl_to_us; 
						}
						break;
				}
			}
			break;
		case PLOT_TYPE_FFT:
			for (int t = 0; t < PLOT_TRACES; t++) {
				mPlotVectorX[t].resize(mNSamples);
				for (int i = 0; i < mNSamples; i++) {
					int m;
					mPlotVectorX[t][i] = i * (1000. / mWaveDump->getDevTSampl_ns(mWaveDump->getPlotTraceDev(t))) / (mFFT.NSamplesFFT(mNSamples, &m)); //MHz
				}
			}
			break;
		case PLOT_TYPE_SAMPLES_HISTO:
			for (int i = 0; i < PLOT_TRACES; i++) {
				mPlotVectorY[i].resize(MAX_PLOT_VECTOR_SIZE);
				mPlotVectorY[i].fill(0);
			}

			switch (mUoM_y) {
				case UOM_SAMPLE:
					for (int t = 0; t < PLOT_TRACES; t++) {
						mPlotVectorX[t].resize(MAX_PLOT_VECTOR_SIZE);
						for (int i = 0; i < MAX_PLOT_VECTOR_SIZE; i++) {
							mPlotVectorX[t][i] = i; //samples
						}
					}
					break;
				case UOM_PHYS_UNIT:
					for (int t = 0; t < PLOT_TRACES; t++) {
						mPlotVectorX[t].resize(MAX_PLOT_VECTOR_SIZE);
						for (int i = 0; i < MAX_PLOT_VECTOR_SIZE; i++) {
							mPlotVectorX[t][i] = i + mWaveDump->getPlotMinX();
						}
					}
					break;
				}
			break;
	}

	mUpdateXUnit = false;
}

void eventBuildThread::updatePlots(int plot, const QString& deviceName, int channel) {
	//int idx;
	for (int dev = 0; dev < mTotDevices; dev++) {
		if (deviceName == mDevices->at(dev)->getName()) {
			QMutexLocker l(&mMtx);
			mHashPlot.remove(plot);
			mHashPlot.insert(plot, dev * MAX_NCH + channel);
			mSample_to_V.replace(plot, mDevices->at(dev)->getSample_to_V(channel));
			mGain.replace(plot, mDevices->at(dev)->getChGain(channel));
			break;
		}
	}
}

void eventBuildThread::fillPlotVector(GlobalEvent *ev) {
	if (mWaveDump->baseline_changing)
		return;
	
	int index;
	int bin;
	int size;
	double MaxY = mWaveDump->getPlotMaxY();
	double baseline = (mDevices->at(mWaveDump->getActivePlotTraceDevIndex())->getBaselineLSB(mWaveDump->getActivePlotTraceCh()) - 0.5 * mDevices->at(mWaveDump->getActivePlotTraceDevIndex())->getVRangeS()) * mSample_to_V.at(mWaveDump->getActiveTrace()) * 1e3;
	if (mUpdateXUnit)
		updateXVector();
	if (mClearVector) {
		for (int i = 0; i < PLOT_TRACES; i++) {
			mPlotVectorY[i].fill(0);
		}
		mClearVector = false;
	}

	QMutexLocker l(&mGMtx);
	for (int dev = 0; dev < mTotDevices; dev++) {
		for (int ch = 0; ch < mDevices->at(dev)->getNumCh(); ch++) {
			index = dev*MAX_NCH + ch;
			int idx = mHashPlot.key(index, -1);	
			
			if (idx != -1) {
				mShowTrace[idx] = mWaveDump->isTraceVisible(idx);
				if (!mShowTrace[idx])
					continue;
				if (!ev->IsDev()[dev]) {
					mShowTrace[idx] = false;
					continue;
				}
				if (ev->Samples(dev).size() == 0)
					break;
				if (ev->Samples(dev)[ch] != 0) {	
						switch (mWaveDump->getPlotType()) {
							case PLOT_TYPE_WAVEFORMS:
								mPlotVectorY[idx].clear();
								switch (mUoM_y) {
								case UOM_SAMPLE:
									std::copy(ev->W()[index], ev->W()[index] + ev->Samples(dev)[ch], std::back_inserter(mPlotVectorY[idx]));
									break;
								case UOM_PHYS_UNIT:
									std::copy(ev->W()[index], ev->W()[index] + ev->Samples(dev)[ch], std::back_inserter(mPlotVectorY[idx]));
									for (unsigned int j = 0; j < ev->Samples(dev)[ch]; j++) {
										mPlotVectorY[idx][j] -= 0.5 * mDevices->at(mWaveDump->getPlotTraceDevIndex(idx))->getVRangeS();	
										mPlotVectorY[idx][j] *= mSample_to_V.at(idx) * 1e3;
										mPlotVectorY[idx][j] -= baseline;
									}
									break;
								}
								mPlotVectorY[idx].insert(ev->Samples(dev)[ch], mNSamples - ev->Samples(dev)[ch], mWaveDump->getPlotMinY());
							break;
							case PLOT_TYPE_FFT:
								mPlotVectorY[idx].fill(0);
								size = mFFT.CalculateFFT(ev->W()[index], ev->Samples(dev)[ch], mWaveDump->getADCNBit(mDevices->at(dev)->getName()), BLACKMAN_FFT_WINDOW, mPlotVectorY[idx].data());
							break;
							case PLOT_TYPE_SAMPLES_HISTO:
								switch (mUoM_y) {
								case UOM_SAMPLE:
									for (uint s = 0; s < ev->Samples(dev)[ch]; s++) {
										mPlotVectorY[idx][ev->W()[index][s]] ++;
										if (mPlotVectorY[idx][ev->W()[index][s]] > MaxY)
											MaxY = mPlotVectorY[idx][ev->W()[index][s]];
									}
									break;
								case UOM_PHYS_UNIT:
									for (uint s = 0; s < ev->Samples(dev)[ch]; s++) {
										bin = qRound(((ev->W()[index][s] - 0.5 * mDevices->at(mWaveDump->getPlotTraceDevIndex(idx))->getVRangeS()) * mSample_to_V.at(idx) * 1e3) - baseline - mWaveDump->getPlotMinX());
										mPlotVectorY[idx][bin] ++;
										if (mPlotVectorY[idx][bin] > MaxY)
											MaxY = mPlotVectorY[idx][bin];
									}
									break;
								}
							break;

						}
				}
			}
		}
	}

	emit plotData(mPlotVectorX, mPlotVectorY, mShowTrace, mNSkew, MaxY);
}


void eventBuildThread::fillPlotVectorSkew(GlobalEvent * ev) {
	if (mWaveDump->baseline_changing)
		return;

	int index;
	int size;
	int bin;
	int skew_samples;
	double MaxY = mWaveDump->getPlotMaxY();
	double baseline = (mDevices->at(mWaveDump->getActivePlotTraceDevIndex())->getBaselineLSB(mWaveDump->getActivePlotTraceCh()) - 0.5 * mDevices->at(mWaveDump->getActivePlotTraceDevIndex())->getVRangeS()) * mSample_to_V.at(mWaveDump->getActivePlotTraceDevIndex()) * 1e3;

	if (mUpdateXUnit)
		updateXVector();

	QMutexLocker l(&mGMtx);
	for (int dev = 0; dev < mTotDevices; dev++) {

		skew_samples = ev->Wskew()[dev] / (1e9 * mDevices->at(dev)->getSample_to_S());
		for (int ch = 0; ch < mDevices->at(dev)->getNumCh(); ch++) {
			index = dev*MAX_NCH + ch;
			int idx = mHashPlot.key(index, -1);			
			if (idx != -1) {
				mShowTrace[idx] = mWaveDump->isTraceVisible(idx);
				if (!mShowTrace[idx])
					continue;
				if (!ev->IsDev()[dev]) {
					mShowTrace[idx] = false;
					continue;
				}
				if (mPlotVectorX[idx].size() < mNSamples) { //restore x vector to the original size
					int k = 0;
					double sampl_to_us = mWaveDump->getDevSampleToS(mWaveDump->getPlotTraceDev(idx)) * mWaveDump->getDevDecimation(mWaveDump->getPlotTraceDev(idx)) * 1e+6; //micro-seconds
					switch (mUoM_x) {
					case UOM_SAMPLE:
						while (mPlotVectorX[idx].size() < mNSamples) {
							mPlotVectorX[idx].insert(0, k);
							k++;
						}
						break;
					case UOM_PHYS_UNIT:
						while (mPlotVectorX[idx].size() < mNSamples) {
							mPlotVectorX[idx].insert(0, k * sampl_to_us);
							k++;
						}
						break;
					}
				}
				if (ev->Samples(dev)[ch] != 0) { //manage the skew samples
					mPlotVectorX[idx].remove(0, skew_samples);
					mNSkew[idx] = (mUoM_x == UOM_SAMPLE)? skew_samples : ev->Wskew()[dev];
					switch (mWaveDump->getPlotType()) {
						case PLOT_TYPE_WAVEFORMS:
							mPlotVectorY[idx].clear();
							switch (mUoM_y) {
							case UOM_SAMPLE:
								std::copy(ev->W()[index], ev->W()[index] + ev->Samples(dev)[ch] - skew_samples, std::back_inserter(mPlotVectorY[idx]));
								break;
							case UOM_PHYS_UNIT:
								std::copy(ev->W()[index], ev->W()[index] + ev->Samples(dev)[ch] - skew_samples, std::back_inserter(mPlotVectorY[idx]));
								for (unsigned int j = 0; j < mPlotVectorY[idx].size(); j++) {
									mPlotVectorY[idx][j] -= 0.5 * mDevices->at(mWaveDump->getPlotTraceDevIndex(idx))->getVRangeS();
									mPlotVectorY[idx][j] *= mSample_to_V.at(idx) * 1e3;
									mPlotVectorY[idx][j] -= baseline;
								}
								break;
							}
						break;
						case PLOT_TYPE_FFT:
							mPlotVectorY[idx].fill(0);
							size = mFFT.CalculateFFT(ev->W()[index], ev->Samples(dev)[ch], mWaveDump->getADCNBit(mDevices->at(dev)->getName()), BLACKMAN_FFT_WINDOW, mPlotVectorY[idx].data());
						break;
						case PLOT_TYPE_SAMPLES_HISTO:
							switch (mUoM_y) {
							case UOM_SAMPLE:
								for (uint s = 0; s < ev->Samples(dev)[ch]; s++) {
									mPlotVectorY[idx][ev->W()[index][s]] ++;
									if (mPlotVectorY[idx][ev->W()[index][s]] > MaxY)
										MaxY = mPlotVectorY[idx][ev->W()[index][s]];
								}
								break;
							case UOM_PHYS_UNIT:
								for (uint s = 0; s < ev->Samples(dev)[ch]; s++) {
									bin = qRound(((ev->W()[index][s] - 0.5 * mDevices->at(mWaveDump->getPlotTraceDevIndex(idx))->getVRangeS()) * mSample_to_V.at(idx) * 1e3) - baseline - mWaveDump->getPlotMinX());
									mPlotVectorY[idx][bin] ++;
									if (mPlotVectorY[idx][bin] > MaxY)
										MaxY = mPlotVectorY[idx][bin];
								}
								break;
							}
						break;
					}
				}
			}
		}
	}
	emit plotData(mPlotVectorX,mPlotVectorY, mShowTrace, mNSkew, MaxY);
}

void eventBuildThread::run() {

	mNSkew.fill(0);

	if (mGlobalStartMode.contains("ASYNC"))
		RunAsyncMode();
	else {
		if (mWaveDump->OnlyMatching) {
			if(!mWaveDump->OfflineMode)
				RunSyncMode();
			else
				RunSyncOfflineMode();
		}			
		else {
			if (!mWaveDump->OfflineMode)
				RunSyncIndependentMode();
			else
				RunSyncIndependentOfflineMode();
		}
	}

	exit(0);
}

void eventBuildThread::RunSyncMode() {
	int Wait;									// One baord has no data but the timeout has not been reached => keep waiting for data
	int AllEmpty;								// All the boards are empty
	int AllPresent;								// All the boards participate to the global event with data
	int AvailableEvent[MAX_NDEV] = { 0 };		// one event has been read from the device
	int Purged[MAX_NDEV] = { 0 };				// A board becomes Purged when it doesn't provide data for a given maximum timeout
	QElapsedTimer nodatatime[MAX_NDEV];		// Time of the oldest attempt to read the board with no data (used for the timeout)
	int Nev[MAX_NDEV] = { 0 };				//event index of the dev queue
	std::uint64_t tstamp[MAX_NDEV] = { 0 };		//event timestamp in ns
	std::uint32_t ev_ID[MAX_NDEV] = { 0 };			//Trigger ID
	std::uint64_t PrevEvID[MAX_NDEV] = { 0 };		// Previous trigger ID
	int ev_id = 0;
	int first_plot = 1;
	int NEvts = 0;

	QDateTime lastPlotTime = QDateTime::currentDateTime();
	for (int dev = 0; dev < mTotDevices; dev++) {
		nodatatime[dev].start();
		mGlobalEvent->IsDev()[dev] = false;
	}
	while (!mEndThread) {
		while (mDoRead) {
			AllEmpty = 1;
			Wait = 0;

			for (int dev = 0; dev < mTotDevices; dev++) {

				if (!mDevices->at(dev)->AcqStarted) {//at least one device has stopped acquiring data--->stop sync mode
					mDoRead = false;
					mEndThread = true;
					break;
				}

				if (!AvailableEvent[dev]) {  //not yet read one evnt from dev 

					if (mDevices->at(dev)->NEvtsInQueue == 0) { //no new data to read from dev
						if (!Purged[dev]) { //timeout not yet reached
							if (nodatatime[dev].elapsed() == 0) { //nodatatime is 0--> the last time it was read dev had some data
								Wait = 1; //keep waiting
							}
							else if (nodatatime[dev].elapsed() > WAITDATA_TIMEOUT_MS) { // timeout --> restart timer and exclude dev from this global event
								Purged[dev] = 1;
								nodatatime[dev].start();
							}
							else {
								Wait = 1; //keep waiting
							}
						}
					}
					else { //dev has some events to read
						if (mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_ID < PrevEvID[dev]) {
							--mDevices->at(dev)->NEvtsInQueue;
							continue;
						}
						AllEmpty = 0;
						nodatatime[dev].start();
						Purged[dev] = 0;
						AvailableEvent[dev] = 1;

						QMutexLocker locker(&mDevices->at(dev)->queue_mutex);
						QMutexLocker l(&mGMtx);
						//read the first event from the queue
						tstamp[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_Timestamp * mDevices->at(dev)->TSamplUnit_ns; //mDevices->at(dev)->getTSampl_clk()
						ev_ID[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_ID;

						std::uint32_t* s = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes();
						mGlobalEvent->NumSamples[dev].clear();
						std::copy(s, s + mDevices->at(dev)->getNumCh(), std::back_inserter(mGlobalEvent->NumSamples[dev]));
						for (int ch = 0; ch < mDevices->at(dev)->getNumCh(); ch++) {
							std::uint16_t* w = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->data()[ch];
							if (w != nullptr)
								std::copy(w, w + mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes()[ch], mGlobalEvent->W()[dev * MAX_NCH + ch]);
							if (mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_chmask & ((uint64_t)1 << ch))
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = true;
							else
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = false;
						}
						const auto max_size_it = std::max_element(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes(), mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes() + mDevices->at(dev)->getNumCh());
						mGlobalEvent->MaxSizes[dev] = *max_size_it;

						mStats->IncrReadTrgCnt(dev, 1);
						mStats->IncrReadByteCnt(dev, mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						mStats->IncrTotBytesCnt(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						if ((ev_ID[dev] > 0) && (ev_ID[dev] - PrevEvID[dev]) > 1) {
							mStats->IncrLostTrgCnt(dev, (ev_ID[dev] - PrevEvID[dev] - 1) % EV_ID_MAX);
						}
						PrevEvID[dev] = ev_ID[dev];

						Nev[dev] = (++Nev[dev]) % MAX_NEVTS;
						if(mDevices->at(dev)->NEvtsInQueue == MAX_NEVTS)
							mDevices->at(dev)->waitCondition.wakeAll();
						
						--mDevices->at(dev)->NEvtsInQueue;
					}
				}
				else { //dev data already collected
					AllEmpty = 0;
				}

			}//dev loop

			if (!mDoRead)
				break;

			if (AllEmpty) { //reset timers if all devs are empty
				int stop = 0;
				for (int dev = 0; dev < mTotDevices; dev++) {
					nodatatime[dev].start();
					if (mDevices->at(dev) == nullptr)
						continue;
					if (mDevices->at(dev)->isQueueClosed())
						stop++;
				}
				if (stop == mTotDevices) {//all the events queues are closed-->readout is finished
					for (int dev = 0; dev < mTotDevices; dev++)
						mDevices->at(dev)->freeEvtsQueue();
					mEndThread = true;
					break;
				}
			}
			if (Wait || AllEmpty) {
				//plot
				QDateTime currentTime = QDateTime::currentDateTime();
				qint64 timeDiff = lastPlotTime.msecsTo(currentTime);
				if(mWaveDump->OfflineMode)
					PLOT_TIME = mWaveDump->PlotInterval_ms;
				if (timeDiff > PLOT_TIME && !mEndThread) {
					mNew_plot = QtConcurrent::run([this] { return fillPlotVectorSkew(mGlobalEvent); });
					lastPlotTime = QDateTime::currentDateTime();
				}

				mNew_plot.waitForFinished();
				continue;
			}

			// Get oldest time stamp and Trigger ID
			std::uint64_t oldest_ts = 0xFFFFFFFFFFFFFFFF;
			std::uint32_t oldest_trgid = 0xFFFFFFFF;
			for (int dev = 0; dev < mTotDevices; dev++) {
				if (AvailableEvent[dev]) {
					if (tstamp[dev] < oldest_ts) 
						oldest_ts = tstamp[dev];
					if (ev_ID[dev] < oldest_trgid) 
						oldest_trgid = ev_ID[dev];
					mGlobalEvent->BoardEvID()[dev] = ev_ID[dev];
				}
			}

			*mGlobalEvent->T() = oldest_ts;
			//*mGlobalEvent->ID() = oldest_trgid;
				
			mStats->SetCurrentTimestamp_us(static_cast<double>(*mGlobalEvent->T()) / 1000.);
			
			AllPresent = 1;
			for (int dev = 0; dev < mTotDevices; dev++) {
				mGlobalEvent->IsDev()[dev] = true;
				mGlobalEvent->Wskew()[dev] = static_cast<uint32_t>(tstamp[dev] - oldest_ts);

				if ((!AvailableEvent[dev]) || ((tstamp[dev] - oldest_ts) > mCoincWindow)) { // No data from this board or this event is ahead in time => keep it for the next call
					mStats->IncrBuildMissCnt(dev, 1);
					AllPresent = 0;
					mGlobalEvent->IsDev()[dev] = false;
					continue;
				}

				mStats->IncrProcTrgCnt(dev, 1);				

				AvailableEvent[dev] = 0;
			}

			
			if (!AllPresent) {
				mStats->IncrOutOfSyncEvCnt(1);
				for (int dev = 0; dev < mTotDevices; dev++)
					mGlobalEvent->IsDev()[dev] = false; //reset for plot
			}
			else {
				mStats->IncrGlobalEvCnt(1);
				*mGlobalEvent->ID() = ev_id;
				ev_id++;

				//save event to file if save is enabled
				if (mSaving) {
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
					if (mWaveDump->mDataSavedEnabled.load() == 1) {
#else
					if (mWaveDump->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
						if (NEvts == mNEvts_file && mNEvts_file) {
							this->mSaveFile->close();
							this->mTestFile->close();
							delete this->mSaveFile;
							delete this->mTestFile;
							InitFiles();
							NEvts = 0;
						}
						//save data to the sync test file
						outtest << *mGlobalEvent->ID() << "\t";
						for (int dev = 0; dev < mTotDevices; dev++)
							outtest << tstamp[dev] << "\t\t\t";
						outtest << ENDL;
						outtest.flush();
						SaveEventToFile(mGlobalEvent);
						NEvts++;
					}
				}
			}


			//plot
			QDateTime currentTime = QDateTime::currentDateTime();
			qint64 timeDiff = lastPlotTime.msecsTo(currentTime);
			if (mWaveDump->OfflineMode)
				PLOT_TIME = mWaveDump->PlotInterval_ms;
			if ((timeDiff > PLOT_TIME || first_plot) && !mEndThread) {
				if (first_plot)
					first_plot = 0;
				mNew_plot = QtConcurrent::run([this] { return fillPlotVectorSkew(mGlobalEvent); });
				lastPlotTime = QDateTime::currentDateTime();
			}

			mNew_plot.waitForFinished();

			for (int dev = 0; dev < mTotDevices; dev++)
				mGlobalEvent->IsDev()[dev] = false; //final reset also if allpresent


		}
		if (mSaving) {
			this->mSaveFile->close();
			this->mTestFile->close();
		}
	}
	for (int dev = 0; dev < mTotDevices; dev++) { //unlock the queue to avoid possible lck
		mDevices->at(dev)->NEvtsInQueue = 0;
		mDevices->at(dev)->waitCondition.wakeAll();
	}
}

void eventBuildThread::RunSyncIndependentMode() {
	int Wait;									// One baord has no data but the timeout has not been reached => keep waiting for data
	int AllEmpty;								// All the boards are empty
	int AllPresent;								// All the boards participate to the global event with data
	int AvailableEvent[MAX_NDEV] = { 0 };		// one event has been read from the device
	int Purged[MAX_NDEV] = { 0 };				// A board becomes Purged when it doesn't provide data for a given maximum timeout
	QElapsedTimer nodatatime[MAX_NDEV];		// Time of the oldest attempt to read the board with no data (used for the timeout)
	int Nev[MAX_NDEV] = { 0 };				//event index of the dev queue
	std::uint64_t tstamp[MAX_NDEV] = { 0 };		//event timestamp in ns
	std::uint32_t ev_ID[MAX_NDEV] = { 0 };			//Trigger ID
	std::uint64_t PrevEvID[MAX_NDEV] = { 0 };		// Previous trigger ID
	int ev_id = 0;
	int first_plot = 1;
	int NEvts = 0;

	QDateTime lastPlotTime = QDateTime::currentDateTime();
	for (int dev = 0; dev < mTotDevices; dev++) {
		nodatatime[dev].start();
		mGlobalEvent->IsDev()[dev] = false;
	}
	while (!mEndThread) {
		while (mDoRead) {
			AllEmpty = 1;
			Wait = 0;

			for (int dev = 0; dev < mTotDevices; dev++) {

				if (!mDevices->at(dev)->AcqStarted) {//at least one device has stopped acquiring data--->stop sync mode
					mDoRead = false;
					mEndThread = true;
					break;
				}
				if (!AvailableEvent[dev]) {  //not yet read one evnt from dev 
					mGlobalEvent->NumSamples[dev].clear();
					if (mDevices->at(dev)->NEvtsInQueue == 0) { //no new data to read from dev
						if (!Purged[dev]) { //timeout not yet reached
							if (nodatatime[dev].elapsed() == 0) { //nodatatime is 0--> the last time it was read dev had some data
								Wait = 1; //keep waiting
							}
							else if (nodatatime[dev].elapsed() > WAITDATA_TIMEOUT_MS) { //timeout --> restart timer and exclude dev from this global event
								Purged[dev] = 1;
								nodatatime[dev].start();
							}
							else {
								Wait = 1; //keep waiting
							}
						}
					}
					else { //dev has some events to read
						//if (mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_ID < PrevEvID[dev]) {
						//	--mDevices->at(dev)->NEvtsInQueue;
						//	continue;
						//}
						AllEmpty = 0;
						nodatatime[dev].start();
						Purged[dev] = 0;
						AvailableEvent[dev] = 1;

						QMutexLocker locker(&mDevices->at(dev)->queue_mutex);
						QMutexLocker l(&mGMtx);
						//read the first event from the queue
						tstamp[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_Timestamp * mDevices->at(dev)->TSamplUnit_ns; //mDevices->at(dev)->getTSampl_clk()
						ev_ID[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_ID;

						std::uint32_t* s = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes();
						
						std::copy(s, s + mDevices->at(dev)->getNumCh(), std::back_inserter(mGlobalEvent->NumSamples[dev]));
						for (int ch = 0; ch < mDevices->at(dev)->getNumCh(); ch++) {
							std::uint16_t* w = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->data()[ch];
							if (w != nullptr)
								std::copy(w, w + mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes()[ch], mGlobalEvent->W()[dev * MAX_NCH + ch]);
							if (mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_chmask & ((uint64_t)1 << ch))
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = true;
							else
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = false;
						}
						const auto max_size_it = std::max_element(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes(), mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes() + mDevices->at(dev)->getNumCh());
						mGlobalEvent->MaxSizes[dev] = *max_size_it;

						mStats->IncrReadTrgCnt(dev, 1);
						mStats->IncrReadByteCnt(dev, mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						mStats->IncrTotBytesCnt(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						if ((ev_ID[dev] > 0) && (ev_ID[dev] - PrevEvID[dev]) > 1) {
							mStats->IncrLostTrgCnt(dev, (ev_ID[dev] - PrevEvID[dev] - 1) % EV_ID_MAX);
						}
						PrevEvID[dev] = ev_ID[dev];

						Nev[dev] = (++Nev[dev]) % MAX_NEVTS;
						if (mDevices->at(dev)->NEvtsInQueue == MAX_NEVTS)
							mDevices->at(dev)->waitCondition.wakeAll();
						--mDevices->at(dev)->NEvtsInQueue;
					}
				}
				else { //dev data already collected
					AllEmpty = 0;
				}

			}//dev loop

			if (AllEmpty) { //reset timers if all devs are empty
				int stop = 0;
				for (int dev = 0; dev < mTotDevices; dev++) {
					nodatatime[dev].start();
					if (mDevices->at(dev) == nullptr)
						continue;
					if (mDevices->at(dev)->isQueueClosed())
						stop++;
				}
				if (stop == mTotDevices) {//all the events queues are closed-->readout is finished
					for (int dev = 0; dev < mTotDevices; dev++)
						mDevices->at(dev)->freeEvtsQueue();
					mEndThread = true;
					break;
				}
			}
			if (Wait || AllEmpty) {
				//plot
				QDateTime currentTime = QDateTime::currentDateTime();
				qint64 timeDiff = lastPlotTime.msecsTo(currentTime);
				if (mWaveDump->OfflineMode)
					PLOT_TIME = mWaveDump->PlotInterval_ms;
				if (timeDiff > PLOT_TIME && !mEndThread) {
					mNew_plot = QtConcurrent::run([this] { return fillPlotVectorSkew(mGlobalEvent); });
					lastPlotTime = QDateTime::currentDateTime();
				}

				mNew_plot.waitForFinished();
				continue;
			}

			// Get oldest time stamp and Trigger ID
			std::uint64_t oldest_ts = 0xFFFFFFFFFFFFFFFF;
			std::uint32_t oldest_trgid = 0xFFFFFFFF;
			for (int dev = 0; dev < mTotDevices; dev++) {
				if (AvailableEvent[dev]) {
					if (tstamp[dev] < oldest_ts)
						oldest_ts = tstamp[dev];
					if (ev_ID[dev] < oldest_trgid)
						oldest_trgid = ev_ID[dev];
					mGlobalEvent->BoardEvID()[dev] = ev_ID[dev];
				}
			}

			*mGlobalEvent->T() = oldest_ts;
			//*mGlobalEvent->ID() = oldest_trgid;
			*mGlobalEvent->ID() = ev_id;
			ev_id++;

			mStats->SetCurrentTimestamp_us(static_cast<double>(*mGlobalEvent->T()) / 1000.);

			AllPresent = 1;
			for (int dev = 0; dev < mTotDevices; dev++) {

				mGlobalEvent->Wskew()[dev] = static_cast<uint32_t>(tstamp[dev] - oldest_ts);

				if ((!AvailableEvent[dev]) || ((tstamp[dev] - oldest_ts) > mCoincWindow)) { // No data from this board or this event is ahead in time => keep it for the next call
					AllPresent = 0;
					mGlobalEvent->IsDev()[dev] = false;
					continue;
				}
				mGlobalEvent->IsDev()[dev] = true;
				mStats->IncrProcTrgCnt(dev, 1);

				AvailableEvent[dev] = 0;
			}

			//save event to file if save is enabled
			if (mSaving) {
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
				if (mWaveDump->mDataSavedEnabled.load() == 1) {
#else
				if (mWaveDump->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
					if (NEvts == mNEvts_file && mNEvts_file) {
						this->mSaveFile->close();
						this->mTestFile->close();
						delete this->mSaveFile;
						delete this->mTestFile;
						InitFiles();
						NEvts = 0;
					}
					//save data to the sync test file
					outtest << *mGlobalEvent->ID() << "\t";
					for (int dev = 0; dev < mTotDevices; dev++) {
						if (mGlobalEvent->IsDev()[dev])
							outtest << tstamp[dev] << "\t\t\t\t\t";
						else
							outtest << "\t\t\t\t\t";
					}
					outtest << ENDL;
					outtest.flush();
					SaveEventToFile(mGlobalEvent);
					NEvts++;
				}
			}
			

			//plot
			QDateTime currentTime = QDateTime::currentDateTime();
			qint64 timeDiff = lastPlotTime.msecsTo(currentTime);
			if (mWaveDump->OfflineMode)
				PLOT_TIME = mWaveDump->PlotInterval_ms;
			if ((timeDiff > PLOT_TIME || first_plot) && !mEndThread) {
				if (first_plot)
					first_plot = 0;
				mNew_plot = QtConcurrent::run([this] { return fillPlotVectorSkew(mGlobalEvent); });
				lastPlotTime = QDateTime::currentDateTime();
			}

			mNew_plot.waitForFinished();

			for (int dev = 0; dev < mTotDevices; dev++) 
				mGlobalEvent->IsDev()[dev] = false; //reset for plot
		}
		if (mSaving) {
			this->mSaveFile->close();
			this->mTestFile->close();
		}
	}
}



void eventBuildThread::RunAsyncMode() {
	int AllEmpty;								// All the boards are empty
	int AllPresent;								// All the boards participate to the global event with data
	int AvailableEvent[MAX_NDEV] = { 0 };		// one event has been read from the device
	int Nev[MAX_NDEV] = { 0 };				//event index of the dev queue
	std::uint64_t tstamp[MAX_NDEV] = { 0 };		//event timestamp in ns
	std::uint32_t ev_ID[MAX_NDEV] = { 0 };			//Trigger ID
	std::uint64_t PrevEvID[MAX_NDEV] = { 0 };		// Previous trigger ID
	int ev_id = 0;
	QDateTime lastPlotTime = QDateTime::currentDateTime();
	int first_plot = true;
	int first_ev = 1;

	while (!mEndThread) {
		while (mDoRead) {
			AllEmpty = 1;
			for (int dev = 0; dev < mTotDevices; dev++) {
				if (!mDevices->at(dev)->AcqStarted)
					continue;
				if (!AvailableEvent[dev]) {  //not yet read one evnt from dev 

					if (mDevices->at(dev)->NEvtsInQueue) { //dev has some events to read
						QMutexLocker locker(&mDevices->at(dev)->queue_mutex);
						QMutexLocker l(&mGMtx);

						//read the first event from the queue
						tstamp[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_Timestamp * mDevices->at(dev)->TSamplUnit_ns; //mDevices->at(dev)->getTSampl_clk()
						ev_ID[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_ID;
						std::uint32_t* s = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes();
						mGlobalEvent->NumSamples[dev].clear();
						std::copy(s, s + mDevices->at(dev)->getNumCh(), std::back_inserter(mGlobalEvent->NumSamples[dev]));
						for (int ch = 0; ch < mDevices->at(dev)->getNumCh(); ch++) {
							std::uint16_t* w = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->data()[ch];
							if (w != nullptr)
								std::copy(w, w + mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes()[ch], mGlobalEvent->W()[dev * MAX_NCH + ch]);
							if (mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_chmask & ((uint64_t)1 << ch))
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = true;
							else
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = false;
						}

						AllEmpty = 0;
						mGlobalEvent->IsDev()[dev] = true;
						mStats->IncrReadTrgCnt(dev, 1);
						mStats->IncrReadByteCnt(dev, mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						mStats->IncrTotBytesCnt(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);

						if (ev_ID[dev] > 0 && (ev_ID[dev] > PrevEvID[dev])) {
							if (first_ev && mWaveDump->OfflineMode) {
								first_ev = false;
							}
							else
								mStats->IncrLostTrgCnt(dev, (ev_ID[dev] - PrevEvID[dev] - 1) % EV_ID_MAX);
						}
						
						PrevEvID[dev] = ev_ID[dev];
						AvailableEvent[dev] = 1;

						Nev[dev] = (++Nev[dev]) % MAX_NEVTS;
						if (mDevices->at(dev)->NEvtsInQueue == MAX_NEVTS) {
							--mDevices->at(dev)->NEvtsInQueue;
							mDevices->at(dev)->waitCondition.wakeAll();
						}
						else	
							--mDevices->at(dev)->NEvtsInQueue;
					}
				}
				else { //dev data already collected
					AllEmpty = 0;
				}

			}//dev loop

			if (AllEmpty) { //reset timers if all devs are empty
				int stop = 0;
				for (int dev = 0; dev < mTotDevices; dev++) {
					if (mDevices->at(dev)->isQueueClosed())
						stop++;
				}
				if (stop == mTotDevices) {//all the events queues are closed-->readout is finished
					for (int dev = 0; dev < mTotDevices; dev++)
						mDevices->at(dev)->freeEvtsQueue();
					mEndThread = true;
					break;
				}
			}
			if (AllEmpty)
				continue;

			*mGlobalEvent->ID() = ev_id;
			ev_id++;

			AllPresent = 1;
			for (int dev = 0; dev < mTotDevices; dev++) {
				if (!mDevices->at(dev)->AcqStarted)
					continue;
				if (!AvailableEvent[dev]) { // No data from this board 
					//mStats->IncrBuildMissCnt(dev, 1);
					AllPresent = 0;
					continue;
				}
				mStats->IncrProcTrgCnt(dev, 1);
				mGlobalEvent->Wskew()[dev] = 0;
				
				AvailableEvent[dev] = 0;
			}
			mStats->IncrGlobalEvCnt(1);

			//plot
			QDateTime currentTime = QDateTime::currentDateTime();
			qint64 timeDiff = lastPlotTime.msecsTo(currentTime);
			if (mWaveDump->OfflineMode)
				PLOT_TIME = mWaveDump->PlotInterval_ms;
			if ((timeDiff > PLOT_TIME || first_plot) && !mEndThread) {
				if(first_plot)
					first_plot = 0;
#if QT_VERSION_MAJOR < 6
				mNew_plot = QtConcurrent::run(this, &eventBuildThread::fillPlotVector, mGlobalEvent);
#else
				mNew_plot = QtConcurrent::run(&eventBuildThread::fillPlotVector, this, mGlobalEvent);
#endif
				lastPlotTime = QDateTime::currentDateTime();
			}

			mNew_plot.waitForFinished();
		}
	}
	
}


void eventBuildThread::stopRun() {
	mNew_plot.waitForFinished();
	mEndThread = true;
	mDoRead = false;
}


void eventBuildThread::startRun() {
	this->mDoRead = true;
}

void eventBuildThread::InitFiles() {
	QString filen;

	this->mTestFile = new QFile(QDir::homePath() + DIRSEP + WAVEDUMP2_DIR + DIRSEP + "SyncTestFile.txt");
	outtest.setDevice(mTestFile);
	this->mTestFile->open(QIODevice::WriteOnly | QIODevice::Truncate);
	outtest << "ID" << "\t";
	for (int dev = 0; dev < mTotDevices; dev++)
		outtest << QString("Board %1 Timestamp").arg(dev)<<"\t";
	outtest << "\n";


	mWaveDump->getOutputSettings(mFile_folder, mFile_prefix, mFile_ftype, mFile_header, mFile_format, mFile_sync, mNEvts_file);
	if (mFile_sync == "NO" || mWaveDump->IsFastSaveEnabled()) {
		mSaving = false;
		return;
	}
	QString ext = (mFile_format.contains("ASCII")) ? ".txt" : ".bin";
	QDateTime date = QDateTime::currentDateTime();
	QString formattedTime = date.toString("yyyyMMddhhmmss");
	int RunId = mWaveDump->RunID;
	if (RunId < 10)
		filen = mFile_prefix + "_SYNC_" + formattedTime + QString("-0%1").arg(RunId) + ext;
	else
		filen = mFile_prefix + "_SYNC_" + formattedTime + QString("-%1").arg(RunId) + ext;

	this->mSaveFile = new QFile(mFile_folder + "/" + filen);
	out.setDevice(mSaveFile);
	if (!this->mSaveFile->open(QIODevice::WriteOnly | QIODevice::Truncate)) {
			emit GenericError("Unable to create file: " + this->mSaveFile->errorString());
			mSaving = false;
			return;
	}
	
	mSaving = true;
}

void eventBuildThread::SaveEventToFile(GlobalEvent* Gevent) {
	uint32_t max_size;
	if (mFile_format.contains("ASCII")) {
			if (mFile_header.contains("YES")) {
				out << "Event n. " << *Gevent->ID() << ENDL;
				out << "TimeStamp: " << *Gevent->T() << ENDL;
			}
			for (int dev = 0; dev < mTotDevices; dev++) {

				if (!Gevent->IsDev()[dev])
					continue;

				if (Gevent->NumSamples[dev].size() == 0)
					continue;
				const auto max_size_it = std::max_element(Gevent->NumSamples[dev].data(), Gevent->NumSamples[dev].data() + mDevices->at(dev)->getNumCh());
				max_size = *max_size_it;
				if (mFile_header.contains("YES")) {
					out << "Dev " << dev << " Samples: " << max_size << ENDL;
					out << "1 Sample = " << mDevices->at(dev)->getSample_to_S() * mDevices->at(dev)->getDecimation() * 1e6 << " us" << ENDL;
					out << "S \t";
					for (uint16_t i = 0; i < mDevices->at(dev)->getNumCh(); i++) {
						const auto size = Gevent->NumSamples[dev][i];
						if (size != 0) {
							out << "CH: " << i << '\t';
						}
					}
					out << ENDL;
				}
				for (uint32_t j = 0; j < max_size; j++) {
					out << j << '\t';
					switch (mWaveDump->getUoMY()) {
					case UOM_SAMPLE:
						for (uint16_t i = 0; i < mDevices->at(dev)->getNumCh(); i++) {
							const auto size = Gevent->NumSamples[dev][i];
							if (size != 0) {
								const auto data = Gevent->W()[dev * MAX_NCH + i];
								out << data[j] << '\t';
							}
						}
						break;
					case UOM_PHYS_UNIT: {
						for (uint16_t i = 0; i < mDevices->at(dev)->getNumCh(); i++) {
							const auto size = Gevent->NumSamples[dev][i];
							if (size != 0) {
								const auto data = Gevent->W()[dev * MAX_NCH + i];
								const auto baseline = mDevices->at(dev)->getBaselineLSB(i);
								const auto sample_to_v = mDevices->at(dev)->getSample_to_V(i);
								float val = (data[j] - baseline) * sample_to_v * 1e3;
								out << val << '\t';
							}
						}
						break;
					}
					}
					out << ENDL;
				}			
		}
		out.flush();
	}
	else { //binary format
		if (mFile_header.contains("YES")) {
			mSaveFile->write((const char*)Gevent->ID(), sizeof(Gevent->ID()));
			mSaveFile->write((const char*)Gevent->T(), sizeof(Gevent->T()));
		}
		for (int dev = 0; dev < mTotDevices; dev++) {

			if (!Gevent->IsDev()[dev])
				continue;

			if (Gevent->NumSamples[dev].size() == 0)
				continue;
			if (mFile_header.contains("YES")) {
				mSaveFile->write((const char*)&Gevent->NumSamples[dev], sizeof(Gevent->NumSamples[dev][0]));
				uint64_t one_sample_ns = mDevices->at(dev)->getSample_to_S() * mDevices->at(dev)->getDecimation() * 1e9;
				mSaveFile->write((const char*)&one_sample_ns, sizeof(one_sample_ns));

				uint32_t ch = 0;
				for (uint16_t i = 0; i < mDevices->at(dev)->getNumCh(); i++)
					if (Gevent->NumSamples[dev][i] != 0)
						++ch;
				mSaveFile->write((const char*)&ch, sizeof(ch));
			}
			for (uint16_t i = 0; i < mDevices->at(dev)->getNumCh(); i++) {
				const auto data = Gevent->W()[dev * MAX_NCH + i];
				const auto size = Gevent->NumSamples[dev][i];
				if (size != 0) {
					mSaveFile->write((const char*)&i, sizeof(i));
					switch (mWaveDump->getUoMY()) {
					case UOM_SAMPLE: {
						mSaveFile->write((const char*)data, size * sizeof(*data));
						break;
					}
					case UOM_PHYS_UNIT: {
						const auto baseline = mDevices->at(dev)->getBaselineLSB(i);
						const auto sample_to_v = mDevices->at(dev)->getSample_to_V(i);
						for (uint32_t j = 0; j < size; j++) {
							float val = (data[j] - baseline) * sample_to_v * 1e3;
							mSaveFile->write((const char*)&val, sizeof(val));
						}
						break;
					}
					}
				}
			}
			
		}
		mSaveFile->flush();
	}
}


void eventBuildThread::RunSyncOfflineMode() {
	int Wait;									// One baord has no data but the timeout has not been reached => keep waiting for data
	int AllEmpty;								// All the boards are empty
	int AllPresent;								// All the boards participate to the global event with data
	int AvailableEvent[MAX_NDEV] = { 0 };		// one event has been read from the device
	int Nev[MAX_NDEV] = { 0 };				//event index of the dev queue
	std::uint64_t tstamp[MAX_NDEV] = { 0 };		//event timestamp in ns
	std::uint32_t ev_ID[MAX_NDEV] = { 0 };			//Trigger ID
	std::uint64_t PrevEvID[MAX_NDEV] = { 0 };		// Previous trigger ID
	int ev_id = 0;
	int first_plot = 1;
	int NEvts = 0;

	QDateTime lastPlotTime = QDateTime::currentDateTime();
	for (int dev = 0; dev < mTotDevices; dev++) {
		mGlobalEvent->IsDev()[dev] = false;
	}
	while (!mEndThread) {
		while (mDoRead) {
			AllEmpty = 1;
			Wait = 0;

			for (int dev = 0; dev < mTotDevices; dev++) {

				if (!mDevices->at(dev)->AcqStarted) {//at least one device has stopped acquiring data--->stop sync mode
					mDoRead = false;
					mEndThread = true;
					break;
				}

				if (!AvailableEvent[dev]) {  //not yet read one evnt from dev 

					if (mDevices->at(dev)->NEvtsInQueue == 0) { //no new data to read from dev
						Wait = 1; //keep waiting						
					}
					else { //dev has some events to read
						AllEmpty = 0;
						AvailableEvent[dev] = 1;

						QMutexLocker locker(&mDevices->at(dev)->queue_mutex);
						QMutexLocker l(&mGMtx);
						//read the first event from the queue
						tstamp[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_Timestamp * mDevices->at(dev)->TSamplUnit_ns; //mDevices->at(dev)->getTSampl_clk()
						ev_ID[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_ID;

						std::uint32_t* s = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes();
						mGlobalEvent->NumSamples[dev].clear();
						std::copy(s, s + mDevices->at(dev)->getNumCh(), std::back_inserter(mGlobalEvent->NumSamples[dev]));
						for (int ch = 0; ch < mDevices->at(dev)->getNumCh(); ch++) {
							std::uint16_t* w = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->data()[ch];
							if (w != nullptr)
								std::copy(w, w + mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes()[ch], mGlobalEvent->W()[dev * MAX_NCH + ch]);
							if (mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_chmask & ((uint64_t)1 << ch))
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = true;
							else
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = false;
						}
						const auto max_size_it = std::max_element(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes(), mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes() + mDevices->at(dev)->getNumCh());
						mGlobalEvent->MaxSizes[dev] = *max_size_it;

						mStats->IncrReadTrgCnt(dev, 1);
						mStats->IncrReadByteCnt(dev, mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						mStats->IncrTotBytesCnt(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						if ((ev_ID[dev] > 0) && (ev_ID[dev] - PrevEvID[dev]) > 1 && !mWaveDump->IsManualOfflineEnabled()) {
							mStats->IncrLostTrgCnt(dev, (ev_ID[dev] - PrevEvID[dev] - 1) % EV_ID_MAX);
						}
						PrevEvID[dev] = ev_ID[dev];

						Nev[dev] = (++Nev[dev]) % MAX_NEVTS;
						if (mDevices->at(dev)->NEvtsInQueue == MAX_NEVTS)
							mDevices->at(dev)->waitCondition.wakeAll();
						--mDevices->at(dev)->NEvtsInQueue;
					}
				}
				else { //dev data already collected
					AllEmpty = 0;
				}

			}//dev loop

			if (!mDoRead)
				break;

			if (AllEmpty) { //reset timers if all devs are empty
				int stop = 0;
				for (int dev = 0; dev < mTotDevices; dev++) {
					if (mDevices->at(dev) == nullptr)
						continue;
					if (mDevices->at(dev)->isQueueClosed())
						stop++;
				}
				if (stop == mTotDevices) {//all the events queues are closed-->readout is finished
					for (int dev = 0; dev < mTotDevices; dev++)
						mDevices->at(dev)->freeEvtsQueue();
					mEndThread = true;
					break;
				}
			}
			if (Wait || AllEmpty) {
				continue;
			}

			// Get oldest time stamp and Trigger ID
			std::uint64_t oldest_ts = 0xFFFFFFFFFFFFFFFF;
			std::uint32_t oldest_trgid = 0xFFFFFFFF;
			for (int dev = 0; dev < mTotDevices; dev++) {
				if (AvailableEvent[dev]) {
					if (tstamp[dev] < oldest_ts)
						oldest_ts = tstamp[dev];
					if (ev_ID[dev] < oldest_trgid)
						oldest_trgid = ev_ID[dev];
					mGlobalEvent->BoardEvID()[dev] = ev_ID[dev];
				}
			}

			*mGlobalEvent->T() = oldest_ts;

			mStats->SetCurrentTimestamp_us(static_cast<double>(*mGlobalEvent->T()) / 1000.);

			AllPresent = 1;
			for (int dev = 0; dev < mTotDevices; dev++) {
				mGlobalEvent->IsDev()[dev] = true;
				mGlobalEvent->Wskew()[dev] = static_cast<uint32_t>(tstamp[dev] - oldest_ts);

				if ((!AvailableEvent[dev]) || ((tstamp[dev] - oldest_ts) > mCoincWindow)) { // No data from this board or this event is ahead in time => keep it for the next call
					mStats->IncrBuildMissCnt(dev, 1);
					AllPresent = 0;
					mGlobalEvent->IsDev()[dev] = false;
					continue;
				}

				mStats->IncrProcTrgCnt(dev, 1);

				AvailableEvent[dev] = 0;
			}


			if (!AllPresent) {
				mStats->IncrOutOfSyncEvCnt(1);
				for (int dev = 0; dev < mTotDevices; dev++)
					mGlobalEvent->IsDev()[dev] = false; //reset for plot
			}
			else {
				mStats->IncrGlobalEvCnt(1);
				*mGlobalEvent->ID() = ev_id;
				ev_id++;

				//save event to file if save is enabled
				if (mSaving) {
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
					if (mWaveDump->mDataSavedEnabled.load() == 1) {
#else
					if (mWaveDump->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
						if (NEvts == mNEvts_file && mNEvts_file) {
							this->mSaveFile->close();
							this->mTestFile->close();
							delete this->mSaveFile;
							delete this->mTestFile;
							InitFiles();
							NEvts = 0;
						}
						//save data to the sync test file
						outtest << *mGlobalEvent->ID() << "\t";
						for (int dev = 0; dev < mTotDevices; dev++)
							outtest << tstamp[dev] << "\t\t\t";
						outtest << ENDL;
						outtest.flush();
						SaveEventToFile(mGlobalEvent);
						NEvts++;
					}
					}
				}


			//plot
			QDateTime currentTime = QDateTime::currentDateTime();
			qint64 timeDiff = lastPlotTime.msecsTo(currentTime);
			if (mWaveDump->OfflineMode)
				PLOT_TIME = mWaveDump->PlotInterval_ms;
			if ((timeDiff > PLOT_TIME || first_plot) && !mEndThread) {
				if (first_plot)
					first_plot = 0;
				mNew_plot = QtConcurrent::run([this] { return fillPlotVectorSkew(mGlobalEvent); });
				lastPlotTime = QDateTime::currentDateTime();
			}

			mNew_plot.waitForFinished();

			for (int dev = 0; dev < mTotDevices; dev++)
				mGlobalEvent->IsDev()[dev] = false; //final reset also if allpresent


			}
		if (mSaving) {
			this->mSaveFile->close();
			this->mTestFile->close();
		}
		}
	for (int dev = 0; dev < mTotDevices; dev++) { //unlock the queue to avoid possible lck
		mDevices->at(dev)->NEvtsInQueue = 0;
		mDevices->at(dev)->waitCondition.wakeAll();
	}
}

void eventBuildThread::RunSyncIndependentOfflineMode() {
	int Wait;									// One baord has no data but the timeout has not been reached => keep waiting for data
	int AllEmpty;								// All the boards are empty
	int AllPresent;								// All the boards participate to the global event with data
	int AvailableEvent[MAX_NDEV] = { 0 };		// one event has been read from the device
	int Nev[MAX_NDEV] = { 0 };				//event index of the dev queue
	std::uint64_t tstamp[MAX_NDEV] = { 0 };		//event timestamp in ns
	std::uint32_t ev_ID[MAX_NDEV] = { 0 };			//Trigger ID
	std::uint64_t PrevEvID[MAX_NDEV] = { 0 };		// Previous trigger ID
	int ev_id = 0;
	int first_plot = 1;
	int NEvts = 0;

	QDateTime lastPlotTime = QDateTime::currentDateTime();
	for (int dev = 0; dev < mTotDevices; dev++) {
		mGlobalEvent->IsDev()[dev] = false;
	}
	while (!mEndThread) {
		while (mDoRead) {
			AllEmpty = 1;
			Wait = 0;

			for (int dev = 0; dev < mTotDevices; dev++) {

				if (!mDevices->at(dev)->AcqStarted) {//at least one device has stopped acquiring data--->stop sync mode
					mDoRead = false;
					mEndThread = true;
					break;
				}
				if (!AvailableEvent[dev]) {  //not yet read one evnt from dev 
					mGlobalEvent->NumSamples[dev].clear();
					if (mDevices->at(dev)->NEvtsInQueue == 0){ //no new data to read from dev
						Wait = 1; //keep waiting					
					}
					else { //dev has some events to read
						AllEmpty = 0;
						AvailableEvent[dev] = 1;

						QMutexLocker locker(&mDevices->at(dev)->queue_mutex);
						QMutexLocker l(&mGMtx);
						//read the first event from the queue
						tstamp[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_Timestamp * mDevices->at(dev)->TSamplUnit_ns; //mDevices->at(dev)->getTSampl_clk()
						ev_ID[dev] = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_ID;

						std::uint32_t* s = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes();

						std::copy(s, s + mDevices->at(dev)->getNumCh(), std::back_inserter(mGlobalEvent->NumSamples[dev]));
						for (int ch = 0; ch < mDevices->at(dev)->getNumCh(); ch++) {
							std::uint16_t* w = mDevices->at(dev)->EventsQueue()->at(Nev[dev])->data()[ch];
							if (w != nullptr)
								std::copy(w, w + mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes()[ch], mGlobalEvent->W()[dev * MAX_NCH + ch]);
							if (mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_chmask & ((uint64_t)1 << ch))
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = true;
							else
								mGlobalEvent->IsCh()[dev * MAX_CHANNELS + ch] = false;
						}
						const auto max_size_it = std::max_element(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes(), mDevices->at(dev)->EventsQueue()->at(Nev[dev])->sizes() + mDevices->at(dev)->getNumCh());
						mGlobalEvent->MaxSizes[dev] = *max_size_it;

						mStats->IncrReadTrgCnt(dev, 1);
						mStats->IncrReadByteCnt(dev, mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						mStats->IncrTotBytesCnt(mDevices->at(dev)->EventsQueue()->at(Nev[dev])->Ev_size);
						if ((ev_ID[dev] > 0) && (ev_ID[dev] - PrevEvID[dev]) > 1 && !mWaveDump->IsManualOfflineEnabled()) {
							mStats->IncrLostTrgCnt(dev, (ev_ID[dev] - PrevEvID[dev] - 1) % EV_ID_MAX);
						}
						PrevEvID[dev] = ev_ID[dev];

						Nev[dev] = (++Nev[dev]) % MAX_NEVTS;
						if (mDevices->at(dev)->NEvtsInQueue == MAX_NEVTS)
							mDevices->at(dev)->waitCondition.wakeAll();
						--mDevices->at(dev)->NEvtsInQueue;
					}
				}
				else { //dev data already collected
					AllEmpty = 0;
				}

			}//dev loop

			if (AllEmpty) { //reset timers if all devs are empty
				int stop = 0;
				for (int dev = 0; dev < mTotDevices; dev++) {
					if (mDevices->at(dev) == nullptr)
						continue;
					if (mDevices->at(dev)->isQueueClosed())
						stop++;
				}
				if (stop == mTotDevices) {//all the events queues are closed-->readout is finished
					for (int dev = 0; dev < mTotDevices; dev++)
						mDevices->at(dev)->freeEvtsQueue();
					mEndThread = true;
					break;
				}
			}
			if (Wait || AllEmpty) {
				continue;
			}

			// Get oldest time stamp and Trigger ID
			std::uint64_t oldest_ts = 0xFFFFFFFFFFFFFFFF;
			std::uint32_t oldest_trgid = 0xFFFFFFFF;
			for (int dev = 0; dev < mTotDevices; dev++) {
				if (AvailableEvent[dev]) {
					if (tstamp[dev] < oldest_ts)
						oldest_ts = tstamp[dev];
					if (ev_ID[dev] < oldest_trgid)
						oldest_trgid = ev_ID[dev];
					mGlobalEvent->BoardEvID()[dev] = ev_ID[dev];
				}
			}

			*mGlobalEvent->T() = oldest_ts;
			//*mGlobalEvent->ID() = oldest_trgid;
			*mGlobalEvent->ID() = ev_id;
			ev_id++;

			mStats->SetCurrentTimestamp_us(static_cast<double>(*mGlobalEvent->T()) / 1000.);

			AllPresent = 1;
			for (int dev = 0; dev < mTotDevices; dev++) {

				mGlobalEvent->Wskew()[dev] = static_cast<uint32_t>(tstamp[dev] - oldest_ts);

				if ((!AvailableEvent[dev]) || ((tstamp[dev] - oldest_ts) > mCoincWindow)) { // No data from this board or this event is ahead in time => keep it for the next call
					AllPresent = 0;
					mGlobalEvent->IsDev()[dev] = false;
					continue;
				}
				mGlobalEvent->IsDev()[dev] = true;
				mStats->IncrProcTrgCnt(dev, 1);

				AvailableEvent[dev] = 0;
			}

			//save event to file if save is enabled
			if (mSaving) {
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
				if (mWaveDump->mDataSavedEnabled.load() == 1) {
#else
				if (mWaveDump->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
					if (NEvts == mNEvts_file && mNEvts_file) {
						this->mSaveFile->close();
						this->mTestFile->close();
						delete this->mSaveFile;
						delete this->mTestFile;
						InitFiles();
						NEvts = 0;
					}
					//save data to the sync test file
					outtest << *mGlobalEvent->ID() << "\t";
					for (int dev = 0; dev < mTotDevices; dev++) {
						if (mGlobalEvent->IsDev()[dev])
							outtest << tstamp[dev] << "\t\t\t\t\t";
						else
							outtest << "\t\t\t\t\t";
					}
					outtest << ENDL;
					outtest.flush();
					SaveEventToFile(mGlobalEvent);
					NEvts++;
				}
				}


			//plot
			QDateTime currentTime = QDateTime::currentDateTime();
			qint64 timeDiff = lastPlotTime.msecsTo(currentTime);
			if (mWaveDump->OfflineMode)
				PLOT_TIME = mWaveDump->PlotInterval_ms;
			if ((timeDiff > PLOT_TIME || first_plot) && !mEndThread) {
				if (first_plot)
					first_plot = 0;
				mNew_plot = QtConcurrent::run([this] { return fillPlotVectorSkew(mGlobalEvent); });
				lastPlotTime = QDateTime::currentDateTime();
			}

			mNew_plot.waitForFinished();

			for (int dev = 0; dev < mTotDevices; dev++)
				mGlobalEvent->IsDev()[dev] = false; //reset for plot
			}
		if (mSaving) {
			this->mSaveFile->close();
			this->mTestFile->close();
		}
		}
	}


eventBuildThread::~eventBuildThread()
{
}
