/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		deviceReadoutThread.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "deviceReadoutThread.h"

#include <QtGlobal>

#include "WaveDump2.h"

deviceReadoutThread::deviceReadoutThread(WaveDump2* w, CAENDevice* dev, std::uint32_t runId, ::uint32_t upRate)
{
	devQueueInited = false;
	this->mDevHandle = dev->getHandle();
	this->mReadPointHandle = dev->getEPHandle();
	mNumOfchannels = dev->getNumCh();
	mEvents = dev->EventsQueue();
	mWave_cuts = dev->Wcut;
	for (int ev = 0; ev < MAX_NEVTS; ev++)
		mWave_cuts[ev] = 0;
	mDevice = dev;
	mWaveDump2 = w;
	mSample_to_s = w->getSampleToS();
	mSample_to_V = 1;
	this->mName = dev->getName();
	this->mUpdateRate = upRate;
	mRunId = runId;
	mStats = w->Stats();

	mSaveFile.resize(mNumOfchannels);
	outFile.resize(mNumOfchannels);
	mChEnableMask = 0;
	for (int ch = 0; ch < mNumOfchannels; ch++) {
		mSaveFile[ch] = NULL;
		if (dev->isChEnabled(ch))
			mChEnableMask |= (uint64_t{ 1 } << ch);
	}

	InitEvtsQueue();
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
	if (mWaveDump2->mDataSavedEnabled.load() == 1) {
#else
	if (mWaveDump2->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
		mSaveRaw = w->IsFastSaveEnabled();
		if (mSaveRaw) {
			this->mReadPointHandle = dev->getRAWHandle();
			dev->MallocRawBuffers();
		}		
	}
	InitFiles();

	mManual = w->IsManualOfflineEnabled();

	connect(this, SIGNAL(finished()), this, SLOT(deleteLater()));
	connect(this, SIGNAL(StopReadout(QString)), mWaveDump2, SLOT(ReaderThreadFinished(QString)));
	connect(this, SIGNAL(ReadoutError(QString, CAEN_FELib_ErrorCode)), mWaveDump2, SLOT(ReadoutError(QString, CAEN_FELib_ErrorCode)));
	connect(this, SIGNAL(GenericError(QString)), mWaveDump2, SLOT(ReadoutError1(QString)));
	connect(this, SIGNAL(NewEvLoaded(QString, int, quint64)), mWaveDump2, SLOT(NewEvLoadedOK(QString, int, quint64)));
	connect(this, SIGNAL(ErrorLoadingEv(QString, int)), mWaveDump2, SLOT(NewEvLoadError(QString, int)));
	connect(this, SIGNAL(LoadChMask(QString,quint64)), mWaveDump2, SLOT(PaletteChMask(QString,quint64)));	
}

void deviceReadoutThread::InitEvtsQueue() {
	mDevice->createEvtsQueue(mWaveDump2->getALLMaxRecLenS_corr(), MAX_NEVTS);
	devQueueInited = true;
}


void deviceReadoutThread::startRun() {
	QMutexLocker l(&mStopMutex);
	mDevice->NEvtsInQueue = 0;
	mDoRead = true;
	mRequestStop = false;
}

void deviceReadoutThread::stopRun() {
	QMutexLocker l(&mStopMutex);
	mRequestStop = true;
	mDevice->waitCondition.wakeAll();
}

void deviceReadoutThread::SaveThreadFinished() {

}

void deviceReadoutThread::run() {
	if (mDevice->isVirtual()) {
		if (!mManual)
			RunFromFile();
		else
			RunFromFileManual();
	}
	else {
		if (mSaveRaw) {
			mSaveThread = new dataSaveThread(this, mWaveDump2, mDevice, mRunId);
			mSaveThread->startRun();
			mSaveThread->start();
			QThread::msleep(50);
			RunSaveRaw();

			mSaveThread->stopRun();
			mSaveThread->wait();
			delete mSaveThread;
			mDevice->FreeRawBuffers();
		}
		else
			RunNormal();
	}

	exit(0);
}

int deviceReadoutThread::CalcJSamplesShift(uint16_t** data, int NCh, uint64_t trgmask,int *thr, int pretrgsamples, int nextrasamples) {
	int FirstSAboveThr = pretrgsamples + nextrasamples;
	for (int ch = 0;ch < NCh;ch++) {
		if (!(trgmask & (1LL << ch)))
			continue;
		for (int s = pretrgsamples; s < (pretrgsamples + nextrasamples); s++) {//inspect the extra samples after pretrigger
			if (data[ch][s] > thr[ch]) { //first sample above threshold for this channel
				if (s < FirstSAboveThr)
					FirstSAboveThr = s;
				break;
			}
		}
	}
	//now the first sample above threshold has been found
	FirstSAboveThr -= pretrgsamples;
	return FirstSAboveThr;  //number of samples to cut on the left
}

void deviceReadoutThread::RunNormal()
{
	QString prefix;
	QString folder;
	QString ftype;
	QString header;
	QString format;
	int Nev = 0;
	int first_fill = 1;
	QString name = mDevice->getName();
	int NEvts = 0;
	bool apply_j_correction = false;
	int pretrgS = 0;
	int thrS[MAX_CHANNELS] = { 0 };
	uint64_t chtmask = 0;
	QString smode = mDevice->getStartSource();
	QString trgsrc = mDevice->getTrgSource();
	if (trgsrc == "SwTrg|ITLA" && mDevice->NJExtraSamples) {
		apply_j_correction = true;
		pretrgS = mDevice->getPreTrg_valueS();
		chtmask = mDevice->getSelfTrgMask().toULong();
		for (int ch = 0; ch < MAX_CHANNELS; ch++) {
			if (!(chtmask & (1LL << ch)))
				continue;
			thrS[ch] = mDevice->getChThreshold(ch) + mDevice->getBaselineLSB(ch);
		}
	}
	

	while (!mEndThread) {
		Nev = 0;
		while (mDoRead) {
			if (mRequestStop || !mDevice->AcqStarted) {
				QMutexLocker l(&mStopMutex);
				mDoRead = false;
				mDevice->closeEvtsQueue();
				mEndThread = true;
				break;
			}
			else {
					QMutexLocker locker(&mDevice->queue_mutex);
					if (first_fill)
						locker.unlock();

				if (mDevice->NEvtsInQueue == MAX_NEVTS) {
					//wait for available space in the queue
					mDevice->waitCondition.wait(&mDevice->queue_mutex);					
				}
				for (uint16_t ch = 0; ch < mNumOfchannels; ch++)
					mEvents->at(Nev)->data()[ch] -= mWave_cuts[Nev];

				//readdata with 300 ms timeout
				CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_ReadData(mReadPointHandle, 100,
					&mEvents->at(Nev)->Ev_Timestamp,
					&mEvents->at(Nev)->Ev_ID,
					mEvents->at(Nev)->data(),
					mEvents->at(Nev)->sizes(),
					&mEvents->at(Nev)->Ev_size,
					&mEvents->at(Nev)->Ev_flags);
				if (err != CAEN_FELib_Success) {
					if (err == CAEN_FELib_Timeout) {
						mDevice->Triggered = false;
					}
					else if (err == CAEN_FELib_Stop && !smode.contains("SINlevel",Qt::CaseInsensitive)) {
						mRequestStop = true;
					}
					else
						emit ReadoutError(mDevice->getName(), err);

					continue;
				}

				mDevice->Triggered = true;
				mEvents->at(Nev)->Ev_chmask = 0;
				for (uint16_t ch = 0; ch < mNumOfchannels; ch++) {
					if (mEvents->at(Nev)->sizes()[ch]) {
						mEvents->at(Nev)->Ev_chmask |= (uint64_t{ 1 } << ch);
					}
					else {
						mEvents->at(Nev)->data()[ch] = nullptr;
					}
				}

				if (apply_j_correction && !mDevice->SWTrg_sent) {
					int cut = CalcJSamplesShift(mEvents->at(Nev)->data(), mNumOfchannels, chtmask, thrS, pretrgS, mDevice->RecLenJIncrS);
					for (uint16_t ch = 0; ch < mNumOfchannels; ch++) {
						mEvents->at(Nev)->sizes()[ch] -= mDevice->RecLenJIncrS; //total size of waveforms decreased removing the added extra samples
						mEvents->at(Nev)->data()[ch] += cut; //start of waveform shifted 
						mWave_cuts[Nev] = cut;
					}
				}
				else{
					if (mDevice->SWTrg_sent)
						mDevice->SWTrg_sent = false;
					for (uint16_t ch = 0; ch < mNumOfchannels; ch++) {
						if(mEvents->at(Nev)->Ev_chmask & (uint64_t{ 1 } << ch))
							mEvents->at(Nev)->sizes()[ch] -= mDevice->RecLenJIncrS; //total size of waveforms decreased removing the added extra samples
					}
				}

				++mDevice->NEvtsInQueue;
			}
			// handle savefile
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
			if (mWaveDump2->mDataSavedEnabled.load() == 1) {
#else
			if (mWaveDump2->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
				if (NEvts == mNEvts_file && mNEvts_file) {
					for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
						if (mSaveFile[i] != nullptr) {
							mSaveFile[i]->close();
							delete mSaveFile[i];
							mSaveFile[i] = nullptr;
							delete outFile[i];
							outFile[i] = nullptr;
						}
					}
					InitFiles();
					NEvts = 0;
				}
				SaveEventToFile(mEvents->at(Nev));
				NEvts++;
			}

			Nev = (++Nev) % MAX_NEVTS;
			if (first_fill && Nev == (MAX_NEVTS - 1)) {
				first_fill = 0;
			}
			

		}//doread

		if (mSaving) {
			for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
				if (mSaveFile[i] != nullptr) {
					mSaveFile[i]->close();
					delete mSaveFile[i];
					mSaveFile[i] = nullptr;
					delete outFile[i];
					outFile[i] = nullptr;
				}
			}
			mSaving = false;
		}
		//test_file->close();
	}

	emit StopReadout(name);
}


void deviceReadoutThread::SaveEventToFile(const CAENDig2Event* event) {
	if (mFile_sync == "YES")
		return;
    //if (!this->mSaveFile[0]->exists()) {
    //	if (!this->mSaveFile[0]->open(QIODevice::WriteOnly)) {
    //		emit GenericError("Unable to create file: " + this->mSaveFile[0]->errorString());
    //		return;
    //	}
    //}
	if (mFile_format.contains("ASCII")) {
		if (mFile_ftype.contains("SINGLE")) {
			const auto max_size_it = std::max_element(event->sizes(), event->sizes() + this->mNumOfchannels);
			const auto max_size = *max_size_it;
			if (mFile_header.contains("YES")) {
				*outFile[0] << "Event n. " << event->Ev_ID << ENDL;
				*outFile[0] << "TimeStamp: " << event->Ev_Timestamp<< ENDL;  //clock units
				*outFile[0] << "Samples: " << max_size << ENDL;
				*outFile[0] << "1 Sample = " << mDevice->getSample_to_S() * mDevice->getDecimation() * 1e6 << " us" << ENDL;
				*outFile[0] << "S \t";
				for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
					const auto size = event->sizes()[i];
					if (size != 0) {
						*outFile[0] << "CH: " << i << '\t';
					}
				}
				*outFile[0] << ENDL;
			}
			for (uint32_t j = 0; j < max_size; j++) {
				*outFile[0] << j << '\t';
				switch (mWaveDump2->getUoMY()) {
				case UOM_SAMPLE:
					for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
						const auto size = event->sizes()[i];
						if (size != 0) {
							const auto data = event->data()[i];
							*outFile[0] << data[j] << '\t';
						}
					}
					break;
				case UOM_PHYS_UNIT: {
					for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
						const auto size = event->sizes()[i];
						if (size != 0) {
							const auto data = event->data()[i];
							const auto baseline = mDevice->getBaselineLSB(i);
							const auto sample_to_v = mDevice->getSample_to_V(i);
							float val = (data[j] - baseline) * sample_to_v * 1e3;
							*outFile[0] << val << '\t';
						}
					}
					break;
				}
				}
				*outFile[0] << ENDL;
			}
			outFile[0]->flush();
		}
		else {//one file per channel
			for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
				if (!(mChEnableMask & (uint64_t{ 1 } << i)))
					continue;
				const auto data = event->data()[i];
				const auto size = event->sizes()[i];
				if (size != 0) {
					if (mFile_header.contains("YES")) {
						*outFile[i] << "Event n. " << event->Ev_ID << ENDL;
						*outFile[i] << "TimeStamp: " << event->Ev_Timestamp << ENDL; //clock units
						*outFile[i] << "Samples: " << size << ENDL;
						*outFile[i] << "1 Sample = " << mDevice->getSample_to_S() * mDevice->getDecimation() * 1e6 << " us" << ENDL;
					}
					switch (mWaveDump2->getUoMY()) {
					case UOM_SAMPLE: {
						for (uint32_t j = 0; j < size; j++)
							*outFile[i] << data[j] << '\t';
						break;
					}
					case UOM_PHYS_UNIT: {
						const auto baseline = mDevice->getBaselineLSB(i);
						const auto sample_to_v = mDevice->getSample_to_V(i);
						for (uint32_t j = 0; j < size; j++) {
							float val = (data[j] - baseline) * sample_to_v * 1e3;
							*outFile[i] << val << '\t';
						}
						break;
					}
					}
				}
				*outFile[i] << ENDL;
				outFile[i]->flush();
			}
		}
	}
	else { //binary format
		if (mFile_ftype.contains("SINGLE")) {
			if (mFile_header.contains("YES")) {
				const auto max_size_it = std::max_element(event->sizes(), event->sizes() + this->mNumOfchannels);
				uint32_t max_size = *max_size_it;
				mSaveFile[0]->write((const char*)&event->Ev_ID, sizeof(event->Ev_ID));
				mSaveFile[0]->write((const char*)&event->Ev_Timestamp, sizeof(event->Ev_Timestamp));
				mSaveFile[0]->write((const char*)&max_size, sizeof(max_size));
				uint64_t one_sample_ns = mDevice->getSample_to_S() * mDevice->getDecimation() * 1e9;
				mSaveFile[0]->write((const char*)&one_sample_ns, sizeof(one_sample_ns));

				uint32_t ch = 0;
				for (uint16_t i = 0; i < this->mNumOfchannels; i++)
					if (event->sizes()[i] != 0)
						++ch;
				mSaveFile[0]->write((const char*)&ch, sizeof(ch));
			}
			for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
				const auto data = event->data()[i];
				const auto size = event->sizes()[i];
				if (size != 0) {
					mSaveFile[0]->write((const char*)&i, sizeof(i));
					switch (mWaveDump2->getUoMY()) {
					case UOM_SAMPLE: {
						mSaveFile[0]->write((const char*)data, size * sizeof(*data));
						break;
					}
					case UOM_PHYS_UNIT: {
						const auto baseline = mDevice->getBaselineLSB(i);
						const auto sample_to_v = mDevice->getSample_to_V(i);
						for (uint32_t j = 0; j < size; j++) {
							float val = (data[j] - baseline) * sample_to_v * 1e3;
							mSaveFile[0]->write((const char*)&val, sizeof(val));
						}
						break;
					}
					}
				}
			}
			mSaveFile[0]->flush();
		}
		else {
			for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
				if (!(mChEnableMask & (uint64_t{ 1 } << i)))
					continue;
				const auto data = event->data()[i];
				const auto size = event->sizes()[i];
				if (size != 0) {
					if (mFile_header.contains("YES")) {
						mSaveFile[i]->write((const char*)&event->Ev_ID, sizeof(event->Ev_ID));
						mSaveFile[i]->write((const char*)&event->Ev_Timestamp, sizeof(event->Ev_Timestamp));
						mSaveFile[i]->write((const char*)&size, sizeof(size));
						uint64_t one_sample_ns = mDevice->getSample_to_S() * mDevice->getDecimation() * 1e9;
						mSaveFile[i]->write((const char*)&one_sample_ns, sizeof(one_sample_ns));
					}
					switch (mWaveDump2->getUoMY()) {
					case UOM_SAMPLE: {
						mSaveFile[i]->write((const char*)data, size * sizeof(*data));
						break;
					}
					case UOM_PHYS_UNIT: {
						const auto baseline = mDevice->getBaselineLSB(i);
						const auto sample_to_v = mDevice->getSample_to_V(i);
						for (uint32_t j = 0; j < size; j++) {
							float val = (data[j] - baseline) * sample_to_v * 1e3;
							mSaveFile[i]->write((const char*)&val, sizeof(val));
						}
						break;
					}
					}
					mSaveFile[i]->flush();
				}
			}
		}
	}

}

void deviceReadoutThread::SaveEventFast(const CAENDig2Event *event) {

	mSaveFile[0]->write((const char*)&event->Ev_ID, sizeof(event->Ev_ID)); //event id
	mSaveFile[0]->write((const char*)&event->Ev_Timestamp, sizeof(event->Ev_Timestamp)); //event timestamp
	const auto max_size_it = std::max_element(event->sizes(), event->sizes() + this->mNumOfchannels);
	uint max_size = *max_size_it;
	mSaveFile[0]->write((const char*)&max_size, sizeof(uint)); //size in samples
	uint64_t one_sample_ns = mDevice->getSample_to_S() * mDevice->getDecimation() * 1e9;
	mSaveFile[0]->write((const char*)&one_sample_ns, sizeof(one_sample_ns)); //1 sample in ns 
	mSaveFile[0]->write((const char*)&event->Ev_chmask, sizeof(event->Ev_chmask)); //channel mask

	
	for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
		const auto data = event->data()[i];
		const auto size = event->sizes()[i];
		if (data != nullptr) {
			const auto baseline = mDevice->getBaselineLSB(i);
			const auto sample_to_mV = mDevice->getSample_to_V(i) *1e3;
			uint32_t factor = sample_to_mV * 1000.;
			mSaveFile[0]->write((const char*)&baseline, sizeof(baseline)); //channel baseline in LSB
			mSaveFile[0]->write((const char*)&factor, sizeof(factor)); //1 sample in uV  ---> (data[j] - baseline) * factor / 1e3 to switch to mV
			mSaveFile[0]->write((const char*)data, size * sizeof(*data)); //samples in LSB
		}
	}
	mSaveFile[0]->flush();
}


void deviceReadoutThread::InitFiles() {
	if (mSaveRaw)
		return;
	QString filen;
	QString devicename = mDevice->getName();
	devicename.replace(":", "");
	devicename.replace("//", "-");
	mWaveDump2->getOutputSettings(mFile_folder, mFile_prefix, mFile_ftype, mFile_header, mFile_format, mFile_sync, mNEvts_file);
	if (mFile_sync == "YES") {
		mSaving = false;
		return;
	}
	QString ext = (mFile_format.contains("ASCII")) ? ".txt" : ".bin";
	QDateTime date = QDateTime::currentDateTime();
	QString formattedTime = date.toString("yyyyMMddhhmmss");
	if (mFile_ftype == "SINGLE") {
		if (mRunId < 10)
			filen = mFile_prefix + "_" + devicename + "_" + formattedTime + QString("-0%1").arg(mRunId) + ext;
		else
			filen = mFile_prefix + "_" + devicename + "_" + formattedTime + QString("-%1").arg(mRunId) + ext;
		this->mSaveFile[0] = new QFile(mFile_folder + "/" + filen);
		outFile[0] = new QTextStream(this->mSaveFile[0]);
		outFile[0]->setRealNumberNotation(QTextStream::FixedNotation);
		int save_status;
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
		save_status = mWaveDump2->mDataSavedEnabled.load();
#else
		save_status= mWaveDump2->mDataSavedEnabled.loadRelaxed();
#endif
		if (save_status == 1) {
			if (!this->mSaveFile[0]->open(QIODevice::WriteOnly)) {
				emit GenericError("Unable to create file: " + this->mSaveFile[0]->errorString());
				return;
			}
		}
	}
	else {

		for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
			if (!(mChEnableMask & (uint64_t{ 1 } << i)))
				continue;
			if (mRunId < 10)
				filen = mFile_prefix + "_" + devicename + "_CH" + QString::number(i) + "_" + formattedTime + QString("-0%1").arg(mRunId) + ext;
			else
				filen = mFile_prefix + "_" + devicename + "_CH" + QString::number(i) + "_" + formattedTime + QString("-%1").arg(mRunId) + ext;

			if (mNEvts_file) {
				if (!QDir(mFile_folder + QString("/CH%1").arg(i)).exists())
					QDir().mkdir(mFile_folder + QString("/CH%1").arg(i));
				this->mSaveFile[i] = new QFile(mFile_folder + QString("/CH%1/").arg(i) + filen);
			}
			else
				this->mSaveFile[i] = new QFile(mFile_folder + "/" + filen);
			outFile[i] = new QTextStream(this->mSaveFile[i]);
			if (!this->mSaveFile[i]->open(QIODevice::WriteOnly)) {
				emit GenericError("Unable to create file: " + this->mSaveFile[i]->errorString());
				return;
			}
		}
	}
	mSaving = true;
}

static inline void swapByteOrder(unsigned short& us)
{
	us = (us >> 8) |
		(us << 8);
}

static inline void swapByteOrder1(unsigned int& ui)
{
	ui = (ui >> 24) |
		((ui << 8) & 0x00FF0000) |
		((ui >> 8) & 0x0000FF00) |
		(ui << 24);
}



void deviceReadoutThread::RunSaveRaw()
{
	QString name = mDevice->getName();
	int NTOTB = 0;
	mStats->clear();
	int dev = mDevice->getIndex();
	int PrevEvID = 0;
	size_t size = 0;
	uint64_t word;
	int ActiveChannels = 0;
	for (int c = 0; c < mDevice->getNumCh(); c++) {
		if (mDevice->isChEnabled(c))
			ActiveChannels++;
	}
	int single_event_size = 1;	uint32_t wave_length = 0, mask1=0, mask2=0, word32;
	uint32_t chmask32=0;
	mDevice->NBuffFilled = 0;
	for(int n=0;n<MAX_RAW_BUFFS;n++)
		mDevice->BuffOccupancy[n] = 0;
	int NSaveBuff = -1;
	char* buffer = nullptr;

	while (!mEndThread) {
		while (mDoRead) {
			if (mRequestStop) {
				QMutexLocker l(&mStopMutex);
				mDoRead = false;
				mDevice->closeEvtsQueue();
				mDevice->AcqStarted = false;
				mEndThread = true;
				break;
			}
			if (mDevice->NBuffFilled == MAX_RAW_BUFFS) {
				continue;
			}


			NSaveBuff = (NSaveBuff + 1) % MAX_RAW_BUFFS;
			buffer = mDevice->RawDataBuffer[NSaveBuff];

			//readdata with 100 ms timeout
			CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_ReadData(mReadPointHandle, 50, buffer, &mDevice->BuffOccupancy[NSaveBuff]);
			if (err != CAEN_FELib_Success) {
				if (err == CAEN_FELib_Timeout) {
					mDevice->Triggered = false;
					if ((mRequestStop || !mDevice->AcqStarted) && (mDevice->DeviceClass == CAEN_DIG1)) {
						QMutexLocker l(&mStopMutex);
						mDoRead = false;
						mDevice->closeEvtsQueue();
						mEndThread = true;
						continue;
					}
				}
				else
					emit ReadoutError(mDevice->getName(), err);
				continue;
			}

			size = mDevice->BuffOccupancy[NSaveBuff];
			
			if (size == 0) {
				NSaveBuff = (NSaveBuff - 1) % MAX_RAW_BUFFS;
				continue;
			}

			mDevice->BuffMutex.lock();
			++mDevice->NBuffFilled;
			mDevice->BuffMutex.unlock();

			memcpy(&word, buffer, sizeof(uint64_t));
			word = mWaveDump2->swapByteOrderWord(word);
			if (mDevice->isInfoStartWord(word)) {//StartRun word 
				if (mDevice->DeviceClass == CAEN_DIG2) {
					uint64_t w[4];
					memcpy(w, buffer, 4*sizeof(uint64_t));
					mSaveThread->StoreStartRunWords(w);

					wave_length = word & 0x1ffffff;
					single_event_size = wave_length * sizeof(uint64_t) * ActiveChannels + 3 * sizeof(uint64_t);
				}
				else {
					memcpy(&word32, buffer, sizeof(uint32_t));
					single_event_size = (word32 & 0xfffffff) * sizeof(uint32_t);
					memcpy(&word32, buffer + sizeof(uint32_t), sizeof(uint32_t));
					chmask32 = word32 & 0xff;
					memcpy(&word32, buffer + 2*sizeof(uint32_t), sizeof(uint32_t));
					chmask32 |= (((word32 >> 24) & 0xff) << 8);
					for (int c = 0; c < MAX_CHANNELS/2; c++) {
						if (chmask32 & (1 << c))
							ActiveChannels++;
					}
				}
			}

			//handle stats
			mStats->IncrReadTrgCnt(dev, (int)(size/single_event_size));
			mStats->IncrReadByteCnt(dev, size);
			mStats->IncrTotBytesCnt(size);

			if (mDevice->DeviceClass == CAEN_DIG2) {
				memcpy(&word, buffer + size - 3 * sizeof(uint64_t), sizeof(uint64_t));
				word = mWaveDump2->swapByteOrderWord(word);
				if (mDevice->isInfoStopWord(word)) {//StopRun word
					QMutexLocker l(&mStopMutex);
					mDoRead = false;
					mEndThread = true;
				}
			}
			else {
				if (!mDevice->AcqStarted || mRequestStop) {//acq has finished
					QMutexLocker l(&mStopMutex);
					mDoRead = false;
					mEndThread = true;
				}
			}

		}//doread


		if (mSaveFile[0] != nullptr) {
			mSaveFile[0]->close();
			delete mSaveFile[0];
			mSaveFile[0] = nullptr;
			delete outFile[0];
			outFile[0] = nullptr;
		}

		}

	emit StopReadout(name);
	}


void deviceReadoutThread::RunFromFile()
{
	QString fname = mDevice->getRawDataFile();
	QString family_code = mDevice->getFamilyCode();
	QFile f(fname);
	f.open(QIODevice::ReadOnly);
	QDataStream in(&f);
	in.setByteOrder(QDataStream::LittleEndian);
	if(mDevice->DeviceClass == CAEN_DIG2)
		in.device()->seek(uint64_t(26) + uint64_t(16)*mNumOfchannels +32);//file header + channel header + startrun word
	else
		in.device()->seek(uint64_t(26) + uint64_t(16) * mNumOfchannels);//file header + channel header
    quint64 word;
	quint32 word32;
	quint16 word16;
	uint32_t size, trigger_id;
	uint64_t Trigger_timetag;
	uint64_t chmask;
	uint32_t gmask = 0;
	int nchannels = 0;
	int Nev = 0;
	int first_fill = 1;
	int NEvts = 0;
	bool mask_set = false, zle=false;
	QVector<int> enabled_ch_list;
	int NRollOver = 0;
	uint64_t prev_TTimetag = 0;
	uint64_t LastTtoAdd = 0;

	mDevice->LastEvTimetag = 0;
	mDevice->AcqStarted = true;
	mDevice->setVirtualAcqStatus("Running Offline");
	while (!mEndThread) {
		while (mDoRead) {
			if (mRequestStop) {
				QMutexLocker l(&mStopMutex);
				mDoRead = false;
				mDevice->closeEvtsQueue();
				mDevice->AcqStarted = false;
				mEndThread = true;
				break;
			}
			QMutexLocker locker(&mDevice->queue_mutex);
			if (first_fill)
				locker.unlock();

			if (mDevice->NEvtsInQueue == MAX_NEVTS) {
				//wait for available space in the queue
				mDevice->waitCondition.wait(&mDevice->queue_mutex);
			}

			if (mDevice->DeviceClass == CAEN_DIG2) {
				in >> word;
				word = mWaveDump2->swapByteOrderWord(word);

				if ((word == 0x3200000200000003) || (in.status() != QTextStream::Ok)) {//StopRun word
					locker.unlock();
					QMutexLocker l(&mStopMutex);
					QThread::msleep(1000);
					mDoRead = false;
					mDevice->closeEvtsQueue();
					mDevice->AcqStarted = false;
					mEndThread = true;
					break;
				}

				//if (((word >> 60) & 0xf) == 1) {//TODO:check fw scope
				size = word & 0xffffffff;
				trigger_id = (word >> 32) & 0xffffff;
				//}
				in >> word;
				word = mWaveDump2->swapByteOrderWord(word);
				Trigger_timetag = word & 0xffffffffffff;
				in >> word;
				word = mWaveDump2->swapByteOrderWord(word);
				chmask = word;
				if (!mask_set) {
					emit LoadChMask(mName, chmask);
					mask_set = true;
				}
				nchannels = 0;
				enabled_ch_list.clear();
				for (int ch = 0; ch < mNumOfchannels; ch++) {
					if ((chmask & ((uint64_t)1 << ch))) {
						nchannels++;
						mEvents->at(Nev)->sizes()[ch] = mDevice->getReclen_valueS();
						enabled_ch_list.append(ch);
					}
					else {
						mEvents->at(Nev)->sizes()[ch] = 0;
						mEvents->at(Nev)->data()[ch] = nullptr;
					}
				}

				mEvents->at(Nev)->Ev_Timestamp = Trigger_timetag;
				mEvents->at(Nev)->Ev_ID = trigger_id;
				mEvents->at(Nev)->Ev_size = size * 8;
				mEvents->at(Nev)->Ev_chmask = chmask;
				mDevice->LastEvTimetag = (Trigger_timetag * 8) / 1000000;//time in ms

				int read_size = 0;
				while (read_size < (size - 3) * 4 / nchannels) {
					for (int ch = 0; ch < nchannels; ch++) {
						in >> word;
						word = mWaveDump2->swapByteOrderWord(word);
						for (int i = 0; i < 4; i++)
							mEvents->at(Nev)->data()[enabled_ch_list.at(ch)][i + read_size] = (word >> (i * 16)) & 0xffff;
					}
					read_size += 4;
				}
			}
			else {//DIG1
				in >> word32;
				if (in.status() != QTextStream::Ok) {//end of file
					locker.unlock();
					QMutexLocker l(&mStopMutex);
					QThread::msleep(1000);
					mDoRead = false;
					mDevice->closeEvtsQueue();
					mDevice->AcqStarted = false;
					mEndThread = true;
					break;
				}
				size = word32 & 0xfffffff;
				in >> word32;
				chmask = word32 & 0xff;
				gmask = chmask; //save group mask if needed
				zle = ((word32 >> 24) & 0x1);
				in >> word32;
				trigger_id = word32 & 0xffffff;
				chmask |= (((word32 >> 24) & 0xff) << 8);
				if (family_code == "740") {
					uint32_t mask = 0;
					for (int g = 0; g < MAX_CHANNELS / 8; g++) {
						if (chmask & (1 << g)) {
							mask |= (0xff << (g * 8));
						}
					}
					chmask = mask;
				}
				if (!mask_set) {
					emit LoadChMask(mName, chmask);
					mask_set = true;
				}
				nchannels = 0;
				enabled_ch_list.clear();
				for (int ch = 0; ch < mNumOfchannels; ch++) {
					if ((chmask & ((uint64_t)1 << ch))) {
						nchannels++;
						mEvents->at(Nev)->sizes()[ch] = mDevice->getReclen_valueS();
						enabled_ch_list.append(ch);
					}
					else {
						mEvents->at(Nev)->sizes()[ch] = 0;
						mEvents->at(Nev)->data()[ch] = nullptr;
					}
				}
				in >> word32;
				Trigger_timetag = (word32 & 0x7FFFFFFF);
				bool rolled_over = ((word32 >> 31) & 0x1);
				if (rolled_over && prev_TTimetag==0) {
					NRollOver++;
					LastTtoAdd += 0x7FFFFFFF;
				}
				if (Trigger_timetag < prev_TTimetag) {
					uint64_t diff = (0x7FFFFFFF - prev_TTimetag);
					LastTtoAdd = ((uint64_t)0x7FFFFFFF * NRollOver) + diff + prev_TTimetag;
					NRollOver++;
				}
				prev_TTimetag = Trigger_timetag;

				mEvents->at(Nev)->Ev_Timestamp = Trigger_timetag + LastTtoAdd;
				mEvents->at(Nev)->Ev_ID = trigger_id;
				mEvents->at(Nev)->Ev_size = size * 4;
				mEvents->at(Nev)->Ev_chmask = chmask;
				mDevice->LastEvTimetag = (mEvents->at(Nev)->Ev_Timestamp * 8) / 1000000;//time in ms

				
				if (family_code == "740") {
					for (int g = 0; g < (MAX_CHANNELS / 8); g++) {
						if ((gmask >> g) & 0x1) {
							for (int i = 0; i < mEvents->at(Nev)->sizes()[g * 8]; i += 3) {
								for (int w = 0; w < 9; w++) {
									in >> word32;
									mBuffer40[w] = word32;
								}
								V1740UnpackEventCicle(mBuffer40, mChsplit40);
								for (int j = 0; j < 8; j++) {
									mEvents->at(Nev)->data()[g * 8 + j][i] = mChsplit40[j][0];
									mEvents->at(Nev)->data()[g * 8 + j][i + 1] = mChsplit40[j][1];
									mEvents->at(Nev)->data()[g * 8 + j][i + 2] = mChsplit40[j][2];
								}
							}
						}
					}
				}
				else if (family_code == "730" || family_code == "725") {
					if (zle) {
						emit ReadoutError(mDevice->getName(), CAEN_FELib_NotImplemented);
						QThread::msleep(300);
						continue;
					}
					for (int i = 0; i < MAX_CHANNELS; i++) {
						if ((chmask >> i) & 0x1) {
							for (int j = 0; j < mEvents->at(Nev)->sizes()[i]; j++) {
								in >> word16;
								mEvents->at(Nev)->data()[i][j] = word16 & 0x3FFF;
							}							
						}
					}
				}


			}
			mDevice->Triggered = true;
			++mDevice->NEvtsInQueue;

			// handle savefile
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
			if (mWaveDump2->mDataSavedEnabled.load() == 1) {
#else
			if (mWaveDump2->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
				if (NEvts == mNEvts_file && mNEvts_file) {
					for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
						if (mSaveFile[i] != nullptr) {
							mSaveFile[i]->close();
							delete mSaveFile[i];
							mSaveFile[i] = nullptr;
							delete outFile[i];
							outFile[i] = nullptr;
						}
					}
					InitFiles();
					NEvts = 0;
				}
				SaveEventToFile(mEvents->at(Nev));
				NEvts++;
			}

			Nev = (++Nev) % MAX_NEVTS;
			if (first_fill && Nev == (MAX_NEVTS - 1)) {
				first_fill = 0;
			}
		}//doread

		if (mSaving) {
			for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
				if (mSaveFile[i] != nullptr) {
					mSaveFile[i]->close();
					delete mSaveFile[i];
					mSaveFile[i] = nullptr;
					delete outFile[i];
					outFile[i] = nullptr;
				}
			}
			mSaving = false;
		}
	}
	mDevice->setVirtualAcqStatus("Ready");
	emit StopReadout(mDevice->getName());
}

void deviceReadoutThread::RunFromFileManual() {
	QString fname = mDevice->getRawDataFile();
	QString family_code = mDevice->getFamilyCode();
	QFile f(fname);
	f.open(QIODevice::ReadOnly);
	QDataStream in(&f);
	in.setByteOrder(QDataStream::LittleEndian);
	if (mDevice->DeviceClass == CAEN_DIG2)
		in.device()->seek(uint64_t(26) + uint64_t(16) * mNumOfchannels + 32);//file header + channel header + startrun word
	else
		in.device()->seek(uint64_t(26) + uint64_t(16) * mNumOfchannels);//file header + channel header
	qint64 file_size = f.size();
	quint64 word;
	quint32 word32;
	quint16 word16;
	quint8 word8;
	uint32_t size, trigger_id;
	uint64_t Trigger_timetag;
	uint64_t chmask;
	uint32_t gmask = 0;
	uint32_t size_bytes;
	bool zle,pack25 = 0;
	int nchannels = 0;
	int Nev = 0;
	int first_fill = 1;
	int NEvts = 0;
	int current_ev = -1;
	bool mask_set = false;
	QVector<int> enabled_ch_list;


	mDevice->LastEvTimetag = 0;
	mDevice->AcqStarted = true;
	mDevice->setVirtualAcqStatus("Running Offline");
	while (!mEndThread) {
		while (mDoRead) {
			if (mRequestStop) {
				QMutexLocker l(&mStopMutex);
				mDoRead = false;
				mDevice->closeEvtsQueue();
				mDevice->AcqStarted = false;
				mEndThread = true;
				break;
			}
			QMutexLocker locker(&mDevice->queue_mutex);
			if (first_fill)
				locker.unlock();

			if (mDevice->NEvtsInQueue == MAX_NEVTS) {
				//wait for available space in the queue
				mDevice->waitCondition.wait(&mDevice->queue_mutex);
			}

			Read_Ev_mutex.lock();
			waitNextEvCondition.wait(&Read_Ev_mutex);//wait for manual step

			if (mRequestStop)
				continue;

			if (Next_Ev_To_Load != (current_ev + 1) && (Next_Ev_To_Load != current_ev)) {
				bool jump_ok;
				int diff = abs(Next_Ev_To_Load - current_ev);
				if (Next_Ev_To_Load < current_ev) {
					jump_ok = in.device()->seek(in.device()->pos() - (size_bytes * (diff + 1)));
				}
				else {
					if (in.device()->pos() + (size_bytes * (diff - 1)) >= file_size)
						jump_ok = false;
					else
						jump_ok = in.device()->seek(in.device()->pos() + (size_bytes * (diff - 1)));

				}

				if (!jump_ok) {
					emit ErrorLoadingEv(mName, current_ev);
					Read_Ev_mutex.unlock();
					continue;
				}
			}
			current_ev = Next_Ev_To_Load;

			if (mDevice->DeviceClass == CAEN_DIG2) {
				in >> word;
				word = mWaveDump2->swapByteOrderWord(word);

				if (mDevice->isInfoStopWord(word)) {//StopRun word
					locker.unlock();
					QMutexLocker l(&mStopMutex);
					QThread::msleep(1000);
					mDoRead = false;
					mDevice->closeEvtsQueue();
					mDevice->AcqStarted = false;
					mEndThread = true;
					break;
				}

				//if (((word >> 60) & 0xf) == 1) {//TODO:check fw scope
				size = word & 0xffffffff;
				size_bytes = size * 8;
				trigger_id = (word >> 32) & 0xffffff;
				//}
				in >> word;
				word = mWaveDump2->swapByteOrderWord(word);
				Trigger_timetag = word & 0xffffffffffff;
				in >> word;
				word = mWaveDump2->swapByteOrderWord(word);
				chmask = word;
				if (!mask_set) {
					emit LoadChMask(mName, chmask);
					mask_set = true;
				}

				nchannels = 0;
				enabled_ch_list.clear();
				for (int ch = 0; ch < mNumOfchannels; ch++) {
					if ((chmask & ((uint64_t)1 << ch))) {
						nchannels++;
						mEvents->at(Nev)->sizes()[ch] = mDevice->getReclen_valueS();
						enabled_ch_list.append(ch);
					}
					else {
						mEvents->at(Nev)->sizes()[ch] = 0;
						mEvents->at(Nev)->data()[ch] = nullptr;
					}
				}

				mEvents->at(Nev)->Ev_Timestamp = Trigger_timetag;
				mEvents->at(Nev)->Ev_ID = trigger_id;
				mEvents->at(Nev)->Ev_size = size * 8;
				mEvents->at(Nev)->Ev_chmask = chmask;
				mDevice->LastEvTimetag = (Trigger_timetag * 8) / 1000000;//time in ms

				int read_size = 0;
				while (read_size < (size - 3) * 4 / nchannels) {
					for (int ch = 0; ch < nchannels; ch++) {
						in >> word;
						word = mWaveDump2->swapByteOrderWord(word);
						for (int i = 0; i < 4; i++)
							mEvents->at(Nev)->data()[enabled_ch_list.at(ch)][i + read_size] = (word >> (i * 16)) & 0xffff;
					}
					read_size += 4;
				}
			}
			else {//DIG1
				in >> word32;
				if (in.status() != QTextStream::Ok) {//end of file
					locker.unlock();
					QMutexLocker l(&mStopMutex);
					QThread::msleep(1000);
					mDoRead = false;
					mDevice->closeEvtsQueue();
					mDevice->AcqStarted = false;
					mEndThread = true;
					break;
				}
				size = word32 & 0xfffffff;
				size_bytes = size * 4;
				in >> word32;
				chmask = word32 & 0xff;
				gmask = chmask; //save group mask if needed
				zle = ((word32 >> 24) & 0x1);
				pack25 = ((word32 >> 25) & 0x1);
				in >> word32;
				trigger_id = word32 & 0xffffff;
				chmask |= (((word32 >> 24) & 0xff) << 8);
				if (family_code == "740") {
					uint32_t mask = 0;
					for (int g = 0; g < MAX_CHANNELS / 8; g++) {
						if (chmask & (1 << g)) {
							mask |= (0xff << (g * 8));
						}
					}
					chmask = mask;
				}
				if (!mask_set) {
					emit LoadChMask(mName, chmask);
					mask_set = true;
				}
				nchannels = 0;
				enabled_ch_list.clear();
				for (int ch = 0; ch < mNumOfchannels; ch++) {
					if ((chmask & ((uint64_t)1 << ch))) {
						nchannels++;
						mEvents->at(Nev)->sizes()[ch] = mDevice->getReclen_valueS();
						enabled_ch_list.append(ch);
					}
					else {
						mEvents->at(Nev)->sizes()[ch] = 0;
						mEvents->at(Nev)->data()[ch] = nullptr;
					}
				}
				in >> word32;
				Trigger_timetag = word32;

				mEvents->at(Nev)->Ev_Timestamp = Trigger_timetag;
				mEvents->at(Nev)->Ev_ID = trigger_id;
				mEvents->at(Nev)->Ev_size = size * 4;
				mEvents->at(Nev)->Ev_chmask = chmask;
				mDevice->LastEvTimetag = (Trigger_timetag * 8) / 1000000;//time in ms


				if (family_code == "740") {
					for (int g = 0; g < (MAX_CHANNELS / 8); g++) {
						if ((gmask >> g) & 0x1) {
							for (int i = 0; i < mEvents->at(Nev)->sizes()[g * 8]; i += 3) {
								for (int w = 0; w < 9; w++) {
									in >> word32;
									mBuffer40[w] = word32;
								}
								V1740UnpackEventCicle(mBuffer40, mChsplit40);
								for (int j = 0; j < 8; j++) {
									mEvents->at(Nev)->data()[g * 8 + j][i] = mChsplit40[j][0];
									mEvents->at(Nev)->data()[g * 8 + j][i + 1] = mChsplit40[j][1];
									mEvents->at(Nev)->data()[g * 8 + j][i + 2] = mChsplit40[j][2];
								}
							}
						}
					}
				}
				else if (family_code == "730" || family_code == "725") {
					if (zle) {
						emit ReadoutError(mDevice->getName(), CAEN_FELib_NotImplemented);
						QThread::msleep(300);
						continue;
					}
					for (int i = 0; i < MAX_CHANNELS; i++) {
						if ((chmask >> i) & 0x1) {
							for (int j = 0; j < mEvents->at(Nev)->sizes()[i]; j++) {
								in >> word16;
								mEvents->at(Nev)->data()[i][j] = word16 & 0x3FFF;
							}
						}
					}
				}
				else if (family_code == "751") {
					uint8_t end = 0, count=0;
					uint32_t s = 0, j=0;
					for (int i = 0; i < MAX_CHANNELS; i++) {
						if ((chmask >> i) & 0x1) {
							end = 0;
							s = 0;
							j = 0;
							while ((!end) && (j < mEvents->at(Nev)->sizes()[i])) {
								in >> word32;
								count = (uint8_t)((word32 & 0xC0000000) >> 30);
								switch (count) {
								case 3:
									mEvents->at(Nev)->data()[i][s] = (uint16_t)(word32 & 0x000003FF);
									mEvents->at(Nev)->data()[i][s + 1] = (uint16_t)((word32 & 0x000FFC00) >> 10);
									mEvents->at(Nev)->data()[i][s + 2] = (uint16_t)((word32 & 0x3FF00000) >> 20);
									s += 3;
									break;
								case 2:
									mEvents->at(Nev)->data()[i][s] = (uint16_t)(word32 & 0x000003FF);
									mEvents->at(Nev)->data()[i][s + 1] = (uint16_t)((word32 & 0x000FFC00) >> 10);
									s += 2;
									end = 1;
									break;
								case 1:
									mEvents->at(Nev)->data()[i][s] = (uint16_t)(word32 & 0x000003FF);
									s += 1;
									end = 1;
								}
								j++;
							}
						}
					}
				}
				else if (family_code == "724" || family_code == "780" || family_code == "782" || family_code == "781") {
					if (zle) {
						emit ReadoutError(mDevice->getName(), CAEN_FELib_NotImplemented);
						QThread::msleep(300);
						continue;
					}
					for (int i = 0; i < MAX_CHANNELS; i++) {
						if ((chmask >> i) & 0x1) {
							for (int j = 0; j < mEvents->at(Nev)->sizes()[i]; j++) {
								in >> word16;
								mEvents->at(Nev)->data()[i][j] = word16;
							}
						}
					}
				}
				else if (family_code == "721" || family_code == "731") {
					if (zle) {
						emit ReadoutError(mDevice->getName(), CAEN_FELib_NotImplemented);
						QThread::msleep(300);
						continue;
					}
					for (int i = 0; i < MAX_CHANNELS; i++) {
						if ((chmask >> i) & 0x1) {							
							for (int j = 0; j < mEvents->at(Nev)->sizes()[i]; j++) {
								in >> word8;
								mEvents->at(Nev)->data()[i][j] = word8;
							}
						}
					}
				}
				else if (family_code == "720" || family_code == "790") {
					if (zle) {
						emit ReadoutError(mDevice->getName(), CAEN_FELib_NotImplemented);
						QThread::msleep(300);
						continue;
					}
					switch (pack25) {
						case 0:
							for (int i = 0; i < MAX_CHANNELS; i++) {
								if ((chmask >> i) & 0x1) {
									for (int j = 0; j < mEvents->at(Nev)->sizes()[i]; j++) {
										in >> word16;
										mEvents->at(Nev)->data()[i][j] = word16;
									}
								}
							}
							break;
						case 1: {
							uint32_t ChannelTwoWords = (uint32_t)(((size - 4) / nchannels) / 2);
							for (int i = 0; i < MAX_CHANNELS; i++) {
								if ((chmask >> i) & 0x1) {
									int u = 0;
									for (int j = 0; j < ChannelTwoWords; j++) {
										in >> word;
										mEvents->at(Nev)->data()[i][u] = (uint16_t)((((word >> 6)& 0x3f) << 6) | (word & 0x3f));
										mEvents->at(Nev)->data()[i][u + 1] = (uint16_t)(((((word >> 18) & 0x3f) << 6) << 6) | ((word >> 12) & 0x3f));
										mEvents->at(Nev)->data()[i][u + 2] = (uint16_t)((((word >> 32) & 0x3f) << 6) | ((word >> 24) & 0x3f));
										mEvents->at(Nev)->data()[i][u + 3] = (uint16_t)((((word >> 44) & 0x3f) << 6) | ((word >> 38) & 0x3f));
										mEvents->at(Nev)->data()[i][u + 4] = (uint16_t)((((word >> 56) & 0x3f) << 6) | ((word >> 50) & 0x3f));
										u += 5;
									}
								}
							}
							}
							break;
					}
				}
				else if (family_code == "761") {
				int end = 0, partialSize=0, j=0;
				uint8_t count;
				for (int i = 0; i < MAX_CHANNELS; i++) {
					uint32_t ch_x761;
					ch_x761 = i / 4; 

					if ((chmask >> i) & 0x1) {
						int sizeOffset = 0;
						end = 0;
						partialSize = 0;
						j = 0;
						if ((i % 4) == 3) {
							sizeOffset = 1;
						}

						while ((!end) && (j < (mEvents->at(Nev)->sizes()[i] / 4))) {
							in >> word32;
							int32_t index = 2 * partialSize + sizeOffset;
							count = (uint8_t)((word32 & 0xC0000000) >> 30);
							switch (count) {
							case 3:
								mEvents->at(Nev)->data()[ch_x761][index] = (uint16_t)(word32 & 0x000003FF);
								mEvents->at(Nev)->data()[ch_x761][index + 2] = (uint16_t)((word32 & 0x000FFC00) >> 10);
								mEvents->at(Nev)->data()[ch_x761][index + 4] = (uint16_t)((word32 & 0x3FF00000) >> 20);
								partialSize += 3;
								break;
							case 2:
								mEvents->at(Nev)->data()[ch_x761][index] = (uint16_t)(word32 & 0x000003FF);
								mEvents->at(Nev)->data()[ch_x761][index + 2] = (uint16_t)((word32 & 0x000FFC00) >> 10);
								partialSize += 2;
								end = 1;
								break;
							case 1:
								mEvents->at(Nev)->data()[ch_x761][index] = (uint16_t)(word32 & 0x000003FF);
								partialSize += 1;
								end = 1;
							}
							j++;
						}
					}
					}
				}

			}

			mDevice->Triggered = true;
			++mDevice->NEvtsInQueue;

			// handle savefile
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
			if (mWaveDump2->mDataSavedEnabled.load() == 1) {
#else
			if (mWaveDump2->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
				if (NEvts == mNEvts_file && mNEvts_file) {
					for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
						if (mSaveFile[i] != nullptr) {
							mSaveFile[i]->close();
							delete mSaveFile[i];
							mSaveFile[i] = nullptr;
							delete outFile[i];
							outFile[i] = nullptr;
						}
					}
					InitFiles();
					NEvts = 0;
				}
				SaveEventToFile(mEvents->at(Nev));
				NEvts++;
			}

			Nev = (++Nev) % MAX_NEVTS;
			if (first_fill && Nev == (MAX_NEVTS - 1)) {
				first_fill = 0;
			}

			Read_Ev_mutex.unlock();
			emit NewEvLoaded(mName, current_ev, (Trigger_timetag * 8));

		}//doread

		if (mSaving) {
			for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
				if (mSaveFile[i] != nullptr) {
					mSaveFile[i]->close();
					delete mSaveFile[i];
					mSaveFile[i] = nullptr;
					delete outFile[i];
					outFile[i] = nullptr;
				}
			}
			mSaving = false;
		}

		mDevice->setVirtualAcqStatus("Ready");
		emit StopReadout(mDevice->getName());
	}
}


void deviceReadoutThread::V1740UnpackEventCicle(uint32_t* buffin, uint16_t buffout[8][3]) {
	int wpnt = 0, rpnt = 0;
	uint16_t buff[8][3];
	buff[0][wpnt] = (uint16_t)(buffin[rpnt] & 0x00000FFF);           /* S0[11:0] - CH0 */
	buff[0][wpnt + 1] = (uint16_t)((buffin[rpnt] & 0x00FFF000) >> 12);     /* S1[11:0] - CH0 */
	buff[0][wpnt + 2] = (uint16_t)((buffin[rpnt] & 0xFF000000) >> 24);     /* S2[ 7:0] - CH0 */
	rpnt++;
	buff[0][wpnt + 2] |= (uint16_t)((buffin[rpnt] & 0x0000000F) << 8);      /* S2[11:8] - CH0 */
	buff[1][wpnt] = (uint16_t)((buffin[rpnt] & 0x0000FFF0) >> 4);      /* S0[11:0] - CH1 */
	buff[1][wpnt + 1] = (uint16_t)((buffin[rpnt] & 0x0FFF0000) >> 16);     /* S1[11:0] - CH1 */
	buff[1][wpnt + 2] = (uint16_t)((buffin[rpnt] & 0xF0000000) >> 28);     /* S2[ 3:0] - CH1 */
	rpnt++;
	buff[1][wpnt + 2] |= (uint16_t)((buffin[rpnt] & 0x000000FF) << 4);      /* S2[11:4] - CH1 */
	buff[2][wpnt] = (uint16_t)((buffin[rpnt] & 0x000FFF00) >> 8);      /* S0[11:0] - CH2 */
	buff[2][wpnt + 1] = (uint16_t)((buffin[rpnt] & 0xFFF00000) >> 20);     /* S1[11:0] - CH2 */
	rpnt++;
	buff[2][wpnt + 2] = (uint16_t)(buffin[rpnt] & 0x00000FFF);            /* S2[11:0] - CH2 */
	buff[3][wpnt] = (uint16_t)((buffin[rpnt] & 0x00FFF000) >> 12);     /* S0[11:0] - CH3 */
	buff[3][wpnt + 1] = (uint16_t)((buffin[rpnt] & 0xFF000000) >> 24);     /* S1[ 7:0] - CH3 */
	rpnt++;
	buff[3][wpnt + 1] |= (uint16_t)((buffin[rpnt] & 0x0000000F) << 8);      /* S1[11:8] - CH3 */
	buff[3][wpnt + 2] = (uint16_t)((buffin[rpnt] & 0x0000FFF0) >> 4);      /* S2[11:0] - CH3 */
	buff[4][wpnt] = (uint16_t)((buffin[rpnt] & 0x0FFF0000) >> 16);     /* S0[11:0] - CH4 */
	buff[4][wpnt + 1] = (uint16_t)((buffin[rpnt] & 0xF0000000) >> 28);     /* S1[ 3:0] - CH4 */
	rpnt++;
	buff[4][wpnt + 1] |= (uint16_t)((buffin[rpnt] & 0x000000FF) << 4);      /* S1[11:4] - CH4 */
	buff[4][wpnt + 2] = (uint16_t)((buffin[rpnt] & 0x000FFF00) >> 8);      /* S2[11:0] - CH4 */
	buff[5][wpnt] = (uint16_t)((buffin[rpnt] & 0xFFF00000) >> 20);     /* S0[11:0] - CH5 */
	rpnt++;
	buff[5][wpnt + 1] = (uint16_t)(buffin[rpnt] & 0x00000FFF);          /* S1[11:0] - CH5 */
	buff[5][wpnt + 2] = (uint16_t)((buffin[rpnt] & 0x00FFF000) >> 12);     /* S2[11:0] - CH5 */
	buff[6][wpnt] = (uint16_t)((buffin[rpnt] & 0xFF000000) >> 24);    /* S0[ 7:0] - CH6 */
	rpnt++;
	buff[6][wpnt] |= (uint16_t)((buffin[rpnt] & 0x0000000F) << 8);     /* S0[11:8] - CH6 */
	buff[6][wpnt + 1] = (uint16_t)((buffin[rpnt] & 0x0000FFF0) >> 4);     /* S1[11:0] - CH6 */
	buff[6][wpnt + 2] = (uint16_t)((buffin[rpnt] & 0x0FFF0000) >> 16);    /* S2[11:0] - CH6 */
	buff[7][wpnt] = (uint16_t)((buffin[rpnt] & 0xF0000000) >> 28);    /* S0[ 3:0] - CH7 */
	rpnt++;
	buff[7][wpnt] |= (uint16_t)((buffin[rpnt] & 0x000000FF) << 4);     /* S0[11:4] - CH7 */
	buff[7][wpnt + 1] = (uint16_t)((buffin[rpnt] & 0x000FFF00) >> 8);     /* S1[11:0] - CH7 */
	buff[7][wpnt + 2] = (uint16_t)((buffin[rpnt] & 0xFFF00000) >> 20);    /* S2[11:0] - CH7 */
	memcpy(buffout, buff, sizeof(buff));
	return;
}

deviceReadoutThread::~deviceReadoutThread()
{
}
