/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		dataSaveThread.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "dataSaveThread.h"

#include <QtGlobal>

#include "WaveDump2.h"

dataSaveThread::dataSaveThread(deviceReadoutThread* pare, WaveDump2* w, CAENDevice* dev, std::uint32_t runId)
{
	mDevice = dev;
	mWaveDump2 = w;
	this->mName = dev->getName();
	mRunId = runId;
	this->mNumOfchannels = dev->getNumCh();
	mSaveFile.resize(mNumOfchannels);
	outFile.resize(mNumOfchannels);
	mChEnableMask = 0;
	for (int ch = 0; ch < mNumOfchannels; ch++) {
		mSaveFile[ch] = NULL;
		if (dev->isChEnabled(ch))
			mChEnableMask |= (uint64_t{ 1 } << ch);
}

#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
	if (mWaveDump2->mDataSavedEnabled.load() == 1) {
#else
	if (mWaveDump2->mDataSavedEnabled.loadRelaxed() == 1) {
#endif
		mSaveRaw = w->IsFastSaveEnabled();		
	}
	InitFiles();

	connect(this, SIGNAL(finished()), this, SLOT(deleteLater()));
	connect(this, SIGNAL(StopSave()), pare, SLOT(SaveThreadFinished()));
	//connect(this, SIGNAL(GenericError(QString)), parent, SLOT(SaveError(QString)));
}


void dataSaveThread::startRun() {
	QMutexLocker l(&mStopMutex);
	mDoSave = true;
	mRequestStop = false;
}

void dataSaveThread::stopRun() {
	QMutexLocker l(&mStopMutex);
	mRequestStop = true;
}

void dataSaveThread::run() {

	if (mSaveRaw)
		RunSaveRaw();	

	exit(0);
}

void dataSaveThread::SaveRawBuffer(char* buffer, size_t size) {
	mSaveFile[0]->write((const char*)buffer, size);
	if (mSaveFile[0]->size() > 4294967296) {
		QString name = mSaveFile[0]->fileName();
		name.remove(".wbin");
		if (mNumFile > 0) {
			name.remove(QString("_%1").arg(mNumFile));
		}
		mSaveFile[0]->close();
		delete mSaveFile[0];
		mNumFile++;
		mSaveFile[0] = new QFile(name + QString("_%1.wbin").arg(mNumFile));
		outFile[0] = new QTextStream(this->mSaveFile[0]);
		mSaveFile[0]->open(QIODevice::WriteOnly);
		mSaveFile[0]->write((const char*)mHeader, mHeaderSize); //Header
		if(mDevice->DeviceClass == CAEN_DIG2)
			mSaveFile[0]->write((const char*)mStartRun, 4 * sizeof(uint64_t)); //StartRun words
	}
}

void dataSaveThread::WriteFileHeader() {
	mHeaderSize = 0;
	QString familyname = mDevice->getFamilyCode();
	QString result = "";
	for (const QChar c : qAsConst(familyname)) {
		if (c.isDigit() || c == '.') {
			result.append(c);
		}
	}
	uint32_t family = result.toUInt();
	mSaveFile[0]->write((const char*)&family, sizeof(family)); //device Family Code
	memcpy(mHeader, (const char*)&family, sizeof(family));
	mHeaderSize += sizeof(family);
	uint32_t pid = mDevice->getSN().toUInt();
	mSaveFile[0]->write((const char*)&pid, sizeof(pid)); //device PID
	memcpy(mHeader + mHeaderSize, (const char*)&pid, sizeof(pid));
	mHeaderSize += sizeof(pid);
	uint32_t ADCNBit = mDevice->getADCNBit();
	mSaveFile[0]->write((const char*)&ADCNBit, sizeof(ADCNBit)); //device ADC nbits (VRange = 2^ADCNBit -1)
	memcpy(mHeader + mHeaderSize, (const char*)&ADCNBit, sizeof(ADCNBit));
	mHeaderSize += sizeof(ADCNBit);
	double SamplToS = mDevice->getSample_to_S();
	uint32_t Tfactor = SamplToS * 1e12;
	mSaveFile[0]->write((const char*)&Tfactor, sizeof(Tfactor)); //device Sample to S
	memcpy(mHeader + mHeaderSize, (const char*)&Tfactor, sizeof(Tfactor));
	mHeaderSize += sizeof(Tfactor);
	uint32_t TSource = mDevice->calcTrgSourceToSave();
	mSaveFile[0]->write((const char*)&TSource, sizeof(TSource)); //device Trigger Source
	memcpy(mHeader + mHeaderSize, (const char*)&TSource, sizeof(TSource));
	mHeaderSize += sizeof(TSource);
	uint32_t pretrg = static_cast<uint32_t>(mDevice->getPreTrgCachedValueS());
	mSaveFile[0]->write((const char*)&pretrg, sizeof(pretrg)); //device PreTrigger
	memcpy(mHeader + mHeaderSize, (const char*)&pretrg, sizeof(pretrg));
	mHeaderSize += sizeof(pretrg);
	mSaveFile[0]->write((const char*)&this->mNumOfchannels, sizeof(this->mNumOfchannels)); //device NChannels
	memcpy(mHeader + mHeaderSize, (const char*)&this->mNumOfchannels, sizeof(this->mNumOfchannels));
	mHeaderSize += sizeof(this->mNumOfchannels);
	for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
		const auto baseline = mDevice->getBaselineLSB(i);
		const auto sample_to_mV = mDevice->getSample_to_V(i) * 1e3;
		uint32_t gain = mDevice->getCachedChGain(i) * 1e6;
		uint32_t factor = sample_to_mV * 1e6;
		int32_t thr = static_cast<int32_t>(mDevice->getThr_value(i));
		mSaveFile[0]->write((const char*)&baseline, sizeof(baseline)); //channel baseline in LSB
		memcpy(mHeader + mHeaderSize, (const char*)&baseline, sizeof(baseline));
		mHeaderSize += sizeof(baseline);
		mSaveFile[0]->write((const char*)&factor, sizeof(factor)); //1 sample in nV  ---> (data[j] - baseline) * factor / 1e6 to switch to mV
		memcpy(mHeader + mHeaderSize, (const char*)&factor, sizeof(factor));
		mHeaderSize += sizeof(factor);
		mSaveFile[0]->write((const char*)&gain, sizeof(gain)); //chgain factor
		memcpy(mHeader + mHeaderSize, (const char*)&gain, sizeof(gain));
		mHeaderSize += sizeof(gain);
		mSaveFile[0]->write((const char*)&thr, sizeof(thr)); //ch threshold (relative to baseline)
		memcpy(mHeader + mHeaderSize, (const char*)&thr, sizeof(thr));
		mHeaderSize += sizeof(thr);
	}
}

void dataSaveThread::InitFiles() {
	QString filen;
	QString devicename = mDevice->getName();
	devicename.replace(":", "");
	devicename.replace("//", "-");
	mWaveDump2->getOutputSettings(mFile_folder, mFile_prefix, mFile_ftype, mFile_header, mFile_format, mFile_sync, mNEvts_file);
	if (mFile_sync == "YES" && !mSaveRaw) {
		mSaving = false;
		return;
	}
	QString ext = (mFile_format.contains("ASCII")) ? ".txt" : ".bin";
	if (mSaveRaw)
		ext = ".wbin";
	QDateTime date = QDateTime::currentDateTime();
	QString formattedTime = date.toString("yyyyMMddhhmmss");
	if (mFile_ftype == "SINGLE") {
		if (mRunId < 10)
			filen = mFile_prefix + "_" + devicename + "_" + formattedTime + QString("-0%1").arg(mRunId) + ext;
		else
			filen = mFile_prefix + "_" + devicename + "_" + formattedTime + QString("-%1").arg(mRunId) + ext;
		this->mSaveFile[0] = new QFile(mFile_folder + "/" + filen);
		outFile[0] = new QTextStream(this->mSaveFile[0]);
		outFile[0]->setRealNumberNotation(QTextStream::FixedNotation);
		int save_status;
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
		save_status = mWaveDump2->mDataSavedEnabled.load();
#else
		save_status= mWaveDump2->mDataSavedEnabled.loadRelaxed();
#endif
		if (save_status == 1) {
			if (!this->mSaveFile[0]->open(QIODevice::WriteOnly)) {
				emit GenericError("Unable to create file: " + this->mSaveFile[0]->errorString());
				return;
			}
		}
		if (mSaveRaw) {
			WriteFileHeader();
		}
	}
	else {

		for (uint16_t i = 0; i < this->mNumOfchannels; i++) {
			if (!(mChEnableMask & (uint64_t{ 1 } << i)))
				continue;
			if (mRunId < 10)
				filen = mFile_prefix + "_" + devicename + "_CH" + QString::number(i) + "_" + formattedTime + QString("-0%1").arg(mRunId) + ext;
			else
				filen = mFile_prefix + "_" + devicename + "_CH" + QString::number(i) + "_" + formattedTime + QString("-%1").arg(mRunId) + ext;

			if (mNEvts_file) {
				if (!QDir(mFile_folder + QString("/CH%1").arg(i)).exists())
					QDir().mkdir(mFile_folder + QString("/CH%1").arg(i));
				this->mSaveFile[i] = new QFile(mFile_folder + QString("/CH%1/").arg(i) + filen);
			}
			else
				this->mSaveFile[i] = new QFile(mFile_folder + "/" + filen);
			outFile[i] = new QTextStream(this->mSaveFile[i]);
			if (!this->mSaveFile[i]->open(QIODevice::WriteOnly)) {
				emit GenericError("Unable to create file: " + this->mSaveFile[i]->errorString());
				return;
			}
		}
	}
	mSaving = true;
}


void dataSaveThread::RunSaveRaw()
{
	QString name = mDevice->getName();
	int NTOTB = 0;
	int dev = mDevice->getIndex();
	size_t size = 0;
	int NSaveBuff = -1;
	mNumFile = 0;

	while (!mEndThread) {
		while (mDoSave) {
			if (mRequestStop) {
				QMutexLocker l(&mStopMutex);
				mDoSave = false;
				mEndThread = true;
				break;
			}

			if (mDevice->NBuffFilled == 0)
				continue;

			NSaveBuff = (NSaveBuff + 1) % MAX_RAW_BUFFS;

			if (mDevice->BuffOccupancy[NSaveBuff]) {
					SaveRawBuffer(mDevice->RawDataBuffer[NSaveBuff], mDevice->BuffOccupancy[NSaveBuff]);

					NTOTB++;

					mDevice->BuffMutex.lock();
					--mDevice->NBuffFilled;
					mDevice->BuffMutex.unlock();
					mDevice->BuffOccupancy[NSaveBuff] = 0;
			}

			

			}//doread


		if (mSaveFile[0] != nullptr) {
			mSaveFile[0]->close();
			delete mSaveFile[0];
			mSaveFile[0] = nullptr;
			delete outFile[0];
			outFile[0] = nullptr;
		}

		}

	emit StopSave();
}


void dataSaveThread::StoreStartRunWords(uint64_t* s) {
	mStartRun[0] = s[0];
	mStartRun[1] = s[1];
	mStartRun[2] = s[2];
	mStartRun[3] = s[3];
}
void dataSaveThread::StoreStopRunWord(uint64_t s) {
	mStopRun = s;
}

dataSaveThread::~dataSaveThread()
{
}
