/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		controlPanel.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "controlPanel.h"
#include "ui_controlPanel.h"

#include <CAEN_FELib.h>

#include "WaveDump2.h"
#include "chmaskDialog.h"

controlPanel::controlPanel(WaveDump2 *parent)
	: QWidget(parent)
	, ui(new Ui::controlPanel)
	, mWaveDump(parent)
	, mMinR{}
	, mMaxR{}
	, mIncrR{}
	, mMinP{}
	, mMaxP{}
	, mIncrP{}
{
	ui->setupUi(this);
	mIgnoreEvent = false;

	ui->dBox_reclen->setDecimals(1);
	ui->dBox_reclen->setValue(mWaveDump->getRecLenS());
	QString mu = QChar(0xbc, 0x03);
	ui->label_reclen->setText("Recordlength [" + mu + "s]");

	ui->dBox_pretrg->setValue(mWaveDump->getPreTrgS());
	ui->label_pretrg->setText("PreTrigger [" + mu + "s]");

	ui->pB_trg_mask->setEnabled(false);

	trigger_timer = new QTimer(this);
	QObject::connect(trigger_timer, SIGNAL(timeout()), mWaveDump, SLOT(SWTrg()));

	ui->pBStartManual->hide();
	ui->pbStopAcq->setEnabled(false);

	connect(this->ui->pbStartAcq, SIGNAL(clicked()), mWaveDump, SLOT(startAcquisition()));
	connect(this->ui->pbStopAcq, SIGNAL(clicked()), mWaveDump, SLOT(stopAcquisition()));
	connect(this->ui->pB_Trigger, SIGNAL(clicked()), mWaveDump, SLOT(SingleSWTrg()));
	connect(this->ui->gBox_cursors, SIGNAL(clicked(bool)), this, SLOT(cursors_checked(bool)));
	connect(this->ui->cBox_x_unit, SIGNAL(currentIndexChanged(int)), this, SLOT(XUnitChanged(int)));
	connect(this->ui->cBox_y_unit, SIGNAL(currentIndexChanged(int)), this, SLOT(YUnitChanged(int)));
	connect(this->ui->dBox_reclen, SIGNAL(valueChanged(double)), this, SLOT(ReclenChanged(double)));
	connect(this->ui->dBox_pretrg, SIGNAL(valueChanged(double)), this, SLOT(PretrgChanged(double)));
	//connect(this->ui->cbDev, SIGNAL(currentTextChanged(QString)), mWaveDump, SLOT(ActiveDeviceChanged(QString)));
	connect(this, SIGNAL(DevChanged(QString)),mWaveDump,SLOT(ActiveDeviceChanged(QString)));
	connect(this->ui->cbDev, SIGNAL(currentTextChanged(QString)), this, SLOT(DeviceChanged(QString)));
	connect(this->ui->cBox_trigger_source, SIGNAL(currentIndexChanged(int)), this, SLOT(TriggerSourceChanged(int)));
	connect(this->ui->cBox_trigger_mode, SIGNAL(currentIndexChanged(int)), this, SLOT(TriggerModeChanged(int)));
	connect(this->ui->pB_trg_mask, SIGNAL(clicked()), this, SLOT(SetSelfTrgMask()));
	connect(this->ui->pBStartManual, SIGNAL(clicked()), mWaveDump, SLOT(startOfflineManualAcq()));
}

controlPanel::~controlPanel()
{
	delete ui;
}

void controlPanel::disableAll(bool f) {
	if(f)
		ui->pbStopAcq->setEnabled(!f);
	ui->pbStartAcq->setEnabled(!f);	
	ui->pB_Trigger->setEnabled(!f);
	ui->pBStartManual->setEnabled(!f);
}

void controlPanel::enableAllStartPb() {
	ui->pbStartAcq->setEnabled(true);
	ui->pBStartManual->setEnabled(true);
}

void controlPanel::toogleStartStop() {
	if (ui->pbStartAcq->isEnabled() || ui->pBStartManual->isEnabled()) { //start acq 
		ui->pbStartAcq->setEnabled(false);
		ui->pBStartManual->setEnabled(false);
		ui->pbStopAcq->setEnabled(true);
		if (!mWaveDump->OfflineMode) {
			if (ui->cBox_trigger_mode->currentIndex() == TRG_MODE_NORMAL)
				ui->pB_Trigger->setEnabled(true);

			if (!mWaveDump->NAutoTrg)
				trigger_timer->stop();
			else {
				if (!trigger_timer->isActive())
					trigger_timer->start(500);
			}
			ui->dBox_reclen->setEnabled(false);
			ui->dBox_pretrg->setEnabled(false);
			ui->pB_trg_mask->setEnabled(false);
		}
	}
	else { //stop acq 		
		ui->pbStopAcq->setEnabled(false);
		if(mWaveDump->IsManualOfflineEnabled())
			ui->pBStartManual->setEnabled(true);
		else
			ui->pbStartAcq->setEnabled(true);
		if (!mWaveDump->OfflineMode) {
			ui->pB_Trigger->setEnabled(false);
			ui->dBox_reclen->setEnabled(true);
			ui->dBox_pretrg->setEnabled(true);
			ui->pB_trg_mask->setEnabled(true);
			if (trigger_timer->isActive() && !mWaveDump->NDevStarted)
				trigger_timer->stop();
		}
	}

}

void controlPanel::setStartStopStatus(int status) {
	if ((status == 1 && ui->pbStartAcq->isEnabled()) || (status == 0 && ui->pbStopAcq->isEnabled())) { 
		toogleStartStop();
	}
}

void controlPanel::UpdateStartOptions() {
	if (mWaveDump->getGlobalSMode().contains("ASYNC")) {
		ui->pBStartManual->setEnabled(true);
		ui->cbDev->setEnabled(true);
	}
	else {
		ui->cbDev->blockSignals(true);
		ui->cbDev->setCurrentText("ALL");
		ui->cbDev->setEnabled(false);
		ui->pBStartManual->setEnabled(true);
		ui->cbDev->blockSignals(false);
	}
}

void controlPanel::DeviceChanged(QString dev){
	if (mWaveDump->OfflineMode){
		if (!mWaveDump->getGlobalSMode().contains("ASYNC")) {
			if (dev == "ALL") {
				ui->pBStartManual->setEnabled(true);
				ui->pbStartAcq->setEnabled(true);
			}
			else {
				ui->pBStartManual->setEnabled(false);
				ui->pbStartAcq->setEnabled(false);
			}
		}
		else {
			ui->pBStartManual->setEnabled(true);
			ui->pbStartAcq->setEnabled(true);
		}
	}
	emit DevChanged(dev);
}


void controlPanel::updateStartableDevices(int cmd, const QString& name) {
	int idx;
	ui->cbDev->blockSignals(true);
	switch (cmd) {
	case 0: // remove
		idx = this->ui->cbDev->findText(name);
		if (idx != -1) {
			this->ui->cbDev->removeItem(idx);
		}
		if ((this->ui->cbDev->count() == 1) && (this->ui->cbDev->currentText() == "ALL")) {
			this->ui->cbDev->clear();
			this->ui->cbDev->setEnabled(false);
			this->ui->pbStartAcq->setEnabled(false);
		}
		if ((this->ui->cbDev->count() == 2) && (this->ui->cbDev->findText("ALL") != -1)) {
			idx = this->ui->cbDev->findText("ALL");
			this->ui->cbDev->removeItem(idx);
		}
		break;
	case 1: // add
		this->ui->cbDev->addItem(name);
		if ((this->ui->cbDev->count() > 1) && (this->ui->cbDev->findText("ALL") == -1)) {
			this->ui->cbDev->addItem("ALL");
			this->ui->cbDev->model()->sort(0);
		}
		this->ui->cbDev->setEnabled(true);
		this->ui->pbStartAcq->setEnabled(true);
		break;
	}
	ui->cbDev->blockSignals(false);

}

void controlPanel::disableALLOption() {
	ui->cbDev->blockSignals(true);
	int idx = this->ui->cbDev->findText("ALL");
	if (idx != -1)
		this->ui->cbDev->removeItem(idx);
	ui->cbDev->blockSignals(false);
}

QString controlPanel::getDevName() const {
	return this->ui->cbDev->currentText();
}

void controlPanel::updateCursorsTrace(QColor color, QString name) {
	ui->cBox_C_trace->clear();
	QPixmap iconPixmap(32, 32);
	iconPixmap.fill(color);
	QIcon itemIcon(iconPixmap);
	ui->cBox_C_trace->addItem(itemIcon, name, color);
}

void controlPanel::cursors_checked(bool checked) {
	mWaveDump->New_Cursors(checked);
	if (!checked) {
		ui->lineEdit_x1->setText("");
		ui->lineEdit_y1->setText("");
		ui->lineEdit_x2->setText("");
		ui->lineEdit_y2->setText("");
		ui->lineEdit_Dx->setText("");
		ui->lineEdit_Dy->setText("");
	}
}

void controlPanel::UpdateCursorsData(QVector<double> pos) {
	if (mXUnit == UOM_SAMPLE) {
		ui->lineEdit_x1->setText(QString::number(qRound(pos.at(0)), 'f', 0));
		ui->lineEdit_x2->setText(QString::number(qRound(pos.at(2)), 'f', 0));
		ui->lineEdit_Dx->setText(QString::number(qRound(fabs(pos.at(0) - pos.at(2))), 'f', 0));
	}
	else {
		ui->lineEdit_x1->setText(QString::number(pos.at(0), 'f', 2));
		ui->lineEdit_x2->setText(QString::number(pos.at(2), 'f', 2));
		ui->lineEdit_Dx->setText(QString::number(fabs(pos.at(0) - pos.at(2)), 'f', 2));
	}

	if (mYUnit == UOM_SAMPLE) {
		ui->lineEdit_y1->setText(QString::number(qRound(pos.at(1)), 'f', 0));
		ui->lineEdit_y2->setText(QString::number(qRound(pos.at(3)), 'f', 0));
		ui->lineEdit_Dy->setText(QString::number(qRound(fabs(pos.at(1) - pos.at(3))), 'f', 0));
	}
	else {
		ui->lineEdit_y1->setText(QString::number(pos.at(1), 'f', 2));
		ui->lineEdit_y2->setText(QString::number(pos.at(3), 'f', 2));
		ui->lineEdit_Dy->setText(QString::number(fabs(pos.at(1) - pos.at(3)), 'f', 2));
	}
}

void controlPanel::ActiveDevChangeCmd(QString dev) {
	int index = ui->cbDev->findText(dev);
	ui->cbDev->setCurrentIndex(index);
}

void controlPanel::TriggerModeChanged(int mode) {
	if (mIgnoreEvent)
		return;
	if (mode == TRG_MODE_NORMAL) {
		if(!mWaveDump->NAutoTrg)
			trigger_timer->stop();
		ui->pB_Trigger->setEnabled(true);
		ui->pB_Trigger->setChecked(false);
	}
	else {
		ui->cBox_trigger_source->setCurrentIndex(Software);
		if(!trigger_timer->isActive())
			trigger_timer->start(500);
		ui->pB_Trigger->setEnabled(false);
		ui->pB_Trigger->setChecked(true);
	}
	mWaveDump->setTrgMode(mode);
	NWidgetTrgMode();
}

void controlPanel::TriggerModeChangeCmd(int mode) {
	ui->cBox_trigger_mode->setCurrentIndex(mode);
	ui->cBox_trigger_mode->currentIndexChanged(mode);
}

void controlPanel::TriggerSourceChanged(int index) {
	if (mIgnoreEvent)
		return;
	ui->pB_trg_mask->setEnabled(false);
	QString trg_source;
	switch (index) {
	case Software:
		trg_source = "SwTrg";
		break;
	case ExtTrgIn:
		trg_source = "SwTrg|TrgIn";
		break;
	case ExtGPIO:
		trg_source = "SwTrg|GPIO";
		break;
	case Self:
		trg_source = "SwTrg|ITLA";
		if (ui->pbStartAcq->isEnabled())
			ui->pB_trg_mask->setEnabled(true);
		break;
	case Internal:
		trg_source = "SwTrg|TestPulse";
		break;
	}
	mWaveDump->setTrgSource(trg_source);
	NWidgetTrgSrc();
	if (index == Software)
		ui->cBox_trigger_mode->setCurrentIndex(TRG_MODE_AUTO);
	else
		ui->cBox_trigger_mode->setCurrentIndex(TRG_MODE_NORMAL);
}

void controlPanel::SetSelfTrgMask() {
	QString current_mask = mWaveDump->getSelfTrgMask();
	uint64_t val = current_mask.toULongLong();
	chmaskDialog *dlg = new chmaskDialog(this, mWaveDump->getNumChannels(), val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask; 
		new_mask.setNum(dlg->getMask());
		mWaveDump->setSelfTrgMask(new_mask);
	}
	delete dlg;
}

void controlPanel::updateTrgSettings(int tmode, QString tsource){
	updateTrgSource(tsource);
	if (tsource.contains("ITL") && ui->pbStartAcq->isEnabled())
		ui->pB_trg_mask->setEnabled(true);
	else
		ui->pB_trg_mask->setEnabled(false);
	mIgnoreEvent = true;
	ui->cBox_trigger_mode->setCurrentIndex(tmode);
	mIgnoreEvent = false;
	NWidgetTrgMode();
}

void controlPanel::updateTrgSource(QString trg_src) {
	mIgnoreEvent = true;
	ui->pB_trg_mask->setEnabled(false);
	if (trg_src.contains("TrgIn", Qt::CaseInsensitive))
		ui->cBox_trigger_source->setCurrentIndex(ExtTrgIn);
	else if (trg_src.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_trigger_source->setCurrentIndex(ExtGPIO);
	else if (trg_src.contains("ITL", Qt::CaseInsensitive)) {
		ui->cBox_trigger_source->setCurrentIndex(Self);
		ui->pB_trg_mask->setEnabled(true);
	}
	else if (trg_src.contains("Pulse", Qt::CaseInsensitive))
		ui->cBox_trigger_source->setCurrentIndex(Internal);
	else {
		ui->cBox_trigger_source->setCurrentIndex(Software);
		
	}
	mIgnoreEvent = false;
	NWidgetTrgSrc();
	if(ui->cBox_trigger_source->currentIndex()==Software)
		ui->cBox_trigger_mode->setCurrentIndex(TRG_MODE_AUTO);
	else
		ui->cBox_trigger_mode->setCurrentIndex(TRG_MODE_NORMAL);
}

void controlPanel::CheckTrgSourceOptions(int dev_type, QString trg_src) {
	if ((dev_type == CAEN_DIG1) && (ui->cBox_trigger_source->count() > 3)) {
		ui->cBox_trigger_source->blockSignals(true);
		ui->cBox_trigger_source->removeItem(ui->cBox_trigger_source->count()-1);
		ui->cBox_trigger_source->removeItem(ui->cBox_trigger_source->count()-1);
		ui->cBox_trigger_source->blockSignals(false);
		updateTrgSource(trg_src);
		return;
	}
	if ((dev_type == CAEN_DIG2) && (ui->cBox_trigger_source->count() < 4)) {
		ui->cBox_trigger_source->blockSignals(true);
		ui->cBox_trigger_source->addItem("Ext-GPIO");
		ui->cBox_trigger_source->addItem("Internal");
		ui->cBox_trigger_source->removeItem(4);
		ui->cBox_trigger_source->blockSignals(false);
		updateTrgSource(trg_src);
		return;
	}
}

void controlPanel::updateReclenWidget(double max, double min, double incr, double value) {
	mIgnoreEvent = true;
	mMinR = min;
	mMaxR = max;
	mIncrR = incr;
	QString mu = QChar(0xbc, 0x03);
	switch (mXUnit) {
		case UOM_SAMPLE:
			ui->dBox_reclen->setDecimals(0);
			ui->label_reclen->setText("Record Length [samples]");
			break;
		case UOM_PHYS_UNIT:
			ui->dBox_reclen->setDecimals(3);
			ui->label_reclen->setText("Record Length [" + mu + "s]");
			break;
	}
	ui->dBox_reclen->setMinimum(min);
	ui->dBox_reclen->setMaximum(max);
	ui->dBox_reclen->setSingleStep(incr);
	ui->dBox_reclen->setValue(value);
	mIgnoreEvent = false;
	NWidgetReclen();
}

void controlPanel::ReclenChanged(double reclen) {
	if (mIgnoreEvent)
		return;
	mWaveDump->ReclenChanged(reclen);
	mIgnoreEvent = true;
	if(mXUnit == UOM_SAMPLE)
		ui->dBox_reclen->setValue(mWaveDump->getRecLenS());
	else
		ui->dBox_reclen->setValue(mWaveDump->getRecLenS() * (mWaveDump->getDecimation() * mWaveDump->getSampleToS() * 1e+6));
	mIgnoreEvent = false;
	NWidgetReclen();
	if (mXUnit == UOM_PHYS_UNIT && (ui->dBox_pretrg->value() != -1))
		PretrgChanged(ui->dBox_pretrg->value());//update post trigger samples so that the value in ns is the same
}

void controlPanel::updatePreTrgWidget(double max, double min, double incr, double value) {
	mIgnoreEvent = true;
	mMinP=min;
	mMaxP=max;
	mIncrP=incr;
	QString mu = QChar(0xbc, 0x03);
	switch (mXUnit) {
	case UOM_SAMPLE:
		ui->dBox_pretrg->setDecimals(0);
		ui->label_pretrg->setText("Pre Trigger [samples]");
		break;
	case UOM_PHYS_UNIT:
		ui->dBox_pretrg->setDecimals(3);
		ui->label_pretrg->setText("Pre Trigger [" + mu + "s]");
		break;
	}
	ui->dBox_pretrg->setMinimum(min);
	ui->dBox_pretrg->setMaximum(max);
	ui->dBox_pretrg->setSingleStep(incr);
	ui->dBox_pretrg->setValue(value);
	mIgnoreEvent = false;
	NWidgetPreTrg();
}

void controlPanel::PretrgChanged(double value) {
	if (mIgnoreEvent)
		return;	
	int scale_factor = 1.;
	if (mXUnit == UOM_SAMPLE && mWaveDump->getActiveDev()->DeviceClass == CAEN_DIG1)
		scale_factor = mWaveDump->getActiveDev()->getDecimation();
	
	mWaveDump->PreTrgChanged(value * scale_factor);
	mIgnoreEvent = true;
	if (mXUnit == UOM_SAMPLE)
		ui->dBox_pretrg->setValue(mWaveDump->getPreTrgS() / scale_factor);
	else {
		if(mWaveDump->getActiveDev()->DeviceClass == CAEN_DIG1)
			scale_factor = 1.;
		else
			scale_factor = mWaveDump->getActiveDev()->getDecimation();
		ui->dBox_pretrg->setValue(mWaveDump->getPreTrgS() * mWaveDump->getActiveDev()->getSample_to_S() * 1e+6);
	}
	mIgnoreEvent = false;
	NWidgetPreTrg();
}

void controlPanel::XUnitChanged(int unit) {
	mXUnit = unit;
	mWaveDump->XUnitChanged(unit);
}

void controlPanel::YUnitChanged(int unit) {
	mIgnoreEvent = true;
	mYUnit = unit;
	mWaveDump->YUnitChanged(unit);
	mIgnoreEvent = false;
}

void controlPanel::YUnitChangeCmd(int unit) {
	ui->cBox_y_unit->setCurrentIndex(unit);
}
void controlPanel::XUnitChangeCmd(int unit) {
	ui->cBox_x_unit->setCurrentIndex(unit);
}

void controlPanel::updateSmodeSettings() {
	ui->cbDev->blockSignals(true);
	if (mWaveDump->getGlobalSMode() != "SW_ASYNC") {
		ui->cbDev->setCurrentText("ALL");
	}
	else {
		ui->cbDev->setCurrentText(mWaveDump->getCurrentDeviceName());
	}
	ui->cbDev->blockSignals(false);
}

void controlPanel::updatePlotType(int type) {
	switch (type) {
		case PLOT_TYPE_WAVEFORMS:
			ui->cBox_x_unit->setEnabled(true);
			ui->cBox_y_unit->setEnabled(true);
			break;
		case PLOT_TYPE_FFT:
			ui->cBox_x_unit->setEnabled(false);
			ui->cBox_y_unit->setEnabled(false);
			break;
		case PLOT_TYPE_SAMPLES_HISTO:
			ui->cBox_x_unit->setEnabled(false);
			ui->cBox_y_unit->setEnabled(false);
			break;
	}

}

void controlPanel::WWidgetReclen() {
	ui->dBox_reclen->setStyleSheet("color: red");
	ui->dBox_reclen->setSpecialValueText("?");
	ui->dBox_reclen->blockSignals(true);
	ui->dBox_reclen->setMinimum(ui->dBox_reclen->minimum() - 1);
	ui->dBox_reclen->setValue(ui->dBox_reclen->minimum());
	ui->dBox_reclen->blockSignals(false);
}

void controlPanel::WWidgetPreTrg() {
	ui->dBox_pretrg->setStyleSheet("color: red");
	ui->dBox_pretrg->setSpecialValueText("?");
	ui->dBox_pretrg->blockSignals(true);
	ui->dBox_pretrg->setMinimum(ui->dBox_pretrg->minimum() - 1);
	ui->dBox_pretrg->setValue(ui->dBox_pretrg->minimum());
	ui->dBox_pretrg->blockSignals(false);
}

void controlPanel::WWidgetTrgSrc() {
	ui->cBox_trigger_source->setStyleSheet("color: red");
	ui->cBox_trigger_source->blockSignals(true);
	ui->cBox_trigger_source->setCurrentIndex(-1);
	ui->cBox_trigger_source->blockSignals(false);
}

void controlPanel::WWidgetTrgMode() {
	ui->cBox_trigger_mode->setStyleSheet("color: red");
	ui->cBox_trigger_mode->blockSignals(true);
	ui->cBox_trigger_mode->setCurrentIndex(-1);
	ui->cBox_trigger_mode->blockSignals(false);
}

void controlPanel::NWidgetReclen() {
	if (ui->dBox_reclen->styleSheet().contains("color: red")) {
		ui->dBox_reclen->blockSignals(true);
		ui->dBox_reclen->setStyleSheet("color: black");
		ui->dBox_reclen->setSpecialValueText("");
		ui->dBox_reclen->setMinimum(ui->dBox_reclen->minimum() + 1);
		ui->dBox_reclen->blockSignals(false);
	}
}
void controlPanel::NWidgetPreTrg() {
	if (ui->dBox_pretrg->styleSheet().contains("color: red")) {
		ui->dBox_pretrg->blockSignals(true);
		ui->dBox_pretrg->setStyleSheet("color: black");
		ui->dBox_pretrg->setSpecialValueText("");
		ui->dBox_pretrg->setMinimum(ui->dBox_pretrg->minimum() + 1);
		ui->dBox_pretrg->blockSignals(false);
	}
}
void controlPanel::NWidgetTrgSrc() {
	if (ui->cBox_trigger_source->styleSheet().contains("color: red")) {
		ui->cBox_trigger_source->setStyleSheet("color: black");
	}
}
void controlPanel::NWidgetTrgMode() {
	if (ui->cBox_trigger_mode->styleSheet().contains("color: red")) {
		ui->cBox_trigger_mode->setStyleSheet("color: black");
	}
}

void controlPanel::updateGUIForOffline(bool enable) {
	if (enable) {
		ui->groupBox_3->setEnabled(false);
		ui->dBox_pretrg->setEnabled(false);
		ui->dBox_reclen->setEnabled(false);
		ui->label_pretrg->setEnabled(false);
		ui->label_reclen->setEnabled(false);
		ui->pbStartAcq->disconnect(SIGNAL(clicked()));
		connect(this->ui->pbStartAcq, SIGNAL(clicked()), mWaveDump, SLOT(startOfflineAcq()));
		ui->pbStopAcq->disconnect(SIGNAL(clicked()));
		connect(this->ui->pbStopAcq, SIGNAL(clicked()), mWaveDump, SLOT(stopOfflineAcq()));
		ui->pBStartManual->show();
	}
	else{
		ui->groupBox_3->setEnabled(true);
		ui->dBox_pretrg->setEnabled(true);
		ui->dBox_reclen->setEnabled(true);
		ui->label_pretrg->setEnabled(true);
		ui->label_reclen->setEnabled(true);
		ui->pBStartManual->hide();
		ui->pbStartAcq->disconnect(SIGNAL(clicked()));
		connect(this->ui->pbStartAcq, SIGNAL(clicked()), mWaveDump, SLOT(startAcquisition()));
		ui->pbStopAcq->disconnect(SIGNAL(clicked()));
		connect(this->ui->pbStopAcq, SIGNAL(clicked()), mWaveDump, SLOT(stopAcquisition()));
	}
}


double controlPanel::getPreTrg() {
  return ui->dBox_pretrg->value(); 
}