/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		WaveDump2.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "WaveDump2.h"

#include <iostream>
#include <QtGlobal>
#include <QtWidgets>
#include <QThread>
#include <QScreen>
#include <QGuiApplication>
#include <CAEN_FELib.h>

#include "manualControllerPanel.h"
#include "controlPanel.h"
#include "deviceManagerPanel.h"
#include "discoveryDevicesDialog.h"
#include "outputSettingsDialog.h"
#include "boardsListDialog.h"
#include "Dx740V2ConfigPanel.h"
#include "Dx740V1ConfigPanel.h"
#include "SystemSettings.h"
#include "CAENFELibException.h"
#include "hash.hpp"
#include "CAENDig2Device.h"
#include "CAENDig1Device.h"
#include "CAENDevice.h"
#include "infoPanel.h"

#include "about.h"
#include <cstring>
#include <locale>
#include <clocale>
#include <iostream>

WaveDump2::WaveDump2(QWidget *parent)
	: QMainWindow(parent)
{
	mCenterWidget = new QWidget;
	mCenterWidget->setStyleSheet(" background-image: url(:/CAEN-logo.png); background-repeat: no-repeat; background-position: center;");
	setCentralWidget(mCenterWidget);
	//{
	//	QTemporaryFile tmpFile;
	//	tmpFile.open();
	//	tmpFile.setAutoRemove(true);
	//	mTmpConfigFileName = tmpFile.fileName();
	//}
	mConnected = 0;
	mCouldSave = false;
	mNumberOfDevice = 0;
	mCoincidence_win = DEFAULT_COINCWINDOW;
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
	mDataSavedEnabled.store(0);
#else
	mDataSavedEnabled.storeRelaxed(0);
#endif
	this->setWindowIcon(QIcon(":WAVEDump2.png"));
	qRegisterMetaType<QVector<QVector<double>>>();
	qRegisterMetaType<QVector<int>>();
	qRegisterMetaType<QVector<double>>();
	qRegisterMetaType<QVector<bool>>();
	this->setWindowTitle("WaveDump2");
	this->setWindowFlags(windowFlags() | Qt::CustomizeWindowHint |
		Qt::WindowMinimizeButtonHint |
		Qt::WindowMaximizeButtonHint |
		Qt::WindowCloseButtonHint);
	QScreen* screen = QGuiApplication::primaryScreen();
	QRect  screenGeometry = screen->geometry();
	int height = screenGeometry.height();
	int width = screenGeometry.width();
	
	//mConfigFile = new QSettings(mTmpConfigFileName, QSettings::IniFormat, this);

	mConfigFileName = QDir::homePath() + DIRSEP + WAVEDUMP2_DIR + DIRSEP + DEFAUL_CONFIG_FILENAME;
	if (!QDir(QDir::homePath() + DIRSEP + WAVEDUMP2_DIR).exists())
		QDir().mkdir(QDir::homePath() + DIRSEP + WAVEDUMP2_DIR);
	QFile f(mConfigFileName);
	if (f.exists())
		f.remove();

	mConfigFile = new QSettings(mConfigFileName, QSettings::IniFormat, this);

	width *= 0.9; // 90% of the screen size
	height *= 0.9; // 80% of the screen size
	QSize newSize(width, height);
	int widthPlot = width *= 0.8; // TODO prossible bug...
	setGeometry(
		QStyle::alignedRect(
			Qt::LeftToRight,
			Qt::AlignCenter,
			newSize,
			screenGeometry
		)
	);

	QGuiApplication::setAttribute(Qt::AA_EnableHighDpiScaling);
	//QGuiApplication::setHighDpiScaleFactorRoundingPolicy(Qt::HighDpiScaleFactorRoundingPolicy::PassThrough);//
	
	mMainToolBar = new QToolBar(this);
	mMainStatusBar = new QStatusBar(this);
	mMainStatusBarText = new QLabel(this);
	mMainStatusBarText->setFixedWidth(110);
	mMainStatusBarText->setText("<html> <b> Disconnected </b></html>");
	mMainProgressBar = new QProgressBar(mMainStatusBar);
	mMainProgressBar->setTextVisible(false);
	mMainProgressBar->setFixedWidth(70);
	mMainProgressBar->setMaximum(100);
	mMainProgressBar->setMinimum(0);
	mMainProgressBar->setValue(0);
	mStatusLED = new QPushButton(this);
	mStatusLED->setEnabled(false);
	mStatusLED->setText(" No Errors ");
	mStatusLED->setStyleSheet("border-radius:6px; font: bold; color: black; background-color: rgb(48, 243, 18)"); //green
	mResetButton = new QPushButton(this);
	mResetButton->setEnabled(true);
	mResetButton->setText(" RESET ");
	connect(mResetButton, SIGNAL(clicked()), this, SLOT(Reset()));

	mMainStatusBar->addPermanentWidget(new QLabel("<html> <b> Status:&nbsp;&nbsp;</b></html>"));
	mMainStatusBar->addPermanentWidget(mMainStatusBarText);
	mMainStatusBar->addPermanentWidget(mStatusLED);
	mMainStatusBar->addPermanentWidget(mMainProgressBar);
	mMainStatusBar->addPermanentWidget(mResetButton);

	mManualControllerPan = new manualControllerPanel(this);
	mDeviceManagerPan = new deviceManagerPanel(this);
	mControlPan = new controlPanel(this);
	mLogPan = new logPanel(this);
	mPlotPan = new plotPanel(this, PLOT_TRACES);
	mStatsPan = new statisticsPanel(this);

	mStatistics = new Statistics();
	mStatistics->init(MAX_NDEV);

	Statistics_Timer = new QTimer(this);
	QObject::connect(Statistics_Timer, SIGNAL(timeout()), this, SLOT(UpdateStatsAndStatus()));

	Check_connection_Timer = new QTimer(this);
	QObject::connect(Check_connection_Timer, SIGNAL(timeout()), this, SLOT(CheckConnections()));

	setCorner(Qt::TopLeftCorner, Qt::LeftDockWidgetArea);
	setCorner(Qt::BottomLeftCorner, Qt::LeftDockWidgetArea);

	connect(mPlotPan, SIGNAL(tracesChanged(QVector<QString>, QVector<int>)), this, SLOT(tracesChanged(QVector<QString>, QVector<int>)));
	connect(mPlotPan, SIGNAL(NewCursorPositions(QVector<double>)), this, SLOT(showCursorsPositions(QVector<double>)));


	this->setStatusBar(mMainStatusBar);
	this->addToolBar(mMainToolBar);
	createDockWindows(widthPlot);
	populateToolBar();
	populateMenu();
	mManualControllerDock->hide();
	mControlPan->disableAll(false);
	mControlPan->setVisible(false);
	resizeDocks({ mLogDock }, { 10 }, Qt::Vertical);
	resizeDocks({ mStatsDock }, { 10 }, Qt::Vertical);
	resizeDocks({ mDeviceManagerDock }, { 200 }, Qt::Horizontal);
	this->setOutputSettings(QDir::homePath(), "Acq", "SINGLE", "YES", "ASCII", "NO", 0, false);
    connect(mDeviceManagerPan, SIGNAL(showManualController(std::uint64_t, const QString&, const QString&, bool)), this, SLOT(showManualController(std::uint64_t, const QString&, const QString&, bool)));

	QAction *confMenu = new QAction("Setup Configuration", this);
	confMenu->setStatusTip(tr("Configure the setup"));
	confMenu->setToolTip(tr("Configure the setup"));
	connect(confMenu, SIGNAL(triggered()), this, SLOT(configureSetup()));
	mConfiguration->addAction(confMenu);

	fillCommonDefValues();	

	CheckDefaultConfigToLoad();
}


void WaveDump2::paintEvent(QPaintEvent *) {
}

void WaveDump2::createDockWindows(int widthPlot)
{

	QDockWidget *dock = new QDockWidget("<html><b>Control</b></html>", this);
	dock->setAllowedAreas(Qt::TopDockWidgetArea);
	dock->setTitleBarWidget(new QWidget());
	//dock->setMaximumHeight(95);
	dock->setMinimumHeight(85);
	dock->setWidget(this->mControlPan);
	addDockWidget(Qt::TopDockWidgetArea, dock);

	mDeviceManagerDock = new CAENDockWidget("Device Manager", this);
	mDeviceManagerDock->setStyleSheet("QDockWidget { font: bold }");
	mDeviceManagerDock->setAllowedAreas(Qt::LeftDockWidgetArea);
	mDeviceManagerDock->setWidget(this->mDeviceManagerPan);
	mDeviceManagerDock->setMinimumWidth(235);
	mDeviceManagerDock->hide();
	addDockWidget(Qt::LeftDockWidgetArea, mDeviceManagerDock);
	connect(mDeviceManagerDock, SIGNAL(dockClosed(QCloseEvent *)), this, SLOT(browseCloseEvent(QCloseEvent *)));

	mPlotDock = new QDockWidget(tr("Plot"), this);
	mPlotDock->setStyleSheet("QDockWidget { font: bold; }");
	mPlotDock->setAllowedAreas(Qt::RightDockWidgetArea);
	mPlotDock->setFeatures(dock->features() & ~QDockWidget::DockWidgetClosable & ~QDockWidget::DockWidgetFloatable);
	//mPlotDock->setMinimumWidth(widthPlot);
	mPlotDock->setWidget(this->mPlotPan);
	addDockWidget(Qt::RightDockWidgetArea, mPlotDock);


	mLogDock = new CAENDockWidget(tr("Log"), this);
	mLogDock->setStyleSheet("QDockWidget { font: bold }");
	mLogDock->setAllowedAreas(Qt::RightDockWidgetArea);
	mLogDock->setMinimumHeight(30);
	mLogDock->setWidget(this->mLogPan);
	addDockWidget(Qt::RightDockWidgetArea, mLogDock);
	connect(mLogDock, SIGNAL(dockClosed(QCloseEvent *)), this, SLOT(logCloseEvent(QCloseEvent *)));

	mManualControllerDock = new CAENDockWidget("Manual Controller", this);
	mManualControllerDock->setAllowedAreas(Qt::RightDockWidgetArea);
	mManualControllerDock->setStyleSheet("QDockWidget { font: bold }");
	mManualControllerDock->setFeatures(dock->features() & ~QDockWidget::DockWidgetFloatable);
	mManualControllerDock->setMaximumHeight(80);
	mManualControllerDock->setMinimumHeight(80);
	mManualControllerDock->setWidget(this->mManualControllerPan);
	addDockWidget(Qt::RightDockWidgetArea, mManualControllerDock);
	connect(mManualControllerDock, SIGNAL(dockClosed(QCloseEvent *)), this, SLOT(manualControllerCloseEvent(QCloseEvent *)));

	mStatsDock = new CAENDockWidget(tr("Statistics"), this);
	mStatsDock->setAllowedAreas(Qt::RightDockWidgetArea);
	//mStatsDock->setMinimumWidth(335);
	mStatsDock->setWidget(this->mStatsPan);
	addDockWidget(Qt::RightDockWidgetArea, mStatsDock);
	connect(mStatsDock, SIGNAL(dockClosed(QCloseEvent *)), this, SLOT(statsCloseEvent(QCloseEvent *)));

	mPlotDock->setVisible(false);
	mLogDock->setVisible(false);
	mStatsDock->setVisible(false);
	mDeviceManagerDock->setVisible(false);
}

void WaveDump2::logCloseEvent(QCloseEvent *e) {
	Q_UNUSED(e);
	this->mLogAct->setChecked(false);
}

void WaveDump2::manualControllerCloseEvent(QCloseEvent *e) {
	Q_UNUSED(e);
	this->mManualControllerAct->setChecked(false);
}

void WaveDump2::browseCloseEvent(QCloseEvent *e) {
	Q_UNUSED(e);
	this->mBrowseAct->setChecked(false);
}

void WaveDump2::statsCloseEvent(QCloseEvent *e) {
	Q_UNUSED(e);
	this->mStatAct->setChecked(false);
}

void WaveDump2::increaseProgressBar() {
	int v = mMainProgressBar->value();
	if (v >= mMainProgressBar->maximum() - 1)
		v = 0;
	else
		v += 5;
	mMainProgressBar->setValue(v);
}

void WaveDump2::InitConfigFile(int dev_type) {
	QString name;
	QHash<QString, QString>::iterator item;
	mConfigFile->beginGroup(name);
	mConfigFile->endGroup();	
	switch (dev_type) {
	case CAEN_DIG1:
		name = "COMMON1";
		mConfigFile->beginGroup(name + "_GLOBAL_SETTINGS");
		for (item = CommonDevValuesmap1.begin(); item != CommonDevValuesmap1.end(); ++item) {
			mConfigFile->setValue(item.key(), item.value());
		}
		mConfigFile->endGroup();
		for (int ch = 0; ch < MAX_CHANNELS; ch++) {
			mConfigFile->beginGroup(name + QString("_CH%1").arg(ch));
			for (item = CommonChValuesmap1[ch].begin(); item != CommonChValuesmap1[ch].end(); ++item) {
				mConfigFile->setValue(item.key(), item.value());
			}
			mConfigFile->endGroup();
		}
		for (int g = 0; g < MAX_GROUPS; g++) {
			mConfigFile->beginGroup(name + QString("_GROUP%1").arg(g));
			for (item = CommonGroupValuesmap1[g].begin(); item != CommonGroupValuesmap1[g].end(); ++item) {
				mConfigFile->setValue(item.key(), item.value());
			}
			mConfigFile->endGroup();
		}
		break;
	case CAEN_DIG2:	
		name = "COMMON";
		mConfigFile->beginGroup(name + "_GLOBAL_SETTINGS");
		for (item = CommonDevValuesmap.begin(); item != CommonDevValuesmap.end(); ++item) {
			mConfigFile->setValue(item.key(), item.value());
		}
		mConfigFile->endGroup();
		for (int ch = 0; ch < MAX_CHANNELS; ch++) {
			mConfigFile->beginGroup(name + QString("_CH%1").arg(ch));
			for (item = CommonChValuesmap[ch].begin(); item != CommonChValuesmap[ch].end(); ++item) {
				mConfigFile->setValue(item.key(), item.value());
			}
			mConfigFile->endGroup();
		}
		for (int g = 0; g < MAX_VGA_GROUPS; g++) {
			mConfigFile->beginGroup(name + QString("_VGA%1").arg(g));
			for (item = CommonGroupValuesmap[g].begin(); item != CommonGroupValuesmap[g].end(); ++item) {
				mConfigFile->setValue(item.key(), item.value());
			}
			mConfigFile->endGroup();
		}
		break;

	}
}

void WaveDump2::fillCommonDefValues() {
	//default values for Dig2 main parameters
	CommonDevValuesmap.insert("/par/RecordLengthS", "1000");
	CommonDevValuesmap.insert("/par/StartSource", "SWCmd");
	CommonDevValuesmap.insert("/par/AcqTriggerSource", "SWTrg");
	CommonDevValuesmap.insert("/par/TstampResetSource", "Start");
	CommonDevValuesmap.insert("/par/PreTriggerS", "125");
	CommonDevValuesmap.insert("/par/TrgOutMode", "Disabled");
	CommonDevValuesmap.insert("/par/GPIOMode", "Disabled");
	CommonDevValuesmap.insert("/par/VetoSource", "Disabled");
	CommonDevValuesmap.insert("/par/VetoWidth", "0");
	CommonDevValuesmap.insert("/par/VetoPolarity", "ActiveHigh");
	CommonDevValuesmap.insert("/par/IOLevel", "NIM");
	CommonDevValuesmap.insert("/par/ITLAMask", "0");
	CommonDevValuesmap.insert("/par/ITLBMask", "0");
	CommonDevValuesmap.insert("/par/EnClockOutFP", "True");
	CommonDevValuesmap.insert("/par/RunDelay", "0");
	CommonDevValuesmap.insert("/par/DecimationFactor", "1");
	CommonDevValuesmap.insert("/par/EnAutoDisarmAcq", "True");

	for (int ch = 0; ch < MAX_CHANNELS; ch++) {
		CommonChValuesmap[ch].insert(QString("/ch/%1/par/ChEnable").arg(ch), "True");
		CommonChValuesmap[ch].insert(QString("/ch/%1/par/SelfTriggerEdge").arg(ch), "RISE");
		CommonChValuesmap[ch].insert(QString("/ch/%1/par/DCOffset").arg(ch), "50");
		CommonChValuesmap[ch].insert(QString("/ch/%1/par/TriggerThrMode").arg(ch), "Relative");
		CommonChValuesmap[ch].insert(QString("/ch/%1/par/TriggerThr").arg(ch), "3277");
		CommonChValuesmap[ch].insert(QString("/ch/%1/par/ChGain").arg(ch), "0");

	}
	for (int g = 0; g < MAX_VGA_GROUPS; g++) {
		CommonGroupValuesmap[g].insert(QString("/vga/%1/par/VgaGain").arg(g), "0");
	}

	//default values for Dig1 main parameters
	CommonDevValuesmap1.insert("/par/RecLen", "8000");
	CommonDevValuesmap1.insert("/par/StartMode", "Start_Mode_Sw");
	CommonDevValuesmap1.insert("/par/Trg_Sw_Enable", "True");
	CommonDevValuesmap1.insert("/par/Trg_Ext_Enable", "False");
	CommonDevValuesmap1.insert("/par/PostTrg", "7000");
	CommonDevValuesmap1.insert("/par/IOLevel", "FPIOType_NIM");
	CommonDevValuesmap1.insert("/par/Start_Delay", "0");
	CommonDevValuesmap1.insert("/par/Self_Trigger_Edge", "RISE");
	CommonDevValuesmap1.insert("/par/Trg_Sw_Out_Propagate", "False");
	CommonDevValuesmap1.insert("/par/Trg_Ext_Out_Propagate", "False");
	CommonDevValuesmap1.insert("/par/Out_Selection", "OUT_PROPAGATION_TRIGGER");
	CommonDevValuesmap1.insert("/par/Decimation_Factor", "DECIM_FACTOR_1");

	for (int ch = 0; ch < MAX_CHANNELS; ch++) {
		CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch), "False");
		CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Enabled").arg(ch), "True");
		CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_DCOffset").arg(ch), "50");
		CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Threshold").arg(ch), "1000");
		//CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_InDyn").arg(ch), "INDYN_2_0_VPP");//common value not set
	}
	for (int g = 0; g < MAX_GROUPS; g++) {
		CommonGroupValuesmap1[g].insert(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), "False");
		CommonGroupValuesmap1[g].insert(QString("/group/%1/par/Gr_Enabled").arg(g), "True");
		CommonGroupValuesmap1[g].insert(QString("/group/%1/par/Gr_DCOffset").arg(g), "50");
		CommonGroupValuesmap1[g].insert(QString("/group/%1/par/Gr_Threshold").arg(g), "3277");
	}
		
}

void WaveDump2::CheckConnectedDevStatus() {
	CAENDevice* dev = mConnectedDevices.at(mConnectedDevices.size() - 1);
	if (dev->ErrorsReported)
		dev->ErrorsReported = false;
		CheckConnections();
	}

int WaveDump2::getDeviceTypeCode(QString type) {
	if (type == "ETHERNET")
		return CTYPE_ETH_DIG2;
	else if (type == "USB")
		return CTYPE_USB_DIG2;
	else if (type == "USB1")
		return CTYPE_USB_DIG1;
	else if (type == "USBA4818")
		return CTYPE_USBA4818_DIG1;
	else if (type == "USBV4718")
		return CTYPE_USBV4718_DIG1;
	else if (type == "OPTILINK")
		return CTYPE_OPTILINK_DIG1;
	else if (type == "ETHERNETV4718")
		return CTYPE_ETHV4718_DIG1;
	else
		return -1;
}

std::uint64_t WaveDump2::openDeviceExec(int type, const QString& address, const QString& name, bool updateConfig) {
	QString connectionStr;
	QString typeString;
	uint64_t DevHandle;
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices[i]->getName() == name) {
			QMessageBox::critical(this, tr("Open Device"),
				tr("A device with Name: ") + name + tr(" is already connected\n"));
			return 0xFFFFFFFFFFFFFFFF;
		}
	}
	if (address.isEmpty()) {
		QMessageBox::critical(this, tr("Open Device"),
			tr("Cannot open the device! Connection parameter is missing!"));
		return 0xFFFFFFFFFFFFFFFF;
	}

	switch (type) {
		case CTYPE_ETH_DIG2: //eth dig2
			typeString = "ETHERNET";
			connectionStr = "dig2://" + address;
			break;
		case CTYPE_USB_DIG2: //USB dig2
			typeString = "USB";
			connectionStr = "dig2://usb:" + address;
			break;
		case CTYPE_USB_DIG1:
			typeString = "USB1";
			connectionStr = "dig1://" + address;
			break;
		case CTYPE_USBA4818_DIG1:
			typeString = "USBA4818";
			connectionStr = "dig1://" + address;
			break;
		case CTYPE_USBV4718_DIG1:
			typeString = "USBV4718";
			connectionStr = "dig1://" + address;
			break;
		case CTYPE_OPTILINK_DIG1:
			typeString = "OPTILINK";
			connectionStr = "dig1://" + address;
			break;
		case CTYPE_ETHV4718_DIG1:
			typeString = "ETHERNETV4718";
			connectionStr = "dig1://" + address;
			break;

	default:
		connectionStr = nullptr;
	}
	/*LANGID test1 = SetThreadUILanguage(MAKELANGID(LANG_ENGLISH, SUBLANG_ENGLISH_UK));
	LCID i1 = GetThreadLocale();
	bool test = SetThreadLocale(MAKELANGID(LANG_ENGLISH, SUBLANG_ENGLISH_UK));

	LCID i = GetThreadLocale();*/

	int ret = CAEN_FELib_Open(qPrintable(connectionStr), &DevHandle); //a new device is opened, it becomes the active device-->mDevHandle is updated
	switch (ret) {
	case ::CAEN_FELib_Success: {
		break;
	}
	case ::CAEN_FELib_BadLibraryVersion: {
		char error[1024];
		CAEN_FELib_GetLastError(error);
		QMessageBox::warning(this, tr("Open Device"),
			tr("Open successful, but you are using an old version of CAEN Dig2. Possible undefined behaviors.\n Please update WaveDump2.0.\n Error at ") + address + QString(" Error code %1: ").arg(ret) + QString(error));
		break;
	}
	default: {
		char error[1024];
		CAEN_FELib_GetLastError(error);
		if (typeString == "ETHERNET") {
			QMessageBox::critical(this, tr("Open Device"),
				tr("Cannot open the device\n at ") + address + QString(" Error code %1").arg(ret) + "Check that the device is powered on with the link cable connected and that the whole ethernet chain (SFP+ Ethernet card + Fiber cable + FSP+ transceiver) is 1Gb or 10Gb compliant depending on the firmware type.\n" + QString(error));
		}
		else {
			if (ret == CAEN_FELib_DeviceLibraryNotAvailable) {
				QMessageBox::critical(this, tr("Open Device"),
					tr("Cannot open the device\n at ") + address + ". CAEN_Dig1 library could not be found. Please install the library and try again.");
			}
			else {
				QMessageBox::critical(this, tr("Open Device"),
					tr("Cannot open the device\n at ") + address + QString(" Error code %1").arg(ret) + "Check that the device is powered on with the link cable connected.\n" + QString(error));
			}
		}
		CAEN_FELib_Close(DevHandle);
		return 0xFFFFFFFFFFFFFFFF;
	}
	}

	mConnectionAbortCmd = false;


	char fw_type[80];
	ret = CAEN_FELib_GetValue(DevHandle, "/par/FwType", fw_type);
	if (!QString(fw_type).contains("Scope", Qt::CaseInsensitive)) {
		QMessageBox::critical(this, tr("FW Error"),
			tr("WaveDump2 can only run with Scope firmware. Device cannot be connected."));
		CAEN_FELib_Close(DevHandle);
		return 0xFFFFFFFFFFFFFFFF;
	}

	QString actual_name = name.isEmpty() ? connectionStr : name;
	if (actual_name.contains("dig1")) {
		char val[80];
		ret = CAEN_FELib_GetValue(DevHandle, "/par/ModelName", val);
		if (ret != CAEN_FELib_Success) {
			QMessageBox::critical(this, tr("Error"),
				tr("Error loading device info."));
			CAEN_FELib_Close(DevHandle);
			return 0xFFFFFFFFFFFFFFFF;
		}
		actual_name = "dig1-" + QString(val) + "-";
		ret = CAEN_FELib_GetValue(DevHandle, "/par/SerialNum", val);
		if (ret != CAEN_FELib_Success) {
			QMessageBox::critical(this, tr("Error"),
				tr("Error loading device info."));
			CAEN_FELib_Close(DevHandle);
			return 0xFFFFFFFFFFFFFFFF;
		}
		actual_name.append(QString(val));
		
	}
	mLoadDeviceManager = 0;
	if (mLoadDeviceManager == -1) {
		QMessageBox::StandardButton resBtn = QMessageBox::question(this, "WaveDump2",
			tr("Do you want to load the Device Manager?\n"),
			QMessageBox::No | QMessageBox::Yes,
			QMessageBox::Yes);
		if (resBtn != QMessageBox::Yes) {
			mLoadDeviceManager = 0;
			mBrowseAct->setChecked(false);
			browseToggle();
		}
		else {
			mLoadDeviceManager = 1;
		}
	}
	bool dm_result = true;
	if (mLoadDeviceManager == 1) {
		mAsyncConn = QtConcurrent::run([this, actual_name, DevHandle] { return createDeviceTree(actual_name, DevHandle); });
		QThread::msleep(1);
		mMainStatusBarText->setText("<html><B>Collecting data</b></html>");
		QMessageBox msgBox(this);
		msgBox.setText("<html><B>Reading information from the devices <br>           ... please wait ...</html>");
		msgBox.setStandardButtons({});
		msgBox.show();
		while (mAsyncConn.isRunning()) {
			QThread::msleep(100);
			this->increaseProgressBar();
			QCoreApplication::processEvents();
		}
		msgBox.close();
		mMainProgressBar->setValue(0);
		dm_result = mAsyncConn.result();
		mBrowseAct->setChecked(true);
		browseToggle();
	}
	if (dm_result) {
		mManualControllerPan->setEnabled(true);
		mControlPan->disableAll(true);
		mConnected++;
		OfflineMode = false;
		if (updateConfig) {
			mConfigFile->beginGroup("DEVICES");
			mConfigFile->setValue("DEV_" + QString::number(mNumberOfDevice) + "_NAME", actual_name);
			mConfigFile->setValue("DEV_" + QString::number(mNumberOfDevice) + "_TYPE", typeString);
			mConfigFile->setValue("DEV_" + QString::number(mNumberOfDevice) + "_ARG", address);
			mConfigFile->endGroup();
			QString n = actual_name;
			if (n.contains("://"))
				n.replace("://", ".");
			mConfigFile->beginGroup(n + "_GLOBAL_SETTINGS");
			mConfigFile->endGroup();
		}

		try {
			CAENDevice* tmp;
			if (type < 100) {
				CAENDig2Device* dig2 = new CAENDig2Device(actual_name, typeString, address, DevHandle, false);
				tmp = dig2;
			}
			else {
				CAENDig1Device* dig1 = new CAENDig1Device(actual_name, typeString, address, DevHandle, false);
				tmp = dig1;
			}
			
			tmp->ClearData();
			
			if (tmp->ConnectionOpen) {
				tmp->Reset();
				QThread::msleep(500);				
				if(tmp->DeviceClass == CAEN_DIG2)
					tmp->ApplyDefaultConfig(CommonDevValuesmap, CommonChValuesmap, CommonGroupValuesmap);//set initial default  settings
				else if (tmp->DeviceClass == CAEN_DIG1)
					tmp->ApplyDefaultConfig(CommonDevValuesmap1, CommonChValuesmap1, CommonGroupValuesmap1);//set initial default  settings
			}
			InitConfigFile(tmp->DeviceClass);
			int index = mConnectedDevices.size();
			tmp->setIndex(index);
			
			mConnectedDevices += tmp;
			if (mDevicesReadoutThreads.size() < mConnectedDevices.size())
				mDevicesReadoutThreads.append(nullptr);
			mStatsPan->fillDevices(mConnectedDevices);
			mIgnoreEvent = true;
			mControlPan->updateStartableDevices(1, tmp->getName());
			mIgnoreEvent = false;
			QVector<QString> channels;
			for (int i = 0; i < tmp->getNumCh(); i++) {//fill plot traces with channels of the first connected device
				channels += QString("CH%1").arg(QString::number(i));
			}
			if (mActiveDevice == nullptr) {//the first connected device becomes the active device
				setActiveDevice(tmp);
			}
			if (tmp->ConnectionOpen)
				mPlotPan->updatePalette(1, tmp->getName(), channels);

			if (QString(fw_type).contains("Open", Qt::CaseInsensitive)) {
				tmp->setScopeChannel("User");
				QMessageBox::information(this, tr("Configure device"), "Custom wave channel enabled by default for Open Scope firmware.\n The standard wave channel can be re-enabled in the device advanced settings.", QMessageBox::Ok, 0);
			}

		}
		catch (CAENFELibException& exc) {
			addLog(exc.ErrorMessage(), false);
			QMessageBox::critical(this, tr("Configure Device"),
				tr("Cannot configure the device\n at ") + address + QString(" : ") + exc.ErrorMessage());
			return 0xFFFFFFFFFFFFFFFF;
		}

		mNumberOfDevice++;
		SaveLastConnectionData(type, address);

		//check if a previous config file is present (only if the user has not selected a default file to load)///
		if (!mAutoLoadFile->isChecked() && updateConfig) {
			CheckLoadLastConfig(mConnectedDevices.at(0)->getSN());
		}

		if (mNumberOfDevice == 1) {
			handleConfFileModified();
			mOpenSettingsAct->setEnabled(true);
			mManualControllerAct->setEnabled(true);
			mControlPan->updateCursorsTrace(mPlotPan->getTraceColor(mPlotPan->ActiveTrace), QString("Trace %1").arg(mPlotPan->ActiveTrace));
			mControlPan->setVisible(true);
			mControlPan->setEnabled(true);
			mPlotPan->setEnabled(true);
			if (mActiveDevice->DeviceClass == CAEN_DIG1) {
				mControlPan->YUnitChangeCmd(UOM_SAMPLE);
				mControlPan->XUnitChangeCmd(UOM_SAMPLE);
			}
			mInfoMenuAct->setEnabled(true);
			mErrorsMenuAct->setEnabled(true);
			mControlPan->updateGUIForOffline(false);
			mPlotPan->enableOfflineMode(false);
			QVector<double>sampl_to_V;
			for(int ch=0;ch<mConnectedDevices.at(0)->getNumCh();ch++)
				sampl_to_V.append(mConnectedDevices.at(0)->getSample_to_V(ch));
			mPlotPan->updateBaselines(mConnectedDevices.at(0)->getName(), mConnectedDevices.at(0)->getADCNBit(), sampl_to_V, mConnectedDevices.at(0)->ChBaselines);
			mPlotPan->updateGains(mConnectedDevices.at(0)->getName(), mConnectedDevices.at(0)->ChGain);
			mBrowseAct->setEnabled(true);
			mStatAct->setEnabled(true);
			mLogAct->setEnabled(true);
			mSaveOutputAct->setEnabled(true);
			mOutputConfigAct->setEnabled(true);
			mDisconnectMenuAct->setEnabled(true);
			mDisconnectAct->setEnabled(true);
			mConfiguration->setEnabled(true);
			mScriptMenu->setEnabled(true);
			mOutputMenu->setEnabled(true);
			mWindowMenu->setEnabled(true);
			mPlotDock->setVisible(true);
			mLogDock->setVisible(true);
			mSaveasSettingsAct->setEnabled(true);
			mResetButton->setEnabled(true);
			mCenterWidget->setMaximumWidth(1);
			if (!this->mConnectAct->isChecked())
				this->mConnectAct->setChecked(true);
			if (!mActiveDevice->ErrorsReported)
				Check_connection_Timer->start(3000);
			mLoadRawMenuAct->setEnabled(false);	
			PaletteChMask(mConnectedDevices.at(0)->getName(), mConnectedDevices.at(0)->getChEnableMask());
		}


		CAENQAction* confMenu = new CAENQAction(actual_name, this);
		confMenu->setStatusTip(tr("Configure ") + actual_name);
		confMenu->setToolTip(tr("Configure ") + actual_name);
		connect(confMenu, SIGNAL(clicked(QString)), this, SLOT(configureDevice(QString)));
		mConfiguration->addAction(confMenu);
		mHashConfig.insert(actual_name, DevHandle);

		mLogPan->add2Log("Connection with " + actual_name + " via:" + typeString + " :" + address, true);
		if (mLoadDeviceManager != 0) {
			mLogPan->add2Log("Adding " + actual_name + " to Device Manager", true);
		}
		mManualControllerPan->addDevice(actual_name, DevHandle);

		return DevHandle;
	}
	else {
		if (mConnectedDevices.size() == 0) {
			mManualControllerPan->setEnabled(false);
			mControlPan->setEnabled(false);
		}
		if (mLoadDeviceManager == 1) mDeviceManagerPan->removeDevice(DevHandle);
		mLogPan->add2Log("Connection with " + actual_name + " via:" + typeString + " :" + address, false);
		CAEN_FELib_Close(DevHandle);
		return 0xFFFFFFFFFFFFFFFF;;
	}
}

void WaveDump2::SaveLastConnectionData(int type, QString conn_data) {
	QString filename = QDir::homePath() + DIRSEP + QString(".cWInfo");

	QFile connection_file(filename);
	connection_file.open(QIODevice::WriteOnly | QIODevice::Truncate);
	QTextStream out(&connection_file);

	out << type;
	out << ENDL;
	out << conn_data;
	out << ENDL;

	connection_file.close();
}

uint64_t WaveDump2::ReConnect() {
	QString conn_data, type;
	QString filename = QDir::homePath() + DIRSEP + QString(".cWInfo");
	QFile connection_file(filename);
	if (connection_file.exists()) {
		connection_file.open(QIODevice::ReadOnly);
		QTextStream in(&connection_file);
		type = in.readLine();
		conn_data = in.readLine();
		connection_file.close();
	}
	else {
		QMessageBox::critical(this,tr("Reconnect"),"No valid connection data found.");
		return 0xFFFFFFFFFFFFFFFF;
	}

	return openDeviceExec(type.toInt(), conn_data, "", true);
}

void WaveDump2::setActiveDevice(CAENDevice *dev) {

	mActiveDevice = dev;

	setCurrentDeviceName(dev->getName());

	mSample_to_s = dev->getSample_to_S();

	mVRangeS = qPow(2, dev->getADCNBit()) - 1;
	if(dev->ConnectionOpen)
		mSample_to_V = dev->getSample_to_V(mPlotPan->getActiveTraceCh());
	updatePlotYAxisRange();
	if (dev->ConnectionOpen)
		updateActiveDevInfo(dev);
	mControlPan->CheckTrgSourceOptions(dev->DeviceClass, dev->getTrgSource());
}

void WaveDump2::updateActiveDevInfo(CAENDevice *dev) { 
	if (dev == NULL)
		return;
	try {
		
		mRecLenS = dev->getReclen_valueS();
		mPretrgS = dev->getPreTrg_valueS();
		mDecimation = dev->getDecimation_value();
		dev->updateBaselines();
		dev->updateGains();

		double factor_dev;//the plot x axis range extends always to the max reclen(max_reclex*max_decimation)
		switch (mUoMX) {
		case UOM_SAMPLE:
			mControlPan->updateReclenWidget(MAX_SW_RECLEN_SAMPLES, dev->getReclen_minS(), dev->getReclen_incrS(), dev->getReclen_valueS());
			mPlotPan->updateXPosInfo(dev->getReclen_maxS(), dev->getReclen_minS(), dev->getReclen_incrS(), dev->getReclen_valueS(), 1.);
			if (dev->DeviceClass == CAEN_DIG1)
				factor_dev = 1. / dev->getDecimation_value();//show value in reclen samples (decimated)
			else
				factor_dev = 1.;
			mControlPan->updatePreTrgWidget(dev->getPreTrg_maxS()*factor_dev, dev->getPreTrg_minS() * factor_dev, dev->getPreTrg_incrS() * factor_dev, dev->getPreTrg_valueS() * factor_dev);
			if (dev->getName() == mPlotPan->getActiveTraceDev())
				mPlotPan->updatePreTrgInfo(dev->getPreTrg_valueS() * factor_dev);
			break;
		case UOM_PHYS_UNIT:			
			factor_dev = dev->getDecimation_value() / 1000.;
			mControlPan->updateReclenWidget(MAX_SW_RECLEN_SAMPLES * factor_dev, dev->getReclen_min() * factor_dev, dev->getReclen_incr() * factor_dev, dev->getReclen_value() * factor_dev);
			mPlotPan->updateXPosInfo(dev->getReclen_max(), dev->getReclen_min(), dev->getReclen_incr(), dev->getReclen_value(), factor_dev);
			if (dev->DeviceClass == CAEN_DIG1)
				factor_dev = 1. / 1000.;
			mControlPan->updatePreTrgWidget(dev->getPreTrg_max() * factor_dev, dev->getPreTrg_min() * factor_dev, dev->getPreTrg_incr() * factor_dev, dev->getPreTrg_value() * factor_dev);
			if (dev->getName() == mPlotPan->getActiveTraceDev())
				mPlotPan->updatePreTrgInfo(dev->getPreTrg_value() * factor_dev);
			break;
		}
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	updatePlotXAxisRange();
	CheckTracesEnableStatus();

	if (mPlotPan->getActiveTraceDev() == dev->getName()) {
		int ch = mPlotPan->getActiveTraceCh();
		mPlotPan->updateOffsetInfo(dev->getChOffset(ch));
		mPlotPan->updateTSampl(dev->getSample_to_S() * 1e9);
		mPlotPan->updateGains(dev->getName(), dev->ChGain);
		switch (mUoMY) {
		case UOM_SAMPLE:
			mPlotPan->updateThrInfo(dev->getThr_max(ch), dev->getThr_min(ch), dev->getThr_incr(ch), dev->getThr_value(ch), 1.);
			break;
		case UOM_PHYS_UNIT:
			if(dev->DeviceClass == CAEN_DIG2)
				mPlotPan->updateThrInfo(dev->getThr_max(ch), dev->getThr_min(ch), dev->getThr_incr(ch), dev->getThr_value(ch), mSample_to_V * 1e3);
			else {
				mPlotPan->updateThrInfo(dev->getThr_max(ch), dev->getThr_min(ch), dev->getThr_incr(ch), dev->getThr_value(ch), dev->getVRangeV(ch) * 1e3);
			}
			break;

		}
	}

	mControlPan->updateTrgSource(dev->getTrgSource());
	UpdateStatus();
}

void WaveDump2::UpdateStatus(){
	QString status = mActiveDevice->getAcqStatus();
	mMainStatusBarText->setText(status);
	if (mActiveDevice->ErrorsReported)
		return;	
	if (status.contains("Running",Qt::CaseInsensitive)) {
		mActiveDevice->resetTime();
		mControlPan->setStartStopStatus(1);
		mConfiguration->setEnabled(false); 
	}
	else {
		mControlPan->setStartStopStatus(0);
		mConfiguration->setEnabled(true);
		if (status.contains("Ready", Qt::CaseInsensitive) && mActiveDevice->AcqStarted)
			mActiveDevice->AcqStarted = false;
	}
}

void WaveDump2::UpdateStatusFast() {
	
	for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
		if (mConnectedDevices.at(dev)->ErrorsReported)
			continue;
		try {
			QString status = mConnectedDevices.at(dev)->getAcqStatus();
			mStatsPan->showDevStatus(dev, status);
			QString shown_status = mMainStatusBarText->text();
			if (shown_status != status) {
				mMainStatusBarText->setText(status);
				if (status != "Armed")
					UpdateStatus();
			}
		}
		catch (CAENFELibException& exc) {
			addLog(exc.ErrorMessage(), false);
		}
	}
}

void WaveDump2::UpdateStatsAndStatus() {
	UpdateStatusFast();
	if (stats_panel_open)
		mStatsPan->showStats();
}

void WaveDump2::ActiveDeviceChanged(QString new_dev) {
	if (mIgnoreEvent)
		return;
	if (mRecordAction)
		mActions->AddSWAction("ACTIVE_DEV_CHANGE", new_dev);
	if (new_dev == "ALL") {//check for different settings
		QString  trgsrc;
		double reclen, pretrg;
		int trgmode;
		bool reclen_ok = true, pretrg_ok=true, trgsrc_ok=true, trgmode_ok=true;
		for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
			if (dev == 0) {
				if (mUoMX == UOM_PHYS_UNIT) {
					reclen = (double)mConnectedDevices[dev]->getReclen_ns() * (double)mConnectedDevices[dev]->getDecimation_value() / 1000.;
					pretrg = (double)mConnectedDevices[dev]->getPreTrgCachedValue() * (double)mConnectedDevices[dev]->getDecimation_value() / 1000.;
				}
				else {
					reclen = (double)mConnectedDevices[dev]->getReclen_valueS();
					pretrg = (double)mConnectedDevices[dev]->getPreTrg_valueS();
				}
				trgsrc = mConnectedDevices[dev]->getTrgSource();
				trgmode = mConnectedDevices[dev]->getTrgMode();
			}
			else {
				if (mUoMX == UOM_PHYS_UNIT) {
					if ((double)mConnectedDevices[dev]->getReclen_ns() * (double)mConnectedDevices[dev]->getDecimation_value() / 1000. != reclen && reclen_ok)
						reclen_ok = false;
					if ((double)mConnectedDevices[dev]->getPreTrgCachedValue() * (double)mConnectedDevices[dev]->getDecimation_value() / 1000. != pretrg && pretrg_ok)
						pretrg_ok = false;
				}
				else{
					if ((double)mConnectedDevices[dev]->getReclen_valueS() != reclen && reclen_ok)
						reclen_ok = false;
					if ((double)mConnectedDevices[dev]->getPreTrg_valueS() != pretrg && pretrg_ok)
						pretrg_ok = false;
				}
				if (mConnectedDevices[dev]->getTrgSource() != trgsrc && trgsrc_ok) {
					if(trgsrc.contains("ITL") && !mConnectedDevices[dev]->getTrgSource().contains("ITL"))
						trgsrc_ok = false;
				}
				if (mConnectedDevices[dev]->getTrgMode() != trgmode && trgmode_ok)
					trgmode_ok = false;
			}
		}
		if (!reclen_ok)
			mControlPan->WWidgetReclen();
		if (!pretrg_ok)
			mControlPan->WWidgetPreTrg();
		if (!trgsrc_ok)
			mControlPan->WWidgetTrgSrc();
		if (!trgmode_ok)
			mControlPan->WWidgetTrgMode();
		updatePlotXAxisRange();

		if (OfflineMode)
			mControlPan->UpdateStartOptions();
		UpdateStatus();
		return;
	}
		
	int i = 0;
	for (i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices[i]->getName() == new_dev)
			break;
	}
	setActiveDevice(mConnectedDevices.at(i));

	UpdateStatus();
}


void WaveDump2::abortConnection() {
	mConnectionAbortCmd = true;
}

bool WaveDump2::createDeviceTree(const QString& addr, uint64_t rootHandle) {
	QString str = QString("/");
	CAENTreeWidgetItem *root = mDeviceManagerPan->addTreeRoot(addr, CAEN_FELib_NodeType_t::CAEN_FELib_FOLDER, rootHandle);
	return this->addNode(root, rootHandle, str);
}

bool WaveDump2::addNode(CAENTreeWidgetItem *father, uint64_t handle, const QString& qry) {
	const size_t handles_size = 1024;
	uint64_t handles[handles_size];
	bool flag = true;
	CAEN_FELib_NodeType_t type;
	int i;
	CAEN_FELib_ErrorCode ErrCode;
	static const QString slash("/");
	if (mConnectionAbortCmd)
		return false;
	int ret = CAEN_FELib_GetChildHandles(handle, "/", handles, handles_size);
	if (ret < 0) {
		//QMessageBox::critical(this, tr("Open Device"), tr("Error retrieving Device properties!"));
		qDebug() << "Error retrieving Device properties!\n";
		return false;
	}
	if (ret > handles_size) {
		//QMessageBox::critical(this, tr("Open Device"), tr("Error retrieving Device properties\n Low memory size!"));
		qDebug() << "Error retrieving Device properties!\n";
		return false;
	}
	if (ret == 0)
		return true;
	for (i = 0; i < ret; i++) {
		char name[32];
		ErrCode = (CAEN_FELib_ErrorCode)CAEN_FELib_GetNodeProperties(handles[i], "", name, &type);
		if (ErrCode != CAEN_FELib_Success) {
			//QMessageBox::critical(this, tr("Open Device"), tr("Error retrieving Device properties\n Missed Item Info!"));
			return false;
		}
		CAENTreeWidgetItem *treeItem = new CAENTreeWidgetItem(father, type);
		treeItem->setText(0, name);
		treeItem->setRootName(father->getRootName());
		treeItem->setType(static_cast<CAEN_FELib_NodeType_t>(type));
		treeItem->setHandle(handles[i]);
		QString tmp = QString(name);
		treeItem->setQry(qry + slash + tmp);
		if (type != CAEN_FELib_NodeType_t::CAEN_FELib_ATTRIBUTE)
			father->addChild(treeItem);

		switch (type) {
		case CAEN_FELib_NodeType_t::CAEN_FELib_PARAMETER:
		case CAEN_FELib_NodeType_t::CAEN_FELib_CHANNEL:
		case CAEN_FELib_NodeType_t::CAEN_FELib_COMMAND:
		case CAEN_FELib_NodeType_t::CAEN_FELib_ENDPOINT:
		case CAEN_FELib_NodeType_t::CAEN_FELib_ATTRIBUTE:
		case CAEN_FELib_NodeType_t::CAEN_FELib_VGA:
		case CAEN_FELib_NodeType_t::CAEN_FELib_LVDS:
			flag &= addNode(treeItem, handles[i], qry + ((qry.endsWith('/')) ? tmp : slash + tmp));
			break;
		case CAEN_FELib_NodeType_t::CAEN_FELib_FOLDER:
			treeItem->setText(0, tmp.toUpper());
			flag &= addNode(treeItem, handles[i], qry + ((qry.endsWith('/')) ? tmp : slash + tmp));
			break;
		default:
			break;
		}
	}
	return flag;
}

void WaveDump2::populateToolBar()
{
	const QIcon connectIcon = QIcon(":Connect.png");
	const QIcon disconnectIcon = QIcon(":Disconnect.png");
	const QIcon scanIcon = QIcon(":scan.png");
	const QIcon openIcon = QIcon(":openconfiguration.png");
	const QIcon saveIcon = QIcon(":saveconfiguration.png");
	const QIcon saveAsIcon = QIcon(":saveasconfiguration.png");
	const QIcon browseIcon = QIcon(":info.png");
	const QIcon exitIcon = QIcon(":exit.png");
	const QIcon logIcon = QIcon(":log.png");
	const QIcon manualControllerIcon = QIcon(":settings.png");
	const QIcon statIcon = QIcon(":statistics.png");
	const QIcon openSettingsIcon = QIcon(":opensettings.png");
	const QIcon saveAsSettingsIcon = QIcon(":saveassettings.png");

	mConnectAct = new QAction(connectIcon, "Connect to device", this);
	mConnectMenuAct = new QAction("Connect to device", this);
	mConnectAct->setStatusTip(tr("Connect to device"));
	mConnectAct->setToolTip("Connect to device");
	mConnectAct->setCheckable(false);

	mDisconnectAct = new QAction(disconnectIcon, "Disconnect device", this);
	mDisconnectMenuAct = new QAction("Disconnect device", this);
	mDisconnectAct->setStatusTip(tr("Disconnect device"));
	mDisconnectAct->setToolTip("Disconnect device");
	mDisconnectAct->setCheckable(false);

	mInfoMenuAct = new QAction("Show active device info", this);
	mInfoMenuAct->setEnabled(false);

	mErrorsMenuAct = new QAction("Show active device errors", this);
	mErrorsMenuAct->setEnabled(false);

	mScanAct = new QAction(scanIcon, "Discovery devices", this);
	mScanMenuAct = new QAction("Discovery devices", this);
	mScanAct->setStatusTip(tr("Discovery devices"));
	mScanAct->setToolTip(tr("Discovery devices"));

	mLoadAct = new QAction(openIcon, "Load", this);
	mLoadMenuAct = new QAction("Load", this);
	mLoadAct->setStatusTip(tr("Load configuration file"));
	mLoadAct->setToolTip(tr("Load configuration file"));

	mLoadRawMenuAct = new QAction("Load Raw Data", this);

	mOpenSettingsAct = new QAction(openSettingsIcon, "Load", this);
	mOpenSettingsMenuAct = new QAction("Load", this);
	mOpenSettingsAct->setStatusTip(tr("Load Settings file"));
	mOpenSettingsAct->setToolTip(tr("Load Settings file"));

	mSaveAct = new QAction(saveIcon, "Save", this);
	mSaveMenuAct = new QAction("Save", this);
	mSaveAct->setStatusTip(tr("Save configuration file"));
	mSaveAct->setToolTip(tr("Save configuration file"));

	mSaveasAct = new QAction(saveAsIcon, "Save As ...", this);
	mSaveasMenuAct = new QAction("Save As ...", this);
	mSaveasAct->setStatusTip(tr("Save configuration file As"));
	mSaveasAct->setToolTip(tr("Save configuration file As"));

	mExitAct = new QAction(exitIcon, "Exit", this);
	mExitMenuAct = new QAction("Exit", this);
	mExitAct->setStatusTip(tr("Exit WaveDump2"));
	mExitAct->setToolTip("Exit WaveDump2");

	mSaveOutputAct = new QAction("Save Data", this);
	mSaveOutputAct->setStatusTip(tr("Save Data to file"));
	mSaveOutputAct->setToolTip(tr("Save Data to file"));
	mSaveOutputAct->setCheckable(true);

	mSaveasSettingsAct = new QAction(saveAsSettingsIcon, nullptr, this);
	mSaveasSettingsAct->setStatusTip(tr("Start/Stop saving as Settings File"));
	mSaveasSettingsAct->setToolTip(tr("Start/Stop saving as Settings File"));
	mSaveasSettingsAct->setCheckable(true);
	mSaveasSettingsAct->setChecked(false);

	mOutputConfigAct = new QAction("Configure", this);
	mOutputConfigAct->setStatusTip(tr("Configure data saved"));
	mOutputConfigAct->setToolTip(tr("Configure data saved"));

	mBrowseAct = new QAction(browseIcon, "Show/Hide Device Manager", this);
	mBrowseAct->setStatusTip(tr("Show/Hide Device Manager"));
	mBrowseAct->setToolTip("Show/Hide Device Manager");
	mBrowseAct->setCheckable(true);
	mBrowseAct->setChecked(false);

	mLogAct = new QAction(logIcon, "Show/Hide Log window", this);
	mLogAct->setStatusTip(tr("Show/Hide Log window"));
	mLogAct->setToolTip("Show/Hide Log window");
	mLogAct->setCheckable(true);
	mLogAct->setChecked(true);

	mManualControllerAct = new QAction(manualControllerIcon, "Show/Hide Manual Controller", this);
	mManualControllerAct->setStatusTip(tr("Show/Hide Manual Controller"));
	mManualControllerAct->setToolTip("Show/Hide Manual Controller");
	mManualControllerAct->setCheckable(true);
	mManualControllerAct->setChecked(false);

	mStatAct = new QAction(statIcon, "Show / Hide Statistics", this);
	mStatAct->setStatusTip(tr("Show/Hide Statistics"));
	mStatAct->setToolTip("Show/Hide Statistics");
	mStatAct->setCheckable(true);
	mStatAct->setChecked(false);

	mAboutAct = new QAction("About WaveDump2", this);
	mAboutAct->setStatusTip(tr("Show About"));
	mAboutAct->setToolTip("Show About");
	mAboutAct->setCheckable(false);

	connect(mConnectAct, SIGNAL(triggered()), this, SLOT(connect2Device()));
	connect(mConnectMenuAct, SIGNAL(triggered()), this, SLOT(connect2Device()));

	connect(mInfoMenuAct, SIGNAL(triggered()), this, SLOT(DeviceInfo()));
	connect(mErrorsMenuAct, SIGNAL(triggered()), this, SLOT(DeviceErrors()));

	connect(mDisconnectAct, SIGNAL(triggered()), this, SLOT(disconnectDevice()));
	connect(mDisconnectMenuAct, SIGNAL(triggered()), this, SLOT(disconnectDevice()));
	
	connect(mScanAct, SIGNAL(triggered()), this, SLOT(scanDevices()));
	connect(mScanMenuAct, SIGNAL(triggered()), this, SLOT(scanDevices()));
	
	connect(mLoadAct, SIGNAL(triggered()), this, SLOT(load()));
	connect(mLoadMenuAct, SIGNAL(triggered()), this, SLOT(load()));

	connect(mLoadRawMenuAct, SIGNAL(triggered()), this, SLOT(LoadRawFromFile()));

	connect(mSaveAct, SIGNAL(triggered()), this, SLOT(save()));
	connect(mSaveMenuAct, SIGNAL(triggered()), this, SLOT(save()));

	connect(mSaveasAct, SIGNAL(triggered()), this, SLOT(saveAs()));
	connect(mSaveasMenuAct, SIGNAL(triggered()), this, SLOT(saveAs()));

	connect(mExitAct, &QAction::triggered, qApp, &QApplication::closeAllWindows);
	connect(mExitMenuAct, &QAction::triggered, qApp, &QApplication::closeAllWindows);

	connect(mSaveOutputAct, SIGNAL(triggered()), this, SLOT(saveOutput()));
	connect(mOutputConfigAct, SIGNAL(triggered()), this, SLOT(openOutputSettings()));

	connect(mBrowseAct, SIGNAL(triggered()), this, SLOT(browseToggle()));
	connect(mManualControllerAct, SIGNAL(triggered()), this, SLOT(manualToggle()));
	connect(mLogAct, SIGNAL(triggered()), this, SLOT(logToggle()));
	connect(mStatAct, SIGNAL(triggered()), this, SLOT(statsToggle()));
	
	connect(mOpenSettingsAct, SIGNAL(triggered()), this, SLOT(openSettings()));
	connect(mOpenSettingsMenuAct, SIGNAL(triggered()), this, SLOT(openSettings()));
	connect(mSaveasSettingsAct, SIGNAL(triggered()), this, SLOT(saveAsSettings()));

	connect(mAboutAct, SIGNAL(triggered()), this, SLOT(AboutWaveDump()));

	/*
	connect(scanAct, &QAction::triggered, this, &MainWindow::newFile);
	connect(uploadAct, &QAction::triggered, this, &MainWindow::newFile);
	connect(infoAct, &QAction::triggered, this, &MainWindow::newFile);*/

	mMainToolBar->addAction(mScanAct);
	mMainToolBar->addAction(mConnectAct);
	mMainToolBar->addAction(mDisconnectAct);
	mMainToolBar->addSeparator();
	mMainToolBar->addAction(mLoadAct);
	mMainToolBar->addAction(mOpenSettingsAct);
	mMainToolBar->addSeparator();
	mMainToolBar->addAction(mSaveAct);
	mMainToolBar->addAction(mSaveasAct);
	mMainToolBar->addAction(mSaveasSettingsAct);
	mMainToolBar->addSeparator();
	mMainToolBar->addSeparator();
	mMainToolBar->addAction(mBrowseAct);
	mMainToolBar->addAction(mLogAct);
	mMainToolBar->addAction(mManualControllerAct);
	mMainToolBar->addSeparator();
	mMainToolBar->addAction(mStatAct);

	//mMainToolBar->addWidget(new QLabel("Default Config:"));
	mDefConfigFile = new QLineEdit();
	mDefConfigFile->setEnabled(false);
	mDefConfigFile->setMaximumSize(600,50);
	mMainToolBar->addWidget(mDefConfigFile);
	mAutoLoadFile = new QCheckBox();
	mAutoLoadFile->setText("use file as default config");
	mAutoLoadFile->setChecked(false);
	mMainToolBar->addWidget(mAutoLoadFile);
	connect(mAutoLoadFile, SIGNAL(clicked(bool)), this, SLOT(DefaultLoadEnable(bool)));


	mSaveAct->setEnabled(false);
	mBrowseAct->setEnabled(false);
	mLogAct->setEnabled(false);
	mOpenSettingsAct->setEnabled(false);
	mSaveasAct->setEnabled(false);
	mSaveasSettingsAct->setEnabled(false);
	mMainToolBar->setMovable(false);
	mManualControllerAct->setEnabled(false);
	mStatAct->setEnabled(false);
	mSaveOutputAct->setEnabled(false);
	mOutputConfigAct->setEnabled(false);
	mResetButton->setEnabled(false);
}

void WaveDump2::CheckLoadLastConfig(QString pid) {
	QDir dir(QDir::homePath() + DIRSEP + WAVEDUMP2_DIR);

	if (!dir.exists()) {
		return;
	}

	QString filename = QString("WD2_%1.wconf").arg(pid);
	QStringList filters;
	filters << filename;
	dir.setNameFilters(filters);

	QFileInfoList files = dir.entryInfoList(QDir::Files);
	if (files.isEmpty())
		return;
	else {
		QMessageBox::StandardButton resBtn = QMessageBox::question(this, "WaveDump2",
			tr("Do you want to load the last used configuration for this device?\n"),
			QMessageBox::No | QMessageBox::Yes, QMessageBox::No);
		if (resBtn == QMessageBox::Yes) {
			loadFile(QDir::homePath() + DIRSEP + WAVEDUMP2_DIR + DIRSEP + files.at(0).fileName(), false);
		}
	}
}

void WaveDump2::CheckDefaultConfigToLoad(){
	QString filename = QDir::homePath() + DIRSEP + QString(".fWInfo");
	QString Cfilename, enable;
	QFile config_info(filename);
	bool file_found = false;
	if (config_info.exists()) {
		config_info.open(QIODevice::ReadOnly);
		QTextStream in(&config_info);
		Cfilename = in.readLine();
		enable = in.readLine();
		config_info.close();
	}
	else
		return;
	QFile config(Cfilename);
	if (config.exists() && enable.toInt()) {
		mDefConfigFile->setText(Cfilename);
		if (enable.toInt()) {	
			loadFile(Cfilename, true);			
			mIgnoreEvent = true;
			mAutoLoadFile->setChecked(true);
			mIgnoreEvent = false;
		}
	}
	else {
		if(enable.toInt())
			QMessageBox::critical(this, tr("Load default configuration"), "Default configuration file not found");
		mDefConfigFile->setText(Cfilename);
		DefaultLoadEnable(false);
	}

}

void WaveDump2::DefaultLoadEnable(bool enable) {
	if (mDefConfigFile->text() == "") {
		mIgnoreEvent = true;
		mAutoLoadFile->setChecked(false);
		mIgnoreEvent = false;
		return;
	}
	QString filename = QDir::homePath() + DIRSEP + QString(".fWInfo");
	QFile f(filename);
	f.open(QIODevice::WriteOnly|QIODevice::Truncate);
	QTextStream out(&f);
	out << mDefConfigFile->text() << ENDL;
	out << static_cast<int>(enable) << ENDL;
	f.close();
}


void WaveDump2::populateMenu() {
	mFileMenu = new QMenu();
	mFileMenu->addAction(mScanMenuAct);
	mFileMenu->addSeparator();
	mFileMenu->addAction(mConnectMenuAct);
	mFileMenu->addAction(mLoadMenuAct);
	mFileMenu->addSeparator();
	mFileMenu->addAction(mLoadRawMenuAct);
	mFileMenu->addSeparator();
	mFileMenu->addAction(mSaveMenuAct);
	mFileMenu->addAction(mSaveasMenuAct);
	mFileMenu->addSeparator();
	mFileMenu->addAction(mInfoMenuAct);
	mFileMenu->addAction(mErrorsMenuAct);
	mFileMenu->addSeparator();
	mFileMenu->addAction(mExitMenuAct);
	this->menuBar()->addMenu(mFileMenu)->setText("&File");

	mConfiguration = new QMenu();
	this->menuBar()->addMenu(mConfiguration)->setText("&Configuration");
	mConfiguration->setEnabled(false);

	mScriptMenu = new QMenu();
	mScriptMenu->addAction(mOpenSettingsMenuAct);
	this->menuBar()->addMenu(mScriptMenu)->setText("&Settings");
	mScriptMenu->setEnabled(false);

	mOutputMenu = new QMenu();
	mOutputMenu->addAction(mSaveOutputAct);
	mOutputMenu->addSeparator();
	mOutputMenu->addAction(mOutputConfigAct);
	this->menuBar()->addMenu(mOutputMenu)->setText("&Output");
	mOutputMenu->setEnabled(false);

	mWindowMenu = new QMenu();
	mWindowMenu->addAction(mBrowseAct);
	mWindowMenu->addAction(mLogAct);
	mWindowMenu->addAction(mManualControllerAct);
	mWindowMenu->addAction(mStatAct);
	this->menuBar()->addMenu(mWindowMenu)->setText("&Window");
	mWindowMenu->setEnabled(false);

	mHelpMenu = new QMenu();
	mHelpMenu->addAction(mAboutAct);
	this->menuBar()->addMenu(mHelpMenu)->setText("&Help");
	mHelpMenu->setEnabled(true);

	mSaveMenuAct->setEnabled(false);
	mSaveasMenuAct->setEnabled(false);
}
void WaveDump2::connect2Device() {
		openDialog  dialog(this);
		dialog.setModal(true);
		dialog.exec();
}

void WaveDump2::disconnectDevice() {
	if (mActiveDevice == nullptr)
		return;
	if (mActiveDevice->AcqStarted) {
		QMessageBox::StandardButton reply = QMessageBox::question(this, "WaveDump2",	tr("Active device is acquiring, disconnect it and stop data acquisition?\n"), QMessageBox::No | QMessageBox::Yes, QMessageBox::Yes);
		if (reply == QMessageBox::Yes) {
			if(!OfflineMode)
				stopAcquisition();
			else {
				if (mManualOfflineAcq)
					stopOfflineManualAcq();
				else
					stopOfflineAcq();
			}
			QThread::msleep(500);
		}
		else 
			return;
	}

	mActiveDevice->CloseConnection();
	mPlotPan->ClearPlot();
	mMainStatusBarText->setText("");
	UpdateAfterDisconnection(mActiveDevice);
}

void WaveDump2::UpdateAfterDisconnection(CAENDevice *device){
	if (mConnected) {
		mConnected--;
		mNumberOfDevice--;
		for (int a = 0; a < mConfiguration->actions().size(); a++) {
			if (mConfiguration->actions().at(a)->text() == device->getName())
				mConfiguration->removeAction(mConfiguration->actions().at(a));
		}
		if (!device->isVirtual())
			mDeviceManagerPan->removeDevice(device->getHandle());
		else
			mPlotPan->removeOfflineDevice(device->getName());
		mManualControllerPan->removeDevice(device->getName());
		mCenterWidget->setMaximumWidth(1);
		mControlPan->updateStartableDevices(0, device->getName());
		QVector<QString> channels;
		mPlotPan->RemoveDevFromPalette(device->getName());
		mPlotPan->updatePalette(0, device->getName(), channels);
		int remove_index = device->getIndex();
		mDevicesReadoutThreads.removeAt(remove_index);
		mConnectedDevices.removeAt(remove_index);
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			if (mConnectedDevices.at(i)->getIndex() >= remove_index)
				mConnectedDevices.at(i)->setIndex(mConnectedDevices.at(i)->getIndex() - 1);
		}

		if (mConnected < 2)
			mGlobalStartMode = "SW_ASYNC";

		if (mConnected == 0) {
			mManualControllerPan->setEnabled(false);
			mControlPan->setEnabled(false);
			mDisconnectMenuAct->setEnabled(false);
			mDisconnectAct->setEnabled(false);
			mInfoMenuAct->setEnabled(false);
			mErrorsMenuAct->setEnabled(false);
			if (stats_panel_open)
				mStatAct->setChecked(false);
			mStatAct->setEnabled(false);
			mPlotPan->setEnabled(false);
			mPlotPan->enableOfflineMode(false);
			mActiveDevice = nullptr;
			mResetButton->setEnabled(false);
			mLoadRawMenuAct->setEnabled(true);
			mLoadAct->setEnabled(true);
			mConnectAct->setEnabled(true);	
			mConnectMenuAct->setEnabled(true);
		}
		else
			setActiveDevice(mConnectedDevices.at(0));

		mStatsPan->fillDevices(mConnectedDevices);
		if(!Check_connection_Timer->isActive() && !OfflineMode)
			Check_connection_Timer->start(3000);
	}

}


void WaveDump2::CheckConnections(){
	uint32_t flags = 0;
	bool critical = false;
	bool reset = false;
	QString err_message;
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		try {
			flags=mConnectedDevices.at(i)->getErrorFlags();
			if (flags != 0) {
				mStatusLED->setText(" Errors ");
				mStatusLED->setStyleSheet("border-radius:6px; font: bold; color: black; background-color: red");
				if (!mConnectedDevices.at(i)->ErrorsReported) {
					mConnectedDevices.at(i)->ErrorsReported = true;
					QString err = mConnectedDevices.at(i)->DecodeFlagsMask(flags);
					if (err.contains("OverTemperature")) {
						QThread::msleep(300);
						flags = mConnectedDevices.at(i)->getErrorFlags();
						err = mConnectedDevices.at(i)->DecodeFlagsMask(flags);
						err_message = " Overtemperature fail detected! The device will be disconnected.\n Try to remove the cause of the overheating before reconnecting, or contact the manufacturer for support.\n\n";
						err_message.append(err);
						critical = true;
					}
					else if (err.contains("ADC ShutDown")) {
						if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG1) {
							err_message = " Device is in failure status due to a previous Overtemperature!\nThe device will be disconnected.\nBefore connecting it, a Reboot is required.\n\n ";
							critical = true;
						}
						else {
							err_message = " Device is in failure status due to a previous Overtemperature!\nBefore connecting it, a Reboot is required.\n You can send a Reboot now or manually Power Cycle the device later.\n\n ";
							reset = true;
						}
						err_message.append(err);						
					}
									
					if (critical) {
						Check_connection_Timer->stop();
						if (mConnectedDevices.at(i)->AcqStarted) {
							stopAcquisition();
							mEventBuildthread->stopRun();							
							QThread::msleep(1000);
						}
						if (mInfo != nullptr) {
							mInfo->Quit();
							QThread::msleep(300);
						}
						QMessageBox::critical(this, tr("Error"), err_message, QMessageBox::Ok, 0);
						mConnectedDevices.at(i)->CloseConnection();
						UpdateAfterDisconnection(mConnectedDevices.at(i));						
					}
					else if (reset) {
						Check_connection_Timer->stop();
						if (mInfo != nullptr) {
							mInfo->Quit();
							QThread::msleep(300);
						}
						QMessageBox msgBox; msgBox.setText(err_message); QAbstractButton* pButtonYes = msgBox.addButton(tr("Reboot and disconnect"), QMessageBox::YesRole); msgBox.addButton(tr("Disconnect"), QMessageBox::NoRole);
						msgBox.exec();
						if (msgBox.clickedButton() == pButtonYes) {
							RebootDevice(mConnectedDevices.at(i)->getName());
						}						
						mConnectedDevices.at(i)->CloseConnection();
						UpdateAfterDisconnection(mConnectedDevices.at(i));
					}					
					else
						QMessageBox::critical(this, tr("Error"), err, QMessageBox::Ok, 0);
				}
			}
			else {
				mStatusLED->setText(" No Errors ");
				mStatusLED->setStyleSheet("border-radius:6px; font: bold; color: black; background-color: rgb(48, 243, 18)");
				mConnectedDevices.at(i)->ErrorsReported = false;
			}
		}
		catch (CAENFELibException& exc) {
			if (mConnectedDevices.at(i)->ConnectionOpen) {
				mConnectedDevices.at(i)->CloseConnection();
				addLog(exc.ErrorMessage(), false);
				QMessageBox::critical(this, tr("Error"), "Communication lost with device " + mConnectedDevices.at(i)->getName(), QMessageBox::Ok, 0);
				UpdateAfterDisconnection(mConnectedDevices.at(i));
			}
		}
	}
}

void WaveDump2::DeviceInfo() {
	mInfo = new infoPanel(0, mConnectedDevices, mControlPan->getDevName());
	mInfo->show();
	connect(mInfo, SIGNAL(destroyed()), this, SLOT(quit_info()));
}

void WaveDump2::quit_info() {
	mInfo = nullptr;
}

void WaveDump2::DeviceErrors() {
	QString line_total = "";
	if (mControlPan->getDevName() == "ALL") {
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			uint32_t err_flags = mConnectedDevices.at(i)->getErrorFlags();
			line_total.append(mConnectedDevices.at(i)->DecodeFlagsMask(err_flags));
			line_total.append("\n");
			line_total.append("\n");
		}
	}
	else {
		uint32_t err_flags = mActiveDevice->getErrorFlags();
		line_total.append(mActiveDevice->DecodeFlagsMask(err_flags));
		line_total.append("\n");
		line_total.append("\n");
	}

	QWidget* win_info = new QWidget(0);
	win_info->setFixedSize(480, 400);
	win_info->setWindowIcon(QIcon(":WAVEDump2.png"));
	QTextEdit* txt_info = new QTextEdit(win_info);
	txt_info->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	txt_info->setStyleSheet("QTextEdit {background-color: rgb(215, 253, 255);}");
	txt_info->setAlignment(Qt::AlignCenter);
	txt_info->setEnabled(false);
	QFont font_text;
	font_text.setPointSize(13);
	txt_info->setFont(font_text);
	QGridLayout* gr_hv = new QGridLayout;
	gr_hv->addWidget(txt_info, 0, 0);
	win_info->setLayout(gr_hv);
	txt_info->setText(line_total);
	txt_info->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
	win_info->setWindowModality(Qt::ApplicationModal);
	win_info->show();
}

void WaveDump2::openOutputSettings() {
		outputSettingsDialog  dialog(this);
		dialog.setModal(true);
		dialog.exec();
}

void WaveDump2::browseToggle() {
	if (mBrowseAct->isChecked()) {
		if (mLoadDeviceManager == 0) {
			QMessageBox::StandardButton resBtn = QMessageBox::question(this, "WaveDump2",
				tr("Do you want to load the Device Manager?\n"),
				QMessageBox::No | QMessageBox::Yes,
				QMessageBox::Yes);
			if (resBtn != QMessageBox::Yes) {
				mBrowseAct->setChecked(false);
				return;
			}
			if (mRecordAction)
				mActions->AddSWAction("DEVMANAGER_SHOW", "TRUE");
			// gestire il load
			mMainStatusBarText->setText("<html><B>Collecting data</b></html>");
			QMessageBox msgBox(this);
			msgBox.setText("<html><B>Reading information from the devices <br>           ... please wait ...</html>");
			msgBox.setStandardButtons({});
			msgBox.show();
			for (int i = 0; i < mConnectedDevices.size(); i++) {
				mAsyncConn = QtConcurrent::run([this, i] { return createDeviceTree(mConnectedDevices[i]->getName(), mConnectedDevices[i]->getHandle()); });
				QThread::msleep(10);
				while (mAsyncConn.isRunning()) {
					QThread::msleep(100);
					this->increaseProgressBar();
					QCoreApplication::processEvents();
				}
				if (mAsyncConn.result()) {
					mLogPan->add2Log("Adding " + mConnectedDevices[i]->getName() + " to Device Manager", true);
				}
				else {
					mDeviceManagerPan->removeDevice(mConnectedDevices[i]->getHandle());
					mLogPan->add2Log("Adding " + mConnectedDevices[i]->getName() + " to Device Manager", false);
				}
			}
			msgBox.close();
			mMainProgressBar->setValue(0);
			mMainStatusBarText->setText("<html><B>Ready</b></html>");
			mLoadDeviceManager = 1;
		}
		mDeviceManagerDock->show();
		resizeDocks({ mDeviceManagerDock }, { 400 }, Qt::Horizontal);
	}
	else {
		mDeviceManagerDock->hide();
		if (mRecordAction)
			mActions->AddSWAction("DEVMANAGER_SHOW", "FALSE");
	}
}

void WaveDump2::logToggle() {
	if (mLogAct->isChecked()) {
		mLogDock->show();
		if (mRecordAction)
			mActions->AddSWAction("LOG_SHOW", "TRUE");
	}
	else {
		mLogDock->hide();
		if (mRecordAction)
			mActions->AddSWAction("LOG_SHOW", "FALSE");
	}

}

void WaveDump2::manualToggle() {
	if (mManualControllerAct->isChecked()) {
		mManualControllerDock->show();
		if (mRecordAction)
			mActions->AddSWAction("MANUAL_SHOW", "TRUE");
	}
	else {
		mManualControllerDock->hide();
		if (mRecordAction)
			mActions->AddSWAction("MANUAL_SHOW", "FALSE");
	}
}

void WaveDump2::statsToggle() {
	if (mStatAct->isChecked()) {
		mStatsDock->show();
		mStatsPan->showStats();
		stats_panel_open = true;
		if (mRecordAction)
			mActions->AddSWAction("STATS_SHOW", "TRUE");
	}
	else {
		mStatsDock->hide();
		stats_panel_open = false;
		if (mRecordAction)
			mActions->AddSWAction("STATS_SHOW", "FALSE");
	}
}

void WaveDump2::OnlyMatchingEnable(bool enable) { 
	OnlyMatching = enable;
	if (mRecordAction) {
		if(enable)
			mActions->AddSWAction("FULLY_MATCHING_EVENTS_ONLY", "TRUE");
		else
			mActions->AddSWAction("FULLY_MATCHING_EVENTS_ONLY", "FALSE");
	}
}

void WaveDump2::closeEvent(QCloseEvent *event)
{
	QMessageBox::StandardButton resBtn = QMessageBox::question(this, "WaveDump2",
		tr("Are you sure you want to quit the application?\n"),
		QMessageBox::Cancel | QMessageBox::No | QMessageBox::Yes,
		QMessageBox::Yes);
	if (resBtn != QMessageBox::Yes) {
		event->ignore();
	}
	else {
		if (mConnected) {
			QMessageBox::StandardButton resBtn = QMessageBox::question(this, "WaveDump2",
				tr("Do you want to save current configuration?\n"),
				QMessageBox::No | QMessageBox::Yes, QMessageBox::No);
			if (resBtn == QMessageBox::Yes) {
				saveAs();
			}
			for (int i = 0; i < mConnectedDevices.size(); i++) {
				mConnected--;
				mConnectedDevices.at(i)->CloseConnection();
			}
		}
		event->accept();
	}
}

void WaveDump2::manageAddLogAndConf(QString param_name, CAENDevice *dev, QString value_log, QString value_to_set) {
	QString val;
	if (param_name == "SelfTriggerMask") {
		if (dev->DeviceClass == CAEN_DIG2) {
			add2LogAndConf("SET", dev->getName(), "/par/ITLAMask", value_log, value_to_set, CAEN_FELib_Success);
			add2LogAndConf("SET", dev->getName(), "/par/ITLAMainLogic", "OR", "OR", CAEN_FELib_Success);
			add2LogAndConf("SET", dev->getName(), "/par/ITLAPairLogic", "None", "None", CAEN_FELib_Success);
			add2LogAndConf("SET", dev->getName(), "/par/ITLAGateWidth", "16", "16", CAEN_FELib_Success);
		}
		else {
			int gsize = static_cast<CAENDig1Device*>(dev)->getChGroupSize();
			uint64_t mask = value_log.toULong();
			if ( gsize > 1) {
				uint32_t group_mask = 0;
				for (int ch = 0; ch < dev->getNumCh(); ch++) {
					if (mask & (1LL << ch))
						group_mask |= (1 << (ch / gsize));
				}
				for (int g = 0; g < dev->getNumCh() / gsize; g++) {
					if (group_mask & (1 << g))
						val = "TRUE";
					else
						val = "FALSE";
					add2LogAndConf("SET", dev->getName(), QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), val, val, CAEN_FELib_Success);
				}
			}
			for (int ch = 0; ch < dev->getNumCh(); ch++) {
				if (mask & (1LL << ch))
					val = "TRUE";
				else
					val = "FALSE";
				add2LogAndConf("SET", dev->getName(), QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch), val, val, CAEN_FELib_Success);
			}
		}
		return;
	}
	else if (param_name == "TrgoutMask") {
		if (dev->DeviceClass == CAEN_DIG2)
			add2LogAndConf("SET", dev->getName(), "ITLBMask", value_log, value_to_set, CAEN_FELib_Success);
		else {
			int gsize = static_cast<CAENDig1Device*>(dev)->getChGroupSize();
			uint64_t mask = value_log.toULong();
			if (gsize > 1) {
				uint32_t group_mask = 0;
				for (int ch = 0; ch < dev->getNumCh(); ch++) {
					if (mask & (1LL << ch))
						group_mask |= (1 << (ch / gsize));
				}
				for (int g = 0; g < dev->getNumCh() / gsize; g++) {
					if (group_mask & (1 << g))
						val = "TRUE";
					else
						val = "FALSE";
					add2LogAndConf("SET", dev->getName(), QString("/group/%1/par/Gr_Out_propagate").arg(g), val, val, CAEN_FELib_Success);
				}
			}
			for (int ch = 0; ch < dev->getNumCh(); ch++) {
				if (mask & (1LL << ch))
					val = "TRUE";
				else
					val = "FALSE";
				add2LogAndConf("SET", dev->getName(), QString("/ch/%1/par/Ch_Out_propagate").arg(ch), val, val, CAEN_FELib_Success);
			}
		}
		return;
	}
	else if (param_name == "RecordLength") {
		if (dev->DeviceClass == CAEN_DIG2)
			add2LogAndConf("SET", dev->getName(), "/par/RecordLengthT", value_log, value_to_set, CAEN_FELib_Success);
		else
			add2LogAndConf("SET", dev->getName(), "/par/RecLen", value_log, value_to_set, CAEN_FELib_Success);
		return;
	}
	else if (param_name == "PreTrigger"){
		if (dev->DeviceClass == CAEN_DIG2)
			add2LogAndConf("SET", dev->getName(), "/par/PreTriggerT", value_log, value_to_set, CAEN_FELib_Success);
		else {
			uint32_t post = (dev->getReclenCachedValue()*dev->getDecimation_value() - value_to_set.toUInt());
			QString val = QString("%1").arg(post);
			add2LogAndConf("SET", dev->getName(), "/par/PostTrg", val, val, CAEN_FELib_Success);
		}
		return;
	}
	else if (param_name.contains("TriggerThr")) {
		if (dev->DeviceClass == CAEN_DIG2)
			add2LogAndConf("SET", dev->getName(), param_name, value_log, value_to_set, CAEN_FELib_Success);
		else {
			QStringList l = param_name.split("/");
			int ch = l.at(1).toInt();
			double val = (value_to_set.toDouble()* (dev->getVRangeV(ch) * 1e3)/ dev->getThr_max(ch) + mPlotPan->getMinY());
			if (dev->getFamilyCode() == "XX740") {
				int g = ch / static_cast<CAENDig1Device*>(dev)->getChGroupSize();
				add2LogAndConf("SET", dev->getName(), QString("/group/%1/par/Gr_Threshold").arg(g), QString("%1").arg(val), value_to_set, CAEN_FELib_Success);
			}
			else {
				QString new_name = param_name;
				new_name.replace("TriggerThr","Ch_Threshold");
				add2LogAndConf("SET", dev->getName(), new_name, QString("%1").arg(val), value_to_set, CAEN_FELib_Success);
			}
		}
		return;
	}
	else if (param_name.contains("DCOffset")) {
		if (dev->DeviceClass == CAEN_DIG2) {
			add2LogAndConf("SET", dev->getName(), param_name, value_log, value_to_set, CAEN_FELib_Success);
		}
		else {
			int gsize = static_cast<CAENDig1Device*>(dev)->getChGroupSize();
			if (gsize > 1) {
				QStringList l = param_name.split("/");
				int ch = l.at(2).toInt();
				int g = ch / static_cast<CAENDig1Device*>(dev)->getChGroupSize();
				add2LogAndConf("SET", dev->getName(), QString("/group/%1/par/Gr_DCOffset").arg(g), value_log, value_to_set, CAEN_FELib_Success);
			}
			else {
				QStringList l = param_name.split("/");
				int ch = l.at(2).toInt();
				add2LogAndConf("SET", dev->getName(), QString("/ch/%1/par/Ch_DCOffset").arg(ch), value_log, value_to_set, CAEN_FELib_Success);
			}
		}
		return;
	}
	else if (param_name == "TriggerSource") {
		if (dev->DeviceClass == CAEN_DIG2) {
			add2LogAndConf("SET", dev->getName(), "/par/AcqTriggerSource", value_log, value_to_set, CAEN_FELib_Success);
		}
		else {
			if (value_to_set.contains("TrgIn", Qt::CaseInsensitive))
				val = "TRUE";			
			else 
				val = "FALSE";
			add2LogAndConf("SET", dev->getName(), "/par/Trg_Ext_Enable", val, val, CAEN_FELib_Success);

			if (value_to_set.contains("Sw", Qt::CaseInsensitive))
				val = "TRUE";
			else
				val = "FALSE";
			add2LogAndConf("SET", dev->getName(), "/par/Trg_Sw_Enable", val, val, CAEN_FELib_Success);

			if (!value_to_set.contains("ITL", Qt::CaseInsensitive)) {
				manageAddLogAndConf("SelfTriggerMask", dev, "0", "0");
				for (int g = 0; g < dev->getNumCh() / static_cast<CAENDig1Device *>(dev)->getChGroupSize(); g++) 
					add2LogAndConf("SET", dev->getName(), QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), "FALSE", "FALSE", CAEN_FELib_Success);
			}
		}
	}
	else if (param_name == "GlobalStartMode") {
		if (dev->DeviceClass == CAEN_DIG2) {
			if (value_log == "SW_ASYNC")
				add2LogAndConf("SET", dev->getName(), "/par/StartSource", "SwCmd", "SwCmd", CAEN_FELib_Success);
			else if (value_log == "SW_CHAIN_SYNC-CLKIN" || value_log == "HW_CHAIN_SYNC-CLKIN") {
				if (dev->DevChainIndex == 0 && (value_log == "SW_CHAIN_SYNC-CLKIN")) { //board  b is the master and start mode is still sw controlled
					add2LogAndConf("SET", dev->getName(), "/par/StartSource", "SWcmd", "SWcmd", CAEN_FELib_Success);
					add2LogAndConf("SET", dev->getName(), "/par/SyncOutMode", "Run", "Run", CAEN_FELib_Success);
				}
				else {
					add2LogAndConf("SET", dev->getName(), "/par/StartSource", "EncodedClkIn", "EncodedClkIn", CAEN_FELib_Success);
					add2LogAndConf("SET", dev->getName(), "/par/SyncOutMode", "SyncIn", "SyncIn", CAEN_FELib_Success);
				}
			}
			else if (value_log == "SW_CHAIN_SIN-GPIO" || value_log == "HW_CHAIN_SIN-GPIO") {
						if (dev->DevChainIndex == 0 && value_log == "SW_CHAIN_SIN-GPIO") { //board  b is the master and start mode is still sw controlled
							add2LogAndConf("SET", dev->getName(), "/par/StartSource", "SWcmd", "SWcmd", CAEN_FELib_Success);
							add2LogAndConf("SET", dev->getName(), "/par/GPIOMode", "Run", "Run", CAEN_FELib_Success);
						}
						else {
							add2LogAndConf("SET", dev->getName(), "/par/StartSource", "SINLevel", "SINLevel", CAEN_FELib_Success);
							add2LogAndConf("SET", dev->getName(), "/par/GPIOMode", "SIN", "SIN", CAEN_FELib_Success);
						}
			}
			else if (value_log == "SW_CHAIN_SIN-TRGOUT" || value_log == "HW_CHAIN_SIN-TRGOUT") {
					if (dev->DevChainIndex == 0 && value_log == "SW_CHAIN_SIN-TRGOUT") { //board  b is the master and start mode is still sw controlled
						add2LogAndConf("SET", dev->getName(), "/par/StartSource", "SWcmd", "SWcmd", CAEN_FELib_Success);
						add2LogAndConf("SET", dev->getName(), "/par/TRGOUTMode", "Run", "Run", CAEN_FELib_Success);
					}
					else {
						add2LogAndConf("SET", dev->getName(), "/par/StartSource", "SINLevel", "SINLevel", CAEN_FELib_Success);
						add2LogAndConf("SET", dev->getName(), "/par/TRGOUTMode", "Run", "Run", CAEN_FELib_Success);
						}
			}
			else if (value_log == "SW_CHAIN_1ST_TRIGGER" || value_log == "HW_CHAIN_1ST_TRIGGER") {
					add2LogAndConf("SET", dev->getName(), "/par/StartSource", "FirstTrigger", "FirstTrigger", CAEN_FELib_Success);

					if (dev->DevChainIndex == 0 && value_log == "SW_CHAIN_1ST_TRIGGER")  //board  b is the master and start mode is still sw controlled
						add2LogAndConf("SET", dev->getName(), "/par/TRGOUTMode", "Run", "Run", CAEN_FELib_Success);
					else 
						add2LogAndConf("SET", dev->getName(), "/par/TRGOUTMode", "TrgIn", "TrgIn", CAEN_FELib_Success);						
			}

		}
		else {
			if (value_log == "SW_ASYNC") { //individual start mode
				add2LogAndConf("SET", dev->getName(), "/par/StartMode", "START_MODE_SW", "START_MODE_SW", CAEN_FELib_Success);
			}
			else if (value_log == "SW_CHAIN_SIN-TRGOUT" || value_log == "HW_CHAIN_SIN-TRGOUT") {
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Ext_Enable", "TRUE", "TRUE", CAEN_FELib_Success);
					if (dev->DevChainIndex == 0 && value_log == "SW_CHAIN_SIN-TRGOUT") { //board  b is the master and start mode is still sw controlled
						add2LogAndConf("SET", dev->getName(), "/par/StartMode", "START_MODE_SW", "START_MODE_SW", CAEN_FELib_Success);
						add2LogAndConf("SET", dev->getName(), "/par/OUT_Selection", "OUT_PROPAGATION_RUN", "OUT_PROPAGATION_RUN", CAEN_FELib_Success);
					}
					else {
						add2LogAndConf("SET", dev->getName(), "/par/StartMode", "START_MODE_S_IN", "START_MODE_S_IN", CAEN_FELib_Success);
						add2LogAndConf("SET", dev->getName(), "/par/OUT_Selection", "OUT_PROPAGATION_RUN", "OUT_PROPAGATION_RUN", CAEN_FELib_Success);
					}
			}
			else if (value_log == "SW_CHAIN_1ST_TRIGGER" || value_log == "HW_CHAIN_1ST_TRIGGER") {
				add2LogAndConf("SET", dev->getName(), "/par/StartMode", "START_MODE_FIRST_TRG", "START_MODE_FIRST_TRG", CAEN_FELib_Success);
				if (dev->DevChainIndex == 0 && value_log == "SW_CHAIN_1ST_TRIGGER") { //board  b is the master and start mode is still sw controlled
					add2LogAndConf("SET", dev->getName(), "/par/OUT_Selection", "OUT_PROPAGATION_RUN", "OUT_PROPAGATION_RUN", CAEN_FELib_Success);
				}
				else {
					add2LogAndConf("SET", dev->getName(), "/par/Trg_Ext_Out_Propagate", "TRUE", "TRUE", CAEN_FELib_Success);
				}
			}
			else if (value_log == "SW_CHAIN_TRGIN-TRGOUT" || value_log == "HW_CHAIN_TRGIN-TRGOUT") {
				if (dev->DevChainIndex == 0) { //board  b is the master, disable ext trigin to avoid start run with ext triggers
					add2LogAndConf("SET", dev->getName(), "/par/Ext_Trg_Inhibit", "TRUE", "TRUE", CAEN_FELib_Success);
				}
				add2LogAndConf("SET", dev->getName(), "/par/StartMode", "START_MODE_FIRST_TRG", "START_MODE_FIRST_TRG", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Ext_Enable", "TRUE", "TRUE", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Sw_Enable", "TRUE", "TRUE", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Ext_Out_Propagate", "TRUE", "TRUE", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Sw_Out_Propagate", "TRUE", "TRUE", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/OUT_Selection", "OUT_PROPAGATION_TRIGGER", "OUT_PROPAGATION_TRIGGER", CAEN_FELib_Success);
			}
			else if (value_log == "SW_CHAIN_ONE2ALL_EXTOR" || value_log == "HW_CHAIN_ONE2ALL_EXTOR") {
				add2LogAndConf("SET", dev->getName(), "/par/StartMode", "START_MODE_FIRST_TRG", "START_MODE_FIRST_TRG", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Ext_Enable", "TRUE", "TRUE", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Sw_Enable", "FALSE", "FALSE", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Ext_Out_Propagate", "FALSE", "FALSE", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/Trg_Sw_Out_Propagate", "TRUE", "TRUE", CAEN_FELib_Success);
				add2LogAndConf("SET", dev->getName(), "/par/OUT_Selection", "OUT_PROPAGATION_TRIGGER", "OUT_PROPAGATION_TRIGGER", CAEN_FELib_Success);
			}
		}
	}
	else if (param_name == "ClockSource") {
		if (dev->DeviceClass == CAEN_DIG2) 
			add2LogAndConf("SET", dev->getName(), "/par/ClockSource", value_log, value_to_set, CAEN_FELib_Success);
		else {
			if(value_to_set == "Internal")
				add2LogAndConf("SET", dev->getName(), "/par/DT_Ext_Clock", "FALSE", "FALSE", CAEN_FELib_Success);
			else
				add2LogAndConf("SET", dev->getName(), "/par/DT_Ext_Clock", "TRUE", "TRUE", CAEN_FELib_Success);
		}
	}	
	else if (param_name == "IOLevel") {
		if (dev->DeviceClass == CAEN_DIG2)
			add2LogAndConf("SET", dev->getName(), "/par/IOLevel", value_log, value_to_set, CAEN_FELib_Success);
		else 
			add2LogAndConf("SET", dev->getName(), "/par/IOLevel", "FPIOType_" + value_log, "FPIOType_" + value_to_set, CAEN_FELib_Success);
	}
	else if (param_name == "RunDelay") {
		if (dev->DeviceClass == CAEN_DIG2)
			add2LogAndConf("SET", dev->getName(), "/par/RunDelay", value_log, value_to_set, CAEN_FELib_Success);
		else
			add2LogAndConf("SET", dev->getName(), "/par/Start_Delay", value_log, value_to_set, CAEN_FELib_Success);
	}
}

void WaveDump2::addLog(const QString& cmd, bool ok) {
	mLogPan->add2Log(cmd, ok);
}

void WaveDump2::add2LogAndConf(const QString& cmd, const QString& bname, const QString& qry, const QString& val, const QString& valS, CAEN_FELib_ErrorCode ErrCode) {
	QString QryLog = qry;
	if (cmd == "CMD") {
		mLogPan->add2Log(cmd, qry, val, bname, ErrCode);
		return;
	}
	QString n = bname;
	if (n.contains("://"))
		n.replace("://", ".");
	if(n.contains(":"))
		n.replace(":", ".");
	if (n.contains("/"))
		n.replace("/", ".");
	QStringList listCH = findChannels(qry);
	QStringList listG = findGroups(qry);
	switch (listCH.size()) {
	case 0: // global sattings
		if (listG.size() == 0) {
			if (ErrCode == CAEN_FELib_Success) {
				mConfigFile->beginGroup(n + "_GLOBAL_SETTINGS");
				mConfigFile->setValue(qry, valS);
				mConfigFile->endGroup();
			}
			if (mRecordAction)
				mActions->AddAction(cmd, bname, qry, valS);
			mLogPan->add2Log(cmd, qry, val, bname, ErrCode);
		}
		else {//group settings
			switch (listG.size()) {
				case 1: // channel settings
				{
					QRegularExpression regular("[\\/]GROUP[\\/]\\d{1,2}(\\/.*)", QRegularExpression::CaseInsensitiveOption);
					QRegularExpressionMatch match;
					match = regular.match(qry);
					if (match.hasMatch()) {
						QryLog = match.captured(1);
					}
					if (ErrCode == CAEN_FELib_Success) {
						mConfigFile->beginGroup(n + "_GROUP" + listG.at(0)); //this->mCurrectDeviceName
						if (n != "COMMON" && n != "COMMON1")
							mConfigFile->setValue(QryLog, valS);
						else
							mConfigFile->setValue(qry, valS);
						mConfigFile->endGroup();
					}
					if (mRecordAction)
						mActions->AddAction(cmd, bname, qry, valS);
					mLogPan->add2Log(cmd, qry, val, bname, ErrCode);
				}
				break;
				case 2: // range settings
				{
					QRegularExpression regular("[\\/]GROUP[\\/]\\d{1,2}(\\/.*)", QRegularExpression::CaseInsensitiveOption);
					QRegularExpressionMatch match;
					int fG = listG.at(0).toShort();
					int tG = listG.at(1).toShort();
					QString pattern = listG.at(0) + ".." + listG.at(1);
					for (int i = fG; i <= tG; i++) {
						QString nqry = qry;
						nqry.replace(pattern, QString::number(i));

						match = regular.match(nqry);
						if (match.hasMatch()) {
							QryLog = match.captured(1);
						}
						else QryLog = nqry;
						if (ErrCode == CAEN_FELib_Success) {
							mConfigFile->beginGroup(n + "_GROUP" + QString::number(i)); //this->mCurrectDeviceName
							if (n != "COMMON" && n != "COMMON1")
								mConfigFile->setValue(QryLog, valS);
							else
								mConfigFile->setValue(qry, valS);
							mConfigFile->endGroup();
						}
						if (mRecordAction)
							mActions->AddAction(cmd, bname, qry, valS);
						mLogPan->add2Log(cmd, nqry, val, bname, ErrCode);
					}
				}
				break;
			}
		}
		break;
	case 1: // channel settings
	{
		QRegularExpression regular("[\\/]CH[\\/]\\d{1,2}(\\/.*)", QRegularExpression::CaseInsensitiveOption);
		QRegularExpressionMatch match;
		match = regular.match(qry);
		if (match.hasMatch()) {
			QryLog = match.captured(1);
		}
		if (ErrCode == CAEN_FELib_Success) {
			mConfigFile->beginGroup(n + "_CH" + listCH.at(0)); //this->mCurrectDeviceName
			if(n != "COMMON" && n != "COMMON1")
				mConfigFile->setValue(QryLog, valS);
			else
				mConfigFile->setValue(qry, valS);
			mConfigFile->endGroup();
		}
		if (mRecordAction)
			mActions->AddAction(cmd, bname, qry, valS);
			//mActions->AddAction(cmd, bname + "_CH" + listCH.at(0), QryLog, valS);
		mLogPan->add2Log(cmd, qry, val, bname, ErrCode);
	}
	break;
	case 2: // range settings
	{
		QRegularExpression regular("[\\/]CH[\\/]\\d{1,2}(\\/.*)", QRegularExpression::CaseInsensitiveOption);
		QRegularExpressionMatch match;
		int fCh = listCH.at(0).toShort();
		int tCh = listCH.at(1).toShort();
		QString pattern = listCH.at(0) + ".." + listCH.at(1);
		for (int i = fCh; i <= tCh; i++) {
			QString nqry = qry;
			nqry.replace(pattern, QString::number(i));

			match = regular.match(nqry);
			if (match.hasMatch()) {
				QryLog = match.captured(1);
			}
			else QryLog = nqry;
			if (ErrCode == CAEN_FELib_Success) {
				mConfigFile->beginGroup(n + "_CH" + QString::number(i)); //this->mCurrectDeviceName
				if(n != "COMMON" && n != "COMMON1")
					mConfigFile->setValue(QryLog, valS);
				else
					mConfigFile->setValue(qry, valS);
				mConfigFile->endGroup();
			}
			if (mRecordAction)
				mActions->AddAction(cmd, bname, qry, valS);
			mLogPan->add2Log(cmd, nqry, val, bname, ErrCode);
		}
	}
	break;
	}
}

void WaveDump2::setCurrentDeviceName(const QString& name) {
	mCurrentDeviceName = name;
}

QStringList WaveDump2::findChannels(const QString& qry) {
	QRegularExpression regular("[\\/]CH[\\/]\\d{1,2}\\/.*", QRegularExpression::CaseInsensitiveOption);
	QRegularExpression regularM("[\\/]CH[\\/](\\d{1,2})\\.\\.(\\d{1,2})\\/.*", QRegularExpression::CaseInsensitiveOption);
	QRegularExpression re("\\d{1,2}");
	QRegularExpression rex("\\d{1,2}");
	QStringList listCh;
	QRegularExpressionMatch match = regular.match(qry);
	if (match.hasMatch()) { // one channel only
		QString matched = match.captured(0);
		QRegularExpressionMatch match_local = re.match(matched);
		listCh = match_local.capturedTexts();
	}
	else {
		match = regularM.match(qry);
		if (match.hasMatch()) { // more than one channel
			QString matched = match.captured(0);
			QRegularExpressionMatchIterator i = rex.globalMatch(matched);
			while (i.hasNext()) {
				QRegularExpressionMatch match_local = i.next();
				if (match_local.hasMatch()) {
					listCh << match.captured(0);
				}
			}
		}
	}
	return listCh;
}

QStringList WaveDump2::findGroups(const QString& qry) {
	QRegularExpression regular("[\\/]GROUP[\\/]\\d{1,2}\\/.*", QRegularExpression::CaseInsensitiveOption);
	QRegularExpression regularM("[\\/]GROUP[\\/](\\d{1,2})\\.\\.(\\d{1,2})\\/.*", QRegularExpression::CaseInsensitiveOption);
	QRegularExpression re("\\d{1,2}");
	QRegularExpression rex("\\d{1,2}");
	QStringList listG;
	QRegularExpressionMatch match = regular.match(qry);
	if (match.hasMatch()) { // one channel only
		QString matched = match.captured(0);
		QRegularExpressionMatch match_local = re.match(matched);
		listG = match_local.capturedTexts();
	}
	else {
		match = regularM.match(qry);
		if (match.hasMatch()) { // more than one channel
			QString matched = match.captured(0);
			QRegularExpressionMatchIterator i = rex.globalMatch(matched);
			while (i.hasNext()) {
				QRegularExpressionMatch match_local = i.next();
				if (match_local.hasMatch()) {
					listG << match.captured(0);
				}
			}
		}
	}
	return listG;
}

void WaveDump2::openSettings() {
	QString fileName = QFileDialog::getOpenFileName(this, tr("Open settings file"), "", tr("WaveDump2 settings file (*.wsets)"));
	if (fileName.isEmpty())
		return;
	openSettingsFile(fileName);
}

void WaveDump2::openSettingsFile(QString fileName) {
	QString new_line;
	QStringList list;
	QFile fsettings(fileName);
	if (!fsettings.exists()) {
		QMessageBox::critical(this, tr("Error"), "File " + fileName + " not found.\n Settings could not be loaded.", QMessageBox::Ok, 0);
		return;
	}
	QTextStream in(&fsettings);
	fsettings.open(QIODevice::ReadOnly);
	bool error_found = false, sw_command_error=false;
	QString error_message = "Errors found while loading settings from file: ";
	int line = 0;
	mRecordStopTime = 0;
	if (Check_connection_Timer->isActive())
		Check_connection_Timer->stop();

	while (!in.atEnd()) {

		new_line = in.readLine();
		list = new_line.split(" ");
		if (list.at(0) == ACTION_SW) {
			if (list.size() < 3) {
				error_found = true;
				error_message.append(QString("line %1 wrong format\n").arg(line));
				continue;
			}
			sw_command_error = false;
			if(list.size() > 3)
				sw_command_error = ExecSWCommand(list.at(1), list.at(2) + " " + list.at(3));
			else
				sw_command_error = ExecSWCommand(list.at(1), list.at(2));
			if(sw_command_error)
				error_message.append(QString("line %1 sw command error\n").arg(line));
			
		}
		else if (list.at(0) == ACTION_HW) {
			if (list.size() < 5) {
				error_found = true;
				error_message.append(QString("line %1 wrong format\n").arg(line));
				continue;
			}
			for (int b = 0; b < mConnectedDevices.size(); b++) {
				if (mConnectedDevices.at(b)->getName() == list.at(1)) {
					if (list.at(2) == "SET") {
						CAEN_FELib_ErrorCode ErrCode = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mConnectedDevices.at(b)->getHandle(), qPrintable(list.at(3)), qPrintable(list.at(4)));
						add2LogAndConf("SET", mConnectedDevices.at(b)->getName(), list.at(3), list.at(4), list.at(4), ErrCode);
						if (list.at(3).contains("DCOffset", Qt::CaseInsensitive)) {
							QVector<double>sampl_to_V;
							for (int ch = 0; ch < mConnectedDevices.at(b)->getNumCh(); ch++) {
								if(list.at(3).contains(QString("/ch/%1/").arg(ch)))
									mConnectedDevices.at(b)->getChOffset(ch);
								sampl_to_V.append(mConnectedDevices.at(0)->getSample_to_V(ch));
							}
							mPlotPan->updateBaselines(mConnectedDevices.at(b)->getName(), mConnectedDevices.at(b)->getADCNBit(), sampl_to_V, mConnectedDevices.at(b)->ChBaselines);
						}
						if (list.at(3).contains("VgaGain", Qt::CaseInsensitive)) { 
							for (int ch = 0; ch < mConnectedDevices.at(b)->getNumCh(); ch++) {
								if (list.at(3).contains(QString("/vga/%1/").arg(ch% mConnectedDevices.at(b)->getVgaGainGroupSize(), Qt::CaseInsensitive)))
									mConnectedDevices.at(b)->getChGain(ch);
							}
							mPlotPan->updateGains(mConnectedDevices.at(b)->getName(), mConnectedDevices.at(b)->ChGain);
						}
					}
				}
			}
		
		}
		line++;

		if (mRecordStopTime) {//wait acq time
			QProgressDialog progress("Acquisition running according to the loaded script file...", "Abort", 0, mRecordStopTime, this);
			progress.setWindowModality(Qt::WindowModal);
			int i = 0;
			while (i < mRecordStopTime) {
				progress.setValue(i);

				if (progress.wasCanceled())
					break;
				QThread::msleep(300);
				i += 300;
				QApplication::processEvents();
			}
			QThread::msleep(500);
			QApplication::processEvents();
			mPlotPan->doReplot();
			progress.setValue(mRecordStopTime);//wait finished
			mRecordStopTime = 0;
		}
	}

	if (error_found) {
		QMessageBox::critical(this, tr("Error"), error_message, QMessageBox::Ok, 0);
	}

	if (mActiveDevice == NULL)
		return;
	updateActiveDevInfo(mActiveDevice);
	QVector<double>sampl_to_V;
	for (int ch = 0; ch < mActiveDevice->getNumCh(); ch++)
		sampl_to_V.append(mActiveDevice->getSample_to_V(ch));
	mPlotPan->updateBaselines(mActiveDevice->getName(), mActiveDevice->getADCNBit(), sampl_to_V, mActiveDevice->ChBaselines);
	mPlotPan->updateGains(mActiveDevice->getName(), mActiveDevice->ChGain);
	if (mGlobalStartMode.contains("ASYNC"))
		CheckStartModes();
	else
		AssignChainIndexes();
	if (!Check_connection_Timer->isActive())
		Check_connection_Timer->start(3000);
		
}


bool WaveDump2::ExecSWCommand(QString command, QString val) {
	caen::hash::generator hasher;

	if (command.contains("TRACE_ENABLE")) {
		QStringList l = command.split("_");
		int N = l.at(2).toInt();
		mPlotPan->TraceEnableCmd(N, val == "TRUE");
	}
	else if (command.contains("TRACE_CHANGE")) {
		QStringList l = command.split("_");
		int N = l.at(2).toInt();
		QStringList l1 = val.split(" ");
		if (l1.size() < 2)
			return false;
		mPlotPan->TraceChangeCmd(N, l1.at(0), l1.at(1).toInt());
	}
	else {
		switch (hasher(command.toUtf8().constData())) {
			using namespace caen::hash::literals;
			case YUNIT_CHANGE:
				mControlPan->YUnitChangeCmd(val.toInt());
				break;
			case XUNIT_CHANGE:
				mControlPan->XUnitChangeCmd(val.toInt());
				break;
			case TOOLS_SHOW:
				mPlotPan->ShowToolsCmd(val == "TRUE");
				break;
			case PALETTE_SHOW:
				mPlotPan->ShowPaletteCmd(val == "TRUE");
				break;
			case PLOT_FREEZE:
				mPlotPan->FreezeCmd(val == "TRUE");
				break;
			case LOG_SHOW:
				mLogAct->setChecked(val == "TRUE");
				logToggle();
				break;
			case STATS_SHOW:
				mStatAct->setChecked(val == "TRUE");
				statsToggle();
				break;
			case FULLY_MATCHING_EVENTS:
				OnlyMatchingEnable(val == "TRUE");
				break;
			case MANUAL_SHOW:
				mManualControllerAct->setChecked(val == "TRUE");
				manualToggle();
				break;
			case DEVMANAGER_SHOW:
				mBrowseAct->setChecked(val == "TRUE");
				browseToggle();
				break;
			case PLOTTYPE_CHANGE:
				mPlotPan->PlotTypeChangeCmd(val.toInt());
				break;
			case NWAVES_SET:
				mPlotPan->NAvgChangeCmd(val.toInt());
				break;
			case TRIGGER_MODE_CHANGE: {
				QStringList l1 = val.split(" ");
				if (l1.size() < 2)
					return false;
				mControlPan->TriggerModeChangeCmd(l1.at(1).toInt());
				break;
				}
			case SEND_SW_TRIGGER:
				SingleSWTrg();
				break;
			case ACTIVE_DEV_CHANGE:
				mControlPan->ActiveDevChangeCmd(val);
				break;
			case START_ACQ:
				QThread::msleep(3000);
				startAcquisition();
				break;
			case STOP_ACQ: {
				QStringList l1 = val.split(" ");
				if (l1.size() < 2)
					return false;
				mRecordStopTime = l1.at(1).toInt();
				QTimer::singleShot(mRecordStopTime+300, this, SLOT(stopAcquisition()));
				break;
				}
			case SAVE_FOLDER_CHANGE:
				setConfigValue("OUTPUT", "folder", val);
				mOutput_folder = val;
				break;
			case FILE_PREFIX_CHANGE:
				setConfigValue("OUTPUT", "prefix", val);
				mOutput_prefix = val;
				break;
			case FILE_TYPE_CHANGE:
				setConfigValue("OUTPUT", "ftype", val);
				mOutput_ftype = val;
				break;
			case FILE_FORMAT_CHANGE:
				setConfigValue("OUTPUT", "format", val);
				mOutput_format = val;
				break;
			case FILE_HEADER_ENABLE:
				setConfigValue("OUTPUT", "header", (val == "TRUE")? "YES" : "NO");
				mOutput_header = (val == "TRUE") ? "YES" : "NO";
				break;	
			case FILE_SYNC_UNIQUE:
				setConfigValue("OUTPUT", "sync", (val == "TRUE") ? "YES" : "NO");
				mOutput_sync = (val == "TRUE") ? "YES" : "NO";
				break;
			case DATA_SAVE_ENABLE:
				enableDataSave((val == "TRUE") ? true : false);
				break;
			case GLOBAL_S_MODE_CHANGE:
				setGlobalSMode(val);
				break;
			case GLOBAL_T_MODE_CHANGE:
				setGlobalTrgMode(val);
				break;
			case CINC_WIN_SET:
				setCoincWin(val.toDouble());
				break;
			case MAX_NEVTS_FILE:
				mNEvts_file = val.toInt();
				break;
		}

	}
	return true;
}


void WaveDump2::saveAsSettings() {
	if (mSaveasSettingsAct->isChecked()) {
		QString fileName = QFileDialog::getSaveFileName(this, tr("Save settings file"), "", tr("WaveDump2 settings file (*.wsets)"));
		if (fileName.isEmpty())
			return;
		if (!fileName.contains(".wsets"))
			fileName.append(".wsets");
		mRecordAction = true;
		mActions = new CAENDig2Script(fileName);
		QString message = "The settings applied from now on will be saved to " + fileName;
		QMessageBox::information(this, tr("Save settings"), message, QMessageBox::Ok, 0);
	}
	else {
		if (mActions != nullptr)
			delete mActions;
		mRecordAction = false;
		QMessageBox::information(this, tr("Save settings"), "Save settings disabled.", QMessageBox::Ok, 0);
	}
}

void WaveDump2::load() {
	int reply;
	if (Check_connection_Timer->isActive())
		Check_connection_Timer->stop();
	if (mCouldSave) {
		reply = QMessageBox::question(this, "Test", "Do you want to save the current configuration?",
			QMessageBox::Yes | QMessageBox::No);
		if (reply == QMessageBox::Yes) {
			saveAs();
		}
	}
	QString fileName = QFileDialog::getOpenFileName(this, tr("Open config file"), "", tr("WaveDump2 conf file (*.wconf);;All Files (*)"));
	if (fileName.isEmpty()) {
		if (!Check_connection_Timer->isActive())
			Check_connection_Timer->start(3000);
		return;
	}
	loadFile(fileName, true);
}

void WaveDump2::loadFile(QString fileName, bool check_connect) {
	uint64_t handle;
	CAEN_FELib_ErrorCode ErrCode;
	CAENDevice* dev;
	QStringList allSection;
	QSettings *ConfigFile = new QSettings(fileName, QSettings::IniFormat, this);

	for (const QString &group : ConfigFile->childGroups()) {
		allSection.append(group);
	}
	if (!allSection.contains("DEVICES")) {
		QMessageBox::critical(this, tr("Error loading config file"), tr("The file ") + fileName + tr(" is not a valid Wavedump Config file"), QMessageBox::Ok, 0);
		if (!Check_connection_Timer->isActive())
			Check_connection_Timer->start(3000);
		delete ConfigFile;
		return;
	}

	// COMMON SECTION
	ConfigFile->beginGroup("COMMON_GLOBAL_SETTINGS");
	QStringList common_sectionList = ConfigFile->allKeys();
	CommonDevValuesmap.clear();
	for (const QString &item : common_sectionList) {
		if (item.startsWith("#")) {
			common_sectionList.removeAt(common_sectionList.indexOf(item));
			continue;
		}
		QString test = item;
		QString test1 = ConfigFile->value(item).toString();
		QString qry = "/" + item;
		qry.replace("\\", "/");
		CommonDevValuesmap.insert(qry, ConfigFile->value(item).toString());
	}
	ConfigFile->endGroup();
	QApplication::processEvents();
	ConfigFile->beginGroup("COMMON1_GLOBAL_SETTINGS");
	common_sectionList = ConfigFile->allKeys();
	CommonDevValuesmap1.clear();
	for (const QString& item : common_sectionList) {
		if (item.startsWith("#")) {
			common_sectionList.removeAt(common_sectionList.indexOf(item));
			continue;
		}
		QString test = item;
		QString test1 = ConfigFile->value(item).toString();
		QString qry = "/" + item;
		qry.replace("\\", "/");
		CommonDevValuesmap1.insert(qry, ConfigFile->value(item).toString());
	}
	ConfigFile->endGroup();
	QApplication::processEvents();
	for (int ch = 0; ch < MAX_CHANNELS; ch++) {
		ConfigFile->beginGroup("COMMON_CH" + QString::number(ch));
		QStringList ch_sectionList = ConfigFile->allKeys();
		CommonChValuesmap[ch].clear();
		for (const QString &item : ch_sectionList) {
			if (item.startsWith("#")) {
				ch_sectionList.removeAt(common_sectionList.indexOf(item));
				continue;
			}
			QString qry = "/" + item;
			qry.replace("\\", "/");
			CommonChValuesmap[ch].insert(qry, ConfigFile->value(item).toString());
		}
		ConfigFile->endGroup();
		ConfigFile->beginGroup("COMMON1_CH" + QString::number(ch));
		ch_sectionList = ConfigFile->allKeys();
		CommonChValuesmap1[ch].clear();
		for (const QString& item : ch_sectionList) {
			if (item.startsWith("#")) {
				ch_sectionList.removeAt(common_sectionList.indexOf(item));
				continue;
			}
			QString qry = "/" + item;
			qry.replace("\\", "/");
			CommonChValuesmap1[ch].insert(qry, ConfigFile->value(item).toString());
		}
		ConfigFile->endGroup();
	}
	for (int g = 0; g < MAX_GROUPS; g++) {
		ConfigFile->beginGroup("COMMON_GROUP" + QString::number(g));
		QStringList g_sectionList = ConfigFile->allKeys();
		CommonGroupValuesmap[g].clear();
		for (const QString& item : g_sectionList) {
			if (item.startsWith("#")) {
				g_sectionList.removeAt(common_sectionList.indexOf(item));
				continue;
			}
			QString qry = "/" + item;
			qry.replace("\\", "/");
			CommonGroupValuesmap[g].insert(qry, ConfigFile->value(item).toString());
		}
		ConfigFile->endGroup();
		ConfigFile->beginGroup("COMMON1_GROUP" + QString::number(g));
		g_sectionList = ConfigFile->allKeys();
		CommonGroupValuesmap1[g].clear();
		for (const QString& item : g_sectionList) {
			if (item.startsWith("#")) {
				g_sectionList.removeAt(common_sectionList.indexOf(item));
				continue;
			}
			QString qry = "/" + item;
			qry.replace("\\", "/");
			CommonGroupValuesmap1[g].insert(qry, ConfigFile->value(item).toString());
		}
		ConfigFile->endGroup();
	}

	QApplication::processEvents();
	// DEVICES SECTION
	ConfigFile->beginGroup("DEVICES");
	QStringList sectionList = ConfigFile->allKeys();
	ConfigFile->endGroup();

	for (const QString &item : sectionList) {
		if (item.startsWith("#")) sectionList.removeAt(sectionList.indexOf(item));
	}
	if ((sectionList.size() % 3) != 0) {
		QMessageBox::critical(this, tr("Error loading config file"), tr("The file ") + fileName + tr(" is not a valid Wavedump Config file"), QMessageBox::Ok, 0);
		if (!Check_connection_Timer->isActive())
			Check_connection_Timer->start(3000);
		delete ConfigFile;
		return;
	}
	QRegularExpression regular("DEV_(\\d{1,2})_\\w*", QRegularExpression::CaseInsensitiveOption);
	QRegularExpressionMatch match;
	QVector<int> devices;
	for (const QString &item : sectionList) {
		match = regular.match(item);
		if (match.hasMatch()) {
			if (std::find(devices.begin(), devices.end(), match.captured(1).toInt()) == devices.end()) {
				devices.push_back(match.captured(1).toInt());
			} 
		}
	}
	std::sort(devices.begin(), devices.end());
	QVector<QString> connected_names;
	QString FileArg,FileCType;
	for (int d = 0; d < devices.size(); d++) {
		//configure each device
		ConfigFile->beginGroup("DEVICES");
		QString name = ConfigFile->value("DEV_" + QString::number(devices[d]) + "_NAME").toString();
		QString Stype = ConfigFile->value("DEV_" + QString::number(devices[d]) + "_TYPE").toString();
		int type = getDeviceTypeCode(Stype);

		QString arg = ConfigFile->value("DEV_" + QString::number(devices[d]) + "_ARG").toString();
		if (d == 0) {
			FileArg = arg;
			FileCType = Stype;
		}
		ConfigFile->endGroup();

		//check if this device is already connected
		bool already_connected = false;
		int b = 0;
		if (check_connect) {
			for (b = 0; b < mConnectedDevices.size(); b++) {
				if (mConnectedDevices.at(b)->getName() == name && mConnectedDevices.at(b)->getAddr() == arg) {
					already_connected = true;
					connected_names.append(name);
					break;
				}
			}
		}
		else {
			already_connected = true;
			connected_names.clear();
			for (b = 0; b < mConnectedDevices.size(); b++)
				connected_names.append(mConnectedDevices.at(b)->getName());
		}

		
		if (!already_connected) {//not connected---> connect it
			handle = openDeviceExec(type, arg, name, false);
			if (handle == 0xFFFFFFFFFFFFFFFF) {
				QMessageBox::critical(this, tr("Connection to device error"), tr("The device ") + name + tr(" cannot be connected"), QMessageBox::Ok, 0);
				continue;
			}
			CheckConnectedDevStatus();
			if (mConnectedDevices.at(mConnectedDevices.size() - 1)->ErrorsReported)
				return;
			connected_names.append(name);
			dev = mConnectedDevices.at(mConnectedDevices.size() - 1);
			if (name.contains("dig1", Qt::CaseInsensitive))
				dev->DeviceClass = CAEN_DIG1;
			else
				dev->DeviceClass = CAEN_DIG2;
			QApplication::processEvents();
		}
		else {//already connected --> get handle and apply settings
			dev = mConnectedDevices.at(mConnectedDevices.size() - 1);
			handle = dev->getHandle();
			if (dev->DeviceClass == CAEN_DIG2)
				dev->ApplyDefaultConfig(CommonDevValuesmap, CommonChValuesmap, CommonGroupValuesmap);//apply the common  settings loaded from config file
			else
				dev->ApplyDefaultConfig(CommonDevValuesmap1, CommonChValuesmap1, CommonGroupValuesmap1);//apply the common  settings loaded from config file
			QApplication::processEvents();
		}

		QString n = name;
		if (n.contains("://"))
			n.replace("://", ".");
		if (n.contains(":"))
			n.replace(":", ".");
		if (n.contains("/"))
			n.replace("/", ".");
		ConfigFile->beginGroup(n + "_GLOBAL_SETTINGS");
		sectionList = ConfigFile->allKeys();
		for (const QString &item : sectionList) {
			QString qry = "/" + item;
			qry.replace("\\", "/");
			QString value = ConfigFile->value(item).toString();
			if (qry.contains("RecordLength") || qry.contains("RecLen")) {
				int s = value.toInt();
				int incr = dev->getReclenParam()->getIncr();
				s += dev->NJExtraSamples;
				s = ((s + incr - 1) / incr) * incr;
				dev->RecLenJIncrS = (s - value.toInt()) / (dev->getSample_to_S() * 1e9); //real increment considering minimum step

				value = QString("%1").arg(s);
			}
			ErrCode = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(handle, qPrintable(qry), qPrintable(value));
			mLogPan->add2Log("SET", qry, value, name, ErrCode);			
		}
		ConfigFile->endGroup();
		QVector<int> channels;
		for (const QString &item : allSection) {
			QRegularExpression regular_item(n + "_CH(\\d{1,2})", QRegularExpression::CaseInsensitiveOption);
			match = regular_item.match(item);
			if (!match.hasMatch()) {
				continue;
			}
			channels.push_back(match.captured(1).toInt());
		}
		std::sort(channels.begin(), channels.end());
		for (int ch : channels) {
			ConfigFile->beginGroup(n + "_CH" + QString::number(ch));
			sectionList = ConfigFile->allKeys();
			for (const QString &item : sectionList) {
				QString qry = "/ch/" + QString::number(ch) + "/" + item;
				qry.replace("\\", "/");
				QString value = ConfigFile->value(item).toString();
				ErrCode = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(handle, qPrintable(qry), qPrintable(value));
				mLogPan->add2Log("SET", qry, value, name, ErrCode);
			}
			ConfigFile->endGroup();
		}
		QApplication::processEvents();
		QVector<int> groups;
		for (const QString& item : allSection) {
			QRegularExpression regular_item(n + "_GROUP(\\d{1,2})", QRegularExpression::CaseInsensitiveOption);
			match = regular_item.match(item);
			if (!match.hasMatch()) {
				continue;
			}
			groups.push_back(match.captured(1).toInt());
		}
		std::sort(groups.begin(), groups.end());
		for (int g : groups) {
			ConfigFile->beginGroup(n + "_GROUP" + QString::number(g));
			sectionList = ConfigFile->allKeys();
			for (const QString& item : sectionList) {
				QString qry = "/group/" + QString::number(g) + "/" + item;
				qry.replace("\\", "/");
				if (dev->DeviceClass == CAEN_DIG1) {
					if ((static_cast<CAENDig1Device*>(dev))->getChGroupSize() == 1)
						continue;
				}
				QString value = ConfigFile->value(item).toString();
				ErrCode = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(handle, qPrintable(qry), qPrintable(value));
				mLogPan->add2Log("SET", qry, value, name, ErrCode);				
			}
			ConfigFile->endGroup();
		}
		QApplication::processEvents();
	}

	//check if there are other devices already connected
	for (int b = 0; b < mConnectedDevices.size(); b++) {
		if (!connected_names.contains(mConnectedDevices.at(b)->getName())) {
			QMessageBox::StandardButton resBtn = QMessageBox::question(this, "WaveDump2", "Do you want to apply the file COMMON settings also to device " + mConnectedDevices.at(b)->getName() + ", which is not present in the loaded file?\n", QMessageBox::No | QMessageBox::Yes, QMessageBox::Yes);
			if (resBtn == QMessageBox::Yes) {
				if(mConnectedDevices.at(b)->DeviceClass == CAEN_DIG2)
					mConnectedDevices.at(b)->ApplyDefaultConfig(CommonDevValuesmap, CommonChValuesmap, CommonGroupValuesmap);//apply the common  settings loaded from config file
				else
					mConnectedDevices.at(b)->ApplyDefaultConfig(CommonDevValuesmap1, CommonChValuesmap1, CommonGroupValuesmap1);//apply the common  settings loaded from config file
			}
		}
	}
	if (mConnectedDevices.size() < devices.size())
		return;

	//replace temp config file with the one that has been loaded
	QFile src(fileName);
	QFile dest(mConfigFileName);
	src.open(QIODevice::ReadOnly);
	QByteArray sdata = src.readAll();
	src.close();
	dest.open(QIODevice::WriteOnly | QIODevice::Truncate);
	dest.write(sdata);
	dest.close();
	if (!check_connect) {
		QFile dest(mConfigFileName);
		dest.open(QIODevice::ReadOnly | QIODevice::Text);
		QTextStream destStream(&dest);

		QList<QString> destLines;
		while (!destStream.atEnd()) {
			destLines.append(destStream.readLine());
		}
		dest.close();

		dest.open(QIODevice::WriteOnly | QIODevice::Truncate);
		destStream.setDevice(&dest);
		QString curr_addr = dev->getAddr();



		for(int l=0; l<destLines.size();l++){
			QString line = destLines.at(l);

			if (line.contains(FileArg)) {
				if (FileCType == "ETHERNET" && dev->getCType().contains("USB")) {
					if (line.contains("//"))
						line.replace(FileArg, "usb:" + curr_addr);
					else if (line.contains("." + FileArg))
						line.replace(FileArg, "usb." + curr_addr);
					else
						line.replace(FileArg, curr_addr);
				}
				else if (FileCType == "USB" && dev->getCType().contains("ETH")) {
					if (line.contains("//"))
						line.remove("usb:");
					else if (line.contains("." + FileArg))
						line.remove("usb.");
					line.replace(FileArg, curr_addr);
				}
				else
					line.replace(FileArg, curr_addr);
			}
			if (line.contains("ETHERNET") && dev->getCType().contains("USB"))
				line.replace("ETHERNET", "USB");
			if (line.contains("USB") && dev->getCType().contains("ETH"))
				line.replace("USB", "ETHERNET");
			destStream << line << "\n";
		}
		dest.close();
	}
	///////////////

	QApplication::processEvents();
	if (allSection.contains("LOAD_SETTINGS_FILE")) {
		ConfigFile->beginGroup("LOAD_SETTINGS_FILE");
		QStringList sectionList = ConfigFile->allKeys();
		sectionList = ConfigFile->allKeys();
		for (const QString& item : sectionList) {
			QString filename = ConfigFile->value(item).toString();
			if(!filename.contains(".wsets"))
				QMessageBox::critical(this, tr("Load settings file error"), tr("Invalid settings file specified in the configuration file, only .wsets files are allowed"), QMessageBox::Ok, 0);
			else
				openSettingsFile(filename);
		}
	}

	QApplication::processEvents();
	updateActiveDevInfo(mActiveDevice);
	QVector<double>sampl_to_V;
	for (int ch = 0; ch < mActiveDevice->getNumCh(); ch++)
		sampl_to_V.append(mActiveDevice->getSample_to_V(ch));
	mPlotPan->updateBaselines(mActiveDevice->getName(), mActiveDevice->getADCNBit(), sampl_to_V, mActiveDevice->ChBaselines);
	mPlotPan->updateGains(mActiveDevice->getName(), mActiveDevice->ChGain);
	if (!Check_connection_Timer->isActive())
		Check_connection_Timer->start(3000);
	delete ConfigFile;
	if (mGlobalStartMode.contains("ASYNC"))
		CheckStartModes();
	else
		AssignChainIndexes();

	mDefConfigFile->setText(fileName);
}

void WaveDump2::CheckStartModes() {

	for (int b = 0; b < mConnectedDevices.size(); b++) {
		if (mConnectedDevices.at(b)->getStartSource() == "EncodedClkIn") {
			setGlobalSMode("SW_CHAIN_SYNC-CLKIN");
			return;
		}
		if (mConnectedDevices.at(b)->getStartSource() == "FirstTrigger") {
			setGlobalSMode("SW_CHAIN_1ST_TRIGGER");
			return;
		}
	}

	for (int b = 0; b < mConnectedDevices.size(); b++) {
		if (mConnectedDevices.at(b)->getGPIOMode() == "SIN") {
			setGlobalSMode("SW_CHAIN_SIN-GPIO");
			return;
		}
		if (mConnectedDevices.at(b)->getTRGOUTMode() == "TrgIn") {
			setGlobalSMode("SW_CHAIN_1ST_TRIGGER");
			return;
		}
	}
}
void WaveDump2::AssignChainIndexes() {
	for (int b = 0; b < mConnectedDevices.size(); b++) {
		mConnectedDevices.at(b)->DevChainIndex = b;		
	}
}

void WaveDump2::save() {
	if (mConfigFileName.isEmpty()) {
		saveConfig(true);
	}
	else
		saveConfig(false);
}

void WaveDump2::saveAs() {
	saveConfig(true);
}

void WaveDump2::saveConfig(bool as) {
	QString destination_filename;
	if (as) {
		QString fileName = QFileDialog::getSaveFileName(this, tr("Save config file"), "", tr("WaveDump2 conf file (*.wconf);;All Files (*)"));
		if (fileName.isEmpty())
			return;
		destination_filename = fileName;
		if (!destination_filename.contains(".wconf"))
			destination_filename.append(".wconf");
	}
	else
		destination_filename = mConfigFileName;
	QFile destinationFile(destination_filename);
	if (destinationFile.exists())
		destinationFile.remove();
	QFile::copy(mConfigFileName, destination_filename);
	QMessageBox::information(this, tr("Save operation"), tr("Config File Saved with success."), QMessageBox::Ok, 0);
	mCouldSave = false;
}

void WaveDump2::startOfflineManualAcq() {
	mManualOfflineAcq = true;
	mPlotPan->resetSumCount();
	mLoadAct->setEnabled(false);
	mLoadMenuAct->setEnabled(false);
	mPlotPan->EnableEvDataTab(false);
	mControlPan->toogleStartStop();

	createReadoutThreadsAndStart();

	mPlotPan->EnableEvDataTab(true);
	QString val = mControlPan->getDevName();
	if (val == "ALL") {
		for (int t = 0; t < mDevicesReadoutThreads.size(); t++) {
			if (mDevicesReadoutThreads.at(t) != nullptr) {
				mDevicesReadoutThreads.at(t)->setNextEvToLoad(0);
				QThread::msleep(100);
				mDevicesReadoutThreads.at(t)->waitNextEvCondition.wakeAll();
				mPlotPan->UnLockDataDevice(t);
			}
			else
				mPlotPan->LockDataDevice(t);
		}
	}
	else {
		for (int t = 0; t < mDevicesReadoutThreads.size(); t++) {
			if (mDevicesReadoutThreads.at(t) != nullptr && (mActiveDevice->getIndex() == t)) {
				mDevicesReadoutThreads.at(t)->setNextEvToLoad(0);
				QThread::msleep(100);
				mDevicesReadoutThreads.at(t)->waitNextEvCondition.wakeAll();
				mPlotPan->UnLockDataDevice(t);
				break;
			}
		}
		for (int t = 0; t < mDevicesReadoutThreads.size(); t++) {
			if (mDevicesReadoutThreads.at(t) == nullptr) {
				mPlotPan->LockDataDevice(t);
			}
		}
	}
}

void WaveDump2::stopOfflineManualAcq() {
	QString val = mControlPan->getDevName();
	if (val == "ALL") {
		for (int b = 0; b < mDevicesReadoutThreads.size(); b++) {
			if (mDevicesReadoutThreads.at(b) != nullptr) {
				mDevicesReadoutThreads.at(b)->stopRun();
				mDevicesReadoutThreads.at(b)->waitNextEvCondition.wakeAll();
				mDevicesReadoutThreads.at(b)->wait();
				mDevicesReadoutThreads[b] = nullptr;
			}
		}
		mManualOfflineAcq = false;
		mPlotPan->EnableEvDataTab(false);
	}
	else {
		for (int b = 0; b < mDevicesReadoutThreads.size(); b++) {
			if (mDevicesReadoutThreads.at(b) != nullptr && (mActiveDevice->getIndex() == b)) {
				mDevicesReadoutThreads.at(b)->stopRun();
				mDevicesReadoutThreads.at(b)->waitNextEvCondition.wakeAll();
				mDevicesReadoutThreads.at(b)->wait();
				mDevicesReadoutThreads[b] = nullptr;
				mPlotPan->LockDataDevice(b);
				break;
			}
		}
		bool all_off = true;
		for (int b = 0; b < mDevicesReadoutThreads.size(); b++) {
			if (mDevicesReadoutThreads[b] != nullptr) {
				all_off = false;
				break;
			}
		}
		if (all_off) {
			mManualOfflineAcq = false;
			mPlotPan->EnableEvDataTab(false);
		}
	}
	QThread::msleep(200);
	QApplication::processEvents();
	QThread::msleep(200);
	UpdateStatus();
	checkStartPbEnable();
}

void WaveDump2::NewEvLoadedOK(QString name, int ev, quint64 t) {
	mPlotPan->showNewEvData(name, ev, t);
}

void WaveDump2::NewEvLoadError(QString name, int ev) {
	QMessageBox::critical(this, tr("Load event error"), tr("Error loading selected event number"), QMessageBox::Ok, 0);
	mPlotPan->showNewEvData(name, ev, 0);
}

void WaveDump2::LoadNewEvent(QString name, int ev) {
	for (int b = 0; b < mConnectedDevices.size(); b++) {
		if (mConnectedDevices.at(b)->getName() == name || (mControlPan->getDevName() == "ALL" && !mGlobalStartMode.contains("ASYNC"))) {
			if (mDevicesReadoutThreads.at(b)->Next_Ev_To_Load != ev) {
				mDevicesReadoutThreads.at(b)->setNextEvToLoad(ev);
				mDevicesReadoutThreads.at(b)->waitNextEvCondition.wakeAll();
			}
		}
	}
}

void WaveDump2::PaletteChMask(QString devname, quint64 chmask) {
	for (int t = 0; t < NPLOTTRACES; t++){
		if (mPlotPan->getTraceDev(t) == devname) {
			if (!(chmask & ((uint64_t)1 << mPlotPan->getTraceCh(t)))) {
				if(mPlotPan->isTraceVisible(t))
					mPlotPan->TraceEnableCmd(t, false);
			}
		}
	}
}

void WaveDump2::stopOfflineAcq() {
	if (mManualOfflineAcq)
		return stopOfflineManualAcq();

	QString val = mControlPan->getDevName();

	this->setCursor(Qt::WaitCursor);

	if (val == "ALL") {
			//stopreadouts threads 
			for (int b = 0; b < mConnectedDevices.size(); b++) {
				if (mDevicesReadoutThreads.at(b) != nullptr) {
					mDevicesReadoutThreads.at(b)->stopRun();
					mDevicesReadoutThreads.at(b)->wait();
					mDevicesReadoutThreads[b] = nullptr;
				}
			}
		}
		else {
			if (mDevicesReadoutThreads.at(mActiveDevice->getIndex()) != nullptr) {
				mDevicesReadoutThreads.at(mActiveDevice->getIndex())->stopRun();
				mDevicesReadoutThreads.at(mActiveDevice->getIndex())->wait();
				mDevicesReadoutThreads[mActiveDevice->getIndex()] = nullptr;
		}
	}
	this->setCursor(Qt::ArrowCursor);
	QThread::msleep(200);
	QApplication::processEvents();
	QThread::msleep(200);
	UpdateStatus();
	checkStartPbEnable();
}

void  WaveDump2::checkStartPbEnable() {
	bool all_stopped = true;
	for (int b = 0; b < mConnectedDevices.size(); b++) {
		if (isDevAcquiring(b)) {
			all_stopped = false;
			break;
		}
	}
	if (all_stopped)
		mControlPan->enableAllStartPb();
}

void WaveDump2::createReadoutThreadsAndStart() {
	QString val = mControlPan->getDevName();
	bool start_builder_th = false;
	RunID++;
	//create readout threads
	const QVector<QString>& devNames = mPlotPan->getDevToPlot();
	const QVector<int>& chanIdx = mPlotPan->getChanToPlot();

	if (mEventBuildthread == nullptr && !mFastSaveEnable) {
		mEventBuildthread = new eventBuildThread(this, &mConnectedDevices);
		connect(mEventBuildthread, SIGNAL(plotData(QVector< QVector< double > >, QVector< QVector< double > >, QVector<bool>, QVector<double>, double)), mPlotPan, SLOT(plotData(QVector< QVector< double > >, QVector< QVector< double > >, QVector<bool>, QVector<double>, double)));
		connect(mEventBuildthread, SIGNAL(finished()), this, SLOT(BuildThreadFinished()));
		for (int i = 0; i < devNames.size(); i++) {
			mEventBuildthread->updatePlots(i, devNames[i], chanIdx[i]);
		}
		mEventBuildthread->setUoMX(mUoMX);
		mEventBuildthread->setUoMY(mUoMY);
		start_builder_th = true;
	}

	if (val == "ALL") {
		for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
			mDevicesReadoutThreads.replace(dev, new deviceReadoutThread(this, mConnectedDevices.at(dev), RunID, 200));
			while (!mDevicesReadoutThreads.at(dev)->devQueueInited) {//wait for the device events queue to be inited
				QThread::msleep(100);
			}
			mDevicesReadoutThreads.at(dev)->startRun();
			mDevicesReadoutThreads.at(dev)->start();
			NDevStarted++;
		}
	}
	else {
		mDevicesReadoutThreads.replace(mActiveDevice->getIndex(), new deviceReadoutThread(this, mActiveDevice, RunID, 200));
		while (!mDevicesReadoutThreads.at(mActiveDevice->getIndex())->devQueueInited) {//wait for the device events queue to be inited
			QThread::msleep(100);
		}
		mDevicesReadoutThreads.at(mActiveDevice->getIndex())->startRun();//0
		mDevicesReadoutThreads.at(mActiveDevice->getIndex())->start();//0
		NDevStarted++;
	}

	mStatsPan->resetTime();

	if (start_builder_th) {
		QThread::msleep(300);
		mEventBuildthread->startRun();
		mEventBuildthread->start();
	}

	if (!Statistics_Timer->isActive())
		Statistics_Timer->start(1000);
}

void WaveDump2::startOfflineAcq() {	
	mControlPan->toogleStartStop();
	mPlotPan->resetSumCount();
	mLoadAct->setEnabled(false);
	mLoadMenuAct->setEnabled(false);
	mPlotPan->EnableEvDataTab(false);

	createReadoutThreadsAndStart();

	//UpdateStatus();
}

void WaveDump2::startAcquisition() {
	bool start_builder_th = false;
	QString val = mControlPan->getDevName();
	QString data_format = "WAVEFORMS";
	if (mFastSaveEnable)
		data_format = "RAW";

	this->setCursor(Qt::WaitCursor);

	///save a copy of the config file if only one board is connected///
	if (mConnectedDevices.size() == 1) {
		QString filedir = QDir::homePath() + DIRSEP + WAVEDUMP2_DIR + DIRSEP;
		QString last_config_filename = filedir + "WD2_" + QString("%1.wconf").arg(mActiveDevice->getSN());
		QFile* conf = new QFile(last_config_filename);
		if (conf->exists())
			conf->remove();
		delete conf;
		QFile::copy(mConfigFileName, last_config_filename);
	}
	//////////////////////////////

	mStatsPan->resetTime();

	NAutoTrg = 0;
	for (int b = 0; b < mConnectedDevices.size(); b++) {
		if (mConnectedDevices.at(b)->getTrgMode() == TRG_MODE_AUTO)
			NAutoTrg++;
	}
	mControlPan->toogleStartStop();
	mPlotPan->resetSumCount();
	mConfiguration->setEnabled(false);
	mOpenSettingsAct->setEnabled(false);
	mScriptMenu->setEnabled(false);
	mSaveasSettingsAct->setEnabled(false);
	mSaveasMenuAct->setEnabled(false);
	mSaveAct->setEnabled(false);
	mSaveMenuAct->setEnabled(false);
	mLoadAct->setEnabled(false);
	mLoadMenuAct->setEnabled(false);
	mResetButton->setEnabled(false);

	if (mRecordAction) {
		mActions->AddSWAction("START_ACQ", val);		
	}
	if(mRecordAction || mRecordStopTime)
		mRecordStopTimer.start();

	RunID++;
	//create readout threads
	const QVector<QString>& devNames = mPlotPan->getDevToPlot();
	const QVector<int>& chanIdx = mPlotPan->getChanToPlot();

	if (mEventBuildthread == nullptr && !mFastSaveEnable) {
		mEventBuildthread = new eventBuildThread(this, &mConnectedDevices);
		connect(mEventBuildthread, SIGNAL(plotData(QVector< QVector< double > >, QVector< QVector< double > >, QVector<bool>, QVector<double>, double)), mPlotPan, SLOT(plotData(QVector< QVector< double > >, QVector< QVector< double > >, QVector<bool>, QVector<double>, double)));
		connect(mEventBuildthread, SIGNAL(finished()), this, SLOT(BuildThreadFinished()));
		for (int i = 0; i < devNames.size(); i++) {
			mEventBuildthread->updatePlots(i, devNames[i], chanIdx[i]);
		}
		mEventBuildthread->setUoMX(mUoMX);
		mEventBuildthread->setUoMY(mUoMY);
		start_builder_th = true;
	}

	if (val == "ALL") {
		for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
			try {
				mConnectedDevices.at(dev)->SetDataFormat(data_format);
			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}
			mDevicesReadoutThreads.replace(dev, new deviceReadoutThread(this, mConnectedDevices.at(dev), RunID, 200));
			while (!mDevicesReadoutThreads.at(dev)->devQueueInited) {//wait for the device events queue to be inited
				QThread::msleep(100);
			}
			if (mFastSaveEnable) {
				mConnectedDevices.at(dev)->ClearData();
				mConnectedDevices.at(dev)->AcqStarted = true;
				mDevicesReadoutThreads.at(dev)->startRun();
				mDevicesReadoutThreads.at(dev)->start();
				NDevStarted++;
			}
		}
	}
	else {
		int check = mActiveDevice->getIndex();
		try {
			mActiveDevice->SetDataFormat(data_format);
		}
		catch (CAENFELibException& exc) {
			addLog(exc.ErrorMessage(), false);
		}
		mDevicesReadoutThreads.replace(mActiveDevice->getIndex(), new deviceReadoutThread(this, mActiveDevice, RunID, 200));
		while (!mDevicesReadoutThreads.at(mActiveDevice->getIndex())->devQueueInited) {//wait for the device events queue to be inited
			QThread::msleep(100);
		}
		if (mFastSaveEnable) {
			mActiveDevice->ClearData();
			mActiveDevice->AcqStarted = true;
			mDevicesReadoutThreads.at(mActiveDevice->getIndex())->startRun();//0
			mDevicesReadoutThreads.at(mActiveDevice->getIndex())->start();//0
			NDevStarted++;
		}
	}

	QThread::msleep(500);
	if (IsFastSaveEnabled()) {
		QMessageBox msgBox(this);
		msgBox.setText("<html><B>Creating raw data file header<br>           ... please wait ...</html>");
		msgBox.setStandardButtons({});
		msgBox.show();
		for (int i = 0; i < 50;i++) {
			QThread::msleep(100);
			this->increaseProgressBar();
			if(i%5 == 0)
				QCoreApplication::processEvents();
		}
		msgBox.close();
		mMainProgressBar->setValue(0);
	}

	//start data acquisition for the device(s)
	try {
		if (val == "ALL") {
			if (mGlobalStartMode == "SW_ASYNC") { //no sync, sw start for all boards one after the other
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					mConnectedDevices.at(b)->StartAcquisition();
					mLogPan->add2Log("CMD", "SwStartAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
				}
			}
			else if (mGlobalStartMode == "SW_CHAIN_SYNC-CLKIN" || mGlobalStartMode == "SW_SYNC-CLKIN" || mGlobalStartMode == "SW_CHAIN_SIN-GPIO" || mGlobalStartMode == "SW_CHAIN_SIN-TRGOUT") {
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					if (mConnectedDevices.at(b)->DevChainIndex != 0) {
						mConnectedDevices.at(b)->ArmAcquisition();
						mLogPan->add2Log("CMD", "ArmAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
					}
				}
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					if (mConnectedDevices.at(b)->DevChainIndex == 0) { //send start command to master board
						mConnectedDevices.at(b)->StartAcquisition();
						mLogPan->add2Log("CMD", "SwStartAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
						break;
					}
				}
			}
			else if (mGlobalStartMode == "SW_CHAIN_1ST_TRIGGER") {
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					mConnectedDevices.at(b)->ArmAcquisition();
					mLogPan->add2Log("CMD", "ArmAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
				}
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					if (mConnectedDevices.at(b)->DevChainIndex == 0) { //send sw trigger to master board
						mConnectedDevices.at(b)->SendSWTrg();
						mLogPan->add2Log("CMD", "SendSWTrigger", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
						break;
					}
				}
			}
			else if (mGlobalStartMode == "SW_CHAIN_TRGIN-TRGOUT" || mGlobalStartMode == "HW_CHAIN_TRGIN-TRGOUT" || mGlobalStartMode == "SW_CHAIN_ONE2ALL_EXTOR" || mGlobalStartMode == "HW_CHAIN_ONE2ALL_EXTOR") {
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					if (mConnectedDevices.at(b)->DevChainIndex != 0) {
						mConnectedDevices.at(b)->ArmAcquisition();
						mLogPan->add2Log("CMD", "ArmAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
					}
					else {
						if(mGlobalStartMode.contains("TRGIN-TRGOUT") && mConnectedDevices.at(b)->DevChainIndex == 0)
							static_cast<CAENDig1Device*>(mConnectedDevices.at(b))->disableExtTrgIn("TRUE");
					}
				}
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					if (mConnectedDevices.at(b)->DevChainIndex == 0) { //send start command to master board
						mConnectedDevices.at(b)->StartAcquisition();
						mConnectedDevices.at(b)->SendSWTrg();
						if (mGlobalStartMode.contains("TRGIN-TRGOUT") && mConnectedDevices.at(b)->DevChainIndex == 0)
							static_cast<CAENDig1Device*>(mConnectedDevices.at(b))->disableExtTrgIn("FALSE");
						mLogPan->add2Log("CMD", "SwStartAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
						break;
					}
				}
			}
			else {
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					mConnectedDevices.at(b)->ArmAcquisition();						
					mLogPan->add2Log("CMD", "ArmAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
				}
			}
		}
		else {
			mActiveDevice->StartAcquisition();
			mLogPan->add2Log("CMD", "SwStartAcquisition", "", mActiveDevice->getName(), CAEN_FELib_Success);
		}
	}
	catch (CAENFELibException& exc) {
		mControlPan->toogleStartStop();
		mConfiguration->setEnabled(true);
		//stopreadouts threads 
		for (int b = 0; b < mConnectedDevices.size(); b++) {
			if (mDevicesReadoutThreads.at(b) != nullptr) {
				if (mDevicesReadoutThreads.at(b)->isRunning()) {
					mDevicesReadoutThreads.at(b)->stopRun();
					mDevicesReadoutThreads.at(b)->wait();
				}
				else
					delete mDevicesReadoutThreads.at(b);
				mDevicesReadoutThreads[b] = nullptr;
			}
		}
		if (mEventBuildthread != nullptr) {
			mEventBuildthread->stopRun();
			mEventBuildthread->wait();
			delete mEventBuildthread;
			mEventBuildthread = nullptr;
		}
		this->setCursor(Qt::ArrowCursor);
		addLog(exc.ErrorMessage(), false);
		return;
	}

	
	
	//start readout threads
	if (!mFastSaveEnable) {
		if (val == "ALL") {
			for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
				mDevicesReadoutThreads.at(dev)->startRun();
				mDevicesReadoutThreads.at(dev)->start();
				NDevStarted++;
			}
		}
		else {
			mDevicesReadoutThreads.at(mActiveDevice->getIndex())->startRun();//0
			mDevicesReadoutThreads.at(mActiveDevice->getIndex())->start();//0
			NDevStarted++;
		}
	}

	if (start_builder_th) {
		mEventBuildthread->startRun();
		mEventBuildthread->start();
	}

	if(!Statistics_Timer->isActive())
		Statistics_Timer->start(1000);	
	QThread::msleep(300);
	UpdateStatus();
	if(!Statistics_Timer->isActive())
		Statistics_Timer->start(1300);		
	this->setCursor(Qt::ArrowCursor);
}

void WaveDump2::stopAcquisition() {
	QString val = mControlPan->getDevName();

	mControlPan->toogleStartStop();
	mConfiguration->setEnabled(true);
	mOpenSettingsAct->setEnabled(true);
	mScriptMenu->setEnabled(true);
	mSaveasSettingsAct->setEnabled(true);
	mSaveasMenuAct->setEnabled(true);
	mSaveAct->setEnabled(true);
	mSaveMenuAct->setEnabled(true);
	mLoadAct->setEnabled(true);
	mLoadMenuAct->setEnabled(true);
	mResetButton->setEnabled(true);

	this->setCursor(Qt::WaitCursor);
	if (mRecordAction) {
		qint64 acq_time = mRecordStopTimer.elapsed();
		mActions->AddSWAction("STOP_ACQ", val + " " + QString("%1").arg(acq_time));
	}
	try {
		if (val == "ALL") {
			if (mGlobalStartMode == "SW_ASYNC") {
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					mConnectedDevices.at(b)->StopAcquisition();
					mLogPan->add2Log("CMD", "SwStopAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
				}
			}
			else if (mGlobalStartMode == "SW_CHAIN_SYNC-CLKIN" || mGlobalStartMode == "SW_CHAIN_SIN-GPIO" || mGlobalStartMode == "SW_CHAIN_SIN-TRGOUT" || mGlobalStartMode == "SW_CHAIN_TRGIN-TRGOUT") {
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					if (mConnectedDevices.at(b)->DevChainIndex == 0) {
						mConnectedDevices.at(b)->StopAcquisition();
						mLogPan->add2Log("CMD", "SwStopAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
					}
					else {
						mConnectedDevices.at(b)->DisarmAcquisition();
						mLogPan->add2Log("CMD", "DisArmAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
					}
				}
			}
			else {
				for (int b = 0; b < mConnectedDevices.size(); b++) {
					mConnectedDevices.at(b)->DisarmAcquisition();
					mLogPan->add2Log("CMD", "DisArmAcquisition", "", mConnectedDevices.at(b)->getName(), CAEN_FELib_Success);
				}
			}
			//stopreadouts threads 
			for (int b = 0; b < mConnectedDevices.size(); b++) {
				if (mDevicesReadoutThreads.at(b) != nullptr) {
					mDevicesReadoutThreads.at(b)->stopRun();
					mDevicesReadoutThreads.at(b)->wait();
					mDevicesReadoutThreads[b] = nullptr;
				}
			}
		}
		else {	
			mActiveDevice->StopAcquisition();
			mLogPan->add2Log("CMD", "SwStopAcquisition", "", mActiveDevice->getName(), CAEN_FELib_Success);
			if (mDevicesReadoutThreads.at(mActiveDevice->getIndex()) != nullptr) {
				mDevicesReadoutThreads.at(mActiveDevice->getIndex())->stopRun();
				mDevicesReadoutThreads.at(mActiveDevice->getIndex())->wait();
				mDevicesReadoutThreads[mActiveDevice->getIndex()] = nullptr;
			}
		}
	}
	catch (CAENFELibException& exc) {
		mControlPan->toogleStartStop();
		addLog(exc.ErrorMessage(), false);
	}

	UpdateStatus();
	checkStartPbEnable();
	this->setCursor(Qt::ArrowCursor);	
}

void WaveDump2::BuildThreadFinished(){
	delete mEventBuildthread;
	mEventBuildthread = nullptr;
	Statistics_Timer->stop();
	UpdateStatsAndStatus(); //final refresh
	UpdateStatsAndStatus(); //final refresh
	if (OfflineMode && mControlPan->getDevName() == "ALL") {
		mControlPan->toogleStartStop();
		for (int b = 0; b < mConnectedDevices.size(); b++) {
			if (mDevicesReadoutThreads.at(b) != nullptr) {
				mDevicesReadoutThreads.at(b)->stopRun();
				mDevicesReadoutThreads.at(b)->wait();
			}
		}
	}
	checkStartPbEnable();
}

void WaveDump2::ReadoutError(QString devname, CAEN_FELib_ErrorCode err) {
	stopAcquisition();
	QMessageBox::critical(this, tr("Readout error"), tr("ReadData error from device ") + devname + QString(" :error %1").arg(err), QMessageBox::Ok, 0);
}

void WaveDump2::ReadoutError1(QString error) {
	stopAcquisition();
	QMessageBox::critical(this, tr("Readout error"), error, QMessageBox::Ok, 0);
}

void WaveDump2::ReaderThreadFinished(QString devname) {
	addLog("Readout stopped for "+devname,true);
	if(OfflineMode && mControlPan->getDevName() == devname)
		mControlPan->toogleStartStop();
	for (int b = 0; b < mConnectedDevices.size(); b++) {
		if (mConnectedDevices.at(b)->getName() == devname) {
			delete mDevicesReadoutThreads[b];
			mDevicesReadoutThreads[b] = nullptr;//the thread of this device has already finished
			if (mConnectedDevices.at(b)->AcqStarted && !mConnectedDevices.at(b)->getStartSource().contains("SIN", Qt::CaseInsensitive)) { //acq has not been stopped by the sw				
				stopAcquisition();
				QMessageBox::warning(this, tr("Warning"), "Unexpected acquisition stop for device "+devname + ".\n Readout stopped. ");
			}
		}
	}
	if (mFastSaveEnable) {
		bool all_stopped = true;
		for (int b = 0; b < mConnectedDevices.size(); b++) {
			if (mConnectedDevices.at(b)->AcqStarted) {
				all_stopped = false;
				break;
			}
		}
		if (Statistics_Timer->isActive() && all_stopped) {
			Statistics_Timer->stop();
			UpdateStatsAndStatus(); //final refresh
		}
	}
}

void WaveDump2::tracesChanged(QVector<QString> dev, QVector<int> ch) {
	if (mEventBuildthread == nullptr)
		return;
	QVector<QString> chan;
	for (int i = 0; i < this->mPlotPan->getTraces(); i++) {
		mEventBuildthread->updatePlots(i, dev[i], ch[i]); 
		chan += QString("CH%1").arg(QString::number(i));
	}

}

void WaveDump2::showManualController(std::uint64_t handle, const QString &rootName, const QString &Qry, bool f) {
	mManualControllerPan->showData(handle, rootName, Qry, f);
	if (!mManualControllerAct->isChecked()) {
		mManualControllerAct->setChecked(true);
		mManualControllerDock->show();
	}
}

void WaveDump2::scanDevices() {
	QMessageBox::information(this, tr("Info"),tr("Not Yet Implemented"));
		return;

	discoveryDevicesDialog dialog(this);
	dialog.setWindowFlags(dialog.windowFlags() & ~Qt::WindowContextHelpButtonHint);
	dialog.setModal(true);
	dialog.exec();
	if (!dialog.getAddress().isEmpty())
		openDeviceExec(dialog.getType(), dialog.getAddress(), "", true);
	CheckConnectedDevStatus();
}

QString WaveDump2::getConfigValue(const QString& section, const QString& key) {
	QString value = "";
	mConfigFile->beginGroup(section);
	value = mConfigFile->value(key).toString();
	mConfigFile->endGroup();
	return value;
}

void WaveDump2::setConfigValue(const QString& section, const QString& key, const QString& value) {
	mConfigFile->beginGroup(section);
	mConfigFile->setValue(key,value);
	mConfigFile->endGroup();
	handleConfFileModified();
}

void WaveDump2::handleConfFileModified() {
	mSaveAct->setEnabled(true);
	mSaveasAct->setEnabled(true);
	mSaveMenuAct->setEnabled(true);
	mSaveasMenuAct->setEnabled(true);
	mCouldSave = true;
}

void WaveDump2::saveOutput() {
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
	mDataSavedEnabled.store((mSaveOutputAct->isChecked()) ? 1 : 0);
#else
	mDataSavedEnabled.storeRelaxed((mSaveOutputAct->isChecked()) ? 1 : 0);
#endif
	mOutputConfigAct->setEnabled(!mSaveOutputAct->isChecked());
	if(mRecordAction)
		mActions->AddSWAction("DATA_SAVE_ENABLE", (mDataSavedEnabled) ? "TRUE" : "FALSE");

	if (mFastSaveEnable) {
		mPlotPan->ShowPaletteCmd(false);
		mPlotPan->ShowToolsCmd(false);
		mPlotPan->ClearPlot();
		mStatistics->clear();
	}
	if (!mDataSavedEnabled && mFastSaveEnable)
		mFastSaveEnable = false;
}

bool WaveDump2::isSaveEnabled() {
	return mSaveOutputAct->isChecked();
}

void WaveDump2::setOutputSettings(QString folder, QString prefix, QString ftype, QString header, QString format, QString sync, int nevts, bool saveconfig) {
	if (folder == "") folder = QDir::homePath();
	if (prefix == "") prefix = "Acq";
	if (ftype == "") ftype = "SINGLE";
	if (header == "") header = "YES";
	if (format == "") format = "ASCII";
	mOutput_folder = folder;
	mOutput_prefix = prefix;
	mOutput_ftype = ftype;
	mOutput_header = header;
	mOutput_format = format;
	mOutput_sync = sync;
	mNEvts_file = nevts;
	if (saveconfig) {
		this->setConfigValue("OUTPUT", "folder", mOutput_folder);
		this->setConfigValue("OUTPUT", "prefix", mOutput_prefix);
		this->setConfigValue("OUTPUT", "ftype", mOutput_ftype);
		this->setConfigValue("OUTPUT", "header", mOutput_header);
		this->setConfigValue("OUTPUT", "format", mOutput_format);
		this->setConfigValue("OUTPUT", "sync", mOutput_sync);
		this->setConfigValue("OUTPUT", "nevts", QString("%1").arg(mNEvts_file));
	}
	if (mRecordAction) {
		mActions->AddSWAction("SAVE_FOLDER_CHANGE", mOutput_folder);
		mActions->AddSWAction("FILE_PREFIX_CHANGE", mOutput_prefix);
		mActions->AddSWAction("FILE_TYPE_CHANGE", mOutput_ftype);
		mActions->AddSWAction("FILE_FORMAT_CHANGE", mOutput_format);
		mActions->AddSWAction("FILE_HEADER_ENABLE", (mOutput_header == "YES")? "TRUE" : "FALSE");
		mActions->AddSWAction("FILE_SYNC_UNIQUE", (mOutput_sync == "YES") ? "TRUE" : "FALSE");
		mActions->AddSWAction("FILE_MAX_NEVTS", QString("%1").arg(mNEvts_file));
	}

}

void WaveDump2::getOutputSettings(QString &folder, QString &prefix, QString &ftype, QString &header, QString &format, QString &sync, int &nevts) {
	folder = mOutput_folder;
	prefix = mOutput_prefix;
	ftype = mOutput_ftype;
	header = mOutput_header;
	format = mOutput_format;
	sync = mOutput_sync;
	nevts = mNEvts_file;
}

void WaveDump2::configureSetup(){
	if (Check_connection_Timer->isActive())
		Check_connection_Timer->stop();

	SystemSettings* settings = new SystemSettings(this, &mConnectedDevices);
	settings->exec();
	settings->setModal(true);

	if (!OfflineMode) {
		RefreshDevSettings();

		if (!Check_connection_Timer->isActive())
			Check_connection_Timer->start(3000);
	}
	else
		mControlPan->UpdateStartOptions();
}

void WaveDump2::RefreshDevSettings() {
	if (mActiveDevice != nullptr) {
		updateActiveDevInfo(mActiveDevice);
		mControlPan->updateSmodeSettings();
		if (mControlPan->getDevName() == "ALL")
			ActiveDeviceChanged("ALL");
			//PreTrgChanged(mActiveDevice->getPreTrgCachedValue() * mActiveDevice->getDecimation() / 1000.);
		for (int b = 0; b < mConnectedDevices.size(); b++) {
			mConnectedDevices.at(b)->updateBaselines();
			mConnectedDevices.at(b)->updateGains();
			mPlotPan->updateGains(mConnectedDevices.at(b)->getName(), mConnectedDevices.at(b)->ChGain);
			QVector<double>sampl_to_V;
			for (int ch = 0; ch < mConnectedDevices.at(b)->getNumCh(); ch++) {
				sampl_to_V.append(mConnectedDevices.at(b)->getSample_to_V(ch));
				mPlotPan->updateOffsetInfo1(mConnectedDevices.at(b)->getName(), ch, mConnectedDevices.at(b)->ChBaselines.at(ch));				
			}
			mPlotPan->updateBaselines(mConnectedDevices.at(b)->getName(), mConnectedDevices.at(b)->getADCNBit(), sampl_to_V, mConnectedDevices.at(b)->ChBaselines);
			if (mConnectedDevices.at(b)->getName() == mPlotPan->getActiveTraceDev()) {
				mPlotPan->updateOffsetInfo(mConnectedDevices.at(b)->getChOffset(mPlotPan->getActiveTraceCh()));
				CAENparameter* thr_ch = mConnectedDevices.at(b)->getThrParam(mPlotPan->getActiveTraceCh());
				mConnectedDevices.at(b)->updateChSample_to_V(mPlotPan->getActiveTraceCh());
				if (mActiveDevice->DeviceClass == CAEN_DIG2)
					mPlotPan->updateThrInfo(thr_ch->getMax(), thr_ch->getMin(), thr_ch->getIncr(), thr_ch->getValue().toDouble(), mConnectedDevices.at(b)->getSample_to_V(mPlotPan->getActiveTraceCh()) * 1e3);
				else {
					mPlotPan->updateThrInfo(thr_ch->getMax(), thr_ch->getMin(), thr_ch->getIncr(), thr_ch->getValue().toDouble(), mConnectedDevices.at(b)->getVRangeV(mPlotPan->getActiveTraceCh()) * 1e3);
				}
			}
		}
		
	}
}

int WaveDump2::getADCNBit(QString dev_name) {
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == dev_name) {
			return mConnectedDevices.at(i)->getADCNBit();			
		}
	}
	return 0;
}

int WaveDump2::getActivePlotTraceDevIndex() {
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == mPlotPan->getActiveTraceDev()) {
			return i;
		}
	}
	return 0;
}

int WaveDump2::getPlotTraceDevIndex(int trace) {
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == mPlotPan->getTraceDev(trace)) {
			return i;
		}
	}
	return 0;
}

CAENDevice* WaveDump2::getPlotTraceDev(int trace) {
	QString name = mPlotPan->getTraceDev(trace);
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == name) {
			return mConnectedDevices.at(i);
		}
	}
	return nullptr;
}

void WaveDump2::configureDevice(QString deviceName) {
	if (Check_connection_Timer->isActive())
		Check_connection_Timer->stop();
	QString family;
	caen::hash::generator hasher;
	int dev = 0;
	for(dev=0; dev<mConnectedDevices.size(); dev++){
		if (mConnectedDevices.at(dev)->getName() == deviceName) {
			family = mConnectedDevices.at(dev)->getFamilyCode();			
			break;
		}	
	}

	uint64_t hash = hasher(family.toUtf8().constData());
	switch (hasher(family.toUtf8().constData())) {
		using namespace caen::hash::literals;
	case "2740"_h:
	case "2745"_h:     //x2745 almost identical to x2740 --> use the same gui with the extra parameter VgaGain
	case "2730"_h:     //x2730 almost identical to x2740 --> use the same gui with the extra parameter ChGain
	case "2751"_h:
		{	
		mConnectedDevices.at(dev)->setThrMode("Relative");
		Dx740V2ConfigPanel dialog(this, static_cast<CAENDig2Device*>(mConnectedDevices.at(dev)), deviceName, true);
		dialog.setModal(true);
		dialog.exec();
		}
		break;
	case "XX740"_h:
	case "XX730"_h:
	case "XX725"_h:
	case "XX751"_h:
	case "XX724"_h:
	case "XX720"_h:
		{
		Dx740V1ConfigPanel dialog(this, static_cast<CAENDig1Device*>(mConnectedDevices.at(dev)), deviceName, true);
		dialog.setModal(true);
		dialog.exec();
		}
		break;
	default:
		QMessageBox::information(this, tr("Configure ") + deviceName,
			tr("The configuration panel for this family board is not implemented in this version\n Please check if there is a newer version of this software in CAEN Website\n If not please contact CAEN software division"), QMessageBox::Ok);
		break;
	}

	if (mActiveDevice->getName() == deviceName)
		CheckSettings();
	else
		RefreshDevSettings();
	if (!Check_connection_Timer->isActive())
		Check_connection_Timer->start(3000);

}

//timer slot for Auto trigger
void WaveDump2::SWTrg() {
	try {
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			if(mConnectedDevices.at(i)->getTrgMode() == TRG_MODE_AUTO && mConnectedDevices.at(i)->AcqStarted && !mConnectedDevices.at(i)->Triggered)
				mConnectedDevices.at(i)->SendSWTrg();
		}
		}
	catch (CAENFELibException& exc) {
					addLog(exc.ErrorMessage(), false);
	}			
}

//manual single sw trigger
void WaveDump2::SingleSWTrg() {
	if (mControlPan->getDevName() == "ALL") {
		if (mRecordAction)
			mActions->AddSWAction("SEND_SW_TRIGGER", "ALL");
		try {
			for (int i = 0; i < mConnectedDevices.size(); i++)
				mConnectedDevices.at(i)->SendSWTrg();
		}
		catch (CAENFELibException& exc) {
			addLog(exc.ErrorMessage(), false);
		}
	}
	else {
		if (mRecordAction)
			mActions->AddSWAction("SEND_SW_TRIGGER", mActiveDevice->getName());
		try {
			mActiveDevice->SendSWTrg();
		}
		catch (CAENFELibException& exc) {
			addLog(exc.ErrorMessage(), false);
		}
	}
}

void WaveDump2::ClearPlotData() {
	if (mEventBuildthread != nullptr)
		mEventBuildthread->ClearPlotArray();
}

void WaveDump2::New_Cursors(bool enable) {
	mPlotPan->NewCursors(enable);
}

void WaveDump2::showCursorsPositions(QVector<double> positions) {
	mControlPan->UpdateCursorsData(positions);
}

void WaveDump2::cursorTraceChanged(int C, int trace) {
	if (C == 1)
		mPlotPan->setC1trace(trace);
	else if (C == 2)
		mPlotPan->setC2trace(trace);
}

void WaveDump2::XUnitChanged(int unit) {
	mUoMX = unit;
	CAENparameter *reclen_param = nullptr,  *max_reclen_param_S = nullptr;
	int active_dev = 0;
	double active_tr_pretrg_valueS=0;
	double factor_s=1;
	uint32_t max_reclen_S = 0, s;
	int extra_samples = 0;
	double Sample_to_s=1;
	double PreTrg=1;
	int Decimation = 1;
	double factor = 1.;
	try {

		for (int i = 0; i < mConnectedDevices.size(); i++) {//pretrg marker always relative to active trace
			if (mConnectedDevices.at(i)->getName() == mPlotPan->getActiveTraceDev()) {
				active_tr_pretrg_valueS = mConnectedDevices.at(i)->getPreTrg_valueS();
				Sample_to_s = mConnectedDevices.at(i)->getSample_to_S();
				Decimation = mConnectedDevices.at(i)->getDecimation();
				break;
			}
		}

		if (mControlPan->getDevName() == "ALL") {
			reclen_param = mConnectedDevices.at(0)->getReclenParam();
			active_dev = 0;
			for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
				s = mConnectedDevices.at(dev)->getReclen_ns() / (mConnectedDevices.at(dev)->getSample_to_S() * 1e9);
				if (s > max_reclen_S) {
					max_reclen_S = s;
					max_reclen_param_S = mConnectedDevices.at(dev)->getReclenParam();
					extra_samples = mConnectedDevices.at(dev)->RecLenJIncrS;
				}
			}
		}
		else {
			for (int i = 0; i < mConnectedDevices.size(); i++) {
				if (mConnectedDevices.at(i)->getName() == mControlPan->getDevName()) {
					reclen_param = mConnectedDevices.at(i)->getReclenParam();
					active_dev = i;
				}
				s = (mConnectedDevices.at(i)->getReclen_ns() / (mConnectedDevices.at(i)->getSample_to_S() * 1e9));
				if ( s > max_reclen_S) {
					max_reclen_param_S = mConnectedDevices.at(i)->getReclenParam();
					factor_s = 1 / (mConnectedDevices.at(i)->getSample_to_S() * 1e9);
					extra_samples = mConnectedDevices.at(i)->RecLenJIncrS;
				}
			}
		}
		
		switch (mUoMX) {
		case UOM_SAMPLE:
			mPlotPan->setXAxis(0, static_cast<double>(getMaxRecLenS()), mUoMX);
			if (max_reclen_param_S != nullptr)
				mPlotPan->updateXPosInfo(max_reclen_param_S->getMax(), max_reclen_param_S->getMin(), max_reclen_param_S->getIncr(), max_reclen_param_S->getValue().toDouble() - (extra_samples*factor_s), factor_s);
			
			mPlotPan->calcNewCursorPositions(1 / (Sample_to_s * Decimation * 1e6));
			mControlPan->updateReclenWidget(mConnectedDevices.at(active_dev)->getReclen_maxS(), mConnectedDevices.at(active_dev)->getReclen_minS(), mConnectedDevices.at(active_dev)->getReclen_incrS(), mConnectedDevices.at(active_dev)->getReclen_valueS());
			if (mConnectedDevices.at(active_dev)->DeviceClass == CAEN_DIG1)
				factor = 1. / mConnectedDevices.at(active_dev)->getDecimation_value();//show value in reclen samples (decimated)
			else
				factor = 1.;
			mPlotPan->updatePreTrgInfo(active_tr_pretrg_valueS * factor);
			mControlPan->updatePreTrgWidget(mConnectedDevices.at(active_dev)->getPreTrg_maxS(), mConnectedDevices.at(active_dev)->getPreTrg_minS(), mConnectedDevices.at(active_dev)->getPreTrg_incrS(), mConnectedDevices.at(active_dev)->getPreTrg_valueS()*factor);
			break;
		case UOM_PHYS_UNIT:
			factor = getMaxDecimation() / 1e3;
			mPlotPan->setXAxis(0, getMaxRecLen_us(), mUoMX);//us			
			mPlotPan->calcNewCursorPositions(Sample_to_s * Decimation * 1e6);
			mControlPan->updateReclenWidget(mConnectedDevices.at(active_dev)->getReclen_max()*factor, mConnectedDevices.at(active_dev)->getReclen_min() *factor, mConnectedDevices.at(active_dev)->getReclen_incr() *factor, mConnectedDevices.at(active_dev)->getReclen_value() *factor);
			if (mConnectedDevices.at(active_dev)->DeviceClass == CAEN_DIG1) {
				factor = 1. / 1e3;
				mPlotPan->updatePreTrgInfo(active_tr_pretrg_valueS * Sample_to_s * 1e6);
			}
			else
				mPlotPan->updatePreTrgInfo(active_tr_pretrg_valueS * Sample_to_s * Decimation * 1e6);
			mControlPan->updatePreTrgWidget(mConnectedDevices.at(active_dev)->getPreTrg_max()*factor, mConnectedDevices.at(active_dev)->getPreTrg_min() *factor, mConnectedDevices.at(active_dev)->getPreTrg_incr() *factor, mConnectedDevices.at(active_dev)->getPreTrg_value() *factor);
			break;
		}
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	mPlotPan->showCursorPositions();

	if (mEventBuildthread != nullptr) {
		mEventBuildthread->setUoMX(unit);
	}
	if(mRecordAction)
		mActions->AddSWAction("XUNIT_CHANGE", QString("%1").arg(mUoMX));
}

void WaveDump2::HandleThrMode() {
	if (mPlotPan->getPlotScale() == PLOT_SCALE_RELATIVE) {
		for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
			try {
				mConnectedDevices.at(dev)->setThrMode("Relative");
			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}
		}
	}
	else {
		for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
			try {
				mConnectedDevices.at(dev)->setThrMode("Absolute");
			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}
		}
	}
}

void WaveDump2::YUnitChanged(int unit) {
	mUoMY = unit;
	CAENparameter* thr_ch=nullptr;
	double Thr=0;//serve thr del device corrispondente alla traccia attiva
	CAENDevice* dev;
	try {
		for (int i = 0; i < mConnectedDevices.size(); i++) {//thr always relative to active trace
			if (mConnectedDevices.at(i)->getName() == mPlotPan->getActiveTraceDev()) {
				if (!mConnectedDevices.at(i)->isVirtual())
					thr_ch = mConnectedDevices.at(i)->getThrParam(mPlotPan->getActiveTraceCh());
				else
					Thr = mConnectedDevices.at(i)->getThrCachedValue(mPlotPan->getActiveTraceCh());
				break;
			}
		}
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	switch (mUoMY) {
	case UOM_SAMPLE:
		updatePlotYAxisRange();
		if(!OfflineMode)
			mPlotPan->updateThrInfo(thr_ch->getMax(), thr_ch->getMin(), thr_ch->getIncr(), thr_ch->getValue().toDouble(), 1.);
		else
			mPlotPan->updateThrInfo(1e9, 0, 1, Thr, 1.);
		break;
	case UOM_PHYS_UNIT:
		updatePlotYAxisRange();
	if (!OfflineMode){
		if (mActiveDevice->DeviceClass == CAEN_DIG2)
			mPlotPan->updateThrInfo(thr_ch->getMax(), thr_ch->getMin(), thr_ch->getIncr(), thr_ch->getValue().toDouble(), mActiveDevice->getSample_to_V(mPlotPan->getActiveTraceCh()) * 1e3);
		else {
			mPlotPan->updateThrInfo(thr_ch->getMax(), thr_ch->getMin(), thr_ch->getIncr(), thr_ch->getValue().toDouble(), mActiveDevice->getVRangeV(mPlotPan->getActiveTraceCh()) * 1e3);
		}
	}
	else
		mPlotPan->updateThrInfo(1e9, -1e9, 0.1, Thr, 1.);
		break;
	}

	mPlotPan->showCursorPositions();

	if (mEventBuildthread != nullptr) {
		mEventBuildthread->setUoMY(unit);
	}
	if (mRecordAction)
		mActions->AddSWAction("YUNIT_CHANGE", QString("%1").arg(mUoMY));
}

void WaveDump2::PlotScaleChanged() {
	HandleThrMode();
	updatePlotYAxisRange();
}

void WaveDump2::updatePlotXAxisRange() {
	if (mPlotPan->PlotType == PLOT_TYPE_FFT) {
		mPlotPan->setXAxis(0, static_cast<double>(getMaxRecLenS()), mUoMX);
		mPlotPan->setYAxis(-140, 0, mUoMY);
	}
	else if (mPlotPan->PlotType == PLOT_TYPE_SAMPLES_HISTO) {//switch x and y axes limits
		switch (mUoMY) {
		case UOM_SAMPLE:
			mPlotPan->setXAxis(0, static_cast<double>(getMaxVRangeS()), mUoMY);
			break;
		case UOM_PHYS_UNIT:
			mPlotPan->setXAxis(-(getMaxVRangeVForPlot() * 1e3) / 2., (getMaxVRangeVForPlot() * 1e3) / 2., mUoMY);
			break;
		}
		mPlotPan->setYAxis(0, 10000, mUoMY);
	}
	else if (mPlotPan->PlotType == PLOT_TYPE_WAVEFORMS) {
		switch (mUoMX) {
		case UOM_SAMPLE:
			mPlotPan->setXAxis(0, static_cast<double>(getMaxRecLenS()), mUoMX);
			break;
		case UOM_PHYS_UNIT:
			mPlotPan->setXAxis(0, getMaxRecLen_us(), mUoMX);
			break;
		}
		updatePlotYAxisRange();
	}

}

void WaveDump2::PlotTypeChanged() {
	baseline_changing = true;
	QThread::msleep(400);
	updatePlotXAxisRange();
	mPlotType = mPlotPan->PlotType;
	if(mEventBuildthread != nullptr)
		mEventBuildthread->setUoMX(mUoMX);
	mControlPan->updatePlotType(mPlotPan->PlotType);
	if (mRecordAction)
		mActions->AddSWAction("PLOTTYPE_CHANGE", QString("%1").arg(mPlotType));
	baseline_changing = false;
}

void WaveDump2::updatePlotYAxisRange() {
	if (mPlotPan->PlotType != PLOT_TYPE_WAVEFORMS)
		return;
	double range_active_dev=1; //rangemax, 
	double active_trace_bl=0;
	switch (mUoMY) {
		case UOM_SAMPLE:
			mPlotPan->setYAxis(0, getMaxVRangeS(), mUoMY);
			break;
		case UOM_PHYS_UNIT:
			//rangemax = getMaxVRangeVForPlot() * 1e3;
			for (int i = 0; i < mConnectedDevices.size(); i++) {
				if (mConnectedDevices.at(i)->getName() == mPlotPan->getActiveTraceDev()) {
					if (!mConnectedDevices.at(i)->ErrorsReported) {
						range_active_dev = mConnectedDevices.at(i)->getVRangeV(mPlotPan->getActiveTraceCh()) * 1e3;
						active_trace_bl = mConnectedDevices.at(i)->ChBaselines[mPlotPan->getActiveTraceCh()];
					}
					break;
				}
			}
			mPlotPan->setYAxis(0 - (range_active_dev * active_trace_bl / 100.), range_active_dev - (range_active_dev * active_trace_bl / 100.), mUoMY);//mV
			break;
	}
}

int WaveDump2::getMaxDecimation() {
	uint32_t max = 0, value;
	for (int t = 0; t < mPlotPan->getTraces(); t++) {
		if (!mPlotPan->isTraceVisible(t))
			continue;
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			if (mPlotPan->getTraceDev(t) == mConnectedDevices.at(i)->getName()) {
				value = mConnectedDevices.at(i)->getDecimation();
				if (value > max)
					max = value;
				break;
			}
		}
	}
	if (max == 0)
		return mActiveDevice->getDecimation();
	return max;
}

uint32_t WaveDump2::getALLMaxRecLenS() {
	uint32_t max = 0, value;
	try {
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			value = static_cast<uint32_t>(mConnectedDevices.at(i)->getReclen_valueS());
			if (value > max)
				max = value;
		}
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	return max;
}

uint32_t WaveDump2::getALLMaxRecLenS_corr() {
	uint32_t max = 0, value;
	try {
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			value = static_cast<uint32_t>(mConnectedDevices.at(i)->getReclen_valueS());
			value += mConnectedDevices.at(i)->RecLenJIncrS;
			if (value > max)
				max = value;
		}
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	return max;
}

uint32_t WaveDump2::getMaxRecLenS() {
	uint32_t max = 0, value;
	try {
		for (int t = 0; t < mPlotPan->getTraces(); t++) {
			if (!mPlotPan->isTraceVisible(t))
				continue;
			for (int i = 0; i < mConnectedDevices.size(); i++) {
				if (mPlotPan->getTraceDev(t) == mConnectedDevices.at(i)->getName()) {
					value = static_cast<uint32_t>(mConnectedDevices.at(i)->getReclen_valueS());
					if (value > max)
						max = value;
					break;
				}
			}
		}
		if (max == 0)
			return static_cast<uint32_t>(mActiveDevice->getReclen_valueS());
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	return max;
}

double WaveDump2::getMaxRecLen_us() {
	double max = 0, value;
	try {
		for (int t = 0; t < mPlotPan->getTraces(); t++) {
			if (!mPlotPan->isTraceVisible(t))
				continue;

				for (int i = 0; i < mConnectedDevices.size(); i++) {
					if (mPlotPan->getTraceDev(t) == mConnectedDevices.at(i)->getName()) {
						value = mConnectedDevices.at(i)->getReclen_ns() / 1000. * mConnectedDevices.at(i)->getDecimation_value();
						if (value > max)
							max = value;
						break;
					}
				}
		}
		if (max == 0)
			return mActiveDevice->getReclen_ns() /1000. * mActiveDevice->getDecimation_value();
		}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	return max;
}

double WaveDump2::getMaxSampleToS() {
	double max = 0, value;
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		value = mConnectedDevices.at(i)->getSample_to_S();
		if (value > max)
			max = value;
	}
	return max;
}

uint32_t WaveDump2::getMaxVRangeS() {
	uint32_t max = 0, r;
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		r = mConnectedDevices.at(i)->getVRangeS();
		if (r > max)
			max = r;
	}
	return max;
}


double WaveDump2::getMaxVRangeVForPlot() { //max Y in Volts calculated considering the current enabled traces in the plot
	double max = 0, value;
	for (int t = 0; t < mPlotPan->getTraces(); t++) {
		if (!mPlotPan->isTraceVisible(t))
			continue;
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			if (mPlotPan->getTraceDev(t) == mConnectedDevices.at(i)->getName()) {
				if (mConnectedDevices.at(i)->ErrorsReported)
					continue;
				value = mConnectedDevices.at(i)->getVRangeV(mPlotPan->getTraceCh(t));
				if (value > max)
					max = value;
				break;
			}
		}
	}
	return max;
}

double WaveDump2::getDevChSampleToV(QString devname, int ch) {
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == devname) {
			if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG1)
				mConnectedDevices.at(i)->getChGain(ch);
			mConnectedDevices.at(i)->updateChSample_to_V(ch);
			return mConnectedDevices.at(i)->getSample_to_V(ch);
		}
	}
	return 1;
}

double WaveDump2::getDevChGain(QString devname, int ch) {
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == devname)
			return mConnectedDevices.at(i)->getChGain(ch);
	}
	return 1;
}

double WaveDump2::getThrSampleTomV_f(QString devname, int ch) {
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == devname) {
			if (ch >= mConnectedDevices.at(i)->getNumCh())
				return 1.;
			if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
				return mConnectedDevices.at(i)->getSample_to_V(ch) * 1e3;
			else {
				double range_dev = mConnectedDevices.at(i)->getVRangeV(ch) * 1e3;
				double ch_bl = mConnectedDevices.at(i)->getChOffset(ch);
				return  range_dev / mConnectedDevices.at(i)->getThr_max(ch);
			}
			break;
		}
	}
	return 1.;
}

double WaveDump2::getThrSampleTomV_off(QString devname, int ch) {
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == devname) {
			if (ch >= mConnectedDevices.at(i)->getNumCh())
				return 0.;
			if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
				return 0;
			else {
				double range_dev = mConnectedDevices.at(i)->getVRangeV(ch) * 1e3;
				double ch_bl = mConnectedDevices.at(i)->getChOffset(ch);
				double minY_value = (0 - (range_dev * ch_bl / 100.));
				return  minY_value;
			}
			break;
		}
	}
	return 0.;
}


void WaveDump2::ReclenChanged(double reclen) {
	QString value_ns = QString("%1").arg(reclen);
	QString pretrg_value_ns;

	if (mControlPan->getDevName() == "ALL") {//ALL selected, to set reclen to all boards
		for (int i = 0; i < mConnectedDevices.size(); i++) {
				try {
					if (mUoMX == UOM_PHYS_UNIT) {
						if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
							value_ns = QString("%1").arg(reclen * 1000. / getDevDecimation(mConnectedDevices.at(i)));
						else {
							value_ns = QString("%1").arg(reclen * 1000.);
							pretrg_value_ns = QString("%1").arg(mControlPan->getPreTrg() * 1000.);
						}
					}
					else {
						if(mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
							value_ns = QString("%1").arg(reclen * (mConnectedDevices.at(i)->getSample_to_S() * 1e9) / mConnectedDevices.at(i)->getDecimation_value());
						else {
							value_ns = QString("%1").arg(reclen * (mConnectedDevices.at(i)->getSample_to_S() * 1e9));
							pretrg_value_ns = QString("%1").arg(mControlPan->getPreTrg() * (mConnectedDevices.at(i)->getSample_to_S() * 1e9) * mConnectedDevices.at(i)->getDecimation_value());
						}
					}
					mConnectedDevices.at(i)->setReclenT(value_ns);
					manageAddLogAndConf("RecordLength", mConnectedDevices.at(i), value_ns, value_ns);
					if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG1) {
						mConnectedDevices.at(i)->setPreTrg(pretrg_value_ns);
						manageAddLogAndConf("PreTrigger", mConnectedDevices.at(i), pretrg_value_ns, pretrg_value_ns);
					}
				}
				catch (CAENFELibException& exc) {
					addLog(exc.ErrorMessage(), false);
				}				
		}
	}
	else {
				try {
					if (mUoMX == UOM_PHYS_UNIT) {
						value_ns = QString("%1").arg(reclen * 1000 / mDecimation);
						if (mActiveDevice->DeviceClass == CAEN_DIG1)
							pretrg_value_ns = QString("%1").arg(mControlPan->getPreTrg() * 1000.);						
					}
					else {
						if (mActiveDevice->DeviceClass == CAEN_DIG2)
							value_ns = QString("%1").arg(reclen * (mActiveDevice->getSample_to_S() * 1e9) / mDecimation);
						else {
							value_ns = QString("%1").arg(reclen * (mActiveDevice->getSample_to_S() * 1e9));
							pretrg_value_ns = QString("%1").arg(mControlPan->getPreTrg() * (mActiveDevice->getSample_to_S() * 1e9) * mDecimation);
						}
					}
					mActiveDevice->setReclenT(value_ns);
					manageAddLogAndConf("RecordLength", mActiveDevice, value_ns, value_ns);
					if (mActiveDevice->DeviceClass == CAEN_DIG1) {
						mActiveDevice->setPreTrg(pretrg_value_ns);
						manageAddLogAndConf("PreTrigger", mActiveDevice, pretrg_value_ns, pretrg_value_ns);
					}
				}
				catch (CAENFELibException& exc) {
					addLog(exc.ErrorMessage(), false);
				}
	}

	try{
		mRecLenS = mActiveDevice->getReclen_valueS();
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}

	if (mPlotPan->PlotType == PLOT_TYPE_WAVEFORMS) {
		if (mUoMX == UOM_PHYS_UNIT)
			mPlotPan->setXAxis(0, getMaxRecLen_us(), mUoMX);
		else
			mPlotPan->setXAxis(0, static_cast<double>(getMaxRecLenS()), mUoMX);
	}
}

void WaveDump2::PreTrgChanged(double val) {
	QString value = QString("%1").arg(qRound(val));

	if (mControlPan->getDevName() == "ALL") {//ALL selected, to set pretrg to all boards
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			try {
				if (mUoMX == UOM_PHYS_UNIT) {
					if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
						value = QString("%1").arg(qRound((val * 1000.) / getDevDecimation(mConnectedDevices.at(i))));
					else
						value = QString("%1").arg(qRound((val * 1000.)));
				}
				else {
					if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
						value = QString("%1").arg(val * (mConnectedDevices.at(i)->getSample_to_S() * 1e9) / getDevDecimation(mConnectedDevices.at(i)));
					else
						value = QString("%1").arg(val * (mConnectedDevices.at(i)->getSample_to_S() * 1e9));
				}
				mConnectedDevices.at(i)->setPreTrg(value);
				manageAddLogAndConf("PreTrigger", mConnectedDevices.at(i), value, value);
				double scale_factor = 1.;
				if (i==0) { //the value is the same for all, also for the active trace
					switch (mUoMX) {
					case UOM_SAMPLE:
						if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG1)
							scale_factor = 1. / mConnectedDevices.at(i)->getDecimation();
						mPlotPan->updatePreTrgInfo(mConnectedDevices.at(i)->getPreTrgCachedValueS() * scale_factor);
						break;
					case UOM_PHYS_UNIT:
						mPlotPan->updatePreTrgInfo(mConnectedDevices.at(i)->getPreTrgCachedValue() * mConnectedDevices.at(i)->getDecimation() / 1000.);
						break;
					}
				}
			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}

		}

	}
	else {
				try {
					if (mUoMX == UOM_PHYS_UNIT) {
						if(mActiveDevice->DeviceClass == CAEN_DIG2)
							value = QString("%1").arg(qRound(val * 1000. / mDecimation));
						else
							value = QString("%1").arg(qRound(val * 1000.));
					}
					else {
						if (mActiveDevice->DeviceClass == CAEN_DIG2)
							value = QString("%1").arg(val * (mActiveDevice->getSample_to_S() * 1e9) / mDecimation);
						else
							value = QString("%1").arg(val * (mActiveDevice->getSample_to_S() * 1e9));
					}
					mActiveDevice->setPreTrg(value);
					manageAddLogAndConf("PreTrigger", mActiveDevice, value, value);
					if (mActiveDevice->getName() == mPlotPan->getActiveTraceDev()) {//pretrg marker is relative to active trace, not active device
						double scale_factor = 1.;
						switch (mUoMX) {
						case UOM_SAMPLE:
							if (mActiveDevice->DeviceClass == CAEN_DIG1)
								scale_factor = 1. / mActiveDevice->getDecimation();
							mPlotPan->updatePreTrgInfo(mActiveDevice->getPreTrgCachedValueS() * scale_factor);
							break;
						case UOM_PHYS_UNIT:
							if (mActiveDevice->DeviceClass == CAEN_DIG1)
								scale_factor = 1.;
							else
								scale_factor = mActiveDevice->getDecimation();
							mPlotPan->updatePreTrgInfo(mActiveDevice->getPreTrgCachedValue() * scale_factor / 1000.);
							break;
						}
					}
				}
				catch (CAENFELibException& exc) {
					addLog(exc.ErrorMessage(), false);
				}
	}
	mPretrgS = mActiveDevice->getPreTrgCachedValueS();
}

void WaveDump2::ThrChanged(double thr, bool all) {
	QString value_s;
	QString active_trace_dev = mPlotPan->getActiveTraceDev();
	int ch = mPlotPan->getActiveTraceCh();
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mUoMY == UOM_PHYS_UNIT)
			value_s = QString("%1").arg(qRound(thr / (mConnectedDevices.at(i)->getSample_to_V(ch) * 1e3)));
		else
			value_s = QString("%1").arg(qRound(thr));

		if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG1 && (mUoMY == UOM_PHYS_UNIT)) {
			double val = (thr - mPlotPan->getMinY()) * mConnectedDevices.at(i)->getThr_max(ch) / (mConnectedDevices.at(i)->getVRangeV(ch) * 1e3);
			value_s = QString("%1").arg(qRound(val));
		}

		if (all) {
			for (int c = 0; c < mConnectedDevices.at(i)->getNumCh(); c++) {
				mConnectedDevices.at(i)->setThr_value(c, value_s);
				switch (mUoMY) {
					case UOM_SAMPLE:
						manageAddLogAndConf(QString("/ch/%1/par/TriggerThr").arg(c), mConnectedDevices.at(i), value_s, value_s);
						break;
					case UOM_PHYS_UNIT:
						manageAddLogAndConf(QString("/ch/%1/par/TriggerThr").arg(c), mConnectedDevices.at(i), QString("%1 mV").arg(mConnectedDevices.at(i)->getThrCachedValue(c) * mConnectedDevices.at(i)->getSample_to_V(c) * 1e3), QString("%1").arg(mConnectedDevices.at(i)->getThrCachedValue(c)));
				}
			}
		}

		if (mConnectedDevices.at(i)->getName() == active_trace_dev) { 
			try {				
				mConnectedDevices.at(i)->setThr_value(ch, value_s);
				
				switch (mUoMY) {
				case UOM_SAMPLE:
					manageAddLogAndConf(QString("/ch/%1/par/TriggerThr").arg(ch), mConnectedDevices.at(i), value_s, value_s);
					mPlotPan->updateThrInfo(mConnectedDevices.at(i)->getThr_max(ch), mConnectedDevices.at(i)->getThr_min(ch), mConnectedDevices.at(i)->getThr_incr(ch), mConnectedDevices.at(i)->getThrCachedValue(ch), 1.);
					break;
				case UOM_PHYS_UNIT:
					manageAddLogAndConf(QString("/ch/%1/par/TriggerThr").arg(ch), mConnectedDevices.at(i), QString("%1 mV").arg(mConnectedDevices.at(i)->getThrCachedValue(ch) * mConnectedDevices.at(i)->getSample_to_V(ch) * 1e3), QString("%1").arg(mConnectedDevices.at(i)->getThrCachedValue(ch)));
					if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
						mPlotPan->updateThrInfo(mConnectedDevices.at(i)->getThr_max(ch), mConnectedDevices.at(i)->getThr_min(ch), mConnectedDevices.at(i)->getThr_incr(ch), mConnectedDevices.at(i)->getThrCachedValue(ch), mConnectedDevices.at(i)->getSample_to_V(ch) * 1e3);
					else {
						mPlotPan->updateThrInfo(mConnectedDevices.at(i)->getThr_max(ch), mConnectedDevices.at(i)->getThr_min(ch), mConnectedDevices.at(i)->getThr_incr(ch), mConnectedDevices.at(i)->getThrCachedValue(ch), mConnectedDevices.at(i)->getVRangeV(ch) * 1e3);
					}

					break;
				}

			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}
			if(!all)
				break;
		}
	}
}

void WaveDump2::DCOffsetChanged(QString device_name, int ch, double value, bool all) {
	baseline_changing = true;
	for (int j = 0; j < mConnectedDevices.size(); j++) {
		if (mConnectedDevices.at(j)->getName() == device_name) {
			try {
				mConnectedDevices.at(j)->setChOffset(ch, value);
				double new_val = mConnectedDevices.at(j)->getChOffset(ch);
				manageAddLogAndConf(QString("/ch/%1/par/DCOffset").arg(ch), mConnectedDevices.at(j), QString("%1").arg(new_val), QString("%1").arg(new_val));
				if (mConnectedDevices.at(j)->DeviceClass == CAEN_DIG2) {
					if (!all) {
						mPlotPan->updateOffsetInfo(new_val);
						updatePlotYAxisRange();
					}
					else {
						mPlotPan->updateOffsetInfo1(device_name, ch, new_val);
					}
				}
				else {
					int gsize = static_cast<CAENDig1Device*>(mConnectedDevices.at(j))->getChGroupSize();
					if (!all) {
						mPlotPan->updateOffsetInfo(new_val);
						updatePlotYAxisRange();
					}
					for(int c=0; c<gsize; c++)
						mPlotPan->updateOffsetInfo1(device_name, c + (ch / gsize) * gsize, new_val);

					if(mUoMY == UOM_PHYS_UNIT)
						mPlotPan->updateThrInfo(mConnectedDevices.at(j)->getThr_max(ch), mConnectedDevices.at(j)->getThr_min(ch), mConnectedDevices.at(j)->getThr_incr(ch), mConnectedDevices.at(j)->getThr_value(ch), mConnectedDevices.at(j)->getVRangeV(ch) * 1e3);
					else
						mPlotPan->updateThrInfo(mConnectedDevices.at(j)->getThr_max(ch), mConnectedDevices.at(j)->getThr_min(ch), mConnectedDevices.at(j)->getThr_incr(ch), mConnectedDevices.at(j)->getThr_value(ch), 1.);
				}

			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
				baseline_changing = false;
			}
			break;
		}
	}
	baseline_changing = false;
}

void WaveDump2::TracePlotChanged(int t, QString dev, int ch) {
	if (mRecordAction)
		mActions->AddSWAction(QString("TRACE_CHANGE_%1").arg(t), dev + QString(" %1").arg(ch));
	for (int b = 0; b < mConnectedDevices.size(); b++) {
		if (mConnectedDevices.at(b)->getName() == dev) {
			mPlotPan->updateGains(dev, mConnectedDevices.at(b)->ChGain);
			QVector<double>sampl_to_V;
			for (int ch = 0; ch < mConnectedDevices.at(b)->getNumCh(); ch++)
				sampl_to_V.append(mConnectedDevices.at(b)->getSample_to_V(ch));
			mPlotPan->updateBaselines(dev, mConnectedDevices.at(b)->getADCNBit(), sampl_to_V, mConnectedDevices.at(b)->ChBaselines);
		}
	}
	updatePlotYAxisRange();
	updatePlotXAxisRange();
}

void WaveDump2::TraceEnableChanged(int t, int enable){
	if (enable) {
		//check if the channel is enabled for acq before enabling the trace
		bool enabled = true;
		for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
			if (mPlotPan->getTraceDev(t) == mConnectedDevices.at(dev)->getName()) {
				if (mConnectedDevices.at(dev)->ErrorsReported)
					break;
				if (!mConnectedDevices.at(dev)->isChEnabled(mPlotPan->getTraceCh(t)))
					enabled = false;
				break;
			}
		}
		if (!enabled) {
			QMessageBox::critical(this, tr("Trace enable error"), tr("Device channel is not enabled for acquisition"), QMessageBox::Ok, 0);
			//disable again the trace
			mPlotPan->TraceEnableCmd(t, false);
			return;
		}
	}
	if (mRecordAction)
		mActions->AddSWAction(QString("TRACE_ENABLE_%1").arg(t), (enable) ? "TRUE" : "FALSE");
	updatePlotYAxisRange();
}

void WaveDump2::CheckTracesEnableStatus() {
	for (int t = 0; t < NPLOTTRACES; t++) {
		if (mPlotPan->isTraceVisible(t)) {
			for (int dev = 0; dev < mConnectedDevices.size(); dev++) {
				if (mPlotPan->getTraceDev(t) == mConnectedDevices.at(dev)->getName()) {
					if (!mConnectedDevices.at(dev)->isChEnabled(mPlotPan->getTraceCh(t)))
						mPlotPan->TraceEnableCmd(t, false);
					break;
				}
			}
		}
	}

}

void WaveDump2::PlotGenericTraceChanged(QString dev_name, int ch){
	if (ch == -1)
		return;
	for (int j = 0; j < mConnectedDevices.size(); j++) {
		if (mConnectedDevices.at(j)->getName() == dev_name) {
			double val = mConnectedDevices.at(j)->getChOffset(ch);
			mPlotPan->updateOffsetInfo1(dev_name, ch, val);
		}
	}
}

void WaveDump2::PlotActiveTraceChanged(QString dev_name, int ch) {
	if (ch == -1)
		return;
	for (int j = 0; j < mConnectedDevices.size(); j++) {
		if (mConnectedDevices.at(j)->getName() == dev_name) {
			mSample_to_V = mConnectedDevices.at(j)->getSample_to_V(ch);
			double val = mConnectedDevices.at(j)->getChOffset(ch);
			mPlotPan->updateOffsetInfo(val);
			mPlotPan->updateTSampl(mConnectedDevices.at(j)->getSample_to_S() * 1e9);
			mPlotPan->setActiveDevClass(mConnectedDevices.at(j)->DeviceClass);
			updatePlotYAxisRange();
			switch (mUoMY) {
			case UOM_SAMPLE:
				mPlotPan->updateThrInfo(mConnectedDevices.at(j)->getThr_max(ch), mConnectedDevices.at(j)->getThr_min(ch), mConnectedDevices.at(j)->getThr_incr(ch), mConnectedDevices.at(j)->getThr_value(ch), 1.);
				break;
			case UOM_PHYS_UNIT:
				if (mConnectedDevices.at(j)->DeviceClass == CAEN_DIG2)
					mPlotPan->updateThrInfo(mConnectedDevices.at(j)->getThr_max(ch), mConnectedDevices.at(j)->getThr_min(ch), mConnectedDevices.at(j)->getThr_incr(ch), mConnectedDevices.at(j)->getThr_value(ch), mConnectedDevices.at(j)->getSample_to_V(ch) * 1e3);
				else {
					mPlotPan->updateThrInfo(mConnectedDevices.at(j)->getThr_max(ch), mConnectedDevices.at(j)->getThr_min(ch), mConnectedDevices.at(j)->getThr_incr(ch), mConnectedDevices.at(j)->getThr_value(ch), mConnectedDevices.at(j)->getVRangeV(ch) * 1e3);
				}
				break;
			}
			switch (mUoMX) {
			case UOM_SAMPLE:
				mPlotPan->updatePreTrgInfo(mConnectedDevices.at(j)->getPreTrgCachedValueS());
				break;
			case UOM_PHYS_UNIT:
				mPlotPan->updatePreTrgInfo(mConnectedDevices.at(j)->getPreTrgCachedValue() * mConnectedDevices.at(j)->getDecimation_value() / 1000.);
				break;
			}
			break;
		}
	}
	mControlPan->updateCursorsTrace(mPlotPan->getTraceColor(mPlotPan->ActiveTrace), QString("Trace %1").arg(mPlotPan->ActiveTrace));
}

void WaveDump2::PlotActiveTraceColorChanged() {
	mControlPan->updateCursorsTrace(mPlotPan->getTraceColor(mPlotPan->ActiveTrace), QString("Trace %1").arg(mPlotPan->ActiveTrace));
	if (mPlotPan->cursorsEnabled()) {
		New_Cursors(false);
		New_Cursors(true);
	}
}

void  WaveDump2::setTrgSource(QString trg_src) {
	if (mControlPan->getDevName() == "ALL") {//ALL selected, to set reclen to all boards
		for (int i = 0; i < mConnectedDevices.size(); i++) {
			try {
				mConnectedDevices.at(i)->setTrgSource(trg_src);
				manageAddLogAndConf("TriggerSource", mConnectedDevices.at(i), trg_src, trg_src);
			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}
		}
	}
	else {
		try {
			mActiveDevice->setTrgSource(trg_src);
			manageAddLogAndConf("TriggerSource", mActiveDevice, trg_src, trg_src);
		}
		catch (CAENFELibException& exc) {
			addLog(exc.ErrorMessage(), false);
		}
	}
}


void  WaveDump2::setTrgMode(int mode) {
	if (mControlPan->getDevName() == "ALL") {//ALL selected
		if(mRecordAction)
			mActions->AddSWAction("TRIGGER_MODE_CHANGE", "ALL" + QString(" %1").arg(mode));
		for (int i = 0; i < mConnectedDevices.size(); i++) {
				mConnectedDevices.at(i)->setTrgMode(mode);
		}
	}
	else {
		if(mRecordAction)
			mActions->AddSWAction("TRIGGER_MODE_CHANGE", mActiveDevice->getName() + QString(" %1").arg(mode));
		mActiveDevice->setTrgMode(mode);
	}
}

QString WaveDump2::getCommonSelfTrgMask() {
	uint64_t mask = 0;
	for (int ch = 0; ch < 64; ch++) {
		if (CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch)).contains("TRUE", Qt::CaseInsensitive))
			mask |= (1LL << ch);
	}
	return QString("%1").arg(mask);
}

QString WaveDump2::getCommonTrgOutMask() {
	uint64_t mask = 0;
	for (int ch = 0; ch < 64; ch++) {
		if (CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Out_Propagate").arg(ch)).contains("TRUE", Qt::CaseInsensitive))
			mask |= (1LL << ch);
	}
	return QString("%1").arg(mask);
}

void WaveDump2::setCommonMask(QString m, QString mask_ref_par){
	uint64_t mask = m.toULong();
	uint32_t group_mask = 0;
	for (int ch = 0; ch < 64; ch++) {
		if (mask & (1LL << ch))
			group_mask |= (1 << (ch / 8));
	}
	for (int g = 0; g < 8; g++) {
		if (group_mask & (1 << g)) {
			CommonGroupValuesmap1[g].insert((QString("/group/%1/par/Gr_").arg(g) + mask_ref_par), "TRUE");
			add2LogAndConf("SET", "COMMON1", (QString("/group/%1/par/Gr_").arg(g) + mask_ref_par), "TRUE", "TRUE", CAEN_FELib_ErrorCode(0));

		}
		else {
			CommonGroupValuesmap1[g].insert((QString("/group/%1/par/Gr_").arg(g) + mask_ref_par), "FALSE");
			add2LogAndConf("SET", "COMMON1", (QString("/group/%1/par/Gr_").arg(g) + mask_ref_par), "FALSE", "FALSE", CAEN_FELib_ErrorCode(0));
		}
	}

	for (int ch = 0; ch < 64; ch++) {
		if (mask & (1LL << ch)) {
			CommonChValuesmap1[ch].insert((QString("/ch/%1/par/Ch_").arg(ch) + mask_ref_par), "TRUE");
			add2LogAndConf("SET", "COMMON1", (QString("/ch/%1/par/Ch_").arg(ch) + mask_ref_par), "TRUE", "TRUE", CAEN_FELib_ErrorCode(0));
		}
		else {
			CommonChValuesmap1[ch].insert((QString("/ch/%1/par/Ch_").arg(ch) + mask_ref_par), "FALSE");
			add2LogAndConf("SET", "COMMON1", (QString("/ch/%1/par/Ch_").arg(ch) + mask_ref_par), "FALSE", "FALSE", CAEN_FELib_ErrorCode(0));
		}
	}

}

QString WaveDump2::getCommonTrgSource() {
	QString value = "";
	if (CommonDevValuesmap1.value("/par/Trg_Ext_Enable").contains("TRUE",Qt::CaseInsensitive))
		value.append("TrgIn");
	if (CommonDevValuesmap1.value("/par/Trg_Sw_Enable").contains("TRUE", Qt::CaseInsensitive)) {
		if (!value.isEmpty())
			value.append("|");
		value.append("SwCmd");
	}
	if (getCommonSelfTrgMask() != "0") {
		if (!value.isEmpty())
			value.append("|");
		value.append("ITL");
	}
	return value;
}

QString WaveDump2::getSelfTrgMask() {
	QString mask = "";
	try {
		mask = mActiveDevice->getSelfTrgMask();
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	return mask;
}

QString WaveDump2::getTrgoutMask() {
	QString mask = "";
	try {
		mask = mActiveDevice->getTrgOutMask();
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	return mask;

}

int WaveDump2::getNumChannels() {
	int CH;
	try {
		CH = mActiveDevice->getNumCh();
	}
	catch (CAENFELibException& exc) {
		addLog(exc.ErrorMessage(), false);
	}
	return CH;
}

qint64 WaveDump2::getAcqtime(int dev) {
	return mConnectedDevices.at(dev)->TimeFromStart();
}

void WaveDump2::setSelfTrgMask(QString mask) {
	if (mControlPan->getDevName() == "ALL") {//ALL selected

		for (int i = 0; i < mConnectedDevices.size(); i++) {
			try {
				mConnectedDevices.at(i)->setSelfTrgMask(mask);
				manageAddLogAndConf("SelfTriggerMask", mConnectedDevices.at(i), mConnectedDevices.at(i)->getSelfTrgMask(), mConnectedDevices.at(i)->getSelfTrgMask());
			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}
		}
	}
	else {
		try {
			mActiveDevice->setSelfTrgMask(mask);
			manageAddLogAndConf("SelfTriggerMask", mActiveDevice, mActiveDevice->getSelfTrgMask(), mActiveDevice->getSelfTrgMask());
		}
		catch (CAENFELibException& exc) {
			addLog(exc.ErrorMessage(), false);
		}
	}

}


void WaveDump2::CheckSettings() {
	updateActiveDevInfo(mActiveDevice);
	mActiveDevice->updateGains();
	for (int ch = 0; ch < mActiveDevice->getNumCh(); ch++)
		mPlotPan->updateOffsetInfo1(mActiveDevice->getName(), ch, mActiveDevice->ChBaselines.at(ch));
	CAENparameter* thr_ch = mActiveDevice->getThrParam(mPlotPan->getActiveTraceCh());
	mActiveDevice->updateChSample_to_V(mPlotPan->getActiveTraceCh());
	if (mUoMY == UOM_PHYS_UNIT) {
		if (mActiveDevice->DeviceClass == CAEN_DIG2)
			mPlotPan->updateThrInfo(thr_ch->getMax(), thr_ch->getMin(), thr_ch->getIncr(), thr_ch->getValue().toDouble(), mActiveDevice->getSample_to_V(mPlotPan->getActiveTraceCh()) * 1e3);
		else {
			mPlotPan->updateThrInfo(thr_ch->getMax(), thr_ch->getMin(), thr_ch->getIncr(), thr_ch->getValue().toDouble(), mActiveDevice->getVRangeV(mPlotPan->getActiveTraceCh()) * 1e3);
		}
	}
	updatePlotYAxisRange();
}

void WaveDump2::Reset() {
	ResetDevice(mControlPan->getDevName());
}

void WaveDump2::ResetDevice(QString devname) {
	QMessageBox::StandardButton resBtn = QMessageBox::question(this, "WaveDump2", " Do you want to reset " + devname + " device now? \n All the registers will be set to the default value and  memories will be cleared.", QMessageBox::No | QMessageBox::Yes, QMessageBox::Yes);
	if (resBtn == QMessageBox::Yes) {
		if (Check_connection_Timer->isActive())
			Check_connection_Timer->stop();

		if (devname == "ALL") {//ALL selected
			for (int i = 0; i < mConnectedDevices.size(); i++) {
				try {
					mConnectedDevices.at(i)->Reset();
					QThread::msleep(1000);
					QApplication::processEvents();
					QThread::msleep(1000);
					QApplication::processEvents();
					QThread::msleep(1000);
					QApplication::processEvents();
					if(mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
						mConnectedDevices.at(i)->ApplyDefaultConfig(CommonDevValuesmap, CommonChValuesmap, CommonGroupValuesmap);
					else
						mConnectedDevices.at(i)->ApplyDefaultConfig(CommonDevValuesmap1, CommonChValuesmap1, CommonGroupValuesmap1);
				}
				catch (CAENFELibException& exc) {
					addLog(exc.ErrorMessage(), false);
				}
			}

		}
		else {
			try {
				mActiveDevice->Reset();
				QThread::msleep(1000);
				QApplication::processEvents();
				QThread::msleep(1000);
				QApplication::processEvents();
				QThread::msleep(1000);
				QApplication::processEvents();
				if (mActiveDevice->DeviceClass == CAEN_DIG2)
					mActiveDevice->ApplyDefaultConfig(CommonDevValuesmap, CommonChValuesmap, CommonGroupValuesmap);
				else
					mActiveDevice->ApplyDefaultConfig(CommonDevValuesmap1, CommonChValuesmap1, CommonGroupValuesmap1);
			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}

		}

		RefreshDevSettings();

		if (!Check_connection_Timer->isActive())
			Check_connection_Timer->start(3000);
		QMessageBox::information(this, tr("WaveDump2"), "Reset executed successfully.", QMessageBox::Ok, 0);
	}

}

void WaveDump2::RebootDevice(QString devname) {
	for (int i = 0; i < mConnectedDevices.size(); i++) {
		if (mConnectedDevices.at(i)->getName() == devname) {
			try {
				mConnectedDevices.at(i)->Reboot();
				QThread::msleep(1000);
				QApplication::processEvents();
				QThread::msleep(1000);
				QApplication::processEvents();
				QThread::msleep(1000);
				QApplication::processEvents();
				QThread::msleep(1000);
				QApplication::processEvents();
				QThread::msleep(1000);
				QApplication::processEvents();
				if (mConnectedDevices.at(i)->DeviceClass == CAEN_DIG2)
					mConnectedDevices.at(i)->ApplyDefaultConfig(CommonDevValuesmap, CommonChValuesmap, CommonGroupValuesmap);
				else
					mConnectedDevices.at(i)->ApplyDefaultConfig(CommonDevValuesmap1, CommonChValuesmap1, CommonGroupValuesmap1);
			}
			catch (CAENFELibException& exc) {
				addLog(exc.ErrorMessage(), false);
			}
		}
	}
}

void WaveDump2::AboutWaveDump(){
	About *winfo = new About(this);
	winfo->exec();
}


void WaveDump2::LoadRawFromFile() {
	QStringList fileNames;
	fileNames = QFileDialog::getOpenFileNames(this, ("select one or more raw data files"), "", tr("WaveDump2 raw file (*.wbin);;All Files (*)"));
	if (fileNames.size() == 0) {
		return;
	}
	for (auto xfile : fileNames)
	{
		if (CreateDeviceFromFile(xfile, mConnected) != 0)
			return;
		mConnected++;
	}
	mPlotPan->EnableEvDataTab(false);
	OfflineMode = true;

	if (mConnectedDevices.size() == 1 && mActiveDevice->DeviceClass == CAEN_DIG1) {
		mControlPan->YUnitChangeCmd(UOM_SAMPLE);
		mControlPan->XUnitChangeCmd(UOM_SAMPLE);
	}
}


int WaveDump2::CreateDeviceFromFile(QString filename, int index) {
	quint32 pid=0, ADCNBit=0, SampleToS=0, PreTrg = 0, TrgSrc=0, RecLen=0, baseline[MAX_CHANNELS] = { 0 }, factor[MAX_CHANNELS] = {0}, gain[MAX_CHANNELS] = { 0 }, family=0;
	qint32 thr[MAX_CHANNELS] = { 0 };
	quint16 NCh = 0, Decim=1;
	quint64 StartRun[4], Mask=0, size_word;
	quint32 StartRun32[4], Mask32 = 0, gmask=0;
	uint16_t exp = 0;
	uint32_t evsize;
	double test = 0;
	QFile f(filename);
	f.open(QIODevice::ReadOnly);
	QDataStream in(&f);
	in.setByteOrder(QDataStream::LittleEndian);
	CAENDig2Device dig2;
	CAENDig1Device dig1;
	int dev_type;

	in >> family;
	in >> pid;
	in >> ADCNBit;
	in >> SampleToS;
	in >> TrgSrc;
	in >> PreTrg;
	in >> NCh;
	for (int ch = 0; ch < NCh; ch++) {
		in >> baseline[ch];
		in >> factor[ch];
		in >> gain[ch];
		in >> thr[ch];
	}
	in >> StartRun[0];
	StartRun[0] = swapByteOrderWord(StartRun[0]);
	if (dig2.isInfoStartWord(StartRun[0]))
		dev_type = CAEN_DIG2;
	else if(dig1.isInfoStartWord(StartRun[0]))
		dev_type = CAEN_DIG1;
	else{
		QMessageBox::critical(this, tr("File error"), "Raw data file" + filename +" is invalid", QMessageBox::Ok, 0);
		f.close();
		return -1;
	}

	bool same_device = false;
	QString new_name = "";
	for (int d = 0; d < mConnectedDevices.size(); d++) {
		QString name = mConnectedDevices.at(d)->getName();
		if (name.contains(QString("%1").arg(pid))) {
			same_device = true;
			QStringList l = name.split(QString("%1").arg(pid));

			new_name = l.at(0) + QString("%1").arg(pid);
			if (l.size() > 1) {
				QString ll = l.at(1);
				ll.remove("_");
				int n = ll.toInt();
				new_name.append(QString("_%1").arg(n + 1));
			}
		}
	}


	CAENDevice* tmp;
	if (dev_type == CAEN_DIG2) {
		in >> StartRun[1];
		StartRun[1] = swapByteOrderWord(StartRun[1]);
		RecLen = (StartRun[1] & 0x1ffffff) * 4;
		exp = ((StartRun[1] >> 27) & 0x1) + 2 * ((StartRun[1] >> 28) & 0x1) + 4 * ((StartRun[1] >> 29) & 0x1) + 8 * ((StartRun[1] >> 30) & 0x1);
		Decim = qPow(2, exp);
		in >> StartRun[2];
		StartRun[2] = swapByteOrderWord(StartRun[2]);
		Mask = StartRun[2] & 0xffffffff;
		in >> StartRun[3];
		StartRun[3] = swapByteOrderWord(StartRun[3]);
		Mask |= ((StartRun[3] & 0xffffffff) << 32);

		if(new_name == "")
			new_name = QString("dig2.Virtual.%1").arg(pid);
		CAENDig2Device* dev = new CAENDig2Device(new_name, "virtual", QString("virtual.%1").arg(pid), 0, true);
		tmp = dev;
	}
	else {
		in.device()->seek(in.device()->pos() - sizeof(quint64));
		in >> StartRun32[0];
		in >> StartRun32[1];
		in >> StartRun32[2];
		in >> StartRun32[3];

		Mask32 = StartRun32[1] & 0xff;
		gmask = Mask32;
		Mask32 |= (((StartRun32[2] >> 24) & 0xff) << 8);
		int ActiveChannels = 0;
		int ActiveGroups = 0;
		if (family == 740) {
			uint32_t chmask = 0;
			for (int g = 0; g < MAX_CHANNELS / 8; g++) {
				if (Mask32 & (1 << g)) {
					chmask |= (0xff << (g*8));
				}
				if (gmask & (1 << g))
					ActiveGroups++;
			}
			Mask32 = chmask;
		}
		
		for (int c = 0; c < MAX_CHANNELS / 2; c++) {
			if (Mask32 & (1 << c))
				ActiveChannels++;
		}
		RecLen = (((StartRun32[0] & 0xfffffff) - 4) * 2 / ActiveChannels);
		if (family == 740)
			RecLen = ((((StartRun32[0] & 0xfffffff) - 4) * (4/ActiveGroups))/ 12); 

		if (new_name == "")
			new_name = QString("dig1.Virtual.%1").arg(pid);
		CAENDig1Device* dev = new CAENDig1Device(new_name, "virtual", QString("virtual.%1").arg(pid), 0, true);
		tmp = dev;
		Mask = Mask32;
	}

	
	tmp->DeviceClass = dev_type;
	tmp->DefineFamily(family);
	tmp->DefineInfo(index, pid, ADCNBit, SampleToS, NCh, baseline, factor);
	tmp->DefineGains(gain);
	tmp->DefineDecimation(Decim);
	tmp->DefineReclen(RecLen);
	tmp->DefineChMask(Mask);
	tmp->DefineTrgSource(TrgSrc);
	tmp->DefineThresholds(thr);
	tmp->DefinePreTrigger(PreTrg);
	tmp->DefineRawDataFile(filename);

	mConnectedDevices += tmp;
	if (mDevicesReadoutThreads.size() < mConnectedDevices.size())
		mDevicesReadoutThreads.append(nullptr);
	mStatsPan->fillDevices(mConnectedDevices);
	mIgnoreEvent = true;
	mControlPan->updateStartableDevices(1, tmp->getName());
	if (same_device)
		mControlPan->disableALLOption();
	mIgnoreEvent = false;
	QVector<QString> channels;
	for (int i = 0; i < tmp->getNumCh(); i++) {//fill plot traces with channels of the first connected device
		channels += QString("CH%1").arg(QString::number(i));
	}
	if (mActiveDevice == nullptr) {//the first connected device becomes the active device
		setActiveDevice(tmp);
	}

	mPlotPan->updatePalette(1, tmp->getName(), channels);
	mNumberOfDevice++;

	mPlotPan->addOfflineDevice(tmp->getName());	
	uint64_t tot_size = f.size();
	uint32_t max_ev;
	if (tmp->DeviceClass == CAEN_DIG2) {
		in >> size_word;
		size_word = swapByteOrderWord(size_word);
		evsize = (size_word & 0xffffffff) * 8;		
		max_ev = (tot_size - (uint64_t(26) + uint64_t(16) * uint64_t(NCh) + 32)) / evsize;		
	}
	else {
		evsize = (StartRun32[0] & 0xfffffff) * 4;
		max_ev = (tot_size - (uint64_t(26) + uint64_t(16) * uint64_t(NCh))) / evsize;
	}
	mPlotPan->setMaxEvNumber(tmp->getName(), max_ev);
	f.close();

	if (mNumberOfDevice == 1) {
		mOpenSettingsAct->setEnabled(false);
		mManualControllerAct->setEnabled(false);
		mControlPan->updateCursorsTrace(mPlotPan->getTraceColor(mPlotPan->ActiveTrace), QString("Trace %1").arg(mPlotPan->ActiveTrace));
		mControlPan->setVisible(true);
		mControlPan->setEnabled(true);
		mControlPan->updateGUIForOffline(true);	
		mPlotPan->enableOfflineMode(true);		
		mPlotPan->setEnabled(true);
		mInfoMenuAct->setEnabled(false);
		mErrorsMenuAct->setEnabled(false);
		QVector<double>sampl_to_V;
		for (int ch = 0; ch < mConnectedDevices.at(0)->getNumCh(); ch++)
			sampl_to_V.append(mConnectedDevices.at(0)->getSample_to_V(ch));
		mPlotPan->updateBaselines(mConnectedDevices.at(0)->getName(), mConnectedDevices.at(0)->getADCNBit(), sampl_to_V, mConnectedDevices.at(0)->ChBaselines);
		mPlotPan->updateGains(mConnectedDevices.at(0)->getName(), mConnectedDevices.at(0)->ChGain);
		mBrowseAct->setEnabled(false);
		mStatAct->setEnabled(true);
		mLogAct->setEnabled(true);
		mSaveOutputAct->setEnabled(true);
		mSaveOutputAct->setChecked(false);
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
		mDataSavedEnabled.store(0);
#else
		mDataSavedEnabled.storeRelaxed(0);
#endif
		mFastSaveEnable = false;
		mOutputConfigAct->setEnabled(true);
		mDisconnectMenuAct->setEnabled(true);
		mDisconnectAct->setEnabled(true);
		mConfiguration->setEnabled(true);
		mScriptMenu->setEnabled(false);
		mOutputMenu->setEnabled(true);
		mWindowMenu->setEnabled(true);
		mPlotDock->setVisible(true);
		mLogDock->setVisible(true);
		mSaveasSettingsAct->setEnabled(false);
		mResetButton->setEnabled(false);
		mConnectAct->setEnabled(false);
		mConnectMenuAct->setEnabled(false);
		mScanAct->setEnabled(false);
		mLoadAct->setEnabled(false);
		mLoadMenuAct->setEnabled(false);
		mCenterWidget->setMaximumWidth(1);
		if (!this->mConnectAct->isChecked())
			this->mConnectAct->setChecked(true);
	}
	return 0;
}


uint64_t WaveDump2::swapByteOrderWord(uint64_t ull)
{
	uint64_t w = ull;
	w = (w >> 56) |
		((w << 40) & 0x00FF000000000000) |
		((w << 24) & 0x0000FF0000000000) |
		((w << 8) & 0x000000FF00000000) |
		((w >> 8) & 0x00000000FF000000) |
		((w >> 24) & 0x0000000000FF0000) |
		((w >> 40) & 0x000000000000FF00) |
		(w << 56);
	return w;
}


void WaveDump2::PlotTimeChanged(double val) {
	PlotInterval_ms = val * 1000;
}