/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		SystemSettings.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "SystemSettings.h"
#include "Dx740V2ConfigPanel.h"
#include "Dx740V1ConfigPanel.h"
#include "WaveDump2.h"
#include "CAENFELibException.h"
#include "boardsListDialog.h"

SystemSettings::SystemSettings(WaveDump2 *parent, QList<CAENDevice *> *devices) // QVector<uint64_t> dev_handles, QVector<QString> dev_names
{
	ui.setupUi(this);
	setWindowIcon(QIcon(":WAVEDump2.png"));

	QMessageBox msgBox(this);
	msgBox.setText("<html><B>Reading device settings <br>           ... please wait ...</html>");
	msgBox.setStandardButtons({});
	msgBox.show();
	QApplication::processEvents();

	mWaveDump = parent;
	mBoards = devices;

	QString smode = mWaveDump->getGlobalSMode();
	smode.remove(0,3);
	int index = ui.cBox_SMode->findText(smode);
	ui.cBox_SMode->setCurrentIndex(index);
	mSMode = ui.cBox_SMode->currentText();
	if (mSMode.contains("CHAIN")) {
		ui.cBox_sw_hw->setEnabled(true);
		ui.checkBox_matching->setEnabled(true);
		ui.doubleSpinBox_coinc_win->setEnabled(true);
	}
	else {
		ui.cBox_sw_hw->setEnabled(false);
		ui.checkBox_matching->setEnabled(false);
		ui.doubleSpinBox_coinc_win->setEnabled(false);
	}
	ui.checkBox_matching->setChecked(mWaveDump->OnlyMatching);

	ui.tabWidget_Boards->clear();
	config_panels.clear();


	if (mWaveDump->OfflineMode) {
		ui.cBox_IOlevel->hide();
	}
	else {
		for (int i = 0; i < devices->size(); i++) {
			mHandles.push_back(devices->at(i)->getHandle());
			mNames.push_back(devices->at(i)->getName());
		}
		for (int dev = 0; dev < mHandles.size(); dev++) {

			if (mBoards->at(dev)->getModel().contains("2740") || mBoards->at(dev)->getModel().contains("2745") || mBoards->at(dev)->getModel().contains("2730")
				|| mBoards->at(dev)->getModel().contains("2751")) {
				Dx740V2ConfigPanel* config = new Dx740V2ConfigPanel(mWaveDump, static_cast<CAENDig2Device *>(devices->at(dev)), mNames.at(dev), false);
				ui.tabWidget_Boards->addTab(config, mNames.at(dev));
				config_panels.push_back(config);
				Ndig2_devs++;
			}
			else if (mBoards->at(dev)->getModel().contains("740") || mBoards->at(dev)->getModel().contains("730") || mBoards->at(dev)->getModel().contains("725") 
				|| mBoards->at(dev)->getModel().contains("724") || mBoards->at(dev)->getModel().contains("751") || mBoards->at(dev)->getModel().contains("720")) {
				Dx740V1ConfigPanel* config = new Dx740V1ConfigPanel(mWaveDump, static_cast<CAENDig1Device*>(devices->at(dev)), mNames.at(dev), false);
				ui.tabWidget_Boards->addTab(config, mNames.at(dev));
				config_panels.push_back(config);
				Ndig1_devs++;
			}
			continue;

		}

		ui.cBox_IOlevel->setToolTip("/par/IOLevel");
		UpdateIOLevel();
		connect(ui.cBox_IOlevel, SIGNAL(currentIndexChanged(int)), this, SLOT(IOLevelChanged(int)));
	}
	
	ui.doubleSpinBox_coinc_win->setValue(mWaveDump->getCoincWin() / 1000.); //value in us
	QString mu = QChar(0xbc, 0x03);
	ui.label_coinc_win->setText(ui.label_coinc_win->text() + " [" + mu + "s]");

	bool same_device = false;
	if (mWaveDump->OfflineMode) {
		QStringList names;
		for (int b = 0; b < mBoards->size(); b++) {
			if (names.contains(mBoards->at(b)->getSN()))
				same_device = true;
			names.append(mBoards->at(b)->getSN());			
		}
	}

	if (mBoards->size() > 1 && !same_device) {
		ui.cBox_SMode->clear();
		ui.cBox_SMode->addItem("ASYNC");
		if (!mWaveDump->OfflineMode) {
			if (Ndig2_devs && !Ndig1_devs) {
				common_config = new CommonConfigPanel(mWaveDump, mHandles);
				ui.tabWidget_Boards->addTab(common_config, "COMMON");
				ui.cBox_SMode->addItem("CHAIN_SYNC-CLKIN");
				ui.cBox_SMode->addItem("CHAIN_SIN-GPIO");
				ui.cBox_SMode->addItem("CHAIN_SIN-TRGOUT");
				ui.cBox_SMode->addItem("CHAIN_1ST_TRIGGER");
			}
			else if (Ndig1_devs && !Ndig2_devs) {
				QVector<CAENDig1Device*>devs;
				for (int i = 0; i < devices->size(); i++)
					devs.append(static_cast<CAENDig1Device *>(devices->at(i)));
				common_config1 = new CommonConfigPanel1(mWaveDump, devs, false);
				ui.tabWidget_Boards->addTab(common_config1, "COMMON");
				ui.cBox_SMode->addItem("CHAIN_TRGIN-TRGOUT");
				ui.cBox_SMode->addItem("CHAIN_ONE2ALL_EXTOR");
				ui.cBox_SMode->addItem("CHAIN_SIN-TRGOUT");
				ui.cBox_SMode->addItem("CHAIN_1ST_TRIGGER");
			}
			connect(ui.tabWidget_Boards, SIGNAL(currentChanged(int)), this, SLOT(currentTabChanged(int)));
		}

		QString smode = mWaveDump->getGlobalSMode();
		smode.remove(0, 3);
		mSMode = smode;
		if (mSMode.contains("CHAIN")) {
			ui.cBox_sw_hw->setEnabled(true);
			ui.checkBox_matching->setEnabled(true);
			ui.doubleSpinBox_coinc_win->setEnabled(true);
		}
		else {
			ui.cBox_sw_hw->setEnabled(false);
			ui.checkBox_matching->setEnabled(false);
			ui.doubleSpinBox_coinc_win->setEnabled(false);
		}
		ui.checkBox_matching->setChecked(mWaveDump->OnlyMatching);
		int index = ui.cBox_SMode->findText(mSMode);
		ui.cBox_SMode->setCurrentIndex(index);
		if(!mWaveDump->OfflineMode)
			connect(ui.cBox_SMode, SIGNAL(currentTextChanged(QString)), this, SLOT(SModeChanged(QString)));
		else
			connect(ui.cBox_SMode, SIGNAL(currentTextChanged(QString)), this, SLOT(OfflineSModeChanged(QString)));
		connect(ui.cBox_sw_hw, SIGNAL(currentTextChanged(QString)), this, SLOT(SWHWChanged(QString)));
		connect(ui.doubleSpinBox_coinc_win, SIGNAL(valueChanged(double)), this, SLOT(WindowChanged(double)));
		connect(ui.checkBox_matching, SIGNAL(clicked(bool)), this, SLOT(FullMatchingEnable(bool)));

		if (ui.cBox_SMode->count() == 1) {
			ui.doubleSpinBox_coinc_win->setEnabled(false);
			ui.checkBox_matching->setEnabled(false);
		}
	}
	else {
		ui.cBox_SMode->setEnabled(false);
		ui.cBox_sw_hw->setEnabled(false);
		ui.doubleSpinBox_coinc_win->setEnabled(false);
		ui.label_coinc_win->setEnabled(false);
		ui.checkBox_matching->setEnabled(false);
	}	

	msgBox.close();
}

void SystemSettings::UpdateIOLevel() {
	QString bval, global_val;
	for (int b = 0; b < mBoards->size(); b++) {
		try {
			bval = mBoards->at(b)->getIOLevel();
		}
		catch (CAENFELibException& exc) {
				mWaveDump->addLog(QString("Get value of /par/IOLevel"), false);
				ui.cBox_IOlevel->setEnabled(false);
				return;
		}

		if (b == 0)
			global_val = bval;
		else {
			if (bval != global_val) {
				ui.cBox_IOlevel->setStyleSheet("color: red");
				mIgnoreEvents = true;
				ui.cBox_IOlevel->setCurrentIndex(-1);
				mIgnoreEvents = false;
				return;
			}			
		}

	}
	int index = ui.cBox_IOlevel->findText(global_val);
	ui.cBox_IOlevel->setCurrentIndex(index);
}


void SystemSettings::currentTabChanged(int index) {
	if (mIgnoreEvents)
		return;
	Dx740V2ConfigPanel *config;
	Dx740V1ConfigPanel* config1;

	QWidget *widget = ui.tabWidget_Boards->widget(index);
	QString name = widget->metaObject()->className();
	if (name == "Dx740V2ConfigPanel") {
		config = (Dx740V2ConfigPanel *)widget;
		config->Update();
	}
	else if (name == "Dx740V1ConfigPanel") {
		config1 = (Dx740V1ConfigPanel*)widget;
		config1->Update();
	}
	
}

void SystemSettings::WindowChanged(double value_us) {
	mWaveDump->setCoincWin(value_us * 1000.);
}

void SystemSettings::FullMatchingEnable(bool enable) {
	mWaveDump->OnlyMatchingEnable(enable);
}

void SystemSettings::SWHWChanged(QString sw_hw) {
	QString new_mode = ui.cBox_SMode->currentText();
	SModeChanged(new_mode);
}

void SystemSettings::IOLevelChanged(int index) {
	if (mIgnoreEvents)
		return;
	for (int b = 0; b < mBoards->size(); b++) {
		try {
			mBoards->at(b)->setIOLevel(qPrintable(ui.cBox_IOlevel->currentText()));
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(QString("Set value of /par/IOLevel"), false);
			return;
		}
		mWaveDump->manageAddLogAndConf("IOLevel", mBoards->at(b), ui.cBox_IOlevel->currentText(), ui.cBox_IOlevel->currentText());
	}
}

void SystemSettings::OfflineSModeChanged(QString mode) {
	QString smode = mode;
	if (ui.cBox_sw_hw->currentText() == "START SOFTWARE")
		smode.prepend("SW_");
	else
		smode.prepend("HW_");
	if (smode.contains("ASYNC")) { //individual start mode
		for (int i = 0; i < mBoards->size(); i++) {
			mBoards->at(i)->DevChainIndex = -1;
		}
		ui.checkBox_matching->setEnabled(false);
	}
	else {//CHAIN start mode
		ui.checkBox_matching->setEnabled(true);
		//boards order in the chain
		QList<QString> bnames_list;
		for (int i = 0; i < mBoards->size(); i++)
			bnames_list << mBoards->at(i)->getName();
		boardsListDialog* bList = new boardsListDialog(mWaveDump, &bnames_list);
		if (bList->exec() == QDialog::Rejected) {
			ui.cBox_SMode->setCurrentText("ASYNC");
			return;
		}
		else {//assign the chain index to every board
			for (int b = 0; b < mBoards->size(); b++) {
				for (int n = 0; n < bnames_list.size(); n++) {
					if (bnames_list.at(n) == mBoards->at(b)->getName()) {
						mBoards->at(b)->DevChainIndex = n;
						break;
					}
				}
			}
		}
		
	}
	mSMode = smode;
	if (mSMode.contains("CHAIN")) {
		ui.doubleSpinBox_coinc_win->setEnabled(true);
	}
	else {
		ui.doubleSpinBox_coinc_win->setEnabled(false);
	}
	mWaveDump->setGlobalSMode(smode);
}

void SystemSettings::SModeChanged(QString mode) {
	QString smode = mode;
	if (ui.cBox_sw_hw->currentText() == "START SOFTWARE")
		smode.prepend("SW_");
	else
		smode.prepend("HW_");

	if (smode == "SW_ASYNC") { //individual start mode
		try {
			for (int i = 0; i < mBoards->size(); i++) {
				mBoards->at(i)->setGlobalStartMode(smode,0);//start mode must be sw controlled
				mWaveDump->manageAddLogAndConf("GlobalStartMode", mBoards->at(i), smode, smode);
				mBoards->at(i)->DevChainIndex = -1;
			}
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
		}
		ui.checkBox_matching->setEnabled(false);
		ui.doubleSpinBox_coinc_win->setEnabled(false);
	}
	else {//CHAIN start mode
		ui.checkBox_matching->setEnabled(true);
		ui.doubleSpinBox_coinc_win->setEnabled(true);
		//boards order in the chain
		QList<QString> bnames_list;
		for (int i = 0; i < mBoards->size(); i++)
			bnames_list << mBoards->at(i)->getName();
		boardsListDialog *bList = new boardsListDialog(mWaveDump, &bnames_list);
		if (bList->exec() == QDialog::Rejected) {
			ui.cBox_SMode->setCurrentText("ASYNC");
			return;
		}
		else {//assign the chain index to every board
			for (int b = 0; b < mBoards->size(); b++) {
				for (int n = 0; n < bnames_list.size(); n++) {
					if (bnames_list.at(n) == mBoards->at(b)->getName()) {
						mBoards->at(b)->DevChainIndex = n;
						break;
					}
				}
			}
		}

		if (smode == "SW_CHAIN_SYNC-CLKIN" || smode == "HW_CHAIN_SYNC-CLKIN") {
			try {
				for (int b = 0; b < mBoards->size(); b++) {
					int delay = (int)((mBoards->size() - mBoards->at(b)->DevChainIndex - 1) * mBoards->at(b)->TSamplUnit_ns * 6); //run delay value 
					mBoards->at(b)->setGlobalStartMode(smode,delay);
					mWaveDump->manageAddLogAndConf("GlobalStartMode", mBoards->at(b), smode, smode);
					mWaveDump->manageAddLogAndConf("RunDelay", mBoards->at(b), QString("%1").arg(delay), QString("%1").arg(delay));
				}
			}
			catch (CAENFELibException& exc) {
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
		}
		else if (smode == "SW_CHAIN_SIN-GPIO" || smode == "HW_CHAIN_SIN-GPIO") {
			try {
				for (int b = 0; b < mBoards->size(); b++) {
					int delay = (int)((mBoards->size() - mBoards->at(b)->DevChainIndex - 1) * mBoards->at(b)->TSamplUnit_ns * 10); //run delay value
					mBoards->at(b)->setGlobalStartMode(smode, delay);
					mWaveDump->manageAddLogAndConf("GlobalStartMode", mBoards->at(b), smode, smode);
					mWaveDump->manageAddLogAndConf("RunDelay", mBoards->at(b), QString("%1").arg(delay), QString("%1").arg(delay));
				}

			}
			catch (CAENFELibException& exc) {
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
		}
		else if (smode == "SW_CHAIN_SIN-TRGOUT" || smode == "HW_CHAIN_SIN-TRGOUT") {
			try {
				for (int b = 0; b < mBoards->size(); b++) {
					int delay = (int)((mBoards->size() - mBoards->at(b)->DevChainIndex - 1) * mBoards->at(b)->TSamplUnit_ns * 10); //run delay value				
					mBoards->at(b)->setGlobalStartMode(smode, delay);
					mWaveDump->manageAddLogAndConf("GlobalStartMode", mBoards->at(b), smode, smode);
					if(Ndig2_devs)
						mWaveDump->manageAddLogAndConf("RunDelay", mBoards->at(b), QString("%1").arg(delay), QString("%1").arg(delay));
				}

			}
			catch (CAENFELibException& exc) {
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
		}
		else if (smode == "SW_CHAIN_1ST_TRIGGER" || smode == "HW_CHAIN_1ST_TRIGGER") {
			try {
				for (int b = 0; b < mBoards->size(); b++) {
					int delay = (int)((mBoards->size() - mBoards->at(b)->DevChainIndex - 1) * mBoards->at(b)->TSamplUnit_ns * 10); //run delay value
					mBoards->at(b)->setGlobalStartMode(smode, delay);
					mWaveDump->manageAddLogAndConf("GlobalStartMode", mBoards->at(b), smode, smode);
					if(Ndig2_devs)
						mWaveDump->manageAddLogAndConf("RunDelay", mBoards->at(b), QString("%1").arg(delay), QString("%1").arg(delay));
				}

			}
			catch (CAENFELibException& exc) {
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
		}
		else if (smode == "SW_CHAIN_TRGIN-TRGOUT" || smode == "HW_CHAIN_TRGIN-TRGOUT") {
			try{
				for (int b = 0; b < mBoards->size(); b++) {
					int delay= (int)((mBoards->size() - mBoards->at(b)->DevChainIndex - 1) * mBoards->at(b)->TSamplUnit_ns * 4); //run delay value
					mBoards->at(b)->setGlobalStartMode(smode, delay);
					mWaveDump->manageAddLogAndConf("GlobalStartMode", mBoards->at(b), smode, smode);
					if (Ndig2_devs)
						mWaveDump->manageAddLogAndConf("RunDelay", mBoards->at(b), QString("%1").arg(delay), QString("%1").arg(delay));
				}
			}
			catch (CAENFELibException& exc) {
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
		}
		else if (smode == "SW_CHAIN_ONE2ALL_EXTOR" || smode == "HW_CHAIN_ONE2ALL_EXTOR") {
			try {
				for (int b = 0; b < mBoards->size(); b++) {
					int delay = 0;
					mBoards->at(b)->setGlobalStartMode(smode, delay);
					mWaveDump->manageAddLogAndConf("GlobalStartMode", mBoards->at(b), smode, smode);
					if (Ndig2_devs)
						mWaveDump->manageAddLogAndConf("RunDelay", mBoards->at(b), QString("%1").arg(delay), QString("%1").arg(delay));
				}
			}
			catch (CAENFELibException& exc) {
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
		}

	}

	mSMode = smode;
	if (mSMode.contains("CHAIN")) {
		ui.cBox_sw_hw->setEnabled(true);
		ui.doubleSpinBox_coinc_win->setEnabled(true);
	}
	else {
		ui.cBox_sw_hw->setEnabled(false);
		ui.doubleSpinBox_coinc_win->setEnabled(false);
	}
	mWaveDump->setGlobalSMode(smode);
	CheckClockSettings();
	//update the individual board settings
	for (int dev = 0; dev < mHandles.size(); dev++) {
		config_panels.at(dev)->Update();
	}
}

void SystemSettings::CheckClockSettings() {
	if (mWaveDump->OfflineMode)
		return;
 try{
	for (int i = 0; i < mBoards->size(); i++) {
		if (mSMode.contains("ASYNC")) {
			if (QString::compare(mBoards->at(i)->getClkSource(), "Internal", Qt::CaseInsensitive) != 0) { //clk source must be internal
				mBoards->at(i)->setClkSource("Internal");
				mWaveDump->manageAddLogAndConf("ClockSource", mBoards->at(i), "Internal", "Internal");
			}
		}
		else {
			if (QString::compare(mBoards->at(i)->getClkOutFP(), "True", Qt::CaseInsensitive) != 0) {//clk out enabled for all boards
				mBoards->at(i)->setClkOutFP("True");
				if(mBoards->at(i)->DeviceClass == CAEN_DIG2)
					mWaveDump->add2LogAndConf("SET", mBoards->at(i)->getName(), "/par/EnClockOutFP", "True", "True", CAEN_FELib_Success);
			}

			if(mBoards->at(i)->DevChainIndex == 0) { //master board, clk source must be internal
				if (QString::compare(mBoards->at(i)->getClkSource(), "Internal", Qt::CaseInsensitive) != 0) {
					mBoards->at(i)->setClkSource("Internal");
					mWaveDump->manageAddLogAndConf("ClockSource", mBoards->at(i), "Internal", "Internal");
				}
			}
			else{ //slave boards, clk source is clkin
				if (QString::compare(mBoards->at(i)->getClkSource(), "FPClkIn", Qt::CaseInsensitive) != 0) {
					mBoards->at(i)->setClkSource("FPClkIn");
					mWaveDump->manageAddLogAndConf("ClockSource", mBoards->at(i), "FPClkIn", "FPClkIn");
				}
			}

		}

	}
 }
 catch (CAENFELibException& exc) {
	 mWaveDump->addLog(exc.ErrorMessage(), false);
 }

}

/*
void SystemSettings::TrgModeChanged(QString mode) {
	if (mIgnoreEvents)
		return;

	try{
		for (int i = 0; i < mBoards->size(); i++) {
			if (mode == "COMMONT_EXT_TRGIN_TRGOUT") {
				mBoards->at(i)->setTrgSource("SwTrg|TrgIn");
				mWaveDump->add2LogAndConf("SET", mBoards->at(i)->getName(), "/par/AcqTriggerSource", "TrgIn", "TrgIn", CAEN_FELib_Success);
				mBoards->at(i)->setTRGOUTMode("TrgIn");
				mWaveDump->add2LogAndConf("SET", mBoards->at(i)->getName(), "/par/TRGOUTMode", "TrgIn", "TrgIn", CAEN_FELib_Success);
			}
			else if (mode == "INDIVIDUAL_SIN_TRGOUT") {
				mBoards->at(i)->setTrgSource("SwTrg|TrgIn|ITLA");
				mWaveDump->add2LogAndConf("SET", mBoards->at(i)->getName(), "/par/AcqTriggerSource", "TrgIn|ITLA","TrgIn|ITLA", CAEN_FELib_Success);
				mBoards->at(i)->setTRGOUTMode("Run");
				mWaveDump->add2LogAndConf("SET", mBoards->at(i)->getName(), "/par/TRGOUTMode", "Run", "Run", CAEN_FELib_Success);
			}
			else {
				mBoards->at(i)->setTrgSource("SwTrg");
				mWaveDump->add2LogAndConf("SET", mBoards->at(i)->getName(), "/par/AcqTriggerSource", "SwTrg","SwTrg", CAEN_FELib_Success);
				mBoards->at(i)->setTRGOUTMode("Disabled");
				mWaveDump->add2LogAndConf("SET", mBoards->at(i)->getName(), "/par/TRGOUTMode", "Disabled", "Disabled", CAEN_FELib_Success);
			}
		}
		mTMode = mode;
		mWaveDump->setGlobalTrgMode(mode);
		//update the individual board settings
		for (int dev = 0; dev < mHandles.size(); dev++) {
			config_panels.at(dev)->Update();
		}
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}
}*/

SystemSettings::~SystemSettings()
{
	config_panels.clear();
}
