/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		OscilloscopeTools.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "OscilloscopeTools.h"
#include "plotPanel.h"


OscilloscopeTools::OscilloscopeTools(plotPanel *parent)
	: QWidget(parent)
	, maxX{}
	, minX{}
	, maxY{}
	, minY{}
	, maxXPos{}
	, maxThr{}
{
	ui.setupUi(this);
	mPlot = parent;

	connect(ui.dial_x, SIGNAL(valueChanged(int)), this, SLOT(XChanged(int)));
	connect(ui.dial_y, SIGNAL(valueChanged(int)), this, SLOT(YChanged(int)));
	connect(ui.Slider_x, SIGNAL(valueChanged(int)), this, SLOT(XPosChanged(int)));
	connect(ui.dBox_xpos, SIGNAL(valueChanged(double)), this, SLOT(XPosChanged(double)));
	connect(ui.Slider_y, SIGNAL(valueChanged(int)), this, SLOT(OffsetChanged(int)));
	connect(ui.dBox_offset, SIGNAL(valueChanged(double)), this, SLOT(OffsetChanged(double)));
	connect(ui.cBox_active_trace, SIGNAL(currentIndexChanged(int)), this, SLOT(ActiveTraceChanged(int)));
	connect(ui.dBox_thr, SIGNAL(valueChanged(double)), this, SLOT(ThrChanged(double)));
	connect(ui.cBox_thr_show, SIGNAL(stateChanged(int)), mPlot, SLOT(ThrShow(int)));
	connect(ui.cBox_xpos_show, SIGNAL(stateChanged(int)), mPlot, SLOT(XPosShow(int)));
	connect(ui.pB_offset_to_all, SIGNAL(clicked()), this, SLOT(OffsetToAll()));
	connect(ui.pB_thr_to_all, SIGNAL(clicked()), this, SLOT(ThresholdToAll()));
}

void OscilloscopeTools::UpdateGUI() {	
	updateXMax();
	updateYMax();
	updateTraces();
}

void OscilloscopeTools::updateTraces() {
	ui.cBox_active_trace->blockSignals(true);
	ui.cBox_active_trace->clear();
	for (int i = 0; i < mPlot->getTraces(); i++) {
		ui.cBox_active_trace->addItem(createIcon(mPlot->getTraceColor(i)), QString("Trace %1").arg(i), mPlot->getTraceColor(i));
	}
	ui.cBox_active_trace->setCurrentIndex(mPlot->ActiveTrace);
	ui.cBox_active_trace->blockSignals(false);
}

QIcon OscilloscopeTools::createIcon(QColor color){
	QPixmap iconPixmap(32, 32);
	iconPixmap.fill(color);
	QIcon itemIcon(iconPixmap);
	return itemIcon;
}


void OscilloscopeTools::updateXMax(){
	mIgnoreEvent = true;
	maxX = mPlot->getMaxX();
	minX = mPlot->getMinX();

	ui.dBox_x->setMaximum((maxX-minX) / X_TICKS);

	mUomx = mPlot->getUoMX();
	mIgnoreEvent = false;
}

void OscilloscopeTools::updateYMax() {
	mIgnoreEvent = true;
	maxY = mPlot->getMaxY();
	minY = mPlot->getMinY();

	ui.dBox_y->setMaximum(fabs(maxY-minY) / Y_TICKS);

	mUomy = mPlot->getUoMY();
	mIgnoreEvent = false;
	YChanged(ui.dial_y->value());
}
void OscilloscopeTools::XChanged(int step){
	if (mIgnoreEvent)
		return;
	double max_value_div = fabs(maxX - minX) / X_TICKS; //no zoom with this value/div
	double new_value_div = max_value_div / (qPow(2, step));
	double new_range = new_value_div * X_TICKS;
	double current_pos = mPlot->getZoomCenter()*(maxX - minX) + minX;
	mPlot->getXAxis()->setRange( current_pos - mPlot->getZoomCenter()*new_range, current_pos + (1- mPlot->getZoomCenter())*new_range);
	mPlot->doReplot();
	mIgnoreEvent = true;
	if(mUomx==UOM_SAMPLE && mPlot->PlotType != PLOT_TYPE_FFT)
		ui.dBox_x->setValue(qRound(new_value_div));
	else
		ui.dBox_x->setValue(new_value_div);
	mIgnoreEvent = false;
}
void OscilloscopeTools::YChanged(int step) {
	if (mIgnoreEvent)
		return;
	double max_value_div = fabs(maxY - minY) / Y_TICKS; //no zoom with this value/div
	double new_value_div = max_value_div / (qPow(2,step));

	mIgnoreEvent = true;
	if (mPlot->PlotType == PLOT_TYPE_WAVEFORMS) {
		if (mUomy == UOM_SAMPLE) {
			ui.dBox_y->setValue(qRound(new_value_div));
			double current_pos = ui.dBox_offset->value() * (fabs(maxY - minY)) / 100.;
			double new_range = new_value_div * Y_TICKS;
			mPlot->getYAxis()->setRange(current_pos - (ui.dBox_offset->value() / 100.) * new_range, current_pos + (1 - ui.dBox_offset->value() / 100.) * new_range);
		}
		else {
			mPlot->getYAxis()->setRange(minY / qPow(2, step), maxY / qPow(2, step));
			ui.dBox_y->setValue(new_value_div);
		}
	}
	mIgnoreEvent = false;
	mPlot->doReplot();
}
void OscilloscopeTools::OffsetChanged(int value) {
	if (mIgnoreEvent)
		return;
	mPlot->setNewOffset(value);
}

void OscilloscopeTools::OffsetChanged(double value) {
	if (mIgnoreEvent)
		return;
	mPlot->setNewOffset(value);
}

void OscilloscopeTools::OffsetToAll() {
	mPlot->OffsetToAll(ui.dBox_offset->value());
}

void OscilloscopeTools::ThresholdToAll() {
	mPlot->ThresholdToAll(ui.dBox_thr->value());
}


void OscilloscopeTools::XPosChanged(int value) {
	if (mIgnoreEvent)
		return;
	mIgnoreEvent = true;
	mPlot->setXPos(static_cast<double>(value) / 100.);
	double c = mPlot->getZoomCenter();
	double test = mPlot->getZoomCenter() * fabs(maxX - minX) + minX;
	ui.dBox_xpos->setValue(mPlot->getZoomCenter()*fabs(maxX-minX) + minX);
	mIgnoreEvent = false;
}

void OscilloscopeTools::XPosChanged(double value) {
	if (mIgnoreEvent)
		return;
	mIgnoreEvent = true;
	double pos = (value - minX) / fabs(maxX - minX);
	mPlot->setXPos(pos);
	ui.Slider_x->setValue(qRound(pos * 100));
	mIgnoreEvent = false;
}

void OscilloscopeTools::updateXUnit(int unit) {
	mUomx = unit;
	maxX = mPlot->getMaxX();
	minX = mPlot->getMinX();
	mIgnoreEvent = true;
	ui.dBox_x->setMaximum(fabs(maxX - minX) / X_TICKS);
	double max_value_div = fabs(maxX - minX) / X_TICKS; //no zoom with this value/div
	double new_value_div = max_value_div / qPow(2, ui.dial_x->value());

	QString mu = QChar(0xbc, 0x03);
	switch (mPlot->PlotType) {
		case PLOT_TYPE_WAVEFORMS:
			switch (mUomx) {
				case UOM_SAMPLE:
					ui.label_x_unit->setText("[Samples/Div]");
					ui.label->setText("Position [Samples]");
					ui.dBox_x->setValue(qRound(new_value_div));
					break;
				case UOM_PHYS_UNIT:
					ui.label_x_unit->setText("[" + mu + "s/Div]");
					ui.label->setText("Position [" + mu + "s]");
					ui.dBox_x->setValue(new_value_div);
					break;
			}
		break;
		case PLOT_TYPE_FFT:
			ui.label_x_unit->setText("[MHz/Div]");
			ui.label->setText("Position [MHz]");
			ui.dBox_x->setValue(new_value_div);
			break;
		case PLOT_TYPE_SAMPLES_HISTO:
			switch (mUomy) {
			case UOM_SAMPLE:
				ui.label_x_unit->setText("[ADC channels/Div]");
				ui.label->setText("Position [Samples]");
				ui.dBox_x->setValue(qRound(new_value_div));
				break;
			case UOM_PHYS_UNIT:
				ui.label_x_unit->setText("[mV/Div]");
				ui.label->setText("Position [mV]");
				ui.dBox_x->setValue(new_value_div);
				break;
			}
			break;
	}
	mIgnoreEvent = false;
}
void OscilloscopeTools::updateYUnit(int unit) {
	mUomy = unit;
	maxY = mPlot->getMaxY();
	minY = mPlot->getMinY();
	mIgnoreEvent = true;
	ui.dBox_y->setMaximum(fabs(maxY-minY) / Y_TICKS);
	double max_value_div = fabs(maxY - minY) / Y_TICKS; //no zoom with this value/div
	double new_value_div = max_value_div / qPow(2, ui.dial_y->value());
	ui.dBox_y->setValue(qRound(new_value_div));
	QString mu = QChar(0xbc, 0x03);
	QString text;
	switch (mPlot->PlotType) {
		case PLOT_TYPE_WAVEFORMS:
			switch (mUomy) {
			case UOM_SAMPLE:
				ui.label_y_unit->setText("[ADC channels/Div]");
				text = "Threshold (rel.) [LSB]";
				if (mDClass == DIG1)
					text.replace("rel","abs");
				ui.label_thr->setText(text);
				break;
			case UOM_PHYS_UNIT:
				ui.label_y_unit->setText("[mV/Div]");
				text = "Threshold (rel.) [mV]";
				if (mDClass == DIG1)
					text.replace("rel", "abs");
				ui.label_thr->setText(text);
				break;
			}
		break;
		case PLOT_TYPE_FFT:
			ui.label_y_unit->setText("[dB/Div]");
			break;
		case PLOT_TYPE_SAMPLES_HISTO:
			ui.label_y_unit->setText("[Counts/Div]");
			break;
	}
	mIgnoreEvent = false;
}

void OscilloscopeTools::updateXPos(QVector<double> info) {
	mIgnoreEvent = true;
	double val;
	if (mPlot->PlotType != PLOT_TYPE_SAMPLES_HISTO) {
		double scale_factor = info.at(4);
		maxXPos = info.at(0) * scale_factor;
		ui.dBox_xpos->setMaximum(maxXPos);
		ui.dBox_xpos->setMinimum(0);//info.at(1) * scale_factor
		ui.dBox_xpos->setSingleStep(info.at(2) * scale_factor);
		val = mPlot->getZoomCenter() * info.at(3) * scale_factor;
	}
	else {
		maxXPos = mPlot->getMaxX();
		ui.dBox_xpos->setMaximum(maxXPos);
		double minXPos = mPlot->getMinX();
		ui.dBox_xpos->setMinimum(0);//minXPos
		ui.dBox_xpos->setSingleStep(1);
		val = mPlot->getZoomCenter() * fabs(maxX - minX) + minX;
	}

	ui.dBox_xpos->setValue(val);
	ui.Slider_x->setValue(qRound(mPlot->getZoomCenter() * 100.));
	mIgnoreEvent = false;
}

void OscilloscopeTools::updateOffset(double val) {
	mIgnoreEvent = true;
	ui.Slider_y->blockSignals(true);
	ui.dBox_offset->blockSignals(true);
	ui.dBox_offset->setValue(val);
	ui.Slider_y->setValue(qRound(val));
	ui.Slider_y->blockSignals(false);
	ui.dBox_offset->blockSignals(false);
	mIgnoreEvent = false;
}

void OscilloscopeTools::updateThr(QVector<double> info) {
	mIgnoreEvent = true;
	double scale_factor = info.at(4);
	maxThr = info.at(0) * scale_factor;
	ui.dBox_thr->setMaximum(maxThr);
	ui.dBox_thr->setMinimum(info.at(1) * scale_factor);
	ui.dBox_thr->setSingleStep(info.at(2) * scale_factor);
	ui.dBox_thr->setValue(info.at(3) * scale_factor);
	mIgnoreEvent = false;
}

void OscilloscopeTools::ThrChanged(double value) {
	if (mIgnoreEvent)
		return;
	mIgnoreEvent = true;
	mPlot->setThr(value);
	mIgnoreEvent = false;
}

void OscilloscopeTools::plotScaleChanged() {
	if (mPlot->getPlotScale() == PLOT_SCALE_ABSOLUTE)
		ui.label_thr->setText("Threshold (abs.) [LSB]");
	else {
		QString text = "Threshold (rel.) ";
		if (mUomy == UOM_SAMPLE) {
			text.append("[LSB]");
			if (mDClass == DIG1)
				text.replace("rel","abs");
			ui.label_thr->setText(text);
		}
		else {
			text.append("[mV]");
			if (mDClass == DIG1)
				text.replace("rel", "abs");
			ui.label_thr->setText(text);
		}
	}
}

void OscilloscopeTools::setActiveDevClass(int dclass) {
	mDClass = dclass;
	if (dclass == DIG2)
		return;
	QString text = ui.label_thr->text();
	if (text.contains("rel.")) {
		text.replace("rel","abs");
		ui.label_thr->setText(text);
	}
	
}

void OscilloscopeTools::refreshXPos() {
	maxXPos = mPlot->getMaxX();
	ui.dBox_xpos->setMaximum(maxXPos);
	double minXPos = mPlot->getMinX();
	ui.dBox_xpos->setMinimum(0);//minXPos
	XPosChanged(ui.Slider_x->value());
}

void OscilloscopeTools::plotTypeChanged(int type) {
	if (type != PLOT_TYPE_WAVEFORMS) {
		ui.groupBox_2->setEnabled(false);
		ui.cBox_thr_show->setChecked(false);
	}
	else
		ui.groupBox_2->setEnabled(true);
	updateXMax();
	updateXUnit(mUomx);
	refreshXPos();
}

void OscilloscopeTools::ActiveTraceChanged(int index) {
	if (!mPlot->isTraceVisible(index)) {
		QMessageBox::critical(this, tr("Trace error"), tr("Trace is not enabled for plotting."), QMessageBox::Ok, 0);
		ui.cBox_active_trace->blockSignals(true);
		ui.cBox_active_trace->setCurrentIndex(mPlot->ActiveTrace);
		ui.cBox_active_trace->blockSignals(false);
		return;
	}
	mPlot->setActiveTrace(index);
}

void OscilloscopeTools::enableOfflineMode(bool enable){
	if (enable) {
		ui.Slider_y->setEnabled(false);
		ui.dBox_offset->setEnabled(false);
		ui.dBox_thr->setEnabled(false);
		ui.cBox_thr_show->setEnabled(false);
		ui.pB_offset_to_all->setEnabled(false);
	}
	else {
		ui.Slider_y->setEnabled(true);
		ui.dBox_offset->setEnabled(true);
		ui.dBox_thr->setEnabled(true);
		ui.cBox_thr_show->setEnabled(true);
		ui.pB_offset_to_all->setEnabled(true);
	}
}

OscilloscopeTools::~OscilloscopeTools()
{
}
