/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		FFT.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "FFT.h"
#include <qmath.h>

FFT::FFT(void)
{
    m_rawDataLength = 0;
};


void FFT::Init(int Wsamples) {
    m_x.resize(Wsamples);
    m_y.resize(Wsamples);
}

//calc the number of samples of the fft given the number of waveform samples
int FFT::NSamplesFFT(int Wsamples, int *M) {
    int n, m;
    // ns should be a power of 2. If it is not, find the maximum m
    // such that n = 2^m < ns and get n samples instead of ns.
    int i = 1;
    while ((1 << i) < Wsamples)
        i++;
    if (Wsamples == (1 << i)) {
        m = i;
        n = Wsamples;
    }
    else {
        m = i - 1;
        n = (1 << m);
    }

    *M = m;
    return n;//fft is symmetric, use half samples
}

int FFT::CalculateFFT(std::uint16_t* wave, int ns, int Nbit, int WindowType, double* fft) {
    int m=0, n, ip, le, le1, nm1, k, l, j, i, nv2;
    double u1, u2, u3, arg, c, s, t1, t2, t3, t4;
    int NormFactor = (1 << Nbit) / 2;  // Max amplitude A of a sin wave is ADC_FullScale/2

    m_x.fill(0);
    m_y.fill(0);

    n = NSamplesFFT(ns, &m);

    double win_sum = 0.;

    // apply the windowing to the input vector
    for (i = 0; i < n; i++) {
        double win_coeff;
        switch (WindowType) {
        case HANNING_FFT_WINDOW:
            win_coeff = (0.5 - 0.5 * cos(2. * M_PI * i / n));
            break;
        case HAMMING_FFT_WINDOW:
            win_coeff = (0.54 - 0.46 * cos(2. * M_PI * i / n));
            break;
        case RECT_FFT_WINDOW:
            win_coeff = 1.;
            break;
        case BLACKMAN_FFT_WINDOW:
        default:
            win_coeff = (0.42 - 0.50 * cos(2. * M_PI * i / n) + 0.08 * cos(4. * M_PI * i / n));
            break;
        }
        m_y[i] = 0.0; // imaginary part of the input vector (always 0)
        m_x[i] = wave[i] * win_coeff;
        win_sum += win_coeff; // needed to normalize the output
    }

    double win_scale = 1. / win_sum;

    // now the vectors x and y represents the input waveform expressed as imaginary numbers
    // after the application of the windowing.

    // calculate the FFT
    for (l = 0; l < m; l++) {
        le = 1 << (m - l);
        le1 = le / 2;
        u1 = 1.0;
        u2 = 0.0;
        arg = M_PI / le1;
        c = cos(arg);
        s = -sin(arg);

        for (j = 0; j < le1; j++) {
            for (i = j; i < n; i = i + le) {
                ip = i + le1;
                t1 = m_x[i] + m_x[ip];
                t2 = m_y[i] + m_y[ip];
                t3 = m_x[i] - m_x[ip];
                t4 = m_y[i] - m_y[ip];
                m_x[ip] = t3 * u1 - t4 * u2;
                m_y[ip] = t4 * u1 + t3 * u2;
                m_x[i] = t1;
                m_y[i] = t2;
            }
            u3 = u1 * c - u2 * s;
            u2 = u2 * c + u1 * s;
            u1 = u3;
        }
    }

    nv2 = n / 2;
    nm1 = n - 1;
    j = 0;
    i = 0;
    while (i < nm1) {
        if (i > j) goto rif;
        t1 = m_x[j];
        t2 = m_y[j];
        m_x[j] = m_x[i];
        m_y[j] = m_y[i];
        m_x[i] = t1;
        m_y[i] = t2;
    rif:
        k = nv2;
    rife:
        if (k > j) goto rifer;
        j = j - k;
        k = k / 2;
        goto rife;
    rifer:
        j = j + k;
        i++;
    }

    // get the amplitude of the FFT and apply window scale
    for (i = 0; i < n / 2; i++) {
        fft[i] = hypot(m_x[i], m_y[i]) * win_scale;
    }

    // real fft is symmetric: scale by sqrt(2) all points except the first that is unpaired
    // important: we need sqrt(2), not 2
    for (i = 1; i < n / 2; i++) {
        fft[i] *= M_SQRT2;
    }

    // Add the baseline, normalize and transform in dBFS
    for (i = 0; i < n / 2; i++) {
        fft[i] = 20 * log10(fft[i] / NormFactor + FFT_BASELINE);
    }

    return (n / 2);
}


FFT::~FFT(void)
{
}

