/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		Dx740V2ConfigPanel.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "Dx740V2ConfigPanel.h"
#include "WaveDump2.h"
#include "CAENSwitch.h"
#include "CAENFELibException.h"
#include "chmaskDialog.h"
#include "DACOUTDialog.h"


Dx740V2ConfigPanel::Dx740V2ConfigPanel(WaveDump2 *parent, CAENDig2Device *dev, QString dev_name, bool show_wait) :
	QDialog(parent),
	ui(new Ui::Dx740V2ConfigPanel) {
	ui->setupUi(this);

	setWindowIcon(QIcon(":WAVEDump2.png"));
	QMessageBox msgBox(this);
	if (show_wait) {		
		msgBox.setText("<html><B>Reading device settings <br>           ... please wait ...</html>");
		msgBox.setStandardButtons({});
		msgBox.show();
		QApplication::processEvents();
	}

	mWaveDump = parent;
	mDevHandle = dev->getHandle();
	mDevName = dev_name;
	mDev = dev;
	mUoMX = mWaveDump->getUoMX();
	mUoMY = mWaveDump->getUoMY();	

	CAENSwitch pippo("on","off",1,this);
	ui->lay_0->addWidget(&pippo);
	
	mNumberOfChannels = mDev->getNumCh();

	for (int i = 0; i < mNumberOfChannels; i++) {
		mPbCh[i] = new CAENPushButton(this, i);
		mPbCh[i]->setMaximumWidth(60);
		mPbCh[i]->setText("CH " + QString::number(i));
		mPbCh[i]->setCheckable(true);
		connect(mPbCh[i], &CAENPushButton::clicked, [=] {channelClicked(mPbCh[i]); });
	}

	for (int i = 0; i < int(mNumberOfChannels / 16); i++) {
		for (int j = 0; j < 16; j++) {
			switch (i) {
				case 0:
					ui->lay_0->addWidget((mPbCh[(i * 16) + j]));
					break;
				case 1:
					ui->lay_1->addWidget(mPbCh[(i * 16) + j]);
					break;
				case 2:
					ui->lay_2->addWidget(mPbCh[(i * 16) + j]);
					break;
				case 3:
					ui->lay_3->addWidget(mPbCh[(i * 16) + j]);
					break;
			}
		}
	}

	QString ModelName = mDev->getModel();
	if (ModelName.contains("2745")) {
		ui->dBox_gain->setVisible(true);
		ui->label_17->setVisible(true);
		mVGAGainVisible = true;
		const size_t handles_size = 1024;
		uint64_t handles[handles_size];
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetChildHandles(mDevHandle, "/vga/", handles, handles_size);
		if (err < 0 || err > handles_size) {
			mWaveDump->addLog("Error getting VgaGain", false);
		}
		else
			mGainGroupSize = mNumberOfChannels / err;
	}
	else if (ModelName.contains("2730") || ModelName.contains("2751")) {
		ui->dBox_gain->setVisible(true);
		ui->label_17->setVisible(true);
		ui->label->setVisible(false);
		ui->cBox_decim->setVisible(false);
		ui->label_freq_decimation->setVisible(false);
		mDecimationVisible = false;
		mCHGainVisible = true;
	}
	else {
		ui->dBox_gain->setVisible(false);
		ui->label_17->setVisible(false);
	}

	mPbCh[0]->setChecked(true);
	mFirstSelectedCh = 0;
	mFirstSelectedGroup.clear();
	mIgnoreEvent = false;
	mMulti = false;

	ui->pB_trg_mask->setEnabled(false);
	ui->pB_trgout_mask->setEnabled(false);


	if (!mDev->getFWType().contains("Open", Qt::CaseInsensitive)) {
			ui->cBox_sel_wave->setEnabled(false);
			ui->cBox_sel_wave->hide();
			ui->label_10->hide();
	}
	

	Init();

	UpdateParamsUoM();
	QApplication::processEvents();
	UpdateDevGUI();
	QApplication::processEvents();
	UpdateChGUI();
	QApplication::processEvents();
	UpdateGroupGUI();
	QApplication::processEvents();
	UpdateMainSettings();
	if(mDecimationVisible)
		UpdateSamplFreq();

	if (show_wait)
		msgBox.close();
}

void Dx740V2ConfigPanel::UpdateMainSettings() {
	UpdateTriggerMode();
	UpdateTrgoutMode();
	UpdateVetoSrc();
	UpdateGpioMode();
}

void Dx740V2ConfigPanel::connectWidgets() {
	//connections
	connect(ui->pbAll, SIGNAL(clicked()), this, SLOT(selectAll()));
	connect(ui->pbMulti, SIGNAL(clicked()), this, SLOT(MultiClicked()));

	connect(ui->lWidget_StartSrc, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemCheckedChanged(QListWidgetItem *)));
	connect(ui->lWidget_TReset_Src, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemCheckedChanged(QListWidgetItem *)));
	connect(ui->lWidget_Trg_Src, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemCheckedChanged(QListWidgetItem *)));
	connect(ui->dBox_reclen, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dBox_pretrg, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));


	connect(ui->cBox_enable, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_polarity, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->dSBox_bsl_level, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dBox_gain, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dBox_thr, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->cBox_decim, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));

	connect(ui->cBox_TRGOUT, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_GPIO, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->lWidget_VETOSrc, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemCheckedChanged(QListWidgetItem *)));
	connect(ui->dBox_VETO_width, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->cBox_VETOPol, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_trg_mode, SIGNAL(currentTextChanged(QString)), this, SLOT(TriggerModeChanged(QString)));
	connect(ui->cBox_hfreq, SIGNAL(stateChanged(int)), this, SLOT(HFRejChanged(int)));
	connect(ui->cBox_trgout1, SIGNAL(currentTextChanged(QString)), this, SLOT(TrgoutChanged(QString)));
	connect(ui->lineEdit_trig_mask, SIGNAL(textChanged(QString)), this, SLOT(TextChanged(QString)));
	connect(ui->lineEdit_trgout_mask, SIGNAL(textChanged(QString)), this, SLOT(TextChanged(QString)));
	connect(ui->pB_trg_mask, SIGNAL(clicked()), this, SLOT(SetMask()));
	connect(ui->pB_trgout_mask, SIGNAL(clicked()), this, SLOT(SetTrgoutMask()));
	connect(ui->cBox_veto1, SIGNAL(currentTextChanged(QString)), this, SLOT(VetoChanged(QString)));
	connect(ui->cBox_gpio1, SIGNAL(currentTextChanged(QString)), this, SLOT(GpioChanged(QString)));
	connect(ui->cBox_clk_source, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->gBox_clock_out, SIGNAL(clicked(bool)), this, SLOT(boolChanged(bool)));
	connect(ui->sBox_v_clk_delay, SIGNAL(valueChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->pushButton_permanent_clock, SIGNAL(clicked()), this, SLOT(SetPermanentClockDelay()));
	if(ui->cBox_sel_wave->isEnabled())
		connect(ui->cBox_sel_wave, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->tabWidget, SIGNAL(currentChanged(int)), this, SLOT(currentTabChanged(int)));

	connect(ui->pBDACOUT, SIGNAL(clicked()), this, SLOT(ConfigureDACOUT()));
}

void Dx740V2ConfigPanel::FillParamsAndWidgetsList() {
	//start creating the list of device params
	mDevParamsList.clear();
	mDevWidgetsList.clear();

	mDevParamsList << "/par/RecordLengthS";
	mDevWidgetsList << ui->dBox_reclen;
	ui->dBox_reclen->setToolTip("/par/RecordLengthS");

	mDevParamsList << "/par/StartSource";
	mDevWidgetsList << ui->lWidget_StartSrc;
	ui->lWidget_StartSrc->setToolTip("/par/StartSource");

	mDevParamsList << "/par/AcqTriggerSource";
	mDevWidgetsList << ui->lWidget_Trg_Src;
	ui->lWidget_Trg_Src->setToolTip("/par/AcqTriggerSource");

	mDevParamsList << "/par/TstampResetSource";
	mDevWidgetsList << ui->lWidget_TReset_Src;
	ui->lWidget_TReset_Src->setToolTip("/par/TstampResetSource");

	mDevParamsList << "/par/PreTriggerS";
	mDevWidgetsList << ui->dBox_pretrg;
	ui->dBox_pretrg->setToolTip("/par/PreTriggerS");

	mDevParamsList << "/par/TrgOutMode";
	mDevWidgetsList << ui->cBox_TRGOUT;
	ui->cBox_TRGOUT->setToolTip("/par/TrgOutMode");

	mDevParamsList << "/par/GPIOMode";
	mDevWidgetsList << ui->cBox_GPIO;
	ui->cBox_GPIO->setToolTip("/par/GPIOMode");

	mDevParamsList << "/par/VetoSource";
	mDevWidgetsList << ui->lWidget_VETOSrc;
	ui->lWidget_VETOSrc->setToolTip("/par/VetoSource");

	mDevParamsList << "/par/VetoWidth";
	mDevWidgetsList << ui->dBox_VETO_width;
	ui->dBox_VETO_width->setToolTip("/par/VetoWidth");

	mDevParamsList << "/par/VetoPolarity";
	mDevWidgetsList << ui->cBox_VETOPol;
	ui->cBox_VETOPol->setToolTip("/par/VetoPolarity");

	mDevParamsList << "/par/ITLAMask";
	mDevWidgetsList << ui->lineEdit_trig_mask;
	ui->lineEdit_trig_mask->setToolTip("/par/ITLAMask");

	mDevParamsList << "/par/ITLBMask";
	mDevWidgetsList << ui->lineEdit_trgout_mask;
	ui->lineEdit_trgout_mask->setToolTip("/par/ITLBMask");

	if (mDecimationVisible) {
		mDevParamsList << "/par/DecimationFactor";
		mDevWidgetsList << ui->cBox_decim;
		ui->cBox_decim->setToolTip("/par/DecimationFactor");
	}

	mDevParamsList << "/par/ClockSource";
	mDevWidgetsList << ui->cBox_clk_source;
	ui->cBox_clk_source->setToolTip("/par/ClockSource");

	mDevParamsList << "/par/EnClockOutFP";
	mDevWidgetsList << ui->gBox_clock_out;
	ui->gBox_clock_out->setToolTip("/par/EnClockOutFP");

	mDevParamsList << "/par/VolatileClockOutDelay";
	mDevWidgetsList << ui->sBox_v_clk_delay;
	ui->sBox_v_clk_delay->setToolTip("/par/VolatileClockOutDelay");

	if (ui->cBox_sel_wave->isEnabled()) {
		mDevParamsList << "/par/SelectWave";
		mDevWidgetsList << ui->cBox_sel_wave;
		ui->cBox_sel_wave->setToolTip("/par/SelectWave");
	}

	//add more device parameters if needed ...

	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mDevWidgetsList.size(); i++)
		mDevWidgetsList.at(i)->setStyleSheet("color: black");

	//create the list of channel params
	mChParamsList.clear();
	mChWidgetsList.clear();

	mChParamsList << QString("/ch/%1/par/ChEnable");
	mChWidgetsList << ui->cBox_enable;
	ui->cBox_enable->setToolTip("/ch/N/par/ChEnable");

	mChParamsList << QString("/ch/%1/par/SelfTriggerEdge");
	mChWidgetsList << ui->cBox_polarity;
	ui->cBox_polarity->setToolTip("/ch/N/par/SelfTriggerEdge");

	mChParamsList << QString("/ch/%1/par/DCOffset");
	mChWidgetsList << ui->dSBox_bsl_level;
	ui->dSBox_bsl_level->setToolTip("/ch/N/par/DCOffset");

	mChParamsList << QString("/ch/%1/par/TriggerThr");
	mChWidgetsList << ui->dBox_thr;
	ui->dBox_thr->setToolTip("/ch/N/par/TriggerThr");

	mChParamsList << QString("/ch/%1/par/SamplesOverThreshold");
	mChWidgetsList << ui->cBox_hfreq;


	if (mCHGainVisible) {
		mChParamsList << QString("/ch/%1/par/ChGain");
		mChWidgetsList << ui->dBox_gain;
		ui->dBox_gain->setToolTip("/ch/N/par/ChGain");
	}

	//add more channel parameters if needed
	mNumberOfChParams = mChParamsList.size();
	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mChWidgetsList.size(); i++)
		mChWidgetsList.at(i)->setStyleSheet("color: black");

	//create the list of group params
	mGroupParamsList.clear();
	mGroupWidgetsList.clear();
	mGroupsize.clear();

	if (mVGAGainVisible) {
		mGroupParamsList << QString("/vga/%1/par/VgaGain");
		mGroupWidgetsList << ui->dBox_gain;
		ui->dBox_gain->setToolTip("/vga/N/par/VgaGain");
		mGroupsize << mGainGroupSize; 
	}

	//add more group parameters if needed
	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mGroupWidgetsList.size(); i++)
		mGroupWidgetsList.at(i)->setStyleSheet("color: black");
	mNumberOfGroupParams = mGroupParamsList.size();
	for (int g = 0; g < mNumberOfGroupParams; g++)
		mFirstSelectedGroup << 0; //for every group param depending on the group size there is a first selected group
}

void Dx740V2ConfigPanel::UpdateParamsUoM() {
	if (mUoMX == UOM_PHYS_UNIT) {
		mDevGUIMap.value(ui->dBox_reclen)->setUoMToGUIUoM(mWaveDump->getSampleToS() * 1e6);
		mDevGUIMap.value(ui->dBox_reclen)->setUoMToGUIUoM_incr(-mDev->RecLenJIncrS*(mDev->getSample_to_S()) * 1e6);
		mDevGUIMap.value(ui->dBox_pretrg)->setUoMToGUIUoM(mWaveDump->getSampleToS() * 1e6);
		mDevGUIMap.value(ui->dBox_VETO_width)->setUoMToGUIUoM(mWaveDump->getSampleToS() * 1e6);
		QString mu = QChar(0xbc, 0x03);
		ui->label_pretrig_unit->setText("[" + mu + "s]");
		ui->label_reclen_unit->setText("[" + mu + "s]");
		ui->label_veto_width_unit->setText("[" + mu + "s]");
	}
	else {
		ui->label_pretrig_unit->setText("[samples]");
		ui->label_reclen_unit->setText("[samples]");
		ui->label_veto_width_unit->setText("[samples]");
		mDevGUIMap.value(ui->dBox_reclen)->setUoMToGUIUoM_incr(-mDev->RecLenJIncrS);
	}

	if (mUoMY == UOM_PHYS_UNIT) {
		int index = mChWidgetsList.indexOf(ui->dBox_thr);
		for (int ch = 0; ch < mNumberOfChannels; ch++) {
			double f = (mWaveDump->getDevChSampleToV(mDevName, ch) * 1e3); 
			mChParams.at((mNumberOfChParams * ch) + index)->setUoMToGUIUoM(f);
		}
		ui->label_thr_unit->setText("[mV]");
	}
	else {
		ui->label_thr_unit->setText("[samples]");
	}

}

void Dx740V2ConfigPanel::createMaps() {
	uint64_t phandle=0;
	CAENparameter *new_param=nullptr;

	for (int p = 0; p < mDevParamsList.size(); p++) {
		phandle = 0;
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle, qPrintable(mDevParamsList.at(p)), &phandle);
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Get Handle of " + mDevParamsList.at(p) + " error %1").arg(err), false);
			mDevWidgetsList.at(p)->setEnabled(false);
			mDevGUIMap.insert(mDevWidgetsList.at(p), nullptr);
			continue;
		}

		try {
			new_param = new CAENparameter(phandle);
			new_param->SetWidget(mDevWidgetsList.at(p)); //assign the widget
		}
		catch (CAENFELibException& exc) {
			mDevGUIMap.insert(mDevWidgetsList.at(p), nullptr);
			mWaveDump->addLog(exc.ErrorMessage(), false);
		}
		mDevGUIMap.insert(mDevWidgetsList.at(p), new_param); //fill the map
		if(new_param == nullptr)
			mWaveDump->addLog("GET " + mDevParamsList.at(p), 0);
	}

	for (int ch = 0; ch < mNumberOfChannels; ch++) {
		for (int p = 0; p < mNumberOfChParams; p++) {
			phandle = 0;
			CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle, qPrintable(mChParamsList.at(p).arg(ch)), &phandle);
			if (err != CAEN_FELib_Success) {
				mWaveDump->addLog(QString("Get Handle of " + mChParamsList.at(p).arg(ch) + " error %1").arg(err), false);
				mChWidgetsList.at(p)->setEnabled(false);
				mChParams.push_back(nullptr);
				continue;
			}
			CAENparameter *new_param = nullptr;
			try {
				new_param = new CAENparameter(phandle);
				new_param->SetWidget(mChWidgetsList.at(p));
			}
			catch (CAENFELibException& exc) {
				mChParams.push_back(new_param);
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
			mChParams.push_back(new_param);
			if (new_param == nullptr)
				mWaveDump->addLog("GET" + mChParamsList.at(p), 0);
		}
	}

	CreateChParamsMap();

	for (int p = 0; p < mNumberOfGroupParams; p++) {
		for (int g = 0; g < (mNumberOfChannels/mGroupsize.at(p)); g++) {
			phandle = 0;
			CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle, qPrintable(mGroupParamsList.at(p).arg(g)), &phandle);
			if (err != CAEN_FELib_Success) {
				mWaveDump->addLog(QString("Get Handle of " + mGroupParamsList.at(p).arg(g) + " error %1").arg(err), false);
				mGroupWidgetsList.at(p)->setEnabled(false);
				mGroupParams.push_back(nullptr);
				continue;
			}
			CAENparameter* new_param = nullptr;
			try {
				new_param = new CAENparameter(phandle);
				new_param->SetWidget(mGroupWidgetsList.at(p));
			}
			catch (CAENFELibException& exc) {
				mGroupParams.push_back(new_param);
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
			mGroupParams.push_back(new_param);
			if (new_param == nullptr)
				mWaveDump->addLog("GET" + mGroupParamsList.at(p), 0);
		}
	}
	CreateGroupParamsMap();

	mDevParamsList.clear();//no more needed when the map is filled
	mDevWidgetsList.clear();
}

void Dx740V2ConfigPanel::defineScaleParams() {
	uint64_t phandle;
	if (mCHGainVisible) {
		mChGUIMap.value(ui->dBox_thr)->addRelatedParam(mChGUIMap.value(ui->dBox_gain));
	}
	if (!mDecimationVisible)
		return;
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle, "/par/DecimationFactor", &phandle);
	if (err != CAEN_FELib_Success) {
		mWaveDump->addLog(QString("Get Handle of /par/DecimationFactor error %1").arg(err), false);
	}
	else {
		mDevGUIMap.value(ui->dBox_reclen)->setScaleFactorParam(phandle);
		mDevGUIMap.value(ui->dBox_pretrg)->setScaleFactorParam(phandle);
		mDevGUIMap.value(ui->cBox_decim)->addRelatedParam(mDevGUIMap.value(ui->dBox_reclen));
		mDevGUIMap.value(ui->cBox_decim)->addRelatedParam(mDevGUIMap.value(ui->dBox_pretrg));
	}

}


void Dx740V2ConfigPanel::UpdateDevGUI() {
	this->ConfigurationPanel::updateDevGUI();
}

void Dx740V2ConfigPanel::UpdateChGUI() {
	this->ConfigurationPanel::updateChGUI();
}

void Dx740V2ConfigPanel::UpdateGroupGUI() {
	this->ConfigurationPanel::updateGroupGUI();
}

void Dx740V2ConfigPanel::Update(){
	for (int p = 0; p < mChParams.size(); p++) {
		if(mChParams.at(p) != nullptr)
			mChParams.at(p)->getValue();
	}
	for (int p = 0; p < mGroupParams.size(); p++) {
		if (mGroupParams.at(p) != nullptr)
			mGroupParams.at(p)->getValue();
	}
	UpdateMainSettings();
	this->ConfigurationPanel::Update();
}

void Dx740V2ConfigPanel::CreateChParamsMap() {
	mChGUIMap.clear();
	for (int p = 0; p < mNumberOfChParams; p++)
		mChGUIMap.insert(mChWidgetsList.at(p), mChParams.at((mNumberOfChParams * mFirstSelectedCh) + p));
	if (mUoMY == UOM_PHYS_UNIT) {
		double f = (mWaveDump->getDevChSampleToV(mDevName, mFirstSelectedCh) * 1e3); 
		mChGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM(f);
	}
}

void Dx740V2ConfigPanel::CreateGroupParamsMap() {
	mGroupGUIMap.clear();
	for (int p = 0; p < mNumberOfGroupParams; p++)
		mGroupGUIMap.insert(mGroupWidgetsList.at(p), mGroupParams.at((mNumberOfGroupParams * mFirstSelectedGroup.at(p)) + p));//qui mGroupParams ha size 0
	//if (mUoMY == UOM_PHYS_UNIT)
	//	mGroupGUIMap.value(xxxxxxx)->setUoMToGUIUoM(xxxxx);////to be used if a group param has a UoM
}

//multi sel. active --> check if the selected channels have the same param values
void Dx740V2ConfigPanel::CheckChParamValues() {
	for (int p = 0; p < mChParamsList.size(); p++) {
		if (mChGUIMap.value(mChWidgetsList.at(p)) == nullptr)
			continue;
		QString first_sel_val = mChGUIMap.value(mChWidgetsList.at(p))->getCachedValue();
		bool same_value = true;
		for (int i = 0; i < mNumberOfChannels; i++) {
			if (!mPbCh[i]->isChecked() || (i == mFirstSelectedCh) || mChParams.at((mNumberOfChParams * i) + p) == nullptr)
				continue;

			if (mChParams.at((mNumberOfChParams * i) + p)->getCachedValue() != first_sel_val) {
				same_value = false;
				break;
			}
		}

		if (!same_value) {
			WarningWidget(mChWidgetsList.at(p), false);
		}
	}
}
//multi sel. active --> check if the selected groups have the same param values
void Dx740V2ConfigPanel::CheckGroupParamValues() {
	for (int p = 0; p < mNumberOfGroupParams; p++) {
		if (mGroupGUIMap.value(mGroupWidgetsList.at(p)) == nullptr)
			continue;
		QString first_sel_val = mGroupGUIMap.value(mGroupWidgetsList.at(p))->getCachedValue();
		bool same_value = true;
		for (int i = 0; i < (mNumberOfChannels / mGroupsize.at(p)); i++) {
			if (!mPbCh[i + mGroupsize.at(p)]->isChecked() || (i == mFirstSelectedGroup.at(p)) || mGroupParams.at((mNumberOfGroupParams * i) + p) == nullptr)
				continue;

			if (mGroupParams.at((mNumberOfGroupParams * i) + p)->getCachedValue() != first_sel_val) {
				same_value = false;
				break;
			}
		}

		if (!same_value) {
			WarningWidget(mGroupWidgetsList.at(p), false);
		}
	}
}

void Dx740V2ConfigPanel::channelClicked(CAENPushButton *pb) {
	int ch = pb->getIndex();

	//single channel selected
	if (!mMulti) {
		for (int i = 0; i < mNumberOfChannels; i++)
			if (i != ch) mPbCh[i]->setChecked(false);
		if (ch != mFirstSelectedCh) {
			mFirstSelectedCh = ch;
			for (int p = 0; p < mNumberOfGroupParams; p++)
				mFirstSelectedGroup.replace(p, (ch / mGroupsize.at(p)));
			CreateChParamsMap();
			UpdateChGUI();
			CreateGroupParamsMap();
			UpdateGroupGUI();
		}
		else
			mPbCh[ch]->setChecked(true);
	}
	else {//multi sel. active
		if (ch == mFirstSelectedCh) {
			for (int i = 0; i < mNumberOfChannels; i++) {
				if (mPbCh[i]->isChecked()) {
					mFirstSelectedCh = i;
					for (int p = 0; p < mNumberOfGroupParams; p++)
						mFirstSelectedGroup.replace(p, (i / mGroupsize.at(p)));
					CreateChParamsMap();
					UpdateChGUI();
					CreateGroupParamsMap();
					UpdateGroupGUI();
					break;
				}
			}
			mPbCh[mFirstSelectedCh]->setChecked(true);
		}

		CheckChParamValues();
		CheckGroupParamValues();
	}
}
void Dx740V2ConfigPanel::selectAll() {
	for (int i = 0; i < mNumberOfChannels; i++)
		mPbCh[i]->setChecked(true);
	ui->pbMulti->setChecked(true);
	CheckChParamValues();
	CheckGroupParamValues();
}

void  Dx740V2ConfigPanel::MultiClicked() {
	mMulti = ui->pbMulti->isChecked();
	if (!mMulti) {
		QHash<QWidget *, CAENparameter *>::iterator item;
		for (item = mChGUIMap.begin(); item != mChGUIMap.end(); ++item) {
			NormalWidget(item.key(), false);
		}
		int memCh = mFirstSelectedCh;
		mFirstSelectedCh = -1;
		emit channelClicked(mPbCh[memCh]);
	}
}

void Dx740V2ConfigPanel::WarningWidget(QWidget *widget, bool bold) {
	if (!widget->styleSheet().contains("color: black"))
		return;
	if(bold)
		widget->setStyleSheet("font-weight: bold; color: red");
	else
		widget->setStyleSheet("color: red");
	mIgnoreEvent = true;
	if (QComboBox *cbox = qobject_cast<QComboBox *>(widget)) {
		cbox->setCurrentIndex(-1);
		mIgnoreEvent = false;
		return;
	}
	if(QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(widget)){
		doubleSpinBox->setSpecialValueText("?");
		doubleSpinBox->setMinimum(-1);
		doubleSpinBox->setValue(-1);
		mIgnoreEvent = false;
		return;
	}
	if(QSpinBox *SpinBox = qobject_cast<QSpinBox *>(widget)){
		SpinBox->setSpecialValueText("?");
		SpinBox->setMinimum(SpinBox->minimum() - 1);
		SpinBox->setValue(SpinBox->minimum());
		mIgnoreEvent = false;
		return;
	}
	mIgnoreEvent = false;
}

void Dx740V2ConfigPanel::NormalWidget(QWidget *widget, bool bold) {
	if (!widget->styleSheet().contains("color: red"))
		return;
	if(bold)
		widget->setStyleSheet("font-weight: bold; color: black");
	else
		widget->setStyleSheet("color: black");
	mIgnoreEvent = true;
	QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(widget);
	if (doubleSpinBox != nullptr) {
		doubleSpinBox->setSpecialValueText("");
		doubleSpinBox->setMinimum(doubleSpinBox->minimum() + 1);
		mIgnoreEvent = false;
		return;
	}
	QSpinBox *SpinBox = qobject_cast<QSpinBox *>(widget);
	if (SpinBox != nullptr) {
		SpinBox->setSpecialValueText("");
		SpinBox->setMinimum(SpinBox->minimum() + 1);
		mIgnoreEvent = false;
		return;
	}
	mIgnoreEvent = false;
	
}

void Dx740V2ConfigPanel::ApplyNewValue(QWidget *widget) {
	try {
		if (mDevGUIMap.contains(widget) && mDevGUIMap.value(widget) != nullptr) {
			mDevGUIMap.value(widget)->SetValueFromWidget();
			QDoubleSpinBox* doubleSpinBox = qobject_cast<QDoubleSpinBox*>(widget);
			if (doubleSpinBox != nullptr) {
				if (doubleSpinBox == ui->dBox_reclen)
					mWaveDump->add2LogAndConf("SET", mDevName, mDevGUIMap.value(widget)->getQry(), QString("%1").arg(ui->dBox_reclen->value()), QString("%1").arg(ui->dBox_reclen->value()), CAEN_FELib_Success);
				else
					mWaveDump->add2LogAndConf("SET", mDevName, mDevGUIMap.value(widget)->getQry(), mDevGUIMap.value(widget)->getCachedValue(), mDevGUIMap.value(widget)->getCachedValue(), CAEN_FELib_Success);
			}
			else
				mWaveDump->add2LogAndConf("SET", mDevName, mDevGUIMap.value(widget)->getQry(), mDevGUIMap.value(widget)->getCachedValue(), mDevGUIMap.value(widget)->getCachedValue(), CAEN_FELib_Success);
			mIgnoreEvent = true;
			mDevGUIMap.value(widget)->UpdateWidget();
			mIgnoreEvent = false;
		}
		else if (mChGUIMap.contains(widget)) {
			if (!ui->pbMulti->isChecked() && mChGUIMap.value(widget) != nullptr) {//single channel selected
				mChGUIMap.value(widget)->SetValueFromWidget();
				mWaveDump->add2LogAndConf("SET", mDevName, mChGUIMap.value(widget)->getQry(), mChGUIMap.value(widget)->getCachedValue(), mChGUIMap.value(widget)->getCachedValue(), CAEN_FELib_Success);
				mIgnoreEvent = true;
				mChGUIMap.value(widget)->UpdateWidget();
				mIgnoreEvent = false;
			}
			else {//multi channel select
				int index = mChWidgetsList.indexOf(widget);
				for (int ch = 0; ch < mNumberOfChannels; ch++) {
					if (!mPbCh[ch]->isChecked())
						continue;
					if (mChParams.at((mNumberOfChParams * ch) + index) == nullptr)
						continue;
					mChParams.at((mNumberOfChParams * ch) + index)->SetValueFromWidget();
					mWaveDump->add2LogAndConf("SET", mDevName, mChParams.at((mNumberOfChParams * ch) + index)->getQry(), mChParams.at((mNumberOfChParams * ch) + index)->getCachedValue(), mChParams.at((mNumberOfChParams * ch) + index)->getCachedValue(), CAEN_FELib_Success);
				}
				mIgnoreEvent = true;
				mChParams.at((mNumberOfChParams * mFirstSelectedCh) + index)->UpdateWidget();
				mIgnoreEvent = false;
				NormalWidget(widget, false);
			}
		}
		else if (mGroupGUIMap.contains(widget)) {
			if (!ui->pbMulti->isChecked() && mGroupGUIMap.value(widget) != nullptr) {//single channel selected
				mGroupGUIMap.value(widget)->SetValueFromWidget();
				mWaveDump->add2LogAndConf("SET", mDevName, mGroupGUIMap.value(widget)->getQry(), mGroupGUIMap.value(widget)->getCachedValue(), mGroupGUIMap.value(widget)->getCachedValue(), CAEN_FELib_Success);
				mIgnoreEvent = true;
				mGroupGUIMap.value(widget)->UpdateWidget();
				mIgnoreEvent = false;
			}
			else {//multi channel select
				int index = mGroupWidgetsList.indexOf(widget);
				for (int gs = 0; gs < mGroupsize.size(); gs++) {
					if (!mPbCh[mNumberOfChannels / mGroupsize.at(gs)]->isChecked())
							continue;
					if (mGroupParams.at((mNumberOfGroupParams * gs) + index) == nullptr)
							continue;
					mGroupParams.at((mNumberOfGroupParams * gs) + index)->SetValueFromWidget();
					mWaveDump->add2LogAndConf("SET", mDevName, mGroupParams.at((mNumberOfGroupParams * gs) + index)->getQry(), mGroupParams.at((mNumberOfGroupParams * gs) + index)->getCachedValue(), mGroupParams.at((mNumberOfGroupParams * gs) + index)->getCachedValue(), CAEN_FELib_Success);
					
				}
				mIgnoreEvent = true;
				mGroupParams.at((mNumberOfGroupParams * mFirstSelectedGroup.at(index)) + index)->UpdateWidget();
				mIgnoreEvent = false;
				NormalWidget(widget, false);
			}
		}
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}
}

void Dx740V2ConfigPanel::UpdateRelatedParams(QWidget *widget) {
	mIgnoreEvent = true;
	if (widget == ui->dBox_gain) {
		if (mUoMY == UOM_PHYS_UNIT) {
			mChGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM(mWaveDump->getDevChSampleToV(mDevName, mFirstSelectedCh) * 1e3);
			mChGUIMap.value(ui->dBox_thr)->UpdateWidget();
		}
	}
	else {
		if (mDevGUIMap.contains(widget)) {
			for (int p = 0; p < mDevGUIMap.value(widget)->RelatedParams.size(); p++)
				mDevGUIMap.value(widget)->RelatedParams.at(p)->UpdateWidget();
		}
		else if (mChGUIMap.contains(widget)) {
			for (int p = 0; p < mChGUIMap.value(widget)->RelatedParams.size(); p++)
				mChGUIMap.value(widget)->RelatedParams.at(p)->UpdateWidget();
		}
		else if (mGroupGUIMap.contains(widget)) {
			for (int p = 0; p < mGroupGUIMap.value(widget)->RelatedParams.size(); p++)
				mGroupGUIMap.value(widget)->RelatedParams.at(p)->UpdateWidget();
		}
	}
	mIgnoreEvent = false;
}

void Dx740V2ConfigPanel::UpdateSamplFreq() {
	char val[32];
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mDevHandle, "/par/ADC_SamplRate", val);
	if (err != CAEN_FELib_Success) {
		char error[1024];
		CAEN_FELib_GetLastError(error);
		mWaveDump->addLog("Error reading ADC sampling rate: " + QString(error), false);
		ui->label_freq_decimation->setText("[Frequency = unknown]");
	}
	else {
		double freq = QString(val).toDouble();
		freq /= ui->cBox_decim->currentText().toInt();
		ui->label_freq_decimation->setText(QString("[Frequency = %1 MHz]").arg(freq));
	}
}

void Dx740V2ConfigPanel::SetPermanentClockDelay() {
	QString value = QString("%1").arg(ui->sBox_v_clk_delay->value());
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/PermanentClockOutDelay", value.toStdString().c_str());
	if (err != CAEN_FELib_Success) {
		mWaveDump->addLog(QString("Set /par/PermanentClockOutDelay error %1").arg(err), false);
		return;
	}
	else
		mWaveDump->add2LogAndConf("SET", mDevName, "/par/PermanentClockOutDelay", value, value, CAEN_FELib_Success);
}

void Dx740V2ConfigPanel::intChanged(int val) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
	UpdateRelatedParams(widget);

	if (widget == ui->cBox_decim)
		UpdateSamplFreq();
}

void Dx740V2ConfigPanel::boolChanged(bool val) {
	if (mIgnoreEvent)
		return;
	QWidget* widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
	UpdateRelatedParams(widget);
}

void Dx740V2ConfigPanel::doubleChanged(double val) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	
	QDoubleSpinBox* doubleSpinBox = qobject_cast<QDoubleSpinBox*>(widget);
	if (doubleSpinBox != nullptr) {
		if (doubleSpinBox == ui->dBox_reclen) {
			if (mUoMX == UOM_SAMPLE && val > MAX_SW_RECLEN_SAMPLES) {
				doubleSpinBox->setValue(MAX_SW_RECLEN_SAMPLES);
				return;
			}
			else if (mUoMX == UOM_PHYS_UNIT && val > (MAX_SW_RECLEN_SAMPLES * mWaveDump->getDevSampleToS(mDev) * 1e6)) {
				doubleSpinBox->setValue(MAX_SW_RECLEN_SAMPLES * mWaveDump->getDevSampleToS(mDev) * 1e6);
				return;
			}
	}
	}
	ApplyNewValue(widget);
	UpdateRelatedParams(widget);
}

void Dx740V2ConfigPanel::itemCheckedChanged(QListWidgetItem *item) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
	QListWidget* lwidget = qobject_cast<QListWidget*>(widget);
	if (lwidget != nullptr) {
		if (lwidget == ui->lWidget_StartSrc) {
			CAEN_FELib_ErrorCode err;
			QString value;
			if (mDev->getStartSource().contains("SINlevel", Qt::CaseInsensitive))
				value = "False";
			else
				value = "True";
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/EnAutoDisarmAcq", value.toStdString().c_str());
			if (err != CAEN_FELib_Success) {
				mWaveDump->addLog(QString("Set /par/EnAutoDisarmAcq error %1").arg(err), false);
			}
			else
				mWaveDump->add2LogAndConf("SET", mDevName, "/par/EnAutoDisarmAcq", value, value, CAEN_FELib_Success);
		}
	}
}

void Dx740V2ConfigPanel::SetMask() {
	QString current_mask = mWaveDump->getSelfTrgMask();
	uint64_t val = current_mask.toULongLong();
	chmaskDialog *dlg = new chmaskDialog(this, mWaveDump->getNumChannels(), val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask = QString("%1").arg(dlg->getMask(),0,16, QLatin1Char('0'));
		ui->lineEdit_trig_mask->setText(new_mask);
	}
	delete dlg;
}

void Dx740V2ConfigPanel::SetTrgoutMask() {
	QString current_mask = mWaveDump->getTrgoutMask();
	uint64_t val = current_mask.toULongLong();
	chmaskDialog* dlg = new chmaskDialog(this, mWaveDump->getNumChannels(), val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask = QString("%1").arg(dlg->getMask(), 0, 16, QLatin1Char('0'));
		ui->lineEdit_trgout_mask->setText(new_mask);
	}
	delete dlg;
}


void Dx740V2ConfigPanel::TextChanged(QString text) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void Dx740V2ConfigPanel::TrgoutChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	CAEN_FELib_ErrorCode err;
	ui->pB_trgout_mask->setEnabled(false);
	if (mode == "DISABLED") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/TRGOUTMode", "Disabled");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "Disabled", "Disabled", CAEN_FELib_Success);
	}
	else if (mode == "TRGIN") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/TRGOUTMode", "TrgIn");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "TrgIn", "TrgIn", CAEN_FELib_Success);
	}
	else if (mode == "SIN") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/TRGOUTMode", "SIN");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "SIN", "SIN", CAEN_FELib_Success);
	}
	else if (mode == "SELF TRIGGER") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/TRGOUTMode", "ITLB");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
		}
		else {
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "ITLB", "ITLB", CAEN_FELib_Success);
			ui->pB_trgout_mask->setEnabled(true);
		}
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/ITLBMainLogic", "OR");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/ITLBMainLogic error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLBMainLogic", "OR", "OR", CAEN_FELib_Success);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/ITLBPairLogic", "None");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/ITLBPairLogic error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLBPairLogic", "None", "None", CAEN_FELib_Success);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/ITLBGateWidth", "16");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/ITLBGateWidth error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLBGateWidth", "16", "16", CAEN_FELib_Success);

	}
	else if (mode == "GPIO") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/TRGOUTMode", "GPIO");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "GPIO", "GPIO", CAEN_FELib_Success);
	}
	else if (mode == "RUN") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/TRGOUTMode", "Run");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "Run", "Run", CAEN_FELib_Success);
	}

	if (err == CAEN_FELib_Success)
		NormalWidget(ui->cBox_trgout1, true);

}

void Dx740V2ConfigPanel::HFRejChanged(int enable) {
	CAEN_FELib_ErrorCode err;
	int index = mChWidgetsList.indexOf(ui->cBox_hfreq);
	QString nsamples = "1";
	if (enable)
		nsamples = "20";
	
	if(ui->pbMulti->isChecked()){
		for (int ch = 0; ch < mNumberOfChannels; ch++) {
			if (!mPbCh[ch]->isChecked())
				continue;
			QString parname = QString("/ch/%1/par/SamplesOverThreshold").arg(ch);
			mChParams.at((mNumberOfChParams * ch) + index)->setValue(nsamples.toStdString().c_str());
			mWaveDump->add2LogAndConf("SET", mDevName, QString("/ch/%1/par/SamplesOverThreshold").arg(ch), nsamples.toStdString().c_str(), nsamples.toStdString().c_str(), CAEN_FELib_Success);
		}
	}	
	else {
		QString parname = QString("/ch/%1/par/SamplesOverThreshold").arg(mFirstSelectedCh);
		mChGUIMap.value(ui->cBox_hfreq)->setValue(nsamples.toStdString().c_str());
		mWaveDump->add2LogAndConf("SET", mDevName, QString("/ch/%1/par/SamplesOverThreshold").arg(mFirstSelectedCh), nsamples.toStdString().c_str(), nsamples.toStdString().c_str(), CAEN_FELib_Success);

	}
	NormalWidget(ui->cBox_hfreq, false);
}

void Dx740V2ConfigPanel::TriggerModeChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	CAEN_FELib_ErrorCode err;
	ui->pB_trg_mask->setEnabled(false);
	if (mode == "INDEPENDENT_SELF_TRIGGERS" || mode == "SW_TRIGGER_ONLY") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/AcqTriggerSource", "SWTrg");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName,"/par/AcqTriggerSource" , "SWTrg", "SWTrg", CAEN_FELib_Success);
	}
	else if (mode == "SELF_TRIGGER") {
		ui->pB_trg_mask->setEnabled(true);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/AcqTriggerSource", "SWTrg|ITLA");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/AcqTriggerSource", "SWTrg|ITLA", "SWTrg|ITLA", CAEN_FELib_Success);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/ITLAMainLogic", "OR");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLAMainLogic", "OR", "OR", CAEN_FELib_Success);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/ITLAPairLogic", "None");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLAPairLogic", "None", "None", CAEN_FELib_Success);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/ITLAGateWidth", "16");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLAGateWidth", "16", "16", CAEN_FELib_Success);
	}
	else if (mode == "EXTERNAL_TRGIN") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/AcqTriggerSource", "SWTrg|TrgIn");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/AcqTriggerSource", "SWTrg|TrgIn", "SWTrg|TrgIn", CAEN_FELib_Success);
	}
	else if (mode == "EXTERNAL_GPIO") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/AcqTriggerSource", "SWTrg|GPIO");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/AcqTriggerSource", "SWTrg|GPIO", "SWTrg|GPIO", CAEN_FELib_Success);
	}
	else if (mode == "INTERNAL_PULSER") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/AcqTriggerSource", "SWTrg|TestPulse");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/AcqTriggerSource", "SWTrg|TestPulse", "SWTrg|TestPulse", CAEN_FELib_Success);
	}

	if (err == CAEN_FELib_Success)
		NormalWidget(ui->cBox_trg_mode, true);
}

void Dx740V2ConfigPanel::VetoChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	CAEN_FELib_ErrorCode err;
	if (mode == "DISABLED") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/VETOSource", "Disabled");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/VETOSource error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/VETOSource", "Disabled", "Disabled", CAEN_FELib_Success);
	}
	else if (mode == "SIN") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/VETOSource", "SIN");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/VETOSource error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/VETOSource", "SIN", "SIN", CAEN_FELib_Success);
	}
	if (mode == "GPIO") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/VETOSource", "GPIO");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/VETOSource error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/VETOSource", "GPIO", "GPIO", CAEN_FELib_Success);
	}
	if (mode == "LVDS") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/VETOSource", "LVDS");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/VETOSource error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/VETOSource", "LVDS", "LVDS", CAEN_FELib_Success);
	}

	if (err == CAEN_FELib_Success)
		NormalWidget(ui->cBox_veto1, true);
}

void Dx740V2ConfigPanel::GpioChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	CAEN_FELib_ErrorCode err;
	if (mode == "DISABLED") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/GPIOMode", "Disabled");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "Disabled", "Disabled", CAEN_FELib_Success);
	}
	else if (mode == "SIN") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/GPIOMode", "SIN");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "SIN", "SIN", CAEN_FELib_Success);
	}
	else if (mode == "LVDS") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/GPIOMode", "LVDS");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "LVDS", "LVDS", CAEN_FELib_Success);
	}
	else if (mode == "TRGIN") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/GPIOMode", "TrgIn");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "TrgIn", "TrgIn", CAEN_FELib_Success);
	}
	else if (mode == "SELF TRIGGER") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/GPIOMode", "ITLA");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "ITLA", "ITLA", CAEN_FELib_Success);
	}
	else if (mode == "RUN") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/GPIOMode", "Run");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "Run", "Run", CAEN_FELib_Success);
	}
	else if (mode == "BUSY") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle, "/par/GPIOMode", "BUSY");
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "BUSY", "BUSY", CAEN_FELib_Success);
	}

	if (err == CAEN_FELib_Success)
		NormalWidget(ui->cBox_gpio1, true);
}

void Dx740V2ConfigPanel::UpdateTriggerMode() {
	char value[200];
	mIgnoreEvent = true;
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mDevHandle, "/par/AcqTriggerSource", value);
	if (err != CAEN_FELib_Success) {
		mWaveDump->addLog(QString("Get /par/AcqTriggerSource error %1").arg(err), false);
		mIgnoreEvent = false;
		return;
	}

	QString TrgMode = QString(value);
	if (TrgMode.contains("ITLA", Qt::CaseInsensitive)) {
		ui->cBox_trg_mode->setCurrentText("SELF_TRIGGER");
		ui->pB_trg_mask->setEnabled(true);
	}
	else if (TrgMode.contains("TrgIn", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("EXTERNAL_TRGIN");
	else if (TrgMode.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("EXTERNAL_GPIO");
	else if (TrgMode.contains("TestPulse", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("INTERNAL_PULSER");
	else if (TrgMode.contains("SwTrg", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("SW_TRIGGER_ONLY");
	else
		WarningWidget(ui->cBox_trg_mode, true);

	mIgnoreEvent = false;	
}

void Dx740V2ConfigPanel::UpdateTrgoutMode() {
	char value[200];
	mIgnoreEvent = true;
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mDevHandle, "/par/TRGOUTMode", value);
	if (err != CAEN_FELib_Success) {
		mWaveDump->addLog(QString("Get /par/TRGOUTMode error %1").arg(err), false);
		mIgnoreEvent = false;
		return;
	}
	QString TrgoutMode = QString(value);
	if (TrgoutMode.contains("ITLB", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("SELF TRIGGER");
		ui->pB_trgout_mask->setEnabled(true);
	}
	else if (TrgoutMode.contains("Disabled", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("DISABLED");
	else if (TrgoutMode.contains("TrgIn", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("TRGIN");
	else if (TrgoutMode.contains("SIN", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("SIN");
	else if (TrgoutMode.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("GPIO");
	else if (TrgoutMode.contains("Run", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("RUN");
	else if (TrgoutMode.contains("RefClk", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("REFCLK");
	else
		WarningWidget(ui->cBox_trgout1, true);
	mIgnoreEvent = false;
}

void Dx740V2ConfigPanel::UpdateVetoSrc() {
	char value[200];
	mIgnoreEvent = true;
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mDevHandle, "/par/VETOSource", value);
	if (err != CAEN_FELib_Success) {
		mWaveDump->addLog(QString("Get /par/VETOSource error %1").arg(err), false);
		mIgnoreEvent = false;
		return;
	}
	QString vetosrc = QString(value);
	if (vetosrc.contains("Disabled", Qt::CaseInsensitive))
		ui->cBox_veto1->setCurrentText("DISABLED");
	else if (vetosrc.contains("SIN", Qt::CaseInsensitive))
		ui->cBox_veto1->setCurrentText("SIN");
	else if (vetosrc.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_veto1->setCurrentText("GPIO");
	else if (vetosrc.contains("LVDS", Qt::CaseInsensitive))
		ui->cBox_veto1->setCurrentText("LVDS");
	else
		WarningWidget(ui->cBox_veto1, true);

	mIgnoreEvent = false;
}

void Dx740V2ConfigPanel::UpdateGpioMode() {
	char value[200];
	mIgnoreEvent = true;
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mDevHandle, "/par/GPIOMode", value);
	if (err != CAEN_FELib_Success) {
		mWaveDump->addLog(QString("Get /par/GPIOMode error %1").arg(err), false);
		mIgnoreEvent = false;
		return;
	}
	QString mode = QString(value);
	if (mode.contains("Disabled", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("DISABLED");
	else if (mode.contains("SIN", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("SIN");
	else if (mode.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("GPIO");
	else if (mode.contains("ITLA", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("SELF TRIGGER");
	else if (mode.contains("TrgIn", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("TRGIN");
	else if (mode.contains("Run", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("RUN");
	else if (mode.contains("Busy", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("BUSY");
	else
		WarningWidget(ui->cBox_gpio1, true);

	mIgnoreEvent = false;
}

void Dx740V2ConfigPanel::currentTabChanged(int tab){
	if (mIgnoreEvent)
		return;
	QWidget *currWidget = ui->tabWidget->currentWidget();
	if (currWidget == ui->tab_main) {
		UpdateMainSettings();
	}
	else if (currWidget == ui->tab_advanced) {
		mDevGUIMap.value(ui->lWidget_StartSrc)->getValue();
		mDevGUIMap.value(ui->lWidget_StartSrc)->UpdateWidget();
		mDevGUIMap.value(ui->lWidget_Trg_Src)->getValue();
		mDevGUIMap.value(ui->lWidget_Trg_Src)->UpdateWidget();
		mDevGUIMap.value(ui->lWidget_TReset_Src)->getValue();
		mDevGUIMap.value(ui->lWidget_TReset_Src)->UpdateWidget();
		mDevGUIMap.value(ui->lWidget_VETOSrc)->getValue();
		mDevGUIMap.value(ui->lWidget_VETOSrc)->UpdateWidget();
		mDevGUIMap.value(ui->cBox_TRGOUT)->getValue();
		mDevGUIMap.value(ui->cBox_TRGOUT)->UpdateWidget();
		mDevGUIMap.value(ui->cBox_GPIO)->getValue();
		mDevGUIMap.value(ui->cBox_GPIO)->UpdateWidget();
	}
}

void Dx740V2ConfigPanel::ConfigureDACOUT() {

	DACOUTDialog* dlg = new DACOUTDialog(this, mDev->getHandle());
	int ok = dlg->exec();
	delete dlg;
}


Dx740V2ConfigPanel::~Dx740V2ConfigPanel()
{
	delete ui;
}
