/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		Dx740V1ConfigPanel.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "Dx740V1ConfigPanel.h"
#include "WaveDump2.h"
#include "CAENSwitch.h"
#include "CAENFELibException.h"
#include "chmaskDialog.h"
#include <QDebug>


Dx740V1ConfigPanel::Dx740V1ConfigPanel(WaveDump2 *parent, CAENDig1Device* dev, QString dev_name, bool show_wait) :
	QDialog(parent),
	ui(new Ui::Dx740V1ConfigPanel) {
	ui->setupUi(this);

	setWindowIcon(QIcon(":WAVEDump2.png"));
	QMessageBox msgBox(this);
	if (show_wait) {		
		msgBox.setText("<html><B>Reading device settings <br>           ... please wait ...</html>");
		msgBox.setStandardButtons({});
		msgBox.show();
		QApplication::processEvents();
	}

	mWaveDump = parent;
	mDevHandle = dev->getHandle();
	mDev = dev;
	mDevName = dev_name;
	mUoMX = mWaveDump->getUoMX();
	mUoMY = mWaveDump->getUoMY();	

	CAENSwitch pippo("on","off",1,this);
	ui->lay_0->addWidget(&pippo);
	
	mNumberOfChannels = dev->getNumCh();

	for (int i = 0; i < mNumberOfChannels; i++) {
		mPbCh[i] = new CAENPushButton(this, i);
		mPbCh[i]->setMaximumWidth(60);
		mPbCh[i]->setText("CH " + QString::number(i));
		mPbCh[i]->setCheckable(true);
		connect(mPbCh[i], &CAENPushButton::clicked, [=] {channelClicked(mPbCh[i]); });
	}

	int nraws = (mNumberOfChannels < 16) ? 1 : int(mNumberOfChannels / 16);
	int size = (mNumberOfChannels < 16) ? mNumberOfChannels : 16;
	for (int i = 0; i < nraws; i++) {
		for (int j = 0; j < size; j++) {
			switch (i) {
				case 0:
					ui->lay_0->addWidget((mPbCh[(i * size) + j]));
					break;
				case 1:
					ui->lay_1->addWidget(mPbCh[(i * size) + j]);
					break;
				case 2:
					ui->lay_2->addWidget(mPbCh[(i * size) + j]);
					break;
				case 3:
					ui->lay_3->addWidget(mPbCh[(i * size) + j]);
					break;
			}
		}
	}

	
	QString Modelname = dev->getFamilyCode();
	if (Modelname.contains("740")) {
		mGroupedChannels = true;
		mChGroupSize = mDev->getChGroupSize();
	}
	else if (Modelname.contains("730") || Modelname.contains("725") || Modelname.contains("724")) {
		ui->cBox_gain->setVisible(true);
		ui->label_17->setVisible(true);
		ui->label->setVisible(false);
		mCHGainVisible = true;
	}
	else if (Modelname.contains("751"))
		ui->label->setVisible(false);
	else {
		ui->cBox_gain->setVisible(false);
		ui->label_17->setVisible(false);
	}

	if (Modelname.contains("740") || Modelname.contains("724"))
		mDecimationVisible = true;

	//if ((Modelname.contains("725") || Modelname.contains("730") || Modelname.contains("751") || Modelname.contains("731") || Modelname.contains("761")) && !Modelname.contains("S"))
	//	ui->pB_CalibADC->setVisible(true);
	//else {
		ui->pB_CalibADC->setVisible(false);
		ui->pB_CalibADC->setEnabled(false);
	//}

	mPbCh[0]->setChecked(true);
	mFirstSelectedCh = 0;
	mFirstSelectedGroup.clear();
	mIgnoreEvent = false;
	mMulti = false;

	ui->pB_trg_mask->setEnabled(false);
	ui->pB_trgout_mask->setEnabled(false);
	ui->cBox_polarity->setEnabled(false);

	Init();

	UpdateParamsUoM();
	QApplication::processEvents();
	UpdateDevGUI();
	QApplication::processEvents();
	UpdateChGUI();
	QApplication::processEvents();
	UpdateGroupGUI();
	QApplication::processEvents();
	UpdateMainSettings();
	if (mDecimationVisible)
		UpdateSamplFreq();
	else
		ui->cBox_decim->hide();

	if (show_wait)
		msgBox.close();
}

void Dx740V1ConfigPanel::UpdateMainSettings() {
	UpdateTriggerMode();
	UpdateTrgoutMode();
	UpdatePreTrigger();
	UpdateClkSource();
}

void Dx740V1ConfigPanel::connectWidgets() {
	//connections
	connect(ui->pbAll, SIGNAL(clicked()), this, SLOT(selectAll()));
	connect(ui->pbMulti, SIGNAL(clicked()), this, SLOT(MultiClicked()));
	connect(ui->dBox_reclen, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dBox_pretrg, SIGNAL(valueChanged(double)), this, SLOT(PreTrgChanged(double)));
	connect(ui->cBox_decim, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_trg_mode, SIGNAL(currentTextChanged(QString)), this, SLOT(TriggerModeChanged(QString)));
	connect(ui->pB_trg_mask, SIGNAL(clicked()), this, SLOT(SetMask()));
	//connect(ui->lineEdit_trig_mask, SIGNAL(textChanged(QString)), this, SLOT(TextChanged(QString)));
	connect(ui->cBox_trgout1, SIGNAL(currentTextChanged(QString)), this, SLOT(TrgoutChanged(QString)));
	connect(ui->pB_trgout_mask, SIGNAL(clicked()), this, SLOT(SetTrgoutMask()));
	connect(ui->cBox_polarity, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_enable, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->dBox_thr, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dSBox_bsl_level, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->cBox_gain, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_StartMode, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->lWidget_Trg_Src, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(TrgSourceChanged(QListWidgetItem*)));
	connect(ui->cBox_clk_source, SIGNAL(currentIndexChanged(int)), this, SLOT(ClockSourceChanged(int)));
	connect(ui->cBox_trg_ext_trgout, SIGNAL(stateChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_trg_sw_trgout, SIGNAL(stateChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_TRGOUT, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	//connect(ui->cBox_GPIO, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	////////
	connect(ui->tabWidget, SIGNAL(currentChanged(int)), this, SLOT(currentTabChanged(int)));
	if(ui->pB_CalibADC->isEnabled())
		connect(ui->pB_CalibADC, SIGNAL(clicked()), this, SLOT(calibrateADC()));
}

void Dx740V1ConfigPanel::FillParamsAndWidgetsList() {
	//start creating the list of device params
	mDevParamsList.clear();
	mDevWidgetsList.clear();

	mDevParamsList << "/par/RecLen";
	mDevWidgetsList << ui->dBox_reclen;
	ui->dBox_reclen->setToolTip("/par/RecLen");

	mDevParamsList << QString("/par/Self_Trigger_Edge");
	mDevWidgetsList << ui->cBox_polarity;
	ui->cBox_polarity->setToolTip("/par/Self_Trigger_Edge");

	mDevParamsList << "/par/StartMode";
	mDevWidgetsList << ui->cBox_StartMode;
	ui->cBox_StartMode->setToolTip("/par/StartMode");

	mDevParamsList << "/par/Trg_Ext_Out_Propagate";
	mDevWidgetsList << ui->cBox_trg_ext_trgout;
	ui->cBox_trg_ext_trgout->setToolTip("/par/Trg_Ext_Out_Propagate");

	mDevParamsList << "/par/Trg_Sw_Out_Propagate";
	mDevWidgetsList << ui->cBox_trg_sw_trgout;
	ui->cBox_trg_sw_trgout->setToolTip("/par/Trg_Sw_Out_Propagate");

	mDevParamsList << "/par/Out_Selection";
	mDevWidgetsList << ui->cBox_TRGOUT;
	ui->cBox_TRGOUT->setToolTip("/par/Out_Selection");

	if (mDecimationVisible) {
		mDevParamsList << "/par/Decimation_Factor";
		mDevWidgetsList << ui->cBox_decim;
		ui->cBox_decim->setToolTip("/par/Decimation_Factor");
	}
	//add more device parameters if needed ...

	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mDevWidgetsList.size(); i++)
		mDevWidgetsList.at(i)->setStyleSheet("color: black");

	//create the list of channel params
	mChParamsList.clear();
	mChWidgetsList.clear();

	if (mChGroupSize == 1) {
		mChParamsList << QString("/ch/%1/par/Ch_Enabled");
		mChWidgetsList << ui->cBox_enable;
		ui->cBox_enable->setToolTip("/ch/N/par/Ch_Enabled");

		mChParamsList << QString("/ch/%1/par/Ch_Threshold");
		mChWidgetsList << ui->dBox_thr;
		ui->dBox_thr->setToolTip("/ch/N/par/Ch_Threshold");

		mChParamsList << QString("/ch/%1/par/Ch_DCOffset");
		mChWidgetsList << ui->dSBox_bsl_level;
		ui->dSBox_bsl_level->setToolTip("/ch/N/par/Ch_DCOffset");
	}

	//if (mCHGainVisible) {
		mChParamsList << QString("/ch/%1/par/Ch_InDyn");
		mChWidgetsList << ui->cBox_gain;
		ui->cBox_gain->setToolTip("/ch/N/par/Ch_InDyn");
	//}

	//add more channel parameters if needed
	mNumberOfChParams = mChParamsList.size();
	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mChWidgetsList.size(); i++)
		mChWidgetsList.at(i)->setStyleSheet("color: black");

	//create the list of group params
	mGroupParamsList.clear();
	mGroupWidgetsList.clear();
	mGroupsize.clear();

	if (mChGroupSize > 1) {
		mGroupParamsList << QString("/group/%1/par/Gr_Enabled");
		mGroupWidgetsList << ui->cBox_enable;
		ui->cBox_enable->setToolTip("/group/N/par/Gr_Enabled");
		mGroupsize << mChGroupSize;

		mGroupParamsList << QString("/group/%1/par/Gr_Threshold");
		mGroupWidgetsList << ui->dBox_thr;
		ui->dBox_thr->setToolTip("/group/N/par/Gr_Threshold");
		mGroupsize << mChGroupSize;

		mGroupParamsList << QString("/group/%1/par/Gr_DCOffset");
		mGroupWidgetsList << ui->dSBox_bsl_level;
		ui->dSBox_bsl_level->setToolTip("/group/N/par/Gr_DCOffset");
		mGroupsize << mChGroupSize;
	}

	//add more group parameters if needed
	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mGroupWidgetsList.size(); i++)
		mGroupWidgetsList.at(i)->setStyleSheet("color: black");
	mNumberOfGroupParams = mGroupParamsList.size();
	for (int g = 0; g < mNumberOfGroupParams; g++)
		mFirstSelectedGroup << 0; //for every group param depending on the group size there is a first selected group
}

void Dx740V1ConfigPanel::UpdateParamsUoM() {
	if (mUoMX == UOM_PHYS_UNIT) {
		mDevGUIMap.value(ui->dBox_reclen)->setUoMToGUIUoM(0.001);
		mDevGUIMap.value(ui->dBox_reclen)->setUoMToGUIUoM_incr(-mDev->RecLenJIncrS * 0.001);
		QString mu = QChar(0xbc, 0x03);
		ui->label_pretrig_unit->setText("[" + mu + "s]");
		ui->label_reclen_unit->setText("[" + mu + "s]");
		ui->dBox_reclen->setDecimals(3);
	}
	else {
		mDevGUIMap.value(ui->dBox_reclen)->setUoMToGUIUoM(1 / (mDev->getSample_to_S() * 1e9));
		mDevGUIMap.value(ui->dBox_reclen)->setUoMToGUIUoM_incr(-mDev->RecLenJIncrS);
		ui->label_pretrig_unit->setText("[samples]");
		ui->label_reclen_unit->setText("[samples]");
		ui->dBox_reclen->setDecimals(0);
	}

	if (mUoMY == UOM_PHYS_UNIT) {
		int index = mChWidgetsList.indexOf(ui->dBox_thr);
		if (index == -1) {
			index = mGroupWidgetsList.indexOf(ui->dBox_thr);
			for (int g = 0; g< (mNumberOfChannels/mGroupsize.at(index)); g++) 
				mGroupParams.at((index * (mNumberOfChannels / mGroupsize.at(index))) + g)->setUoMToGUIUoM(mWaveDump->getDevChSampleToV(mDevName, g*mChGroupSize) * 1e3);
		}
		else {
			for (int ch = 0; ch < mNumberOfChannels; ch++)
				mChParams.at((mNumberOfChParams * ch) + index)->setUoMToGUIUoM(mWaveDump->getDevChSampleToV(mDevName, ch) * 1e3);
		}
		ui->label_thr_unit->setText("[mV]");
	}
	else {
		ui->label_thr_unit->setText("[samples]");
	}

}

void Dx740V1ConfigPanel::createMaps() {
	uint64_t phandle=0;
	CAENparameter *new_param=nullptr;

	for (int p = 0; p < mDevParamsList.size(); p++) {
		phandle = 0;
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle, qPrintable(mDevParamsList.at(p)), &phandle);
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Get Handle of " + mDevParamsList.at(p) + " error %1").arg(err), false);
			mDevWidgetsList.at(p)->setEnabled(false);
			mDevGUIMap.insert(mDevWidgetsList.at(p), nullptr);
			continue;
		}

		try {
			new_param = new CAENparameter(phandle);
			new_param->SetWidget(mDevWidgetsList.at(p)); //assign the widget
		}
		catch (CAENFELibException& exc) {
			mDevGUIMap.insert(mDevWidgetsList.at(p), nullptr);
			mWaveDump->addLog(exc.ErrorMessage(), false);
		}
		mDevGUIMap.insert(mDevWidgetsList.at(p), new_param); //fill the map
		if(new_param == nullptr)
			mWaveDump->addLog("GET " + mDevParamsList.at(p), 0);
	}

	for (int ch = 0; ch < mNumberOfChannels; ch++) {
		for (int p = 0; p < mNumberOfChParams; p++) {
			phandle = 0;
			CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle, qPrintable(mChParamsList.at(p).arg(ch)), &phandle);
			if (err != CAEN_FELib_Success) {
				mWaveDump->addLog(QString("Get Handle of " + mChParamsList.at(p).arg(ch) + " error %1").arg(err), false);
				mChWidgetsList.at(p)->setEnabled(false);
				mChParams.push_back(nullptr);
				continue;
			}
			CAENparameter *new_param = nullptr;
			try {
				new_param = new CAENparameter(phandle);
				new_param->SetWidget(mChWidgetsList.at(p));
			}
			catch (CAENFELibException& exc) {
				mChParams.push_back(new_param);
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
			mChParams.push_back(new_param);
			if (new_param == nullptr)
				mWaveDump->addLog("GET" + mChParamsList.at(p), 0);
		}
	}

	CreateChParamsMap();

	for (int p = 0; p < mNumberOfGroupParams; p++) {
		for (int g = 0; g < (mNumberOfChannels/mGroupsize.at(p)); g++) {
			phandle = 0;
			CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle, qPrintable(mGroupParamsList.at(p).arg(g)), &phandle);
			if (err != CAEN_FELib_Success) {
				mWaveDump->addLog(QString("Get Handle of " + mGroupParamsList.at(p).arg(g) + " error %1").arg(err), false);
				mGroupWidgetsList.at(p)->setEnabled(false);
				mGroupParams.push_back(nullptr);
				continue;
			}
			CAENparameter* new_param = nullptr;
			try {
				new_param = new CAENparameter(phandle);
				new_param->SetWidget(mGroupWidgetsList.at(p));
			}
			catch (CAENFELibException& exc) {
				mGroupParams.push_back(new_param);
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
			mGroupParams.push_back(new_param);
			if (new_param == nullptr)
				mWaveDump->addLog("GET" + mGroupParamsList.at(p), 0);
		}
	}
	CreateGroupParamsMap();

	mDevParamsList.clear();//no more needed when the map is filled
	mDevWidgetsList.clear();
}

void Dx740V1ConfigPanel::defineScaleParams() {
	uint64_t phandle;
	if (mDecimationVisible) {

		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle, "/par/Decimation_Factor", &phandle);
		if (err != CAEN_FELib_Success) {
			mWaveDump->addLog(QString("Get Handle of /par/Decimation_Factor error %1").arg(err), false);
		}
		else {
			if (mDevGUIMap.value(ui->dBox_reclen) != nullptr && mUoMX == UOM_PHYS_UNIT) {
				mDevGUIMap.value(ui->dBox_reclen)->setScaleFactorParam(phandle);
				mDevGUIMap.value(ui->cBox_decim)->addRelatedParam(mDevGUIMap.value(ui->dBox_reclen));
			}
		}
	}
	
	if(mChGUIMap.value(ui->dSBox_bsl_level) != nullptr)
		mChGUIMap.value(ui->dSBox_bsl_level)->addRelatedParam(mChGUIMap.value(ui->dBox_thr));
	else if (mGroupGUIMap.value(ui->dSBox_bsl_level) != nullptr)
		mGroupGUIMap.value(ui->dSBox_bsl_level)->addRelatedParam(mGroupGUIMap.value(ui->dBox_thr));
}


void Dx740V1ConfigPanel::UpdateDevGUI() {
	this->ConfigurationPanel::updateDevGUI();
}

void Dx740V1ConfigPanel::UpdateChGUI() {
	this->ConfigurationPanel::updateChGUI();
}

void Dx740V1ConfigPanel::UpdateGroupGUI() {
	this->ConfigurationPanel::updateGroupGUI();
}

void Dx740V1ConfigPanel::Update(){
	for (int p = 0; p < mChParams.size(); p++) {
		if(mChParams.at(p) != nullptr)
			mChParams.at(p)->getValue();
	}
	for (int p = 0; p < mGroupParams.size(); p++) {
		if (mGroupParams.at(p) != nullptr)
			mGroupParams.at(p)->getValue();
	}
	UpdateMainSettings();
	this->ConfigurationPanel::Update();
}

void Dx740V1ConfigPanel::CreateChParamsMap() {
	mChGUIMap.clear();
	for (int p = 0; p < mNumberOfChParams; p++)
		mChGUIMap.insert(mChWidgetsList.at(p), mChParams.at((mNumberOfChParams * mFirstSelectedCh) + p));
	if (mUoMY == UOM_PHYS_UNIT && (mChGroupSize == 1)) {
			mChGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM(mWaveDump->getThrSampleTomV_f(mDevName, mFirstSelectedCh));
			mChGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM_incr(mWaveDump->getThrSampleTomV_off(mDevName, mFirstSelectedCh));
	}
	defineScaleParams();
}

void Dx740V1ConfigPanel::CreateGroupParamsMap() {
	mGroupGUIMap.clear();
	for (int p = 0; p < mNumberOfGroupParams; p++) {
		mGroupGUIMap.insert(mGroupWidgetsList.at(p), mGroupParams.at((p * (mNumberOfChannels / mGroupsize.at(p)) + mFirstSelectedGroup.at(p))));
	}
	if (mUoMY == UOM_PHYS_UNIT && (mChGroupSize > 1)) {
		mGroupGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM(mWaveDump->getThrSampleTomV_f(mDevName, mFirstSelectedCh));
		mGroupGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM_incr(mWaveDump->getThrSampleTomV_off(mDevName, mFirstSelectedCh));
	}
	defineScaleParams();
}

//multi sel. active --> check if the selected channels have the same param values
void Dx740V1ConfigPanel::CheckChParamValues() {
	for (int p = 0; p < mChParamsList.size(); p++) {
		if (mChGUIMap.value(mChWidgetsList.at(p)) == nullptr)
			continue;
		QString first_sel_val = mChGUIMap.value(mChWidgetsList.at(p))->getCachedValue();
		bool same_value = true;
		for (int i = 0; i < mNumberOfChannels; i++) {
			if (!mPbCh[i]->isChecked() || (i == mFirstSelectedCh) || mChParams.at((mNumberOfChParams * i) + p) == nullptr)
				continue;

			if (mChParams.at((mNumberOfChParams * i) + p)->getCachedValue() != first_sel_val) {
				same_value = false;
				break;
			}
		}

		if (!same_value) {
			WarningWidget(mChWidgetsList.at(p), false);
		}
	}
}
//multi sel. active --> check if the selected groups have the same param values
void Dx740V1ConfigPanel::CheckGroupParamValues() {
	for (int p = 0; p < mNumberOfGroupParams; p++) {
		if (mGroupGUIMap.value(mGroupWidgetsList.at(p)) == nullptr)
			continue;
		QString first_sel_val = mGroupGUIMap.value(mGroupWidgetsList.at(p))->getCachedValue();
		bool same_value = true;
		for (int i = 0; i < (mNumberOfChannels / mGroupsize.at(p)); i++) {
			bool some_channels_selected = false;
			if ((i == mFirstSelectedGroup.at(p)) || mGroupParams.at(p * (mNumberOfChannels / mGroupsize.at(p)) + i) == nullptr)
				continue;
			for (int ch = 0; ch < mGroupsize.at(p); ch++)
				if (mPbCh[(i * mGroupsize.at(p)) + ch]->isChecked()) {
					some_channels_selected = true;
					break;
			}
			if (!some_channels_selected)
				continue;
			QString val = mGroupParams.at(p * (mNumberOfChannels / mGroupsize.at(p)) + i)->getCachedValue();
			if ( val != first_sel_val) {
				same_value = false;
				break;
			}
		}

		if (!same_value) {
			WarningWidget(mGroupWidgetsList.at(p), false);
		}
	}
}

void Dx740V1ConfigPanel::channelClicked(CAENPushButton *pb) {
	int ch = pb->getIndex();

	//single channel selected
	if (!mMulti) {
		for (int i = 0; i < mNumberOfChannels; i++)
			if (i != ch) mPbCh[i]->setChecked(false);
		if (ch != mFirstSelectedCh) {
			mFirstSelectedCh = ch;
			for (int p = 0; p < mNumberOfGroupParams; p++)
				mFirstSelectedGroup.replace(p, (ch / mGroupsize.at(p)));
			CreateChParamsMap();
			UpdateChGUI();
			CreateGroupParamsMap();
			UpdateGroupGUI();
		}
		else
			mPbCh[ch]->setChecked(true);
	}
	else {//multi sel. active
		if (ch == mFirstSelectedCh) {
			for (int i = 0; i < mNumberOfChannels; i++) {
				if (mPbCh[i]->isChecked()) {
					mFirstSelectedCh = i;
					for (int p = 0; p < mNumberOfGroupParams; p++)
						mFirstSelectedGroup.replace(p, (i / mGroupsize.at(p)));
					CreateChParamsMap();
					UpdateChGUI();
					CreateGroupParamsMap();
					UpdateGroupGUI();
					break;
				}
			}
			mPbCh[mFirstSelectedCh]->setChecked(true);
		}

		CheckChParamValues();
		CheckGroupParamValues();
	}
}
void Dx740V1ConfigPanel::selectAll() {
	for (int i = 0; i < mNumberOfChannels; i++)
		mPbCh[i]->setChecked(true);
	ui->pbMulti->setChecked(true);
	CheckChParamValues();
	CheckGroupParamValues();
}

void  Dx740V1ConfigPanel::MultiClicked() {
	mMulti = ui->pbMulti->isChecked();
	if (!mMulti) {
		QHash<QWidget *, CAENparameter *>::iterator item;
		for (item = mChGUIMap.begin(); item != mChGUIMap.end(); ++item) {
			NormalWidget(item.key(), false);
		}
		int memCh = mFirstSelectedCh;
		mFirstSelectedCh = -1;
		emit channelClicked(mPbCh[memCh]);
	}
}

void Dx740V1ConfigPanel::WarningWidget(QWidget *widget, bool bold) {
	if (!widget->styleSheet().contains("color: black"))
		return;
	if(bold)
		widget->setStyleSheet("font-weight: bold; color: red");
	else
		widget->setStyleSheet("color: red");
	mIgnoreEvent = true;
	if (QComboBox *cbox = qobject_cast<QComboBox *>(widget)) {
		cbox->setCurrentIndex(-1);
		mIgnoreEvent = false;
		return;
	}
	if(QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(widget)){
		doubleSpinBox->setSpecialValueText("?");
		doubleSpinBox->setMinimum(doubleSpinBox->minimum() - 1);
		doubleSpinBox->setValue(doubleSpinBox->minimum());
		mIgnoreEvent = false;
		return;
	}
	if(QSpinBox *SpinBox = qobject_cast<QSpinBox *>(widget)){
		SpinBox->setSpecialValueText("?");
		SpinBox->setMinimum(SpinBox->minimum() - 1);
		SpinBox->setValue(SpinBox->minimum());
		mIgnoreEvent = false;
		return;
	}
	mIgnoreEvent = false;
}

void Dx740V1ConfigPanel::NormalWidget(QWidget *widget, bool bold) {
	if (!widget->styleSheet().contains("color: red"))
		return;
	if(bold)
		widget->setStyleSheet("font-weight: bold; color: black");
	else
		widget->setStyleSheet("color: black");
	mIgnoreEvent = true;
	QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(widget);
	if (doubleSpinBox != nullptr) {
		doubleSpinBox->setSpecialValueText("");
		doubleSpinBox->setMinimum(doubleSpinBox->minimum() + 1);
		mIgnoreEvent = false;
		return;
	}
	QSpinBox *SpinBox = qobject_cast<QSpinBox *>(widget);
	if (SpinBox != nullptr) {
		SpinBox->setSpecialValueText("");
		SpinBox->setMinimum(SpinBox->minimum() + 1);
		mIgnoreEvent = false;
		return;
	}
	mIgnoreEvent = false;
	
}

void Dx740V1ConfigPanel::ApplyNewValue(QWidget *widget) {
	try {
		if (mDevGUIMap.contains(widget) && mDevGUIMap.value(widget) != nullptr) {
			mDevGUIMap.value(widget)->SetValueFromWidget();
			QDoubleSpinBox* doubleSpinBox = qobject_cast<QDoubleSpinBox*>(widget);
			if (doubleSpinBox != nullptr) {
				if (doubleSpinBox == ui->dBox_reclen)
					mWaveDump->add2LogAndConf("SET", mDevName, mDevGUIMap.value(widget)->getQry(), QString("%1").arg(ui->dBox_reclen->value()), QString("%1").arg(ui->dBox_reclen->value()), CAEN_FELib_Success);
				else
					mWaveDump->add2LogAndConf("SET", mDevName, mDevGUIMap.value(widget)->getQry(), mDevGUIMap.value(widget)->getCachedValue(), mDevGUIMap.value(widget)->getCachedValue(), CAEN_FELib_Success);
			}
			else
				mWaveDump->add2LogAndConf("SET", mDevName, mDevGUIMap.value(widget)->getQry(), mDevGUIMap.value(widget)->getCachedValue(), mDevGUIMap.value(widget)->getCachedValue(), CAEN_FELib_Success);
			mIgnoreEvent = true;
			mDevGUIMap.value(widget)->UpdateWidget();
			mIgnoreEvent = false;
		}
		else if (mChGUIMap.contains(widget)) {
			if (!ui->pbMulti->isChecked() && mChGUIMap.value(widget) != nullptr) {//single channel selected
				mChGUIMap.value(widget)->SetValueFromWidget();
				mWaveDump->add2LogAndConf("SET", mDevName, mChGUIMap.value(widget)->getQry(), mChGUIMap.value(widget)->getCachedValue(), mChGUIMap.value(widget)->getCachedValue(), CAEN_FELib_Success);
				mIgnoreEvent = true;
				mChGUIMap.value(widget)->UpdateWidget();
				mIgnoreEvent = false;
			}
			else {//multi channel select
				int index = mChWidgetsList.indexOf(widget);
				for (int ch = 0; ch < mNumberOfChannels; ch++) {
					if (!mPbCh[ch]->isChecked())
						continue;
					if (mChParams.at((mNumberOfChParams * ch) + index) == nullptr)
						continue;
					mChParams.at((mNumberOfChParams * ch) + index)->SetValueFromWidget();
					mWaveDump->add2LogAndConf("SET", mDevName, mChParams.at((mNumberOfChParams * ch) + index)->getQry(), mChParams.at((mNumberOfChParams * ch) + index)->getCachedValue(), mChParams.at((mNumberOfChParams * ch) + index)->getCachedValue(), CAEN_FELib_Success);
				}
				mIgnoreEvent = true;
				mChParams.at((mNumberOfChParams * mFirstSelectedCh) + index)->UpdateWidget();
				mIgnoreEvent = false;
				NormalWidget(widget, false);
			}
		}
		else if (mGroupGUIMap.contains(widget)) {
			if (!ui->pbMulti->isChecked() && mGroupGUIMap.value(widget) != nullptr) {//single channel selected
				mGroupGUIMap.value(widget)->SetValueFromWidget();
				mWaveDump->add2LogAndConf("SET", mDevName, mGroupGUIMap.value(widget)->getQry(), mGroupGUIMap.value(widget)->getCachedValue(), mGroupGUIMap.value(widget)->getCachedValue(), CAEN_FELib_Success);
				mIgnoreEvent = true;
				mGroupGUIMap.value(widget)->UpdateWidget();
				mIgnoreEvent = false;
			}
			else {//multi channel select
				int index = mGroupWidgetsList.indexOf(widget);
				int group = -1;
				for (int ch = 0; ch < mNumberOfChannels; ch++) {
						if (!mPbCh[ch]->isChecked())
							continue;
						if (ch / mGroupsize.at(index) == group)
							continue;
						group = ch / mGroupsize.at(index);
						int k = (index * (mNumberOfChannels / mGroupsize.at(index)) + group);
						if (mGroupParams.at(k) == nullptr)
							continue;
						mGroupParams.at(k)->SetValueFromWidget();
						mWaveDump->add2LogAndConf("SET", mDevName, mGroupParams.at(k)->getQry(), mGroupParams.at(k)->getCachedValue(), mGroupParams.at(k)->getCachedValue(), CAEN_FELib_Success);
				}				
				mIgnoreEvent = true;
				mGroupParams.at(index * (mNumberOfChannels / mGroupsize.at(index)) + mFirstSelectedGroup.at(index))->UpdateWidget();
				mIgnoreEvent = false;
				NormalWidget(widget, false);
			}
		}
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}
}

void Dx740V1ConfigPanel::UpdateRelatedParams(QWidget *widget) {
	mIgnoreEvent = true;
	if (widget == ui->cBox_gain) {
		if (mUoMY == UOM_PHYS_UNIT) {
			mChGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM(mWaveDump->getDevChSampleToV(mDevName, mFirstSelectedCh) * 1e3);
			mChGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM_incr(mWaveDump->getThrSampleTomV_off(mDevName, mFirstSelectedCh));
			mChGUIMap.value(ui->dBox_thr)->UpdateWidget();
		}
	}
	if (mDevGUIMap.contains(widget)) {
		for (int p = 0; p < mDevGUIMap.value(widget)->RelatedParams.size(); p++)
			mDevGUIMap.value(widget)->RelatedParams.at(p)->UpdateWidget();
	}
	else if (mChGUIMap.contains(widget)) {
		if (mUoMY == UOM_PHYS_UNIT && (mChGroupSize == 1)) {//update thr uom that depends on the baseline
				mChGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM(mWaveDump->getThrSampleTomV_f(mDevName, mFirstSelectedCh));
				mChGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM_incr(mWaveDump->getThrSampleTomV_off(mDevName, mFirstSelectedCh));
		}
		for (int p = 0; p < mChGUIMap.value(widget)->RelatedParams.size(); p++)
			mChGUIMap.value(widget)->RelatedParams.at(p)->UpdateWidget();
	}
	else if (mGroupGUIMap.contains(widget)) {
		if (mUoMY == UOM_PHYS_UNIT && (mChGroupSize > 1)) {//update thr uom that depends on the baseline
			mGroupGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM(mWaveDump->getThrSampleTomV_f(mDevName, mFirstSelectedCh));
			mGroupGUIMap.value(ui->dBox_thr)->setUoMToGUIUoM_incr(mWaveDump->getThrSampleTomV_off(mDevName, mFirstSelectedCh));
		}
		for (int p = 0; p < mGroupGUIMap.value(widget)->RelatedParams.size(); p++)
			mGroupGUIMap.value(widget)->RelatedParams.at(p)->UpdateWidget();
	}
	mIgnoreEvent = false;
}

void Dx740V1ConfigPanel::UpdateSamplFreq() {
	char val[32];
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mDevHandle, "/par/ADC_SamplRate", val);
	if (err != CAEN_FELib_Success) {
		char error[1024];
		CAEN_FELib_GetLastError(error);
		mWaveDump->addLog("Error reading ADC sampling rate: " + QString(error), false);
		ui->label_freq_decimation->setText("[Frequency = unknown]");
	}
	else {
		double freq = QString(val).toDouble();
		QString vals = ui->cBox_decim->currentText();
		vals.remove("DECIM_FACTOR_");
		freq /= vals.toInt();
		ui->label_freq_decimation->setText(QString("[Frequency = %1 MHz]").arg(freq));
	}
}

void Dx740V1ConfigPanel::intChanged(int val) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	int mem_post_val;
	if (widget == ui->cBox_decim)
		mem_post_val = (mDev->getPostTrgParam()->getCachedValue().toInt() / mDev->getDecimation());
	ApplyNewValue(widget);
	UpdateRelatedParams(widget);

	if (widget == ui->cBox_decim) {
		UpdateSamplFreq();
		int new_val = (mem_post_val * mDev->getDecimation());
		mDev->getPostTrgParam()->setValue(QString("%1").arg(new_val));
		UpdatePreTrigger();
		mWaveDump->manageAddLogAndConf("PreTrigger", mDev, QString("%1").arg(mDev->getPreTrgCachedValue()), QString("%1").arg(mDev->getPreTrgCachedValue()));
	}
}

void Dx740V1ConfigPanel::boolChanged(bool val) {
	if (mIgnoreEvent)
		return;
	QWidget* widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
	UpdateRelatedParams(widget);
}

void Dx740V1ConfigPanel::doubleChanged(double val) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	QDoubleSpinBox* doubleSpinBox = qobject_cast<QDoubleSpinBox*>(widget);
	if (doubleSpinBox != nullptr) {
		if (doubleSpinBox == ui->dBox_reclen) {
			if (mUoMX == UOM_SAMPLE && val > MAX_SW_RECLEN_SAMPLES) {
				doubleSpinBox->setValue(MAX_SW_RECLEN_SAMPLES);
				return;
			}
			else if (mUoMX == UOM_PHYS_UNIT && val > (MAX_SW_RECLEN_SAMPLES * mWaveDump->getDevSampleToS(mDev) * 1e6)) {
				doubleSpinBox->setValue(MAX_SW_RECLEN_SAMPLES * mWaveDump->getDevSampleToS(mDev) * 1e6);
				return;
			}
		}
	}
	ApplyNewValue(widget);
	UpdateRelatedParams(widget);
	if (widget == ui->dBox_reclen && mUoMX == UOM_PHYS_UNIT)//update post trigger samples so that pretrg in ns remais the same
		PreTrgChanged(ui->dBox_pretrg->value());
}

void Dx740V1ConfigPanel::itemCheckedChanged(QListWidgetItem *item) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void Dx740V1ConfigPanel::TrgSourceChanged(QListWidgetItem* item) {
	if (mIgnoreEvent)
		return;
	QString val = "";
	bool check_ok = false;
	for (int i = 0; i < ui->lWidget_Trg_Src->count(); ++i) {
		if (ui->lWidget_Trg_Src->item(i)->checkState() == Qt::Checked) {
			check_ok = true;
			if (val == "")
				val.append(ui->lWidget_Trg_Src->item(i)->text());
			else
				val.append("|" + ui->lWidget_Trg_Src->item(i)->text());
		}
	}
	if (check_ok) {
		if (val.contains("Disabled"))
			mDev->setTrgSource("Disabled");
		else
			mDev->setTrgSource(val);
	}
	else {
		mDev->setTrgSource("Disabled");
	}
}

void Dx740V1ConfigPanel::ClockSourceChanged(int) {
	if (mIgnoreEvent)
		return;
	try{
		if (ui->cBox_clk_source->currentText() == "Internal")
			mDev->setExtClkSource("False");
		else
			mDev->setExtClkSource("True");
		}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}
}

void Dx740V1ConfigPanel::PreTrgChanged(double value) {
	if (mIgnoreEvent)
		return;
	int decim = mDev->getDecimation();
	double pretrg_value;
	if(mUoMX == UOM_PHYS_UNIT)
		pretrg_value = value * 1000. / decim;
	else
		pretrg_value = (value * mDev->getSample_to_S() * 1e9) * decim;
	try {
		mDev->setPreTrg(QString("%1").arg(pretrg_value));
		ui->dBox_pretrg->blockSignals(true);
		if (mUoMX == UOM_PHYS_UNIT) {
			ui->dBox_pretrg->setValue((mDev->getPreTrgCachedValue() / 1000.) * decim);
		}
		else {
			ui->dBox_pretrg->setValue(mDev->getPreTrgCachedValueS() / decim);
		}
		mWaveDump->manageAddLogAndConf("PreTrigger", mDev, QString("%1").arg(mDev->getPreTrgCachedValue()), QString("%1").arg(mDev->getPreTrgCachedValue()));
		ui->dBox_pretrg->blockSignals(false);
	}
	catch (CAENFELibException& exc) {
		ui->dBox_pretrg->setEnabled(false);
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}
}

void Dx740V1ConfigPanel::SetMask() {
	QString current_mask = mDev->getSelfTrgMask();
	uint64_t val = current_mask.toULongLong();
	chmaskDialog *dlg = new chmaskDialog(this, mDev->getNumCh(), val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask;
		new_mask.setNum(dlg->getMask());
		ui->lineEdit_trig_mask->setText(QString("%1").arg(dlg->getMask(), 0, 16, QLatin1Char('0')));
		try {
			mDev->setSelfTrgMask(new_mask);
			mWaveDump->manageAddLogAndConf("SelfTriggerMask", mDev, mDev->getSelfTrgMask(), mDev->getSelfTrgMask());
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
		}
	}
	delete dlg;
}

void Dx740V1ConfigPanel::SetTrgoutMask() {
	QString current_mask = mDev->getTrgOutMask();
	uint64_t val = current_mask.toULongLong();
	chmaskDialog* dlg = new chmaskDialog(this, mDev->getNumCh(), val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask;
		new_mask.setNum(dlg->getMask());
		ui->lineEdit_trgout_mask->setText(QString("%1").arg(dlg->getMask(), 0, 16, QLatin1Char('0')));
		try {
			mDev->setTrgOutMask(new_mask);
			mWaveDump->manageAddLogAndConf("TrgoutMask", mDev, mDev->getTrgOutMask(), mDev->getTrgOutMask());
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
		}
	}
	delete dlg;
}


void Dx740V1ConfigPanel::TextChanged(QString text) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void Dx740V1ConfigPanel::calibrateADC() {
	try {
		mDev->CalibrateADC();
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
		QMessageBox::critical(this,"ADC calibration","Error executing ADC Calibration.");
		return;
	}
	QMessageBox::information(this, "ADC calibration", "ADC Calibration completed successfully.");
}

void Dx740V1ConfigPanel::TrgoutChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	CAEN_FELib_ErrorCode err= CAEN_FELib_Success;
	QString value;
	ui->pB_trgout_mask->setEnabled(false);
	try{
		if (mode == "TRIGGER") {
			value = "OUT_PROPAGATION_TRIGGER";			
			uint64_t m = mDev->getTrgOutMask().toULongLong();
			ui->lineEdit_trgout_mask->setText(QString("%1").arg(m, 0, 16, QLatin1Char('0')));
		}
		else if (mode == "SIN") {
			value = "OUT_PROPAGATION_SYNCIN";
		}
		else if (mode == "RUN") {
			value = "OUT_PROPAGATION_RUN";
		}
		else if (mode == "REFCLK") {
			value = "OUT_PROPAGATION_SAMPLE_CLK";
		}

		mDev->setTRGOUTMode(value);
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
		err = (CAEN_FELib_ErrorCode)exc.ErrorCode();
		return;
	}
	mWaveDump->add2LogAndConf("SET", mDevName, "/par/OUT_Selection", mode, value, CAEN_FELib_Success);
	NormalWidget(ui->cBox_trgout1, true);
}

void Dx740V1ConfigPanel::TriggerModeChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	ui->pB_trg_mask->setEnabled(false);
	ui->cBox_polarity->setEnabled(false);
	if (mode == "SW_TRIGGER_ONLY") {
		try {
			mDev->setTrgSource("Sw");
			mWaveDump->manageAddLogAndConf("TriggerSource", mDev, "Sw", "Sw");
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
			err = (CAEN_FELib_ErrorCode)exc.ErrorCode();
		}
	}
	else if (mode == "SELF_TRIGGER") {
		ui->pB_trg_mask->setEnabled(true);
		ui->cBox_polarity->setEnabled(true);
		mIgnoreEvent = true;
		uint64_t m = mDev->getSelfTrgMask().toULongLong();
		ui->lineEdit_trig_mask->setText(QString("%1").arg(m, 0, 16, QLatin1Char('0')));
		mIgnoreEvent = false;
		try {
			mDev->setTrgSource("ITL");
			mWaveDump->manageAddLogAndConf("TriggerSource", mDev, "ITL", "ITL");
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
			err = (CAEN_FELib_ErrorCode)exc.ErrorCode();
		}
	}
	else if (mode == "EXTERNAL_TRGIN") {
		try {
			mDev->setTrgSource("TrgIn");
			ui->lineEdit_trig_mask->setText("0");
			mWaveDump->manageAddLogAndConf("TriggerSource", mDev, "TrgIn", "TrgIn");
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
			err = (CAEN_FELib_ErrorCode)exc.ErrorCode();
		}
	}

	if (err == CAEN_FELib_Success)
		NormalWidget(ui->cBox_trg_mode, true);
}


void Dx740V1ConfigPanel::UpdateTriggerMode() {
	mIgnoreEvent = true;
	QString TrgMode = mDev->getTrgSource();

	if (TrgMode.contains("ITL", Qt::CaseInsensitive)) {
		ui->cBox_trg_mode->setCurrentText("SELF_TRIGGER");
		ui->pB_trg_mask->setEnabled(true);
		ui->cBox_polarity->setEnabled(true);
		uint64_t m = mDev->getSelfTrgMask().toULongLong();
		ui->lineEdit_trig_mask->setText(QString("%1").arg(m, 0, 16, QLatin1Char('0')));
	}
	else if (TrgMode.contains("TrgIn", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("EXTERNAL_TRGIN");
	else if (TrgMode.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("EXTERNAL_GPO");
	else if (TrgMode.contains("Sw", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("SW_TRIGGER_ONLY");
	else
		WarningWidget(ui->cBox_trg_mode, true);
	QStringList values = TrgMode.split("|");
	QVector<QString>mValues;
	mValues.append("SwCmd");
	mValues.append("ITL");
	mValues.append("TrgIn");
	
	ui->lWidget_Trg_Src->clear();
	for (int val = 0; val < mValues.size(); val++) {
		QListWidgetItem* newItem = new QListWidgetItem;
		newItem->setText(mValues.at(val));
		newItem->setFlags(newItem->flags() | Qt::ItemIsUserCheckable);
		newItem->setTextAlignment(Qt::AlignCenter);
		
		if (values.contains(mValues.at(val)))
			newItem->setCheckState(Qt::Checked);
		else
			newItem->setCheckState(Qt::Unchecked);
		
		ui->lWidget_Trg_Src->insertItem(val, newItem);
	}
	ui->lWidget_Trg_Src->setMaximumWidth(ui->lWidget_Trg_Src->sizeHintForColumn(0) + 5 * ui->lWidget_Trg_Src->frameWidth());
	
	mIgnoreEvent = false;	
}

void Dx740V1ConfigPanel::UpdateTrgoutMode() {
	mIgnoreEvent = true;

	QString TrgoutMode = mDev->getTRGOUTMode();
	if (TrgoutMode.contains("TRIGGER", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("TRIGGER");
		ui->pB_trgout_mask->setEnabled(true);
		uint64_t m = mDev->getTrgOutMask().toULongLong();
		ui->lineEdit_trgout_mask->setText(QString("%1").arg(m, 0, 16, QLatin1Char('0')));
	}
	else if (TrgoutMode.contains("SYNCIN", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("SIN");
	else if (TrgoutMode.contains("RUN", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("RUN");
	else if (TrgoutMode.contains("SAMPLE_CLK", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("REFCLK");
	else
		WarningWidget(ui->cBox_trgout1, true);
	mIgnoreEvent = false;
}

void Dx740V1ConfigPanel::UpdatePreTrigger() {
	mIgnoreEvent = true;
	if (mUoMX == UOM_SAMPLE) {
		ui->dBox_pretrg->setMinimum(0);
		ui->dBox_pretrg->setDecimals(0);
		ui->dBox_pretrg->setMaximum(mDev->getPreTrg_max() / (mDev->getSample_to_S() * 1e9));
		ui->dBox_pretrg->setSingleStep(1);
		ui->dBox_pretrg->setValue(mDev->getPreTrg_value() / (mDev->getSample_to_S() * 1e9 * mDev->getDecimation()));//show value in reclen samples(decimated)
	}
	else {
		ui->dBox_pretrg->setDecimals(3);
		ui->dBox_pretrg->setMinimum(mDev->getPreTrg_min() / 1000.);
		ui->dBox_pretrg->setMaximum(mDev->getPreTrg_max() / 1000.);
		ui->dBox_pretrg->setSingleStep(mDev->getPreTrg_incr() / 1000.);
		ui->dBox_pretrg->setValue(mDev->getPreTrg_value() / 1000.);
	}
	mIgnoreEvent = false;
}

void Dx740V1ConfigPanel::UpdateClkSource() {
	mIgnoreEvent = true;
	try {
		QString clk = mDev->getExtClkSource();
		if (clk == "CLK_SRC_FRONTPANEL" || clk == "TRUE")
			ui->cBox_clk_source->setCurrentText("External");
		else
			ui->cBox_clk_source->setCurrentText("Internal");

		if (clk == "CLK_SRC_FRONTPANEL" || clk == "CLK_SRC_INTERNAL")
			ui->cBox_clk_source->setEnabled(false);
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
		ui->cBox_clk_source->setEnabled(false);
	}
	
	mIgnoreEvent = false;
}

void Dx740V1ConfigPanel::currentTabChanged(int tab){
	if (mIgnoreEvent)
		return;
	QWidget *currWidget = ui->tabWidget->currentWidget();
	if (currWidget == ui->tab_main) {
		UpdateMainSettings();
	}
	else if (currWidget == ui->tab_advanced) {
		mDevGUIMap.value(ui->cBox_StartMode)->getValue();
		mDevGUIMap.value(ui->cBox_StartMode)->UpdateWidget();
		mDevGUIMap.value(ui->cBox_TRGOUT)->getValue();
		mDevGUIMap.value(ui->cBox_TRGOUT)->UpdateWidget();
		//mDevGUIMap.value(ui->cBox_GPIO)->getValue();
		//mDevGUIMap.value(ui->cBox_GPIO)->UpdateWidget();
	}
}


Dx740V1ConfigPanel::~Dx740V1ConfigPanel()
{
	delete ui;
}
