/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CommonConfigPanel1.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "CommonConfigPanel1.h"
#include "WaveDump2.h"
#include "CAENSwitch.h"
#include "CAENFELibException.h"
#include "chmaskDialog.h"
#include <QDebug>


CommonConfigPanel1::CommonConfigPanel1(WaveDump2 *parent, QVector<CAENDig1Device*> devs, bool show_wait) :
	QDialog(parent),
	ui(new Ui::CommonConfigPanel1) {
	ui->setupUi(this);

	mWaveDump = parent;
	for (int d = 0; d < devs.size(); d++)
		mDevs.append(devs.at(d));
	mUoMX = mWaveDump->getUoMX();
	mUoMY = mWaveDump->getUoMY();	

	CAENSwitch pippo("on","off",1,this);
	ui->lay_0->addWidget(&pippo);
	for (int i = 0; i < 64; i++) {
		mPbCh[i] = new CAENPushButton(this, i);
		mPbCh[i]->setMaximumWidth(43);
		mPbCh[i]->setText("CH " + QString::number(i));
		mPbCh[i]->setCheckable(true);
		connect(mPbCh[i], &CAENPushButton::clicked, [=] {channelClicked(mPbCh[i]); });
	}

	mNumBrds = devs.size();
	ui->cBox_gain->setVisible(false);
	ui->label_17->setVisible(false);
	ui->label->setVisible(false);
	ui->cBox_decim->setVisible(false);
	mDevWithCHGain.resize(mNumBrds);
	mDevWithCHGain.fill(0);
	mDevWithDecim.resize(mNumBrds);
	mDevWithDecim.fill(0);
	mDevWithGroupedChannels.resize(mNumBrds);
	mDevWithGroupedChannels.fill(0);
	mChGroupSize.resize(mNumBrds);
	mChGroupSize.fill(1);
	for (int b = 0; b < mNumBrds; b++) {
		mNumberOfChannels.push_back(mDevs.at(b)->getNumCh());

		QString val = mDevs.at(b)->getModel();
		if (val.contains("730") || val.contains("725")) {
			if (!mCHGainVisible) {
				ui->cBox_gain->setVisible(true);
				ui->label_17->setVisible(true);
				//ui->label->setVisible(false);
				mCHGainVisible = true;
			}
			mDevWithCHGain.replace(b, 1);
		}
		else if (val.contains("740")) {
			if (!mDecimationVisible) {
				mDecimationVisible = true;
				ui->label->setVisible(true);
				ui->cBox_decim->setVisible(true);
				ui->cBox_decim->setEnabled(true);
				mDevWithDecim.replace(b, 1);
			}
			mDevWithGroupedChannels.replace(b,1);
			mChGroupSize.replace(b,mDevs.at(b)->getChGroupSize());
		}
		else {
			ui->cBox_gain->setVisible(false);
			ui->label_17->setVisible(false);
		}
	}

	for (int b = 0; b < mNumBrds; b++) {
		if ((mChGroupSize.at(b) > 1) && (mChGroupSize.at(b) < mMinGroupSize))
			mMinGroupSize = mChGroupSize.at(b);
	}
	if (mMinGroupSize == 100)
		mMinGroupSize = 1;

	for (int i = 0; i < 4; i++) {
		for (int j = 0; j < 16; j++) {
			switch (i) {
				case 0:
					ui->lay_0->addWidget((mPbCh[(i * 16) + j]));
					break;
				case 1:
					ui->lay_1->addWidget(mPbCh[(i * 16) + j]);
					break;
				case 2:
					ui->lay_2->addWidget(mPbCh[(i * 16) + j]);
					break;
				case 3:
					ui->lay_3->addWidget(mPbCh[(i * 16) + j]);
					break;
			}
		}
	}

	mPbCh[0]->setChecked(true);
	mFirstSelectedCh = 0;
	mFirstSelectedGroup.clear();
	mIgnoreEvent = false;
	mMulti = false;

	ui->pB_trg_mask->setEnabled(false);
	ui->pB_trgout_mask->setEnabled(false);

	connectWidgets();
	FillParamsAndWidgetsList();
	createMaps();
	defineScaleParams();
	FillParamsInfo();

	UpdateParamsUoM();
	QApplication::processEvents();
	UpdateDevGUI();
	QApplication::processEvents();
	UpdateChGUI();
	QApplication::processEvents();
	UpdateGroupGUI();
	QApplication::processEvents();
	UpdateMainSettings();
}

void CommonConfigPanel1::UpdateMainSettings() {
	UpdateTriggerMode();
	UpdateTrgoutMode();
	UpdatePreTrigger();
	UpdateClkSource();
}

void CommonConfigPanel1::connectWidgets() {
	//connections
	connect(ui->pbAll, SIGNAL(clicked()), this, SLOT(selectAll()));
	connect(ui->pbMulti, SIGNAL(clicked()), this, SLOT(MultiClicked()));
	connect(ui->dBox_reclen, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dBox_pretrg, SIGNAL(valueChanged(double)), this, SLOT(PreTrgChanged(double)));
	connect(ui->cBox_decim, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_trg_mode, SIGNAL(currentTextChanged(QString)), this, SLOT(TriggerModeChanged(QString)));
	connect(ui->pB_trg_mask, SIGNAL(clicked()), this, SLOT(SetMask()));
	//connect(ui->lineEdit_trig_mask, SIGNAL(textChanged(QString)), this, SLOT(TextChanged(QString)));
	connect(ui->cBox_trgout1, SIGNAL(currentTextChanged(QString)), this, SLOT(TrgoutChanged(QString)));
	connect(ui->pB_trgout_mask, SIGNAL(clicked()), this, SLOT(SetTrgoutMask()));
	connect(ui->cBox_polarity, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_enable, SIGNAL(currentIndexChanged(int)), this, SLOT(EnableChanged(int)));
	connect(ui->dBox_thr, SIGNAL(valueChanged(double)), this, SLOT(ThrChanged(double)));
	connect(ui->dSBox_bsl_level, SIGNAL(valueChanged(double)), this, SLOT(DCOChanged(double)));
	connect(ui->cBox_gain, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_StartMode, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->lWidget_Trg_Src, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(TrgSourceChanged(QListWidgetItem*)));
	connect(ui->cBox_clk_source, SIGNAL(currentIndexChanged(int)), this, SLOT(ClockSourceChanged(int)));
	connect(ui->cBox_trg_ext_trgout, SIGNAL(stateChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_trg_sw_trgout, SIGNAL(stateChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_TRGOUT, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	//connect(ui->cBox_GPIO, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	////////
	connect(ui->tabWidget, SIGNAL(currentChanged(int)), this, SLOT(currentTabChanged(int)));
}

void CommonConfigPanel1::FillParamsAndWidgetsList() {
	//start creating the list of device params
	mDevParamsList.clear();
	mDevWidgetsList.clear();

	mDevParamsList << "/par/RecLen";
	mDevWidgetsList << ui->dBox_reclen;
	ui->dBox_reclen->setToolTip("/par/RecLen");

	mDevParamsList << QString("/par/Self_Trigger_Edge");
	mDevWidgetsList << ui->cBox_polarity;
	ui->cBox_polarity->setToolTip("/par/Self_Trigger_Edge");

	mDevParamsList << "/par/StartMode";
	mDevWidgetsList << ui->cBox_StartMode;
	ui->cBox_StartMode->setToolTip("/par/StartMode");

	mDevParamsList << "/par/Trg_Ext_Out_Propagate";
	mDevWidgetsList << ui->cBox_trg_ext_trgout;
	ui->cBox_trg_ext_trgout->setToolTip("/par/Trg_Ext_Out_Propagate");

	mDevParamsList << "/par/Trg_Sw_Out_Propagate";
	mDevWidgetsList << ui->cBox_trg_sw_trgout;
	ui->cBox_trg_sw_trgout->setToolTip("/par/Trg_Sw_Out_Propagate");

	mDevParamsList << "/par/Out_Selection";
	mDevWidgetsList << ui->cBox_TRGOUT;
	ui->cBox_TRGOUT->setToolTip("/par/Out_Selection");

	if (mDecimationVisible) {
		mDevParamsList << "/par/Decimation_Factor";
		mDevWidgetsList << ui->cBox_decim;
		ui->cBox_decim->setToolTip("/par/Decimation_Factor");
	}
	//add more device parameters if needed ...

	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mDevWidgetsList.size(); i++)
		mDevWidgetsList.at(i)->setStyleSheet("color: black");

	//create the list of channel params
	mChParamsList.clear();
	mChWidgetsList.clear();	

	mChParamsList << QString("/ch/%1/par/Ch_InDyn");
	mChWidgetsList << ui->cBox_gain;
	ui->cBox_gain->setToolTip("/ch/N/par/Ch_InDyn");

	//add more channel parameters if needed
	mNumberOfChParams = mChParamsList.size();
	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mChWidgetsList.size(); i++)
		mChWidgetsList.at(i)->setStyleSheet("color: black");

	//create the list of group params
	mGroupParamsList.clear();
	mGroupWidgetsList.clear();
	mGroupsize.clear();
	

	//add more group parameters if needed
	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mGroupWidgetsList.size(); i++)
		mGroupWidgetsList.at(i)->setStyleSheet("color: black");
	mNumberOfGroupParams = mGroupParamsList.size();
	for (int g = 0; g < mNumberOfGroupParams; g++)
		mFirstSelectedGroup << 0; //for every group param depending on the group size there is a first selected group
}

void CommonConfigPanel1::FillParamsInfo() {
	CAEN_FELib_ErrorCode err;
	char val[200];
	uint64_t phandle;

	for (int p = 0; p < mDevParamsList.size(); p++) {

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevs.at(0)->getHandle(), qPrintable(mDevParamsList.at(p)), &phandle);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/dataType", val);

		if (QString(val) == "STRING") {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/multipleValue", val);
			if (err == CAEN_FELib_Success)
				multipleValue << 1;
			else
				multipleValue << 0;
		}
		else
			multipleValue << 0;

		ParamInfo info;
		info.min = "1000000000";
		info.incr = "1000000000";
		info.max = "0";
		fillInfo(mDevParamsList.at(p), &info);

		mDevParamsInfo.append(info);
	}

	for (int p = 0; p < mChParamsList.size(); p++) {

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevs.at(0)->getHandle(), qPrintable(mChParamsList.at(p).arg(0)), &phandle);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/dataType", val);

		if (QString(val) == "STRING") {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/multipleValue", val);
			if (err == CAEN_FELib_Success)
				multipleValue << 1;
			else
				multipleValue << 0;
		}
		else
			multipleValue << 0;

		ParamInfo info;
		info.min = "1000000000";
		info.incr = "1000000000";
		info.max = "0";
		fillInfo(mChParamsList.at(p).arg(0), &info);

		mChParamsInfo.append(info);
	}

	for (int p = 0; p < mGroupParamsList.size(); p++) {

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevs.at(0)->getHandle(), qPrintable(mGroupParamsList.at(p).arg(0)), &phandle);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/dataType", val);

		if (QString(val) == "STRING") {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/multipleValue", val);
			if (err == CAEN_FELib_Success)
				multipleValue << 1;
			else
				multipleValue << 0;
		}
		else
			multipleValue << 0;

		ParamInfo info;
		info.min = "1000000000";
		info.incr = "1000000000";
		info.max = "0";
		fillInfo(mGroupParamsList.at(p).arg(0), &info);

		mGroupParamsInfo.append(info);
	}
}

void CommonConfigPanel1::fillInfo(QString qry, ParamInfo* info) {
	CAEN_FELib_ErrorCode err;
	char val[200];
	uint64_t phandle;
	for (int b = 0; b < mNumBrds; b++) {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevs.at(0)->getHandle(), qPrintable(qry), &phandle);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/allowedValues", val);
		if (err == CAEN_FELib_Success) {//for list parameter fill allowed values vector
			if (atoi(val)) {
				uint64_t hallowed[32];
				int nvalues = (CAEN_FELib_ErrorCode)CAEN_FELib_GetChildHandles(phandle, "/allowedValues/", hallowed, 32);
				if (nvalues > 0) {
					for (int i = 0; i < nvalues; i++) {
						err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(hallowed[i], "", val);
						if (err != CAEN_FELib_Success)
							break;
						if (!info->allowed_values.contains(QString(val)))
							info->allowed_values.append(QString(val));
					}
				}
			}
		}
		else {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/minValue", val);
			if (err != CAEN_FELib_Success)
				info->min = "0";
			else {
				if (atoi(val) < info->min.toInt())
					info->min = QString(val);
			}
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/maxValue", val);
			if (err != CAEN_FELib_Success)
				info->max = "1000000000";
			else {
				if (atoi(val) > info->max.toInt())
					info->max = QString(val);
			}

			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/increment", val);
			if (err != CAEN_FELib_Success)
				info->incr = "1";
			else {
				if (atoi(val) < info->incr.toInt())
					info->incr = QString(val);
			}
		}
	}
}

void CommonConfigPanel1::UpdateParamsUoM() {
	if (mUoMX == UOM_PHYS_UNIT) {
		mWwithUoMX << ui->dBox_reclen;
		QString mu = QChar(0xbc, 0x03);
		ui->label_pretrig_unit->setText("[" + mu + "s]");
		ui->label_reclen_unit->setText("[" + mu + "s]");
	}
	else {
		ui->label_pretrig_unit->setText("[samples]");
		ui->label_reclen_unit->setText("[samples]");
	}

	ui->label_thr_unit->setText("[samples]");//for common config use samples	
}

void CommonConfigPanel1::createMaps() {
	uint64_t phandle=0;
	CAENparameter *new_param=nullptr;

	for (int p = 0; p < mDevParamsList.size(); p++) {
		phandle = 0;
		for (int b = 0; b < mNumBrds; b++) {
			CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevs.at(b)->getHandle(), qPrintable(mDevParamsList.at(p)), &phandle);
			if (err != CAEN_FELib_Success) {
				mDevWidgetsList.at(p)->setEnabled(false);
				break;
			}
		}
		mDevGUIMap.insert(mDevWidgetsList.at(p), mDevParamsList.at(p));
	}

	for (int p= 0; p < mChParamsList.size(); p++) {
		for (int b = 0; b < mNumBrds; b++) {
			CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
			for (int ch = 0; ch < mNumberOfChannels.at(b); ch++) {
				phandle = 0;
				CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevs.at(b)->getHandle(), qPrintable(mChParamsList.at(p).arg(ch)), &phandle);
				if (err != CAEN_FELib_Success) {
					mChWidgetsList.at(p)->setEnabled(false);
					break;;
				}
			}
			mChGUIMap.insert(mChWidgetsList.at(p), mChParamsList.at(p));
		}
	}

	for (int p = 0; p < mNumberOfGroupParams; p++) {
		mGroupGUIMap.insert(mGroupWidgetsList.at(p), mGroupParamsList.at(p));
	}
}

void CommonConfigPanel1::defineScaleParams() {
	mDevRelatedParamsList.clear();
	for (int p = 0; p < mDevParamsList.size(); p++) {
		if (mDevParamsList.at(p) == "/par/RecLen" && mDecimationVisible)
			mDevRelatedParamsList << "/par/Decimation_Factor";
		else
			mDevRelatedParamsList << "";
	}
	mChRelatedParamsList.clear();
	for (int p = 0; p < mChParamsList.size(); p++) {
		if (mChParamsList.at(p) == "/ch/%1/par/TriggerThr")
			mChRelatedParamsList << "/ch/%1/par/Ch_DCOffset";
		else
			mChRelatedParamsList << "";
	}
	mGroupRelatedParamsList.clear();
	for (int p = 0; p < mGroupParamsList.size(); p++) {
		if (mGroupParamsList.at(p) == "/ch/%1/par/TriggerThr")
			mGroupRelatedParamsList << "/ch/%1/par/Ch_DCOffset";
		else
			mGroupRelatedParamsList << "";
	}
}

double CommonConfigPanel1::getScaleParamValue(int b, QString qry) {
	char val[256] = { 0 };
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	uint64_t phandle;
	if (qry.contains("decimation", Qt::CaseInsensitive) && !mDevWithDecim.at(b))
		return 1.0;
	
	return mDevs.at(b)->getDecimation();
}

void CommonConfigPanel1::UpdateWidget(QWidget* widget, ParamInfo info, QString value) {
	if (widget == nullptr)
		return;
	double UoM_to_GUIUoM = 1.;

	if (QComboBox* cbox = qobject_cast<QComboBox*>(widget)) {
		cbox->clear();
		int index = -1;
		for (int i = 0; i < info.allowed_values.size(); i++) {
			cbox->addItem(info.allowed_values.at(i));
			if (value == info.allowed_values.at(i))
				index = (int)i;
		}
		cbox->setCurrentIndex(index);
	}

	else if (QSpinBox* sbox = qobject_cast<QSpinBox*>(widget)) {
		sbox->setMinimum(info.min.toInt() * UoM_to_GUIUoM);
		sbox->setMaximum(info.max.toInt() * UoM_to_GUIUoM);
		sbox->setSingleStep(info.incr.toInt() * UoM_to_GUIUoM);
		sbox->setValue(qRound(value.toDouble() * UoM_to_GUIUoM));
	}
	else if (QDoubleSpinBox* dbox = qobject_cast<QDoubleSpinBox*>(widget)) {
		dbox->setMinimum(info.min.toDouble() * UoM_to_GUIUoM);
		dbox->setMaximum(info.max.toDouble() * UoM_to_GUIUoM);
		dbox->setSingleStep(info.incr.toDouble() * UoM_to_GUIUoM);
		int i = info.incr.lastIndexOf('.');
		int decimals = (i > 0) ? 2 : 0;
		dbox->setDecimals(decimals);
		dbox->setValue(value.toDouble() * UoM_to_GUIUoM);
	}
	else if (QLineEdit* ledit = qobject_cast<QLineEdit*>(widget))
		ledit->setText(value);
	else if (QListWidget* lwidget = qobject_cast<QListWidget*>(widget)) {
		lwidget->clear();
		for (int val = 0; val < info.allowed_values.size(); val++) {
			QListWidgetItem* newItem = new QListWidgetItem;
			newItem->setText(info.allowed_values.at(val));
			newItem->setFlags(newItem->flags() | Qt::ItemIsUserCheckable);
			newItem->setTextAlignment(Qt::AlignCenter);
			if (value.contains(info.allowed_values.at(val), Qt::CaseInsensitive))
				newItem->setCheckState(Qt::Checked);
			else
				newItem->setCheckState(Qt::Unchecked);
			lwidget->insertItem(val, newItem);
		}
		lwidget->setMaximumWidth(lwidget->sizeHintForColumn(0) + 5 * lwidget->frameWidth());

	}

	widget->setStyleSheet("color: black");
}


void CommonConfigPanel1::UpdateDevGUI() {
	mIgnoreEvent = true;
	QHash<QWidget*, QString>::iterator item;
	for (item = mDevGUIMap.begin(); item != mDevGUIMap.end(); ++item) {
		if (item.key()->isEnabled()) {
			int index = mDevWidgetsList.indexOf(item.key());
			QString test = item.value();
			QString pvalue = mWaveDump->CommonDevValuesmap1.value(item.value());
			if (mDevRelatedParamsList.at(index) != "") {
				if (mUoMX == UOM_PHYS_UNIT && mWwithUoMX.contains(item.key())) {
					double scaled_value = getScaleParamValue(0, mDevRelatedParamsList.at(index)) * pvalue.toDouble();
					pvalue = QString("%1").arg(scaled_value);
				}
			}
			UpdateWidget(item.key(), mDevParamsInfo.at(index), pvalue);//update value depending on the widget type				
		}

	}
	if (mUoMX == UOM_PHYS_UNIT) {
		ui->dBox_reclen->setMinimum(0.);
		ui->dBox_reclen->setValue(ui->dBox_reclen->value() / 1000.);
		ui->dBox_pretrg->setValue(ui->dBox_pretrg->value() / 1000.);
		ui->dBox_reclen->setDecimals(2);
		ui->dBox_pretrg->setMinimum(0.);
		ui->dBox_pretrg->setDecimals(3);
	}
	else {
		ui->dBox_reclen->setValue(ui->dBox_reclen->value() / (mWaveDump->getSampleToS() * 1e9));
		ui->dBox_pretrg->setValue(ui->dBox_pretrg->value() / (mWaveDump->getSampleToS() * 1e9));
		
		ui->dBox_reclen->setDecimals(0);
		ui->dBox_pretrg->setDecimals(0);
	}
	mIgnoreEvent = false;
}

void CommonConfigPanel1::UpdateChGUI() {
	mIgnoreEvent = true;
	QHash<QWidget*, QString>::iterator item;
	for (item = mChGUIMap.begin(); item != mChGUIMap.end(); ++item) {
		if (item.key()->isEnabled()) {
			int index = mChWidgetsList.indexOf(item.key());
			UpdateWidget(item.key(), mChParamsInfo.at(index), mWaveDump->CommonChValuesmap1[mFirstSelectedCh].value(item.value().arg(mFirstSelectedCh)));//update value depending on the widget type	
		}
	}

	UpdateThr();
	UpdateEnable();
	UpdateDCO();
	mIgnoreEvent = false;
}

void CommonConfigPanel1::UpdateGroupGUI() {
	mIgnoreEvent = true;
	QHash<QWidget*, QString>::iterator item;
	for (item = mGroupGUIMap.begin(); item != mGroupGUIMap.end(); ++item) {
		if (item.key()->isEnabled()) {
			int index = mGroupWidgetsList.indexOf(item.key());
			UpdateWidget(item.key(), mGroupParamsInfo.at(index), mWaveDump->CommonGroupValuesmap1[mFirstSelectedGroup.at(index)].value(item.value().arg(mFirstSelectedGroup.at(index))));//update value depending on the widget type	
		}
	}
	mIgnoreEvent = false;
}

//multi sel. active --> check if the selected channels have the same param values
void CommonConfigPanel1::CheckChParamValues() {
	for (int p = 0; p < mChParamsList.size(); p++) {
		if (!mChWidgetsList.at(p)->isEnabled())
			continue;
		QString first_sel_val = mWaveDump->CommonChValuesmap1[mFirstSelectedCh].value(mChParamsList.at(p).arg(mFirstSelectedCh));
		bool same_value = true;
		for (int b = 0; b < mNumBrds; b++) {
			for (int i = 0; i < mNumberOfChannels.at(b); i++) {
				if (!mPbCh[i]->isChecked() || (i == mFirstSelectedCh))
					continue;

				if (mWaveDump->CommonChValuesmap1[i].value(mChParamsList.at(p).arg(i)) != first_sel_val) {
					same_value = false;
					break;
				}
			}
			if (!same_value)
				break;
		}

		if (!same_value) {
			WarningWidget(mChWidgetsList.at(p), false);
		}
	}

	NormalWidget(ui->cBox_enable,false);
	for (int b = 0; b < mNumBrds; b++) {
		for (int i = 0; i < mNumberOfChannels.at(b); i++) {
			if (!mPbCh[i]->isChecked() || (i == mFirstSelectedCh))
				continue;
			if (mWaveDump->CommonChValuesmap1[i].value(QString("/ch/%1/par/Ch_Enabled").arg(i)) != mWaveDump->CommonChValuesmap1[mFirstSelectedCh].value(QString("/ch/%1/par/Ch_Enabled").arg(mFirstSelectedCh))) {
				WarningWidget(ui->cBox_enable, false);
				return;
			}
		}
	}
}

//multi sel. active --> check if the selected groups have the same param values
void CommonConfigPanel1::CheckGroupParamValues() {
	for (int p = 0; p < mGroupParamsList.size(); p++) {
		if (!mGroupWidgetsList.at(p)->isEnabled())
			continue;
		QString first_sel_val = mWaveDump->CommonGroupValuesmap1[mFirstSelectedGroup.at(p)].value(mGroupParamsList.at(p).arg(mFirstSelectedGroup.at(p)));
		bool same_value = true;
		for (int b = 0; b < mNumBrds; b++) {
			for (int i = 0; i < (mNumberOfChannels.at(b) / mGroupsize.at(p + b*mGroupParamsList.size())); i++) {//check this index!!
				//bool some_channels_selected = false;
				if (!mPbCh[i]->isChecked() || (i == mFirstSelectedGroup.at(p)))
					continue;
				//for (int ch = 0; ch < mGroupsize.at(p); ch++)
				//	if (mPbCh[(i * mGroupsize.at(p)) + ch]->isChecked()) {
				//		some_channels_selected = true;
				//		break;
				//	}
				//if (!some_channels_selected)
				//	continue;
				if (mWaveDump->CommonGroupValuesmap1[i].value(mGroupParamsList.at(p).arg(i)) != first_sel_val) {
					same_value = false;
					break;
				}
			}
			if (!same_value)
				break;
		}

		if (!same_value) {
			WarningWidget(mGroupWidgetsList.at(p), false);
		}
	}
}

void CommonConfigPanel1::channelClicked(CAENPushButton *pb) {
	int ch = pb->getIndex();

	//single channel selected
	if (!mMulti) {
		for (int i = 0; i < 64; i++) {
			if (i != ch)
				mPbCh[i]->setChecked(false);
		}
		if (ch != mFirstSelectedCh) {
			mFirstSelectedCh = ch;
			UpdateChGUI();
		}
		else
			mPbCh[ch]->setChecked(true);
		for (int p = 0; p < mNumberOfGroupParams; p++)
			mFirstSelectedGroup.replace(p, (ch / mGroupsize.at(p)));
		UpdateGroupGUI();
	}
	else {//multi sel. active
		if (ch == mFirstSelectedCh) {
			for (int i = 0; i < 64; i++) {
				if (mPbCh[i]->isChecked()) {
					mFirstSelectedCh = i;
					UpdateChGUI();
					break;
				}
			}
			mPbCh[mFirstSelectedCh]->setChecked(true);
		}
		for (int p = 0; p < mNumberOfGroupParams; p++)
			mFirstSelectedGroup.replace(p, (ch / mGroupsize.at(p)));
		UpdateGroupGUI();
		CheckChParamValues();
		CheckGroupParamValues();
	}
}
void CommonConfigPanel1::selectAll() {
	for (int i = 0; i < 64; i++)
		mPbCh[i]->setChecked(true);
	ui->pbMulti->setChecked(true);
	CheckChParamValues();
	CheckGroupParamValues();
}

void  CommonConfigPanel1::MultiClicked() {
	mMulti = ui->pbMulti->isChecked();
	if (!mMulti) {
		QHash<QWidget *, QString>::iterator item;
		for (item = mChGUIMap.begin(); item != mChGUIMap.end(); ++item) {
			NormalWidget(item.key(), false);
		}
		int memCh = mFirstSelectedCh;
		mFirstSelectedCh = -1;
		emit channelClicked(mPbCh[memCh]);
	}
}

void CommonConfigPanel1::WarningWidget(QWidget *widget, bool bold) {
	if (!widget->styleSheet().contains("color: black"))
		return;
	if(bold)
		widget->setStyleSheet("font-weight: bold; color: red");
	else
		widget->setStyleSheet("color: red");
	mIgnoreEvent = true;
	if (QComboBox *cbox = qobject_cast<QComboBox *>(widget)) {
		cbox->setCurrentIndex(-1);
		mIgnoreEvent = false;
		return;
	}
	if(QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(widget)){
		doubleSpinBox->setSpecialValueText("?");
		doubleSpinBox->setMinimum(doubleSpinBox->minimum() - 1);
		doubleSpinBox->setValue(doubleSpinBox->minimum());
		mIgnoreEvent = false;
		return;
	}
	if(QSpinBox *SpinBox = qobject_cast<QSpinBox *>(widget)){
		SpinBox->setSpecialValueText("?");
		SpinBox->setMinimum(SpinBox->minimum() - 1);
		SpinBox->setValue(SpinBox->minimum());
		mIgnoreEvent = false;
		return;
	}
	mIgnoreEvent = false;
}

void CommonConfigPanel1::NormalWidget(QWidget *widget, bool bold) {
	if (!widget->styleSheet().contains("color: red"))
		return;
	if(bold)
		widget->setStyleSheet("font-weight: bold; color: black");
	else
		widget->setStyleSheet("color: black");
	mIgnoreEvent = true;
	QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(widget);
	if (doubleSpinBox != nullptr) {
		doubleSpinBox->setSpecialValueText("");
		doubleSpinBox->setMinimum(doubleSpinBox->minimum() + 1);
		mIgnoreEvent = false;
		return;
	}
	QSpinBox *SpinBox = qobject_cast<QSpinBox *>(widget);
	if (SpinBox != nullptr) {
		SpinBox->setSpecialValueText("");
		SpinBox->setMinimum(SpinBox->minimum() + 1);
		mIgnoreEvent = false;
		return;
	}
	mIgnoreEvent = false;
	
}

void CommonConfigPanel1::ApplyNewValue(QWidget *widget) {
		if (mDevGUIMap.contains(widget)) {
			int ok = SetDevParamValueFromWidget(widget, mDevGUIMap.value(widget));		
			mWaveDump->add2LogAndConf("SET", mDevName, mDevGUIMap.value(widget), mWaveDump->CommonDevValuesmap1.value(mDevGUIMap.value(widget)), mWaveDump->CommonDevValuesmap1.value(mDevGUIMap.value(widget)), CAEN_FELib_ErrorCode(ok));
		}
		else if (mChGUIMap.contains(widget)) {
			if (!ui->pbMulti->isChecked() && mChGUIMap.value(widget) != nullptr) {//single channel selected
				int ok = SetDevParamValueFromWidget(widget, mChGUIMap.value(widget).arg(mFirstSelectedCh));
				mWaveDump->add2LogAndConf("SET", mDevName, mChGUIMap.value(widget).arg(mFirstSelectedCh), mWaveDump->CommonChValuesmap1[mFirstSelectedCh].value(mChGUIMap.value(widget).arg(mFirstSelectedCh)), mWaveDump->CommonChValuesmap1[mFirstSelectedCh].value(mChGUIMap.value(widget).arg(mFirstSelectedCh)), CAEN_FELib_ErrorCode(ok));
			}
			else {//multi channel select
				for (int ch = 0; ch < 64; ch++) {
					if (!mPbCh[ch]->isChecked())
						continue;
					int ok = SetDevParamValueFromWidget(widget, mChGUIMap.value(widget).arg(ch));
					mWaveDump->add2LogAndConf("SET", mDevName, mChGUIMap.value(widget).arg(ch), mWaveDump->CommonChValuesmap1[ch].value(mChGUIMap.value(widget).arg(ch)), mWaveDump->CommonChValuesmap1[ch].value(mChGUIMap.value(widget).arg(ch)), CAEN_FELib_ErrorCode(ok));
				}
				NormalWidget(widget, false);
			}
		}
		else if (mGroupGUIMap.contains(widget)) {
			if (!ui->pbMulti->isChecked() && mGroupGUIMap.value(widget) != nullptr) {//single channel selected
				int index = mGroupWidgetsList.indexOf(widget);
				int ok = SetDevParamValueFromWidget(widget, mGroupGUIMap.value(widget).arg(mFirstSelectedGroup.at(index)));
				mWaveDump->add2LogAndConf("SET", mDevName, mGroupGUIMap.value(widget).arg(mFirstSelectedGroup.at(index)), mWaveDump->CommonGroupValuesmap1[mFirstSelectedGroup.at(index)].value(mGroupGUIMap.value(widget).arg(mFirstSelectedGroup.at(index))), mWaveDump->CommonGroupValuesmap1[mFirstSelectedGroup.at(index)].value(mGroupGUIMap.value(widget).arg(mFirstSelectedGroup.at(index))), CAEN_FELib_ErrorCode(ok));
			}
			else {//multi channel select
				int index = mGroupWidgetsList.indexOf(widget);
				for (int g = 0; g < 64 / mGroupsize.at(index); g++) {
					if (!mPbCh[g * mGroupsize.at(index)]->isChecked())
						continue;
					for (int b = 0; b < mNumBrds; b++) {
						if (!mDevWithGroupedChannels.at(b))
							continue;
						int ok = SetDevParamValueFromWidget(widget, mGroupGUIMap.value(widget).arg(g));
						mWaveDump->add2LogAndConf("SET", mDevName, mGroupGUIMap.value(widget).arg(g), mWaveDump->CommonGroupValuesmap1[g].value(mGroupGUIMap.value(widget).arg(g)), mWaveDump->CommonGroupValuesmap1[g].value(mGroupGUIMap.value(widget).arg(g)), CAEN_FELib_ErrorCode(ok));
					}
				}
				NormalWidget(widget, false);
			}
		}
}

QString CommonConfigPanel1::GetValueToSetFromWidget(QWidget* widget, bool* check_uom) {
	*check_uom = false;
	QString value = "";
	if (QComboBox* cbox = qobject_cast<QComboBox*>(widget)) {
		value = cbox->currentText();
	}
	else if (QSpinBox* sBox = qobject_cast<QSpinBox*>(widget)) {
		value = QString("%1").arg(qRound(static_cast<double>(sBox->value())));
		*check_uom = true;
	}
	else if (QDoubleSpinBox* dbox = qobject_cast<QDoubleSpinBox*>(widget)) {
		value = QString("%1").arg(dbox->value());
		*check_uom = true;
	}
	else if (QLineEdit* lineEdit = qobject_cast<QLineEdit*>(widget)) {
		if (mDevGUIMap.value(widget).contains("mask", Qt::CaseInsensitive))
			value = "0x" + lineEdit->text();
		else
			value = lineEdit->text();
	}
	else if (QGroupBox* gBox = qobject_cast<QGroupBox*>(widget)) {
		if (gBox->isCheckable())
			value = gBox->isChecked() ? "TRUE" : "FALSE";
		else
			value = gBox->isEnabled() ? "TRUE" : "FALSE";
	}
	else if (QCheckBox* cBox = qobject_cast<QCheckBox*>(widget)) {
			value = cBox->isChecked() ? "TRUE" : "FALSE";
	}
	else if (QListWidget* lwidget = qobject_cast<QListWidget*>(widget)) {
		bool check_ok = false;
		for (int i = 0; i < lwidget->count(); ++i) {
			if (lwidget->item(i)->checkState() == Qt::Checked) {
				check_ok = true;
				if (value == "")
					value.append(lwidget->item(i)->text());
				else
					value.append("|" + lwidget->item(i)->text());
			}
		}

		QString current_value;
		if (mDevGUIMap.contains(widget))
			current_value = mWaveDump->CommonDevValuesmap.value(mDevGUIMap.value(widget));
		else
			current_value = mWaveDump->CommonChValuesmap[mFirstSelectedCh].value(mChGUIMap.value(widget));
		if (check_ok) {
			if (value.contains("Disabled", Qt::CaseInsensitive) && !current_value.contains("Disabled")) {
				value = "Disabled";
				for (int i = 0; i < lwidget->count(); ++i) {
					if (lwidget->item(i)->checkState() == Qt::Checked && lwidget->item(i)->text() != "Disabled") {
						lwidget->blockSignals(true);
						lwidget->item(i)->setCheckState(Qt::Unchecked);
						lwidget->blockSignals(false);
					}
				}
			}
			if (value.contains("Disabled", Qt::CaseInsensitive) && value != "Disabled") {
				value.replace("Disabled|", "");
				for (int i = 0; i < lwidget->count(); ++i) {
					if (lwidget->item(i)->checkState() == Qt::Checked && lwidget->item(i)->text() == "Disabled") {
						lwidget->blockSignals(true);
						lwidget->item(i)->setCheckState(Qt::Unchecked);
						lwidget->blockSignals(false);
						break;
					}
				}
			}
		}
		else {
			QVector<ParamInfo>* info;
			int index = mDevWidgetsList.indexOf(widget);
			if (index != -1)
				info = &mDevParamsInfo;
			else {
				index = mChWidgetsList.indexOf(widget);
				info = &mChParamsInfo;
				if (index == -1) {
					index = mGroupWidgetsList.indexOf(widget);
					info = &mGroupParamsInfo;
				}
			}

			if (multipleValue[index] && current_value.contains("Disabled", Qt::CaseInsensitive))
				value = "Disabled";
		}
	}
	return value;
}

int CommonConfigPanel1::SetDevParamValueFromWidget(QWidget* widget, QString qry) {
	double UoM_to_GUIUoM = 1.0;
	bool check_UoM;
	int err = 0;
	QString related_param = "";

	QString value = GetValueToSetFromWidget(widget, &check_UoM);
	if (check_UoM && mUoMX == UOM_PHYS_UNIT) {
		if (mWwithUoMX.contains(widget)) {
			double val = value.toDouble();
			val = val * 1000.;
			value = QString("%1").arg(val);
		}
		const auto it = mDevGUIMap.find(widget);
		const auto index = std::distance(mDevGUIMap.begin(), it);
		if (index < mDevRelatedParamsList.size()) { //if >= size the widget has not been found
			if (mDevRelatedParamsList.at(index) != "") {  //check if this param has a related param
				related_param = mDevRelatedParamsList.at(index);
			}
		}
	}
	else if (check_UoM && mUoMX == UOM_SAMPLE) {
		double value_ns = value.toDouble() * (mWaveDump->getSampleToS() * 1e9);
		value = QString("%1").arg(value_ns);
	}
	if (check_UoM && mUoMY == UOM_PHYS_UNIT) {
		if (mWwithUoMY.contains(widget)) {
			double val = value.toDouble();
			val = val / (mWaveDump->getSampleToV() * 1e3);
			value = QString("%1").arg(val);
		}
	}

	int ch = -1;
	if (qry.contains("/ch/")) {
		QStringList l = qry.split("/ch/");
		QStringList l1 = l.at(1).split("/");
		ch = l1.at(0).toInt();
	}
	else if (qry.contains("/group/")) {
		QStringList l = qry.split("/group/");
		QStringList l1 = l.at(1).split("/");
		ch = l1.at(0).toInt();
	}

	QString set_val;
	if (err = CommonSetParam(qry, value, ch, related_param, &set_val) != 0)
		return err;

	bool found = false;
	QHash<QString, QString>::iterator item;
	for (item = mWaveDump->CommonDevValuesmap1.begin(); item != mWaveDump->CommonDevValuesmap1.end(); ++item) {
		if (item.key() == qry) {
			mWaveDump->CommonDevValuesmap1.insert(item.key(), set_val);
			found = true;
			break;
		}
	}
	if (!found) {
		QHash<QString, QString>::iterator item;
		for (item = mWaveDump->CommonChValuesmap1[ch].begin(); item != mWaveDump->CommonChValuesmap1[ch].end(); ++item) {
			if (item.key() == qry) {
				mWaveDump->CommonChValuesmap1[ch].insert(item.key(), value);
				found = true;
				break;
			}
		}
	}
	if (!found) {
		QHash<QString, QString>::iterator item;
		for (item = mWaveDump->CommonGroupValuesmap1[ch].begin(); item != mWaveDump->CommonGroupValuesmap1[ch].end(); ++item) {
			if (item.key() == qry) {
				mWaveDump->CommonGroupValuesmap1[ch].insert(item.key(), value);
				found = true;
				break;
			}
		}
	}

	return 0;
}

int CommonConfigPanel1::CommonSetParam(QString qry, QString value, int ch, QString rpar, QString* set_value) {
	for (int b = 0; b < mNumBrds; b++) {
		if (ch > (mNumberOfChannels[b] - 1) && ch != -1)
			continue;
		if (qry.contains("group", Qt::CaseInsensitive) && !mDevWithGroupedChannels.at(b))
			continue;
		if (qry.contains("chgain", Qt::CaseInsensitive) && !mDevWithCHGain.at(b))
			continue;
		if (qry.contains("decimation", Qt::CaseInsensitive) && !mDevWithDecim.at(b))
			continue;

		QString val = value;
		double factor, scaled_val;
		if (rpar != "") {
			factor = getScaleParamValue(b, rpar);
			scaled_val = val.toDouble() / factor;
			val = QString("%1").arg(scaled_val);
		}

		CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevs.at(b)->getHandle(), qPrintable(qry), qPrintable(val));
		if (err != CAEN_FELib_Success)
			return err;

		*set_value = val;
	}
	return 0;
}


void CommonConfigPanel1::intChanged(int val) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel1::boolChanged(bool val) {
	if (mIgnoreEvent)
		return;
	QWidget* widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel1::doubleChanged(double val) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
	if (widget == ui->dBox_reclen)//update post trigger samples so that pretrg in ns remais the same
		PreTrgChanged(ui->dBox_pretrg->value());
}

void CommonConfigPanel1::itemCheckedChanged(QListWidgetItem *item) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel1::TrgSourceChanged(QListWidgetItem* item) {
	if (mIgnoreEvent)
		return;
	QString val = "";
	bool check_ok = false;
	for (int i = 0; i < ui->lWidget_Trg_Src->count(); ++i) {
		if (ui->lWidget_Trg_Src->item(i)->checkState() == Qt::Checked) {
			check_ok = true;
			if (val == "")
				val.append(ui->lWidget_Trg_Src->item(i)->text());
			else
				val.append("|" + ui->lWidget_Trg_Src->item(i)->text());
		}
	}
	try {
		for (int b = 0; b < mNumBrds; b++) {
			if (check_ok) {
				if (val.contains("Disabled")) {
					mDevs.at(b)->setTrgSource("Disabled");
					mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Ext_Enable", "FALSE");
					mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Sw_Enable", "FALSE");
					for (int ch = 0; ch < 64; ch++) {
						mWaveDump->CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch), "FALSE");
					}
					for (int g = 0; g < 32; g++) {
						mWaveDump->CommonGroupValuesmap1[g].insert(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), "FALSE");
					}
				}
				else {
					mDevs.at(b)->setTrgSource(val);
					mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Ext_Enable", val.contains("TrgIn",Qt::CaseInsensitive)? "TRUE" : "FALSE");
					mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Sw_Enable", val.contains("Sw", Qt::CaseInsensitive) ? "TRUE" : "FALSE");
				}
			}
			else {
				mDevs.at(b)->setTrgSource("Disabled");
				mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Ext_Enable", "FALSE");
				mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Sw_Enable", "FALSE");
				for (int ch = 0; ch < 64; ch++)
					mWaveDump->CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch), "FALSE");
				for (int g = 0; g < 32; g++)
					mWaveDump->CommonGroupValuesmap1[g].insert(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), "FALSE");
			}
		}
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}

	mWaveDump->add2LogAndConf("SET", mDevName, "/par/Trg_Ext_Enable", mWaveDump->CommonDevValuesmap1.value("/par/Trg_Ext_Enable"), mWaveDump->CommonDevValuesmap1.value("/par/Trg_Ext_Enable"), CAEN_FELib_ErrorCode(0));
	mWaveDump->add2LogAndConf("SET", mDevName, "/par/Trg_Sw_Enable", mWaveDump->CommonDevValuesmap1.value("/par/Trg_Sw_Enable"), mWaveDump->CommonDevValuesmap1.value("/par/Trg_Sw_Enable"), CAEN_FELib_ErrorCode(0));
	for (int ch = 0; ch < 64; ch++) 
		mWaveDump->add2LogAndConf("SET", mDevName, QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch)), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch)), CAEN_FELib_ErrorCode(0));
	for (int g = 0; g < 32; g++) 
		mWaveDump->add2LogAndConf("SET", mDevName, QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), mWaveDump->CommonGroupValuesmap1[g].value(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g)), mWaveDump->CommonGroupValuesmap1[g].value(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g)), CAEN_FELib_ErrorCode(0));
}

void CommonConfigPanel1::ClockSourceChanged(int) {
	if (mIgnoreEvent)
		return;
	try{
		for (int b = 0; b < mNumBrds; b++) {
			if (ui->cBox_clk_source->currentText() == "Internal") {
				mDevs.at(b)->setExtClkSource("False");
				mWaveDump->CommonDevValuesmap1.insert("/par/DT_Ext_Clock", "FALSE");
			}
			else {
				mDevs.at(b)->setExtClkSource("True");
				mWaveDump->CommonDevValuesmap1.insert("/par/DT_Ext_Clock", "FALSE");
			}
		}
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}
	mWaveDump->add2LogAndConf("SET", mDevName, "/par/DT_Ext_Clock", mWaveDump->CommonDevValuesmap1.value("/par/DT_Ext_Clock"), mWaveDump->CommonDevValuesmap1.value("/par/DT_Ext_Clock"), CAEN_FELib_ErrorCode(0));
}

void CommonConfigPanel1::EnableChanged(int index) {
	for (int ch = 0; ch < 64; ch++) {
		if (!mPbCh[ch]->isChecked())
			continue;
		for (int b = 0; b < mNumBrds; b++){
			if (ui->cBox_enable->currentText() == "TRUE") {
				mDevs.at(b)->setChEnabled(ch, true);
				mWaveDump->CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Enabled").arg(ch), "TRUE");
				mWaveDump->add2LogAndConf("SET", mDevName, QString("/ch/%1/par/Ch_Enabled").arg(ch), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Enabled").arg(ch)), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Enabled").arg(ch)), CAEN_FELib_ErrorCode(0));
				if (mMinGroupSize != 1) {
					mWaveDump->CommonGroupValuesmap1[ch].insert(QString("/group/%1/par/Gr_Enabled").arg(ch / mMinGroupSize), "TRUE");
					mWaveDump->add2LogAndConf("SET", mDevName, QString("/group/%1/par/Gr_Enabled").arg(ch / mMinGroupSize), mWaveDump->CommonGroupValuesmap1[ch].value(QString("/group/%1/par/Gr_Enabled").arg(ch / mMinGroupSize)), mWaveDump->CommonGroupValuesmap1[ch].value(QString("/group/%1/par/Gr_Enabled").arg(ch / mMinGroupSize)), CAEN_FELib_ErrorCode(0));
				}
			}
			else {
				mDevs.at(b)->setChEnabled(ch, false);
				mWaveDump->CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Enabled").arg(ch), "FALSE");
				mWaveDump->add2LogAndConf("SET", mDevName, QString("/ch/%1/par/Ch_Enabled").arg(ch), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Enabled").arg(ch)), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Enabled").arg(ch)), CAEN_FELib_ErrorCode(0));
				if (mMinGroupSize != 1) {
					mWaveDump->CommonGroupValuesmap1[ch].insert(QString("/group/%1/par/Gr_Enabled").arg(ch / mMinGroupSize), "FALSE");
					mWaveDump->add2LogAndConf("SET", mDevName, QString("/group/%1/par/Gr_Enabled").arg(ch / mMinGroupSize), mWaveDump->CommonGroupValuesmap1[ch].value(QString("/group/%1/par/Gr_Enabled").arg(ch / mMinGroupSize)), mWaveDump->CommonGroupValuesmap1[ch].value(QString("/group/%1/par/Gr_Enabled").arg(ch / mMinGroupSize)), CAEN_FELib_ErrorCode(0));
				}
			}
		}
	}
}

void CommonConfigPanel1::UpdateThr() {
	ui->dBox_thr->blockSignals(true);
	QString vals = mWaveDump->CommonChValuesmap1[mFirstSelectedCh].value(QString("/ch/%1/par/Ch_Threshold").arg(mFirstSelectedCh));
	ui->dBox_thr->setValue(vals.toDouble());
	ui->dBox_thr->blockSignals(false);
}

void CommonConfigPanel1::ThrChanged(double val) {
	if (mIgnoreEvent)
		return;
	for (int ch = 0; ch < 64; ch++) {
		if (!mPbCh[ch]->isChecked())
			continue;
		for (int b = 0; b < mNumBrds; b++) {
			try{
				mDevs.at(b)->setChThreshold(ch, val);
			}
			catch (CAENFELibException& exc) {
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}	
			mWaveDump->CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Threshold").arg(ch), QString("%1").arg(val));
			mWaveDump->add2LogAndConf("SET", mDevName, QString("/ch/%1/par/Ch_Threshold").arg(ch), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Threshold").arg(ch)), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Threshold").arg(ch)), CAEN_FELib_ErrorCode(0));
			if (mMinGroupSize != 1) {
				mWaveDump->CommonGroupValuesmap1[ch].insert(QString("/group/%1/par/Gr_Threshold").arg(ch / mMinGroupSize), "TRUE");
				mWaveDump->add2LogAndConf("SET", mDevName, QString("/group/%1/par/Gr_Threshold").arg(ch / mMinGroupSize), mWaveDump->CommonGroupValuesmap1[ch].value(QString("/group/%1/par/Gr_Threshold").arg(ch / mMinGroupSize)), mWaveDump->CommonGroupValuesmap1[ch].value(QString("/group/%1/par/Gr_Threshold").arg(ch / mMinGroupSize)), CAEN_FELib_ErrorCode(0));
			}
		}

	}
}

void CommonConfigPanel1::DCOChanged(double val) {
	if (mIgnoreEvent)
		return;
	for (int ch = 0; ch < 64; ch++) {
		if (!mPbCh[ch]->isChecked())
			continue;
		for (int b = 0; b < mNumBrds; b++) {
			try {
				mDevs.at(b)->setChOffset(ch, val);
			}
			catch (CAENFELibException& exc) {
				mWaveDump->addLog(exc.ErrorMessage(), false);
			}
			mWaveDump->CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_DCOffset").arg(ch), QString("%1").arg(val));
			mWaveDump->add2LogAndConf("SET", mDevName, QString("/ch/%1/par/Ch_DCOffset").arg(ch), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_DCOffset").arg(ch)), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_DCOffset").arg(ch)), CAEN_FELib_ErrorCode(0));
			if (mMinGroupSize != 1) {
				mWaveDump->CommonGroupValuesmap1[ch].insert(QString("/group/%1/par/Gr_DCOffset").arg(ch / mMinGroupSize), "TRUE");
				mWaveDump->add2LogAndConf("SET", mDevName, QString("/group/%1/par/Gr_DCOffset").arg(ch / mMinGroupSize), mWaveDump->CommonGroupValuesmap1[ch].value(QString("/group/%1/par/Gr_DCOffset").arg(ch / mMinGroupSize)), mWaveDump->CommonGroupValuesmap1[ch].value(QString("/group/%1/par/Gr_DCOffset").arg(ch / mMinGroupSize)), CAEN_FELib_ErrorCode(0));
			}
		}

	}
}

void CommonConfigPanel1::UpdateDCO() {
	ui->dSBox_bsl_level->blockSignals(true);
	QString vals = mWaveDump->CommonChValuesmap1[mFirstSelectedCh].value(QString("/ch/%1/par/Ch_DCOffset").arg(mFirstSelectedCh));
	ui->dSBox_bsl_level->setValue(vals.toDouble());
	ui->dSBox_bsl_level->blockSignals(false);
}

void CommonConfigPanel1::PreTrgChanged(double value) {
	if (mIgnoreEvent)
		return;
	double pretrg_value_ns;
	if (mUoMX == UOM_PHYS_UNIT)
		pretrg_value_ns = value * 1000.;
	else
		pretrg_value_ns = value * (mWaveDump->getSampleToS() * 1e9);
	try {
		for(int b=0;b<mNumBrds;b++)
			mDevs.at(b)->setPreTrg(QString("%1").arg(pretrg_value_ns));
		ui->dBox_pretrg->blockSignals(true);
		if (mUoMX == UOM_PHYS_UNIT)
			ui->dBox_pretrg->setValue(mDevs.at(0)->getPreTrgCachedValue()/1000.);
		else
			ui->dBox_pretrg->setValue(mDevs.at(0)->getPreTrgCachedValue() / (mWaveDump->getSampleToS() * 1e9));
		mWaveDump->addLog(QString("SET PreTrigger to %1 us").arg(ui->dBox_pretrg->value()), true);
		mWaveDump->CommonDevValuesmap1.insert("/par/PostTrg", mDevs.at(0)->getPostTrgParam()->getCachedValue());
		ui->dBox_pretrg->blockSignals(false);
	}
	catch (CAENFELibException& exc) {
		ui->dBox_pretrg->setEnabled(false);
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}
	mWaveDump->add2LogAndConf("SET", mDevName, "/par/PostTrg", mWaveDump->CommonDevValuesmap1.value("/par/PostTrg"), mWaveDump->CommonDevValuesmap1.value("/par/PostTrg"), CAEN_FELib_ErrorCode(0));
}

void CommonConfigPanel1::SetMask() {
	QString current_mask = mWaveDump->getCommonSelfTrgMask();
	uint64_t val = current_mask.toULongLong();
	chmaskDialog *dlg = new chmaskDialog(this, mWaveDump->getNumChannels(), val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask;
		new_mask.setNum(dlg->getMask());
		ui->lineEdit_trig_mask->setText(QString("%1").arg(dlg->getMask(), 0, 16, QLatin1Char('0')));
		try {
			for(int b=0;b<mNumBrds;b++)
				mDevs.at(b)->setSelfTrgMask(new_mask);
			mWaveDump->setCommonMask(new_mask, "Trg_Global_Gen");
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
		}
	}
	delete dlg;
}

void CommonConfigPanel1::SetTrgoutMask() {
	QString current_mask = mWaveDump->getCommonTrgOutMask();
	uint64_t val = current_mask.toULongLong();
	chmaskDialog* dlg = new chmaskDialog(this, mWaveDump->getNumChannels(), val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask;
		new_mask.setNum(dlg->getMask());
		ui->lineEdit_trgout_mask->setText(QString("%1").arg(dlg->getMask(), 0, 16, QLatin1Char('0')));
		try {
			for(int b=0;b<mNumBrds;b++)
				mDevs.at(b)->setTrgOutMask(new_mask);
			mWaveDump->setCommonMask(new_mask, "Out_Propagate");
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
		}
	}
	delete dlg;
}


void CommonConfigPanel1::TextChanged(QString text) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel1::TrgoutChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	CAEN_FELib_ErrorCode err= CAEN_FELib_Success;
	QString value;
	ui->pB_trgout_mask->setEnabled(false);
	try{
		if (mode == "TRIGGER") {
			value = "OUT_PROPAGATION_TRIGGER";			
			uint64_t m = mWaveDump->getCommonTrgOutMask().toULong();
			ui->lineEdit_trgout_mask->setText(QString("%1").arg(m, 0, 16, QLatin1Char('0')));
		}
		else if (mode == "SIN") {
			value = "OUT_PROPAGATION_SYNCIN";
		}
		else if (mode == "RUN") {
			value = "OUT_PROPAGATION_RUN";
		}
		else if (mode == "REFCLK") {
			value = "OUT_PROPAGATION_SAMPLE_CLK";
		}
		for(int b=0;b<mNumBrds;b++)
			mDevs.at(b)->setTRGOUTMode(value);
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
		err = (CAEN_FELib_ErrorCode)exc.ErrorCode();
		return;
	}
	mWaveDump->CommonDevValuesmap1.insert("/par/OUT_Selection", value);
	mWaveDump->add2LogAndConf("SET", mDevName, "/par/OUT_Selection", mWaveDump->CommonDevValuesmap1.value("/par/OUT_Selection"), mWaveDump->CommonDevValuesmap1.value("/par/OUT_Selection"), CAEN_FELib_ErrorCode(0));
	NormalWidget(ui->cBox_trgout1, true);
}

void CommonConfigPanel1::TriggerModeChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	ui->pB_trg_mask->setEnabled(false);
	ui->cBox_polarity->setEnabled(false);
	if (mode == "SW_TRIGGER_ONLY") {
		try {
			for (int b = 0; b < mNumBrds; b++)
				mDevs.at(b)->setTrgSource("Sw");
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
			err = (CAEN_FELib_ErrorCode)exc.ErrorCode();
		}
		mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Ext_Enable", "FALSE");
		mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Sw_Enable", "TRUE");
		for (int ch = 0; ch < 64; ch++)
			mWaveDump->CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch), "FALSE");
		for (int g = 0; g < 32; g++)
			mWaveDump->CommonGroupValuesmap1[g].insert(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), "FALSE");
	}
	else if (mode == "SELF_TRIGGER") {
		ui->pB_trg_mask->setEnabled(true);
		ui->cBox_polarity->setEnabled(true);
		mIgnoreEvent = true;
		uint64_t m = mWaveDump->getCommonSelfTrgMask().toULong();
		ui->lineEdit_trig_mask->setText(QString("%1").arg(m, 0, 16, QLatin1Char('0')));
		mIgnoreEvent = false;
		try {
			for (int b = 0; b < mNumBrds; b++)
				mDevs.at(b)->setTrgSource("ITL");
			mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Ext_Enable", "FALSE");
			mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Sw_Enable", "FALSE");
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
			err = (CAEN_FELib_ErrorCode)exc.ErrorCode();
		}
	}
	else if (mode == "EXTERNAL_TRGIN") {
		try {
			for (int b = 0; b < mNumBrds; b++)
				mDevs.at(b)->setTrgSource("TrgIn");
			ui->lineEdit_trig_mask->setText("0");
			mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Ext_Enable", "TRUE");
			mWaveDump->CommonDevValuesmap1.insert("/par/Trg_Sw_Enable", "FALSE");
			for (int ch = 0; ch < 64; ch++)
				mWaveDump->CommonChValuesmap1[ch].insert(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch), "FALSE");
			for (int g = 0; g < 32; g++)
				mWaveDump->CommonGroupValuesmap1[g].insert(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), "FALSE");
		}
		catch (CAENFELibException& exc) {
			mWaveDump->addLog(exc.ErrorMessage(), false);
			err = (CAEN_FELib_ErrorCode)exc.ErrorCode();
		}
	}

	if (err == CAEN_FELib_Success) {
		NormalWidget(ui->cBox_trg_mode, true);
		mWaveDump->add2LogAndConf("SET", mDevName, "/par/Trg_Ext_Enable", mWaveDump->CommonDevValuesmap1.value("/par/Trg_Ext_Enable"), mWaveDump->CommonDevValuesmap1.value("/par/Trg_Ext_Enable"), CAEN_FELib_ErrorCode(0));
		mWaveDump->add2LogAndConf("SET", mDevName, "/par/Trg_Sw_Enable", mWaveDump->CommonDevValuesmap1.value("/par/Trg_Sw_Enable"), mWaveDump->CommonDevValuesmap1.value("/par/Trg_Sw_Enable"), CAEN_FELib_ErrorCode(0));
		for (int ch = 0; ch < 64; ch++)
			mWaveDump->add2LogAndConf("SET", mDevName, QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch)), mWaveDump->CommonChValuesmap1[ch].value(QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch)), CAEN_FELib_ErrorCode(0));
		for (int g = 0; g < 32; g++)
			mWaveDump->add2LogAndConf("SET", mDevName, QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g), mWaveDump->CommonGroupValuesmap1[g].value(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g)), mWaveDump->CommonGroupValuesmap1[g].value(QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g)), CAEN_FELib_ErrorCode(0));

	}
}


void CommonConfigPanel1::UpdateTriggerMode() {
	mIgnoreEvent = true;
	QString TrgMode = mWaveDump->getCommonTrgSource();

	if (TrgMode.contains("ITL", Qt::CaseInsensitive)) {
		ui->cBox_trg_mode->setCurrentText("SELF_TRIGGER");
		ui->pB_trg_mask->setEnabled(true);
		ui->cBox_polarity->setEnabled(true);
		uint64_t m = mWaveDump->getCommonSelfTrgMask().toULongLong();
		ui->lineEdit_trig_mask->setText(QString("%1").arg(m, 0, 16, QLatin1Char('0')));
	}
	else if (TrgMode.contains("TrgIn", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("EXTERNAL_TRGIN");
	else if (TrgMode.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("EXTERNAL_GPO");
	else if (TrgMode.contains("Sw", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("SW_TRIGGER_ONLY");
	else
		WarningWidget(ui->cBox_trg_mode, true);
	QStringList values = TrgMode.split("|");
	QVector<QString>mValues;
	mValues.append("SwCmd");
	mValues.append("ITL");
	mValues.append("TrgIn");
	
	ui->lWidget_Trg_Src->clear();
	for (int val = 0; val < mValues.size(); val++) {
		QListWidgetItem* newItem = new QListWidgetItem;
		newItem->setText(mValues.at(val));
		newItem->setFlags(newItem->flags() | Qt::ItemIsUserCheckable);
		newItem->setTextAlignment(Qt::AlignCenter);
		
		if (values.contains(mValues.at(val)))
			newItem->setCheckState(Qt::Checked);
		else
			newItem->setCheckState(Qt::Unchecked);
		
		ui->lWidget_Trg_Src->insertItem(val, newItem);
	}
	ui->lWidget_Trg_Src->setMaximumWidth(ui->lWidget_Trg_Src->sizeHintForColumn(0) + 5 * ui->lWidget_Trg_Src->frameWidth());
	
	mIgnoreEvent = false;	
}

void CommonConfigPanel1::UpdateTrgoutMode() {
	mIgnoreEvent = true;

	QString TrgoutMode = mWaveDump->CommonDevValuesmap1.value("/par/Out_Selection");
	if (TrgoutMode.contains("TRIGGER", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("TRIGGER");
		ui->pB_trgout_mask->setEnabled(true);
		uint64_t m = mWaveDump->getCommonTrgOutMask().toULong();
		ui->lineEdit_trgout_mask->setText(QString("%1").arg(m, 0, 16, QLatin1Char('0')));
	}
	else if (TrgoutMode.contains("SYNCIN", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("SIN");
	else if (TrgoutMode.contains("RUN", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("RUN");
	else if (TrgoutMode.contains("SAMPLE_CLK", Qt::CaseInsensitive))
		ui->cBox_trgout1->setCurrentText("REFCLK");
	else
		WarningWidget(ui->cBox_trgout1, true);
	mIgnoreEvent = false;
}

void CommonConfigPanel1::UpdatePreTrigger() {
	mIgnoreEvent = true;
	ui->dBox_pretrg->setMinimum(0);
	ui->dBox_pretrg->setMaximum(1000000);
	ui->dBox_pretrg->setSingleStep(1);
	double reclen = mWaveDump->CommonDevValuesmap1.value("/par/RecLen").toDouble()/1000.;
	double post = mWaveDump->CommonDevValuesmap1.value("/par/PostTrg").toDouble() / 1000.;
	if(mUoMX == UOM_PHYS_UNIT)
		ui->dBox_pretrg->setValue(reclen-post);
	else
		ui->dBox_pretrg->setValue((reclen - post) / (mWaveDump->getSampleToS() * 1e6));
	mIgnoreEvent = false;
}

void CommonConfigPanel1::UpdateEnable() {
	ui->cBox_enable->blockSignals(true);
	if (mWaveDump->CommonChValuesmap1[mFirstSelectedCh].value(QString("/ch/%1/par/Ch_Enabled").arg(mFirstSelectedCh)) == "True")
		ui->cBox_enable->setCurrentIndex(0);
	else
		ui->cBox_enable->setCurrentIndex(1);
	ui->cBox_enable->blockSignals(false);
}

void CommonConfigPanel1::UpdateClkSource() {
	mIgnoreEvent = true;
	try {
		QString clk = mWaveDump->CommonDevValuesmap1.value("/par/DT_Ext_Clock");
		if (clk == "TRUE")
			ui->cBox_clk_source->setCurrentText("External");
		else
			ui->cBox_clk_source->setCurrentText("Internal");
	}
	catch (CAENFELibException& exc) {
		mWaveDump->addLog(exc.ErrorMessage(), false);
	}
	mIgnoreEvent = false;
}

void CommonConfigPanel1::currentTabChanged(int tab){
	if (mIgnoreEvent)
		return;
	QWidget *currWidget = ui->tabWidget->currentWidget();
	if (currWidget == ui->tab_main) {
		UpdateMainSettings();
	}
	else if (currWidget == ui->tab_advanced) {
		UpdateDevGUI();
	}
}


CommonConfigPanel1::~CommonConfigPanel1()
{
	delete ui;
}
