/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CommonConfigPanel.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "CommonConfigPanel.h"
#include "WaveDump2.h"
#include "CAENSwitch.h"
#include "CAENFELibException.h"
#include "chmaskDialog.h"


CommonConfigPanel::CommonConfigPanel(WaveDump2 *parent, QVector<uint64_t> dev_handles) :
	QDialog(parent),
	ui(new Ui::CommonConfigPanel) {
	ui->setupUi(this);

	mWaveDump = parent;
	mDevHandle = dev_handles;
	mNumBrds = dev_handles.size();
	mUoMX = mWaveDump->getUoMX();
	mUoMY = mWaveDump->getUoMY();
	mDevName = "COMMON";


	CAENSwitch pippo("on","off",1,this);
	ui->lay_0->addWidget(&pippo);
	for (int i = 0; i < 64; i++) {
		mPbCh[i] = new CAENPushButton(this, i);
		mPbCh[i]->setMaximumWidth(43);
		mPbCh[i]->setText("CH " + QString::number(i));
		mPbCh[i]->setCheckable(true);
		connect(mPbCh[i], &CAENPushButton::clicked, [=] {channelClicked(mPbCh[i]); });
	}
	

	char val[32];
	ui->dBox_gain->setVisible(false);
	ui->label_17->setVisible(false);
	ui->label->setVisible(false);
	ui->cBox_decim->setVisible(false);
	mDevWithVGAGain.resize(mNumBrds);
	mDevWithVGAGain.fill(0);
	mDevWithCHGain.resize(mNumBrds);
	mDevWithCHGain.fill(0);
	mDevWithDecim.resize(mNumBrds);
	mDevWithDecim.fill(0);
	for(int b=0; b < mNumBrds; b++){
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mDevHandle[b], "/par/NumCh",val);
		if (err != CAEN_FELib_Success) {
			mNumberOfChannels.push_back(0);
		}
		else
			mNumberOfChannels.push_back(QString(val).toInt());
		
		 
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mDevHandle[b], "/par/ModelName", val);
		if (err != CAEN_FELib_Success) {
				char error[1024];
				CAEN_FELib_GetLastError(error);
				mWaveDump->addLog("Error getting device model" + QString(error), false);
		}
		if (QString(val).contains("2745")) {
			if (!mVGAGainVisible) {
				mVGAGainVisible = true;
				ui->dBox_gain->setVisible(true);
				ui->label_17->setVisible(true);
			}
			mDevWithVGAGain.replace(b,1);
			if (!mDecimationVisible) {
				mDecimationVisible = true;
				ui->label->setVisible(true);
				ui->cBox_decim->setVisible(true);
				mDevWithDecim.replace(b, 1);
			}
		}
		else if (QString(val).contains("2730") || QString(val).contains("2751")) {
			if (!mCHGainVisible) {
				mCHGainVisible = true;
				ui->dBox_gain->setVisible(true);
				ui->label_17->setVisible(true);
			}
			mDevWithCHGain.replace(b, 1);
		}
		else if (QString(val).contains("2740")) {
			if (!mDecimationVisible) {
				mDecimationVisible = true;
				ui->label->setVisible(true);
				ui->cBox_decim->setVisible(true);
				mDevWithDecim.replace(b, 1);
			}
		}
	}
	

	for (int i = 0; i < 4; i++) {
		for (int j = 0; j < 16; j++) {
			switch (i) {
				case 0:
					ui->lay_0->addWidget((mPbCh[(i * 16) + j]));
					break;
				case 1:
					ui->lay_1->addWidget(mPbCh[(i * 16) + j]);
					break;
				case 2:
					ui->lay_2->addWidget(mPbCh[(i * 16) + j]);
					break;
				case 3:
					ui->lay_3->addWidget(mPbCh[(i * 16) + j]);
					break;
			}
		}
	}
	mPbCh[0]->setChecked(true);
	mFirstSelectedCh = 0;
	mIgnoreEvent = false;
	mMulti = false;

	ui->pB_trg_mask->setEnabled(false);
	ui->pB_trgout_mask->setEnabled(false);

	connectWidgets();
	FillParamsAndWidgetsList();
	createMaps();
	defineScaleParams();

	UpdateParamsUoM();
	FillParamsInfo();
	UpdateDevGUI();
	UpdateChGUI();
	UpdateGroupGUI();
	UpdateMainSettings();
}

void CommonConfigPanel::UpdateMainSettings() {
	UpdateTriggerMode();
	UpdateTrgoutMode();
	UpdateVetoSrc();
	UpdateGpioMode();
}

void CommonConfigPanel::connectWidgets() {
	//connections
	connect(ui->pbAll, SIGNAL(clicked()), this, SLOT(selectAll()));
	connect(ui->pbMulti, SIGNAL(clicked()), this, SLOT(MultiClicked()));

	connect(ui->lWidget_StartSrc, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemCheckedChanged(QListWidgetItem *)));
	connect(ui->lWidget_TReset_Src, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemCheckedChanged(QListWidgetItem *)));
	connect(ui->lWidget_Trg_Src, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemCheckedChanged(QListWidgetItem *)));
	connect(ui->dBox_reclen, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dBox_pretrg, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->cBox_decim, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));

	connect(ui->cBox_enable, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_polarity, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->dSBox_bsl_level, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dBox_thr, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->dBox_gain, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));

	connect(ui->cBox_TRGOUT, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_GPIO, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->lWidget_VETOSrc, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemCheckedChanged(QListWidgetItem *)));
	connect(ui->dBox_VETO_width, SIGNAL(valueChanged(double)), this, SLOT(doubleChanged(double)));
	connect(ui->cBox_VETOPol, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_trg_mode, SIGNAL(currentTextChanged(QString)), this, SLOT(TriggerModeChanged(QString)));
	connect(ui->cBox_trgout1, SIGNAL(currentTextChanged(QString)), this, SLOT(TrgoutChanged(QString)));
	connect(ui->lineEdit_trig_mask, SIGNAL(textChanged(QString)), this, SLOT(TextChanged(QString)));
	connect(ui->lineEdit_trgout_mask, SIGNAL(textChanged(QString)), this, SLOT(TextChanged(QString)));
	connect(ui->pB_trg_mask, SIGNAL(clicked()), this, SLOT(SetMask()));
	connect(ui->pB_trgout_mask, SIGNAL(clicked()), this, SLOT(SetTrgoutMask()));
	connect(ui->cBox_veto1, SIGNAL(currentTextChanged(QString)), this, SLOT(VetoChanged(QString)));
	connect(ui->cBox_gpio1, SIGNAL(currentTextChanged(QString)), this, SLOT(GpioChanged(QString)));
	connect(ui->cBox_clk_source, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->gBox_clock_out, SIGNAL(clicked(bool)), this, SLOT(boolChanged(bool)));
	connect(ui->pushButton_permanent_clock, SIGNAL(clicked()), this, SLOT(SetPermanentClockDelay()));
	connect(ui->sBox_v_clk_delay, SIGNAL(valueChanged(int)), this, SLOT(intChanged(int)));
	connect(ui->cBox_DACout, SIGNAL(currentIndexChanged(int)), this, SLOT(intChanged(int)));

	connect(ui->tabWidget, SIGNAL(currentChanged(int)), this, SLOT(currentTabChanged(int)));
}

void CommonConfigPanel::FillParamsAndWidgetsList() {
	//start creating the list of device params
	mDevParamsList.clear();
	mDevWidgetsList.clear();

	mDevParamsList << "/par/RecordLengthS";
	mDevWidgetsList << ui->dBox_reclen;
	ui->dBox_reclen->setToolTip("/par/RecordLengthS");

	mDevParamsList << "/par/StartSource";
	mDevWidgetsList << ui->lWidget_StartSrc;
	ui->lWidget_StartSrc->setToolTip("/par/StartSource");

	mDevParamsList << "/par/AcqTriggerSource";
	mDevWidgetsList << ui->lWidget_Trg_Src;
	ui->lWidget_Trg_Src->setToolTip("/par/AcqTriggerSource");

	mDevParamsList << "/par/TstampResetSource";
	mDevWidgetsList << ui->lWidget_TReset_Src;
	ui->lWidget_TReset_Src->setToolTip("/par/TstampResetSource");

	mDevParamsList << "/par/PreTriggerS";
	mDevWidgetsList << ui->dBox_pretrg;
	ui->dBox_pretrg->setToolTip("/par/PreTriggerS");

	mDevParamsList << "/par/TrgOutMode";
	mDevWidgetsList << ui->cBox_TRGOUT;
	ui->cBox_TRGOUT->setToolTip("/par/TrgOutMode");

	mDevParamsList << "/par/GPIOMode";
	mDevWidgetsList << ui->cBox_GPIO;
	ui->cBox_GPIO->setToolTip("/par/GPIOMode");

	mDevParamsList << "/par/VetoSource";
	mDevWidgetsList << ui->lWidget_VETOSrc;
	ui->lWidget_VETOSrc->setToolTip("/par/VetoSource");

	mDevParamsList << "/par/VetoWidth";
	mDevWidgetsList << ui->dBox_VETO_width;
	ui->dBox_VETO_width->setToolTip("/par/VetoWidth");

	mDevParamsList << "/par/VetoPolarity";
	mDevWidgetsList << ui->cBox_VETOPol;
	ui->cBox_VETOPol->setToolTip("/par/VetoPolarity");

	mDevParamsList << "/par/ITLAMask";
	mDevWidgetsList << ui->lineEdit_trig_mask;
	ui->lineEdit_trig_mask->setToolTip("/par/ITLAMask");

	mDevParamsList << "/par/ITLBMask";
	mDevWidgetsList << ui->lineEdit_trgout_mask;
	ui->lineEdit_trgout_mask->setToolTip("/par/ITLBMask");

	if (mDecimationVisible) {
		mDevParamsList << "/par/DecimationFactor";
		mDevWidgetsList << ui->cBox_decim;
		ui->cBox_decim->setToolTip("/par/DecimationFactor");
	}

	mDevParamsList << "/par/ClockSource";
	mDevWidgetsList << ui->cBox_clk_source;
	ui->cBox_clk_source->setToolTip("/par/ClockSource");

	mDevParamsList << "/par/EnClockOutFP";
	mDevWidgetsList << ui->gBox_clock_out;
	ui->gBox_clock_out->setToolTip("/par/EnClockOutFP");

	mDevParamsList << "/par/VolatileClockOutDelay";
	mDevWidgetsList << ui->sBox_v_clk_delay;
	ui->sBox_v_clk_delay->setToolTip("/par/VolatileClockOutDelay");

	mDevParamsList << "/par/DACoutMode";
	mDevWidgetsList << ui->cBox_DACout;
	ui->cBox_DACout->setToolTip("/par/DACoutMode");

	//add more device parameters if needed ...

	//create the list of channel params
	mChParamsList.clear();
	mChWidgetsList.clear();

	mChParamsList << QString("/ch/%1/par/ChEnable");
	mChWidgetsList << ui->cBox_enable;
	ui->cBox_enable->setToolTip("/ch/N/par/ChEnable");

	mChParamsList << QString("/ch/%1/par/SelfTriggerEdge");
	mChWidgetsList << ui->cBox_polarity;
	ui->cBox_polarity->setToolTip("/ch/N/par/SelfTriggerEdge");

	mChParamsList << QString("/ch/%1/par/DCOffset");
	mChWidgetsList << ui->dSBox_bsl_level;
	ui->dSBox_bsl_level->setToolTip("/ch/N/par/DCOffset");

	mChParamsList << QString("/ch/%1/par/TriggerThr");
	mChWidgetsList << ui->dBox_thr;
	ui->dBox_thr->setToolTip("/ch/N/par/TriggerThr");

	if (mCHGainVisible) {
		mChParamsList << QString("/ch/%1/par/ChGain");
		mChWidgetsList << ui->dBox_gain;
		ui->dBox_thr->setToolTip("/ch/N/par/ChGain");
	}

	//create the list of group params
	mGroupParamsList.clear();
	mGroupWidgetsList.clear();
	mGroupsize.clear();

	if (mVGAGainVisible) {
		mGroupParamsList << QString("/vga/%1/par/VgaGain");
		mGroupWidgetsList << ui->dBox_gain;
		ui->dBox_gain->setToolTip("/vga/N/par/VgaGain");

		const size_t handles_size = 1024;
		uint64_t handles[handles_size];
		for (int b = 0; b < mNumBrds; b++) {
			if (!mDevWithVGAGain.at(b))
				continue;			

			CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetChildHandles(mDevHandle[b], "/vga/", handles, handles_size); //get group size from first device supporting it
			if (err < 0 || err > handles_size) {
				mWaveDump->addLog("Error getting VgaGain", false);
			}
			else {
				mGroupsize << mNumberOfChannels.at(b) / err;//gain group size (supposing it is the same for all boards supporting it)
				break;
			}
		}
	}

	//add more group parameters if needed
	//set default stylesheet color...needed for Qt6
	for (int i = 0; i < mGroupWidgetsList.size(); i++)
		mGroupWidgetsList.at(i)->setStyleSheet("color: black");
	mNumberOfGroupParams = mGroupParamsList.size();
	for (int g = 0; g < mNumberOfGroupParams; g++)
		mFirstSelectedGroup << 0; //for every group param depending on the group size there is a first selected group
}

void CommonConfigPanel::fillInfo(QString qry, ParamInfo *info) {
	CAEN_FELib_ErrorCode err;
	char val[200];
	uint64_t phandle;
	for (int b = 0; b < mNumBrds; b++) {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle[b], qPrintable(qry), &phandle);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/allowedValues", val);
		if (err == CAEN_FELib_Success) {//for list parameter fill allowed values vector
			if (atoi(val)) {
				uint64_t hallowed[32];
				int nvalues = (CAEN_FELib_ErrorCode)CAEN_FELib_GetChildHandles(phandle, "/allowedValues/", hallowed, 32);
				if (nvalues > 0) {
					for (int i = 0; i < nvalues; i++) {
						err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(hallowed[i], "", val);
						if (err != CAEN_FELib_Success)
							break;
						if (!info->allowed_values.contains(QString(val)))
							info->allowed_values.append(QString(val));
					}
				}
			}
		}
		else {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/minValue", val);
			if (err != CAEN_FELib_Success)
				info->min = "0";
			else {
				if (atoi(val) < info->min.toInt())
					info->min = QString(val);
			}
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/maxValue", val);
			if (err != CAEN_FELib_Success)
				info->max = "1000000000";
			else {
				if (atoi(val) > info->max.toInt())
					info->max = QString(val);
			}

			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/increment", val);
			if (err != CAEN_FELib_Success)
				info->incr = "1";
			else {
				if (atoi(val) < info->incr.toInt())
					info->incr = QString(val);
			}
		}
	}
}

void CommonConfigPanel::FillParamsInfo() {
	CAEN_FELib_ErrorCode err;
	char val[200];
	uint64_t phandle;

	for (int p = 0; p < mDevParamsList.size(); p++) {

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle[0], qPrintable(mDevParamsList.at(p)), &phandle);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/dataType", val);

		if (QString(val) == "STRING") {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/multipleValue", val);
			if (err == CAEN_FELib_Success)
				multipleValue << 1;
			else
				multipleValue << 0;
		}
		else
			multipleValue << 0;

		ParamInfo info;
		info.min = "1000000000";
		info.incr = "1000000000";
		info.max = "0";
		fillInfo(mDevParamsList.at(p), &info);		

		mDevParamsInfo.append(info);
	}

	for (int p = 0; p < mChParamsList.size(); p++) {

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle[0], qPrintable(mChParamsList.at(p).arg(0)), &phandle);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/dataType", val);

		if (QString(val) == "STRING") {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/multipleValue", val);
			if (err == CAEN_FELib_Success)
				multipleValue << 1;
			else
				multipleValue << 0;
		}
		else
			multipleValue << 0;

		ParamInfo info;
		info.min = "1000000000";
		info.incr = "1000000000";
		info.max = "0";
		fillInfo(mChParamsList.at(p).arg(0), &info); 

		mChParamsInfo.append(info);
	}

	for (int p = 0; p < mGroupParamsList.size(); p++) {

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle[0], qPrintable(mGroupParamsList.at(p).arg(0)), &phandle);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/dataType", val);

		if (QString(val) == "STRING") {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "/multipleValue", val);
			if (err == CAEN_FELib_Success)
				multipleValue << 1;
			else
				multipleValue << 0;
		}
		else
			multipleValue << 0;

		ParamInfo info;
		info.min = "1000000000";
		info.incr = "1000000000";
		info.max = "0";
		fillInfo(mGroupParamsList.at(p).arg(0), &info);

		mGroupParamsInfo.append(info);
	}
}

void CommonConfigPanel::UpdateParamsUoM() {
	if (mUoMX == UOM_PHYS_UNIT) {
		mWwithUoMX << ui->dBox_reclen;
		mWwithUoMX << ui->dBox_pretrg;
		mWwithUoMX << ui->dBox_VETO_width;
		QString mu = QChar(0xbc, 0x03);
		ui->label_pretrig_unit->setText("[" + mu + "s]");
		ui->label_reclen_unit->setText("[" + mu + "s]");
		ui->label_thr_unit->setText("[" + mu + "s]");
		ui->label_veto_width_unit->setText("[" + mu + "s]");
	}
	else {
		ui->label_pretrig_unit->setText("[samples]");
		ui->label_reclen_unit->setText("[samples]");
		ui->label_veto_width_unit->setText("[samples]");
	}

	if (mUoMY == UOM_PHYS_UNIT) {
		mWwithUoMY << ui->dBox_thr;
		ui->label_thr_unit->setText("[mV]");
	}
	else {
		ui->label_thr_unit->setText("[samples]");
	}

}

void CommonConfigPanel::createMaps() {
	uint64_t phandle=0;

	for (int p = 0; p < mDevParamsList.size(); p++) {
		for(int b=0;b<mNumBrds;b++){
			CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle.at(b), qPrintable(mDevParamsList.at(p)), &phandle);
			if (err != CAEN_FELib_Success) {
				mDevWidgetsList.at(p)->setEnabled(false);
				break;
			}
		}
		
		mDevGUIMap.insert(mDevWidgetsList.at(p), mDevParamsList.at(p));
	}

	for (int p = 0; p < mChParamsList.size(); p++) {
		for(int b=0;b<mNumBrds;b++){
			CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
			for (int ch = 0; ch < mNumberOfChannels.at(b); ch++) {
				phandle = 0;
				err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle.at(b), qPrintable(mChParamsList.at(p).arg(ch)), &phandle);
				if (err != CAEN_FELib_Success) {
					mChWidgetsList.at(p)->setEnabled(false);
					break;
				}
			}
			if (err != CAEN_FELib_Success)
				break;
		}

		mChGUIMap.insert(mChWidgetsList.at(p), mChParamsList.at(p));
	}

	for (int p = 0; p < mNumberOfGroupParams; p++) {
		mGroupGUIMap.insert(mGroupWidgetsList.at(p), mGroupParamsList.at(p));
	}


}

void CommonConfigPanel::defineScaleParams() {
	mDevRelatedParamsList.clear();
	for (int p = 0; p < mDevParamsList.size(); p++) {
		if (mDevParamsList.at(p) == "/par/RecordLengthS" || mDevParamsList.at(p) == "/par/PreTriggerS" && mDecimationVisible)
			mDevRelatedParamsList << "/par/DecimationFactor";
		else
			mDevRelatedParamsList << "";
	}
}


void CommonConfigPanel::UpdateDevGUI() {
	mIgnoreEvent = true;
	QHash<QWidget *, QString>::iterator item;
	for (item = mDevGUIMap.begin(); item != mDevGUIMap.end(); ++item) {
		if(item.key()->isEnabled()){
			int index = mDevWidgetsList.indexOf(item.key());
			QString pvalue = mWaveDump->CommonDevValuesmap.value(item.value());
			if (mDevRelatedParamsList.at(index) != "") {				
				if (mUoMX == UOM_PHYS_UNIT && mWwithUoMX.contains(item.key())) {
					double scaled_value = getScaleParamValue(0, mDevRelatedParamsList.at(index)) * pvalue.toDouble();
					pvalue = QString("%1").arg(scaled_value);
				}
			}
			UpdateWidget(item.key(), mDevParamsInfo.at(index),  pvalue);//update value depending on the widget type				
		}
			
	}
	if (mUoMX == UOM_PHYS_UNIT) {
		ui->dBox_reclen->setValue(ui->dBox_reclen->value() * mWaveDump->getSampleToS() * 1e6);
		ui->dBox_pretrg->setValue(ui->dBox_pretrg->value() * mWaveDump->getSampleToS() * 1e6);
		ui->dBox_VETO_width->setValue(ui->dBox_VETO_width->value() * mWaveDump->getSampleToS() * 1e6);
		ui->dBox_reclen->setDecimals(2);
		ui->dBox_pretrg->setDecimals(2);
		ui->dBox_VETO_width->setDecimals(2);
	}
	else {
		ui->dBox_reclen->setDecimals(0);
		ui->dBox_pretrg->setDecimals(0);
		ui->dBox_VETO_width->setDecimals(0);
	}
	mIgnoreEvent = false;
}

void CommonConfigPanel::UpdateChGUI() {
	mIgnoreEvent = true;
	QHash<QWidget *, QString>::iterator item;
	for (item = mChGUIMap.begin(); item != mChGUIMap.end(); ++item) {
		if (item.key()->isEnabled()) {
			int index = mChWidgetsList.indexOf(item.key());
			UpdateWidget(item.key(), mChParamsInfo.at(index), mWaveDump->CommonChValuesmap[mFirstSelectedCh].value(item.value().arg(mFirstSelectedCh)));//update value depending on the widget type	
		}			
	}

	if (mUoMY == UOM_PHYS_UNIT) {
		ui->dBox_thr->setValue(ui->dBox_thr->value() * mWaveDump->getSampleToV() * 1e3);
		ui->dBox_thr->setDecimals(2);
	}
	else {
		ui->dBox_thr->setDecimals(0);
	}
	mIgnoreEvent = false;
}

void CommonConfigPanel::UpdateGroupGUI() {
	mIgnoreEvent = true;
	QHash<QWidget*, QString>::iterator item;
	for (item = mGroupGUIMap.begin(); item != mGroupGUIMap.end(); ++item) {
		if (item.key()->isEnabled()) {
			int index = mGroupWidgetsList.indexOf(item.key());
			UpdateWidget(item.key(), mGroupParamsInfo.at(index), mWaveDump->CommonGroupValuesmap[mFirstSelectedGroup.at(index)].value(item.value().arg(mFirstSelectedGroup.at(index))));//update value depending on the widget type	
		}
	}
	mIgnoreEvent = false;
}

void CommonConfigPanel::UpdateWidget(QWidget *widget, ParamInfo info, QString value){
		if (widget == nullptr)
			return;
		double UoM_to_GUIUoM=1.;

		if (QComboBox *cbox = qobject_cast<QComboBox *>(widget)){
				cbox->clear();
				int index = -1;
				for (int i = 0; i < info.allowed_values.size(); i++) {
					cbox->addItem(info.allowed_values.at(i));
					if (value == info.allowed_values.at(i))
						index = (int)i;
				}
				cbox->setCurrentIndex(index);
		}
			
		else if (QSpinBox *sbox = qobject_cast<QSpinBox *>(widget)){
				sbox->setMinimum(info.min.toInt()  * UoM_to_GUIUoM);
				sbox->setMaximum(info.max.toInt() * UoM_to_GUIUoM);
				sbox->setSingleStep(info.incr.toInt() * UoM_to_GUIUoM);
				sbox->setValue(qRound(value.toDouble() * UoM_to_GUIUoM));
		}
		else if (QDoubleSpinBox *dbox = qobject_cast<QDoubleSpinBox *>(widget)){
				dbox->setMinimum(info.min.toDouble() * UoM_to_GUIUoM);
				dbox->setMaximum(info.max.toDouble() * UoM_to_GUIUoM);
				dbox->setSingleStep(info.incr.toDouble() * UoM_to_GUIUoM);
				int i = info.incr.lastIndexOf('.');
				int decimals = (i > 0) ? 2 : 0;
				dbox->setDecimals(decimals);
				dbox->setValue(value.toDouble() * UoM_to_GUIUoM);
		}
		else if (QLineEdit *ledit = qobject_cast<QLineEdit *>(widget))
			ledit->setText(value);
		else if (QListWidget *lwidget = qobject_cast<QListWidget *>(widget)){
				lwidget->clear();
				for (int val = 0; val < info.allowed_values.size(); val ++) {
					QListWidgetItem *newItem = new QListWidgetItem;
					newItem->setText(info.allowed_values.at(val));
					newItem->setFlags(newItem->flags() | Qt::ItemIsUserCheckable);
					newItem->setTextAlignment(Qt::AlignCenter);
					if(value.contains(info.allowed_values.at(val), Qt::CaseInsensitive))
						newItem->setCheckState(Qt::Checked);
					else
						newItem->setCheckState(Qt::Unchecked);
					lwidget->insertItem(val, newItem);
				}
				lwidget->setMaximumWidth(lwidget->sizeHintForColumn(0) + 5 * lwidget->frameWidth());
			
		}
	
		widget->setStyleSheet("color: black");
}

//multi sel. active --> check if the selected channels have the same param values
void CommonConfigPanel::CheckChParamValues() {
	for (int p = 0; p < mChParamsList.size(); p++) {
		if (!mChWidgetsList.at(p)->isEnabled())
			continue;
		QString first_sel_val = mWaveDump->CommonChValuesmap[mFirstSelectedCh].value(mChParamsList.at(p).arg(mFirstSelectedCh));
		bool same_value = true;
		for(int b=0;b<mNumBrds;b++){
			for (int i = 0; i < mNumberOfChannels.at(b); i++) {
				if (!mPbCh[i]->isChecked() || (i == mFirstSelectedCh))
					continue;

				if (mWaveDump->CommonChValuesmap[i].value(mChParamsList.at(p).arg(i)) != first_sel_val) {
					same_value = false;
					break;
				}
			}
			if(!same_value)
				break;
		}

		if (!same_value) {
			WarningWidget(mChWidgetsList.at(p),false);
		}
	}
}

void CommonConfigPanel::channelClicked(CAENPushButton *pb) {
	int ch = pb->getIndex();

	//single channel selected
	if (!mMulti) {
		for (int i = 0; i < 64; i++)
			if (i != ch) mPbCh[i]->setChecked(false);
		if (ch != mFirstSelectedCh) {
			mFirstSelectedCh = ch;
			UpdateChGUI();
		}
		else
			mPbCh[ch]->setChecked(true);
		for (int p = 0; p < mNumberOfGroupParams; p++)
			mFirstSelectedGroup.replace(p, (ch / mGroupsize.at(p)));
		UpdateGroupGUI();
	}
	else {//multi sel. active
		if (ch == mFirstSelectedCh) {
			for (int i = 0; i < 64; i++) {
				if (mPbCh[i]->isChecked()) {
					mFirstSelectedCh = i;
					UpdateChGUI();
					break;
				}
			}
			mPbCh[mFirstSelectedCh]->setChecked(true);
		}
		for (int p = 0; p < mNumberOfGroupParams; p++)
			mFirstSelectedGroup.replace(p, (ch / mGroupsize.at(p)));
		UpdateGroupGUI();
		CheckChParamValues();
		CheckGroupParamValues();
	}
}

//multi sel. active --> check if the selected channels have the same param values
void CommonConfigPanel::CheckGroupParamValues() {
	for (int p = 0; p < mGroupParamsList.size(); p++) {
		if (!mGroupWidgetsList.at(p)->isEnabled())
			continue;
		QString first_sel_val = mWaveDump->CommonGroupValuesmap[mFirstSelectedGroup.at(p)].value(mGroupParamsList.at(p).arg(mFirstSelectedGroup.at(p)));
		bool same_value = true;
		for (int b = 0; b < mNumBrds; b++) {
			for (int i = 0; i < mNumberOfChannels.at(b)/mGroupsize.at(p); i++) {
				if (!mPbCh[i]->isChecked() || (i == mFirstSelectedGroup.at(p)))
					continue;

				if (mWaveDump->CommonGroupValuesmap[i].value(mGroupParamsList.at(p).arg(i)) != first_sel_val) {
					same_value = false;
					break;
				}
			}
			if (!same_value)
				break;
		}

		if (!same_value) {
			WarningWidget(mGroupWidgetsList.at(p), false);
		}
	}
}
void CommonConfigPanel::selectAll() {
	for (int i = 0; i < 64; i++)
		mPbCh[i]->setChecked(true);
	ui->pbMulti->setChecked(true);
	CheckChParamValues();
	CheckGroupParamValues();
}

void  CommonConfigPanel::MultiClicked() {
	mMulti = ui->pbMulti->isChecked();
	if (!mMulti) {
		QHash<QWidget *, QString>::iterator item;
		for (item = mChGUIMap.begin(); item != mChGUIMap.end(); ++item) {
			NormalWidget(item.key(), false);
		}
		int memCh = mFirstSelectedCh;
		mFirstSelectedCh = -1;
		emit channelClicked(mPbCh[memCh]);
	}
}

void CommonConfigPanel::WarningWidget(QWidget *widget, bool bold) {
	if (!widget->styleSheet().contains("color: black"))
		return;
	if(bold)
		widget->setStyleSheet("font-weight: bold; color: red");
	else
		widget->setStyleSheet("color: red");
	mIgnoreEvent = true;
	if (QComboBox *cbox = qobject_cast<QComboBox *>(widget)) {
		cbox->setCurrentIndex(-1);
		mIgnoreEvent = false;
		return;
	}
	if(QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(widget)){
		doubleSpinBox->setSpecialValueText("?");
		doubleSpinBox->setMinimum(doubleSpinBox->minimum() - 1);
		doubleSpinBox->setValue(doubleSpinBox->minimum());
		mIgnoreEvent = false;
		return;
	}
	if(QSpinBox *SpinBox = qobject_cast<QSpinBox *>(widget)){
		SpinBox->setSpecialValueText("?");
		SpinBox->setMinimum(SpinBox->minimum() - 1);
		SpinBox->setValue(SpinBox->minimum());
		mIgnoreEvent = false;
		return;
	}
	mIgnoreEvent = false;
}

void CommonConfigPanel::NormalWidget(QWidget *widget, bool bold) {
	if (!widget->styleSheet().contains("color: red"))
		return;
	if(bold)
		widget->setStyleSheet("font-weight: bold; color: black");
	else
		widget->setStyleSheet("color: black");
	mIgnoreEvent = true;
	QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(widget);
	if (doubleSpinBox != nullptr) {
		doubleSpinBox->setSpecialValueText("");
		doubleSpinBox->setMinimum(doubleSpinBox->minimum() + 1);
		mIgnoreEvent = false;
		return;
	}
	QSpinBox *SpinBox = qobject_cast<QSpinBox *>(widget);
	if (SpinBox != nullptr) {
		SpinBox->setSpecialValueText("");
		SpinBox->setMinimum(SpinBox->minimum() + 1);
		mIgnoreEvent = false;
		return;
	}
	mIgnoreEvent = false;
	
}

void CommonConfigPanel::ApplyNewValue(QWidget *widget) {

		if (mDevGUIMap.contains(widget)) {
			int ok = SetDevParamValueFromWidget(widget,  mDevGUIMap.value(widget));
			mWaveDump->add2LogAndConf("SET", mDevName, mDevGUIMap.value(widget), mWaveDump->CommonDevValuesmap.value(mDevGUIMap.value(widget)), mWaveDump->CommonDevValuesmap.value(mDevGUIMap.value(widget)), CAEN_FELib_ErrorCode(ok));
		}
		else if (mChGUIMap.contains(widget)) {
			if (!ui->pbMulti->isChecked()) {//single channel selected
				int ok = SetDevParamValueFromWidget(widget, mChGUIMap.value(widget).arg(mFirstSelectedCh));
				mWaveDump->add2LogAndConf("SET", mDevName, mChGUIMap.value(widget).arg(mFirstSelectedCh), mWaveDump->CommonChValuesmap[mFirstSelectedCh].value(mChGUIMap.value(widget).arg(mFirstSelectedCh)), mWaveDump->CommonChValuesmap[mFirstSelectedCh].value(mChGUIMap.value(widget).arg(mFirstSelectedCh)), CAEN_FELib_ErrorCode(ok));
			}
			else {//multi channel select
				for (int ch = 0; ch < 64; ch++) {
					if (!mPbCh[ch]->isChecked())
						continue;
					int ok = SetDevParamValueFromWidget(widget, mChGUIMap.value(widget).arg(ch));
					mWaveDump->add2LogAndConf("SET", mDevName, mChGUIMap.value(widget).arg(ch), mWaveDump->CommonChValuesmap[ch].value(mChGUIMap.value(widget).arg(ch)), mWaveDump->CommonChValuesmap[ch].value(mChGUIMap.value(widget).arg(ch)), CAEN_FELib_ErrorCode(ok));
				}
				NormalWidget(widget, false);
			}
		}
		else if (mGroupGUIMap.contains(widget)) {
			if (!ui->pbMulti->isChecked() && mGroupGUIMap.value(widget) != nullptr) {//single channel selected
				int index = mGroupWidgetsList.indexOf(widget);
				int ok = SetDevParamValueFromWidget(widget, mGroupGUIMap.value(widget).arg(mFirstSelectedGroup.at(index)));
				mWaveDump->add2LogAndConf("SET", mDevName, mGroupGUIMap.value(widget).arg(mFirstSelectedGroup.at(index)), mWaveDump->CommonGroupValuesmap[mFirstSelectedGroup.at(index)].value(mGroupGUIMap.value(widget).arg(mFirstSelectedGroup.at(index))), mWaveDump->CommonGroupValuesmap[mFirstSelectedGroup.at(index)].value(mGroupGUIMap.value(widget).arg(mFirstSelectedGroup.at(index))), CAEN_FELib_ErrorCode(ok));

			}
			else {//multi channel select
				int index = mGroupWidgetsList.indexOf(widget);
				for (int g = 0; g < 64 / mGroupsize.at(index); g++) {
					if (!mPbCh[g* mGroupsize.at(index)]->isChecked())
						continue;
					for (int b = 0; b < mNumBrds; b++) {
						if (!mDevWithVGAGain.at(b))
							continue;
						int ok = SetDevParamValueFromWidget(widget, mGroupGUIMap.value(widget).arg(g));
						mWaveDump->add2LogAndConf("SET", mDevName, mGroupGUIMap.value(widget).arg(g), mWaveDump->CommonGroupValuesmap[g].value(mGroupGUIMap.value(widget).arg(g)), mWaveDump->CommonGroupValuesmap[g].value(mGroupGUIMap.value(widget).arg(g)), CAEN_FELib_ErrorCode(ok));
					}
				}
				NormalWidget(widget, false);
			}
		}

}

QString CommonConfigPanel::GetValueToSetFromWidget(QWidget *widget, bool *check_uom){
	*check_uom=false;
	QString value="";
	if (QComboBox *cbox = qobject_cast<QComboBox *>(widget)){
			value = cbox->currentText();
		}			
		else if (QSpinBox *sBox = qobject_cast<QSpinBox *>(widget)){
			value = QString("%1").arg(qRound(static_cast<double>(sBox->value())));
			*check_uom=true;
		}
		else if (QDoubleSpinBox *dbox = qobject_cast<QDoubleSpinBox *>(widget)){
			value = QString("%1").arg(dbox->value());
			*check_uom=true;
		}
		else if (QLineEdit *lineEdit = qobject_cast<QLineEdit *>(widget)){
			if(mDevGUIMap.value(widget).contains("mask", Qt::CaseInsensitive))
				value = "0x" + lineEdit->text();
			else
				value = lineEdit->text();
		}
		else if (QGroupBox* gBox = qobject_cast<QGroupBox*>(widget)) {
			if (gBox->isCheckable())
				value = gBox->isChecked() ? "TRUE" : "FALSE";
			else
				value = gBox->isEnabled() ? "TRUE" : "FALSE";
		}
		else if (QListWidget *lwidget = qobject_cast<QListWidget *>(widget)){
				bool check_ok = false;
				for (int i = 0; i < lwidget->count(); ++i){
					if (lwidget->item(i)->checkState() == Qt::Checked) {
						check_ok = true;
						if (value == "")
							value.append(lwidget->item(i)->text());
						else
							value.append("|" + lwidget->item(i)->text());
					}		
				}

				QString current_value;
				if (mDevGUIMap.contains(widget))
					current_value = mWaveDump->CommonDevValuesmap.value(mDevGUIMap.value(widget));
				else
					current_value = mWaveDump->CommonChValuesmap[mFirstSelectedCh].value(mChGUIMap.value(widget));
				if (check_ok) {
					if (value.contains("Disabled", Qt::CaseInsensitive) && !current_value.contains("Disabled")) {
						value = "Disabled";
						for (int i = 0; i < lwidget->count(); ++i) {
							if (lwidget->item(i)->checkState() == Qt::Checked && lwidget->item(i)->text() != "Disabled") {
								lwidget->blockSignals(true);
								lwidget->item(i)->setCheckState(Qt::Unchecked);
								lwidget->blockSignals(false);
							}
						}
					}
					if (value.contains("Disabled", Qt::CaseInsensitive) && value != "Disabled") {
						value.replace("Disabled|", "");
						for (int i = 0; i < lwidget->count(); ++i) {
							if (lwidget->item(i)->checkState() == Qt::Checked && lwidget->item(i)->text()=="Disabled") {
								lwidget->blockSignals(true);
								lwidget->item(i)->setCheckState(Qt::Unchecked);
								lwidget->blockSignals(false);
								break;
							}
						}
					}
				}
				else {
					QVector<ParamInfo> *info;
					int index = mDevWidgetsList.indexOf(widget);
					if (index != -1)
						info = &mDevParamsInfo;
					else{
						index = mChWidgetsList.indexOf(widget);
						info = &mChParamsInfo;
						if (index == -1) {
							index = mGroupWidgetsList.indexOf(widget);
							info = &mGroupParamsInfo;						
						}
					}
						
					if(multipleValue[index] && current_value.contains("Disabled", Qt::CaseInsensitive))
						value = "Disabled";
				}
		}
		return value;
}

int CommonConfigPanel::CommonSetParam(QString qry, QString value, int ch, QString rpar, QString *set_value){
		for(int b=0;b<mNumBrds;b++){
			if (ch > (mNumberOfChannels[b] - 1) && ch != -1)
				continue;
			if (qry.contains("vga",Qt::CaseInsensitive) && !mDevWithVGAGain.at(b))
				continue;
			if (qry.contains("chgain",Qt::CaseInsensitive) && !mDevWithCHGain.at(b))
				continue;
			if (qry.contains("decimation", Qt::CaseInsensitive) && !mDevWithDecim.at(b))
				continue;

			QString val = value;
			double factor, scaled_val;
			if (rpar != "") {
				factor = getScaleParamValue(b, rpar);
				scaled_val = val.toDouble() / factor;
				val = QString("%1").arg(scaled_val);
			}
			
			CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mDevHandle[b], qPrintable(qry), qPrintable(val));
			if (err != CAEN_FELib_Success)
				return err;

			*set_value = val;
		}
		return 0;
}

double CommonConfigPanel::getScaleParamValue(int b, QString qry) {
	char val[256] = { 0 };
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	uint64_t phandle;
	if(qry.contains("decimation", Qt::CaseInsensitive) && !mDevWithDecim.at(b))
		return 1.0;
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mDevHandle[b], qPrintable(qry), &phandle);
	if (err != CAEN_FELib_Success) {
		mWaveDump->addLog(QString("Get Handle of " + qry + " error %1").arg(err), false);
	}
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(phandle, "", val);
	if (err != CAEN_FELib_Success) {
		char error[1024];
		CAEN_FELib_GetLastError(error);
		throw CAENFELibException("Error getting value of param " + qry, err);
	}
	return QString(val).toDouble();
}

int CommonConfigPanel::SetDevParamValueFromWidget(QWidget *widget, QString qry) {		
		double UoM_to_GUIUoM = 1.0;
		bool check_UoM;
		int err = 0;
		QString related_param = "";

		QString value=GetValueToSetFromWidget(widget, &check_UoM);
		if(check_UoM && mUoMX==UOM_PHYS_UNIT){
			if (mWwithUoMX.contains(widget)) {
				double val = value.toDouble();
				val = val /(mWaveDump->getSampleToS() * 1e6);
				value = QString("%1").arg(val);
			}
			const auto it = mDevGUIMap.find(widget);
			const auto index = std::distance(mDevGUIMap.begin(), it);
			if (index < mDevRelatedParamsList.size()) { //if >= size the widget has not been found
				if (mDevRelatedParamsList.at(index) != "") {  //check if this param has a related param
					related_param = mDevRelatedParamsList.at(index);
				}
			}
		}
		if (check_UoM && mUoMY == UOM_PHYS_UNIT) {
			if (mWwithUoMY.contains(widget)) {
				double val = value.toDouble();
				val = val / (mWaveDump->getSampleToV() * 1e3);
				value = QString("%1").arg(val);
			}
		}

		int ch = -1;
		if (qry.contains("/ch/")) {
			QStringList l = qry.split("/ch/");
			QStringList l1 = l.at(1).split("/");
			ch = l1.at(0).toInt();
		}
		else if (qry.contains("/vga/")) {
			QStringList l = qry.split("/vga/");
			QStringList l1 = l.at(1).split("/");
			ch = l1.at(0).toInt();
		}

		QString set_val;
	    if(err=CommonSetParam(qry, value, ch, related_param, &set_val) != 0)
			return err;
		
		bool found=false;
		QHash<QString, QString>::iterator item;
		for (item = mWaveDump->CommonDevValuesmap.begin(); item != mWaveDump->CommonDevValuesmap.end(); ++item) {
			if(item.key() == qry){
				mWaveDump->CommonDevValuesmap.insert(item.key(),set_val);
				found=true;
				break;
			}
		}
		if(!found){
			QHash<QString, QString>::iterator item;
			for (item = mWaveDump->CommonChValuesmap[ch].begin(); item != mWaveDump->CommonChValuesmap[ch].end(); ++item) {
				if(item.key() == qry){
					mWaveDump->CommonChValuesmap[ch].insert(item.key(),value);
					found=true;
					break;
				}
			}
		}
		if (!found) {
			QHash<QString, QString>::iterator item;
			for (item = mWaveDump->CommonGroupValuesmap[ch].begin(); item != mWaveDump->CommonGroupValuesmap[ch].end(); ++item) {
				if (item.key() == qry) {
					mWaveDump->CommonGroupValuesmap[ch].insert(item.key(), value);
					found = true;
					break;
				}
			}
		}
		
		return 0;
}

void CommonConfigPanel::intChanged(int val) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel::boolChanged(bool val) {
	if (mIgnoreEvent)
		return;
	QWidget* widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel::doubleChanged(double val) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel::itemCheckedChanged(QListWidgetItem *item) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel::SetMask() {
	QString current_mask = mWaveDump->CommonDevValuesmap.value("/par/ITLAMask");
	uint64_t val = current_mask.toULongLong();
	chmaskDialog *dlg = new chmaskDialog(this, 64, val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask = QString("%1").arg(dlg->getMask(),0,16, QLatin1Char('0'));
		ui->lineEdit_trig_mask->setText(new_mask);
	}
	delete dlg;
}

void CommonConfigPanel::SetTrgoutMask() {
	QString current_mask = mWaveDump->CommonDevValuesmap.value("/par/ITLBMask");
	uint64_t val = current_mask.toULongLong();
	chmaskDialog* dlg = new chmaskDialog(this, 64, val);
	int ok = dlg->exec();
	if (ok == QDialog::Accepted) {
		QString new_mask = QString("%1").arg(dlg->getMask(), 0, 16, QLatin1Char('0'));
		ui->lineEdit_trgout_mask->setText(new_mask);
	}
	delete dlg;
}


void CommonConfigPanel::TextChanged(QString text) {
	if (mIgnoreEvent)
		return;
	QWidget *widget = qobject_cast<QWidget*>(sender());
	ApplyNewValue(widget);
}

void CommonConfigPanel::TrgoutChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	int err = 0;
	QString set_val;
	ui->pB_trgout_mask->setEnabled(false);
	if (mode == "DISABLED") { 
		if (err=CommonSetParam("/par/TRGOUTMode", "Disabled", -1,"", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "Disabled","Disabled", CAEN_FELib_Success);
		mWaveDump->CommonDevValuesmap.insert("/par/TrgOutMode", mode);
	}
	else if (mode == "TRGIN") {
		if (err=CommonSetParam("/par/TRGOUTMode", "TrgIn", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "TrgIn","TrgIn", CAEN_FELib_Success);
		mWaveDump->CommonDevValuesmap.insert("/par/TrgOutMode", "TrgIn");
	}
	else if (mode == "SIN") {
		if (err=CommonSetParam("/par/TRGOUTMode", "SIN", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "SIN","SIN", CAEN_FELib_Success);
		mWaveDump->CommonDevValuesmap.insert("/par/TrgOutMode", mode);
	}
	else if (mode == "SELF TRIGGER") {
		if (err=CommonSetParam("/par/TRGOUTMode", "ITLB", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
			return;
		}
		else {
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "ITLB","ITLB", CAEN_FELib_Success);
			ui->pB_trgout_mask->setEnabled(true);
		}
		if (err = CommonSetParam("/par/ITLBMainLogic", "OR", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/ITLBMainLogic error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLBMainLogic", "OR","OR", CAEN_FELib_Success);
		if (err = CommonSetParam("/par/ITLBPairLogic", "None", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/ITLBPairLogic error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLBPairLogic", "None", "None", CAEN_FELib_Success);
		if (err = CommonSetParam("/par/ITLBGateWidth", "16", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/ITLBGateWidth error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/ITLBGateWidth", "16", "16", CAEN_FELib_Success);

		mWaveDump->CommonDevValuesmap.insert("/par/TrgOutMode", "ITLB");
		ui->pB_trgout_mask->setEnabled(true);
	}
	else if (mode == "GPIO") {
		if (err=CommonSetParam("/par/TRGOUTMode", "GPIO", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "GPIO", "GPIO", CAEN_FELib_Success);
		mWaveDump->CommonDevValuesmap.insert("/par/TrgOutMode", mode);
	}
	else if (mode == "RUN") {
		if (err=CommonSetParam("/par/TRGOUTMode", "Run", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "Run", "Run", CAEN_FELib_Success);
		mWaveDump->CommonDevValuesmap.insert("/par/TrgOutMode", mode);
	}
	else if (mode == "REFCLK") {
		if (err = CommonSetParam("/par/TRGOUTMode", "RefClk", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/TRGOUTMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/TRGOUTMode", "RefClk", "RefClk", CAEN_FELib_Success);
		mWaveDump->CommonDevValuesmap.insert("/par/TrgOutMode", mode);
	}
	
	NormalWidget(ui->cBox_trgout1, true);

}

void CommonConfigPanel::TriggerModeChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	int err=0;
	QString set_val;
	ui->pB_trg_mask->setEnabled(false);

	if (mode == "INDEPENDENT_SELF_TRIGGERS" || mode == "SW_TRIGGER_ONLY") {
		if(err=CommonSetParam("/par/AcqTriggerSource", "SWTrg", -1, "", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		mWaveDump->add2LogAndConf("SET", mDevName,"/par/AcqTriggerSource" , "SWTrg", "SWTrg", CAEN_FELib_Success);
	}
	else if (mode == "SELF_TRIGGER") {		
		if(err=CommonSetParam("/par/AcqTriggerSource", "SWTrg|ITLA", -1, "", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		mWaveDump->add2LogAndConf("SET", mDevName,"/par/AcqTriggerSource" , "SWTrg|ITLA", "SWTrg|ITLA", CAEN_FELib_Success);

		if(err=CommonSetParam("/par/ITLAMainLogic", "OR", -1, "", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/ITLAMainLogic error %1").arg(err), false);
			return;
		}		
		mWaveDump->add2LogAndConf("SET", mDevName,"/par/ITLAMainLogic" , "OR", "OR", CAEN_FELib_Success);
		
		if(err=CommonSetParam("/par/ITLAPairLogic", "None", -1, "", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/ITLAPairLogic error %1").arg(err), false);
			return;
		}		
		mWaveDump->add2LogAndConf("SET", mDevName,"/par/ITLAPairLogic" , "None", "None", CAEN_FELib_Success);

		if(err=CommonSetParam("/par/ITLAGateWidth", "16", -1, "", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/ITLAGateWidth error %1").arg(err), false);
			return;
		}
		mWaveDump->add2LogAndConf("SET", mDevName,"/par/ITLAGateWidth" , "16", "16", CAEN_FELib_Success);

		ui->pB_trg_mask->setEnabled(true);
	}
	else if (mode == "EXTERNAL_TRGIN") {
		if(err=CommonSetParam("/par/AcqTriggerSource", "SWTrg|TrgIn", -1, "", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		mWaveDump->add2LogAndConf("SET", mDevName,"/par/AcqTriggerSource" , "SWTrg|TrgIn", "SWTrg|TrgIn", CAEN_FELib_Success);
	}
	else if (mode == "EXTERNAL_GPIO") {
		if(err=CommonSetParam("/par/AcqTriggerSource", "SWTrg|GPIO", -1, "", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		mWaveDump->add2LogAndConf("SET", mDevName,"/par/AcqTriggerSource" , "SWTrg|GPIO", "SWTrg|GPIO", CAEN_FELib_Success);
	}
	else if (mode == "INTERNAL_PULSER") {
		if(err=CommonSetParam("/par/AcqTriggerSource", "SWTrg|TestPulse", -1, "", &set_val) != 0){
			mWaveDump->addLog(QString("Set /par/AcqTriggerSource error %1").arg(err), false);
			return;
		}
		mWaveDump->add2LogAndConf("SET", mDevName,"/par/AcqTriggerSource" , "SWTrg|TestPulse", "SWTrg|TestPulse", CAEN_FELib_Success);
	}

	mWaveDump->CommonDevValuesmap.insert("/par/AcqTriggerSource",mode);	
	NormalWidget(ui->cBox_trg_mode, true);
}

void CommonConfigPanel::VetoChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	int err=0;
	QString set_val;
	if (mode == "DISABLED") {
		if (err = CommonSetParam("/par/VETOSource", "Disabled", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/VETOSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/VETOSource", "Disabled", "Disabled", CAEN_FELib_Success);
	}
	else if (mode == "SIN") {
		if (err = CommonSetParam("/par/VETOSource", "SIN", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/VETOSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/VETOSource", "SIN", "SIN", CAEN_FELib_Success);
	}
	if (mode == "GPIO") {
		if (err = CommonSetParam("/par/VETOSource", "GPIO", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/VETOSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/VETOSource", "GPIO", "GPIO", CAEN_FELib_Success);
	}
	if (mode == "LVDS") {
		if (err = CommonSetParam("/par/VETOSource", "LVDS", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/VETOSource error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/VETOSource", "LVDS", "LVDS", CAEN_FELib_Success);
	}

	mWaveDump->CommonDevValuesmap.insert("/par/VETOSource", mode);
	NormalWidget(ui->cBox_veto1, true);
}

void CommonConfigPanel::GpioChanged(QString mode) {
	if (mIgnoreEvent)
		return;
	int err=0;
	QString set_val;
	if (mode == "DISABLED") {
		if (err = CommonSetParam("/par/GPIOMode", "Disabled", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "Disabled", "Disabled", CAEN_FELib_Success);
	}
	else if (mode == "SIN") {
		if (err = CommonSetParam("/par/GPIOMode", "SIN", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "SIN", "SIN", CAEN_FELib_Success);
	}
	else if (mode == "LVDS") {
		if (err = CommonSetParam("/par/GPIOMode", "LVDS", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "LVDS", "LVDS", CAEN_FELib_Success);
	}
	else if (mode == "TRGIN") {
		if (err = CommonSetParam("/par/GPIOMode", "TrgIn", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "TrgIn", "TrgIn", CAEN_FELib_Success);
	}
	else if (mode == "SELF TRIGGER") {
		if (err = CommonSetParam("/par/GPIOMode", "ITLA", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "ITLA", "ITLA", CAEN_FELib_Success);
	}
	else if (mode == "RUN") {
		if (err = CommonSetParam("/par/GPIOMode", "Run", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "Run", "Run", CAEN_FELib_Success);
	}
	else if (mode == "BUSY") {
		if (err = CommonSetParam("/par/GPIOMode", "Busy", -1, "", &set_val) != 0) {
			mWaveDump->addLog(QString("Set /par/GPIOMode error %1").arg(err), false);
			return;
		}
		else
			mWaveDump->add2LogAndConf("SET", mDevName, "/par/GPIOMode", "Busy", "Busy", CAEN_FELib_Success);
	}

	mWaveDump->CommonDevValuesmap.insert("/par/GPIOMode", mode);
	NormalWidget(ui->cBox_gpio1, true);
}

void CommonConfigPanel::UpdateTriggerMode() {
	mIgnoreEvent = true;
	QString TrgMode = mWaveDump->CommonDevValuesmap.value("/par/AcqTriggerSource");

	if (TrgMode.contains("ITLA", Qt::CaseInsensitive)) {
		ui->cBox_trg_mode->setCurrentText("SELF_TRIGGER");
		ui->pB_trg_mask->setEnabled(true);
	}
	else if (TrgMode.contains("TrgIn", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("EXTERNAL_TRGIN");
	else if (TrgMode.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("EXTERNAL_GPIO");
	else if (TrgMode.contains("TestPulse", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("INTERNAL_PULSER");
	else if (TrgMode.contains("SwTrg", Qt::CaseInsensitive))
		ui->cBox_trg_mode->setCurrentText("SW_TRIGGER_ONLY");
	else
		WarningWidget(ui->cBox_trg_mode, true);
	mIgnoreEvent = false;
}

void CommonConfigPanel::UpdateTrgoutMode() {
	mIgnoreEvent = true;
	
	QString TrgoutMode = mWaveDump->CommonDevValuesmap.value("/par/TrgOutMode");

	if (TrgoutMode.contains("ITLB", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("SELF TRIGGER");
		ui->cBox_TRGOUT->setCurrentText("SELF TRIGGER");
		ui->pB_trgout_mask->setEnabled(true);
	}
	else if (TrgoutMode.contains("Disabled", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("DISABLED");
		ui->cBox_TRGOUT->setCurrentText("DISABLED");
	}
	else if (TrgoutMode.contains("TrgIn", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("TRGIN");
		ui->cBox_TRGOUT->setCurrentText("TRGIN");
	}
	else if (TrgoutMode.contains("SIN", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("SIN");
		ui->cBox_TRGOUT->setCurrentText("SIN");
	}
	else if (TrgoutMode.contains("GPIO", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("GPIO");
		ui->cBox_TRGOUT->setCurrentText("GPIO");
	}
	else if (TrgoutMode.contains("Run", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("RUN");
		ui->cBox_TRGOUT->setCurrentText("RUN");
	}
	else if (TrgoutMode.contains("RefClk", Qt::CaseInsensitive)) {
		ui->cBox_trgout1->setCurrentText("REFCLK");
		ui->cBox_TRGOUT->setCurrentText("REFCLK");
	}
	else {
		WarningWidget(ui->cBox_trgout1, true);
		WarningWidget(ui->cBox_TRGOUT, true);
	}
	mIgnoreEvent = false;
}

void CommonConfigPanel::UpdateVetoSrc() {
	mIgnoreEvent = true;

	QString vetosrc = mWaveDump->CommonDevValuesmap.value("/par/VetoSource");

	if (vetosrc.contains("Disabled", Qt::CaseInsensitive))
		ui->cBox_veto1->setCurrentText("DISABLED");
	else if (vetosrc.contains("SIN", Qt::CaseInsensitive))
		ui->cBox_veto1->setCurrentText("SIN");
	else if (vetosrc.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_veto1->setCurrentText("GPIO");
	else if (vetosrc.contains("LVDS", Qt::CaseInsensitive))
		ui->cBox_veto1->setCurrentText("LVDS");
	else
		WarningWidget(ui->cBox_veto1, true);

	mIgnoreEvent = false;
}

void CommonConfigPanel::UpdateGpioMode() {
	mIgnoreEvent = true;

	QString mode = mWaveDump->CommonDevValuesmap.value("/par/GPIOMode");

	if (mode.contains("Disabled", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("DISABLED");
	else if (mode.contains("SIN", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("SIN");
	else if (mode.contains("GPIO", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("GPIO");
	else if (mode.contains("ITLA", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("SELF TRIGGER");
	else if (mode.contains("TrgIn", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("TRGIN");
	else if (mode.contains("Run", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("RUN");
	else if (mode.contains("Busy", Qt::CaseInsensitive))
		ui->cBox_gpio1->setCurrentText("BUSY");
	else
		WarningWidget(ui->cBox_gpio1, true);

	mIgnoreEvent = false;
}

void CommonConfigPanel::SetPermanentClockDelay() {
	int err;
	QString set_val;
	QString value = QString("%1").arg(ui->sBox_v_clk_delay->value());
	if (err = CommonSetParam("/par/PermanentClockOutDelay", value.toStdString().c_str(), -1, "", &set_val) != 0) {
		mWaveDump->addLog(QString("Set /par/PermanentClockOutDelay error %1").arg(err), false);
		return;
	}
}

void CommonConfigPanel::currentTabChanged(int tab){
	if (mIgnoreEvent)
		return;
	QWidget *currWidget = ui->tabWidget->currentWidget();
	if (currWidget == ui->tab_main) {
		UpdateMainSettings();
	}
	else if (currWidget == ui->tab_advanced) {
		UpdateDevGUI();
	}
}

CommonConfigPanel::~CommonConfigPanel()
{
	delete ui;
}
