/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENparameter.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "CAENparameter.h"
#include "CAENFELibException.h"
#include <CAEN_FELib.h>
#include <QDateTime>
#include <QDebug>
#include <limits>


CAENparameter::CAENparameter(uint64_t pHandle){

	mHandle = pHandle;
	mName = "";
	mPath = "";
	mMin = 0;
	mMax = 1000.;
	mIncr = 0;
	mValue = "unknown";
	mWidget = nullptr;
	mMultiValue = "false";
	mDataType = "unknown";
	mUoM_to_GUIUoM = 1.;
	mUoM_to_GUIUoM_offset = 0.;
	RelatedParams.clear();

	fillInfo();

	fillAllowedValues();	

	getValue();
}

QString CAENparameter::getValue() {
	char val[256] = {0};
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "", val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting value of param " + mName, err);
	}
	
	mValue = QString(val);
	return mValue;
}

void CAENparameter::setValue(QString val) {
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "", qPrintable(val));
	if (err != CAEN_FELib_Success)
		throw CAENFELibException("Error setting value of param " + mName, err);
	getValue();
}

void CAENparameter::fillInfo() {
	char val[256];
	char val1[256];
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	CAEN_FELib_NodeType_t type;

	try {
	    err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetPath(mHandle, val);
		if (err == CAEN_FELib_Success)
			mPath = QString(val);
		else
			throw CAENFELibException("Error getting Parameter Path", err);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetNodeProperties(mHandle, "/", val, &type);
		if (err == CAEN_FELib_Success){
			mName = QString(val);
			mType = type;
		}			
		else
			throw CAENFELibException("Error getting Parameter node properties", err);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/accessmode", val);
		if (err == CAEN_FELib_Success)
			mAccType = QString(val);
		else
			throw CAENFELibException("Error getting accessMode for param " + mName, err);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/dataType", val);
		if (err == CAEN_FELib_Success)
			mDataType = QString(val);
		else
			throw CAENFELibException("Error getting dataType for param " + mName, err);

		if(mDataType == "STRING"){
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/multipleValue", val);
			if (err == CAEN_FELib_Success)
				mMultiValue = QString(val);
			else
				mMultiValue = "false";
		}
		else {
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/UOM", val);
			if (err == CAEN_FELib_Success)
				mUoM = QString(val);
			else
				throw CAENFELibException("Error getting UoM for param " + mName, err);

			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/ExpUOM", val1);
			if (err == CAEN_FELib_Success) {
				mUoMExp = QString(val1).toInt();
			}
			else
				throw CAENFELibException("Error getting ExpUoM for param " + mName, err);
		}
	}
	catch (const CAENFELibException& exc) {
		throw exc;
	}
}

QString CAENparameter::UoMshort(QString uom) {
	if (QString::compare(uom, "seconds", Qt::CaseInsensitive) == 0)
		return "s";
	else if (QString::compare(uom, "samples per second", Qt::CaseInsensitive) == 0)
		return "samples/s";
	else if (QString::compare(uom, "volts", Qt::CaseInsensitive) == 0)
		return "V";
	else if (QString::compare(uom, "celsius", Qt::CaseInsensitive) == 0)
		return "�C";
	else if (QString::compare(uom, "ohm", Qt::CaseInsensitive) == 0)
		return QString::fromUtf8(u8"\u03A9");		
	else if (QString::compare(uom, "bytes", Qt::CaseInsensitive) == 0)
		return "B";
	else if (QString::compare(uom, "ampere", Qt::CaseInsensitive) == 0)
		return "A";
	else if (QString::compare(uom, "hertz", Qt::CaseInsensitive) == 0)
		return "Hz";
	else if (QString::compare(uom, "percent", Qt::CaseInsensitive) == 0)
		return "%";
	else if (QString::compare(uom, "ADCcounts", Qt::CaseInsensitive) == 0)
		return "LSB";
	else
		return uom;
}

QString CAENparameter::GetUoMString(QString uom, int exp) {
	QString uomstring = "";
	switch (exp) {
	case -9:
		uomstring = "n";
		break;
	case -6:
		uomstring = QChar(0xbc, 0x03);
		break;
	case -3:
		uomstring = "m";
		break;
	case 0:
		break;
	case 3:
		uomstring = "k";
		break;
	case 6:
		uomstring = "M";
		break;
	case 9:
		uomstring = "G";
		break;
	default:
		break;
	}
	uomstring.append(UoMshort(uom));
	return uomstring;
}

void CAENparameter::fillAllowedValues() {
	CAEN_FELib_ErrorCode err;
	char val[256];
	mValues.clear();
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/allowedValues", val);
	if (err == CAEN_FELib_Success) {//for list parameter fill allowed values vector
		if (atoi(val)) {
			uint64_t hallowed[32];
			int nvalues = (CAEN_FELib_ErrorCode)CAEN_FELib_GetChildHandles(mHandle, "/allowedValues/", hallowed, 32);
			if (nvalues > 0) {
				for (int i = 0; i < nvalues; i++) {
					err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(hallowed[i], "", val);
					if (err != CAEN_FELib_Success)
						break;
					mValues.append(QString(val));
				}
			}
			else
				throw CAENFELibException("Error getting allowed values for param " + mName, err);
		}
	}
	else {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/minValue", val);
		if (err != CAEN_FELib_Success) {
			if (mDataType != "STRING")
				throw CAENFELibException("Error getting minValue for param " + mName, err);
			else
				mMin = 0;
		}
		else
			mMin = QString(val).toDouble();
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/maxValue", val);
		if (err != CAEN_FELib_Success) {
			if (mDataType != "STRING")
				throw CAENFELibException("Error getting maxValue for param " + mName, err);
			else
                mMax = 9999999999999;
		}
		else
			mMax = QString(val).toDouble();
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/increment", val);
		if (err != CAEN_FELib_Success) {
			if (mDataType != "STRING")
				throw CAENFELibException("Error getting increment for param " + mName, err);
			else {
				mIncr = 1;
				mIncrStr = "1";
			}
		}
		else {
			mIncr = QString(val).toDouble();
			mIncrStr = QString(val);
		}
	}

}

double CAENparameter::getScaleParamValue() {
	char val[256] = { 0 };
	CAEN_FELib_ErrorCode err= CAEN_FELib_Success;
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mScaleParamHandle,"", val);
	if (err != CAEN_FELib_Success) {
		char error[1024];
		CAEN_FELib_GetLastError(error);
		throw CAENFELibException("Error getting value of param " + mName, err);
	}
	QString result;
	QString v = QString(val);
	for (const QChar c : qAsConst(val)) {
		if (c.isDigit()) {
			result.append(c);
		}
	}
	return result.toDouble();
}

void CAENparameter::SetWidget(QWidget *widget) {
	mWidget = widget;
}

void CAENparameter::UpdateWidget() {
	if (mWidget == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);
	try {
		if (QComboBox *cbox = qobject_cast<QComboBox *>(mWidget))
			updateComboBox(cbox);
		else if (QSpinBox *spinBox = qobject_cast<QSpinBox *>(mWidget))
			updateSpinBox(spinBox);
		else if (QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(mWidget))
			updateDoubleSpinBox(doubleSpinBox);
		else if (QGroupBox *groupBox = qobject_cast<QGroupBox *>(mWidget))
			updateGroupBox(groupBox);
		else if (QLineEdit *lineEdit = qobject_cast<QLineEdit *>(mWidget))
			updateLineEdit(lineEdit);
		else if (QListWidget *listWidget = qobject_cast<QListWidget *>(mWidget))
			updateListWidget(listWidget);
		else if (QCheckBox* cBox = qobject_cast<QCheckBox*>(mWidget))
			updateCheckBox(cBox);
	
		QPalette palette = mWidget->palette();
		palette.setColor(QPalette::Text, Qt::black);
		mWidget->setPalette(palette);
	}
	catch (const CAENFELibException& exc) {
		mWidget->setEnabled(false);
		throw exc;
	}


}

void CAENparameter::updateComboBox(QComboBox *cbox) {
	if (cbox == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);
	if(mDataType!= "STRING")
		throw CAENFELibException("Parameter" + mName + " of type" + mDataType + "not supported by comboBox", CAEN_FELib_GenericError);
	cbox->blockSignals(true);
	cbox->clear();
	const QString pvalue = mValue;
	int index = -1;
	for (int i = 0; i < mValues.size(); i++) {
		cbox->addItem(mValues.at(i));
		if (pvalue == mValues.at(i))
			index = (int)i;
	}
	cbox->setCurrentIndex(index);
	cbox->blockSignals(false);
}


void CAENparameter::updateSpinBox(QSpinBox *sbox) {
	if (sbox == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);
	if(mDataType != "NUMBER")
		throw CAENFELibException("Parameter" + mName + " of type" + mDataType + "not supported by spinBox", CAEN_FELib_GenericError);

	QString pvalue = mValue;
	if (mHasScaleFactor && (mUoM_to_GUIUoM != 1 || mUoM_to_GUIUoM_offset != 0)) {
		double scaled_value = getScaleParamValue() * mValue.toDouble();
		pvalue = QString("%1").arg(scaled_value);
	}

	sbox->blockSignals(true);
	sbox->setMinimum((int)mMin  * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
	sbox->setMaximum((int)mMax  * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
	sbox->setSingleStep((int)mIncr  * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
	sbox->setValue(qRound(pvalue.toDouble() * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset));
	sbox->blockSignals(false);
}

void CAENparameter::updateDoubleSpinBox(QDoubleSpinBox *dbox) {
	if (dbox == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);
	if(mDataType != "NUMBER")
		throw CAENFELibException("Parameter" + mName + " of type" + mDataType + "not supported by doublespinBox", CAEN_FELib_GenericError);

	QString pvalue = mValue;
	if (mHasScaleFactor && (mUoM_to_GUIUoM != 1 || mUoM_to_GUIUoM_offset!= 0)) {
		double scaled_value = getScaleParamValue() * mValue.toDouble();
		pvalue = QString("%1").arg(scaled_value);
	}
	dbox->blockSignals(true);
	dbox->setMinimum(mMin * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
	dbox->setMaximum(mMax * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
	dbox->setSingleStep(mIncr * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
	mIncrStr = QString("%1").arg(mIncr * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
	int i = mIncrStr.lastIndexOf('.');
	int decimals = (i > 0) ? 2 : 0;
	dbox->setDecimals(decimals);
	dbox->setValue(pvalue.toDouble() * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
	dbox->blockSignals(false);
}

void CAENparameter::updateGroupBox(QGroupBox *gbox) {
	if (gbox == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);

	const QString pvalue = mValue;

	gbox->blockSignals(true);
	if (gbox->isCheckable())
		gbox->setChecked(pvalue == "True");
	else
		gbox->setEnabled(pvalue == "False");
	gbox->blockSignals(false);
}

void CAENparameter::updateCheckBox(QCheckBox* cbox) {
	if (cbox == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);

	const QString pvalue = mValue;

	cbox->blockSignals(true);
	if((pvalue == "TRUE") || (pvalue == "FALSE"))
		cbox->setChecked(pvalue == "TRUE");
	else
		cbox->setChecked(pvalue != "1");
	cbox->blockSignals(false);

}

void CAENparameter::updateLineEdit(QLineEdit *ledit) {
	if (ledit == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);
	if(mDataType != "STRING")
		throw CAENFELibException("Parameter" + mName + " of type" + QString("%1").arg(mType) + " not supported by lineEdit", CAEN_FELib_GenericError);
	bool ok = false;

	if (mName.contains("mask", Qt::CaseInsensitive)) {
		QString h;
		h.setNum(mValue.toULongLong(), 16);
		ledit->setText(h);
	}
	else {
		const QString pvalue = QString("%1").arg(mValue.toDouble() * mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset);
		ledit->setText(pvalue);
	}
}

void CAENparameter::updateListWidget(QListWidget *lwidget) {
	if (lwidget == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);
	if(mDataType != "STRING")
		throw CAENFELibException("Parameter" + mName + " of type" + QString("%1").arg(mType) + "not supported by listWidget", CAEN_FELib_GenericError);
	lwidget->clear();
	for (int val = 0; val < mValues.size(); val ++) {
		QListWidgetItem *newItem = new QListWidgetItem;
		newItem->setText(mValues.at(val));
		newItem->setFlags(newItem->flags() | Qt::ItemIsUserCheckable);
		newItem->setTextAlignment(Qt::AlignCenter);
		if(mValue.contains(mValues.at(val)))
			newItem->setCheckState(Qt::Checked);
		else
			newItem->setCheckState(Qt::Unchecked);
		lwidget->insertItem(val, newItem);
	}
	lwidget->setMaximumWidth(lwidget->sizeHintForColumn(0) + 5 * lwidget->frameWidth());
}


void CAENparameter::SetValueFromWidget() {
	if (mWidget == nullptr)
		throw CAENFELibException("Parameter" + mName + " not supported by the device", CAEN_FELib_GenericError);
	if (mAccType == "READ_WRITE") {
		if (QComboBox *cbox = qobject_cast<QComboBox *>(mWidget))
			setValuefromComboBox(cbox);
		else if (QSpinBox *spinBox = qobject_cast<QSpinBox *>(mWidget))
			setValuefromSpinBox(spinBox);
		else if (QDoubleSpinBox *doubleSpinBox = qobject_cast<QDoubleSpinBox *>(mWidget))
			setValuefromDoubleSpinBox(doubleSpinBox);
		else if (QGroupBox *groupBox = qobject_cast<QGroupBox *>(mWidget))
			setValuefromGroupBox(groupBox);
		else if (QLineEdit *lineEdit = qobject_cast<QLineEdit *>(mWidget))
			setValuefromLineEdit(lineEdit);
		else if (QListWidget *listWidget = qobject_cast<QListWidget *>(mWidget))
			setValuefromListWidget(listWidget);
		else if (QCheckBox *cBox = qobject_cast<QCheckBox *>(mWidget))
			setValuefromCheckBox(cBox);
	}
}

void CAENparameter::setValuefromCheckBox(QCheckBox* cbox) {
	if (cbox == nullptr)
		return;
	if(cbox->isChecked())
		setValue("TRUE");
	else
		setValue("FALSE");
}

void CAENparameter::setValuefromComboBox(QComboBox *cbox) {
	if (cbox == nullptr)
		return;
	QString valueCodename = cbox->currentText();
	if (valueCodename.isEmpty())
		return;
	setValue(valueCodename);
}

void CAENparameter::setValuefromSpinBox(QSpinBox *sbox) {
	if (sbox == nullptr)
		return;
	if(mHasScaleFactor && (mUoM_to_GUIUoM != -1 || mUoM_to_GUIUoM_offset != 0))
		setValue(QString("%1").arg(qRound((static_cast<double>(sbox->value()) - mUoM_to_GUIUoM_offset) / (mUoM_to_GUIUoM * getScaleParamValue()))));
	else
		setValue(QString("%1").arg(qRound((static_cast<double>(sbox->value()) - mUoM_to_GUIUoM_offset) / mUoM_to_GUIUoM)));
}

void CAENparameter::setValuefromDoubleSpinBox(QDoubleSpinBox *dbox) {
	if (dbox == nullptr)
		return;
	if (mHasScaleFactor && (mUoM_to_GUIUoM != -1 || mUoM_to_GUIUoM_offset !=0))
		setValue(QString("%1").arg((dbox->value() - mUoM_to_GUIUoM_offset) / (mUoM_to_GUIUoM * getScaleParamValue())));
	else
		setValue(QString("%1").arg((dbox->value() - mUoM_to_GUIUoM_offset) / mUoM_to_GUIUoM ));
}

void CAENparameter::setValuefromGroupBox(QGroupBox *gbox) {
	if (gbox == nullptr)
		return;
	QString val;
	if (gbox->isCheckable())
		val = gbox->isChecked() ? "TRUE" : "FALSE";
	else
		val = gbox->isEnabled() ? "TRUE" : "FALSE";
	setValue(val);
}

void CAENparameter::setValuefromLineEdit(QLineEdit *ledit) {
	if (ledit == nullptr)
		return;
	if (mDataType == "NUMBER")
		setValue(QString("%1").arg(ledit->text().toDouble() / mUoM_to_GUIUoM + mUoM_to_GUIUoM_offset));
	else {
		if (mName.contains("mask", Qt::CaseInsensitive))
			setValue("0x" + ledit->text());
		else
			setValue(ledit->text());

	}
}

void CAENparameter::setValuefromListWidget(QListWidget *lwidget) {
	if (lwidget == nullptr)
		return;
	QString val = "";
	bool check_ok = false;
	for (int i = 0; i < lwidget->count(); ++i){
		if (lwidget->item(i)->checkState() == Qt::Checked) {
			check_ok = true;
			if (val == "")
				val.append(lwidget->item(i)->text());
			else
				val.append("|" + lwidget->item(i)->text());
		}		
	}
	if (check_ok) {
		if (val.contains("Disabled") && !mValue.contains("Disabled"))
			setValue("Disabled");
		else
			setValue(val);
	}
	else {
		if(mMultiValue == "true" && mValues.contains("Disabled"))
			setValue("Disabled");
	}
}

CAENparameter::~CAENparameter()
{
	RelatedParams.clear();
}
