/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENDig2Device.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "CAENDig2Device.h"
#include "CAENFELibException.h"
#include <CAEN_FELib.h>
#include <QDebug>
#include <QVector>
#include <QtMath>

CAENDig2Device::CAENDig2Device(QString name, QString CType, QString address, uint64_t handle, bool isvirtual) :
	CAENDevice(name, CType, address, handle, isvirtual)
{
	DeviceClass = CAEN_DIG2;
	if (!mIsVirtualDevice) {
		try {
			uint32_t errors = getErrorFlags();
			if (errors) {
				ErrorsReported = true;
			}

			FillInfo();
			CreateParams();
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}

void CAENDig2Device::FillInfo(){
	char val[256] = { 0 };
	char val1[256] = { 0 };
	CAEN_FELib_ErrorCode err;
	const size_t handles_size = 1024;
	uint64_t handles[handles_size];
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/NumCh", val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting number of channels for " + mName, err);
	}
	mNumCh = atoi(val);
	ChBaselines.resize(mNumCh);
	mChThreshold.resize(mNumCh);
	updateBaselines();

	TSamplUnit_ns = TSAMPL_UNIT_NS_DIG2;

	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ADC_SamplRate", val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting sampling rate for " + mName, err);
	}
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ADC_SamplRate/ExpUoM", val1);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting sampling rate for " + mName, err);
	}
    mSample_to_S = 1 / (QString::fromLatin1(val).toDouble() * qPow(10, atoi(val1)));
	NJExtraSamples = TSamplUnit_ns / (mSample_to_S *1e9);
	if (NJExtraSamples == 1)
		NJExtraSamples = 0;

	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ADC_Nbit", val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting ADC nbit for " + mName, err);
	}
	mADC_NBit = atoi(val);

	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/InputRange", val1);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting input range for " + mName, err);
	}
	mInputRange = QString::fromLatin1(val1).toDouble();

	for (int i = 0; i < mNumCh; i++){
		QString qry = QString("/ch/%1/par/GainFactor").arg(i);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), val1);
		if (err != CAEN_FELib_Success)
			mChADCGain.push_back(1.0);
		else
			mChADCGain.push_back(QString(val1).toFloat());

		qry = QString("/ch/%1/par/ADCToVolts").arg(i);
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), val1);
		if (err != CAEN_FELib_Success)
			mChADCToVolts.push_back(1.);
		else
			mChADCToVolts.push_back(QString(val1).toDouble());
	}


	mModel = getFamilyCode();
	if (mModel == "2745") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetChildHandles(mHandle, "/vga/", handles, handles_size);
		if (err < 0 || err > handles_size) {
			throw CAENFELibException("Error getting VgaGain for " + mName, err);
		}
		else
			mVgaGainGroupSize = mNumCh / err;

		for (int i = 0; i < mNumCh; i++) {
			int g = i % mVgaGainGroupSize; //group number to which this channel belongs to for this specific parameter 
			QString qry = QString("/vga/%1/par/VgaGain").arg(g);
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), val1);
			if (err != CAEN_FELib_Success)
				ChGain.push_back(1.0);
			else
				ChGain.push_back(std::pow(10., QString(val1).toDouble() / 20.));
		}
	}
	else if ((mModel == "2730") || (mModel == "2751")) {
		for (int i = 0; i < mNumCh; i++) {
			QString qry = QString("/ch/%1/par/ChGain").arg(i);
			err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), val1);
			if (err != CAEN_FELib_Success)
				ChGain.push_back(1.0);
			else
				ChGain.push_back(std::pow(10., QString(val1).toDouble() / 20.));
		}
	}
	else {
		for (int i = 0; i < mNumCh; i++)
			ChGain.push_back(1.0);
	}
	updateGains();

	mWHandle = getWHandle();

	mRawHandle = getRawHandle();

	mSN = getSN();
}

QString CAENDig2Device::getFormFactor() {
	if (mIsVirtualDevice)
		return "";
	return "";
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/FormFactor", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting form factor of " + mName, err);
	}
	int f = QString(value).toInt();
	switch (f) {
	case 0:
		return "VME";
	case 1:
		return "VME64X";
	case 2:
		return "DT";
	default:
		return "unknown";
	}
}

void CAENDig2Device::CreateParams() {
	uint64_t phandle;
	try {
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, "/par/RecordLengthT", &phandle);
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error creating recordlen param for " + mName, err);
		}
		mReclen = new CAENparameter(phandle);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, "/par/PreTriggerT", &phandle);
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error creating pretrigger param for " + mName, err);
		}
		mPreTrg = new CAENparameter(phandle);
	}
	catch (const CAENFELibException& exc) {
		throw exc;
	}

}

void CAENDig2Device::ApplyDefaultConfig(QHash<QString, QString> dpars, QHash<QString, QString> *cpars, QHash<QString, QString>* gpars) {
	CAEN_FELib_ErrorCode ret = CAEN_FELib_Success;
	char family[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/FamilyCode", family);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting family code of " + mName, err);
	}
	for (auto item = dpars.begin(); item != dpars.end(); ++item) {
		if (item.key() == "/TRIGGER_MODE")
			continue;
		if (item.key().contains("ClockOutDelay", Qt::CaseInsensitive))
			continue;
		if ((mModel == "2730" || mModel == "2751") && item.key().contains("Decimation",Qt::CaseInsensitive))
			continue;
		if (item.key().contains("RecordLengthS")) {
			int s = item.value().toInt();
			s += NJExtraSamples;
			int incr = mReclen->getIncr();
			s = ((s + incr - 1) / incr) * incr;
			RecLenJIncrS = (s - item.value().toInt()) / (mSample_to_S * 1e9); //real increment considering minimum step
			
			item.value() = QString("%1").arg(s);
		}
		ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(item.key()), qPrintable(item.value()));
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting" + item.key() + " param value for " + mName, ret);
		}
	}
	for (int ch = 0; ch < mNumCh; ch++) {
		for (auto item = cpars[ch].begin(); item != cpars[ch].end(); ++item) {
			auto x = *item;		
			if (mModel != "2730" && mModel != "2751" && item.key().contains("ChGain"))
				continue;
			ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(item.key()), qPrintable(item.value()));
			if (ret != CAEN_FELib_Success) {
				throw CAENFELibException("Error setting" + item.key() + " param value for " + mName, ret);
			}
		}
	}
	if (mVgaGainGroupSize > 1) {
		for (int g = 0; g < mNumCh / mVgaGainGroupSize; g++) {
			for (auto item = gpars[g].begin(); item != gpars[g].end(); ++item) {
				auto x = *item;
				ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(item.key()), qPrintable(item.value()));
				if (ret != CAEN_FELib_Success) {
					throw CAENFELibException("Error setting " + item.key() + " param value for " + mName, ret);
				}
			}
		}
	}
	//update the baselines if modified
	updateBaselines();
	//updateInputGains if modified
	updateGains();

	//update cached values for params
	mReclen_valS = mReclen->getValue().toUInt() / (1e9 * mSample_to_S) - RecLenJIncrS;
	mPreTrg_valS = mPreTrg->getValue().toUInt() / (1e9 * mSample_to_S);
	//debug
	//ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/par/RegisterAcqCh", "0x000808=0"); //disable the veto for self trigger
	//ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/par/RegisterAcqCh", "0x000800=0"); //disable the veto for self trigger
}

void CAENDig2Device::updateBaselines() {
	if (mIsVirtualDevice)
		return;
	char value[32];
	for (int ch = 0; ch < mNumCh; ch++) {
		QString qry = QString("/ch/%1/par/DCOffset").arg(ch);
		CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error getting param value for " + mName, ret);
		}
		double val = QString::fromLatin1(value).toDouble();
		ChBaselines.replace(ch, val);
	}
}

double CAENDig2Device::getReclen_ns() {
	if(mIsVirtualDevice)
		return mReclen_valS * mSample_to_S * 1e9;
	double val = mReclen->getCachedValue().toDouble();
	val -= RecLenJIncrS * mSample_to_S * 1e9;
	return val;
}

double CAENDig2Device::getReclen_us() {
	double val = 0;
	if (mIsVirtualDevice)
		return mReclen_valS * mSample_to_S * 1e6;
	if (mReclen != nullptr) {
		val = getReclen_ns();
		val /= 1000.;
		if (mModel != "2730" && mModel != "2751") {
			char value[256] = { 0 };
			CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/DecimationFactor"), value);
			if (ret != CAEN_FELib_Success) {
				throw CAENFELibException("Error getting param value for " + mName, ret);
				return -1;
			}
			val *= QString(value).toInt();
		}
	}
	return val;
}

double CAENDig2Device::getChOffset(int ch) const{
	if (mIsVirtualDevice)
		return ChBaselines.at(ch);
	QString qry = QString("/ch/%1/par/DCOffset").arg(ch);
	char value[32];
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting param value for " + mName, ret);
	}
	double val = QString::fromLatin1(value).toDouble();
	ChBaselines.replace(ch,val);
	return val;
}

void CAENDig2Device::updateChSample_to_V(int ch) {
    QString qry = QString("/ch/%1/par/ADCToVolts").arg(ch);
    char value[32]={0};
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting param value for " + mName, err);
    }
    QString vs = QString::fromLatin1(value);
    double val = vs.toDouble();
	mChADCToVolts.replace(ch,val);
}

double CAENDig2Device::getChGain(int ch){
	if (mIsVirtualDevice)
		return ChGain.at(ch);
	int vga_ch;
	QString qry;
	if (mModel == "2745") {//not supported if 0		
		vga_ch = ch % mVgaGainGroupSize;
		qry = QString("/vga/%1/par/VgaGain").arg(vga_ch);
	}
	else if (mModel == "2730" || mModel == "2751") {
		qry = QString("/ch/%1/par/ChGain").arg(ch);
	}
	else
		return 1.0;

	char value[32];
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
	if (ret != CAEN_FELib_Success) {
		// default value
		return 1.;
	}
	double val_db = QString::fromLatin1(value).toDouble();
	double val = std::pow(10., val_db / 20.);
	ChGain.replace(ch, val);

	updateChSample_to_V(ch);

	return ChGain.at(ch);
}

void CAENDig2Device::setChGain(int ch, double value) {
	int vga_ch;
	QString qry;
	if(mModel == "2745") {
	 vga_ch = ch % mVgaGainGroupSize;
	 qry = QString("/vga/%1/par/VgaGain").arg(vga_ch);
	}
	else if(mModel == "2730" || mModel == "2751")
		qry = QString("/ch/%1/par/ChGain").arg(ch);
	double val_to_set = std::log10(value * 20.);
	QString val = QString("%1").arg(val_to_set);
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(val));
	if (ret != CAEN_FELib_Success) {
		// no-op
		return;
	}
	ChGain.replace(ch, value);

	updateChSample_to_V(ch);
}

void CAENDig2Device::setChOffset(int ch, double value) {
	QString qry = QString("/ch/%1/par/DCOffset").arg(ch);
	QString val = QString("%1").arg(value);
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(val));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting dcoffset param value for " + mName, ret);
	}
}

int CAENDig2Device::getChThreshold(int ch){
	QString qry = QString("/ch/%1/par/TriggerThr").arg(ch);
	char value[32];
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting param value for " + mName, ret);
	}
	mChThreshold.replace(ch, QString::fromLatin1(value).toDouble());
	return atoi(value);
}

void CAENDig2Device::setChThreshold(int ch, int val) {
	QString qry = QString("/ch/%1/par/TriggerThr").arg(ch);
	QString value = QString("%1").arg(val);
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting" + qry + "param value for " + mName, ret);
	}
	mChThreshold.replace(ch,static_cast<double>(val));
}

CAENparameter *CAENDig2Device::getThrParam(int ch) {
	uint64_t phandle;
	QString qry = QString("/ch/%1/par/TriggerThr").arg(ch);
	CAENparameter *thr = nullptr;
	try {
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, qPrintable(qry), &phandle);
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error creating thr param for " + mName, err);
		}
		thr = new CAENparameter(phandle);
	}
	catch (const CAENFELibException& exc) {
		throw exc;
	}
	return thr;
}

void CAENDig2Device::setThrMode(QString mode) {
	QString qry = QString("/ch/0..%1/par/TriggerThrMode").arg(mNumCh -1);
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(mode));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting" + qry + " param value for " + mName, ret);
	}
}

void CAENDig2Device::setStartSource(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/StartSource"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting startsource param value for " + mName, ret);
	}
	if (value.contains("SINlevel", Qt::CaseInsensitive)) {
		ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/EnAutodisarmAcq"), "False");
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting EnAutodisarmAcq param value for " + mName, ret);
		}
	}
	else {
		ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/EnAutodisarmAcq"), "True");
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting EnAutodisarmAcq param value for " + mName, ret);
		}
	}
}
QString CAENDig2Device::getStartSource() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/StartSource", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting start source of " + mName, err);
	}
	return QString(value);
}

void CAENDig2Device::setRunDelay(int value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/RunDelay"), qPrintable(QString("%1").arg(value)));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting rundelay param value for " + mName, ret);
	}
}

void CAENDig2Device::setGPIOMode(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/GPIOMode"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting gpiomode param value for " + mName, ret);
	}
}
QString CAENDig2Device::getGPIOMode() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/GPIOMode", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting GPIO mode of " + mName, err);
	}
	return QString(value);
}

void CAENDig2Device::setTRGOUTMode(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/TRGOUTMode"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting trgoutmode param value for " + mName, ret);
	}
}
QString CAENDig2Device::getTRGOUTMode() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/TRGOUTMode", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting TRGOUT mode of " + mName, err);
	}
	return QString(value);
}

void CAENDig2Device::setSyncOutMode(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/SyncOutMode"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error settingsyncoutmode param value for " + mName, ret);
	}
}

QString CAENDig2Device::getClkOutFP() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/EnClockOutFP", value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting EnClockOutFP param value for " + mName, ret);
	}
	return QString(value);
}
void CAENDig2Device::setClkOutFP(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/EnClockOutFP"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting EnClockOutFP param value for " + mName, ret);
	}
}
QString CAENDig2Device::getClkSource() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ClockSource", value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting ClockSource param value for " + mName, ret);
	}
	return QString(value);
}
void CAENDig2Device::setClkSource(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/ClockSource"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting ClockSource param value for " + mName, ret);
	}
}

void CAENDig2Device::setTrgSource(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/AcqtriggerSource"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting AcqtriggerSource param value for " + mName, ret);
	}
	mTrgSource = value;
}
QString CAENDig2Device::getTrgSource() {
	if (mIsVirtualDevice)
		return mTrgSource;
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/AcqtriggerSource", value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting AcqtriggerSource param value for " + mName, ret);
	}
	mTrgSource = QString(value);
	return QString(value);
}

uint32_t CAENDig2Device::calcTrgSourceToSave() {
	QString ts = getTrgSourceCachedValue();
	uint32_t code = 0;
	if (ts.contains("TrgIn", Qt::CaseInsensitive))
		code |= TSource_TrgIn;
	if (ts.contains("P0", Qt::CaseInsensitive))
		code |= TSource_P0;
	if (ts.contains("SwTrg", Qt::CaseInsensitive))
		code |= TSource_SwTrg;
	if (ts.contains("LVDS", Qt::CaseInsensitive))
		code |= TSource_LVDS;
	if (ts.contains("EncodedClkIn", Qt::CaseInsensitive))
		code |= TSource_EncodedClkIn;
	if (ts.contains("GPIO", Qt::CaseInsensitive))
		code |= TSource_GPIO;
	if (ts.contains("TestPulse", Qt::CaseInsensitive))
		code |= TSource_TestPulse;
	if (ts.contains("ITLA_AND_ITLB", Qt::CaseInsensitive))
		code |= TSource_ITLA_AND_ITLB;
	else if (ts.contains("ITLA_OR_ITLB", Qt::CaseInsensitive))
		code |= TSource_ITLA_OR_ITLB;
	else if (ts.contains("ITLA", Qt::CaseInsensitive))
		code |= TSource_ITLA;
	else if (ts.contains("ITLB", Qt::CaseInsensitive))
		code |= TSource_ITLB;
	return code;
}

void CAENDig2Device::setSelfTrgMask(QString val){
	setTrgITLAMask(val);
	setITLALogicOR();
}
QString CAENDig2Device::getSelfTrgMask() {
	return getTrgITLAMask();
}

QString CAENDig2Device::getTrgOutMask() {
	return getTrgITLBMask();
}

void CAENDig2Device::setITLALogicOR() {
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/par/ITLAMainLogic", "OR");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting ITLAMainLogic param value for " + mName, err);
	}
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/par/ITLAPairLogic", "None");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting ITLAPairLogic param value for " + mName, err);
	}
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/par/ITLAGateWidth", "16");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting ITLAGateWidth param value for " + mName, err);
	}
}

void CAENDig2Device::setTrgITLAMask(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/ITLAMask"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting ITLAMask param value for " + mName, ret);
	}
}

void CAENDig2Device::setTrgITLBMask(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/ITLBMask"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting ITLBMask param value for " + mName, ret);
	}
}

QString CAENDig2Device::getTrgITLAMask() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/ITLAMask"), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting ITLAMask param value for " + mName, ret);
	}
	return QString(value);
}

QString CAENDig2Device::getTrgITLBMask() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/ITLBMask"), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting ITLBMask param value for " + mName, ret);
	}
	return QString(value);
}

void CAENDig2Device::setDecimation(int val) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/DecimationFactor"), qPrintable(QString("%1").arg(val)));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting DecimationFactor param value for " + mName, ret);
	}
}
int CAENDig2Device::getDecimation() {
	if (mIsVirtualDevice)
		return mDecim;
	if (mModel == "2730" || mModel == "2751") {
		return 1;
	}
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/DecimationFactor"), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting param value for " + mName, ret);
		return -1;
	}
	return QString(value).toInt();
	return 1;
}

void CAENDig2Device::StartAcquisition() {
	ArmAcquisition();
	QString smode = getStartSource();
	if (smode.contains("SwCmd", Qt::CaseInsensitive)) {
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/SwStartAcquisition");
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error starting acquisition for " + mName, err);
		}
	}
}

void CAENDig2Device::StopAcquisition() {
	if (ErrorsReported)
		return;
	QString smode = getStartSource();
	if (smode.contains("SwCmd", Qt::CaseInsensitive)) {
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/SwStopAcquisition");
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error stopping acquisition for " + mName, err);
		}
	}
	DisarmAcquisition();
}

void CAENDig2Device::Reboot() {
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/Reboot");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error in reboot for " + mName, err);
	}
}

QString CAENDig2Device::getAcqStatus() {
	if (mIsVirtualDevice)
		return mVirtualAcqStatus;
	char value[256] = { 0 };
	QString acq_status="";
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/AcquisitionStatus", value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting AcquisitionStatus param value for " + mName, ret);
		return "";
	}
	uint32_t status = atoi(value);
	if (status & (1 << ACQ_STATUS_BUSY_BIT))//give priority to bust state
		acq_status="Busy";
	else if (status & (1 << ACQ_STATUS_RUN_BIT))
		acq_status = "Running";
	else if (status & (1 << ACQ_STATUS_ARMED_BIT))
		acq_status = "Armed";
	else
		acq_status = "Ready";
	return acq_status;
}

void CAENDig2Device::SetDataFormat(QString format) {
	CAEN_FELib_ErrorCode err;
	if (format != "RAW") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/endpoint/par/activeendpoint", "scope");
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting data format for " + mName, err);
		}
		QString data_format = "[  { \"name\" : \"TIMESTAMP\", \"type\" : \"U64\" }, \
							{ \"name\" : \"TRIGGER_ID\", \"type\" : \"U32\" }, \
							{ \"name\" : \"WAVEFORM\", \"type\" : \"U16\", \"dim\" : 2 }, \
							{ \"name\" : \"WAVEFORM_SIZE\", \"type\" : \"U32\", \"dim\" : 1 },	\
							{ \"name\" : \"EVENT_SIZE\", \"type\" : \"U32\" },  \
							{ \"name\" : \"FLAGS\", \"type\" : \"U16\" } ]";
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetReadDataFormat(mWHandle, qPrintable(data_format));
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting data format for " + mName, err);
		}
	}
	else {
		EnableRawDataFormat();
	}
}

QString CAENDig2Device::getFWGbE() {
	if (mIsVirtualDevice)
		return "";
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/SFPLinkProtocol", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting FW GbE of " + mName, err);
	}
	return QString(value);
}
QString CAENDig2Device::getFWRel() {
	if (mIsVirtualDevice)
		return "";
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/CupVer", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting FW rel of " + mName, err);
	}
	return QString(value);
}

uint32_t CAENDig2Device::getErrorFlags(){
	if (mIsVirtualDevice)
		return 0;
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ErrorFlags", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting Error Flags of " + mName, err);
	}
	uint32_t num=QString(value).toUInt();
	return num;
}

uint64_t CAENDig2Device::getChEnableMask() {
	uint64_t mask = 0;
	for (int ch = 0; ch < mNumCh; ch++) {
		if (isChEnabled(ch))
			mask |= (1LL << ch);
	}
	return mask;
}

bool CAENDig2Device::isChEnabled(int ch) {
	if (mIsVirtualDevice)
		return (mChMask & ((uint64_t)1 << ch)) ? true : false;
	if (ch >= mNumCh)
		return false;
	char value[256] = { 0 };
	QString qry = QString("/ch/%1/par/ChEnable").arg(ch);
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qry.toStdString().c_str(), value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting enable status of channel ", err);
	}
	if (QString(value) == "True")
		return true;
	else
		return false;
}

QString CAENDig2Device::DecodeFlagsMask(uint32_t mask) {
	QString message;
	if(mask == 0)
		return "No errors reported by device " + mName;
	else
		message = "Errors reported by device " + mName + ":\n";
	if (mask & FLAGS_POWERFAIL)
		message.append("Power Failure\n");
	if (mask & FLAGS_INITFAULT)
		message.append("Init Failure\n");
	if (mask & FLAGS_SI5341_unlock)
		message.append("SI5341 PLL Unlock\n");
	if (mask & FLAGS_SI5395_unlock)
		message.append("SI5395 PLL Unlock\n");
	if (mask & FLAGS_LMK04832_unlock)
		message.append("LMK04832 PLL Unlock\n");
	if (mask & FLAGS_JESD_unlock)
		message.append("JESD204B Unlock\n");
	if (mask & FLAGS_DDR_PL_BANK0_CALIB_FAIL)
		message.append("DDR4 PL Bank0 Calibration Failure\n");
	if (mask & FLAGS_DDR_PL_BANK1_CALIB_FAIL)
		message.append("DDR4 PL Bank1 Calibration Failure\n");
	if (mask & FLAGS_DDR_PS_CALIB_FAIL)
		message.append("DDR4 PS Calibration Failure\n");
	if (mask & FLAGS_FPGA_CONFIG_FAIL)
		message.append("FPGA Configuration Failure\n");
	if (mask & FLAGS_BIC_ERROR)
		message.append("BIC Check Error\n");
	if (mask & FLAGS_ADC_OVT)
		message.append("ADC OverTemperature\n");
	if (mask & FLAGS_AIR_OVT)
		message.append("AIR Outlet OverTemperature\n");
	if (mask & FLAGS_FPGA_OVT)
		message.append("FPGA OverTemperature\n");
	if (mask & FLAGS_DCDC_OVT)
		message.append("DC/DC OverTemperature\n");
	if (mask & FLAGS_CLKIN_MISS)
		message.append("Clock In Missing\n");
	if (mask & FLAGS_ADC_SHUTDOWN)
		message.append("ADC ShutDown\n");

	return message;
}

QString CAENDig2Device::getTAirIn() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/TempSensAirIn", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting Temperature air in of " + mName, err);
	}
	return QString(value);
}
QString CAENDig2Device::getTAirOut() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/TempSensAirOut", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting Temperature air out of " + mName, err);
	}
	return QString(value);
}
QString CAENDig2Device::getTCore() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/TempSensCore", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting Temperature Core of " + mName, err);
	}
	return QString(value);
}
QString CAENDig2Device::getTFirstADC() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/TempSensFirstADC", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting Temperature first ADC of " + mName, err);
	}
	return QString(value);
}
QString CAENDig2Device::getTLastADC() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/TempSensLastADC", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting Temperature last ADC of " + mName, err);
	}
	return QString(value);
}
QString CAENDig2Device::getTDCDC(){
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/TempSensDCDC", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting Temperature DCDC of " + mName, err);
	}
	return QString(value);
}

void CAENDig2Device::DefinePreTrigger(uint32_t p) {
	mPreTrg_valS = p;
}


void CAENDig2Device::DefineTrgSource(uint32_t t) {
	QString tr = "";
	if (t & TSource_TrgIn)
		tr.append("|TrgIn");
	if (t & TSource_P0)
		tr.append("|P0");
	if (t & TSource_SwTrg)
		tr.append("|SwTrg");
	if (t & TSource_LVDS)
		tr.append("|LVDS");
	if (t & TSource_EncodedClkIn)
		tr.append("|EncodedClkIn");
	if (t & TSource_GPIO)
		tr.append("|GPIO");
	if (t & TSource_TestPulse)
		tr.append("|TestPulse");
	if (t & TSource_ITLA_AND_ITLB)
		tr.append("|ITLA_AND_ITLB");
	else if (t & TSource_ITLA_OR_ITLB)
		tr.append("|ITLA_OR_ITLB");
	else if (t & TSource_ITLA)
		tr.append("|ITLA");
	else if (t & TSource_ITLB)
		tr.append("|ITLB");

	if (tr.startsWith("|"))
		tr.remove(0,1);
	mTrgSource = tr;
}
void CAENDig2Device::setPreTrg(QString val) {
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/par/PreTriggerT", qPrintable(val));
	if (err != CAEN_FELib_Success)
		throw CAENFELibException("Error setting value of pre trigger param " + mName, err);
	getPreTrg_value();
}

double CAENDig2Device::getPreTrg_value() {
	if (mIsVirtualDevice)
		return mPreTrg_valS * (mSample_to_S * 1e9);
	else {
		uint32_t val = getPreTrgParam()->getValue().toUInt();
		mPreTrg_valS = val / (mSample_to_S * 1e9);
		return static_cast<double>(val);
	}
}
double CAENDig2Device::getPreTrg_min() {
	if (mIsVirtualDevice)
		return 0;
	else
		return getPreTrgParam()->getMin();
}
double CAENDig2Device::getPreTrg_max() {
	if (mIsVirtualDevice)
		return 1e6;
	else
		return getPreTrgParam()->getMax();
}
double CAENDig2Device::getPreTrg_incr() {
	if (mIsVirtualDevice)
		return 1;
	else
		return getPreTrgParam()->getIncr();
}

double CAENDig2Device::getPreTrg_valueS() {
	if (mIsVirtualDevice)
		return mPreTrg_valS;
	else {
		uint32_t val = getPreTrgParam()->getValue().toUInt() / (mSample_to_S * 1e9);
		mPreTrg_valS = val;
		return static_cast<double>(val);
	}
}
double CAENDig2Device::getPreTrg_minS() {
	if (mIsVirtualDevice)
		return 0;
	else
		return getPreTrgParam()->getMin() / (mSample_to_S * 1e9);
}
double CAENDig2Device::getPreTrg_maxS() {
	if (mIsVirtualDevice)
		return 1e6;
	else
		return getPreTrgParam()->getMax() / (mSample_to_S * 1e9);
}
double CAENDig2Device::getPreTrg_incrS() {
	if (mIsVirtualDevice)
		return 1;
	else
		return getPreTrgParam()->getIncr() / (mSample_to_S * 1e9);
}

void CAENDig2Device::setScopeChannel(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/SelectWave"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting SelectWave param value for " + mName, ret);
	}
}
QString CAENDig2Device::getScopeChannel() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/SelectWave", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting start source of " + mName, err);
	}
	return QString(value);
}

void CAENDig2Device::setVETOSource(QString veto) {

}

QString CAENDig2Device::getVETOSource() {
	char value[200];
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/VETOSource", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting VETO source of " + mName, err);
	}
	return QString(value);
}

void CAENDig2Device::setIOLevel(QString val) {
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/IOLevel"), qPrintable(val));
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting IOLevel of " + mName, err);
	}
}
QString CAENDig2Device::getIOLevel() {
	char val[200] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/IOLevel"), val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting IOLevel of " + mName, err);
	}
	return QString(val);
}

void CAENDig2Device::setGlobalStartMode(QString mode, int delay) {
	if (mode == "SW_ASYNC") { //individual start mode
		try {
			setStartSource("SWcmd");//start mode must be sw controlled
			setRunDelay(delay);
		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}
	else if (mode == "SW_CHAIN_SYNC-CLKIN" || mode == "HW_CHAIN_SYNC-CLKIN") {
			try {
				setRunDelay(delay);
				if (DevChainIndex == 0 && (mode == "SW_CHAIN_SYNC-CLKIN")) { //board  b is the master and start mode is still sw controlled
						setStartSource("SWcmd");
						setSyncOutMode("Run");
				}
				else {
						setStartSource("EncodedClkIn");
						setSyncOutMode("SyncIn");					
				}
			}
			catch (CAENFELibException& exc) {
				throw CAENFELibException("Error setting StartSource of " + mName, -1);
			}
	}
	else if (mode == "SW_CHAIN_SIN-GPIO" || mode == "HW_CHAIN_SIN-GPIO") {
		try {
			setRunDelay(delay);
			if (DevChainIndex == 0 && mode == "SW_CHAIN_SIN-GPIO") { //board  b is the master and start mode is still sw controlled
					setStartSource("SWcmd");
					setGPIOMode("Run");
				}
				else {
					setStartSource("SINLevel");
					setGPIOMode("SIN");
				}
		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}
	else if (mode == "SW_CHAIN_SIN-TRGOUT" || mode == "HW_CHAIN_SIN-TRGOUT") {
		try {
			setRunDelay(delay);
			if (DevChainIndex == 0 && mode == "SW_CHAIN_SIN-TRGOUT") { //board  b is the master and start mode is still sw controlled
					setStartSource("SWcmd");
					setTRGOUTMode("Run");
				}
				else {
					setStartSource("SINLevel");
					setTRGOUTMode("Run");
				}
		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}
	else if (mode == "SW_CHAIN_1ST_TRIGGER" || mode == "HW_CHAIN_1ST_TRIGGER") {
		try {
			setRunDelay(delay);
			setStartSource("FirstTrigger");
			if (DevChainIndex == 0 && mode == "SW_CHAIN_1ST_TRIGGER") { //board  b is the master and start mode is still sw controlled
					setTRGOUTMode("Run");
				}
				else {
					setTRGOUTMode("TrgIn");
				}			

		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}

}

bool CAENDig2Device::isInfoStartWord(uint64_t word) {
	if (word == 0x3000000300000004)
		return true;
	else
		return false;
}

bool CAENDig2Device::isInfoStopWord(uint64_t word) {
	if (word == 0x3200000200000003)
		return true;
	else
		return false;
}

CAENDig2Device::~CAENDig2Device()
{
}
