/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENDig1Device.cpp
*	\brief
*	\author
*
******************************************************************************/

#include "CAENDig1Device.h"
#include "CAENFELibException.h"
#include <CAEN_FELib.h>
#include <QDebug>
#include <QVector>
#include <QtMath>
#include <QThread>

CAENDig1Device::CAENDig1Device(QString name, QString CType, QString address, uint64_t handle, bool isvirtual) :
	CAENDevice(name, CType, address, handle, isvirtual)
{
	DeviceClass = CAEN_DIG1;
	if (!mIsVirtualDevice) {
		try {
			uint32_t errors = getErrorFlags();
			if (errors) {
				ErrorsReported = true;
				return;
			}

			FillInfo();
			CreateParams();
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}

QString CAENDig1Device::getFormFactor() {
	if (mIsVirtualDevice)
		return "";
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/FormFactor", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting form factor of " + mName, err);
	}
	return	QString(value);
}

void CAENDig1Device::CalibrateADC() {
	if (mIsVirtualDevice)
		return;
	if ((mModel.contains("725") || mModel.contains("730") || mModel.contains("751") || mModel.contains("731") || mModel.contains("761")) && !mModel.contains("S")) {
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/CalibrateADC");
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error executing ADC calibration " + mName, err);
		}
	}
	return;
}

void CAENDig1Device::FillInfo(){
	char val[256] = { 0 };
	char val1[256] = { 0 };
	CAEN_FELib_ErrorCode err;
	const size_t handles_size = 1024;
	uint64_t handles[handles_size];
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/NumCh", val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting number of channels for " + mName, err);
	}
	mNumCh = atoi(val);
	ChBaselines.resize(mNumCh);
	mChThreshold.resize(mNumCh);
	mChADCToVolts.resize(mNumCh);

	if (mModel.contains("724"))
		TSamplUnit_ns = TSAMPL_UNIT_NS_DIG1_24;
	else
		TSamplUnit_ns = TSAMPL_UNIT_NS_DIG1;

	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ADC_SamplRate", val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting sampling rate for " + mName, err);
	}
	
	int exp_uom = 6; //MS/s  ---> /par/ADC_SamplRate/ExpUom not defined, unit is MS/s
    mSample_to_S = 1 / (QString::fromLatin1(val).toDouble() * qPow(10, exp_uom));

	NJExtraSamples = TSamplUnit_ns / (mSample_to_S * 1e9);
	if (NJExtraSamples == 1)
		NJExtraSamples = 0;

	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ADC_Nbit", val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting ADC nbit for " + mName, err);
	}
	mADC_NBit = atoi(val);

	mInputRange = 1; //fixed to 1 Vpp and multiplied by channel gainfactor (INDYN)

	mModel = getFamilyCode();
	if (mModel == "XX740") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetChildHandles(mHandle, "/group/", handles, handles_size);
		if (err < 0 || err > handles_size) {
			throw CAENFELibException("Error getting group size " + mName, err);
		}
		else
			mGroupSize = mNumCh / err;
	}

	mFormFactor = getFormFactor();
	

	for (int i = 0; i < mNumCh; i++) {
		QString qry = QString("/ch/%1/par/Ch_InDyn").arg(i);
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), val1);
		if (err != CAEN_FELib_Success)
			ChGain.push_back(1.0);
		else
			ChGain.push_back(QString(val1).toDouble());
	}

	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/ch/0/par/Ch_Couple_Trg_Mode"), val1);
	if (err != CAEN_FELib_Success)
		mPairedChannels = false;
	else
		mPairedChannels = true;	
	

	updateBaselines();
	updateGains();

	mWHandle = getWHandle();

	mRawHandle = getRawHandle();

	mSN = getSN();
}

void CAENDig1Device::CreateParams() {
	uint64_t phandle;
	try {
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, "/par/RecLen", &phandle); 
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error creating recordlen param for " + mName, err);
		}
		mReclen = new CAENparameter(phandle);

		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, "/par/PostTrg", &phandle); 
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error creating pretrigger param for " + mName, err);
		}
		mPostTrg = new CAENparameter(phandle);
	}
	catch (const CAENFELibException& exc) {
		throw exc;
	}

}

void CAENDig1Device::ApplyDefaultConfig(QHash<QString, QString> dpars, QHash<QString, QString> *cpars, QHash<QString, QString>* gpars) {
	CAEN_FELib_ErrorCode ret = CAEN_FELib_Success;
	char family[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/FamilyCode", family);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting family code of " + mName, err);
	}
	for (auto item = dpars.begin(); item != dpars.end(); ++item) {
		if (item.key().contains("Decimation", Qt::CaseInsensitive) && !mModel.contains("740"))
			continue;
		if (item.key() == "/TRIGGER_MODE")
			continue;
		if (item.key().contains("RecLen")) {
			int s = item.value().toInt();
			s += NJExtraSamples;
			int incr = mReclen->getIncr();
			s = ((s + incr - 1) / incr) * incr;
			RecLenJIncrS = (s - item.value().toInt()) / (mSample_to_S * 1e9); //real increment considering minimum step
			item.value() = QString("%1").arg(s);
		}
		ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(item.key()), qPrintable(item.value()));
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting param " + item.key() + " for " + mName, ret);
		}
	}
	for (int ch = 0; ch < mNumCh; ch++) {
		for (auto item = cpars[ch].begin(); item != cpars[ch].end(); ++item) {
			auto x = *item;	
			if ((mGroupSize > 1) && (item.key().contains("Ch_DCOffset",Qt::CaseInsensitive) || item.key().contains("Ch_Threshold", Qt::CaseInsensitive) || item.key().contains("Ch_Enabled", Qt::CaseInsensitive) || item.key().contains("Ch_Out", Qt::CaseInsensitive)))
				continue;
			ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(item.key()), qPrintable(item.value()));
			if (ret != CAEN_FELib_Success) {
				throw CAENFELibException("Error setting param " + item.key() + " for " + mName, ret);
			}
		}
	}
	if (mGroupSize > 1) {
		for (int g = 0; g < mNumCh / mGroupSize; g++) {
			for (auto item = gpars[g].begin(); item != gpars[g].end(); ++item) {
				auto x = *item;
				ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(item.key()), qPrintable(item.value()));
				if (ret != CAEN_FELib_Success) {
					throw CAENFELibException("Error setting param " + item.key() + " for " + mName, ret);
				}
			}
		}
	}
	//update the baselines if modified
	updateBaselines();
	//updateInputGains if modified
	updateGains();

	//update cached values for params
	mReclen_valS = mReclen->getValue().toUInt() / (mSample_to_S * 1e9) - RecLenJIncrS;
	mPreTrg_valS = mReclen_valS - mPostTrg->getValue().toUInt() / (mSample_to_S * 1e9);
}

void CAENDig1Device::updateBaselines() {
	if (mIsVirtualDevice)
		return;
	QString qry;
	int g;
	char value[32];
	for (int ch = 0; ch < mNumCh; ch++) {
		g = ch / mGroupSize;
		if (mModel == "XX740")
			qry = QString("/group/%1/par/Gr_DCOffset").arg(g);
		else
			qry = QString("/ch/%1/par/Ch_DCOffset").arg(ch);
		CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error getting param value for " + mName, ret);
		}
		double val = QString::fromLatin1(value).toDouble();
		ChBaselines.replace(ch, val);
	}
}

double CAENDig1Device::getReclen_ns() {//value in ns set by the user (without extra samples)
	double ns = mReclen->getCachedValue().toDouble();
	ns -= RecLenJIncrS * (mSample_to_S * 1e9);
	return ns;
}

double CAENDig1Device::getReclen_us() {
	double val = 0;
	if (mReclen != nullptr) {
		val = getReclen_ns();
		val /= 1000.; //convert from ns to us
		getDecimation();
		val *= mDecim;		
	}
	return val;
}

double CAENDig1Device::getChOffset(int ch) const{
	if (mIsVirtualDevice)
		return ChBaselines.at(ch);
	int g = ch / mGroupSize;
	QString qry;
	if (mModel == "XX740")
		qry = QString("/group/%1/par/Gr_DCOffset").arg(g);
	else
		qry = QString("/ch/%1/par/Ch_DCOffset").arg(ch);
	char value[32];
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting param value for " + mName, ret);
	}
	double val = QString::fromLatin1(value).toDouble();
	for(int c = 0; c < mGroupSize; c ++)
		ChBaselines.replace(c + g*mGroupSize,val);
	return val;
}

void CAENDig1Device::setChOffset(int ch, double value) {
	if (ch >= mNumCh)
		return;
	int g = ch / mGroupSize;
	QString qry;
	if (mModel == "XX740")
		qry = QString("/group/%1/par/Gr_DCOffset").arg(g);
	else
		qry = QString("/ch/%1/par/Ch_DCOffset").arg(ch);
	QString val = QString("%1").arg(value);
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(val));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting DCO param value for " + mName, ret);
	}
}

void CAENDig1Device::updateChSample_to_V(int ch) {
	double val= (mInputRange * ChGain.at(ch) )/ ((1U << mADC_NBit) - 1); 
	mChADCToVolts.replace(ch,val);
}

double CAENDig1Device::getChADCTemp(int ch) {
	QString qry = QString("/ch/%1/par/Ch_ADC_Temperature").arg(ch);
	char value[32];
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
	if (ret != CAEN_FELib_Success) {
		return 0.;
	}
	else
		return QString(value).toDouble();

}

double CAENDig1Device::getChGain(int ch){
	QString qry = QString("/ch/%1/par/Ch_InDyn").arg(ch);
	char value[32];
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
	if (ret != CAEN_FELib_Success) {
		// default value
		return 1.;
	}

	ChGain.replace(ch, VppNumFromString(QString(value)));

	updateChSample_to_V(ch);

	return ChGain.at(ch);
}

void CAENDig1Device::setChGain(int ch, double value) {
	QString qry = QString("/ch/%1/par/Ch_InDyn").arg(ch);
	QString val=VppStringFromNum(value);
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(val));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting gain param value for " + mName, ret);
		return;
	}
	ChGain.replace(ch, value);

	updateChSample_to_V(ch);
}

double CAENDig1Device::VppNumFromString(QString inrange) {
	if (inrange == "INDYN_0_3_VPP")
		return 0.3;
	if (inrange == "INDYN_0_5_VPP")
		return 0.5;
	if (inrange == "INDYN_0_6_VPP")
		return 0.6;
	if (inrange == "INDYN_1_0_VPP")
		return 1.0;
	if (inrange == "INDYN_1_4_VPP")
		return 1.4;
	if (inrange == "INDYN_2_0_VPP")
		return 2.0;
	if (inrange == "INDYN_2_25_VPP")
		return 2.25;
	if (inrange == "INDYN_3_0_VPP")
		return 3.0;
	if (inrange == "INDYN_3_7_VPP")
		return 3.7;
	if (inrange == "INDYN_9_5_VPP")
		return 9.5;
	if (inrange == "INDYN_10_0_VPP")
		return 10.0;
	return 1.;
}

QString CAENDig1Device::VppStringFromNum(double num) {
	if (num == 0.3)
		return "INDYN_0_3_VPP";
	if (num == 0.5)
		return "INDYN_0_5_VPP";
	if (num == 0.6)
		return "INDYN_0_6_VPP";
	if (num == 1.0)
		return "INDYN_1_0_VPP";
	if (num == 1.4)
		return "INDYN_1_4_VPP";
	if (num == 2.0)
		return "INDYN_2_0_VPP";
	if (num == 2.25)
		return "INDYN_2_25_VPP";
	if (num == 3.0)
		return "INDYN_3_0_VPP";
	if (num == 3.7)
		return "INDYN_3_7_VPP";
	if (num == 9.5)
		return "INDYN_9_5_VPP";
	if (num == 10.0)
		return "INDYN_10_0_VPP";
	return "";
}



int CAENDig1Device::getChThreshold(int ch){
	int g = ch / mGroupSize;
	QString qry;
	if (mModel == "XX740")
		qry = QString("/group/%1/par/Gr_Threshold").arg(g);
	else
		qry = QString("/ch/%1/par/Ch_Threshold").arg(ch);
	char value[32];
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting threshold param value for " + mName, ret);
	}
	mChThreshold.replace(ch, QString::fromLatin1(value).toDouble());
	return atoi(value);
}

void CAENDig1Device::setChThreshold(int ch, int val) {
	if (ch >= mNumCh)
		return;
	int g = ch / mGroupSize;
	QString qry;
	if (mModel.contains("740"))
		qry = QString("/group/%1/par/Gr_Threshold").arg(g);
	else
		qry = QString("/ch/%1/par/Ch_Threshold").arg(ch);
	QString value = QString("%1").arg(val);
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting threshold param value for " + mName, ret);
		return;
	}
	mChThreshold.replace(ch,static_cast<double>(val));
}

CAENparameter *CAENDig1Device::getThrParam(int ch) {
	uint64_t phandle;
	int g = ch / mGroupSize;
	QString qry;
	if (mModel == "XX740")
		qry = QString("/group/%1/par/Gr_Threshold").arg(g);
	else
		qry = QString("/ch/%1/par/Ch_Threshold").arg(ch);
	CAENparameter *thr = nullptr;
	try {
		CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, qPrintable(qry), &phandle);
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error creating thr param for " + mName, err);
		}
		thr = new CAENparameter(phandle);
	}
	catch (const CAENFELibException& exc) {
		throw exc;
	}
	return thr;
}

void CAENDig1Device::setStartSource(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/StartMode"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting startmode param value for " + mName, ret);
	}
}
QString CAENDig1Device::getStartSource() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/StartMode", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting start source of " + mName, err);
	}
	return QString(value);
}

void CAENDig1Device::setRunDelay(int value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/Start_Delay"), qPrintable(QString("%1").arg(value)));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting start_delay param value for " + mName, ret);
	}
}

void CAENDig1Device::setTRGOUTMode(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/OUT_Selection"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting out_selction param value for " + mName, ret);
	}
}
QString CAENDig1Device::getTRGOUTMode() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/OUT_Selection", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting TRGOUT mode of " + mName, err);
	}
	return QString(value);
}
QString CAENDig1Device::getTrgOutMask() {
	uint64_t mask = 0;
	QString qry;
	CAEN_FELib_ErrorCode ret;
	char value[256] = { 0 };
	for (int ch = 0; ch < mNumCh; ch++) {
		if (mGroupSize > 1)
			qry = QString("/group/%1/par/Gr_Out_Propagate").arg(ch / mGroupSize);
		else
			qry = QString("/ch/%1/par/Ch_Out_Propagate").arg(ch);
		ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error getting trgout mask param value for " + mName, ret);
		}
		if (QString(value) == "TRUE")
			mask |= (1LL << ch);
	}
	return QString("%1").arg(mask);
	return 0;
}

void CAENDig1Device::setTrgOutMask(QString value) {
	uint64_t mask = value.toULong();
	QString qry;
	CAEN_FELib_ErrorCode ret;
	if (mGroupSize > 1) {
		uint32_t group_mask = 0;
		for (int ch = 0; ch < mNumCh; ch++) {
			if (mask & (1LL << ch))
				group_mask |= (1 << (ch / mGroupSize));
		}
		for (int g = 0; g < mNumCh / mGroupSize; g++) {
			if (group_mask & (1 << g))
				value = "TRUE";
			else
				value = "FALSE";
			qry = QString("/group/%1/par/Gr_Out_Propagate").arg(g);
			ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(value));
			if (ret != CAEN_FELib_Success) {
				throw CAENFELibException("Error setting trgout mask param value for " + mName, ret);
			}
		}
	}
	else {
		for (int ch = 0; ch < mNumCh; ch++) {
			if (mPairedChannels && (ch % 2 != 0))
				continue;
			if (mask & (1LL << ch))
				value = "TRUE";
			else
				value = "FALSE";

			qry = QString("/ch/%1/par/Ch_Out_Propagate").arg(ch);
			ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(value));
			if (ret != CAEN_FELib_Success) {
				throw CAENFELibException("Error setting ch_out_propagate param value for " + mName, ret);
			}
		}
	}
}

QString CAENDig1Device::getExtClkSource() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/DT_Ext_Clock", value);
	if (ret != CAEN_FELib_Success) {
		//for VME boards try to read from status 
		ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/Clock_Source", value);
		if (ret != CAEN_FELib_Success)
			throw CAENFELibException("Error getting clock source for " + mName, ret);
	}
	return QString(value);
}
void CAENDig1Device::setExtClkSource(QString value) {
	if (mFormFactor.contains("VME", Qt::CaseInsensitive))
		return;
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/DT_Ext_Clock"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting DT_Ext_Clock param value for " + mName, ret);
	}
}
void CAENDig1Device::setExtTrgPropagate(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/Trg_Ext_Out_Propagate"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting Trg_Ext_Out_Propagate for " + mName, ret);
	}
}

void CAENDig1Device::setSwTrgPropagate(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/Trg_Sw_Out_Propagate"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting Trg_Sw_Out_Propagate for " + mName, ret);
	}
}

void CAENDig1Device::disableExtTrgIn(QString value){
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/Ext_Trg_Inhibit"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting ext trigger inhibit for " + mName, ret);
	}
}

void CAENDig1Device::setExtTrgEnable(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/Trg_Ext_Enable"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting trg_ext_enable param value for " + mName, ret);
	}
}
QString CAENDig1Device::getExtTrgEnable() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/Trg_Ext_Enable"), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting trg_ext_enable param value for " + mName, ret);
	}
	return QString(value);
}
void CAENDig1Device::setSwTrgEnable(QString value) {
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/Trg_Sw_Enable"), qPrintable(value));
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting trg_sw_enable param value for " + mName, ret);
	}
}
QString CAENDig1Device::getSwTrgEnable() {
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/Trg_Sw_Enable"), value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting trg_sw_enable param value for " + mName, ret);
	}
	return QString(value);
}

void CAENDig1Device::setSelfTrgMask(QString value) {
	uint64_t mask = value.toULong();
	QString val;
	QString qry;
	CAEN_FELib_ErrorCode ret;
	if (mGroupSize > 1) {
		uint32_t group_mask = 0;
		for (int ch = 0; ch < mNumCh; ch++) {
			if (mask & (1LL << ch))
				group_mask |= (1 << (ch / mGroupSize));
		}
		for (int g = 0; g < mNumCh / mGroupSize; g++) {
			if (group_mask & (1 << g))
				val = "TRUE";
			else
				val = "FALSE";
			qry = QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g);
			ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(val));
			if (ret != CAEN_FELib_Success) {
				throw CAENFELibException("Error setting self trigger mask param value for " + mName, ret);
			}			
		}
	}

	for (int ch = 0; ch < mNumCh; ch++) {
		if (mPairedChannels && (ch % 2 != 0)) {
			if (mask & (1LL << (ch-1)))
			continue;
		}
		
		if (mask & (1LL << ch))
			val = "TRUE";
		else
			val = "FALSE";
		
		qry = QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch);
		ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), qPrintable(val));
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting ch_trg_global_gen param value for " + mName, ret);
		}
	}
}

QString CAENDig1Device::getSelfTrgMask() {
	uint64_t mask = 0;
	QString qry;
	//int g;
	CAEN_FELib_ErrorCode ret;
	char value[256] = { 0 };
	for (int ch = 0; ch < mNumCh; ch++) {
		//g = ch / mGroupSize;
		//if (mModel == "XX740")
		//	qry = QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g);
		//else
			qry = QString("/ch/%1/par/Ch_Trg_Global_Gen").arg(ch);
		ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable(qry), value);
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error getting ch_trg_global_gen param value for " + mName, ret);
		}
		if (QString(value) == "TRUE")
			mask |= (1LL << ch);
	}
	return QString("%1").arg(mask);
}

void CAENDig1Device::setTrgSource(QString value) {
	QString qry;
	if (value.contains("TrgIn", Qt::CaseInsensitive)) {
		setExtTrgEnable("TRUE");
	}
	else {
		setExtTrgEnable("FALSE");
	}

	if (value.contains("Sw", Qt::CaseInsensitive)) {
		setSwTrgEnable("TRUE");
	}
	else {
		setSwTrgEnable("FALSE");
	}

	if (!value.contains("ITL", Qt::CaseInsensitive)) {
		setSelfTrgMask("0"); //group channels enablemasks = 0
		if (mGroupSize > 1) {
			for (int g = 0; g < mNumCh / mGroupSize; g++) {
				qry = QString("/group/%1/par/Gr_Trg_Global_Gen").arg(g); //groups do not participate to global trigger
				CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable(qry), "FALSE");
				if (ret != CAEN_FELib_Success) {
					throw CAENFELibException("Error setting Gr_Trg_Global_Gen param value for " + mName, ret);
				}
			}
		}
	}

	mTrgSource = value;
}
QString CAENDig1Device::getTrgSource() {
	if (mIsVirtualDevice)
		return mTrgSource;
	QString value="";
	
	if (getExtTrgEnable() == "TRUE")
		value.append("TrgIn");
	if (getSwTrgEnable() == "TRUE") {
		if (!value.isEmpty())
			value.append("|");
		value.append("SwCmd");
	}
	if (getSelfTrgMask() != "0") {
		if (!value.isEmpty())
			value.append("|");
		value.append("ITL");	
	}
	mTrgSource = value;
	return value;
}

uint32_t CAENDig1Device::calcTrgSourceToSave() {
	QString ts = getTrgSourceCachedValue();
	uint32_t code = 0;
	if (ts.contains("TrgIn", Qt::CaseInsensitive))
		code |= TSource_TrgIn;
	if (ts.contains("Sw", Qt::CaseInsensitive))
		code |= TSource_SwTrg;
	if (ts.contains("LVDS", Qt::CaseInsensitive))
		code |= TSource_LVDS;
	if (ts.contains("GPIO", Qt::CaseInsensitive))
		code |= TSource_GPIO;
	else if (ts.contains("ITL", Qt::CaseInsensitive))
		code |= TSource_ITLA;
	return code;
}

void CAENDig1Device::setDecimation(int val) {
	if (mModel.contains("740") || mModel.contains("724")) {
		CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/Decimation_Factor"), qPrintable(QString("DECIM_FACTOR_%1").arg(val)));
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting Decimation_Factor param value for " + mName, ret);
		}
		mDecim = val;
	}
	else 
		return;
}
int CAENDig1Device::getDecimation() {
	if (mIsVirtualDevice)
		return mDecim;
	if (mModel.contains("740") || mModel.contains("724")) {
		char value[256] = { 0 };
		CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/Decimation_Factor"), value);
		if (ret != CAEN_FELib_Success) {
			throw CAENFELibException("Error getting param value for " + mName, ret);
			return 1;
		}
		QString vals = QString(value);
		vals.remove("DECIM_FACTOR_");
		mDecim = vals.toInt();
	}
	return mDecim;
}

void CAENDig1Device::StartAcquisition() {
	ClearData();
	ArmAcquisition();
}

void CAENDig1Device::StopAcquisition() {
	if (ErrorsReported)
		return;
	DisarmAcquisition();
	ClearData();
	ClearData();
}


QString CAENDig1Device::getAcqStatus() {
	if (mIsVirtualDevice)
		return mVirtualAcqStatus;
	char value[256] = { 0 };
	QString acq_status="";
	CAEN_FELib_ErrorCode ret = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/AcqRunning", value);
	if (ret != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting AcqRunning param value for " + mName, ret);
		return "";
	}
	if(QString(value) == "TRUE")
		acq_status = "Running";
	else
		acq_status = "Ready";
	/*
	uint32_t status = atoi(value);
	if (status & (1 << ACQ_STATUS_BUSY_BIT))//give priority to bust state TODO: check busy for dig1 (reg 8104 full condition)
		acq_status="Busy";
	else if (status & (1 << ACQ_STATUS_RUN_BIT))
		acq_status = "Running";
	else if (status & (1 << ACQ_STATUS_ARMED_BIT))
		acq_status = "Armed";
	else
		acq_status = "Ready";
		*/
	return acq_status;
}


void CAENDig1Device::SetDataFormat(QString format) {
	CAEN_FELib_ErrorCode err;
	if (format != "RAW") {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/endpoint/par/activeendpoint", "scope");
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting data format for " + mName, err);
			return;
		}
		
		QString data_format = "[  { \"name\" : \"TIMESTAMP\", \"type\" : \"U64\" }, \
							{ \"name\" : \"TRIGGER_ID\", \"type\" : \"U32\" }, \
							{ \"name\" : \"WAVEFORM\", \"type\" : \"U16\", \"dim\" : 2 }, \
							{ \"name\" : \"WAVEFORM_SIZE\", \"type\" : \"U32\", \"dim\" : 1 },	\
							{ \"name\" : \"EVENT_SIZE\", \"type\" : \"U32\" },  \
							{ \"name\" : \"EXTRA\", \"type\" : \"U16\" } ]";
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetReadDataFormat(mWHandle, qPrintable(data_format));
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error setting data format for " + mName, err);
			return;
		}
	}
	else {
		EnableRawDataFormat();
	}
}

QString CAENDig1Device::getFWRel() {
	if (mIsVirtualDevice)
		return "";
	char value[256] = { 0 };
	QString fw_ver = "AMC ";
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/AMC_FwVer", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting FW rel of " + mName, err);
	}
	fw_ver.append(QString(value));
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ROC_FwVer", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting FW rel of " + mName, err);
	}
	fw_ver.append("  ROC ");
	fw_ver.append(QString(value));
	return fw_ver;
}


uint32_t CAENDig1Device::getErrorFlags(){
	if (mIsVirtualDevice)
		return 0;
	uint32_t num=0;
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err;
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/PLLLocked", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting Error Flags of " + mName, err);
	}
	if (QString(value) == "FALSE")
		num |= FLAGS_PLLUNLOCK;

	if (mModel.contains("730") || mModel.contains("725")) {
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ChShutDown", value);
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error getting Error Flags of " + mName, err);
		}
		if (QString(value) == "TRUE")
			num |= FLAGS_CHSHUTDOWN;
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ChOverTemp", value);
		if (err != CAEN_FELib_Success) {
			throw CAENFELibException("Error getting Error Flags of " + mName, err);
		}
		if (QString(value) == "TRUE")
			num |= FLAGS_CHOVT;
	}
	return num;	
}

uint64_t CAENDig1Device::getChEnableMask() {
	uint64_t mask = 0;
	for (int ch = 0; ch < mNumCh; ch++) {
		if (isChEnabled(ch))
			mask |= (1LL << ch);
	}
	return mask;
}

bool CAENDig1Device::isChEnabled(int ch) {
	if (mIsVirtualDevice)
		return (mChMask & ((uint64_t)1 << ch)) ? true : false;
	if (ch >= mNumCh)
		return false;
	char value[256] = { 0 };
	QString qry;
	int g = ch / mGroupSize;
	if (mModel.contains("740"))
		qry = QString("/group/%1/par/Gr_Enabled").arg(g);
	else
		qry = QString("/ch/%1/par/Ch_Enabled").arg(ch);
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qry.toStdString().c_str(), value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting enable status of channel ", err);
	}
	if (QString(value) == "TRUE")
		return true;
	else
		return false;
}

void CAENDig1Device::setChEnabled(int ch, bool enable) {
	if (ch >= mNumCh)
		return;
	QString enab = (enable) ? "TRUE" : "FALSE";
	QString qry;
	int g = ch / mGroupSize;
	if (mModel.contains("740"))
		qry = QString("/group/%1/par/Gr_Enabled").arg(g);
	else
		qry = QString("/ch/%1/par/Ch_Enabled").arg(ch);
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qry.toStdString().c_str(), enab.toStdString().c_str());
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting enable status of channel ", err);
	}
}

QString CAENDig1Device::DecodeFlagsMask(uint32_t mask) {
	QString message;
	if(mask == 0)
		return "No errors reported by device " + mName;
	else
		message = "Errors reported by device " + mName + ":\n";
	if (mask & FLAGS_CHSHUTDOWN)
		message.append("ADC ShutDown\n");
	if (mask & FLAGS_PLLUNLOCK)
		message.append("PLL Not Locked\n");
	if (mask & FLAGS_CHOVT)
		message.append("OverTemperature\n");

	return message;
}

void CAENDig1Device::DefinePreTrigger(uint32_t p) {
	mPreTrg_valS = p;
}

void CAENDig1Device::DefineTrgSource(uint32_t t) {
	QString tr = "";
	if (t & TSource_TrgIn)
		tr.append("|TrgIn");
	if (t & TSource_SwTrg)
		tr.append("|SwTrg");
	if (t & TSource_ITLA)
		tr.append("|ITL");

	if (tr.startsWith("|"))
		tr.remove(0,1);
	mTrgSource = tr;
}
//val in ns
void CAENDig1Device::setPreTrg(QString val) {
	CAEN_FELib_ErrorCode err = CAEN_FELib_Success;
	uint32_t value = (getReclen_ns() * getDecimation() - val.toUInt());
	QString value_to_set = QString("%1").arg(value);
	err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/par/PostTrg", qPrintable(value_to_set));
	if (err != CAEN_FELib_Success)
		throw CAENFELibException("Error setting value of pre trigger param " + mName, err);
	getPreTrg_value();
}

double CAENDig1Device::getPreTrg_value() {
	if (mIsVirtualDevice)
		return mPreTrg_valS * (mSample_to_S * 1e9);
	else {
		int decim = getDecimation();
		uint32_t val = (getReclen_ns() * decim) - getPostTrgParam()->getValue().toUInt();
		mPreTrg_valS = val / (mSample_to_S * 1e9 * decim);
		return static_cast<double>(val);
	}
}
double CAENDig1Device::getPreTrg_min() {
	double val = 0;
	if (mIsVirtualDevice)
		return 0;
	else {
		val = getReclen_min() - getPostTrgParam()->getMax();
	}
	return (val < 0) ? 0 : val;
}
double CAENDig1Device::getPreTrg_max() {
	double val = 0;
	if (mIsVirtualDevice)
		return 1e6;
	else {
		val = getReclen_max() * getDecimation() - getPostTrgParam()->getMin();
	}
	return val;
}
double CAENDig1Device::getPreTrg_incr() {
	if (mIsVirtualDevice)
		return 1;
	else
		return getPostTrgParam()->getIncr();
}

double CAENDig1Device::getPreTrg_valueS() {
	if (mIsVirtualDevice)
		return mPreTrg_valS;
	else {
		uint32_t val = (getReclen_ns() * mDecim) - getPostTrgParam()->getValue().toUInt();
		mPreTrg_valS = val / (mSample_to_S * 1e9);
		return static_cast<double>(mPreTrg_valS);
	}
}
double CAENDig1Device::getPreTrg_minS() {
	double val = 0;
	if (mIsVirtualDevice)
		return 0;
	else {
		val = (getReclen_ns() * mDecim - getPostTrgParam()->getMax()) / (mSample_to_S * 1e9);
	}
	return val;
}
double CAENDig1Device::getPreTrg_maxS() {
	double val = 0;
	if (mIsVirtualDevice)
		return 1e6;
	else {
		val = (getReclen_ns() * mDecim - getPostTrgParam()->getMin()) / (mSample_to_S * 1e9);
	}
	return val;
}
double CAENDig1Device::getPreTrg_incrS() {
	if (mIsVirtualDevice)
		return 1;
	else
		return getPostTrgParam()->getIncr() / (mSample_to_S * 1e9);
}
void CAENDig1Device::setScopeChannel(QString value) {
	return;
}
QString CAENDig1Device::getScopeChannel() {
	return "";
}

QString CAENDig1Device::getClkSource() {
	QString val = getExtClkSource();
	if (val == "FALSE")
		return "Internal";
	else
		return "External";
}
void CAENDig1Device::setClkSource(QString value) {
	QString val_bool;
	if (value == "Internal")
		val_bool = "FALSE";
	else
		val_bool = "TRUE";
	setExtClkSource(val_bool);
}

void CAENDig1Device::setGPIOMode(QString value) {
	//TODO add definition
}
QString CAENDig1Device::getGPIOMode() {
	char value[256] = { 0 };
	//TODO add definition
	return QString(value);
}

QString CAENDig1Device::getClkOutFP() {
	return "True"; //return true by default TODO: check!
}

void CAENDig1Device::setClkOutFP(QString value) {
	return;
}

void CAENDig1Device::setVETOSource(QString veto) {
	//TODO: define

}
QString CAENDig1Device::getVETOSource() {
	return "DISABLED";//TODO: define

}

void CAENDig1Device::setIOLevel(QString val) {
	QString text = val;
	text.prepend("FPIOTYPE_");
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, qPrintable("/par/IOLevel"), qPrintable(text));
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error setting IOLevel of " + mName, err);
	}
}
QString CAENDig1Device::getIOLevel() {
	char val[200] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, qPrintable("/par/IOLevel"), val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting IOLevel of " + mName, err);
	}
	QString text = QString(val);
	text.remove("FPIOTYPE_");
	return text;
}


QString CAENDig1Device::getFWGbE() {
	return "";
}

void CAENDig1Device::Reboot() {
	//TODO
	return;
}
void CAENDig1Device::setSyncOutMode(QString value) {
	return;
}

void CAENDig1Device::setGlobalStartMode(QString mode, int delay) {
	Q_UNUSED(delay)
	if (mode == "SW_ASYNC") { //individual start mode
		try {
			setStartSource("START_MODE_SW");//start mode must be sw controlled
			//setRunDelay(delay);
		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}
	else if (mode == "SW_CHAIN_SIN-TRGOUT" || mode == "HW_CHAIN_SIN-TRGOUT") {
		try {
			//setRunDelay(delay);
			setExtTrgEnable("TRUE");
			setExtTrgPropagate("FALSE");
			setSwTrgPropagate("FALSE");
			if (DevChainIndex == 0 && mode == "SW_CHAIN_SIN-TRGOUT") { //board  b is the master and start mode is still sw controlled
				setStartSource("START_MODE_SW");
				setTRGOUTMode("OUT_PROPAGATION_RUN");
			}
			else {
				setStartSource("START_MODE_S_IN");
				setTRGOUTMode("OUT_PROPAGATION_RUN");
			}
		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}
	else if (mode == "SW_CHAIN_1ST_TRIGGER" || mode == "HW_CHAIN_1ST_TRIGGER") {
		try {
			//setRunDelay(delay);
			setStartSource("START_MODE_FIRST_TRG");
			if (DevChainIndex == 0 && mode == "SW_CHAIN_1ST_TRIGGER") { //board  b is the master and start mode is still sw controlled
				setTRGOUTMode("OUT_PROPAGATION_RUN");
			}
			else {
				setExtTrgPropagate("TRUE");
			}

		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}
	else if (mode == "SW_CHAIN_TRGIN-TRGOUT" || mode == "HW_CHAIN_TRGIN-TRGOUT") {
		try {
			//setRunDelay(delay);
			if (DevChainIndex == 0) { //board  b is the master, disable ext trigin to avoid start run with ext triggers
				disableExtTrgIn("TRUE");		
			}
			
			setStartSource("START_MODE_FIRST_TRG");			
			setExtTrgEnable("TRUE");
			setSwTrgEnable("TRUE");
			setExtTrgPropagate("TRUE");
			setSwTrgPropagate("TRUE");
			setTRGOUTMode("OUT_PROPAGATION_TRIGGER");
		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}
	else if (mode == "SW_CHAIN_ONE2ALL_EXTOR" || mode == "HW_CHAIN_ONE2ALL_EXTOR") {
		try {
			//setRunDelay(delay);
			setSwTrgPropagate("TRUE");
			setExtTrgPropagate("FALSE");
			setExtTrgEnable("TRUE");
			setSwTrgEnable("FALSE");
			setStartSource("START_MODE_FIRST_TRG");
			setTRGOUTMode("OUT_PROPAGATION_TRIGGER");
		}
		catch (CAENFELibException& exc) {
			throw CAENFELibException("Error setting StartSource of " + mName, -1);
		}
	}
}

bool CAENDig1Device::isInfoStartWord(uint64_t word) {
	uint32_t word32 = (word >> 32) & 0xffffffff;
	if (((word32 >> 4) & UINT32_C(0xF)) == 0xA)
		return true;
	else
		return false;
}

bool CAENDig1Device::isInfoStopWord(uint64_t word) {
		return false;
}


CAENDig1Device::~CAENDig1Device()
{
}
