#include "CAENDevice.h"
#include "CAENFELibException.h"
#include <CAEN_FELib.h>
#include <QtMath>


CAENDevice::CAENDevice(QString name, QString CType, QString address, uint64_t handle, bool isvirtual) {
	mName = name;
	mConnType = CType;
	mAddress = address;
	mHandle = handle;
	mIndex = 0;
	mIsVirtualDevice = isvirtual;
}

void CAENDevice::MallocRawBuffers() {
	uint64_t size = getMaxRawDataSize();
	for (int b = 0; b < MAX_RAW_BUFFS; b++)
		RawDataBuffer[b] = (char*)malloc(size);
}

void CAENDevice::FreeRawBuffers() {
	for (int b = 0; b < MAX_RAW_BUFFS; b++) {
		free(RawDataBuffer[b]);
		RawDataBuffer[b] = nullptr;
	}
}

void CAENDevice::CloseConnection() {
	int err;
	if (!ConnectionOpen || mIsVirtualDevice)
		return;
	err = CAEN_FELib_Close(mHandle);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error disconnecting device " + mName, err);
	}
	ConnectionOpen = false;
}

void CAENDevice::createEvtsQueue(std::size_t size, int length) {
	mCloseQueue = false;
	NEvtsInQueue = 0;
	mDevEventsQueue.clear();
	for (int ev = 0; ev < length; ev++) {
		mDevEventsQueue.append(new CAENDig2Event(mNumCh, size));
	}
}

void  CAENDevice::freeEvtsQueue() {
	if (mCloseQueue)
		return;
	for (auto ev : mDevEventsQueue)
		delete ev;
	mDevEventsQueue.clear();
	mCloseQueue = true;
	NEvtsInQueue = 0;
	NTotEvents = 0;
	NTotBytesRead = 0;
}

double CAENDevice::getSample_to_V(int ch) {
	if (ErrorsReported || ch < 0)
		return 0;
	//double val= mInputRange / ((1U << mADC_NBit) - 1) / ChGain.at(ch);
	//val /= mChADCGain.at(ch);

	return mChADCToVolts.at(ch);
}

uint32_t CAENDevice::getVRangeS() {
	return qPow(2, mADC_NBit) - 1;
}

double CAENDevice::getVRangeV(int ch) {
	if (ch < 0 || ch > mNumCh)
		return 0;
	return static_cast<double>(getVRangeS()) * getSample_to_V(ch);
}

void CAENDevice::updateGains() {
	if (mIsVirtualDevice)
		return;
	for (int ch = 0; ch < mNumCh; ch++)
		getChGain(ch);
}

double CAENDevice::getMaxGain() {
	double g, max = 0;
	for (int ch = 0; ch < mNumCh; ch++) {
		g = getChGain(ch);
		if (g > max)
			max = g;
	}
	return max;
}

int CAENDevice::getDecimation_value() {
	if (mIsVirtualDevice)
		return mDecim;
	else
		return getDecimation();
}

void CAENDevice::ArmAcquisition() {
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/ArmAcquisition");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error arming acquisition for " + mName, err);
	}
	AcqStarted = 1;
	resetTime();
}
void CAENDevice::DisarmAcquisition() {
	if (ErrorsReported)
		return;
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/DisarmAcquisition");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error disarming acquisition for " + mName, err);
	}
	AcqStarted = 0;
	ClearData();
	ClearData();
}

void CAENDevice::ClearData() {
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/ClearData");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error in cleardata for " + mName, err);
	}
}

void CAENDevice::Reset() {
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/Reset");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error in reset for " + mName, err);
	}
}

uint64_t CAENDevice::getWHandle() {
	uint64_t ep_handle;
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, "/endpoint/scope", &ep_handle);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting waveform endpoint " + mName, err);
	}
	return ep_handle;
}

uint64_t CAENDevice::getRawHandle() {
	uint64_t ep_handle;
	CAEN_FELib_ErrorCode err;
	QString fwtype = getFWType();
	if (fwtype.contains("UDP", Qt::CaseInsensitive))
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, "/endpoint/rawudp", &ep_handle);
	else
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetHandle(mHandle, "/endpoint/raw", &ep_handle);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting waveform endpoint " + mName, err);
	}
	return ep_handle;
}

void CAENDevice::EnableRawDataFormat() {
	CAEN_FELib_ErrorCode err;
	QString fwtype = getFWType();
	if(fwtype.contains("UDP",Qt::CaseInsensitive))
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/endpoint/par/activeendpoint", "rawudp");
	else
		err = (CAEN_FELib_ErrorCode)CAEN_FELib_SetValue(mHandle, "/endpoint/par/activeendpoint", "raw");
	if (err != CAEN_FELib_Success)
		throw CAENFELibException("Error enabling raw data format for " + mName, err);
}

void CAENDevice::SendSWTrg() {
	SWTrg_sent = true;
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_SendCommand(mHandle, "/cmd/SendSWTrigger");
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error sending SW trigger to " + mName, err);
	}
}

QString CAENDevice::getFamilyCode() {
	if (mIsVirtualDevice)
		return mModel;
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/FamilyCode", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting family code of " + mName, err);
	}
	return QString(value);
}

QString CAENDevice::getModel() {
	if (mIsVirtualDevice)
		return "";
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/ModelName", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting model of " + mName, err);
	}
	return QString(value);
}
QString CAENDevice::getFWType() {
	if (mIsVirtualDevice)
		return "Scope";
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/FwType", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting FW type of " + mName, err);
	}
	return QString(value);
}

QString CAENDevice::getSN() {
	if (mIsVirtualDevice)
		return mSN;
	char value[256] = { 0 };
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/SerialNum", value);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting serial number of " + mName, err);
	}
	return QString(value);
}

void CAENDevice::DefineInfo(int index, uint32_t pid, uint32_t ADCNBit, uint32_t SampleToS, uint16_t NCh, uint32_t* baseline, uint32_t* factor) {
	mIndex = index;
	mSN = pid;
	mADC_NBit = ADCNBit;
	mSample_to_S = static_cast<double>(SampleToS) / 1e12;
	mNumCh = NCh;
	ChBaselines.resize(mNumCh);
	mChADCToVolts.clear();
	mChADCToVolts.resize(mNumCh);
	ChGain.resize(mNumCh);
	mChThreshold.resize(mNumCh);
	for (int ch = 0; ch < mNumCh; ch++) {
		double percent = static_cast<double>(baseline[ch]) / (qPow(2, mADC_NBit) - 1) * 100.;
		ChBaselines.replace(ch, percent);
		mChADCToVolts.replace(ch, static_cast<double>(factor[ch]) / 1e12);
	}

}

void CAENDevice::DefineGains(uint32_t* gains) {
	for (int ch = 0; ch < mNumCh; ch++) {
		ChGain.replace(ch, static_cast<double>(gains[ch]) / 1e6);
	}
}
void CAENDevice::DefineThresholds(int32_t* thrs) {
	for (int ch = 0; ch < mNumCh; ch++) {
		mChThreshold.replace(ch, static_cast<double>(thrs[ch]));
	}
}

void CAENDevice::DefineFamily(uint32_t code) {
	mModel = QString("%1").arg(code);
}

void CAENDevice::DefineDecimation(uint16_t d) {
	mDecim = d;
}
void CAENDevice::DefineReclen(uint32_t r) {
	mReclen_valS = r;
}

void CAENDevice::DefineChMask(uint64_t mask) {
	mChMask = mask;
}

double CAENDevice::getReclen_valueS() {
	if (mIsVirtualDevice)
		return mReclen_valS;
	else {
		try {
			double sampl = getReclenParam()->getValue().toDouble() / (1e9 * mSample_to_S);
			return sampl - RecLenJIncrS;
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}
double CAENDevice::getReclen_value() {
	if (mIsVirtualDevice)
		return mReclen_valS * (mSample_to_S * 1e9);
	else {
		try {
			double ns = getReclenParam()->getValue().toDouble();
			return ns - (RecLenJIncrS * (mSample_to_S * 1e9));
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}

void CAENDevice::setReclenT(QString value_ns) {
	if (mIsVirtualDevice)
		return;
	QString val_to_set = value_ns;
	int extra_ns = extra_ns = NJExtraSamples * (mSample_to_S * 1e9);
	int tot_ns = value_ns.toInt() + extra_ns;
	int incr = getReclenParam()->getIncr();
	tot_ns = ((tot_ns + incr - 1) / incr) * incr;
	RecLenJIncrS = (tot_ns - val_to_set.toInt()) / (mSample_to_S * 1e9); //real increment considering minimum step

	val_to_set = QString("%1").arg(tot_ns);

	try {
			getReclenParam()->setValue(val_to_set);
		}
	catch (const CAENFELibException& exc) {
			throw exc;
	}

}

double CAENDevice::getReclen_min() {
	if (mIsVirtualDevice)
		return 0;
	else {
		try {
			return getReclenParam()->getMin();		
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}
double CAENDevice::getReclen_max() {
	if (mIsVirtualDevice)
		return 1e6;
	else {
		try{
		return getReclenParam()->getMax();
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}
double CAENDevice::getReclen_incr() {
	if (mIsVirtualDevice)
		return 1;
	else {
		try{
			return getReclenParam()->getIncr();
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}
double CAENDevice::getReclen_minS() {
	if (mIsVirtualDevice)
		return 0;
	else {
		try{
			return getReclenParam()->getMin() / (1e9 * mSample_to_S);
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}
double CAENDevice::getReclen_maxS() {
	if (mIsVirtualDevice)
		return 1e6;
	else {
		try{
			return getReclenParam()->getMax() / (1e9 * mSample_to_S);
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}
	}
}
double CAENDevice::getReclen_incrS() {
	if (mIsVirtualDevice)
		return 1;
	else {
		try {
			return getReclenParam()->getIncr() / (1e9 * mSample_to_S);
		}
		catch (const CAENFELibException& exc) {
			throw exc;
		}

	}
}

double CAENDevice::getThr_value(int ch) {
	if (mIsVirtualDevice)
		return mChThreshold.at(ch);
	else {
		double th = getThrParam(ch)->getValue().toDouble();
		mChThreshold.replace(ch, th);
		return th;
	}
}
void CAENDevice::setThr_value(int ch, QString value) {
	if (mIsVirtualDevice)
		mChThreshold.replace(ch, value.toDouble());
	else {
		getThrParam(ch)->setValue(value);
		mChThreshold.replace(ch, value.toDouble());
	}
}
double CAENDevice::getThr_min(int ch) {
	if (mIsVirtualDevice)
		return 0;
	else
		return getThrParam(ch)->getMin();
}
double CAENDevice::getThr_max(int ch) {
	if (mIsVirtualDevice)
		return 1e6;
	else
		return getThrParam(ch)->getMax();
}
double CAENDevice::getThr_incr(int ch) {
	if (mIsVirtualDevice)
		return 1;
	else
		return getThrParam(ch)->getIncr();
}

uint64_t CAENDevice::getMaxRawDataSize() {
	if (mIsVirtualDevice)
		return 0;
	char val[32];
	CAEN_FELib_ErrorCode err = (CAEN_FELib_ErrorCode)CAEN_FELib_GetValue(mHandle, "/par/MaxRawDataSize", val);
	if (err != CAEN_FELib_Success) {
		throw CAENFELibException("Error getting max raw data size for " + mName, err);
	}
	return (atoll(val));
}


CAENDevice::~CAENDevice(){
}


























