/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		eventBuildThread.h
*	\brief
*	\author
*
******************************************************************************/

#ifndef EVENTBUILDTHREAD_H
#define EVENTBUILDTHREAD_H

#include <QThread>
#include <QFuture>
#include <QFile>
#include "CAENDevice.h"
#include "Statistics.h"
#include "FFT.h"

#define MAX_NDEV	10
#define MAX_NCH		64

#define WAITDATA_TIMEOUT_MS				100			// Timeout while wating for data from a device (in ms)
#define STATS_UPDATE_TIME	1000 //statistics time in ms
#define DEFAULT_COINCWINDOW	100
#define MAX_PLOT_VECTOR_SIZE 65536

class WaveDump2;

class GlobalEvent {
	std::uint64_t TimeStamp_ns;								// Timestamp of the global event (ns)
	uint32_t EventID;									// ID of the global event
	uint16_t GlobalFlags;								// Flags of the global event
	QVector<bool> IsDevPresent;							// Dev is present in the event
	QVector<bool> IsChPresent;							// Channel is present in the event (per board channel)
	QVector<uint32_t> WaveSkew_ns;						// Waveform Time Skew in ns (respect to TimeStamp_ns) 
	QVector<int> BEvID;									// Event ID of the single boards
	//uint16_t ChFlags[DGTZ_MAX_NBRD][DGTZ_MAX_NCH];		// Channel Flags
	QVector<uint16_t> Energy;		// Energy of the global event (charge or pulse height), one for each channel
	QVector<uint16_t *> Wave;		// Pointer to waveform

public:
	QVector<QVector<uint32_t>> NumSamples;	// Num of samples in the waveform
	QVector<uint> MaxSizes;		// max sizes of the events of the deices
	~GlobalEvent() {
		Energy.clear();
		NumSamples.clear();
		std::for_each(Wave.begin(), Wave.end(), [](std::uint16_t* p) { delete[] p; });
	}

	void init(int ndev, int nch, int numsamples) {
		int size = ndev * nch;
		Energy.resize(size);
		NumSamples.resize(ndev);
		for (int s = 0; s < ndev; s++) {
			NumSamples[s].resize(nch);
			NumSamples[s].fill(0);
		}
		Wave.resize(size);
		IsChPresent.resize(size);
		IsChPresent.fill(false);
		IsDevPresent.resize(size);
		IsDevPresent.fill(false);
		WaveSkew_ns.resize(ndev);
		WaveSkew_ns.fill(0);
		std::generate(Wave.begin(), Wave.end(), [numsamples] { return new std::uint16_t[numsamples]; });
		MaxSizes.resize(ndev);
		BEvID.resize(ndev);
		BEvID.fill(0);
	}
	void clearDevSamples(int dev) { NumSamples[dev].fill(0); }
	std::uint64_t *T() { return &TimeStamp_ns; }
	std::uint32_t *ID() { return &EventID; }
	std::uint16_t *Fl() { return &GlobalFlags; }
	std::uint16_t *E() { return Energy.data(); }
	//std::uint32_t *Nsampl() { return NumSamples.data(); }
	std::uint16_t **W() { return Wave.data(); }
	bool *IsCh() { return IsChPresent.data(); }
	bool* IsDev() { return IsDevPresent.data(); }
	std::uint32_t *Wskew() { return WaveSkew_ns.data(); }
	int* BoardEvID() { return BEvID.data(); }
	QVector<uint32_t> Samples(int dev) { return NumSamples[dev]; }
};

class eventBuildThread : public QThread
{
	Q_OBJECT

public:
	eventBuildThread(WaveDump2 *w, QList<CAENDevice *> *devices);
	~eventBuildThread();
	void setUoMX(int uom) { mUoM_x = uom; mUpdateXUnit = true; }
	void setUoMY(int uom) { mUoM_y = uom; }
	void updatePlots(int plot, const QString& deviceName, int channel);
	void run();
	void ClearPlotArray() { mClearVector = true; }

private:
	QList<CAENDevice *> *mDevices;
	WaveDump2 *mWaveDump;
	bool mEndThread = false;
	bool mDoRead = false;
	int mTotDevices;
	int mNSamples;
	int mUoM_x = 0;
	int mUoM_y = 0;
	bool mUpdateXUnit = false;
	int mPlotType;
	double mCoincWindow;
	bool mOnly_matching = false;
	int PLOT_TIME;
	QString mGlobalStartMode;
	QMutex mMtx;
	QVector<double> mSample_to_V;
	QVector<double> mGain;
	QHash<int, int> mHashPlot; //plot index (key), dev_ch index (value)
	QVector<QVector<double>> mPlotVectorY;
	QVector<QVector<double>> mPlotVectorX;
	QVector<bool> mShowTrace;
	QVector<double> mNSkew;
	QMutex mGMtx;
	GlobalEvent *mGlobalEvent;
	Statistics *mStats;
	QFuture<void> mNew_plot;				// plot vector thread
	FFT mFFT;
	QFile *mSaveFile;
	QTextStream out;
	QString mFile_prefix;
	QString mFile_folder;
	QString mFile_ftype;
	QString mFile_header;
	QString mFile_format;
	QString mFile_sync;
	int mNEvts_file = 0;
	QFile *mTestFile;
	QTextStream outtest;

	bool mSaving = false;
	bool mClearVector = false;
	void RunSyncMode();
	void RunAsyncMode();
	void RunSyncIndependentMode();
	void fillPlotVector(GlobalEvent * ev);
	void fillPlotVectorSkew(GlobalEvent * ev); //fill the vector taking into account the delay
	void updateXVector();
	void InitFiles();
	void SaveEventToFile(GlobalEvent* event);
	void RunSyncOfflineMode();
	void RunSyncIndependentOfflineMode();

signals:
	void plotData(const QVector<QVector<double>>&, const QVector<QVector<double>>&, QVector<bool>, QVector<double>, double MaxY);
	void GenericError(QString);

public slots:
	void stopRun();
	void startRun();
};


#endif
