/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2020-2022 CAEN SpA
*
*	This file is part of WaveDump2.
*
*	WaveDump2 is free software; you can redistribute it and/or
*	it under the terms of the GNU General Public License as published
*	by the Free Software Foundation; either version 3 of the License, or
*	(at your option) any later version.
*
*	WaveDump2 is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with WaveDump2; if not, see https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: GPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		WaveDump2.h
*	\brief
*	\author
*
******************************************************************************/

#ifndef WAVEDUMP2_H
#define WAVEDUMP2_H

#include <cstdint>

#include <QtWidgets/QMainWindow>
#include <QStringList>
#include <QString>
#include <QtConcurrent/QtConcurrentRun>

#include <CAEN_FELib.h>

#include "CAENDockWidget.h"
#include "CAENQAction.h"
#include "manualControllerPanel.h"
#include "controlPanel.h"
#include "deviceManagerPanel.h"
#include "logPanel.h"
#include "plotPanel.h"
#include "CAENparameter.h"
#include "CAENDig2Device.h"
#include "CAENDevice.h"
#include "statisticsPanel.h"
#include "deviceReadoutThread.h"
#include "eventBuildThread.h"
#include "Statistics.h"
#include "CAENDig2Script.h"
#include "infoPanel.h"


#define UOM_PHYS_UNIT 0
#define UOM_SAMPLE    1


#define MAX_CHANNELS 64
#define MAX_GROUPS (MAX_CHANNELS / 2)
#define MAX_VGA_GROUPS 4

#define	PLOT_TRACES		8
#define MAX_SW_RECLEN_SAMPLES	6250000

#ifdef WIN32
	#define ENDL  "\r\n"
	#define DIRSEP "/"
#else
	#define ENDL  "\n"
	#define DIRSEP "/"
#endif

#define WAVEDUMP2_DIR "WaveDump2"
#define DEFAUL_CONFIG_FILENAME "WaveDump2_Config.wconf"

class WaveDump2 : public QMainWindow
{
	Q_OBJECT

public:
	WaveDump2(QWidget *parent = Q_NULLPTR);
	QWidget *centerWidget;
	
	QAtomicInt mDataSavedEnabled;

	std::uint64_t openDeviceExec(int type, const QString& address, const QString& name, bool updateConfig);
	uint64_t ReConnect();
	void abortConnection();
	void add2LogAndConf(const QString& cmd, const QString& bname, const QString& qry, const QString& val, const QString& valS, CAEN_FELib_ErrorCode ErrCode);
	void addLog(const QString& cmd, bool ok);
	void setCurrentDeviceName(const QString& name);
	QString getCurrentDeviceName() { return mCurrentDeviceName; }
	int getNConnectedDevices() { return mConnectedDevices.size(); }
	QString getConfigValue(const QString& section, const QString& key);
	void setConfigValue(const QString& section, const QString& key, const QString& value);
    void setOutputSettings(QString folder, QString prefix, QString ftype, QString header, QString format, QString sync, int nevts, bool saveconfig);
	void getOutputSettings(QString &folder, QString &prefix, QString &ftype, QString &header, QString &format, QString &sync, int &nevts);
	void XUnitChanged(int);
	void YUnitChanged(int);
	int getUoMX() { return mUoMX; }
	int getUoMY() { return mUoMY; }
	int getPlotScaleMode() { return mPlotPan->getPlotScale(); }
	double getPlotMinX() { return mPlotPan->getMinX(); }
	double getPlotMaxY() { return mPlotPan->getMaxY(); }
	double getPlotMinY() { return mPlotPan->getMinY(); }
	void ReclenChanged(double);
	void updatePlotYAxisRange();
	void updatePlotXAxisRange();
	uint32_t getRecLenS() { return mRecLenS; } //get the recordlength of the active device
	uint32_t getMaxRecLenS();////get the reclen to be used to resize the plot (max reclen)
	double getMaxRecLen_us();////get the reclen to be used to resize the plot (max reclen in us)
	uint32_t getALLMaxRecLenS();////get the max reclen considering ALL devices(for start ALL in case of different reclens)
	uint32_t getALLMaxRecLenS_corr();////get the max reclen including the extra samples considering ALL devices
	uint32_t getDevReclenS(CAENDevice* dev) {
		return dev->getReclenParam()->getCachedValue().toUInt();
	}
	uint32_t getPreTrgS() { return mPretrgS; } //get the pretrigger of the active device
	uint32_t getDevPreTrgS(CAENDevice* dev) {
		return dev->getPreTrgCachedValue();
	}
	double getSampleToS() { return mSample_to_s; } //get sample_to_s of active device
	double getDevSampleToS(CAENDevice* dev) {
		return dev->getSample_to_S();
	}
	double getMaxSampleToS();////get the sample_to_s to be used to resize the plot (max sample_to_s)
	double getSampleToV() { return mSample_to_V; } 
	//double getMaxSampleToV();////get the sample_to_V to be used to resize the plot (max sample_to_V)
	double getDevChSampleToV(QString devname, int ch);
	double getDevChGain(QString devname, int ch);
	double getThrSampleTomV_f(QString devname, int ch);
	double getThrSampleTomV_off(QString devname, int ch);
	double getTSampl_ns() { return mSample_to_s * 1e9; }
	double getDevTSampl_ns(CAENDevice* dev) {
		return dev->getSample_to_S() *1e9;
	}
	int getDecimation() { return mDecimation; } //get the decimation factor of the active device
	int getMaxDecimation();////get the decimation to be used to resize the plot (max decimation)
	int getDevDecimation(CAENDevice* dev) {
		return dev->getDecimation();
	}
	double getCoincWin() { return mCoincidence_win; }
	void setCoincWin(double value) { mCoincidence_win = value; }
	int getADCNBit(QString dev_name);
	int getActivePlotTraceDevIndex();
	int getPlotTraceDevIndex(int trace);
	int getActivePlotTraceCh() { return mPlotPan->getActiveTraceCh(); }
	int getActiveTrace() { return mPlotPan->ActiveTrace; }
	CAENDevice* getPlotTraceDev(int trace);
	CAENDevice* getActiveDev() { return mActiveDevice; }
	bool isTraceVisible(int t) { return mPlotPan->isTraceVisible(t); }
	QString getGlobalSMode() { return mGlobalStartMode; }
	void setGlobalSMode(QString mode) {
		mGlobalStartMode = mode;
		if (mRecordAction) 
			mActions->AddSWAction("GLOBAL_START_MODE_CHANGE", mode);
		if (mGlobalStartMode != "SW_ASYNC")
			mOutput_sync = "YES";
		else
			mOutput_sync = "NO";
	}
	void setGlobalTrgMode(QString mode) {
		mGlobalTrgMode = mode; 
		if (mRecordAction)
			mActions->AddSWAction("GLOBAL_TRIGGER_MODE_CHANGE", mode);
	}
	QString getGlobalTrgMode() { return mGlobalTrgMode; }
	void setTrgSource(QString new_trg_src);
	void setTrgMode(int mode);
	QString getSelfTrgMask();
	QString getCommonSelfTrgMask();
	QString getCommonTrgOutMask();
	QString getCommonTrgSource();
	void setCommonMask(QString m, QString mask_ref_par);
	QString getTrgoutMask();
	int getNumChannels();
	void setSelfTrgMask(QString mask);
	void HandleThrMode();
	qint64 getAcqtime(int dev);
	int getPlotType() { return mPlotType; }
	Statistics *Stats() { return mStatistics; }
	bool stats_panel_open = false;
	bool baseline_changing = false;
	int RunID = 0;
	int NDevStarted = 0;
	int NAutoTrg = 0;
	bool OnlyMatching = false;
	void OnlyMatchingEnable(bool enable);
	bool OfflineMode = false;
	QHash<QString, QString>CommonGroupValuesmap[MAX_CHANNELS/2];
	QHash<QString, QString>CommonChValuesmap[MAX_CHANNELS];
	QHash<QString, QString>CommonDevValuesmap;
	QHash<QString, QString>CommonGroupValuesmap1[MAX_CHANNELS / 2];
	QHash<QString, QString>CommonChValuesmap1[MAX_CHANNELS];
	QHash<QString, QString>CommonDevValuesmap1;
	QTimer *Statistics_Timer;
	QTimer* Check_connection_Timer;
	double PlotInterval_ms = 300;
	bool isSavingActions() { return mRecordAction; }
	CAENDig2Script* getScript() { return mActions; }
	//double getMaxGain();////get the Gain to be used to resize the plot (max gain)
	uint32_t getMaxVRangeS();
	double getMaxVRangeVForPlot();
	void CheckConnectedDevStatus();
	bool IsFastSaveEnabled() { return mFastSaveEnable; }
	bool IsManualOfflineEnabled() { return mManualOfflineAcq; }
	void enableFastSave(bool enable) { mFastSaveEnable = enable; }
	void enableDataSave(bool enable) {
		mSaveOutputAct->setChecked(enable);
		saveOutput();
	}
	uint64_t swapByteOrderWord(uint64_t ull);
	uint64_t getLastEvTimetag(int dev) { return mConnectedDevices.at(dev)->LastEvTimetag; }
	bool isDevAcquiring(int dev) { return (bool)mConnectedDevices.at(dev)->AcqStarted; }
	void manageAddLogAndConf(QString param_name, CAENDevice* dev, QString value, QString valueS);


private:
	QMenu *mFileMenu;
	QMenu *mConfiguration;
	QMenu *mScriptMenu;
	QMenu *mOutputMenu;
	QMenu *mWindowMenu;
	QMenu *mHelpMenu;

	QAction *mConnectAct;
	QAction *mDisconnectAct;
	QAction *mDisconnectMenuAct;
	QAction *mConnectMenuAct;
	QAction *mInfoMenuAct;
	QAction *mScanAct;
	QAction *mInfoAct;
	QAction *mScanMenuAct;
	QAction *mLoadAct;
	QAction *mLoadMenuAct;
	QAction *mLoadRawMenuAct;
	QAction *mSaveAct;
	QAction *mSaveMenuAct;
	QAction *mSaveasAct;
	QAction *mSaveasMenuAct;
	QAction *mExitAct;
	QAction *mExitMenuAct;
	QAction *mBrowseAct;
	QAction *mManualControllerAct;
	QAction *mLogAct;
	QAction *mStatAct;
	QAction *mSaveasSettingsAct;
	QAction *mOpenSettingsAct;
	QAction* mOpenSettingsMenuAct;
	QAction *mSaveOutputAct;
	QAction *mOutputConfigAct;
	QAction* mAboutAct;
	QWidget *mCenterWidget;
	QAction* mErrorsMenuAct;

	// Data Save Global Variable
	QString mOutput_folder;
	QString mOutput_prefix;
	QString mOutput_ftype;
	QString mOutput_header;
	QString mOutput_format;
	QString mOutput_sync;
	int mNEvts_file=0;
	
	//
	manualControllerPanel *mManualControllerPan;
	controlPanel *mControlPan;
	deviceManagerPanel *mDeviceManagerPan;
	logPanel *mLogPan;
	plotPanel *mPlotPan;
	QMenuBar *mMenu;
	QString mDataSaveFilename = "";
	QToolBar *mMainToolBar;
	QLineEdit* mDefConfigFile;
	QCheckBox* mAutoLoadFile;
	QStatusBar *mMainStatusBar;
	QPushButton* mStatusLED;
	QPushButton* mResetButton;
	QFuture<bool> mAsyncConn;
	CAENDockWidget *mDeviceManagerDock;
	CAENDockWidget *mLogDock;
	QDockWidget *mPlotDock;
	CAENDockWidget *mManualControllerDock;
	statisticsPanel *mStatsPan;
	Statistics *mStatistics;
	CAENDockWidget  *mStatsDock;
	//QString mTmpConfigFileName;
	bool mCouldSave = false;
	QString mConfigFileName;
	QSettings *mConfigFile;
	QProgressBar *mMainProgressBar;
	QLabel *mMainStatusBarText;
	QString mCurrentDeviceName;
	int mConnected=0;
	int mNumberOfDevice;
	bool mConnectionAbortCmd = false;
	eventBuildThread *mEventBuildthread = nullptr;
	QList<deviceReadoutThread *> mDevicesReadoutThreads;
	QList<CAENDevice *> mConnectedDevices;
	CAENDevice *mActiveDevice= nullptr;
	infoPanel* mInfo=nullptr;
	int mLoadDeviceManager = -1;
	QHash<QString, uint64_t> mHashConfig;
	double mRecLenS;
	double mPretrgS;
	int mDecimation=1;
	double mSample_to_s = 1.0;
	uint32_t mVRangeS; //range of the active device (given by ADC nbits)
	double mSample_to_V = 1.0; //sample to v of the active device
	double mCoincidence_win; //coincidence window for sync (ns unit)
	int mUoMX = UOM_PHYS_UNIT;
	int mUoMY = UOM_PHYS_UNIT;
	int mPlotType = PLOT_TYPE_WAVEFORMS;
	QString mGlobalStartMode = "SW_ASYNC";
	QString mGlobalTrgMode = "INDEPENDENT";
	CAENDig2Script * mActions = nullptr;
	bool mIgnoreEvent = false;
	bool mRecordAction = false;
	double mRecordStopTime = 0;
	bool mFastSaveEnable = false;
	bool mManualOfflineAcq = false;
	QElapsedTimer mRecordStopTimer;
	void populateToolBar();
	void populateMenu();
	void paintEvent(QPaintEvent *);
	void increaseProgressBar();
	void createDockWindows(int widthPlot);
	bool createDeviceTree(const QString& addr, uint64_t rootHandle);
	bool addNode(CAENTreeWidgetItem *father, uint64_t handle, const QString& qry);
	void handleConfFileModified();
	void closeEvent(QCloseEvent *event);
	QStringList findChannels(const QString& qry);
	QStringList findGroups(const QString& qry);
	void saveConfig(bool as);
	void setActiveDevice(CAENDevice *dev);
	void updateActiveDevInfo(CAENDevice *dev);
	void fillCommonDefValues();
	void InitConfigFile(int dev_type);
	void UpdateStatus();
	void UpdateStatusFast();
	void SaveLastConnectionData(int type, QString conn_data);
	bool ExecSWCommand(QString command, QString val);
	void CheckDefaultConfigToLoad();
	void RefreshDevSettings();
	int CreateDeviceFromFile(QString filename, int index);
	int getDeviceTypeCode(QString type);
	void createReadoutThreadsAndStart();
	void checkStartPbEnable();
	void CheckLoadLastConfig(QString pid);

signals:

public slots:
	
	void connect2Device();
	void disconnectDevice();
	void UpdateAfterDisconnection(CAENDevice* device);
	void browseToggle();
	void logToggle();
	void manualToggle();
	void statsToggle();
	void logCloseEvent(QCloseEvent *e);
	void browseCloseEvent(QCloseEvent *e);
	void manualControllerCloseEvent(QCloseEvent *e);
	void statsCloseEvent(QCloseEvent *e);
	void save();
	void load();
	void LoadRawFromFile();
	void loadFile(QString filename, bool check_connect);
	void scanDevices();
	void openSettings();
	void openSettingsFile(QString filename);
	void saveAs();
	void saveOutput();
	void openOutputSettings();
	void saveAsSettings();
	void startAcquisition();
	void stopAcquisition();
	void startOfflineAcq();
	void stopOfflineAcq();
	void startOfflineManualAcq();
	void stopOfflineManualAcq();
	bool isSaveEnabled();
	void tracesChanged(QVector<QString>, QVector<int>);
    void showManualController(std::uint64_t handle, const QString& rootName, const QString& Qry, bool f);
	void configureDevice(QString);
	void configureSetup();
	void SWTrg();
	void SingleSWTrg();
	void CheckStartModes();
	void AssignChainIndexes();
	void New_Cursors(bool enable);
	void showCursorsPositions(QVector<double> positions);
	void cursorTraceChanged(int, int);
	void PreTrgChanged(double);
	void ThrChanged(double, bool);
	void DCOffsetChanged(QString, int, double, bool);
	void PlotActiveTraceChanged(QString, int);
	void PlotGenericTraceChanged(QString, int);
	void PlotActiveTraceColorChanged();
	void ActiveDeviceChanged(QString new_dev);
	void UpdateStatsAndStatus();
	void BuildThreadFinished();
	void PlotScaleChanged();
	void DeviceInfo();
	void DeviceErrors();
	void PlotTypeChanged();
	void ReadoutError(QString devname, CAEN_FELib_ErrorCode err);
	void ReadoutError1(QString);
	void ReaderThreadFinished(QString);
	void CheckSettings();
	void AboutWaveDump();
	void CheckConnections();
	void ClearPlotData();
	void PlotToolsOpen(bool open) {
		if (mRecordAction)
			mActions->AddSWAction("TOOLS_SHOW", (open)? "TRUE" : "FALSE");
	}
	void FreezePlot(bool f) {
		if (mRecordAction)
			mActions->AddSWAction("PLOT_FREEZE", (f)? "TRUE" : "FALSE");
	}
	void OpenPalette(bool f) {
		if (mRecordAction)
			mActions->AddSWAction("PALETTE_SHOW", (f) ? "TRUE" : "FALSE");
	}
	void NWavesAvgChanged(int n) {
		if (mRecordAction)
			mActions->AddSWAction("NWAVES_SET", QString("%1").arg(n));
	}
	void TraceEnableChanged(int t, int enable); 
	void TracePlotChanged(int t, QString dev, int ch);
	void DefaultLoadEnable(bool enable);
	void CheckTracesEnableStatus();
	void ResetDevice(QString devname);
	void RebootDevice(QString devname);
	void Reset();
	void quit_info();
	void PlotTimeChanged(double val);
	void NewEvLoadedOK(QString name, int ev, quint64 t);
	void LoadNewEvent(QString, int);
	void NewEvLoadError(QString name, int ev);
	void PaletteChMask(QString, quint64);
};
#endif
