#pragma once
#ifndef CAENDEVICE_H
#define CAENDEVICE_H
#include <QObject>
#include <QVector>
#include <QMutex>
#include <QElapsedTimer>
#include <QWaitCondition>
#include "CAENDig2Event.h"
#include "CAENparameter.h"

#define MAX_RAW_BUFFS 6
#define MAX_NEVTS	30

#define CAEN_DIG2	0
#define CAEN_DIG1	1

#define TSAMPL_UNIT_NS_DIG2 8
#define TSAMPL_UNIT_NS_DIG1_24	20
#define TSAMPL_UNIT_NS_DIG1	16

#define CTYPE_ETH_DIG2		0
#define CTYPE_USB_DIG2		1
#define CTYPE_USB_DIG1		100
#define CTYPE_USBA4818_DIG1	101
#define CTYPE_USBV4718_DIG1	102
#define CTYPE_OPTILINK_DIG1	103
#define CTYPE_ETHV4718_DIG1 104

#define TSource_TrgIn		  0x1
#define TSource_P0			  0x2
#define TSource_SwTrg		  0x4
#define TSource_LVDS		  0x8
#define TSource_ITLA		  0x10
#define TSource_ITLB		  0x20
#define TSource_ITLA_AND_ITLB 0x40
#define TSource_ITLA_OR_ITLB  0x80
#define TSource_EncodedClkIn  0x100
#define TSource_GPIO          0x200
#define TSource_TestPulse     0x400


class CAENDevice
{
public:
	CAENDevice() {};
	~CAENDevice();
	void CloseConnection();
	void createEvtsQueue(std::size_t size, int length);
	void freeEvtsQueue();
	uint32_t getVRangeS();
	double getVRangeV(int ch); //takes into account the ADC calibration factor for ch (values should be stored in flash) and the Input Gain of ch (user selectable)
	double getSample_to_V(int ch);
	uint64_t getMaxRawDataSize();
	void updateGains();
	double getMaxGain();
	int getDecimation_value();
	void ArmAcquisition();
	void DisarmAcquisition();
	void ClearData();
	void Reset();
	uint64_t getWHandle();
	uint64_t getRawHandle();
	void EnableRawDataFormat();
	void SendSWTrg();
	QString getFamilyCode();
	QString getModel();
	QString getFWType();
	QString getSN();
	void DefineInfo(int index, uint32_t pid, uint32_t ADCNBit, uint32_t SampleToS, uint16_t NCh, uint32_t* baseline, uint32_t* factor);
	void DefineGains(uint32_t* gains);
	void DefineDecimation(uint16_t d);
	void DefineReclen(uint32_t r);
	void DefineChMask(uint64_t mask);
	void DefineThresholds(int32_t* thrs);
	void DefineFamily(uint32_t code);
	double getReclen_valueS();//value in samples
	double getReclen_value();//value in ns
	double getReclen_min();
	double getReclen_max();
	double getReclen_incr();
	double getReclen_minS();
	double getReclen_maxS();
	double getReclen_incrS();
	void setReclenT(QString value_ns);
	CAENparameter* getReclenParam() { return mReclen; }
	double getThr_value(int ch);
	double getThrCachedValue(int ch) { return mChThreshold.at(ch); }
	void setThr_value(int ch, QString value);
	double getThr_min(int ch);
	double getThr_max(int ch);
	double getThr_incr(int ch);
	void resetTime() { mStartAcqTimer.restart(); }
	void setVirtualAcqStatus(QString status) { mVirtualAcqStatus = status; }
	const QString& getName() const noexcept { return mName; }
	const QString& getCType() const noexcept { return mConnType; }
	const QString& getAddr() const noexcept { return mAddress; }
	std::uint64_t getHandle() const noexcept { return mHandle; }
	int getNumCh() const noexcept { return mNumCh; }
	int getADCNBit() const noexcept { return mADC_NBit; }	
	double getSample_to_S() const noexcept { return mSample_to_S; }
	void setIndex(int index) noexcept { mIndex = index; }
	int getIndex() const noexcept { return mIndex; }
	QString getTrgSourceCachedValue() { return mTrgSource; }
	uint64_t getEPHandle() const noexcept { return mWHandle; }
	uint64_t getRAWHandle() const noexcept { return mRawHandle; }
	void closeEvtsQueue() { mCloseQueue = true; }
	bool isQueueClosed() const noexcept { return mCloseQueue; }
	QVector<CAENDig2Event *> *EventsQueue() { return &mDevEventsQueue; }	
	QString getSNFast() { return mSN; }
	uint32_t getReclenCachedValue() {
		if (mIsVirtualDevice)
			return mReclen_valS * (mSample_to_S * 1e9);
		else
			return QString(mReclen->getCachedValue()).toUInt();
	}
	int getBaselineLSB(int ch) const noexcept { return (1LL << mADC_NBit) * ChBaselines[ch] / 100.; }
	qint64 TimeFromStart() {
		if (mIsVirtualDevice)
			return LastEvTimetag;
		else
			return mStartAcqTimer.elapsed();
	}
	void DefineRawDataFile(QString filename) { mRawDataFile = filename; }
	QString getRawDataFile() { return mRawDataFile; }
	bool isVirtual() { return mIsVirtualDevice; }
	double getCachedChGain(int ch) { return ChGain.at(ch); }
	double getPreTrgCachedValue() { return mPreTrg_valS * (mSample_to_S * 1e9); }
	double getPreTrgCachedValueS() { return mPreTrg_valS; }
	void setTrgMode(int mode) { mTrgMode = mode; }
	int getTrgMode() { return mTrgMode; }
	int getVgaGainGroupSize() const noexcept { return mVgaGainGroupSize; }
	void MallocRawBuffers();
	void FreeRawBuffers();
	int getTSampl_ns() { return TSamplUnit_ns; }
	
	virtual uint32_t getErrorFlags() = 0;
	virtual void ApplyDefaultConfig(QHash<QString, QString> dpars, QHash<QString, QString>* cpars, QHash<QString, QString>* gpars) = 0;
	virtual void updateBaselines() = 0;
	virtual double getChOffset(int ch) const = 0;
	virtual double getReclen_us() = 0;
	virtual void setChOffset(int ch, double value) = 0;
	virtual void updateChSample_to_V(int ch) = 0;
	virtual double getChGain(int ch) = 0;
	virtual void setChGain(int ch, double value) = 0;
	virtual int getChThreshold(int ch) = 0;
	virtual void setChThreshold(int ch, int val) = 0;
	virtual CAENparameter* getThrParam(int ch) = 0;
	//virtual CAENparameter* getReclenParam(int ch) = 0;
	virtual void setStartSource(QString value) = 0;
	virtual QString getStartSource() = 0;
	virtual void setRunDelay(int value) = 0;
	virtual void setTRGOUTMode(QString value) = 0;
	virtual QString getTRGOUTMode() = 0;
	virtual QString getClkSource() = 0;
	virtual void setClkSource(QString value) = 0;
	virtual QString getTrgSource() = 0;
	virtual void setTrgSource(QString value) = 0;
	virtual uint32_t calcTrgSourceToSave() = 0;
	virtual void setSelfTrgMask(QString val) = 0;
	virtual QString getSelfTrgMask() = 0;
	virtual void setDecimation(int val) = 0;
	virtual int getDecimation() = 0;
	virtual void StartAcquisition()=0;
	virtual void StopAcquisition()=0;
	virtual QString getAcqStatus() = 0;
	virtual void SetDataFormat(QString format) = 0;
	virtual QString getFWRel() = 0;
	virtual QString getFormFactor() =0;
	virtual bool isChEnabled(int ch) = 0;
	virtual QString DecodeFlagsMask(uint32_t mask) = 0;
	virtual void DefinePreTrigger(uint32_t p) = 0;
	virtual void DefineTrgSource(uint32_t t) = 0;
	virtual double getPreTrg_value()=0;
	virtual double getPreTrg_valueS() = 0;
	virtual double getPreTrg_min()=0;
	virtual double getPreTrg_max()=0;
	virtual double getPreTrg_incr()=0;
	virtual double getPreTrg_minS() = 0;
	virtual double getPreTrg_maxS() = 0;
	virtual double getPreTrg_incrS() = 0;
	virtual void setScopeChannel(QString value) = 0;
	virtual QString getScopeChannel() = 0;
	virtual void setGPIOMode(QString value) = 0;
	virtual QString getGPIOMode() = 0;
	virtual QString getFWGbE() = 0;
	virtual void setThrMode(QString mode) = 0;
	virtual void setPreTrg(QString value) = 0;
	virtual QString getTrgOutMask() = 0;
	virtual void Reboot() = 0;
	virtual void setSyncOutMode(QString value) = 0;
	virtual QString getClkOutFP() = 0;
	virtual void setClkOutFP(QString value) = 0;
	virtual void setVETOSource(QString veto) = 0;
	virtual QString getVETOSource() = 0;
	virtual void setIOLevel(QString val) = 0;
	virtual QString getIOLevel() = 0;
	virtual void setGlobalStartMode(QString mode, int delay)=0;
	virtual bool isInfoStartWord(uint64_t word) = 0;
	virtual bool isInfoStopWord(uint64_t word) = 0;
	virtual uint64_t getChEnableMask() = 0;
	virtual double getReclen_ns() = 0;

	int DeviceClass;
	bool ErrorsReported = false;
	bool ConnectionOpen = true;
	QMutex queue_mutex;
	QWaitCondition waitCondition;
	bool Triggered = false;
	int NEvtsInQueue = 0;
	int NTotEvents = 0;
	int NTotBytesRead = 0;
	int AcqStarted = 0;
	mutable QVector<double> ChBaselines;
	mutable QVector<double> ChGain;
	int DevChainIndex = -1;
	uint64_t LastEvTimetag = 0;//time in ms
	int NJExtraSamples = 0;
	int RecLenJIncrS = 0;
	int TSamplUnit_ns;
	int Wcut[MAX_NEVTS] = { 0 };
	bool SWTrg_sent = false;

	char *RawDataBuffer[MAX_RAW_BUFFS] = { 0 };
	int NBuffFilled = 0;
	QMutex BuffMutex;
	int BuffOccupancy[MAX_RAW_BUFFS] = { 0 };
	



protected:
	CAENDevice(QString name, QString CType, QString address, uint64_t handle, bool isvirtual);

	virtual void FillInfo() = 0;
	virtual void CreateParams() = 0;
	

	uint64_t mHandle;
	uint64_t mWHandle;
	uint64_t mRawHandle;
	QString mName;
	QString mConnType;
	QString mAddress;
	QString mSN;
	QString mFormFactor;
	int mDecim=1;
	uint32_t mReclen_valS = 0;
	uint32_t mPreTrg_valS = 0;
	uint64_t mChMask = 0;
	QString mTrgSource = "";
	int mVgaGainGroupSize = 1;
	int mTrgMode = 0;
	int mIndex=0; //index inside a list
	int mNumCh = 0;
	double mSample_to_S = 1.;
	int mADC_NBit = 14;
	double mInputRange = 0;
	bool mIsVirtualDevice = false;
	QString mModel;
	CAENparameter* mReclen=nullptr;
	QVector<CAENDig2Event*> mDevEventsQueue;
	bool mCloseQueue = true;
	QVector<float> mChADCGain;
	QVector<double> mChADCToVolts;
	QVector<double> mChThreshold;
	QElapsedTimer mStartAcqTimer;
	QString mVirtualAcqStatus="Ready";
	QString mRawDataFile;



};


#endif

