/******************************************************************************
*
* CAEN SpA - Front End Division
* Via Vetraia, 11 - 55049 - Viareggio ITALY
* +390594388398 - www.caen.it
*
***************************************************************************//**
* \note TERMS OF USE:
* This program is free software; you can redistribute it and/or modify it under
* the terms of the GNU General Public License as published by the Free Software
* Foundation. This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. The user relies on the
* software, documentation and results solely at his own risk.
******************************************************************************/

#define ZLErunner_Release        "1.5"
#define ZLErunner_Release_Date   "September 2025"

#include "CAENDigitizer.h"
#include "CAENDigitizerType.h"
#include "ZLEDemoFunc.h"
#include <sys/stat.h>

char path[128];

static long get_time()
{
	long time_ms;
#ifdef WIN32
	struct _timeb timebuffer;
	_ftime(&timebuffer);
	time_ms = (long)timebuffer.time * 1000 + (long)timebuffer.millitm;
#else
	struct timeval t1;
	struct timezone tz;
	gettimeofday(&t1, &tz);
	time_ms = (t1.tv_sec) * 1000 + t1.tv_usec / 1000;
#endif
	return time_ms;
}

int main(int argc, char *argv[])
{
	int ev, board, chan; // loop indices
	int *handle = NULL; // handle pointer
	ERROR_CODES ErrCode = ERR_NONE; // global demo error code
	// Data and config structures
	CAEN_DGTZ_730_ZLE_Event_t      **Event = NULL; // Events equal to the maximum allowed BLT transfer must be allocated 
	ZLEPlot_t                       PlotVar; // struct containing the plot options
	ZLEConfig_t 					ConfigVar; // struct containing the config file options
	// File I/O and related variables
	FILE **RawFile;
	FILE ***WaveFile;
	FILE *f_ini = NULL;
	int *RawFileIndex;
	struct stat buf;
	int FileIndex = 0;
	uint64_t PrevCheckTime; // printout and plot time
	Counter_t *Counter, *CounterOld;
	int PrintFlag,PlotFlag;
	uint32_t AllocatedSize, BufferSize;
	char *buffer = NULL;
	char ConfigFileName[255];
	char tmpConfigFileName[255];
	int *NumEvents;
	int EventPlotted;
	int BLTn, MaxBLTn = 0, MaxBLTnHIndex;
	int RL, MaxRL = 0, MaxRLHIndex;
	uint32_t MSize, MaxMSize = 0;
	int MaxMSizeHIndex;
	struct stat info;
	uint32_t BoardInfo;

	printf("**************************************************************\n");
	printf("                      X725/X730 ZLE Demo %s\n", ZLErunner_Release);
	printf("**************************************************************\n");

#ifdef  WIN32
	sprintf(path, "%s\\ZLE_DEMO\\", getenv("USERPROFILE"));
	_mkdir(path);
#else
	sprintf(path,"");
#endif

	// Open and parse configuration file, init parameters
	if (argc > 1) strcpy(ConfigFileName, argv[1]);
	else {
		strcpy(tmpConfigFileName, DEFAULT_CONFIG_FILE);
		sprintf(ConfigFileName, "%s%s", path, tmpConfigFileName);
	}
	ErrCode = OpenConfigFile(&f_ini, ConfigFileName);
	if (ErrCode == ERR_NONE) ErrCode = ParseConfigFile(f_ini, &ConfigVar);
	if (f_ini != NULL) fclose(f_ini);
	
	// Allocate space for handles and counters according to the number of boards in the acquisition chain
	if (ErrCode == ERR_NONE) {
		if (((handle = calloc(ConfigVar.Nhandle, sizeof(*handle))) == NULL) ||
			((Counter = calloc(ConfigVar.Nhandle, sizeof(*Counter))) == NULL) ||
			((CounterOld = calloc(ConfigVar.Nhandle, sizeof(*CounterOld))) == NULL) ||
			((NumEvents = calloc(ConfigVar.Nhandle, sizeof(*NumEvents))) == NULL) ||
			((Event = calloc(ConfigVar.Nhandle, sizeof(*Event))) == NULL) ||
			((RawFileIndex = calloc(ConfigVar.Nhandle, sizeof(*RawFileIndex))) == NULL) ||
			((RawFile = calloc(ConfigVar.Nhandle, sizeof(*RawFile))) == NULL) ||
			((WaveFile = calloc(ConfigVar.Nhandle, sizeof(*WaveFile))) == NULL)
			) ErrCode = ERR_MALLOC;
	}
	
	// Open the digitizer and read the board information
	if (ErrCode == ERR_NONE) {
		printf("Open digitizers\n");
		if (OpenDigitizer(handle,ConfigVar)) ErrCode = ERR_DGZ_OPEN;
	}
	
	// Print board info
	if (ErrCode == ERR_NONE) {
		printf("Get board info and set board-specific parameters\n");
		for (board = 0; board < ConfigVar.Nhandle; board++) {
			if (CAEN_DGTZ_GetInfo(handle[board],&ConfigVar.BoardConfigVar[board]->BoardInfo)) ErrCode = ERR_BOARD_INFO_READ;		
			else {
				printf("****************************************\n");
				printf("Connected to CAEN Digitizer Model %s\n", ConfigVar.BoardConfigVar[board]->BoardInfo.ModelName);
				printf("Board serial number %d\n", ConfigVar.BoardConfigVar[board]->BoardInfo.SerialNumber);
				printf("ROC FPGA Release is %s\n", ConfigVar.BoardConfigVar[board]->BoardInfo.ROC_FirmwareRel);
				printf("AMC FPGA Release is %s\n", ConfigVar.BoardConfigVar[board]->BoardInfo.AMC_FirmwareRel);
				CAEN_DGTZ_ReadRegister(handle[board], 0x108C, &BoardInfo);
				if (((BoardInfo >> 8) & 0xff) != ZLE_FW_ID) {ErrCode = ERR_WRONG_FW; break;}
			}
		}
		printf("****************************************\n");
	}
	
	// Program the digitizers
	if (ErrCode == ERR_NONE) {
		printf("Program the digitizers\n");
		if (ProgramDigitizers(handle,&ConfigVar)) ErrCode = ERR_DGZ_PROGRAM;
		// get the handles with the highest values of event size, events/block transfer and record length
		for (board = 0; board < ConfigVar.Nhandle; board++) {		
			CAEN_DGTZ_GetMaxNumEventsBLT(handle[board], &BLTn); if (BLTn > MaxBLTn) { MaxBLTn = BLTn; MaxBLTnHIndex = board; }
			CAEN_DGTZ_GetRecordLength(handle[board], &RL); if (RL > MaxRL) { MaxRL = RL; MaxRLHIndex = board; }
			if ((MSize = CheckMallocSize(handle[board])) > MaxMSize) { MaxMSize = MSize; MaxMSizeHIndex = board; }
		}
	}

	// Open the output files
	if (ErrCode == ERR_NONE) {
		if (stat(ConfigVar.OutFilePath, &info) != 0) {
			#ifdef  WIN32
			if (_mkdir(ConfigVar.OutFilePath) != 0) { printf("Output directory %s could not be created. Please verify that the path exists and is writable\n", ConfigVar.OutFilePath); ErrCode = ERR_OUTDIR_OPEN; }
			else printf("Output directory %s created\n", ConfigVar.OutFilePath);
			#else 
			if (mkdir(ConfigVar.OutFilePath, 0777) != 0) { printf("Output directory %s could not be created. Please verify that the path exists and is writable\n", ConfigVar.OutFilePath); ErrCode = ERR_OUTDIR_OPEN; }
			else printf("Output directory %s created\n", ConfigVar.OutFilePath);
			#endif
		}
		if (ErrCode == ERR_NONE) {
			for (board = 0; board < ConfigVar.Nhandle; board++) {
				RawFileIndex[board] = 0;
				if (ConfigVar.OFRawEnable) { if ((ErrCode = OpenRawFile(RawFile + board, board, RawFileIndex[board], ConfigVar.OutFilePath, ConfigVar.OutFileName)) != ERR_NONE) break; }
				if (ConfigVar.OFWaveEnable) {
					if ((*(WaveFile + board) = (FILE**)calloc(ConfigVar.BoardConfigVar[board]->BoardInfo.Channels, sizeof(FILE*))) == NULL) { ErrCode = ERR_MALLOC; break; }
					if ((ErrCode = OpenWaveFile(WaveFile + board, board, ConfigVar.BoardConfigVar[board], ConfigVar.OutFilePath, ConfigVar.OutFileName)) != ERR_NONE) break;
				}
			}
		}
	}
	
	// WARNING: These mallocs must be done after the digitizer programming
	// Allocate memory for the readout buffer. The malloc must be based on the largest record-length set for the boards
	if (ErrCode == ERR_NONE) {
		printf("Readout buffer malloc\n");
		if (CAEN_DGTZ_MallocReadoutBuffer(handle[MaxMSizeHIndex], &buffer, &AllocatedSize)) ErrCode = ERR_MALLOC;
	}
	// Allocate memory for the events. The malloc must be based on the largest event/BLT set for the boards
	if (ErrCode == ERR_NONE) {
		printf("Event malloc\n");
		for (board = 0; board < ConfigVar.Nhandle; board++) {
			if (CAEN_DGTZ_MallocZLEEvents(handle[MaxBLTnHIndex], (void**)&Event[board], &AllocatedSize)) ErrCode = ERR_MALLOC;
		}
	}

	// reset counters
	if (ErrCode == ERR_NONE) {
		for (board = 0; board < ConfigVar.Nhandle; board++) {
			ResetCounter(Counter + board);
			ResetCounter(CounterOld + board);
		}
	}

	// Allocate memory for the waveforms. The malloc must be based on the largest record length set for the boards
	if (ErrCode == ERR_NONE) {
		printf("Waveform malloc\n");
		for (board = 0; board < ConfigVar.Nhandle; board++) {
			for (ev = 0; ev < MaxBLTn; ev++) {
				for (chan = 0; chan < MAX_V1730_CHANNEL_SIZE; chan++) {
					if (CAEN_DGTZ_MallocZLEWaveforms(handle[MaxRLHIndex], (void**)&(Event[board][ev].Channel[chan]->Waveforms), &AllocatedSize)) {ErrCode = ERR_MALLOC;break;}
				}
			}
		}
	}
	// Open the plotter and configure its options
	if (ErrCode == ERR_NONE) {
		printf("Open plotter\n");
		ErrCode = OpenPlotter(&ConfigVar, &PlotVar);
	}
	
	// Readout Loop         
	if (ErrCode == ERR_NONE) {
		printf("[s] start/stop the acquisition, [q] quit, [space key] help\n");
		PrevCheckTime = get_time();
		while (!ConfigVar.Quit) {
			CheckKeyboardCommands(handle, &ConfigVar, &PlotVar);
			if (UpdateTime(ConfigVar.PlotRefreshTime, &PrevCheckTime) && ConfigVar.AcqRun) PrintFlag = 1;
			// if continuous trigger is enabled, send software triggers 
			if (ConfigVar.ContTrigger) for (board = 0; board < ConfigVar.Nhandle; board++) CAEN_DGTZ_SendSWtrigger(handle[board]);
			// Read data from the board 
			for (board = 0; board < ConfigVar.Nhandle; board++) {
				if (CAEN_DGTZ_ReadData(handle[board], CAEN_DGTZ_SLAVE_TERMINATED_READOUT_MBLT, buffer, &BufferSize)) {
					ErrCode = ERR_READOUT;
					break;
				}
				if (board == ConfigVar.BoardPlotted) EventPlotted = -1;
				if (BufferSize > 0) { // The BLT returned data
					Counter[board].ByteCnt += BufferSize;
					// allocate the buffered data in the struct and return the number of events
					if (CAEN_DGTZ_GetZLEEvents(handle[board], buffer, BufferSize / 4, (void**)&Event[board], &NumEvents[board])) {
						ErrCode = ERR_EVENT_BUILD;
						break;
					}
					Counter[board].MB_TS = Event[board][NumEvents[board] - 1].timeStamp;
					Counter[board].TrgCnt += NumEvents[board];
					if (ConfigVar.OFRawEnable) fwrite(buffer, BufferSize, 1, RawFile[board]);
				} else {
					NumEvents[board] = 0;
					continue; // the BLT returned no data, acquisition still on
				} 
				// check file size and open new file if the file size is larger than the value set in the config file
				if (RawFile[board] != NULL) {
					fstat(fileno(RawFile[board]), &buf);
					if ((int)(buf.st_size / MB_SIZE) > ConfigVar.MaxFileSize) { RawFileIndex[board]++; OpenRawFile(&RawFile[board], board, RawFileIndex[board], ConfigVar.OutFilePath, ConfigVar.OutFileName); }
				}
				// Analyze data for each event
				for (ev = 0; ev < NumEvents[board]; ev++) {
					// increase truncate counters if the related header flag is set
					for (chan = 0; chan < MAX_V1730_CHANNEL_SIZE; chan++) {
						if ((ConfigVar.BoardConfigVar[board]->EnableMask & (1 << chan)) && (Event[board][ev].Channel[chan]->fifo_full)) Counter[board].OFCnt[chan]++;
					}
					// Decode raw data into waveforms
					CAEN_DGTZ_DecodeZLEWaveforms(handle[board], Event[board] + ev, NULL);
				}
				// remember the last event of the plotted-board buffer
				if (board == ConfigVar.BoardPlotted) EventPlotted = NumEvents[board]-1;
			}
		
			// Print event info for the selected board
			if ((PrintFlag || ConfigVar.SinglePlot) && ConfigVar.AcqRun) {
				// if plotflag is still 1 no event from the channel selected for plotting was found
				if (PlotFlag == 1 && Counter[ConfigVar.BoardPlotted].ByteCnt != CounterOld[ConfigVar.BoardPlotted].ByteCnt) {
					printf("The board selected for plotting (board #%d) was not present in data\n", ConfigVar.BoardPlotted);
					PlotFlag = 0;
				}
				printf("==========================================\n");
				for (board = 0; board < ConfigVar.Nhandle; board++) {
					printf("Board %d", board); if (board == ConfigVar.BoardPlotted) printf("(plotted) : \n"); else printf("          : \n");
					PrintData(Counter + board, CounterOld + board);
					for (chan = 0; chan < MAX_V1730_CHANNEL_SIZE; chan++) {
						if ((ConfigVar.BoardConfigVar[board]->EnableMask & (1 << chan)) && (Counter[board].OFCnt[chan]!=CounterOld[board].OFCnt[chan])) {
							printf("fraction of truncated events in board %d, channel %d: %.2f%%\n", board, chan,(float)(100 * (float)(Counter[board].OFCnt[chan]-CounterOld[board].OFCnt[chan]))/(float)NumEvents[board]);
						}
					}
					// save waves
					if (ConfigVar.OFWaveEnable) {
						OpenWaveFile(WaveFile + board, board, ConfigVar.BoardConfigVar[board], ConfigVar.OutFilePath, ConfigVar.OutFileName);
						WaveWrite(*(WaveFile + board),*(Event+board), ConfigVar.BoardConfigVar[board]);
					}
				}
				PrintFlag = 0;
				if (ConfigVar.PlotEnable || ConfigVar.SinglePlot) PlotFlag = 1;
				ConfigVar.SinglePlot = 0; // reset one-shot plot ("p" key during acquisition) 
		
			}
			if(PlotFlag==1 && EventPlotted != -1) {
				PlotEvent(&ConfigVar, &PlotVar, Event, ConfigVar.PlotType);
				PlotFlag = 0;
			}
		}
	}

	if (ErrCode) {
		printf("\a%s\n", ErrMsg[ErrCode]);
		printf("Press a key to quit\n");
		getch();
	}

	// stop the acquisition
	if (ErrCode != ERR_DGZ_OPEN) for (board = 0; board < ConfigVar.Nhandle; board++) CAEN_DGTZ_SWStopAcquisition(handle[board]);
	// close the output files (if allocated)
	for (board = 0; board < ConfigVar.Nhandle; board++) {
		if (RawFile[board] != NULL) fclose(RawFile[board]);
	}
	// close the plotter
	if (PlotVar.plotpipe != NULL) ClosePlotter(&PlotVar.plotpipe);
	// free buffers (if allocated)
	if (buffer != NULL) CAEN_DGTZ_FreeReadoutBuffer(&buffer);
	// free events and waveforms (if allocated)
	for (board = 0; board < ConfigVar.Nhandle; board++) {
		if (Event[board] != NULL) {
			for (ev = 0; ev < MaxBLTn; ev++) {
				if ((&Event[board][ev]) != NULL) {
					for (chan = 0; chan < MAX_V1730_CHANNEL_SIZE; chan++) {
						if (Event[board][ev].Channel[chan] != NULL) {
							if (Event[board][ev].Channel[chan]->Waveforms != NULL) CAEN_DGTZ_FreeZLEWaveforms(handle[MaxRLHIndex], Event[board][ev].Channel[chan]->Waveforms);
						}
					}
				}
			}
			CAEN_DGTZ_FreeZLEEvents(handle[MaxBLTnHIndex], (void**)&Event[board]);
		}
	}
	if (ErrCode != ERR_DGZ_OPEN) for (board = 0; board < ConfigVar.Nhandle; board++) CAEN_DGTZ_CloseDigitizer(handle[board]);
    return 0;
}