#include "ZLEDemoFunc.h"
#include <inttypes.h>
#include <sys/types.h>
#include <sys/stat.h>

#define DEBUG 0

extern char path[128];

static long get_time()
{
	long time_ms;
#ifdef WIN32
	struct _timeb timebuffer;
	_ftime(&timebuffer);
	time_ms = (long)timebuffer.time * 1000 + (long)timebuffer.millitm;
#else
	struct timeval t1;
	struct timezone tz;
	gettimeofday(&t1, &tz);
	time_ms = (t1.tv_sec) * 1000 + t1.tv_usec / 1000;
#endif
	return time_ms;
}

ERROR_CODES OpenConfigFile(FILE **f_ini, char *ConfigFileName) {
	ERROR_CODES return_code = ERR_NONE;
	printf("Opening Configuration File %s\n", ConfigFileName);
	if ((*f_ini = fopen(ConfigFileName, "r")) == NULL) return_code = ERR_CONF_FILE_NOT_FOUND;
	return return_code;
}

ERROR_CODES ParseConfigFile(FILE *f_ini, ZLEConfig_t *ConfigVar) {
	ERROR_CODES return_code = ERR_NONE;
	char *str, str1[1000];
	char strline[1000];
	uint32_t addr, data;
	int line = 0;
	int i, j, ch = -1, board = -1, val, Off = 0;
	int DefConfig = 0;
	/* Default settings */
	memset(ConfigVar, 0, sizeof(*ConfigVar));
	ConfigVar->Nhandle = 0;
	#ifdef WIN32
		sprintf(ConfigVar->OutFilePath, "%s%s", path, OUTFILE_PATH);
	#else
		sprintf(ConfigVar->OutFilePath, "%s%s", getenv("HOME"), OUTFILE_PATH);
	#endif
	sprintf(ConfigVar->OutFileName, "%s", OUTFILE_NAME);
	sprintf(ConfigVar->GnuPlotPath, "%s%s", path, PLOTTER_PATH);
	/* read config file and assign parameters */
	while (fgets(strline, 1000, f_ini) != NULL) { // get a line
		line++;
		if (!strline || strline[0] == '#' || strline[0] == ' ' || strline[0] == '\t' || strline[0] == '\n' || strline[0] == '\r') continue;
		str = strtok(strline, " \r\t\n");
		if (str[0] == '[') {
			fscanf(f_ini, "%d", &val);
			if (strstr(str, "COMMON")) { ch = -1; board = -1;}
			else if (strstr(str, "BOARD")) { ch = -1; board = (int)strtol(strtok(NULL, " \r\t\n"), NULL, 10);}
			else if (strstr(str, "CHANNEL")) {
				ch = (int)strtol(strtok(NULL, " \r\t\n"), NULL, 10);
				if (ch < 0 || ch >= MAX_CH) printf("%s: Invalid channel number\n", str);
			}
			continue;
		}

		// OPEN: malloc memory for the board config variable, init it to default and read the details of physical path to the digitizer
		if (!strcmp(strcpy(str1, str), "OPEN")) {
			// malloc board config variable
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle] = (ZLEBoardConfig_t*)malloc(sizeof(ZLEBoardConfig_t));
			// initialize parameters
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ExtTriggerMode = CAEN_DGTZ_TRGMODE_ACQ_ONLY;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->StartMode = CAEN_DGTZ_SW_CONTROLLED;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->FPIOtype = CAEN_DGTZ_IOLevel_TTL;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->EnableMask = 0x0;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->GainFactor = 0;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->RecordLength = 256;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->TrigCoupleMask = 0xff;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->GWn = 0;
			for (j = 0; j < MAX_CH; j++) {
				if (!(j / 2)) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->TrigCoupleLogic[j / 2] = 3;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->PulsePolarity[j] = 1;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->preTrgg[j] = 12;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->NSampBck[j] = 4;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->NSampAhe[j] = 4;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ZleTrigThr[j] = 1000;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ZleThr[j] = 1000;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BLineMode[j] = 0;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BLineDefValue[j] = 0x2000;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BLineNoise[j] = 4;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->DCoffset[j] = 0;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->NoThreshold[j] = 0;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->TP_Enable[j] = 0;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->TP_Polarity[j] = 1;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->TP_Rate[j] = 0;
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->TP_Scale[j] = 0;
			}
			// end of initalization
			if ((str = strtok(NULL, " \r\t\n")) == NULL) {printf("No 1st argument for %s. The command will be ignored\n", str1); continue;}
			if (strcmp(str, "USB") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_DGTZ_USB;
			else if (strcmp(str, "PCI") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_DGTZ_OpticalLink;
			else if (strcmp(str, "USB_A4818") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_DGTZ_USB_A4818;
			else if (strcmp(str, "USB_V4718") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_DGTZ_USB_V4718;
			else if (strcmp(str, "ETH_V4718") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_DGTZ_ETH_V4718;
			else {printf("%s %s: Invalid connection type\n", str, str1);return ERR_INVALID_BOARD_TYPE;}

			switch (ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType) {
			case CAEN_DGTZ_USB:
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 2nd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkNum = (int)strtol(str, NULL, 10);
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ConetNode = 0;
				if ((str = strtok(NULL, " \r\t\n")) == NULL) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = 0; // Optional BA
				else ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = (uint32_t)strtoul(str, NULL, 0);
				break;
			case CAEN_DGTZ_USB_V4718:
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 2nd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkNum = (int)strtol(str, NULL, 10); // PID
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ConetNode = 0;
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 3rd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = (uint32_t)strtoul(str, NULL, 0);
				break;
			case CAEN_DGTZ_OpticalLink:
			case CAEN_DGTZ_USB_A4818:
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 2nd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkNum = (int)strtol(str, NULL, 10); // PID
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 3rd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ConetNode = (int)strtol(str, NULL, 10);
				if ((str = strtok(NULL, " \r\t\n")) == NULL) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = 0; // Optional BA
				else ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = (uint32_t)strtoul(str, NULL, 0);
				break;
			case CAEN_DGTZ_ETH_V4718:
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 2nd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->HostName[0] = '\0';
				strncat(ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->HostName, str, sizeof(ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->HostName) - 1);
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ConetNode = 0;
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 3rd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = (uint32_t)strtoul(str, NULL, 0);
				break;
			default:
				break;
			}
			ConfigVar->Nhandle++;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1,str);
			continue;
		}

		// Generic VME Write (address offset + data, both exadecimal)
		if (!strcmp(strcpy(str1, str), "WRITE_REGISTER")) {
			addr = (int)strtol(strtok(NULL, " \r\t\n"), NULL, 0);
			data = (int)strtol(strtok(NULL, " \r\t\n"), NULL, 0);
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (ConfigVar->BoardConfigVar[i]->GWn < MAX_GW) {
					if (i == board || board == -1) {
						ConfigVar->BoardConfigVar[i]->GWaddr[ConfigVar->BoardConfigVar[i]->GWn] = addr;
						ConfigVar->BoardConfigVar[i]->GWdata[ConfigVar->BoardConfigVar[i]->GWn] = data;
						ConfigVar->BoardConfigVar[i]->GWn++;
					}
				}
				else {
					printf("Maximum number of register locations reached\n");
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1,str);
			continue;
		}

		// Enable periodic plot (YES/NO)
		if (!strcmp(strcpy(str1, str), "PERIODIC_PLOT")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->PlotEnable = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->PlotEnable = 0;
			else printf("%s: invalid option\n", str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable sync procedure (YES/NO)
		if (!strcmp(strcpy(str1, str), "SYNC_ENABLE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->SyncEnable = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->SyncEnable = 0;
			else printf("%s: invalid option\n", str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Choose the type of plot (BOARDPLOT/SYNCPLOT)
		if (!strcmp(strcpy(str1, str), "PLOT_TYPE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "BOARDPLOT") == 0) ConfigVar->PlotType = 0;
			else if (strcmp(str, "SYNCPLOT") == 0) ConfigVar->PlotType = 1;
			else printf("%s: invalid option\n", str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable raw output (YES/NO)
		if (!strcmp(strcpy(str1, str), "OUTFILE_RAW")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->OFRawEnable = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->OFRawEnable = 0;
			else printf("%s: invalid option\n", str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable raw output (YES/NO)
		if (!strcmp(strcpy(str1, str), "OUTFILE_WAVE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->OFWaveEnable = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->OFWaveEnable = 0;
			else printf("%s: invalid option\n", str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Output file path
		if (!strcmp(strcpy(str1, str), "OUTFILE_PATH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			#ifdef WIN32
				sprintf(ConfigVar->OutFilePath, "%s%s", path, str);
			#else
				sprintf(ConfigVar->OutFilePath, "%s%s", getenv("HOME"), str);
			#endif
			strcpy(ConfigVar->OutFilePath, str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Output file name
		if (!strcmp(strcpy(str1, str), "OUTFILE_NAME")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			strcpy(ConfigVar->OutFileName, str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Output file max size
		if (!strcmp(strcpy(str1, str), "OUTFILE_MAXSIZE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			ConfigVar->MaxFileSize = (int)strtol(str, NULL, 10);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}
		// continuous software trigger (YES/NO)
		if (!strcmp(strcpy(str1, str), "CONT_SWTRIGGER")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->ContTrigger = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->ContTrigger = 0;
			else printf("%s: invalid option\n", str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "GNUPLOT_PATH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			strcpy(ConfigVar->GnuPlotPath, str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}


		if (!strcmp(strcpy(str1, str), "STAT_REFRESH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			ConfigVar->PlotRefreshTime = (int)strtol(str, NULL, 10);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Front Panel LEMO I/O level (NIM, TTL)
		if (!strcmp(strcpy(str1, str), "FPIO_LEVEL")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "TTL") == 0) ConfigVar->BoardConfigVar[i]->FPIOtype = 1;
					else if (strcmp(str, "NIM") == 0) ConfigVar->BoardConfigVar[i]->FPIOtype = 0;
					else { printf("%s: invalid option\n", str); break; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// External Trigger (DISABLED, ACQUISITION_ONLY, ACQUISITION_AND_TRGOUT)
		if (!strcmp(strcpy(str1, str), "EXTERNAL_TRIGGER")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "DISABLED") == 0)
						ConfigVar->BoardConfigVar[i]->ExtTriggerMode = CAEN_DGTZ_TRGMODE_DISABLED;
					else if (strcmp(str, "ACQUISITION_ONLY") == 0)
						ConfigVar->BoardConfigVar[i]->ExtTriggerMode = CAEN_DGTZ_TRGMODE_ACQ_ONLY;
					else if (strcmp(str, "ACQUISITION_AND_TRGOUT") == 0)
						ConfigVar->BoardConfigVar[i]->ExtTriggerMode = CAEN_DGTZ_TRGMODE_ACQ_AND_EXTOUT;
					else { printf("%s: Invalid Parameter\n", str); break; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Software Trigger (DISABLED, ACQUISITION_ONLY, TRGOUT_ONLY, ACQUISITION_AND_TRGOUT)
		if (!strcmp(strcpy(str1, str), "SW_TRIGGER")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "DISABLED") == 0)
						ConfigVar->BoardConfigVar[i]->SWTriggerMode = CAEN_DGTZ_TRGMODE_DISABLED;
					else if (strcmp(str, "ACQUISITION_ONLY") == 0)
						ConfigVar->BoardConfigVar[i]->SWTriggerMode = CAEN_DGTZ_TRGMODE_ACQ_ONLY;
					else if (strcmp(str, "TRGOUT_ONLY") == 0)
						ConfigVar->BoardConfigVar[i]->SWTriggerMode = CAEN_DGTZ_TRGMODE_EXTOUT_ONLY;
					else if (strcmp(str, "ACQUISITION_AND_TRGOUT") == 0)
						ConfigVar->BoardConfigVar[i]->SWTriggerMode = CAEN_DGTZ_TRGMODE_ACQ_AND_EXTOUT;
					else { printf("%s: Invalid Parameter\n", str); break; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}


		if (!strcmp(strcpy(str1, str), "START_ACQ")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "SW") == 0) ConfigVar->BoardConfigVar[i]->StartMode = CAEN_DGTZ_SW_CONTROLLED;
					else if (strcmp(str, "S_IN") == 0) ConfigVar->BoardConfigVar[i]->StartMode = CAEN_DGTZ_S_IN_CONTROLLED;
					else if (strcmp(str, "FIRST_TRG") == 0) ConfigVar->BoardConfigVar[i]->StartMode = CAEN_DGTZ_FIRST_TRG_CONTROLLED;
					else if (strcmp(str, "LVDS") == 0) ConfigVar->BoardConfigVar[i]->StartMode = CAEN_DGTZ_LVDS_CONTROLLED;
					else { printf("%s: Invalid Parameter\n", str); break; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// BASELINE DEFAULT ENABLED (YES/NO)
		if (!strcmp(strcpy(str1, str), "BLINE_DEFMODE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) {
						if (i == ch || ch == -1) {
							if (strcmp(str, "YES") == 0) ConfigVar->BoardConfigVar[j]->BLineMode[i] = 1;
							else if (strcmp(str, "NO") == 0) ConfigVar->BoardConfigVar[j]->BLineMode[i] = 0;
							else { printf("%s: invalid option\n", str); break; }
						}
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// BASELINE DEFAULT VALUE
		if (!strcmp(strcpy(str1, str), "BLINE_DEFVALUE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) if (i == ch || ch == -1) ConfigVar->BoardConfigVar[j]->BLineDefValue[i] = (int)strtol(str, NULL, 10);
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "BLINE_NOISE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) if (i == ch || ch == -1) ConfigVar->BoardConfigVar[j]->BLineNoise[i] = val;
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// PULSE POLARITY (POSITIVE/NEGATIVE)
		if (!strcmp(strcpy(str1, str), "PULSE_POLARITY")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) {
						if (i == ch || ch == -1) {
							if (strcmp(str, "POSITIVE") == 0) ConfigVar->BoardConfigVar[j]->PulsePolarity[i] = 1;
							else if (strcmp(str, "NEGATIVE") == 0) ConfigVar->BoardConfigVar[j]->PulsePolarity[i] = 0;
							else printf("%s: invalid option\n", str); continue;
						}
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// NO THRESHOLD OPTION (YES/NO)
		if (!strcmp(strcpy(str1, str), "NO_THRESHOLD")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) {
						if (i == ch || ch == -1) {
							if (strcmp(str, "YES") == 0) ConfigVar->BoardConfigVar[j]->NoThreshold[i] = 1;
							else if (strcmp(str, "NO") == 0) ConfigVar->BoardConfigVar[j]->NoThreshold[i] = 0;
							else printf("%s: invalid option\n", str); continue;
						}
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// TEST PULSE ENABLE (YES/NO)
		if (!strcmp(strcpy(str1, str), "TEST_PULSE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) {
						if (i == ch || ch == -1) {
							if (strcmp(str, "YES") == 0) ConfigVar->BoardConfigVar[j]->TP_Enable[i] = 1;
							else if (strcmp(str, "NO") == 0) ConfigVar->BoardConfigVar[j]->TP_Enable[i] = 0;
							else printf("%s: invalid option\n", str); continue;
						}
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "TP_POLARITY")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) {
						if (i == ch || ch == -1) {
							if (strcmp(str, "POSITIVE") == 0) ConfigVar->BoardConfigVar[j]->TP_Polarity[i] = 1;
							else if (strcmp(str, "NEGATIVE") == 0) ConfigVar->BoardConfigVar[j]->TP_Polarity[i] = 0;
							else printf("%s: invalid option\n", str); continue;
						}
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Test Pulse rate
		if (!strcmp(strcpy(str1, str), "TP_RATE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) {
						if (ch == -1 || i == ch) {
							ConfigVar->BoardConfigVar[j]->TP_Rate[i] = val;
						}
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Test Pulse scale
		if (!strcmp(strcpy(str1, str), "TP_SCALE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) {
						if (ch == -1 || i == ch) {
							ConfigVar->BoardConfigVar[j]->TP_Scale[i] = val;
						}
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "RECORD_LENGTH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) ConfigVar->BoardConfigVar[j]->RecordLength = val;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "TRIGCOUPLE_MASK")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 16);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) ConfigVar->BoardConfigVar[j]->TrigCoupleMask = val;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "TRIGOUTCOUPLE_MASK")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 16);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) ConfigVar->BoardConfigVar[j]->TrigOutCoupleMask = val;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "TRIGLOGIC_COUPLE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No first argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 0);
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No second argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					if (strcmp(str, "AND") == 0) ConfigVar->BoardConfigVar[j]->TrigCoupleLogic[val] = 0;
					else if (strcmp(str, "FIRSTONLY") == 0) ConfigVar->BoardConfigVar[j]->TrigCoupleLogic[val] = 1;
					else if (strcmp(str, "SECONDONLY") == 0) ConfigVar->BoardConfigVar[j]->TrigCoupleLogic[val] = 2;
					else if (strcmp(str, "OR") == 0) ConfigVar->BoardConfigVar[j]->TrigCoupleLogic[val] = 3;
					else printf("invalid parameter\n");
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "GAIN_FACTOR")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) ConfigVar->BoardConfigVar[j]->GainFactor=val;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// ZLE_thr 
		if (!strcmp(strcpy(str1, str), "ZLE_THRESHOLD")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) if (ch == -1 || i == ch) ConfigVar->BoardConfigVar[j]->ZleThr[i] = val;
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Trigger threshold
		if (!strcmp(strcpy(str1, str), "TRG_THRESHOLD")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) if (ch == -1 || i == ch) ConfigVar->BoardConfigVar[j]->ZleTrigThr[i] = val;
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Nsamp back
		if (!strcmp(strcpy(str1, str), "N_LBK")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) if (ch == -1 || i == ch) ConfigVar->BoardConfigVar[j]->NSampBck[i] = val;
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}


		//Nsamp ahead
		if (!strcmp(strcpy(str1, str), "N_LFW")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) if (ch == -1 || i == ch) ConfigVar->BoardConfigVar[j]->NSampAhe[i] = val;
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Pretrigger
		if (!strcmp(strcpy(str1, str), "PRE_TRIGGER")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) if (ch == -1 || i == ch) ConfigVar->BoardConfigVar[j]->preTrgg[i] = val;
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "DC_OFFSET")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			float dc = strtof(str, NULL);
			val = (int)((dc + 50) * 65535 / 100);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < MAX_CH; i++) if (ch == -1 || i == ch) ConfigVar->BoardConfigVar[j]->DCoffset[i] = val;
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "ENABLE_INPUT")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					if (strcmp(str, "YES") == 0) {
						if (ch == -1) ConfigVar->BoardConfigVar[j]->EnableMask = 0xFFFF;
						else ConfigVar->BoardConfigVar[j]->EnableMask = ConfigVar->BoardConfigVar[j]->EnableMask | (1 << ch);
					} else if (strcmp(str, "NO") == 0) {
						if (ch == -1) ConfigVar->BoardConfigVar[j]->EnableMask = 0x0000;
						else ConfigVar->BoardConfigVar[j]->EnableMask = ConfigVar->BoardConfigVar[j]->EnableMask & ~(1 << ch);
					}
					else {
						printf("%s: invalid option\n", str);
						break;
					}
				}
			}		
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "ENABLE_GRAPH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					if (strcmp(str, "YES") == 0) {
						if (ch == -1) ConfigVar->EnableTrack = 0xffff;
						else ConfigVar->EnableTrack = ConfigVar->EnableTrack | (1 << ch);
					}
					else if (strcmp(str, "NO") == 0) {
						if (ch == -1) ConfigVar->EnableTrack = 0x0000;
						else ConfigVar->EnableTrack = ConfigVar->EnableTrack & ~(1 << ch);
					}
					else {
						printf("%s: invalid option\n", str);
						break;
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		printf("%s: invalid setting at line %d\n", str,line);
		return_code = ERR_PARSE_CONFIG;
		break;
	}
	return return_code;
}

ERROR_CODES OpenRawFile(FILE **outfile, int BoardIndex, int FileIndex, char *path, char *fname) {
	ERROR_CODES return_code = ERR_NONE;
	char filename[400];
	struct stat info;
	if (stat(path, &info) != 0) {
		printf("path %s cannot be accessed. Please verify that the selected directory exists and is writable\n", path); return ERR_OUTFILE_OPEN;
	}
	if (*outfile != NULL) fclose(*outfile);
	sprintf(filename, "%s%s_raw_b%d_seg%d.bin", path, fname, BoardIndex, FileIndex);
	if ((*outfile = fopen(filename, "w")) == NULL) {
		printf("output file %s could not be created.\n", filename);
		return_code = ERR_OUTFILE_OPEN;
	}
	return return_code;
}

ERROR_CODES OpenWaveFile(FILE ***outfile, int BoardIndex, ZLEBoardConfig_t *BoardConfigVar, char *path, char *fname) {
	ERROR_CODES return_code = ERR_NONE;
	char filename[400];
	int channel;
	struct stat info;
	if (stat(path, &info) != 0) {
		printf("path %s cannot be accessed. Please verify that the selected directory exists and is writable\n", path); return ERR_OUTFILE_OPEN;
	}
	for (channel = 0; channel < BoardConfigVar->BoardInfo.Channels; channel++) {
		if ((BoardConfigVar->EnableMask >> channel) & 0x1) {
			sprintf(filename, "%s%s_wave_b%d_ch%d.txt", path, fname, BoardIndex, channel);
			if ((*(*(outfile)+channel)) != NULL) {
				fclose((*(*(outfile)+channel)));
			}
			if ((*(*(outfile)+channel) = fopen(filename, "w")) == NULL) {
				printf("output file %s could not be created.\n", filename);
				return ERR_OUTFILE_OPEN;
			}
		}
		else *(*(outfile)+channel) = NULL;
	}
	return return_code;
}

void WaveWrite(FILE **WaveFile, CAEN_DGTZ_730_ZLE_Event_t *Event, ZLEBoardConfig_t *BoardConfigVar) {
	int channel;
	int sample;
	int sampnum=0;
	for (channel = 0; channel < BoardConfigVar->BoardInfo.Channels; channel++) {
		sampnum = 0;
		if ((BoardConfigVar->EnableMask >> channel) & 0x1) {
			fprintf(*(WaveFile + channel), "baseline value: %*d\n\n", 7,Event[0].Channel[channel]->baseline);
			for (sample = 0; sample < 4*BoardConfigVar->RecordLength; sample++) {
				fprintf(*(WaveFile + channel), "%*d\t", 7, sample);
				if (sampnum < Event[0].Channel[channel]->Waveforms->TraceNumber && sample == Event[0].Channel[channel]->Waveforms->TraceIndex[sampnum]) {
					fprintf(*(WaveFile + channel), "%*u\n", 12, Event[0].Channel[channel]->Waveforms->Trace[sampnum]);
					sampnum++;
				} else fprintf(*(WaveFile + channel), "%*u\n", 12, 0);
			}
		}
	}
}

int XX2530_ZLE_SetPreSamples(int handle,uint32_t samples, int channel) {
	int ret;
	if (samples < 256) {
		ret = CAEN_DGTZ_WriteRegister(handle, 0x1054 | (channel << 8),samples);
		if (DEBUG) printf("pre-signal sample number of channel %d: %d\n", channel,samples);
	} else {
		printf("invalid value for pre-signal sample number of channel %d\n",channel);
	}
	return ret;
}

int XX2530_ZLE_SetPostSamples(int handle, uint32_t samples, int channel) {
	int ret;
	if (samples < 256) {
		ret = CAEN_DGTZ_WriteRegister(handle, 0x1058 | (channel << 8),samples);
		if (DEBUG) printf("post-signal sample number of channel %d: %d\n", channel,samples);
	} else {
		printf("invalid value for post-signal sample number of channel %d\n",channel);
	}
	return ret;
}

int XX2530_ZLE_SetDataThreshold(int handle, uint16_t threshold, int channel) {
	int ret;
	if (threshold < 16384) {
		ret = CAEN_DGTZ_WriteRegister(handle, 0x105C | (channel << 8), (uint32_t)(threshold & 0x3FFF));
		if (DEBUG) printf("data threshold of channel %d: %d\n", channel,threshold);
	} else {
		printf("invalid value for data threshold of channel %d\n", channel);
	}
	return ret;
}

int XX2530_ZLE_SetTriggerThreshold(int handle, uint16_t threshold, int channel) {
	int ret;
	if (threshold < 16384) {
		ret = CAEN_DGTZ_WriteRegister(handle, 0x1060 | (channel << 8), (uint32_t)(threshold & 0x3FFF));
		if (DEBUG) printf("trigger threshold of channel %d: %d\n",channel, threshold);
	} else {
		printf("invalid value for trigger threshold of channel %d\n", channel);
	}
	return ret;
}

int XX2530_ZLE_SetPreTrigger(int handle, uint32_t samples, int channel) {
	int ret;
	if (samples < 256) {
		ret = CAEN_DGTZ_WriteRegister(handle, 0x1038 | (channel << 8), samples);
		if (DEBUG) printf("pretrigger sample number of channel %d: %d\n", channel,samples);
	} else {
		printf("invalid value for pretrigger sample number of channel %d\n", channel);
	}
	return ret;
}

int XX2530_ZLE_SetTrigCoupleLogic(int handle, int couple, uint32_t logic) {
	int ret;
	uint32_t regvalue;
	if (logic < 4) {
		ret = CAEN_DGTZ_ReadRegister(handle, 0x1068, &regvalue);
		regvalue = (regvalue & (uint32_t)(~(0x00000003 << (2 * couple)))) | ((uint32_t)(logic << (2 * couple))); // replace only the two bits affecting the selected couple's logic.
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x8068,regvalue);
	} else {
		printf("invalid value for trigger logic of couple %d\n", couple);
	}
	return ret;
}

int XX2530_ZLE_SetBLineMode(int handle,uint32_t mode,int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1034 | (channel << 8), &regvalue);
	if (mode) ret |= CAEN_DGTZ_WriteRegister(handle, 0x1034 | (channel << 8), regvalue | (uint32_t)(1 << 24));
	else ret |= CAEN_DGTZ_WriteRegister(handle, 0x1034 | (channel << 8), (regvalue & (~(uint32_t)(1 << 24))));
	return ret;
}

int XX2530_ZLE_SetBLineDefValue(int handle, uint32_t bl, int channel) {
	int ret;
	uint32_t regvalue;
	if (bl < 16384) {
		ret = CAEN_DGTZ_ReadRegister(handle, 0x1034 | (channel << 8), &regvalue);
		regvalue = (regvalue & (uint32_t)(~(0x00003fff))) | (uint32_t)(bl & 0x3fff); // replace only the two bits affecting the selected couple's logic.
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x1034 | (channel << 8), regvalue);
	} else {
		printf("invalid value for default baseline of channel %d\n", channel);
	}
	return ret;
}

int XX2530_ZLE_SetBLineNoise(int handle, uint32_t noise, int channel) {
	int ret;
	uint32_t regvalue;
	if (noise < 256) {
		ret = CAEN_DGTZ_ReadRegister(handle, 0x1034 | (channel << 8), &regvalue);
		regvalue = (regvalue & (uint32_t)(~(0x00ff0000))) | (uint32_t)(noise << 16); // replace only the two bits affecting the selected couple's logic.
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x1034 | (channel << 8), regvalue);
	} else {
		printf("invalid value for baseline noise of channel %d\n", channel);
	}
	return ret;
}

int XX2530_ZLE_NoThresholdEnable(int handle, int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1064 | (channel << 8), &regvalue);
	ret = CAEN_DGTZ_WriteRegister(handle, 0x1064 | (channel << 8), regvalue | (uint32_t)(0x80));
	return ret;
}

int XX2530_ZLE_NoThresholdDisable(int handle, int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1064 | (channel << 8), &regvalue);
	ret = CAEN_DGTZ_WriteRegister(handle, 0x1064 | (channel << 8), regvalue & 0xffffff7f);
	return ret;
}

int XX2530_ZLE_TestPulseEnable(int handle, int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1064 | (channel << 8), &regvalue);
	ret = CAEN_DGTZ_WriteRegister(handle, 0x1064 | (channel << 8), regvalue | (uint32_t)(0x1));
	return ret;
}

int XX2530_ZLE_TestPulseDisable(int handle, int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1064 | (channel << 8), &regvalue);
	ret = CAEN_DGTZ_WriteRegister(handle, 0x1064 | (channel << 8), regvalue & 0xfffffffe);
	return ret;
}

int XX2530_ZLE_SetTestPulseRate(int handle, uint32_t rate, int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1064 | (channel << 8), &regvalue);
	if (rate < 8) {
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x1064 | (channel << 8), (regvalue & ~(0x0000000e)) | (uint32_t)(rate << 1));
	}
	else {
		printf("invalid value for test pulse rate channel %d\n", channel);
	}
	return ret;
}

int XX2530_ZLE_SetTestPulseScale(int handle, uint32_t scale, int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1064 | (channel << 8), &regvalue);
	if (scale<4) {
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x1064 | (channel << 8), (regvalue & ~(0x00000030)) | (uint32_t)(scale << 4));
	} else {
		printf("invalid value for test pulse scale channel %d\n", channel);
	}
	return ret;
}

int XX2530_ZLE_SetTestPulsePolarity(int handle, uint32_t polarity, int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1064 | (channel << 8), &regvalue);
	(polarity) ? (regvalue = regvalue | 0x00000040) : (regvalue = regvalue & ~(0x00000040));
	ret = CAEN_DGTZ_WriteRegister(handle, 0x1064 | (channel << 8),regvalue);
	return ret;
}

int XX2530_ZLE_SetPulsePolarity(int handle, uint32_t polarity, int channel) {
	int ret;
	uint32_t regvalue;
	ret = CAEN_DGTZ_ReadRegister(handle, 0x1064 | (channel << 8), &regvalue);
	(polarity) ? (regvalue = regvalue | 0x00000100) : (regvalue = regvalue & ~(0x00000100));
	ret = CAEN_DGTZ_WriteRegister(handle, 0x1064 | (channel << 8), regvalue);
	return ret;
}

int XX2530_ZLE_SetTriggerMask(int handle, uint32_t mask) {
	int ret;
	uint32_t regvalue;
	if (mask < 0x100) {
		ret = CAEN_DGTZ_ReadRegister(handle, 0x810C, &regvalue);
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x810C, (regvalue & ~(0x000000ff)) | (uint32_t)(mask));
	}
	else {
		printf("invalid value for channel couple trigger mask\n");
	}
	return ret;
}

int XX2530_ZLE_SetTriggerOutMask(int handle, uint32_t mask) {
	int ret;
	uint32_t regvalue;
	if (mask < 0x100) {
		ret = CAEN_DGTZ_ReadRegister(handle, 0x8110, &regvalue);
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x8110,(regvalue & ~(0x000000ff)) | (uint32_t)(mask));
	} else {
		printf("invalid value for channel couple trigger mask\n");
	}
	return ret;
}

int OpenDigitizer(int *handle, ZLEConfig_t ConfigVar)
{
	int Nboard,ret=0;
	for (Nboard = 0; Nboard < ConfigVar.Nhandle;Nboard++) {
		printf("Board %d: ", Nboard);
		const void* arg;
		uint32_t linkNum;
		if (ConfigVar.BoardConfigVar[Nboard]->LinkType == CAEN_DGTZ_ETH_V4718) {
			printf("hostname: %s;", ConfigVar.BoardConfigVar[Nboard]->HostName);
			arg = ConfigVar.BoardConfigVar[Nboard]->HostName;
		} else {
			printf("link ID: %d;", ConfigVar.BoardConfigVar[Nboard]->LinkNum);
			linkNum = ConfigVar.BoardConfigVar[Nboard]->LinkNum;
			arg = &linkNum;
		}
		printf(" Node ID: %d;", ConfigVar.BoardConfigVar[Nboard]->ConetNode);
		printf(" Base Address: %x\n", ConfigVar.BoardConfigVar[Nboard]->BaseAddress);
		if (ret |= CAEN_DGTZ_OpenDigitizer2(ConfigVar.BoardConfigVar[Nboard]->LinkType, arg, ConfigVar.BoardConfigVar[Nboard]->ConetNode, ConfigVar.BoardConfigVar[Nboard]->BaseAddress,handle+Nboard)) return ret;
	}
	return ret;
}

int ProgramDigitizers(int *handle, ZLEConfig_t *ConfigVar)
{
	int board;
	int i, ret = 0;
	uint32_t regval;
	for (board = 0; board < ConfigVar->Nhandle; board++) {
		// set board sampling period (for plotting purposes)
		if(strstr(ConfigVar->BoardConfigVar[board]->BoardInfo.ModelName,"25")!=NULL) ConfigVar->BoardConfigVar[board]->tSampl = 4;
		else if(strstr(ConfigVar->BoardConfigVar[board]->BoardInfo.ModelName,"30") != NULL) ConfigVar->BoardConfigVar[board]->tSampl = 2;
		else return -1;
		// mask possibly-configured channels that are not present in this digitizer model
		ConfigVar->BoardConfigVar[board]->EnableMask = ConfigVar->BoardConfigVar[board]->EnableMask & (uint16_t)((1 << ConfigVar->BoardConfigVar[board]->BoardInfo.Channels) - 1);
		// reset
		ret |= CAEN_DGTZ_Reset(handle[board]);
		ret |= CAEN_DGTZ_SetIOLevel(handle[board], ConfigVar->BoardConfigVar[board]->FPIOtype);
		//ret |= CAEN_DGTZ_SetAcquisitionMode(handle[board], ConfigVar->BoardConfigVar[board]->StartMode);
		ret |= CAEN_DGTZ_SetSWTriggerMode(handle[board], ConfigVar->BoardConfigVar[board]->SWTriggerMode);
		ret |= CAEN_DGTZ_SetExtTriggerInputMode(handle[board], ConfigVar->BoardConfigVar[board]->ExtTriggerMode);
		ret |= XX2530_ZLE_SetTriggerMask(handle[board], ConfigVar->BoardConfigVar[board]->TrigCoupleMask);
		ret |= XX2530_ZLE_SetTriggerOutMask(handle[board], ConfigVar->BoardConfigVar[board]->TrigOutCoupleMask);
		// Active channels
		ret |= CAEN_DGTZ_SetChannelEnableMask(handle[board], (uint32_t)ConfigVar->BoardConfigVar[board]->EnableMask);
		// gain control
		ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x8028, ConfigVar->BoardConfigVar[board]->GainFactor);
		// record length
		ret |= CAEN_DGTZ_SetRecordLength(handle[board], ConfigVar->BoardConfigVar[board]->RecordLength);
		// max BLT events
		ret |= CAEN_DGTZ_SetMaxNumEventsBLT(handle[board], 1023);
		
		if (ConfigVar->SyncEnable) {
			// sets whether the LVDS quartets are input or output (bits [5:2]): 1st quartet is input, other outputs here
			// sets the LVDS "new" mode (bit 8)
			// TRG OUT is used to propagate signals (bits [17:16])
			// signal propagated through the trgout (bits [19:18]) is the busy signal
			// enable extended time-stamp (bits [22:21] = "10")
			// the other two quartets (not used) are also set to output
			ret |= CAEN_DGTZ_ReadRegister(handle[board], 0x811C, &regval);
			ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x811C, regval | 0x4d0138);
			// acquisition mode is sw-controlled for the first board, LVDS-controlled for the others
			ret |= CAEN_DGTZ_ReadRegister(handle[board], 0x8100, &regval);
			ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x8100, regval | (board == 0 ? 0x00000100 : 0x00000107));
			// register 0x816C: reduces the threshold at which the BUSY is raised at 2^buffer organization-10 events
			ret |= CAEN_DGTZ_ReadRegister(handle[board], 0x800C, &regval);
			ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x816C, (uint32_t)(pow(2., regval) - 10));
			// register 0x8170: timestamp offset
			ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x8170, 3*(ConfigVar->Nhandle - board - 1) + (board == 0 ? -1 : 0));
			// register 0x81A0: select the lowest two quartet as "nBUSY/nVETO" type. BEWARE: set ALL the quartet bits to 2
			ret |= CAEN_DGTZ_ReadRegister(handle[board], 0x81A0, &regval);
			ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x81A0, regval | 0x00002222);
		}	else {
			// enable extended timestamp (bits [22:21] = "10")
			ret |= CAEN_DGTZ_ReadRegister(handle[board], 0x811C, &regval);
			ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x811C, regval | 0x400000);
			// set acquisition mode
			ret |= CAEN_DGTZ_SetAcquisitionMode(handle[board], ConfigVar->BoardConfigVar[board]->StartMode);
			// register 0x8100: set bit 2 to 1 if not in sw-controlled mode
			ret |= CAEN_DGTZ_ReadRegister(handle[board], 0x8100, &regval);
			if (ConfigVar->BoardConfigVar[board]->StartMode == CAEN_DGTZ_SW_CONTROLLED) ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x00008100, regval | 0x000100);
			else ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x8100, regval | 0x00000104);	
		}
		
		// channel-specific settings	
		for (i = 0; i < ConfigVar->BoardConfigVar[board]->BoardInfo.Channels; i++) {
			//trigger generation mode from channel couples
			if ((i % 2) == 0) XX2530_ZLE_SetTrigCoupleLogic(handle[board], (int)(i / 2), ConfigVar->BoardConfigVar[board]->TrigCoupleLogic[(int)(i / 2)]);
			if (ConfigVar->BoardConfigVar[board]->EnableMask & (1 << i)) {
				// set DC offset
				ret |= CAEN_DGTZ_SetChannelDCOffset(handle[board], i, ConfigVar->BoardConfigVar[board]->DCoffset[i]);
				// pretrigger
				ret |= XX2530_ZLE_SetPreTrigger(handle[board], ConfigVar->BoardConfigVar[board]->preTrgg[i], i);
				//ZLE baseline register
				ret |= XX2530_ZLE_SetBLineMode(handle[board], ConfigVar->BoardConfigVar[board]->BLineMode[i], i);
				ret |= XX2530_ZLE_SetBLineDefValue(handle[board], ConfigVar->BoardConfigVar[board]->BLineDefValue[i], i);
				ret |= XX2530_ZLE_SetBLineNoise(handle[board], ConfigVar->BoardConfigVar[board]->BLineNoise[i], i);
				//NSampBack 
				ret |= XX2530_ZLE_SetPreSamples(handle[board], ConfigVar->BoardConfigVar[board]->NSampBck[i], i);
				//NSampAhead
				ret |= XX2530_ZLE_SetPostSamples(handle[board], ConfigVar->BoardConfigVar[board]->NSampAhe[i], i);
				//ZLE Threshold
				ret |= XX2530_ZLE_SetDataThreshold(handle[board], ConfigVar->BoardConfigVar[board]->ZleThr[i], i);
				//ZLE Trigger Threshold
				ret |= XX2530_ZLE_SetTriggerThreshold(handle[board], ConfigVar->BoardConfigVar[board]->ZleTrigThr[i], i);
				//ZLE signal logic register
				ret |= XX2530_ZLE_SetPulsePolarity(handle[board], ConfigVar->BoardConfigVar[board]->PulsePolarity[i], i);
				// collect the whole event, independently from threshold or trigger polarity
				ConfigVar->BoardConfigVar[board]->NoThreshold[i] ? ret |= XX2530_ZLE_NoThresholdEnable(handle[board], i) : XX2530_ZLE_NoThresholdDisable(handle[board], i);
				// pulse emulator
				ConfigVar->BoardConfigVar[board]->TP_Enable[i] ? ret |= XX2530_ZLE_TestPulseEnable(handle[board], i) : XX2530_ZLE_TestPulseDisable(handle[board], i);
				ret |= XX2530_ZLE_SetTestPulsePolarity(handle[board], ConfigVar->BoardConfigVar[board]->TP_Polarity[i], i);
				ret |= XX2530_ZLE_SetTestPulseRate(handle[board], ConfigVar->BoardConfigVar[board]->TP_Rate[i], i);
				ret |= XX2530_ZLE_SetTestPulseScale(handle[board], ConfigVar->BoardConfigVar[board]->TP_Scale[i], i);
			}
		}
		// Write register commands in the config file (possibily overwriting previous settings)
		for (i = 0; i < ConfigVar->BoardConfigVar[board]->GWn; i++) ret |= CAEN_DGTZ_WriteRegister(handle[board], ConfigVar->BoardConfigVar[board]->GWaddr[i], ConfigVar->BoardConfigVar[board]->GWdata[i]);
		// ADC calibration
		ret |= CAEN_DGTZ_WriteRegister(handle[board], 0x809c, 0x1);
		if (ret) printf("Warning: errors found during the programming of the digitizer.\nSome settings may not be executed\n");
	}
	return ret;
}

int PlotEvent(ZLEConfig_t  *ConfigVar, ZLEPlot_t *PlotVar, CAEN_DGTZ_730_ZLE_Event_t **Event, int mode)
{
	int i, t = 0, comma = 0, c, npts = 0, WaitTime = 0;
	int NumTraces = 0;
	int board;
	char fname[100];
	uint32_t s;
	uint32_t SampIndex[16];
	FILE *fplot;
	int *shift, *dtime;
	if((shift = (int*)calloc(ConfigVar->BoardPlotted, sizeof(int)))==NULL) return -1;
	if ((dtime = (int*)calloc(ConfigVar->BoardPlotted, sizeof(int))) == NULL) return -1;
	if (mode == 0) {
		for (i = 0; i < ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->BoardInfo.Channels; i++) {
			SampIndex[i] = 0;
			PlotVar->TraceEnable[i] = 0;
			if (ConfigVar->EnableTrack & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask & (1 << i)) {
				NumTraces += 1;
				sprintf(PlotVar->Title, "%s%d", "Waveforms Ch ", i);
				PlotVar->TraceEnable[i] = 1;
				sprintf(PlotVar->TraceName[i], "Ch %d %s", i, "Input Signal");
			}
		}
	} else { // plot first channel of each board
		for(board=0;board<ConfigVar->Nhandle;board++) {
			SampIndex[board] = 0;
			PlotVar->TraceEnable[board] = 0;
			if (ConfigVar->BoardConfigVar[board]->EnableMask & (1 << 0)) {
				NumTraces += 1;
				sprintf(PlotVar->Title,"Waveforms Ch %d",0);
				PlotVar->TraceEnable[board] = 1;
				sprintf(PlotVar->TraceName[board], "Board %d %s", board, "Channel 0");
			}
		}
	}

	if (NumTraces == 0) {
		printf("all traces disabled\n");
		return -1;
	}
	else {
		sprintf(fname, "%s%s", path, PLOT_DATA_FILE);
		if ((fplot = fopen(fname, "w")) == NULL) {printf("file not allocated\n");return -1;}
	}

	uint32_t max=Event[0][0].tcounter;
	for (board = 0; board < ConfigVar->Nhandle; board++) if (Event[board][0].tcounter > max) max = Event[board][0].tcounter;
	for (board = 0; board < ConfigVar->Nhandle; board++) {
		if((max - Event[board][0].tcounter)>1023) {
			printf("event trigger alignment for board %d was not possible\n", board);
			shift[board] = 0;
		} else shift[board] = max - Event[board][0].tcounter;
		dtime[board] = 0;// 2 * ((int)Event[0][shift[0]].timeStamp - (int)Event[board][shift[board]].timeStamp);
		printf("Trigger index    : %u\n", Event[board][shift[board]].tcounter);
		printf("Timestamp        : %llu ns\n", Event[board][shift[board]].timeStamp * 8);
	}

	if (mode == 0) {
		for (s = 0; s < 4 * ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->RecordLength; s++) {
			fprintf(fplot, "%d,\t", s);
			t = 0;
			for (i = 0; i < NumTraces; i++) { // loop on enabled tracks
				while (PlotVar->TraceEnable[t] == 0) t++;
				if ((s >= 0 && s <= 5) && (s != Event[ConfigVar->BoardPlotted][0].Channel[t]->Waveforms->TraceIndex[SampIndex[t]] || Event[ConfigVar->BoardPlotted][0].Channel[t]->Waveforms->TraceNumber==0)) {
					fprintf(fplot, "%u,\t", Event[ConfigVar->BoardPlotted][0].Channel[t]->baseline);
				} else if (s == Event[ConfigVar->BoardPlotted][0].Channel[t]->Waveforms->TraceIndex[SampIndex[t]] && SampIndex[t] < Event[ConfigVar->BoardPlotted][0].Channel[t]->Waveforms->TraceNumber) {
					fprintf(fplot, "%u,\t", Event[ConfigVar->BoardPlotted][0].Channel[t]->Waveforms->Trace[SampIndex[t]] * PlotVar->Gain[t] + PlotVar->Offset[t]);
					SampIndex[t]++;
				} else {
					fprintf(fplot, " ,\t");
				}
				t++;
			}
			fprintf(fplot, "\n");
		}
	} else {
		for (s = 0; s < 4 * ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->RecordLength; s++) {
			fprintf(fplot, "%d,\t", s);
			t = 0;
			for (i = 0; i < NumTraces; i++) { // loop on enabled tracks
				while (PlotVar->TraceEnable[t] == 0) t++;
				/*while (Event[i][shift[i]].Channel[0]->Waveforms->TraceIndex[SampIndex[t]]
				if ((s+dtime[i])<0)  fprintf(fplot, " ,\t"); // dtime negative: add empty samples
				else {
					if ((s-dtime[i])<0) { // dtime positive, s<dtime
						if (s + dtime[i] > (Event[i][shift[i]].Channel[0]->Waveforms->TraceIndex[SampIndex[t]])) {
							//fprintf(fplot, "%u,\t", Event[i][shift[i]].Channel[0]->Waveforms->Trace[SampIndex[t]] * PlotVar->Gain[t] + PlotVar->Offset[t]);
							printf("entra\n");
							SampIndex[t]++;
						}
						else {
							//fprintf(fplot, " ,\t");
						}
						//}*/
				if ((s >= 0 && s <= 5) && s != (Event[i][shift[i]].Channel[0]->Waveforms->TraceIndex[SampIndex[t]])) {
					fprintf(fplot, "%u,\t", Event[i][shift[i]].Channel[0]->baseline);
					/*} else if ((s-6) < dtime[i]) {
						printf("entra\n");
						fprintf(fplot, " ,\t");
						SampIndex[t]++;*/
				}
				else if ((s == ((Event[i][shift[i]].Channel[0]->Waveforms->TraceIndex[SampIndex[t]])) && SampIndex[t] < Event[i][shift[i]].Channel[0]->Waveforms->TraceNumber)) {
					fprintf(fplot, "%u,\t", Event[i][shift[i]].Channel[0]->Waveforms->Trace[SampIndex[t]] * PlotVar->Gain[t] + PlotVar->Offset[t]);
					SampIndex[t]++;
				}
				else {
					fprintf(fplot, " ,\t");
				}
				t++;
			}
		
			fprintf(fplot, "\n");
		}
	}
	fclose(fplot);
	t = 0;
	fprintf(PlotVar->plotpipe, "plot ");
	c = 2; // first data column
	for (i = 0; i < NumTraces; i++) {
		while (PlotVar->TraceEnable[t] == 0) t++;
		if (comma) fprintf(PlotVar->plotpipe, ", ");
		fprintf(PlotVar->plotpipe, "'%s' using ($1*%f):($%d*%f) title '%s' with step lc %d",fname, PlotVar->Xscale, c++, PlotVar->Yscale, PlotVar->TraceName[t], t + 1);
		comma = 1;
		t++;
	}

	fprintf(PlotVar->plotpipe, "\n");
	fflush(PlotVar->plotpipe);
	/* set the time gnuplot will have finished */
	WaitTime = npts / 20;
	if (WaitTime < 100)
		WaitTime = 100;
	return 0;
}

void CheckKeyboardCommands(int *handle, ZLEConfig_t  *ConfigVar, ZLEPlot_t *PlotVar)
{
	int c = 0, d = 0, ch = 0, i = 0, run = 0;
	int h2dcnt = 0;

	if (!kbhit()) return;
	c = getch();
	switch (c) {
	case 'q':
		ConfigVar->Quit = 1;
		break;
	case '1':
		ConfigVar->EnableTrack = ConfigVar->EnableTrack ^ ((0x0001 <<(8*ConfigVar->EnableHalf)) & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask);
		break;
	case '2':
		ConfigVar->EnableTrack = ConfigVar->EnableTrack ^ ((0x0002 << (8 * ConfigVar->EnableHalf)) & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask);
		break;
	case '3':
		ConfigVar->EnableTrack = ConfigVar->EnableTrack ^ ((0x0004 << (8 * ConfigVar->EnableHalf)) & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask);
		break;
	case '4':
		ConfigVar->EnableTrack = ConfigVar->EnableTrack ^ ((0x0008 << (8 * ConfigVar->EnableHalf)) & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask);
		break;
	case '5':
		ConfigVar->EnableTrack = ConfigVar->EnableTrack ^ ((0x0010 << (8 * ConfigVar->EnableHalf)) & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask);
		break;
	case '6':
		ConfigVar->EnableTrack = ConfigVar->EnableTrack ^ ((0x0020 << (8 * ConfigVar->EnableHalf)) & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask);
		break;
	case '7':
		ConfigVar->EnableTrack = ConfigVar->EnableTrack ^ ((0x0040 << (8 * ConfigVar->EnableHalf)) & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask);
		break;
	case '8':
		ConfigVar->EnableTrack = ConfigVar->EnableTrack ^ ((0x0080 << (8 * ConfigVar->EnableHalf)) & ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask);
		break;
	case 't':
		if (!ConfigVar->ContTrigger) {
			for (i = 0; i < ConfigVar->Nhandle; i++) CAEN_DGTZ_SendSWtrigger(handle[i]);
			printf("Single Software Trigger issued\n");
		}
		break;
	case 'g':
		ConfigVar->EnableHalf = (ConfigVar->EnableHalf == 0 ? 1 : 0);
		printf("Channels from octet %d can be selected for plotting through keys 1-8\n", ConfigVar->EnableHalf);
		break;
	case 'p':
		ConfigVar->SinglePlot = 1;
		break;
	case 's':
		if (ConfigVar->AcqRun == 0) {
			if(ConfigVar->SyncEnable) CAEN_DGTZ_SWStartAcquisition(handle[0]);
			else { for (i = 0; i < ConfigVar->Nhandle; i++) { if (ConfigVar->BoardConfigVar[i]->StartMode == CAEN_DGTZ_SW_CONTROLLED) CAEN_DGTZ_SWStartAcquisition(handle[i]); }}
			printf("Acquisition started\n");
			// board is in RUN mode
			ConfigVar->AcqRun = 1;
		} else {
			if (ConfigVar->SyncEnable) CAEN_DGTZ_SWStopAcquisition(handle[0]);
			else { for (i = 0; i < ConfigVar->Nhandle; i++) { if (ConfigVar->BoardConfigVar[i]->StartMode == CAEN_DGTZ_SW_CONTROLLED) CAEN_DGTZ_SWStopAcquisition(handle[i]); }}
			printf("Acquisition stopped\n");
			ConfigVar->AcqRun = 0;
		}
		break;
	case '+':	
		if (ConfigVar->BoardPlotted < ConfigVar->Nhandle-1) ConfigVar->BoardPlotted++; else ConfigVar->BoardPlotted = 0;
		break;
	case '-':
		if (ConfigVar->BoardPlotted > 0) ConfigVar->BoardPlotted--; else ConfigVar->BoardPlotted = ConfigVar->Nhandle-1;
		break;
	case 32:
		printf("\nBindkey help\n");
		printf("[q]          Quit\n");
		printf("[s]          Start/Stop acquisition\n");
		printf("[t]          Send a software trigger (single shot)\n");
		printf("[p]          Plot one event\n");
		printf("[g]          Switch octet for channel plotting selection\n");
		printf("[F1-F8,1-8]  Plot: include/exclude channels [0-16] (if they are enabled)\n");
		printf("[+]/[-]      Plot: move to the next/previous board (if any)\n");
		printf("[space]  This help\n\n");
		printf("Press a key to continue\n");
		getch();
		break;
	default:   break;
	}
	// Modify the X-axis in case the record length is different
	PlotVar->Xmax = (float)(4 * ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->RecordLength * 5 / 4)*ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->tSampl / 1000;
}

ERROR_CODES OpenPlotter(ZLEConfig_t *ConfigVar, ZLEPlot_t *PlotVar)
{
	char str[1000];
	int i;
	strcpy(str, ConfigVar->GnuPlotPath);
	strcat(str, GNUPLOT_COMMAND);
	if ((PlotVar->plotpipe = popen(str, "w")) == NULL) return ERR_WAVE_MALLOC;

	strcpy(PlotVar->Title, "XX725/XX730 ZLE Firmware Demo");
	strcpy(PlotVar->Xlabel, "us");
	strcpy(PlotVar->Ylabel, "ADC counts");
	for (i = 0; i<ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->BoardInfo.Channels; i++) {
		PlotVar->Gain[i] = 1;
		PlotVar->Offset[i] = 0;
	}
	PlotVar->Xscale = (float)(ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->tSampl / 1000.);
	PlotVar->Yscale = 1.0;
	PlotVar->Xmin = 0.;
	PlotVar->Xmax = (float)(4 * ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->RecordLength * 5 / 4)*ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->tSampl / 1000;
	PlotVar->Ymin = 0;
	PlotVar->Ymax = 16384;
	PlotVar->Xautoscale = 0;
	PlotVar->Yautoscale = 0;
	SetPlotOptions(PlotVar);
	return ERR_NONE;
}

void SetPlotOptions(ZLEPlot_t *PlotVar)
{
	fprintf(PlotVar->plotpipe, "set grid\n");
	fprintf(PlotVar->plotpipe, "set mouse\n");
	fprintf(PlotVar->plotpipe, "set xlabel '%s'\n", PlotVar->Xlabel);
	fprintf(PlotVar->plotpipe, "set ylabel '%s'\n", PlotVar->Ylabel);
	fprintf(PlotVar->plotpipe, "set title '%s'\n", PlotVar->Title);
	fprintf(PlotVar->plotpipe, "Xs = %f\n", PlotVar->Xscale);
	fprintf(PlotVar->plotpipe, "Ys = %f\n", PlotVar->Yscale);
	fprintf(PlotVar->plotpipe, "Xmax = %f\n", PlotVar->Xmax);
	fprintf(PlotVar->plotpipe, "Ymax = %f\n", PlotVar->Ymax);
	fprintf(PlotVar->plotpipe, "Xmin = %f\n", PlotVar->Xmin);
	fprintf(PlotVar->plotpipe, "Ymin = %f\n", PlotVar->Ymin);
	if (PlotVar->Xautoscale) {
		fprintf(PlotVar->plotpipe, "set autoscale x\n");
		fprintf(PlotVar->plotpipe, "bind x 'set autoscale x'\n");
	}
	else {
		fprintf(PlotVar->plotpipe, "set xrange [Xmin:Xmax]\n");
		fprintf(PlotVar->plotpipe, "bind x 'set xrange [Xmin:Xmax]'\n");
	}
	if (PlotVar->Yautoscale) {
		fprintf(PlotVar->plotpipe, "set autoscale y\n");
		fprintf(PlotVar->plotpipe, "bind y 'set autoscale y'\n");
	}
	else {
		fprintf(PlotVar->plotpipe, "set yrange [Ymin:Ymax]\n");
		fprintf(PlotVar->plotpipe, "bind y 'set yrange [Ymin:Ymax]'\n");
	}
	fflush(PlotVar->plotpipe);
}

int ClosePlotter(FILE **gnuplot)
{
	if (*gnuplot != NULL)
		pclose(*gnuplot);
	return 0;
}

int UpdateTime(int RefreshTime, uint64_t *PrevTime) {
	uint64_t CurrentTime = get_time();
	if ((CurrentTime - *PrevTime) > RefreshTime) {
		*PrevTime = CurrentTime;
		return 1;
	}
	else {
		return 0;
	}
}

void PrintData(Counter_t *Counter, Counter_t *CounterOld) {

	double RoRate;
	double TrigRate;
	if (Counter->ByteCnt == CounterOld->ByteCnt) printf("No data...\n");
	else {
		float ElapsedTime = 8 * (float)(Counter->MB_TS - CounterOld->MB_TS)/(float)1000000000.;
		RoRate = Counter->ByteCnt != CounterOld->ByteCnt ? (float)(Counter->ByteCnt - CounterOld->ByteCnt) / (ElapsedTime*1048576):0.;
		TrigRate = Counter->TrgCnt != CounterOld->TrgCnt ? (float)(Counter->TrgCnt - CounterOld->TrgCnt) / (ElapsedTime*1000.):0.;
		printf("Board data rate  : %.2f MB/s\n", RoRate);
		printf("Trigger Rate     : %.2f KHz\n",TrigRate);
		*CounterOld = *Counter;
	}
}

uint32_t CheckMallocSize(int handle) {
	uint32_t size, lsize, maxEvents, nChEnabled = 0, u32;
	if (((CAEN_DGTZ_GetRecordLength(handle, &lsize)) != CAEN_DGTZ_Success) ||
		((CAEN_DGTZ_GetChannelEnableMask(handle, &u32)) != CAEN_DGTZ_Success) ||
		((CAEN_DGTZ_GetMaxNumEventsBLT(handle, &maxEvents)) != CAEN_DGTZ_Success)) return 0;
	for (; u32 > 0; u32 >>= 1) if ((u32 & 0x1) != 0) nChEnabled++;
	size = (4 * ((((lsize * 2) + 1)*nChEnabled) + 4)*maxEvents);
	return size;
}

uint32_t min3(uint32_t a, uint32_t b, uint32_t c) {
	uint32_t m = a;
	if (m > b) m = b;
	if (m > c) m = c;
	return m;
}

uint32_t min2(uint32_t a, uint32_t b) {
	uint32_t m = a;
	if (m > b) m = b;
	return m;
}

void ResetCounter(Counter_t *Counter) {
	int i;
	Counter->MB_TS = 0;
	Counter->ByteCnt = 0;
	Counter->TrgCnt = 0;
	for (i = 0; i < 16; i++) Counter->OFCnt[i];
}
//  Check if plot has finished
/*int IsPlotterBusy()
{
	if (get_time() > Tfinish)
		Busy = 0;
	return Busy;
}*/
