/******************************************************************************
*
* CAEN SpA - Front End Division
* Via Vetraia, 11 - 55049 - Viareggio ITALY
* +390594388398 - www.caen.it
*
***************************************************************************//**
* \note TERMS OF USE:
* This program is free software; you can redistribute it and/or modify it under
* the terms of the GNU General Public License as published by the Free Software
* Foundation. This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. The user relies on the
* software, documentation and results solely at his own risk.
******************************************************************************/

#include <CAENDigitizer.h>

#include "WaveCut.h"
#include "keyb.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef LINUX
	#define Sleep(x) usleep((x)*1000)
	#define DEFAULT_CONFIG_FILE "/etc/WaveCut/WaveCut_Config.txt"
#else
	#define DEFAULT_CONFIG_FILE "WaveCut_Config.txt"
#endif



/* ###########################################################################
*  Global Variables (running flags set/reset by keyboard menu)
*  ########################################################################### */

int Quit=0;
int AcqRun=0;
int ChToPlot=0, BrdToPlot=0;
int ContinuousTrigger=0;
int SingleTrigger=0;
int DoPlotWave=0;
int DoSaveWave=0;
int ChannelToPlot[MAX_NBRD];
char path[255];

/* ###########################################################################
*  Functions
*  ########################################################################### */

/* --------------------------------------------------------------------------------------------------------- */
/*! \fn      int ProgramDigitizer(int handle, DigitizerParams_t Params, CAEN_DGTZ_DPPParamsPHA_t DPPParams)
*   \brief   Program the registers of the digitizer with the relevant parameters
*   \return  0=success; -1=error */
/* ---------------------------.------------------------------------------------------------------------------ */
int ProgramDigitizer(int *handle, int brd, DPP_Config_t WDcfg)
{
	/* This function uses the CAENDigitizer API functions to perform the digitizer's initial configuration */
	int i, ret = 0, EnableMask=0;  
	uint32_t reg;

	for(i=0; i<MAX_NCH; i++)
		EnableMask |= (WDcfg.EnableInput[brd][i]<<i);


	// Set the I/O level (CAEN_DGTZ_IOLevel_NIM or CAEN_DGTZ_IOLevel_TTL)
	ret |= CAEN_DGTZ_SetIOLevel(handle[brd], (CAEN_DGTZ_IOLevel_t)WDcfg.FPIOtype);

	// settings for the synchronization
	if (WDcfg.StartMode == START_MODE_INDEP_SW) {
		ret |= CAEN_DGTZ_SetAcquisitionMode(handle[brd], CAEN_DGTZ_SW_CONTROLLED);
		ret |= CAEN_DGTZ_SetRunSynchronizationMode(handle[brd], CAEN_DGTZ_RUN_SYNC_Disabled); 
	} else {
		ret |= CAEN_DGTZ_SetAcquisitionMode(handle[brd],CAEN_DGTZ_S_IN_CONTROLLED);  // Arm acquisition (Run will start with a level on S_IN/GPI FPIO)
		ret |= CAEN_DGTZ_WriteRegister(handle[brd], 0x8170, 0);   // Run Delay = 0 for all. To be modified ofr the slave boards to take into account the Run signal propagation delay
	}
	// settings for the Trigger Mode
	if (WDcfg.TriggerMode == TRIGGER_MODE_SELF)
		ret |= CAEN_DGTZ_WriteRegister(handle[brd], CAEN_DGTZ_TRIGGER_SRC_ENABLE_ADD, 0x00000000);         // accept SW trg only (ext trig is disabled)  
	else
		ret |= CAEN_DGTZ_WriteRegister(handle[brd], CAEN_DGTZ_TRIGGER_SRC_ENABLE_ADD, 0xC0000000);         // accept ext trg_in (from trg OR) or SW trg

	// direct TRGIN_to_PB when VETO enabled
	ret |= CAEN_DGTZ_ReadRegister(handle[brd], CAEN_DGTZ_FRONT_PANEL_IO_CTRL_ADD, &reg);  
	if (WDcfg.TriggerMode == TRIGGER_MODE_SELF)
      reg = RegisterSetBits(reg, 11, 11, 1);
	else
      reg = RegisterSetBits(reg, 11, 11,0);
	ret |= CAEN_DGTZ_WriteRegister(handle[brd], CAEN_DGTZ_FRONT_PANEL_IO_CTRL_ADD, reg);         

    reg = 0x00000110; //Control Register (0x8000): individual trg and sequential access
	// settings for the Veto Mode
	if (WDcfg.TriggerMode == TRIGGER_MODE_SELF) //veto mode disabled when EXT_TRG
      reg = RegisterSetBits(reg, 12, 13, WDcfg.VetoMode);
    reg = RegisterSetBits(reg, 3, 3, WDcfg.TestWave); //test waveform
    ret |= CAEN_DGTZ_WriteRegister(handle[brd], CAEN_DGTZ_BROAD_CH_CTRL_ADD, reg);  

	ret |= CAEN_DGTZ_WriteRegister(handle[brd], CAEN_DGTZ_FP_TRIGGER_OUT_ENABLE_ADD, EnableMask);    // propagate self trg to TRGOUT

	// Set the enabled channels
	ret |= CAEN_DGTZ_SetChannelEnableMask(handle[brd], EnableMask);

	// Set the number of samples for each waveform (fixed to 1024 sample)
	// The CAENDigitizerLib function CAEN_DGTZ_SetRecordLength cannot be used because:
	// -) Buffer Organization must always be 0xA (the motherbosrd counts 1024 events fot the "full" condition)
	// -) Custom Size cannot be 0, it must be RecordtLength/2.
	ret |= CAEN_DGTZ_WriteRegister(handle[brd], CAEN_DGTZ_BROAD_NUM_BLOCK_ADD, 0xA);    // Buffer organization (fixed to 1024 buffers)
	ret |= CAEN_DGTZ_WriteRegister(handle[brd], CAEN_DGTZ_CUSTOM_SIZE_REG, (WDcfg.RecordLength/2));   //0x8020

	ret |= CAEN_DGTZ_SetMaxNumAggregatesBLT(handle[brd], 1023);  // Number of buffers per BLT

	// Set the input delay size (in pairs of samples)
	ret |= CAEN_DGTZ_WriteRegister(handle[brd], 0x8034 , WDcfg.InputDelay);

	// Channel individual settings
	for(i=0; i<MAX_NCH; i++) {
		if (WDcfg.EnableInput[brd][i]) { 
			// DC offset
			ret |= CAEN_DGTZ_SetChannelDCOffset(handle[brd], i, WDcfg.DCoffset[brd][i]);

			// Set the Pre-Trigger size (in pairs of samples)
			ret |= _CAEN_DGTZ_SetDPPPreTriggerSize(handle[brd], i, WDcfg.PreTrigger);

			// Set the Ns under THR (in pairs of samples)
			ret |= CAEN_DGTZ_WriteRegister(handle[brd], 0x1078 + (i<<8) , WDcfg.N_LFW[brd][i]);

			ret |= CAEN_DGTZ_WriteRegister(handle[brd], 0x1060 + (i<<8), WDcfg.TrgThreshold[brd][i]); 

            ret |= CAEN_DGTZ_ReadRegister (handle[brd], 0x1080 + (i<<8), &reg);   // Alg Ctrl
            if (WDcfg.TriggerMode == TRIGGER_MODE_SELF) // AUTO_TRG disabled when EXT_TRG
			  reg = RegisterSetBits(reg, 24, 24, 0);
			else
			  reg = RegisterSetBits(reg, 24, 24, 1);
            reg = RegisterSetBits(reg, 16, 16, WDcfg.PulsePolarity[brd][i]);
            reg = RegisterSetBits(reg, 20, 22, WDcfg.NsBaseline[brd][i]);
            ret |= CAEN_DGTZ_WriteRegister(handle[brd], 0x1080 + (i<<8), reg);

		}
	}

	// Generic Write accesses with mask
	for(i=0; i<WDcfg.GWn; i++) {
		ret |= CAEN_DGTZ_ReadRegister(handle[brd], WDcfg.GWaddr[brd][i], &reg);
		reg = (reg & ~WDcfg.GWmask[brd][i]) | (WDcfg.GWdata[brd][i] & WDcfg.GWmask[brd][i]);
		ret |= CAEN_DGTZ_WriteRegister(handle[brd], WDcfg.GWaddr[brd][i], reg);
	}

	if (ret) {
		printf("Warning: errors found during the programming of the digitizer.\nSome settings may not be executed\n");
		return ret;
	} else {
		return 0;
	}

}

/* --------------------------------------------------------------------------------------------------------- */
/*! \fn      int ForceClockSync(int handle)
*   \brief   force clock sync in one board
*   \return  0=success; -1=error */
/* --------------------------------------------------------------------------------------------------------- */
int ForceClockSync(int handle)
{    
    int ret;
    Sleep(500);
    /* Force clock phase alignment */
    ret = CAEN_DGTZ_WriteRegister(handle, 0x813C, 1);
    /* Wait an appropriate time before proceeding */
    Sleep(2000);
    return ret;
}


/* --------------------------------------------------------------------------------------------------------- */
/* Get and execute keyboard commands 
/* --------------------------------------------------------------------------------------------------------- */
void CheckKeyboardCommands(int *handle, DPP_Config_t WDcfg)
{
	char c;
	int ch, b;
	uint32_t rdata[MAX_NBRD];

	// Check keyboard
	if(kbhit()) {
		c = getch();

		switch(c) {
		case 'q':  
			Quit = 1; 
			break;

		case 'i':
			for(b=0; b<WDcfg.NumBrd; b++)
				SaveRegImage(handle[b]);
			printf("Register Images saved to file");
			break;

		case 'c':  
			printf("Enter Channel number: ");
			scanf("%d", &ch);
			if (WDcfg.EnableInput[BrdToPlot][ch]) {
				ChannelToPlot[BrdToPlot] = ch;
				printf("Active Channel for plotting is now %d of Board %d\n", ch, BrdToPlot);
			} else {
				printf("Channel %d is not enabled\n", ch);
			}
			break;

		case 'b':  
			printf("Enter board number: ");
			scanf("%d", &BrdToPlot);
			printf("Active Board for plotting is now %d\n", BrdToPlot);
			break;

		case 'w':
			DoPlotWave ^= 1;
			break;

		case 'W':
			DoSaveWave ^= 1;
			printf("Waveform Saving = %d\n", DoSaveWave);
			break;

		case 'T':
			ContinuousTrigger = ContinuousTrigger ? 0 : 1;
			printf("Continuous Trigger = %d\n", ContinuousTrigger);
			break;

		case 't':
			SingleTrigger = 1;
			break;

		case 's':
			// Start Acquisition
			if (WDcfg.StartMode == START_MODE_INDEP_SW) {
				for(b=0; b<WDcfg.NumBrd; b++) {
					CAEN_DGTZ_SWStartAcquisition(handle[b]);
					printf("Acquisition Started for Board %d\n", b);
				}
			} else if (WDcfg.StartMode == START_MODE_SYNC_S_IN) {
				uint32_t d32a = 0.;
				uint32_t mask,bitshift,value; //value to write in the bits corresponding to mask 				
				for(b=0; b<WDcfg.NumBrd; b++) {
					CAEN_DGTZ_ReadRegister(handle[b], 0x811C, &d32a);
					mask = 0x38000; 
					bitshift = 0;
					value = 0x30000;
					d32a &= ~(mask << bitshift);
					d32a |= (value << bitshift);
					CAEN_DGTZ_WriteRegister(handle[b], 0x811C, d32a);
					CAEN_DGTZ_WriteRegister(handle[b], 0x8100, 0x5);
					CAEN_DGTZ_WriteRegister(handle[b], 0x8110, 0x0);
					d32a = 0.;
				}
			}
			AcqRun = 1;
			break;

		case 'S':
			for(b=0; b<WDcfg.NumBrd; b++) {
				// Stop Acquisition
				CAEN_DGTZ_SWStopAcquisition(handle[b]); 
				printf("Acquisition Stopped for Board %d\n", b);
			}
			AcqRun = 0;
			break;

		case ' ':
			printf("\n");
			printf("s )  Start acquisition\n");
			printf("S )  Stop acquisition\n");
			printf("t )  Send software trigger (one shot)\n");
			printf("T )  Enable/Disable continuous software trigger\n");
			printf("W )  Enable/Disable continuous Waveforms saving to file\n");
			printf("b )  Choose board on the plot\n");
			printf("c )  Choose channel on the plot\n");
			printf("w )  Enable/Disable continuous plot\n");
			printf("i )  Save Registers Image\n");
			printf("q )  Quit\n");
			printf("Space )  Print Menu\n\n");
			getch();
			break;

		default: break;
		}
	}
}



/* ########################################################################### */
/* MAIN                                                                        */
/* ########################################################################### */
int main(int argc, char *argv[])
{
	CAEN_DGTZ_ErrorCode ret = 0;
	CAEN_DGTZ_BoardInfo_t BoardInfo[MAX_NBRD];
	uint16_t *Waveform = NULL;
	char *buffer = NULL;  // readout buffer
	WaveCutEvent_t *Events[MAX_NCH];
	int handle[MAX_NBRD];
	uint32_t d32,run_cnt;
	int ChTref=0, BrdTref=0;

	uint64_t PrevTime[MAX_NBRD][MAX_NCH];
	uint64_t ExtendedTT[MAX_NBRD][MAX_NCH];
	int TrgCnt[MAX_NBRD][MAX_NCH];
	uint32_t Nb=0, EventCount=0;

	/* Other variables */
	int i, b, ch, ev;
	uint32_t AllocatedSize, BufferSize;
	int MajorNumber;
	uint64_t CurrentTime, PrevRateTime, ElapsedTime, PrevPlotTime;
	uint32_t NumEvents[MAX_NCH];
	
	// for Config Parsing
	char ConfigFileName[100];
	char tmpConfigFileName[100];
	DPP_Config_t  WDcfg;
	FILE *f_ini;
	FILE *wplot=NULL;
	FILE *wave[MAX_NBRD][MAX_NCH];

#ifdef WIN32
	sprintf(path, "%s\\WaveCut\\", getenv("USERPROFILE"));
	_mkdir(path);
#else
	sprintf(path, "");
#endif

	/* *************************************************************************************** */
	/* Open and parse configuration file                                                       */
	/* *************************************************************************************** */
	printf("************************************************************** \n");
	printf("*********       X724 Wavecut Demo %s %s     **********  \n", WAVECUT_VER, WAVECUT_REL_DATE);
	printf("************************************************************** \n \n");


	if (argc > 1) strcpy(ConfigFileName, argv[1]);
        else {
                strcpy(tmpConfigFileName, DEFAULT_CONFIG_FILE);
                sprintf(ConfigFileName, "%s%s", path, tmpConfigFileName);
        }


	printf("Opening Configuration File %s\n", ConfigFileName);
	f_ini = fopen(ConfigFileName, "r");
	if (f_ini == NULL ) {
		printf("No config file %s in the folder found \n", ConfigFileName);
		printf("\n>>> The software will quit in 10 seconds. \n");
		Sleep(10000);
		return ret;
	}

	ParseConfigFile(f_ini, &WDcfg);
	printf("Found settings for %d board(s) in the config file\n", WDcfg.NumBrd);
	fclose(f_ini);


    /* *************************************************************************************** */
    /* Open the digitizers and read board information                                           */
    /* *************************************************************************************** */
    /* The following function is used to open the digitizer with the given connection parameters
    and get the handler to it */
 		for(b=0; b<WDcfg.NumBrd; b++) {
		const void* arg;
		uint32_t linkNum;
		if (WDcfg.LinkType[b] == CAEN_DGTZ_ETH_V4718) {
			arg = WDcfg.HostName[b];
		}
		else {
			linkNum = WDcfg.LinkNum[b];
			arg = &linkNum;
		}
        ret = CAEN_DGTZ_OpenDigitizer2((CAEN_DGTZ_ConnectionType)WDcfg.LinkType[b], arg, WDcfg.ConetNode[b], WDcfg.BaseAddress[b], &handle[b]);
        if (ret) {
            printf("Can't open digitizer. Stopping at Board.%d\n",b);
            Sleep(3000);
            goto QuitProgram;    
        }

        // Reset the digitizer
        ret = CAEN_DGTZ_Reset(handle[b]);
        if (ret != 0) {
            printf("ERROR: can't reset the digitizer.\n");
            Sleep(3000);
            goto QuitProgram;    
        }

        /* Once we have the handler to the digitizer, we use it to call the other functions */
        ret = CAEN_DGTZ_GetInfo(handle[b], &BoardInfo[b]);
        if (ret) {
            printf("Can't read board info\n");
            Sleep(3000);
            goto QuitProgram;
        }
    }


	for(b=0; b<WDcfg.NumBrd; b++) {

		if (BoardInfo[b].FamilyCode != 0) {
			printf("ERROR: Digitizer model not valid\n");
			Sleep(3000);
			goto QuitProgram;
		}

		/* Check firmware revision (only DPP firmware can be used with this Demo) */
		sscanf(BoardInfo[b].AMC_FirmwareRel, "%d", &MajorNumber);
		if (MajorNumber == 137) {
			printf("This digitizer has a valid WaveCut firmware\n");
		} else {
			printf("ERROR: This digitizer has not a DPP firmware\n");
			Sleep(3000);
			goto QuitProgram;
		}

		uint32_t ser_num;
		uint32_t ser_num_1;
		
		ret = CAEN_DGTZ_ReadRegister(handle[b], 0xF080, &ser_num_1);
		ret = CAEN_DGTZ_ReadRegister(handle[b], 0xF084, &ser_num);

		ser_num &= ~(0xFF << 8);
		ser_num |= (ser_num_1 << 8);

		printf("\nConnected to CAEN Digitizer Model %s; board num. %d; ser. num. %d\n", BoardInfo[b].ModelName, b, ser_num);
		printf("ROC FPGA Release is %s\n", BoardInfo[b].ROC_FirmwareRel);
		printf("AMC FPGA Release is %s\n", BoardInfo[b].AMC_FirmwareRel);

        CAEN_DGTZ_ReadRegister(handle[b], 0x8158, &d32);
        if (d32 == 0x53D4)
            printf("The DPP is licensed\n");
		else { 
			if (d32 == 0){
              printf("\n>>> WARNING: DPP not licensed and Time Expired");
              printf("\n>>> Power ON the board.\n");
 	  		  Sleep(10000);
		      goto QuitProgram;   
			}
			else
			  printf("\n>>> WARNING: DPP not licensed: %d minutes remaining\n\n", (int)((float)d32/0x53D4 * 30));
		}
		// mask the channels not available for this model
		WDcfg.NumCh = BoardInfo[b].Channels;
		for(i=WDcfg.NumCh; i<MAX_NCH; i++)
	  	  WDcfg.EnableInput[b][i] = 0;
		BrdToPlot = 0;
		for(i=0; i<MAX_NCH; i++) {
			if (WDcfg.EnableInput[b][i]) {
				ChannelToPlot[b] = i;
				break;
				
			}
		}
		for(i=0; i<MAX_NCH; i++) {
			PrevTime[b][i]=0;
			ExtendedTT[b][i]=0; 
			wave[b][i]=NULL;
		}
	}


	/* open gnuplot in a pipe and the data file */
#ifdef LINUX
		wplot = popen("/usr/bin/gnuplot", "w");
#else
	{
		char tmp[255];
		sprintf(tmp, "%s\\pgnuplot.exe", path);
		wplot = _popen(tmp, "w");
	}
#endif
		if (wplot == NULL) {
			printf("Can't open gnuplot\n\n");
			goto QuitProgram;
		}
	fprintf(wplot, "set grid\n");
	fprintf(wplot, "set yrange [0:16384]\n");
	fprintf(wplot, "set title 'Board %d - Channel %d'\n", BrdToPlot, ChToPlot);
	DoPlotWave=1;

	/* *************************************************************************************** */
	/* Program the digitizer (see function ProgramDigitizer)                                   */
	/* *************************************************************************************** */
    for(b=0; b<WDcfg.NumBrd; b++) {
        ret = ProgramDigitizer(handle, b, WDcfg);
        if (ret) {
            printf("Failed to program the digitizer\n");
            Sleep(3000);
            goto QuitProgram;
        }
    }
	

	ret |= _CAEN_DGTZ_MallocWaveCutEvents(handle[0], Events, &AllocatedSize); 

//	ret |= _CAEN_DGTZ_MallocWaveCutWaveform(handle[0], &Waveform, &AllocatedSize); 
	if ((Waveform = (int16_t *)malloc(4*1024*1024*sizeof(uint16_t))) == NULL) { // (channel big memory)
        ret |= CAEN_DGTZ_OutOfMemory;
    }
	if ((buffer = (char *)malloc(32*1024*1024)) == NULL) { // (board big memory)
        ret |= CAEN_DGTZ_OutOfMemory;
     }
	buffer = (char *)malloc(32*1024*1024); // (channel big memory)
	
	if (ret) {
		printf("Can't allocate memory buffers\n");
		Sleep(3000);
		goto QuitProgram;    
	}


	/* *************************************************************************************** */
	/* Readout Loop                                                                            */
	/* *************************************************************************************** */
	// Clear  counters
	for(b=0; b<WDcfg.NumBrd; b++) {
		for(ch=0; ch<WDcfg.NumCh; ch++) {
            TrgCnt[b][ch] = 0;
		}
	}
	PrevRateTime = get_time();
	PrevPlotTime = PrevRateTime;
	AcqRun = 0;
	EventCount=0;
	printf("Press [Space] for help\n\n");
	run_cnt=0;
	while(!Quit) {

		// Check from commands from the keyboard 
		CheckKeyboardCommands(handle, WDcfg);

		if (!AcqRun) {
			Sleep(10);
			continue;
		}

		/* Send a software trigger to each board */
		if ((ContinuousTrigger) || (SingleTrigger)) {
			for(b=0; b<WDcfg.NumBrd; b++)
				CAEN_DGTZ_SendSWtrigger(handle[b]);
			if (SingleTrigger)				
				printf("Single Software Trigger issued\n");
			SingleTrigger = 0;
		}

		/* Log throughput and trigger rate to screen (every second) */
		CurrentTime = get_time();
		ElapsedTime = CurrentTime - PrevRateTime; /* milliseconds */
		if (ElapsedTime > 1000) {
#ifdef WIN32
			system("cls");
#else
			system("clear");
#endif
			printf("Press [Space] for help\n\n");
			printf("Readout Rate=%.2f MB\n", (float)Nb/((float)ElapsedTime*1048.576f));
			//printf("EventCount = %ld\n", EventCount);
			for(b=0; b<WDcfg.NumBrd; b++) {
				printf("\nBoard %d:\n",b);
				for(i=0; i<WDcfg.NumCh; i++) {
					if (!WDcfg.EnableInput[b][i]) 
						printf("\tCh %d:\tDisabled\n", i);
					else if (TrgCnt[b][i]>0)
						printf("\tCh %d:\tTrgRate=%.2f KHz\n", i, (float)TrgCnt[b][i]/(float)ElapsedTime);
					else
						printf("\tCh %d:\tNo Data\n", i);
					TrgCnt[b][i]=0;
				}
			}
//			printf("run counter %d\n\n", run_cnt++);
			printf("\n\n");
			Nb = 0;
			PrevRateTime = CurrentTime;
		}
	
		/* Read data from the boards or from the files */
		for(b=0; b<WDcfg.NumBrd; b++) {
			ret = CAEN_DGTZ_ReadData(handle[b], CAEN_DGTZ_SLAVE_TERMINATED_READOUT_MBLT, buffer, &BufferSize);
			if (ret) {
				printf("Readout Error\n");
				Sleep(10000);
				goto QuitProgram;    
			}
			if (BufferSize == 0)
				continue;

			Nb += BufferSize;
			ret = _CAEN_DGTZ_GetDPPEvents(handle[b], buffer, BufferSize, Events, NumEvents);
			if (ret) {
				printf("Data Error: %d\n", ret);
				Sleep(3000);	
				goto QuitProgram;
			}
			ChToPlot=ChannelToPlot[b];
  	        /* Analyze data */
			for(ch=0; ch<WDcfg.NumCh; ch++) {
				if (!WDcfg.EnableInput[b][ch])  
					continue;

				for(ev=0; ev<(int)NumEvents[ch]; ev++) {  // go through entire events buffer
					int PlotThisWaveform = 0;

					TrgCnt[b][ch]++;
					// Extend time stamp to 64 bit
					if ((Events[ch][ev].TimeStamp & 0x3FFFFFFF) < PrevTime[b][ch]) 
						ExtendedTT[b][ch]++;
					PrevTime[b][ch] = Events[ch][ev].TimeStamp & 0x3FFFFFFF;
					Events[ch][ev].TimeStamp |= ExtendedTT[b][ch]<<30;

					if (DoPlotWave && (b == BrdToPlot) && (ch == ChToPlot) && ((CurrentTime-PrevPlotTime)>200))
						PlotThisWaveform = 1;

					// Fill data in the Generic Event Struct
					ConvertWaveCutEventData(handle[b], DoSaveWave | PlotThisWaveform, Events[ch][ev], Waveform);

					/* Plot Waveform */
					if (PlotThisWaveform) {
  					   // Fill data in the Generic Event Struct
                        FILE *wpdata;
						char tmp[255];
						sprintf(tmp, "%sPlotData.txt", path);
                        wpdata = fopen(tmp, "w");
                        /* Save Waveform and Plot it using gnuplot */
                        for(i=0; i<(int)Events[ch][ev].Ns; i++)
                            fprintf(wpdata, "%d\n", Waveform[i]);
                        fclose(wpdata);
                        fprintf(wplot, "set title '1: Channel %d'\n", ChToPlot);
                        fprintf(wplot, "plot '%s' u 1 t 'Channel %d' w step 1", tmp, ChToPlot);
                        fprintf(wplot, "\n");
                        fflush(wplot);
						PrevPlotTime = CurrentTime;
					} 

					/* Save Waveform */
					if (DoSaveWave) {
						SaveWaveform(wave, b, ch, Waveform, Events[ch][ev].Ns, Events[ch][ev].TimeStamp, WDcfg.OutFileFormat);
                    }

				} // loop on events
			} // loop on channels
		} // loop on boards
	} // End of readout loop


QuitProgram:
	/* stop the acquisition, close the device and free the buffers */
	for(b=0; b<WDcfg.NumBrd; b++) {
		CAEN_DGTZ_SWStopAcquisition(handle[b]);
		CAEN_DGTZ_CloseDigitizer(handle[b]);
	}
	free(buffer);
	free(Waveform);
	CAEN_DGTZ_FreeDPPEvents(handle[0], Events);
	if (wplot != NULL)	
		fclose(wplot);
	Sleep(100);
	return ret;
}

