/******************************************************************************
* 
* CAEN SpA - Front End Division
* Via Vetraia, 11 - 55049 - Viareggio ITALY
* +390594388398 - www.caen.it
*
***************************************************************************//**
* \note TERMS OF USE:
* This program is free software; you can redistribute it and/or modify it under
* the terms of the GNU General Public License as published by the Free Software
* Foundation. This program is distributed in the hope that it will be useful, 
* but WITHOUT ANY WARRANTY; without even the implied warranty of 
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. The user relies on the 
* software, documentation and results solely at his own risk.
******************************************************************************/

#ifndef WAVECUT_H
#define WAVECUT_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "CAENDigitizer.h"


#ifdef WIN32
    #include <time.h>
    #include <sys/timeb.h>
    #include <conio.h>
    #include <process.h>
	#define		_PACKED_
	#define		_INLINE_		
#else
    #include <unistd.h>
    #include <stdint.h>   /* C99 compliant compilers: uint64_t */
    #include <ctype.h>    /* toupper() */
    #include <sys/time.h>
	#define		_PACKED_		__attribute__ ((packed, aligned(1)))
	#define		_INLINE_		__inline__ 
#endif

#define WAVECUT_VER "1.1"
#define WAVECUT_REL_DATE "November 2023"

//****************************************************************************
// Definition of limits
//****************************************************************************
#define ENABLE_TRGOUT_TEST_PULSE    0 

#define MAX_NBRD	20         /* max. number of boards */
#define MAX_NCH		8          /* max. number of channels per board */

#define MAX_GW		1000       /* max. number of generic write commads */


//****************************************************************************
// Definition of options for some parameters
//****************************************************************************
#define START_MODE_INDEP_SW		0
#define START_MODE_SYNC_S_IN	1
#define START_MODE_SYNC_HW		2

#define TRIGGER_MODE_SELF		0
#define TRIGGER_MODE_EXTERNAL	1

#define VETO_DISABLED 		    0
#define VETO_ACTIVE_HIGH  	    3
#define VETO_ACTIVE_LOW  	    1

#define OUTFILE_BINARY			0
#define OUTFILE_ASCII			1

#define TESTWAVE_DISABLED   	0
#define TESTWAVE_ENABLED		1

//****************************************************************************
// Definition of Start of Run Modes
//****************************************************************************
// start on software command 
#define RUN_START_ON_SOFTWARE_COMMAND     0xC 
// start on S-IN level (logical high = run; logical low = stop)
#define RUN_START_ON_SIN_LEVEL            0xD
// start on first TRG-IN or Software Trigger 
#define RUN_START_ON_TRGIN_RISING_EDGE    0xE
// start on LVDS I/O level
#define RUN_START_ON_LVDS_IO              0xF


//****************************************************************************
// Definition of DPP configuration parameter struct
//****************************************************************************
typedef struct DPP_Config_t {
	// Parameters for the connection (one per board)
	int LinkType[MAX_NBRD];
    int LinkNum[MAX_NBRD];
	char HostName[MAX_NBRD][64];
    int ConetNode[MAX_NBRD];
    uint32_t BaseAddress[MAX_NBRD];	// for VME boards only

	int NumBrd;                     // Tot number of boards
	int NumCh;                      // Num of channel per board

	// Output Files
	int OutFileFormat;				// BINARY or ASCII

	// Acquisition modes
	int TriggerMode;				// SELF, EXTERNAL
	int StartMode;					// INDEP_SW, SYNC_SW, SYNC_HW
	int StopOnTime;					// Stop after N seconds (never stop when 0)
	int StopOnEvents;				// Stop after N events acquired (never stop when 0)

	int VetoMode;			    	// Disabled, VETO_HIGH, VETO_LOW

	// Common Settings
    int FPIOtype;					// Front Panel IOtype (NIM/TTL)
    int InputDelay;					// input delay (pairs of samples)
    int RecordLength;				// Num of samples in the waveform (or pairs of samples)
    int PreTrigger;					// Num of samples in the pre trigger (or pairs of samples)

	int TestWave;			    	// Disabled, Enabled

	// Channel parameters
	int EnableInput[MAX_NBRD][MAX_NCH];         // Enable input 
    int TrgThreshold[MAX_NBRD][MAX_NCH];		// Trigger Threshold
	int PulsePolarity[MAX_NBRD][MAX_NCH];		// Pulse Polarity (0=pos, 1=neg)
	int NsBaseline[MAX_NBRD][MAX_NCH];			// Num of Samples for the baseline calculation (value depends on FW type)
	int N_LFW[MAX_NBRD][MAX_NCH];				// Num of pairs of samples unde thr (value depends on FW type)
    int DCoffset[MAX_NBRD][MAX_NCH];			// input DC offset (from 0 to 65535)


	// Coincidences and trigger logic
//	int EnableCoinc;							// Enable coincidences
//	int CoincWindow;							// Coincidence Window (in ns)
//  int CoincMode[MAX_NBRD][MAX_NCH+1];			// Coincidence Mode (0=disabled, 1=Coincidence, 2=AntiCoincidence)
//	int CoincMask[MAX_NBRD][MAX_NCH+1];			// Coincidence Mask (combines channel and external triggers using AND, OR or Majority)
    
	// Generic write accesses
	int GWn;
    uint32_t GWaddr[MAX_NBRD][MAX_GW];			// Register Address
    uint32_t GWdata[MAX_NBRD][MAX_GW];			// Data to write
    uint32_t GWmask[MAX_NBRD][MAX_GW];			// Bit Mask
	
} DPP_Config_t;

// Event Data Structure
typedef struct
{
	uint64_t TimeStamp;
    uint32_t Ns;
    uint32_t *Wave;
} WaveCutEvent_t;

//****************************************************************************
// Function prototypes
//****************************************************************************
int ParseConfigFile(FILE *f_ini, DPP_Config_t *WRcfg);
long get_time();
uint32_t RegisterSetBits(uint32_t reg, int start_bit, int end_bit, int val);
int SaveWaveform(FILE *wave[MAX_NBRD][MAX_NCH], int b, int ch, uint16_t *Waveform, int Ns, uint64_t TimeStamp, int format);
int SaveRegImage(int handle);

CAEN_DGTZ_ErrorCode _CAEN_DGTZ_MallocWaveCutEvents(int handle, WaveCutEvent_t **Events, uint32_t *AllocatedSize);
CAEN_DGTZ_ErrorCode _CAEN_DGTZ_FreeWaveCutEvents(int handle, WaveCutEvent_t **Events); 

int ConvertWaveCutEventData(int handle, int DecodeWave, WaveCutEvent_t Event, uint16_t *Waveform);
static CAEN_DGTZ_ErrorCode CAEN_DGTZ_WaveCutDecodeEvent(int handle, int ch, uint32_t *data, WaveCutEvent_t *events);
CAEN_DGTZ_ErrorCode _CAEN_DGTZ_GetDPPEvents(int handle, char *buffer, int BufferSize, WaveCutEvent_t **Events, uint32_t *NumEvents);

CAEN_DGTZ_ErrorCode _CAEN_DGTZ_SetDPPPreTriggerSize(int handle, int ch, uint32_t samples);

#endif
