
#define PeakSensing_Release        "1.1.0"
#define PeakSensing_Release_Date   "November 2023"
#define DBG_TIME
#pragma once
#define _CRT_SECURE_NO_WARNINGS 1 
#define _WINSOCK_DEPRECATED_NO_WARNINGS 1 
#define WIN32_LEAN_AND_MEAN

#define FILES_IN_LOCAL_FOLDER	0

#include "CAENPeakSensingType.h"
#include "CAENPeakSensing.h"
#include "PeakSensingDAQConst.h"
#include "PeakSensingDAQFunc.h"
#include <keyb.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>

char path[128];

static long get_time()
{
long time_ms;
#ifdef WIN32
struct _timeb timebuffer;
_ftime(&timebuffer);
time_ms = (long)timebuffer.time * 1000 + (long)timebuffer.millitm;
#else
struct timeval t1;
struct timezone tz;
gettimeofday(&t1, &tz);
time_ms = (t1.tv_sec) * 1000 + t1.tv_usec / 1000;
#endif
return time_ms;
}

int main(int argc, char *argv[]) {
	
	int board,channel; // board, channel index
	int *handle = NULL; // handle pointer
	ERROR_CODES ErrCode = ERR_NONE; // global demo error code 
	// Data, Counter and Config structures
	CAEN_PADC_Event_t *Event;
	Counter_t *Counter,*CounterOld; 
	CAEN_PADC_BoardInfo_t BoardInfo;
	PeakSensingConfig_t ConfigVar;
	PeakSensingPlot_t PlotVar;
	// File I/O and related variables
	FILE **RawFile;
	FILE **ListFile;
	FILE ***HistoFile;
	FILE *f_ini = NULL;
	int *RawFileIndex, *ListFileIndex;
	int nline = 0, HistoFileIndex=0;
	char ConfigFileName[255];
	char tmpConfigFileName[255];
	struct stat buf;
	struct stat info;
	// readout buffer and related variables
	char *buffer = NULL;
	uint32_t BufferSize, NumEvents;
	// other variables
	uint32_t *MB_Cnt_Roll = NULL;
	uint64_t PrevCheckTime, PrevRefreshTime, LastDataTime;

	printf("**************************************************************\n");
	printf("                PeakSensing DAQ %s\n", PeakSensing_Release);
	printf("**************************************************************\n");

#ifdef  WIN32
	sprintf(path, "%s\\PeakSensing_DAQ\\", getenv("USERPROFILE"));
	_mkdir(path);
#else
	sprintf(path, "");
#endif

	// Open and parse configuration file 
	if (argc > 1) strcpy(ConfigFileName, argv[1]);
	else {
		strcpy(tmpConfigFileName, DEFAULT_CONFIG_FILE);
		sprintf(ConfigFileName, "%s%s", path, tmpConfigFileName);
	}
	ErrCode = OpenConfigFile(&f_ini, ConfigFileName);
	if (ErrCode == ERR_NONE) ErrCode = ParseConfigFile(f_ini, &ConfigVar);
	if (f_ini != NULL) fclose(f_ini);

	// Allocate space for handles and counters according to the number of boards in the acquisition chain
	if (ErrCode == ERR_NONE) {
		if (((handle = (int*)calloc(ConfigVar.Nhandle, sizeof(int)))==NULL) ||
			((MB_Cnt_Roll = (uint32_t*)calloc(ConfigVar.Nhandle, sizeof(uint32_t))) == NULL) ||
			((Counter = (Counter_t*)calloc(ConfigVar.Nhandle,sizeof(Counter_t)))==NULL) ||
			((CounterOld = (Counter_t*)calloc(ConfigVar.Nhandle, sizeof(Counter_t))) == NULL) ||
			((RawFileIndex = (int*)calloc(ConfigVar.Nhandle, sizeof(int))) == NULL) ||
			((ListFileIndex = (int*)calloc(ConfigVar.Nhandle, sizeof(int))) == NULL) ||
			((RawFile = (FILE**)calloc(ConfigVar.Nhandle, sizeof(FILE*))) == NULL) ||
			((ListFile = (FILE**)calloc(ConfigVar.Nhandle, sizeof(FILE*))) == NULL) ||
			((HistoFile = (FILE***)calloc(ConfigVar.Nhandle, sizeof(FILE**))) == NULL)
			) ErrCode = ERR_MALLOC;
	}

	// Open the peak-sensing devices and read the board information                                  
	if (ErrCode == ERR_NONE) {
		printf("Open peak-sensing devices\n");
		if (OpenPeakSensing(handle,ConfigVar)) {ErrCode = ERR_PADC_OPEN;}	
	}
	
	// Print board info
	if (ErrCode == ERR_NONE) {
		printf("Get board info and set board-specific parameters\n");
		for (board = 0; board < ConfigVar.Nhandle; board++) {
			if (CAEN_PADC_GetInfo(handle[board], &BoardInfo)) ErrCode = ERR_BOARD_INFO_READ;
			else {
				printf("****************************************\n");
				printf("Connected to CAEN Peak-Sensing Model %s\n", BoardInfo.ModelName);
				printf("ROC FPGA Release is %s\n", BoardInfo.ROC_FirmwareRel);
				printf("AMC FPGA Release is %s\n", BoardInfo.AMC_FirmwareRel);
			}
		}
		printf("****************************************\n");
	}

	// Program the peak-sensing devices                                                                  
	if (ErrCode == ERR_NONE) {
		printf("Program the peak-sensing\n");
		for (board = 0; board < ConfigVar.Nhandle; board++) if (ProgramPeakSensing(handle[board], ConfigVar.BoardConfigVar[board])) ErrCode = ERR_PS_PROGRAM;	
	}

	// Open the output files
	if (ErrCode == ERR_NONE) {
		if (stat(ConfigVar.OutFilePath, &info) != 0) {
			#ifdef  WIN32
			if (_mkdir(ConfigVar.OutFilePath) != 0) { printf("Output directory %s could not be created. Please verify that the path exists and is writable\n", ConfigVar.OutFilePath); ErrCode = ERR_OUTDIR_OPEN; }
				else printf("Output directory %s created\n", ConfigVar.OutFilePath);
			#else 
			if (mkdir(ConfigVar.OutFilePath, 0777) != 0) { printf("Output directory %s could not be created. Please verify that the path exists and is writable\n", ConfigVar.OutFilePath); ErrCode = ERR_OUTDIR_OPEN;}
			else printf("Output directory %s created\n", ConfigVar.OutFilePath);
			#endif
		}
		if (ErrCode == ERR_NONE) {
			for (board = 0; board < ConfigVar.Nhandle; board++) {
				RawFileIndex[board] = ListFileIndex[board] = 0;
				if (ConfigVar.OFRawEnable) { if ((ErrCode = OpenRawFile(RawFile + board, board, RawFileIndex[board], ConfigVar.OutFilePath, ConfigVar.OutFileName)) != ERR_NONE) break; }
				if (ConfigVar.OFListEnable) { if ((ErrCode = OpenListFile(ListFile + board, board, ConfigVar.BoardConfigVar[board]->ChMax, ListFileIndex[board], ConfigVar.OutFilePath, ConfigVar.OutFileName)) != ERR_NONE) break; }
				if (ConfigVar.OFHistoEnable) {
					if ((*(HistoFile + board) = (FILE**)calloc(ConfigVar.BoardConfigVar[board]->ChMax, sizeof(FILE*))) == NULL) { ErrCode = ERR_MALLOC; break; }
					if ((ErrCode = OpenHistoFile(HistoFile + board, board, ConfigVar.BoardConfigVar[board], ConfigVar.OutFilePath, ConfigVar.OutFileName)) != ERR_NONE) break;
				}
			}
		}
	}

	if (ErrCode == ERR_NONE) {
		printf("Readout buffer malloc\n");
		Event = (CAEN_PADC_Event_t*)malloc(MAX_BLT_EVENTS*sizeof(CAEN_PADC_Event_t));
		if (CAEN_PADC_MallocReadoutBuffer(handle[0], &buffer, &BufferSize)!=CAEN_PADC_Success) ErrCode = ERR_MALLOC; // WARNING: This malloc must be done after the peak-sensing programming
	}

	// malloc and reset counters
	if(ErrCode== ERR_NONE) {
		printf("Counter malloc\n");
		for (board = 0; board < ConfigVar.Nhandle; board++) {
			if (MallocCounter(Counter + board) || MallocCounter(CounterOld + board)) {ErrCode = ERR_MALLOC; break;}
			ResetCounter(Counter+board);
			ResetCounter(CounterOld+board);
		}
	}

	// Calibration Upload into SRAM
	if(ErrCode== ERR_NONE) {
		printf("Calibration upload\n");	
		for (board = 0; board < ConfigVar.Nhandle; board++) {
			if (CAEN_PADC_LoadOffset(handle[board]) != CAEN_PADC_Success) { ErrCode = ERR_CALIB_UPLOAD; break; }
			if (CAEN_PADC_LoadCalibration(handle[board]) != CAEN_PADC_Success) {ErrCode = ERR_CALIB_UPLOAD; break;}
		}
	}

	// Open the plotter and configure its options
	if (ErrCode == ERR_NONE) {
		printf("Open plotter\n");
		ErrCode = OpenPlotter(&ConfigVar, &PlotVar);
	}
	// Readout Loop                                                                            
	if (ErrCode == ERR_NONE) {
		printf("[s] start/stop the acquisition, [q] quit, [space key] help\n");
		PrevCheckTime = PrevRefreshTime = get_time();
		while (!ConfigVar.Quit) {
			if (UpdateTime(100, &PrevRefreshTime)) CheckKeyboardCommands(handle, &ConfigVar, Counter, CounterOld);
			
			if (ConfigVar.ContTrigger) for (board = 0; board < ConfigVar.Nhandle; board++) CAEN_PADC_SendSWtrigger(handle[board]);
			for (board = 0; board < ConfigVar.Nhandle; board++) {
				// Data Readout
				if (CAEN_PADC_ReadData(handle[board], CAEN_PS_SLAVE_TERMINATED_READOUT_MBLT, buffer, &BufferSize) != CAEN_PADC_Success) { ErrCode = ERR_READOUT; break; }
				if (BufferSize > 0) { // The BLT returned data
					LastDataTime = get_time();
					Counter[board].Byte_Cnt += BufferSize;
					// Event decoding
					if (CAEN_PADC_GetEvents(handle[board], buffer, BufferSize / 4, Event, &NumEvents) != CAEN_PADC_Success) {ErrCode = ERR_EVENT_BUILD; break;}
					// event counter roll-over
					if (Counter[board].MB_Cnt < CounterOld[board].MB_Cnt % ((uint32_t)pow(2., 24.)*(MB_Cnt_Roll[board] + 1))) MB_Cnt_Roll[board]++;
					Counter[board].MB_Cnt = (uint64_t)(pow(2., 24.)*MB_Cnt_Roll[board]) + (uint64_t)Event[NumEvents - 1].tcounter;
					Counter[board].MB_TS = Event[NumEvents - 1].timeStamp;
					// Add Events to counters
					AddEvents(Event, NumEvents, Counter + board, ConfigVar.BoardConfigVar[board]->ChMax);
					// Print counters and plot data
					if (ConfigVar.AcqRun && (UpdateTime(ConfigVar.PlotRefreshTime, &PrevCheckTime) || ConfigVar.SinglePlot)) {
						printf("==========================================\n");
						printf("Board %d", board); if (board == ConfigVar.BoardPlotted) printf("(plotted)\n"); else printf("\n");
						printf("Real time              : %*.2f s\n", 7, (get_time() - ConfigVar.StartTime) / 1000.000f);
						PrintData(handle[board], Counter + board, CounterOld + board, ConfigVar);
						CounterOld[board] = Counter[board];
						if ((ConfigVar.PlotEnable || ConfigVar.SinglePlot) && (board == ConfigVar.BoardPlotted)) {
							printf("Last plotted event\n");
							printf("Trigger index          :      %u\n", Event[0].tcounter);
							printf("Event timestamp        : %llu ns\n", 8 * Event[0].timeStamp);
							PlotEvent(&ConfigVar, &PlotVar, Counter + board);
							ConfigVar.SinglePlot = 0;
						}
						// save histograms
						if (ConfigVar.OFHistoEnable) {
							OpenHistoFile(HistoFile + board,board,ConfigVar.BoardConfigVar[board],ConfigVar.OutFilePath, ConfigVar.OutFileName);
							HistoWrite(*(HistoFile + board),Counter + board,ConfigVar.BoardConfigVar[board]);
						}
					}
					// write events on disk (if enabled)
					if (ConfigVar.OFRawEnable) fwrite(buffer, BufferSize, 1, RawFile[board]);
					if (ConfigVar.OFListEnable) ListWrite(ListFile[board], NumEvents, ConfigVar.BoardConfigVar[board], Event);
				} else { // no data from BLT	
					if (ConfigVar.AcqRun && UpdateTime(ConfigVar.PlotRefreshTime, &LastDataTime)) {
						printf("==========================================\n");
						printf("Time since reset: %.2f s\n", (get_time() - ConfigVar.StartTime) / 1000.000f);
						printf("No readout data in board %d\n", board);
					}
				}

				// Break the acquisition loop in case of errors
				if (ErrCode) break;
				// check file size and open new file if the file size is larger than the value set in the config file
				if (RawFile[board] != NULL ) {
					fstat(fileno(RawFile[board]),&buf);
					if ((int)(buf.st_size / MB_SIZE) > ConfigVar.MaxFileSize) {RawFileIndex[board]++;OpenRawFile(&RawFile[board],board,RawFileIndex[board],ConfigVar.OutFilePath, ConfigVar.OutFileName);}
				}
				if (ListFile[board] != NULL) {
					fstat(fileno(ListFile[board]),&buf);
					if ((int)(buf.st_size / MB_SIZE) > ConfigVar.MaxFileSize) {ListFileIndex[board]++;OpenListFile(&ListFile[board], board,ConfigVar.BoardConfigVar[board]->ChMax,ListFileIndex[board],ConfigVar.OutFilePath, ConfigVar.OutFileName);}
				}
			}
			if (ErrCode) break;
		}
	} 
	
	if (ErrCode) {printf("\a%s\n", ErrMsg[ErrCode]);printf("Press a key to quit\n");getch();}
	// stop the acquisition
	if (ErrCode == 0 || ErrCode > ERR_PADC_OPEN) for (board = 0; board < ConfigVar.Nhandle; board++) CAEN_PADC_SWStopAcquisition(handle[board]);
	// close the output files (if allocated)
	for (board = 0; board < ConfigVar.Nhandle; board++) {
		if (RawFile[board] != NULL) fclose(RawFile[board]);
		if (ListFile[board] != NULL) fclose(ListFile[board]);
		if (HistoFile[board] != NULL) {
			for (channel = 0; channel < ConfigVar.BoardConfigVar[board]->ChMax; channel++) {
				if (HistoFile[board][channel] != NULL) fclose(HistoFile[board][channel]);
			}
		}
	}
	// close the plotter
	ClosePlotter(&PlotVar);
	// free buffers (if allocated)
	if (buffer != NULL) CAEN_PADC_FreeReadoutBuffer(&buffer);
	// close the handle(s)
	if (ErrCode==0 || ErrCode > ERR_PADC_OPEN) for (board = 0; board < ConfigVar.Nhandle; board++) CAEN_PADC_ClosePeakSensing(handle[board]);
	return 0;
}
