#include "PeakSensingDAQFunc.h"

#include <inttypes.h>
#include <sys/types.h>
#include <sys/stat.h>

#define _CRT_SECURE_NO_WARNINGS

extern char path[128];

static long get_time()
{
	long time_ms;
#ifdef _WIN32
	struct _timeb timebuffer;
	_ftime64(&timebuffer);
	time_ms = (long)timebuffer.time * 1000 + (long)timebuffer.millitm;
#else
	struct timeval t1;
	struct timezone tz;
	gettimeofday(&t1, &tz);
	time_ms = (t1.tv_sec) * 1000 + t1.tv_usec / 1000;
#endif
	return time_ms;
}

void CheckKeyboardCommands(int *handle, PeakSensingConfig_t  *ConfigVar, Counter_t *Counter, Counter_t *CounterOld) {
	int c = 0;
	int board;
	char *buffer = NULL;
	if (!kbhit()) return;
	c = getch();
	switch (c) {
	case '0':
		ConfigVar->TrackPlotted = ConfigVar->TrackPlotted ^ (0x01);
		break;
	case '1':
		ConfigVar->TrackPlotted = ConfigVar->TrackPlotted ^ (0x02);
		break;
	case '2':
		ConfigVar->TrackPlotted = ConfigVar->TrackPlotted ^ (0x04);
		break;
	case '3':
		ConfigVar->TrackPlotted = ConfigVar->TrackPlotted ^ (0x08);
		break;
	case '4':
		ConfigVar->TrackPlotted= ConfigVar->TrackPlotted ^ (0x10);
		break;
	case '5':
		ConfigVar->TrackPlotted = ConfigVar->TrackPlotted ^ (0x20);
		break;
	case '6':
		ConfigVar->TrackPlotted = ConfigVar->TrackPlotted ^ (0x40);
		break;
	case '7':
		ConfigVar->TrackPlotted = ConfigVar->TrackPlotted ^ (0x80);
		break;
	case 'p':
		ConfigVar->SinglePlot = 1;
		break;
	case 'q':
		ConfigVar->Quit = 1;
		break;
	case 'c':
		printf("Select a channel index between 00 and 63 (enter two digits): ");
		int chan;
		char number[2];
		number[0] = getch();
		if (number[0]<'0' || number[0]>'9') {
			printf("Please enter digits only\n");
			Sleep(2000);
			break;
		}
		number[1] = getch();
		if (number[1]<'0' || number[1]>'9') {
			printf("Please enter digits only\n");
			Sleep(2000);
			break;
		}
		chan=strtoul(number, NULL, 0);
		if (chan < 0 || chan>63) {
			printf("\n Not a valid channel index. Please enter a number between 0 and 63\n");
			Sleep(2000);
			break;
		} else {
			ConfigVar->GroupPlotted = (uint16_t)(chan/8);
			ConfigVar->TrackPlotted = (uint8_t)(1<<(chan%8));
			break;
		}
	case 'r':
		for (board = 0; board < ConfigVar->Nhandle; board++) {
			ResetCounter(Counter + board);
			ResetCounter(CounterOld + board);
			CAEN_PADC_GetDeadTime(handle[board],&(ConfigVar->PileUpTime));
			CAEN_PADC_GetTotalTime(handle[board],&(ConfigVar->TotalTime));
		}
		ConfigVar->StartTime = get_time();
		;
		break;
	case 's':
		ConfigVar->StartTime = get_time();
		if (ConfigVar->AcqRun == 0) {
			for (board = 0; board < ConfigVar->Nhandle; board++) CAEN_PADC_SWStartAcquisition(handle[board]);
			printf("Acquisition started\n");
			ConfigVar->AcqRun = 1;
			for (board = 0; board < ConfigVar->Nhandle; board++) {
				ResetCounter(Counter+board);
				ResetCounter(CounterOld+board);
				CAEN_PADC_GetDeadTime(handle[board], &(ConfigVar->PileUpTime));
				CAEN_PADC_GetTotalTime(handle[board], &(ConfigVar->TotalTime));
			}
		} else {
			for (board = 0; board < ConfigVar->Nhandle; board++) CAEN_PADC_SWStopAcquisition(handle[board]);
			printf("Acquisition stopped\n");
			ConfigVar->AcqRun = 0;
		}
		break;
	case 't':
		if (!ConfigVar->ContTrigger) {
			for (board = 0; board < ConfigVar->Nhandle; board++) CAEN_PADC_SendSWtrigger(handle[board]);
			printf("Single Software Trigger issued\n");
		}
		break;
	case 'T':
			for (board = 0; board < ConfigVar->Nhandle; board++) ConfigVar->ContTrigger = (ConfigVar->ContTrigger == 1) ? 0 : 1;
		break;
	case '+':
		if (ConfigVar->BoardPlotted < ConfigVar->Nhandle - 1) ConfigVar->BoardPlotted++; else ConfigVar->BoardPlotted = 0;
		break;
	case '-':
		if (ConfigVar->BoardPlotted > 0) ConfigVar->BoardPlotted--; else ConfigVar->BoardPlotted = ConfigVar->Nhandle - 1;
		break;
	case '9':
		if (ConfigVar->GroupPlotted < ((int)(ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMax/8)-1)) ConfigVar->GroupPlotted++; else ConfigVar->GroupPlotted = 0;
		break;
	case '8':
		if (ConfigVar->GroupPlotted > 0) ConfigVar->GroupPlotted--; else ConfigVar->GroupPlotted = (int)(ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMax/8)-1;
		break;
	case 32:
		printf("\nBindkey help\n");
		printf("[p]          Plot one event\n");
		printf("[q]          Quit\n");
		printf("[r]          Reset Counters\n");
		printf("[s]          Start/Stop acquisition\n");
		printf("[t]          Send a software trigger (single shot)\n");
		printf("[T]          Send a software trigger (continuous)\n");
		printf("[c]          Plot: enter a channel ID (0/63). The corresponding channel will be plotted alone\n");
		printf("[0-7]        Plot: include/exclude group channels in the plot (if enabled)\n");
		printf("[8]/[9]      Plot: move to the previous/next channel group\n");
		printf("[+]/[-]      Plot: move to the next/previous board\n");
		printf("[space]  This help\n\n");
		printf("Press a key to continue\n");
		getch();
		break;
	default:   break;
	}
}

int ProgramPeakSensing(int handle, PeakSensingBoardConfig_t *BoardConfigVar) {
	int group;
	CAEN_PADC_ErrorCode ret = CAEN_PADC_Success;
	// reset
	ret |= CAEN_PADC_Reset(handle);
	// Front panel I/O level
	ret |= CAEN_PADC_SetIOLevel(handle, BoardConfigVar->FPIOtype);
	ret |= CAEN_PADC_SetAcquisitionMode(handle, CAEN_PADC_SW_CONTROLLED);
	ret |= CAEN_PADC_SetSWTriggerMode(handle,CAEN_PADC_TRGMODE_ACQ_ONLY);
	ret |= CAEN_PADC_SetExtTriggerInputMode(handle, CAEN_PADC_TRGMODE_ACQ_ONLY);
	// Record length: if the "gate" mode is set, the r.l. must be set as small as possible in order not to lose triggers
	ret |= BoardConfigVar->TrigMode? CAEN_PADC_SetRecordLength(handle, 1) : CAEN_PADC_SetRecordLength(handle, BoardConfigVar->RecordLength);
	// set DAC mode (sliding scale or fixed)
	ret |= BoardConfigVar->SlScaleEnable? CAEN_PADC_SlScaleEnable(handle) : CAEN_PADC_SlScaleDisable(handle);
	// set Trigger mode (edge+register value or gate)
	ret |= CAEN_PADC_SetTriggerMode(handle,BoardConfigVar->TrigMode);
	// set Channel mode (bin resolution)
	ret |= CAEN_PADC_SetChMode(handle, BoardConfigVar->ChMode);
	// enable zero suppression
	ret |= BoardConfigVar->ZSEnable? CAEN_PADC_ZSEnable(handle) : CAEN_PADC_ZSDisable(handle);
	// set polarity
	ret |= CAEN_PADC_SetPolarity(handle, BoardConfigVar->Polarity);
	// set input range (0: 8V; 1: 4V)
	ret |= CAEN_PADC_SetInputRange(handle, BoardConfigVar->InputRange);
	// set group-dependent variables
	ret |= CAEN_PADC_GetMaxChannels(handle,&BoardConfigVar->ChMax);
	for (group = 0; group < BoardConfigVar->ChMax/8; group++) {
		ret |= CAEN_PADC_SetGroupZSThreshold(handle, group, BoardConfigVar->ZSThreshold[group]);
		ret |= CAEN_PADC_SetChannelEnableMask(handle, group,(uint32_t)(BoardConfigVar->EnableMask[group]));
	}
	if (ret) printf("Errors found during the programming of the digitizer\n");
	return ret;
}

int AddEvents(CAEN_PADC_Event_t *Event, uint32_t NumEvents, Counter_t *Counter, int MaxChannels) {
	int group, chan, chanptr = 0;
	uint32_t ev;
	int index = 0;
	for (ev = 0; ev < NumEvents; ev++) {
		Counter->Tot_Ev++;
		for (group = 0; group < MaxChannels / 16; group++) {
			chanptr = 0;
			// PU info in group header while the global one comes from mb. Only the first header is looked for PU (they are all the same anyhow)
			if (Event[ev].PS_Group[group].PileUp) {
				for (chan = 0; chan < MaxChannels; chan++) {
					if (((Event[ev].PS_Group[chan / 16].ChMask >> (chan % 16)) & 0x1)) {
						//Counter->Tot_Cnt[chan]++;
						//Counter->Pu_Cnt[chan]++; // il contatore di PU � incrementato SOLO se il canale � presente (non ZS)
					}
				}
				Counter->Pu_Ev++;
				break;
			}
			else {
				if(group==0) Counter->NotRej_Ev++;
				for (chan = 0; chan < 16; chan++) {
					if ((Event[ev].PS_Group[group].ChMask >> chan) & 0x1) {
						Counter->Tot_Cnt[(16 * group) + chan]++;
						if ((int)((*(Event[ev].PS_Group[group].DataPtr + chanptr)) & 0x8000)) {
							Counter->Of_Cnt[(16 * group) + chan]++;
							chanptr++;
							continue;
						}
						index = (int)((*(Event[ev].PS_Group[group].DataPtr + chanptr)) & 0x3fff);
						Counter->Amp_Cnt[(16 * group) + chan][index > 0 ? index : 0]++;
						chanptr++;
					}
				}
			}
		}
	}
	return 0;
}

void PrintData(int handle, Counter_t *Counter, Counter_t *CounterOld, PeakSensingConfig_t ConfigVar) {
	if ((Counter->Byte_Cnt - CounterOld->Byte_Cnt) == 0) printf("No data...\n");
	else {
		double TRate;      // rate of triggers seen by the MB		
		double TProcRate;  // rate of triggers processed by the DB (equal to the number of delivered event packets, including suppressed ones)
		double OCRate[8];  // rate of enabled channels (including saturated events)
		double OCRate_NoSat[8]; // rate of enabled channels (excluding saturated events. This is the rate of plotted events)
		double ICRate[8];  // estimated input count rate
		double PuRate;     // Pile-up rate. It is common to all channels (since there is a single pile-up reject connector)
		double SUPRate[8]; // rate of suppressed events
		double OFRate[8];  // rate of overflow events
		uint32_t TotalTime, PileUpTime;
		uint8_t GroupMask = ConfigVar.BoardConfigVar[ConfigVar.BoardPlotted]->EnableMask[ConfigVar.GroupPlotted];
		int i, index;
		double timeCnt = 8 * (double)(Counter->MB_TS - CounterOld->MB_TS);
		PuRate= (double)(Counter->Pu_Ev - CounterOld->Pu_Ev)*1000000. / timeCnt;
		for (i = 0; i<8; i++) {
			if ((GroupMask >> i) & 0x1) {
				index = (8 * ConfigVar.GroupPlotted) + i;
				if (Counter->NotRej_Ev == CounterOld->NotRej_Ev) SUPRate[i] = 0;
				else SUPRate[i] = 100.*(1 - ((double)(Counter->Tot_Cnt[index] - CounterOld->Tot_Cnt[index]) / (double)(Counter->NotRej_Ev - CounterOld->NotRej_Ev)));				
				OCRate[i] = ((double)(Counter->Tot_Cnt[index] - CounterOld->Tot_Cnt[index]))*1000000. / timeCnt;
				OCRate_NoSat[i] = ((double)(Counter->Tot_Cnt[index] - CounterOld->Tot_Cnt[index]) - (double)(Counter->Of_Cnt[index] - CounterOld->Of_Cnt[index]))*1000000. / timeCnt;
				if (Counter->Tot_Cnt[i]==CounterOld->Tot_Cnt[i]) OFRate[i] = 0;
				else OFRate[i] = 100.*(double)(Counter->Of_Cnt[i] - CounterOld->Of_Cnt[i]) / (double)(Counter->Tot_Cnt[i] - CounterOld->Tot_Cnt[i]);		
			} else { OCRate[i] = ICRate[i] = OCRate_NoSat[i] = 0; }
		}
		TRate = (double)(Counter->MB_Cnt - CounterOld->MB_Cnt)*1000000. / timeCnt;
		TProcRate = (double)(Counter->Tot_Ev - CounterOld->Tot_Ev)*1000000. / timeCnt;
		PuRate = (double)(Counter->Pu_Ev - CounterOld->Pu_Ev)*1000000. / timeCnt;
		CAEN_PADC_GetDeadTime(handle, &PileUpTime);
		CAEN_PADC_GetTotalTime(handle, &TotalTime);
		printf("Live Time              : %*.2f s\n", 7, (double)((TotalTime - ConfigVar.TotalTime) - (PileUpTime - ConfigVar.PileUpTime))*8.*65.536 / 1000000.);
		printf("Dead Time              : %*.2f %%\n", 7, (double)(100.*(PileUpTime - ConfigVar.PileUpTime) / (TotalTime - ConfigVar.TotalTime)));
		printf("Throughput             : %*.3f MB/s\n", 7, (double)(Counter->Byte_Cnt - CounterOld->Byte_Cnt) / timeCnt*1048.576f);
		printf("Detected trigger rate  : %*.3f kHz\n", 7, TRate);
		printf("Processed trigger rate : %*.3f kHz\n", 7, TProcRate);
		printf("Pile-up Rate           : %*.3f kHz\n", 7, PuRate);
		printf("Channel index          : "); for (i = 8 * ConfigVar.GroupPlotted; i < 8 * (1 + ConfigVar.GroupPlotted); i++) printf("%*d ", 7, i); printf("\n");
		printf("Saturated(SAT)         : "); for (i = 0; i < 8; i++) { if ((GroupMask >> i) & 0x1) printf("%*.2f ", 7, OFRate[i]); else printf("    N/A "); } printf(" %%\n");
		printf("Suppressed             : "); for (i = 0; i < 8; i++) { if ((GroupMask >> i) & 0x1) printf("%*.2f ", 7, SUPRate[i]); else printf("    N/A "); } printf(" %%\n");
		printf("OCR                    : "); for (i = 0; i < 8; i++) { if ((GroupMask >> i) & 0x1) printf("%*.3f ", 7, OCRate_NoSat[i]); else printf("    N/A "); } printf(" kcps\n");
		printf("OCR(incl. SAT)         : "); for (i = 0; i < 8; i++) { if ((GroupMask >> i) & 0x1) printf("%*.3f ", 7, OCRate[i]); else printf("    N/A "); } printf(" kcps\n");
	}
}

int UpdateTime(int RefreshTime, uint64_t *PrevTime) {
	uint64_t CurrentTime = get_time();
	if ((CurrentTime - *PrevTime) > RefreshTime) {
		*PrevTime = CurrentTime;
		return 1;
	}
	else {
		return 0;
	}
}

int OpenPeakSensing(int *handle, PeakSensingConfig_t ConfigVar)
{
	int board;
	for (board = 0; board < ConfigVar.Nhandle; board++) {
		printf("Board %d: ", board);
		const void* arg;
		uint32_t linkNum;
		if (ConfigVar.BoardConfigVar[board]->LinkType == CAEN_PADC_ETH_V4718) {
			printf("hostname: %s;", ConfigVar.BoardConfigVar[board]->HostName);
			arg = ConfigVar.BoardConfigVar[board]->HostName;
		}
		else {
			printf("link ID: %d;", ConfigVar.BoardConfigVar[board]->LinkNum);
			linkNum = ConfigVar.BoardConfigVar[board]->LinkNum;
			arg = &linkNum;
		}
		printf(" Node ID: %d;", ConfigVar.BoardConfigVar[board]->ConetNode);
		printf(" Base Address: %x\n", ConfigVar.BoardConfigVar[board]->BaseAddress);
		if (CAEN_PADC_OpenPeakSensing2(ConfigVar.BoardConfigVar[board]->LinkType, arg, ConfigVar.BoardConfigVar[board]->ConetNode, ConfigVar.BoardConfigVar[board]->BaseAddress, handle + board)!=CAEN_PADC_Success) return 1;
	}
	return 0;
}

void SetPlotOptions(PeakSensingPlot_t *PlotVar)
{
	fprintf(PlotVar->plotpipe, "set grid\n");
	fprintf(PlotVar->plotpipe, "set mouse\n");
	fprintf(PlotVar->plotpipe, "set xlabel '%s'\n", PlotVar->Xlabel);
	fprintf(PlotVar->plotpipe, "set ylabel '%s'\n", PlotVar->Ylabel);
	fprintf(PlotVar->plotpipe, "set title '%s'\n", PlotVar->Title);
	fprintf(PlotVar->plotpipe, "Xs = %f\n", PlotVar->Xscale);
	fprintf(PlotVar->plotpipe, "Ys = %f\n", PlotVar->Yscale);
	fprintf(PlotVar->plotpipe, "Xmax = %f\n", PlotVar->Xmax);
	fprintf(PlotVar->plotpipe, "Ymax = %f\n", PlotVar->Ymax);
	fprintf(PlotVar->plotpipe, "Xmin = %f\n", PlotVar->Xmin);
	fprintf(PlotVar->plotpipe, "Ymin = %f\n", PlotVar->Ymin);
	if (PlotVar->Xautoscale) {
		fprintf(PlotVar->plotpipe, "set autoscale x\n");
		fprintf(PlotVar->plotpipe, "bind x 'set autoscale x'\n");
	}
	else {
		fprintf(PlotVar->plotpipe, "set xrange [Xmin:Xmax]\n");
		fprintf(PlotVar->plotpipe, "bind x 'set xrange [Xmin:Xmax]'\n");
	}
	if (PlotVar->Yautoscale) {
		fprintf(PlotVar->plotpipe, "set autoscale y\n");
		fprintf(PlotVar->plotpipe, "bind y 'set autoscale y'\n");
	}
	else {
		fprintf(PlotVar->plotpipe, "set yrange [Ymin:Ymax]\n");
		fprintf(PlotVar->plotpipe, "bind y 'set yrange [Ymin:Ymax]'\n");
	}
	fflush(PlotVar->plotpipe);
}

ERROR_CODES OpenPlotter(PeakSensingConfig_t *ConfigVar, PeakSensingPlot_t *PlotVar)
{
	char str[200];
	int i;
	strcpy(str, ConfigVar->GnuPlotPath);
	strcat(str, GNUPLOT_COMMAND);
	if ((PlotVar->plotpipe = popen(str, "w")) == NULL) return ERR_HISTO_MALLOC;

	strcpy(PlotVar->Title, "PeakSensing DAQ Software");
	strcpy(PlotVar->Xlabel, "ADC channels");
	strcpy(PlotVar->Ylabel, "counts");
	for (i = 0; i<MAX_CH_PLOT; i++) {
		PlotVar->Gain[i] = 1;
		PlotVar->Offset[i] = 0;
	}
	
	PlotVar->Xscale = 1.;
	PlotVar->Yscale = 1.;
	PlotVar->Xmin = 0.;
	PlotVar->Xmax = (float)((int)pow(2., 14 - (double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode));
	PlotVar->Ymin = 0;
	PlotVar->Ymax = 16000;
	PlotVar->Xautoscale = 0;
	PlotVar->Yautoscale = 1;
	SetPlotOptions(PlotVar);
	return ERR_NONE;
}

int ClosePlotter(PeakSensingPlot_t *PlotVar)//FILE *gnuplot)
{
	if (PlotVar->plotpipe != NULL)
		pclose(PlotVar->plotpipe);
	return 0;
}

int PlotEvent(PeakSensingConfig_t *ConfigVar, PeakSensingPlot_t *PlotVar, Counter_t *Counter)
{
	int comma = 0;
	int c = 2;
	int i;
	uint32_t s;
	char fname[100];
	double Avg[MAX_CH_PLOT];
	uint64_t Num[MAX_CH_PLOT];
	uint64_t Sum[MAX_CH_PLOT];
	long double RMS[MAX_CH_PLOT];
	FILE *fplot;
	uint8_t TraceEnable = 0;
	if (PlotVar->Xmax != (float)((int)pow(2., 14 - (double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode))) {
		PlotVar->Xmax = (float)((int)pow(2., 14 - (double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode));
		ClosePlotter(PlotVar);
		OpenPlotter(ConfigVar, PlotVar);
	}

	for (i = 0; i<MAX_CH_PLOT; i++) {
		Avg[i] = 0; Sum[i] = 0; RMS[i] = 0; Num[i] = 0;
		if (ConfigVar->TrackPlotted & (1 << i)) {
			if (ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask[ConfigVar->GroupPlotted] & (1 << i)) {
				TraceEnable = TraceEnable | (uint8_t)(1 << i);
			}
			else {
				printf("Trace %d enabled for plotting but masked in data\n", i + (MAX_CH_PLOT * ConfigVar->GroupPlotted));
			}
		}
	}

	if (!TraceEnable) { printf("All traces disabled\n"); return -1; }
	
	sprintf(fname, "%s%s", path, PLOT_DATA_FILE);
	if ((fplot = fopen(fname, "w")) == NULL) { printf("file not allocated\n"); return -1; }

	for (s = 0; s<(int)pow(2., 14 - (double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode); s++) {
		fprintf(fplot, "%d,\t", s);
		for (i = 0; i<MAX_CH_PLOT; i++) {
			if (TraceEnable&(uint8_t)(1 << i)) {
				fprintf(fplot, "%lu,\t", (uint32_t)Counter->Amp_Cnt[i + (MAX_CH_PLOT * ConfigVar->GroupPlotted)][s]);
				Num[i] += ((uint64_t)s*(uint64_t)Counter->Amp_Cnt[i + (MAX_CH_PLOT * ConfigVar->GroupPlotted)][s]);
				Sum[i] += (uint64_t)Counter->Amp_Cnt[i + (MAX_CH_PLOT * ConfigVar->GroupPlotted)][s];
			}
		}
		fprintf(fplot, "\n");
	}
	fclose(fplot);

	fprintf(PlotVar->plotpipe, " plot ");
	for (i = 0; i<MAX_CH_PLOT; i++) {
		if (TraceEnable&(uint8_t)(1 << i)) {
			Avg[i] = Sum[i]>0 ? ((double)Num[i] / (double)Sum[i]) : 0.;
			for (s = 0; s < (int)pow(2., 14 - (double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode); s++) RMS[i] += (long double)(((double)s - Avg[i])*((double)s - Avg[i])*(double)Counter->Amp_Cnt[i + (MAX_CH_PLOT * ConfigVar->GroupPlotted)][s]);
			RMS[i] = Sum[i]>0 ? sqrt(RMS[i] / (long double)Sum[i]) : 0.;
			sprintf(PlotVar->Title, "Amplitude distributions group %d\n", ConfigVar->GroupPlotted);
			sprintf(PlotVar->TraceName[i], "Ch. %d - Events: %d, Avg: %.2f, RMS: %.2Lf", i + (MAX_CH_PLOT*ConfigVar->GroupPlotted), Counter->Tot_Cnt[i + (MAX_CH_PLOT*ConfigVar->GroupPlotted)] - Counter->Pu_Cnt[i + (MAX_CH_PLOT*ConfigVar->GroupPlotted)], Avg[i], RMS[i]);
			if (comma) fprintf(PlotVar->plotpipe, ", ");
			fprintf(PlotVar->plotpipe, "'%s' using ($1*%f):($%d*%f) title '%s' with step lc %d", fname, PlotVar->Xscale, c++, PlotVar->Yscale, PlotVar->TraceName[i], i + 1);
			comma = 1;
		}
	}

	fprintf(PlotVar->plotpipe, "\n");
	fflush(PlotVar->plotpipe);
	return 0;
}

// this version does not require to save a file on disk, but turns out to be slower because of multiple calls to gnuplot

/*int PlotEvent_nofile(PeakSensingConfig_t *ConfigVar, PeakSensingPlot_t *PlotVar, Counter_t *Counter) {
	int comma = 0;
	int c = 2;
	int i;
	uint32_t s;
	double Avg[MAX_CH_PLOT];
	uint64_t Num[MAX_CH_PLOT];
	uint64_t Sum[MAX_CH_PLOT];
	long double RMS[MAX_CH_PLOT];
	uint8_t TraceEnable = 0;

	if (PlotVar->Xmax != (float)((int)pow(2.,14-(double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode))) {
		PlotVar->Xmax = (float)((int)pow(2.,14-(double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode));
		ClosePlotter(PlotVar);
		OpenPlotter(ConfigVar,PlotVar);
	}

	for (i = 0; i<MAX_CH_PLOT; i++) {
		Avg[i] = 0; Sum[i] = 0; RMS[i] = 0; Num[i] = 0;
		if (ConfigVar->TrackPlotted & (1 << i)) {
			if (ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->EnableMask[ConfigVar->GroupPlotted] & (1 << i)) TraceEnable = TraceEnable | (uint8_t)(1 << i);
			else printf("Trace %d enabled for plotting but masked in data\n", i + (MAX_CH_PLOT * ConfigVar->GroupPlotted));			
		}
	}

	if (!TraceEnable) { printf("All traces disabled\n"); return -1; }

	for (s = 0; s<(int)pow(2.,14-(double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode); s++) {
		for (i = 0; i<MAX_CH_PLOT; i++) {
			if (TraceEnable&(uint8_t)(1 << i)) {
				Num[i] += (uint64_t)s*(uint64_t)(Counter->Amp_Cnt[i + (MAX_CH_PLOT * ConfigVar->GroupPlotted)][s]);
				Sum[i] += (uint64_t)Counter->Amp_Cnt[i + (MAX_CH_PLOT * ConfigVar->GroupPlotted)][s];
			}
		}
	}

	sprintf(PlotVar->Title, "Amplitude distributions group %d\n", ConfigVar->GroupPlotted);
	fprintf(PlotVar->plotpipe, "plot");
	for (i = 0; i < MAX_CH_PLOT; i++) {
		if (TraceEnable&(uint8_t)(1 << i)) {
			// compute avg and RMS to be put in the plot title
			Avg[i] = Sum[i] > 0 ? ((double)Num[i] / (double)Sum[i]) : 0.;
			for (s = 0; s < (int)pow(2., 14 - (double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode); s++) RMS[i] += (long double)(((double)s - Avg[i])*((double)s - Avg[i])*(double)Counter->Amp_Cnt[i + (MAX_CH_PLOT * ConfigVar->GroupPlotted)][s]);
			RMS[i] = Sum[i] > 0 ? sqrt(RMS[i] / (long double)Sum[i]) : 0.;
			// pipe plot titles and options
			sprintf(PlotVar->TraceName[i], "Ch. %d - Events: %d, Avg: %.2f, RMS: %.2Lf", i + (MAX_CH_PLOT*ConfigVar->GroupPlotted), Counter->Tot_Cnt[i + (MAX_CH_PLOT*ConfigVar->GroupPlotted)] - Counter->Pu_Cnt[i + (MAX_CH_PLOT*ConfigVar->GroupPlotted)], Avg[i], RMS[i]);
			fprintf(PlotVar->plotpipe, " '-' title '%s' w step lc %d", PlotVar->TraceName[i], i);
			(i == MAX_CH_PLOT - 1) ? fprintf(PlotVar->plotpipe, "\n") : fprintf(PlotVar->plotpipe, ",");
		}
	}

	for (i = 0; i < MAX_CH_PLOT; i++) {
		if (TraceEnable&(uint8_t)(1 << i)) {
			for (s = 0; s < (int)pow(2., 14 - (double)ConfigVar->BoardConfigVar[ConfigVar->BoardPlotted]->ChMode); s++) {
				fprintf(PlotVar->plotpipe, "%f %f\n", s*PlotVar->Xscale, (uint32_t)Counter->Amp_Cnt[i + (MAX_CH_PLOT * ConfigVar->GroupPlotted)][s] * PlotVar->Yscale);
			}
			fflush(PlotVar->plotpipe);
			fprintf(PlotVar->plotpipe, "e\n");
		}
	}
	
	return 0;
}*/

ERROR_CODES OpenConfigFile(FILE **f_ini, char *ConfigFileName) {
	ERROR_CODES return_code = ERR_NONE;
	printf("Open configuration file %s\n", ConfigFileName);
	if ((*f_ini = fopen(ConfigFileName, "r")) == NULL) return_code = ERR_CONF_FILE_NOT_FOUND;
	return return_code;
}

ERROR_CODES ParseConfigFile(FILE *f_ini, PeakSensingConfig_t *ConfigVar) {
	ERROR_CODES return_code = ERR_NONE;
	char *str, str1[1000];
	char strline[1000];
	int line = 0;
	int i, j, group=-1, ch = -1, board = -1, val, Off = 0;
	uint64_t val64;
	float fval;
	int DefConfig = 0;
	/* Default settings */
	memset(ConfigVar, 0, sizeof(*ConfigVar));
	ConfigVar->Nhandle = 0;
    #ifdef WIN32
		sprintf(ConfigVar->OutFilePath,"%s%s",path,OUTFILE_PATH);
    #else
		sprintf(ConfigVar->OutFilePath, "%s%s", getenv("HOME"), OUTFILE_PATH);
    #endif
	sprintf(ConfigVar->OutFileName, "%s", OUTFILE_NAME);
	sprintf(ConfigVar->GnuPlotPath, "%s%s", path, PLOTTER_PATH);
	/* read config file and assign parameters */
	while (fgets(strline, 1000, f_ini) != NULL) { // get a line
		line++;
		if (!strline || strline[0] == '#' || strline[0] == ' ' || strline[0] == '\t' || strline[0] == '\n' || strline[0] == '\r') continue;
		str = strtok(strline, " \r\t\n");
		if (str[0] == '[') {
			fscanf(f_ini, "%d", &val);
			if (strstr(str, "COMMON")) { ch = -1; group = -1; board = -1; }
			else if (strstr(str, "BOARD")) { ch = -1; group = -1; board = (int)strtol(strtok(NULL, " \r\t\n"), NULL, 10); }
			else if (strstr(str, "GROUP")) { ch = -1; group = (int)strtol(strtok(NULL, " \r\t\n"), NULL, 10);}
			continue;
		}

		// OPEN: malloc memory for the board config variable, init it to default and read the details of physical path to the digitizer
		if (!strcmp(strcpy(str1, str), "OPEN")) {
			// malloc board config variable	
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle] = (PeakSensingBoardConfig_t*)malloc(sizeof(PeakSensingBoardConfig_t));
			// initialize parameters
			//ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ExtTriggerMode = CAEN_PADC_TRGMODE_ACQ_ONLY;
			//ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->StartMode = CAEN_PADC_SW_CONTROLLED;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->FPIOtype = CAEN_PADC_IOLevel_TTL;
			for(i=0;i<8;i++) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->EnableMask[i] = 0x0;
			//ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->GainFactor = 0;
			ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->RecordLength = 256;
			// end of initalization
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 1st argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "USB") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_PADC_USB;
			else if (strcmp(str, "PCI") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_PADC_OpticalLink;
			else if (strcmp(str, "USB_A4818") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_PADC_USB_A4818;
			else if (strcmp(str, "USB_V4718") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_PADC_USB_V4718;
			else if (strcmp(str, "ETH_V4718") == 0) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType = CAEN_PADC_ETH_V4718;
			else { printf("%s %s: Invalid connection type\n", str, str1); return -1; }

			switch (ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkType) {
			case CAEN_PADC_USB:
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 2nd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkNum = (int)strtol(str, NULL, 10);
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ConetNode = 0;
				if ((str = strtok(NULL, " \r\t\n")) == NULL) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = 0; // Optional BA
				else ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = (uint32_t)strtoul(str, NULL, 0);
				break;
			case CAEN_PADC_USB_V4718:
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 2nd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkNum = (int)strtol(str, NULL, 10); // PID
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ConetNode = 0;
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 3rd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = (uint32_t)strtoul(str, NULL, 0);
				break;
			case CAEN_PADC_OpticalLink:
			case CAEN_PADC_USB_A4818:
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 2nd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->LinkNum = (int)strtol(str, NULL, 10); // PID
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 3rd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ConetNode = (int)strtol(str, NULL, 10);
				if ((str = strtok(NULL, " \r\t\n")) == NULL) ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = 0; // Optional BA
				else ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = (uint32_t)strtoul(str, NULL, 0);
				break;
			case CAEN_PADC_ETH_V4718:
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 2nd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->HostName[0] = '\0';
				strncat(ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->HostName, str, sizeof(ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->HostName) - 1);
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->ConetNode = 0;
				if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No 3rd argument for %s. The command will be ignored\n", str1); continue; }
				ConfigVar->BoardConfigVar[ConfigVar->Nhandle]->BaseAddress = (uint32_t)strtoul(str, NULL, 0);
				break;
			default:
				break;
			}
			ConfigVar->Nhandle++;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// gnuplot exec path
		if (!strcmp(strcpy(str1, str), "GNUPLOT_PATH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			strcpy(ConfigVar->GnuPlotPath, str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable gnuplot (YES/NO)
		if (!strcmp(strcpy(str1, str), "PERIODIC_PLOT")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->PlotEnable = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->PlotEnable = 0;
			else { printf("%s = %s: invalid option\n", str1, str); return ERR_PARSE_CONFIG; }
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// gnuplot refresh time 
		if (!strcmp(strcpy(str1, str), "STAT_REFRESH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			ConfigVar->PlotRefreshTime = (int)strtol(str, NULL, 10);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable raw output (YES/NO)
		if (!strcmp(strcpy(str1, str), "OUTFILE_RAW")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->OFRawEnable = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->OFRawEnable = 0;
			else {printf("%s = %s: invalid option\n",str1,str); return ERR_PARSE_CONFIG;}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable list output (YES/NO)
		if (!strcmp(strcpy(str1, str), "OUTFILE_LIST")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->OFListEnable = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->OFListEnable = 0;
			else { printf("%s = %s: invalid option\n", str1, str); return ERR_PARSE_CONFIG; }
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable histo output (YES/NO)
		if (!strcmp(strcpy(str1, str), "OUTFILE_HISTO")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->OFHistoEnable = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->OFHistoEnable = 0;
			else { printf("%s = %s: invalid option\n", str1, str); return ERR_PARSE_CONFIG; }
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Output file path
		if (!strcmp(strcpy(str1, str), "OUTFILE_PATH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			#ifdef WIN32
				sprintf(ConfigVar->OutFilePath, "%s%s",path,str);
			#else
				sprintf(ConfigVar->OutFilePath, "%s%s", getenv("HOME"),str);
			#endif
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Output file name
		if (!strcmp(strcpy(str1, str), "OUTFILE_NAME")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			strcpy(ConfigVar->OutFileName, str);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Output file max size
		if (!strcmp(strcpy(str1, str), "OUTFILE_MAXSIZE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			ConfigVar->MaxFileSize = (int)strtol(str, NULL, 10);
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable sliding scale (YES/NO)
		if (!strcmp(strcpy(str1, str), "SLSCALE_ENABLE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "YES") == 0) ConfigVar->BoardConfigVar[i]->SlScaleEnable = 1;
					else if (strcmp(str, "NO") == 0) ConfigVar->BoardConfigVar[i]->SlScaleEnable = 0;
					else {printf("%s = %s: invalid option\n",str1,str); return ERR_PARSE_CONFIG;}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Enable zero-suppression (YES/NO)
		if (!strcmp(strcpy(str1, str), "ZS_ENABLE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "YES") == 0) ConfigVar->BoardConfigVar[i]->ZSEnable = 1;
					else if (strcmp(str, "NO") == 0) ConfigVar->BoardConfigVar[i]->ZSEnable = 0;
					else { printf("%s = %s: invalid option\n",str1,str); return ERR_PARSE_CONFIG; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Signal Polarity
		if (!strcmp(strcpy(str1, str), "POLARITY")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "POSITIVE") == 0) ConfigVar->BoardConfigVar[i]->Polarity = 1;
					else if (strcmp(str, "NEGATIVE") == 0) ConfigVar->BoardConfigVar[i]->Polarity = 0;
					else { printf("%s = %s: invalid option\n",str1,str); return ERR_PARSE_CONFIG; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Signal Input Range
		if (!strcmp(strcpy(str1, str), "INPUT_RANGE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "4V") == 0) ConfigVar->BoardConfigVar[i]->InputRange = 1;
					else if (strcmp(str, "8V") == 0) ConfigVar->BoardConfigVar[i]->InputRange = 0;
					else { printf("%s = %s: invalid option\n", str1, str); return ERR_PARSE_CONFIG; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "SPECTRUM_CHANNEL")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "1k") == 0) ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_10BITMODE;
					else if (strcmp(str, "2k") == 0) ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_11BITMODE;
					else if (strcmp(str, "4k") == 0) ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_12BITMODE;
					else if (strcmp(str, "8k") == 0) ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_13BITMODE;
					else if (strcmp(str, "16k") == 0) ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_14BITMODE;
					else { printf("%s = %s: invalid option\n", str1, str); return ERR_PARSE_CONFIG; }
					/*switch ((int)strtol(str, NULL, 10)) {
					case 14:
						ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_14BITMODE;
						break;
					case 13:
						ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_13BITMODE;
						break;
					case 12:
						ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_12BITMODE;
						break;
					case 11:
						ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_11BITMODE;
						break;
					case 10:
						ConfigVar->BoardConfigVar[i]->ChMode = CAEN_PADC_10BITMODE;
						break;
					default:
						printf("%s = %s: invalid option\n",str1,str);
						return ERR_PARSE_CONFIG;
						break;
					}*/
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		/*if (!strcmp(strcpy(str1, str), "SIGNAL_MODE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					ConfigVar->BoardConfigVar[i]->SignalMode = (int)strtol(str, NULL, 10);
					if(ConfigVar->BoardConfigVar[i]->SignalMode>4) printf("%s: invalid option\n", str);
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}*/

		/*
		// trigger mode (YES/NO)
		if (!strcmp(strcpy(str1, str), "TRIGGER_MODE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "GATE") == 0) ConfigVar->BoardConfigVar[i]->TrigMode = CAEN_PADC_GATE;
					else if (strcmp(str, "PULSE") == 0) ConfigVar->BoardConfigVar[i]->TrigMode = CAEN_PADC_PULSE;
					else {printf("%s = %s: invalid option\n", str1, str); return ERR_PARSE_CONFIG;}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}*/

		// Enable sliding scale (YES/NO)
		if (!strcmp(strcpy(str1, str), "GATE_PARALIZABLE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "YES") == 0) ConfigVar->BoardConfigVar[i]->GateParalizable = 1;
					else if (strcmp(str, "NO") == 0) ConfigVar->BoardConfigVar[i]->GateParalizable = 0;
					else {printf("%s = %s: invalid option\n", str1, str); return ERR_PARSE_CONFIG;}				
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "GATE_VALUE")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			fval = strtof(str, NULL);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) ConfigVar->BoardConfigVar[j]->GateValue = fval;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// continuous software trigger (YES/NO)
		/*if (!strcmp(strcpy(str1, str), "CONT_SWTRIGGER")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			if (strcmp(str, "YES") == 0) ConfigVar->ContTrigger = 1;
			else if (strcmp(str, "NO") == 0) ConfigVar->ContTrigger = 0;
			else {printf("%s = %s: invalid option\n",str1,str); return ERR_PARSE_CONFIG;}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}*/

		// Front Panel LEMO I/O level (NIM, TTL)
		if (!strcmp(strcpy(str1, str), "FPIO_LEVEL")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "TTL") == 0) ConfigVar->BoardConfigVar[i]->FPIOtype = 1;
					else if (strcmp(str, "NIM") == 0) ConfigVar->BoardConfigVar[i]->FPIOtype = 0;
					else { printf("%s = %s: invalid option\n", str1,str); return ERR_PARSE_CONFIG;}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		/*
		// External Trigger (DISABLED, ACQUISITION_ONLY, ACQUISITION_AND_TRGOUT)
		if (!strcmp(strcpy(str1, str), "EXTERNAL_TRIGGER")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "DISABLED") == 0)
						ConfigVar->BoardConfigVar[i]->ExtTriggerMode = CAEN_PADC_TRGMODE_DISABLED;
					else if (strcmp(str, "ACQUISITION_ONLY") == 0)
						ConfigVar->BoardConfigVar[i]->ExtTriggerMode = CAEN_PADC_TRGMODE_ACQ_ONLY;
					else if (strcmp(str, "ACQUISITION_AND_TRGOUT") == 0)
						ConfigVar->BoardConfigVar[i]->ExtTriggerMode = CAEN_PADC_TRGMODE_ACQ_AND_EXTOUT;
					else { printf("%s: Invalid Parameter\n", str); break; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		// Software Trigger (DISABLED, ACQUISITION_ONLY, TRGOUT_ONLY, ACQUISITION_AND_TRGOUT)
		if (!strcmp(strcpy(str1, str), "SW_TRIGGER")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "DISABLED") == 0)
						ConfigVar->BoardConfigVar[i]->SWTriggerMode = CAEN_PADC_TRGMODE_DISABLED;
					else if (strcmp(str, "ACQUISITION_ONLY") == 0)
						ConfigVar->BoardConfigVar[i]->SWTriggerMode = CAEN_PADC_TRGMODE_ACQ_ONLY;
					else if (strcmp(str, "TRGOUT_ONLY") == 0)
						ConfigVar->BoardConfigVar[i]->SWTriggerMode = CAEN_PADC_TRGMODE_EXTOUT_ONLY;
					else if (strcmp(str, "ACQUISITION_AND_TRGOUT") == 0)
						ConfigVar->BoardConfigVar[i]->SWTriggerMode = CAEN_PADC_TRGMODE_ACQ_AND_EXTOUT;
					else { printf("%s: Invalid Parameter\n", str); break; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "START_ACQ")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			for (i = 0; i < ConfigVar->Nhandle; i++) {
				if (i == board || board == -1) {
					if (strcmp(str, "SW") == 0) ConfigVar->BoardConfigVar[i]->StartMode = CAEN_PADC_SW_CONTROLLED;
					else if (strcmp(str, "S_IN") == 0) ConfigVar->BoardConfigVar[i]->StartMode = CAEN_PADC_S_IN_CONTROLLED;
					else if (strcmp(str, "FIRST_TRG") == 0) ConfigVar->BoardConfigVar[i]->StartMode = CAEN_PADC_FIRST_TRG_CONTROLLED;
					//else if (strcmp(str, "LVDS") == 0) ConfigVar->BoardConfigVar[i]->StartMode = CAEN_PS_LVDS_CONTROLLED;
					else { printf("%s: Invalid Parameter\n", str); break; }
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}
		*/
		if (!strcmp(strcpy(str1, str), "GATE_WIDTH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			fval = strtof(str, NULL);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) {
				if (fval < 0.) printf("Warning: gate width cannot be negative\n");
				else if (fval == 0.)  ConfigVar->BoardConfigVar[j]->TrigMode = CAEN_PADC_GATE;
				else {
					ConfigVar->BoardConfigVar[j]->TrigMode = CAEN_PADC_PULSE;
					ConfigVar->BoardConfigVar[j]->RecordLength = (uint32_t)(fval*1000/16);
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		/*if (!strcmp(strcpy(str1, str), "GAIN_FACTOR")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) ConfigVar->BoardConfigVar[j]->GainFactor = val;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}*/

		if (!strcmp(strcpy(str1, str), "ENABLE_INPUT")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val64 = (uint64_t)strtoull(str, NULL, 16);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) {
				for (i=0;i<8;i++) ConfigVar->BoardConfigVar[j]->EnableMask[i] = (val64 >> (8*i)) & 0xff;
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "ENABLE_GRAPH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val64 = (uint64_t)strtoull(str, NULL, 16);
			ConfigVar->TrackPlotted = (uint8_t)val64;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "GROUP_GRAPH")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) if (j == board || board == -1) ConfigVar->GroupPlotted = val;
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		if (!strcmp(strcpy(str1, str), "ZS_THRESHOLD")) {
			if ((str = strtok(NULL, " \r\t\n")) == NULL) { printf("No argument for %s. The command will be ignored\n", str1); continue; }
			val = (int)strtol(str, NULL, 10);
			for (j = 0; j < ConfigVar->Nhandle; j++) {
				if (j == board || board == -1) {
					for (i = 0; i < 2*MAX_FPGA_GROUPS; i++) {
						if (group == -1 || i == group) {
							ConfigVar->BoardConfigVar[j]->ZSThreshold[i] = val;
						}
					}
				}
			}
			if ((str = strtok(NULL, " \r\t\n")) != NULL) printf("WARNING: too many arguments in %s. the first exceeding argument is %s\n", str1, str);
			continue;
		}

		printf("%s: invalid setting at line %d\n", str, line);
		return_code = ERR_PARSE_CONFIG;
		break;
	}
	return return_code;
}

ERROR_CODES OpenRawFile(FILE **outfile, int BoardIndex, int FileIndex, char *path, char *fname) {
	ERROR_CODES return_code = ERR_NONE;
	char filename[400];
	struct stat info;
	if (stat(path, &info) != 0) {
		printf("path %s cannot be accessed. Please verify that the selected directory exists and is writable\n", path); return ERR_OUTFILE_OPEN;
	}
	if (*outfile != NULL) fclose(*outfile);
	sprintf(filename,"%s%s_raw_b%d_seg%d.bin",path,fname,BoardIndex,FileIndex);
	if ((*outfile = fopen(filename, "w")) == NULL) {
		printf("output file %s could not be created.\n",filename);
		return_code = ERR_OUTFILE_OPEN;
	}
	return return_code;
}

ERROR_CODES OpenListFile(FILE **outfile, int BoardIndex, int ChMax, int FileIndex, char *path, char *fname) {
	ERROR_CODES return_code = ERR_NONE;
	char filename[200];
	int channel;
	if (*outfile != NULL) fclose(*outfile);
	struct stat info;
	if (stat(path, &info) != 0) {
		printf("path %s cannot be accessed. Please verify that the selected directory exists and is writable\n", path); return ERR_OUTFILE_OPEN;
	}
	sprintf(filename, "%s%s_list_b%d_seg%d.txt", path, fname, BoardIndex, FileIndex);
	if ((*outfile = fopen(filename, "w")) == NULL) {
		printf("output file %s could not be created.\n",filename);
		return_code = ERR_OUTFILE_OPEN;
	}
	fprintf(*outfile, "%*s ", 14, "Trigger index");
	fprintf(*outfile, "%*s ", 20, "Timestamp");
	for (channel = 0; channel < ChMax; channel++) fprintf(*outfile, "%*s %*d", 7, "Ch.", 2, channel);
	fprintf(*outfile, "\n\n");
	return return_code;
}

ERROR_CODES OpenHistoFile(FILE ***outfile, int BoardIndex, PeakSensingBoardConfig_t *BoardConfigVar, char *path, char *fname) {
	ERROR_CODES return_code = ERR_NONE;
	char filename[400];
	int channel;
	struct stat info;
	if (stat(path, &info) != 0) {
		printf("path %s cannot be accessed. Please verify that the selected directory exists and is writable\n", path); return ERR_OUTFILE_OPEN;
	}
	for (channel = 0; channel < BoardConfigVar->ChMax; channel++) {
		if ((BoardConfigVar->EnableMask[(int)(channel / 8)] >> (int)(channel % 8)) & 0x1) {
			sprintf(filename, "%s%s_histo_b%d_ch%d.txt", path,fname,BoardIndex, channel);
			if ((*(*(outfile)+channel)) != NULL) fclose((*(*(outfile)+channel)));
			if ((*(*(outfile)+channel) = fopen(filename, "w")) == NULL) {
				printf("output file %s could not be created.\n", filename);
				return ERR_OUTFILE_OPEN;
			}
		}
		else *(*(outfile)+channel) = NULL;
	}
	return return_code;
}

void HistoWrite(FILE **HistoFile, Counter_t *Counter, PeakSensingBoardConfig_t *BoardConfigVar) {
	int Channel, Bin;
	for (Channel = 0; Channel < BoardConfigVar->ChMax; Channel++) {
		if ((BoardConfigVar->EnableMask[(int)(Channel / 8)] >> (int)(Channel % 8)) & 0x1) {
			for (Bin = 0; Bin < (int)pow(2., 14 - (double)BoardConfigVar->ChMode); Bin++) {
				fprintf(*(HistoFile+Channel), "%*d\t", 7, Bin);
				fprintf(*(HistoFile+Channel), "%*lu\n", 12, (uint32_t)Counter->Amp_Cnt[Channel][Bin]);
			}
		}		
	}
}

void ListWrite(FILE *ListFile, int NumEvents, PeakSensingBoardConfig_t *BoardConfigVar, CAEN_PADC_Event_t *Event) {
	int EvIndex;
	int Group;
	int Channel;
	int dataptr;
	for (EvIndex = 0; EvIndex<NumEvents; EvIndex++) {
		fprintf(ListFile, "%*u ", 14, Event[EvIndex].tcounter);
		fprintf(ListFile, "%*llu ", 20, Event[EvIndex].timeStamp);
		// aggiunta
		dataptr = 0;
		for (Channel = 0; Channel < BoardConfigVar->ChMax; Channel++) {
			if (Channel % 16 == 0) dataptr = 0;
			if ((BoardConfigVar->EnableMask[(int)(Channel / 8)] >> (int)(Channel % 8)) & 0x1) {
				if ((Event[EvIndex].PS_Group[(int)(Channel / 16)].ChMask >> (int)(Channel % 16)) & 0x1) {
					fprintf(ListFile, "%*d", 10, *(Event[EvIndex].PS_Group[(int)(Channel / 16)].DataPtr + dataptr));
					dataptr++;
				} else fprintf(ListFile, "%*s", 10, " ZS");
			}
			else fprintf(ListFile, "%*s", 10, "N/A");
		}
		// fine aggiunta
		/*for (Group = 0; Group < 4; Group++) {
			for (Channel = 0; Channel < 16; Channel++) {
				if ((Event[EvIndex].PS_Group[Group].ChMask >> Channel) & 0x1) fprintf(ListFile,"%*d", 10, *(Event[EvIndex].PS_Group[Group].DataPtr + Channel));
				else fprintf(ListFile, "%*s",10, "N/A");
			}
		}*/
		fprintf(ListFile, "\n");
	}
}

int MallocCounter(Counter_t *Counter) {
	int i;
	for (i = 0; i < 64; i++) if ((Counter->Amp_Cnt[i] = (double*)malloc(16384*sizeof(double))) == NULL) return 1;	
	return 0;
}



int ResetCounter(Counter_t *Counter) {
	int i;
	Counter->Pu_Ev = 0;
	Counter->NotRej_Ev = 0;
	Counter->Tot_Ev = 0;
	Counter->MB_Cnt = 0;
	Counter->MB_TS = 0;
	for (i = 0; i < 64; i++) {
		Counter->Tot_Cnt[i] = 0;
		Counter->Pu_Cnt[i] = 0;
		Counter->Of_Cnt[i] = 0;
		memset(Counter->Amp_Cnt[i],0., 16384 * sizeof(double));
	}
	return 0;
}



