#include "A2795Upgrade.h"
#include "CAENComm.h"
#include <string.h>
#include <math.h>
#include <fstream>

A2795_flash::A2795_flash(int handle, controller_t controller_offset) : handle(handle)
{
	uint32_t idcode;

	controller_base_address = USER_CONTROLLER_OFFSET;
	bitstream_length = USER_FIRMWARE_BITSTREAM_LENGTH;

	bitstream = new uint8_t[bitstream_length];

	// Se il controllore  accessibie
	// si deve poter rileggere un IDCODE univoco 
	CAENComm_Read32(handle, controller_base_address + IDCODE_OFFSET, &idcode);

	// Check controller ID code
	if (idcode != FLASH_CONTROLLER_IDCODE)
		_flash_controller_present = 0; // controller not present/mapped
	else
		_flash_controller_present = 1;

	// MUST enable flash access from controller!
	enable_flash_access(); // HACK giusto farlo nel costruttore????

	// Sblocca accesso al controllore flash
	CAENComm_Write32(handle, controller_base_address + UNLOCK_OFFSET, 0xABBA5511);


}


A2795_flash::~A2795_flash()
{
	// MUST disable flash access from controller!
	disable_flash_access(); // HACK giusto farlo nel distruttore?

}

int A2795_flash::get_flash_status(uint32_t * status)
{
	if (!_flash_controller_present) return CONTROLLER_NOT_PRESENT;

	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, READ_STATUS_OPCODE);
	CAENComm_Read32(handle, controller_base_address + OPCODE_OFFSET, status);

	*status >>= 8;

	return 0;
}


int A2795_flash::load_bitstream_from_file(char *filename) {

	ifstream bitstream_file(filename, ios::in | ios::binary);

	if (!(bitstream_file.read((char *)bitstream, bitstream_length))) { // HACK conversione uint8_t * => char *
		return -1; // TODO gestione errori
	}
	return 0;
}

int A2795_flash::get_controller_status(uint32_t * status)
{
	if (!_flash_controller_present) return CONTROLLER_NOT_PRESENT;

	CAENComm_Read32(handle, controller_base_address + OPCODE_OFFSET, status);

	return 0;
}

int A2795_flash::sector_erase(uint32_t start_address)
{
	if (!_flash_controller_present) return CONTROLLER_NOT_PRESENT;

	CAENComm_Write32(handle, controller_base_address + ADDRESS_OFFSET, start_address);
	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, WRITE_ENABLE_OPCODE);
	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, SECTOR_ERASE_OPCODE);

	wait_flash();

	return 0;
}


int A2795_flash::write_page(uint32_t start_address, uint8_t  *buf)
{
	CAENComm_ErrorCode errs[64];
	uint32_t addrs[64];
	uint32_t datas[64];

	if (!_flash_controller_present) return CONTROLLER_NOT_PRESENT;

	CAENComm_Write32(handle, controller_base_address + ADDRESS_OFFSET, start_address);
	CAENComm_Write32(handle, controller_base_address + PAYLOAD_OFFSET, PAGE_SIZE-1); // 256 bytes payload

	// HACK implementare con MultiWrite32/BLT
	//for (int i = 0; i < 64; ++i)
	//	CAENComm_Write32(handle, controller_base_address + BRAM_START_OFFSET + 4 * i, *((uint32_t *)buf + i));

	for (int i = 0; i < 64; ++i) {
		addrs[i] = controller_base_address + BRAM_START_OFFSET + 4 * i;
		datas[i] = *((uint32_t *)buf + i);
	}

	CAENComm_MultiWrite32(handle, addrs, 64, datas, errs);

	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, WRITE_ENABLE_OPCODE);

	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, WRITE_PAGE_OPCODE);

	wait_flash();

	return 0;
}


int A2795_flash::read_page(uint32_t start_address, uint8_t*  buf)
{

	CAENComm_ErrorCode errs[64];
	uint32_t addrs[64];
	uint32_t datas[64];

	if (!_flash_controller_present) return CONTROLLER_NOT_PRESENT;

	CAENComm_Write32(handle, controller_base_address + ADDRESS_OFFSET, start_address);
	CAENComm_Write32(handle, controller_base_address + PAYLOAD_OFFSET, PAGE_SIZE - 1); // 256 bytes payload


	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, READ_PAGE_OPCODE);

	wait_controller();

	// HACK implementare con MultiRead32/BLT
	//for (int i = 0; i < 64; ++i)
	//	CAENComm_Read32(handle, controller_base_address + BRAM_START_OFFSET + 4 * i, (uint32_t *)buf + i);

	for (int i = 0; i < 64; ++i) {
		addrs[i] = controller_base_address + BRAM_START_OFFSET + 4 * i;
	}

	CAENComm_MultiRead32(handle, addrs, 64, datas, errs);

	memcpy(buf, datas, 64 * sizeof(uint32_t));

	return 0;
}


int A2795_flash::set_flash_status(uint8_t status)
{
	if (!_flash_controller_present) return CONTROLLER_NOT_PRESENT;

	CAENComm_Write32(handle, controller_base_address + ADDRESS_OFFSET, (uint32_t)status);
	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, WRITE_STATUS_OPCODE);

	return 0;
}



int A2795_flash::wait_controller()
{
	uint32_t data;


	while (1) { // TODO potential deadlock!!! => timeout

		// Attende che il controllore flash sia pronto ad accettare un nuovo comando
		CAENComm_Read32(handle, controller_base_address + OPCODE_OFFSET, &data);
		if ((data & 0xFE) == 0) 
			break;
	}

	return 0;
}


int A2795_flash::wait_flash()
{
	uint32_t data;

	while (1) { //  TODO potential deadlock!!! => timeout

		// Attende che il controllore della flash abbia terminato una eventuale 
		// operazione in corso.
		wait_controller();

		// Rilegge  registro di status della flash per verificare che
		// abbia finito l'operazione di scrittura.
		CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, READ_STATUS_OPCODE);
		CAENComm_Read32(handle, controller_base_address + OPCODE_OFFSET, &data);
		if (((data >> 8) & 1) == 0)
			return 0;
	}

	return 0;
}

uint8_t  inline A2795_flash::rev_byte(uint8_t v) {
	uint8_t t = v;
	for (int i = sizeof(v) * 8 - 1; i; i--)
	{
		t <<= 1;
		v >>= 1;
		t |= v & 1;
	}
	return	t;
}


int A2795_flash::program_firmware(fw_region_t region, char *filename, int verify, int no_bit_reverse, int skip_erase) {

	uint8_t * buf;
	uint8_t * buf_ver;
	buf = new uint8_t[PAGE_SIZE];
	buf_ver = new uint8_t[PAGE_SIZE];
	int sectors_to_write;
	int bytes_to_write;

	uint32_t start_address;


	if (load_bitstream_from_file(filename))
		return -1; // TODO gestione errori

	if (!bitstream)
		return -1; // Gestione errori => bitstream non caricato

	switch (controller_base_address) {
	case USER_CONTROLLER_OFFSET:
		sectors_to_write = USER_FIRMWARE_SECTORS;

		switch (region) {
		case BOOT_FW_REGION:
			start_address = USER_FACTORY_START_ADDRESS;
			break;
		case APPLICATION1_FW_REGION:
			start_address = USER_APPLICATION1_START_ADDRESS;
			break;
		case APPLICATION2_FW_REGION:
			start_address = USER_APPLICATION2_START_ADDRESS;
			break;
		case APPLICATION3_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION4_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION5_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		default:
			// TODO
			break;
		}

		break;
	default:
		// TODO
		break;
	}

	// Se si deve aggiornare l'iimagine di boot bisogna
	// sproteggere i settori dedicati al firmware FACTORY (BOOT)
	if (region == BOOT_FW_REGION)
		write_unprotect();

	// Cancella i settori a partire da quello pi basso,
	// in modo da lasciare "corrotta" la flash in caso di interruzione prematura
	// della cancellazione.
	if (!skip_erase)
		// Erase sectors
		for (int i = 0; i < sectors_to_write; ++i) {
			sector_erase(start_address + i * SECTOR_SIZE);
		}

	// Programma le pagine di ciascun settore
	// Programma i settori a partire da quello pi alto
	// in modo da lasciare "corrotta" la flash in caso di interruzione prematura
	// della programmazione.
	for (int sector = sectors_to_write - 1; sector >= 0; --sector){
		for (int page = SECTOR_SIZE / PAGE_SIZE - 1; page >= 0; --page) {
			int offset = sector * SECTOR_SIZE + page * PAGE_SIZE;

			int remain = (bitstream_length - offset) > 0 ? bitstream_length - offset : 0;

			if (!remain) continue;

			bytes_to_write = (remain < PAGE_SIZE) ? remain : PAGE_SIZE;

			memset(buf, 0, PAGE_SIZE);

			// Get next data chunk in bitstream buffer
			memcpy(buf, bitstream + offset, bytes_to_write);

			if (!no_bit_reverse) {
				// Bit reversal
				for (int k = 0; k < bytes_to_write; ++k)
					buf[k] = rev_byte(buf[k]);
			}

			// Write buffer into flash page
			write_page(start_address + offset, buf);

			if (verify) {
				read_page(start_address + offset, buf_ver);
				for (int ii = 0; ii < bytes_to_write; ii++) {
					if (buf_ver[ii] != buf[ii]) return -1; // TODO error code per verify fallito
				}
			}
		}
	}

	// Nel caso di programmazione del boot
	// al termine si proteggono nuovamente i suoi settori
	if (region == BOOT_FW_REGION)
		write_protect();

	return 0;

}


int A2795_flash::verify_firmware(fw_region_t region, char *filename, int no_bit_reverse) {

	if (load_bitstream_from_file(filename))
		return -1; // TODO gestione errori

	uint8_t * buf;
	uint8_t * buf_ver;
	buf = new uint8_t[PAGE_SIZE];
	buf_ver = new uint8_t[PAGE_SIZE];

	int sectors_to_read;
	int bytes_to_read;

	uint32_t start_address;

	if (!bitstream)
		return -1; // Gestione errori => bitstream non caricato

	switch (controller_base_address) {
	case USER_CONTROLLER_OFFSET:
		sectors_to_read = USER_FIRMWARE_SECTORS;

		switch (region) {
		case BOOT_FW_REGION:
			start_address = USER_FACTORY_START_ADDRESS;
			break;
		case APPLICATION1_FW_REGION:
			start_address = USER_APPLICATION1_START_ADDRESS;
			break;
		case APPLICATION2_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION3_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION4_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION5_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		default:
			// TODO
			break;
		}

		break;
	default:
		// TODO
		break;
	}


	// Programma le pagine di ciascun settore
	// Programma i settori a partire da quello pi alto
	for (int sector = 0; sector < sectors_to_read; ++sector){
		for (int page = 0; page < SECTOR_SIZE / PAGE_SIZE; ++page) {
			int offset = sector * SECTOR_SIZE + page * PAGE_SIZE;

			int remain = (bitstream_length - offset) > 0 ? bitstream_length - offset : 0;

			if (!remain) continue;

			bytes_to_read = (remain < PAGE_SIZE) ? remain : PAGE_SIZE;

			// Point to next data chunk in bitstream buffer
			//buf = bitstream + i * 64 * 1024 + j * 256;
			// Get next data chunk in bitstream buffer
			memcpy(buf, bitstream + offset, bytes_to_read);


			if (!no_bit_reverse) {
				// Bit reversal
				for (int k = 0; k < bytes_to_read; ++k)
					buf[k] = rev_byte(buf[k]);
			}

			read_page(start_address + offset, buf_ver);
			for (int ii = 0; ii < bytes_to_read; ii++) {
				if (buf_ver[ii] != buf[ii])
					return -1; // TODO error code per verify fallito
			}

		}
	}


	return 0;

}

int A2795_flash::erase_firmware(fw_region_t region) {

	int sectors_to_erase;
	uint32_t start_address;

	switch (controller_base_address) {
	case USER_CONTROLLER_OFFSET:
		sectors_to_erase = USER_FIRMWARE_SECTORS;
		switch (region) {
		case BOOT_FW_REGION:
			start_address = USER_FACTORY_START_ADDRESS;
			break;
		case APPLICATION1_FW_REGION:
			start_address = USER_APPLICATION1_START_ADDRESS;
			break;
		case APPLICATION2_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION3_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION4_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION5_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		default:
			// TODO
			break;
		}


		break;
	default:
		// TODO
		break;
	}


	// Se si deve aggiornare l'iimagine di boot bisogna
	// sproteggere i settori dedicati al firmware FACTORY (BOOT)
	if (region == BOOT_FW_REGION)
		write_unprotect();

	// Erase sectors
	for (int i = 0; i < sectors_to_erase; ++i) {
		sector_erase(start_address + i * SECTOR_SIZE);
	}

	// Nel caso di programmazione del boot
	// al termine si proteggono nuovamente i suoi settori
	if (region == BOOT_FW_REGION)
		write_protect();

	return 0;

}

int A2795_flash::dump_firmware(fw_region_t region, char *filename, int no_bit_reverse) {

	// TODO

	uint32_t start_address;
	uint32_t sectors_to_dump;

	switch (controller_base_address) {
	case USER_CONTROLLER_OFFSET:
		sectors_to_dump = USER_FIRMWARE_SECTORS;

		switch (region) {
		case BOOT_FW_REGION:
			start_address = USER_FACTORY_START_ADDRESS;
			break;
		case APPLICATION1_FW_REGION:
			start_address = USER_APPLICATION1_START_ADDRESS;
			break;
		case APPLICATION2_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION3_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION4_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		case APPLICATION5_FW_REGION:
			return -1; // gestione errori/eccezioni => regione non ammessa per MAIN
			break;
		default:
			// TODO
			break;
		}

		break;

	default:
		// TODO
		break;
	}

	return 0;

}


int A2795_flash::write_protect() {

	uint32_t data;
	uint32_t region;


	switch (controller_base_address) {
	case USER_CONTROLLER_OFFSET:
		region = PROTECT_SECTORS_0_127 << 2;
		CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, WRITE_ENABLE_OPCODE);
		CAENComm_Write32(handle, controller_base_address + ADDRESS_OFFSET, region);
		CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, WRITE_STATUS_OPCODE);
		wait_flash();
		break;
	default:
		// TODO
		break;
	}

	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, READ_STATUS_OPCODE);
	CAENComm_Read32(handle, controller_base_address + OPCODE_OFFSET, &data);

	if (((data >> 8) & 0xFC) != region)
		return -1; // TODO gestione errori => scrittura errata in flash


	return 0;
}

int A2795_flash::write_unprotect() {

	uint32_t region;
	uint32_t data;


	region = UNPROTECT_ALL << 2;
	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, WRITE_ENABLE_OPCODE);
	CAENComm_Write32(handle, controller_base_address + ADDRESS_OFFSET, region);
	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, WRITE_STATUS_OPCODE);
	wait_flash();


	CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, READ_STATUS_OPCODE);
	CAENComm_Read32(handle, controller_base_address + OPCODE_OFFSET, &data);

	if (((data >> 8) & 0xFC) != region)
		return -1; // TODO gestione errori => scrittura errata in flash

	return 0;
}

int A2795_flash::get_protection_status(uint32_t& status) {
	uint32_t data;


	switch (controller_base_address) {
	case USER_CONTROLLER_OFFSET:
		CAENComm_Write32(handle, controller_base_address + OPCODE_OFFSET, READ_STATUS_OPCODE);
		CAENComm_Read32(handle, controller_base_address + OPCODE_OFFSET, &data);
		status = data >> 10;
		break;
	default:
		break;
	}

	return 0;

}

int A2795_flash::enable_flash_access() {
	// Sconfigura FPGA
	// Evita che la flash sia inaccessibile perch FPGA User non programmata o pin in conflitto
	CAENComm_Write32(handle, controller_base_address + FPGA_ACCESS_OFFSET, 0);

	// enable flash access (remove tristate)
	CAENComm_Write32(handle, controller_base_address + FLASH_ACCESS_OFFSET, 1);
	return 0;
}


int A2795_flash::disable_flash_access() {
	// disable flash access (tristate)
	CAENComm_Write32(handle, controller_base_address + FLASH_ACCESS_OFFSET, 0);

	// restart fpga 
	CAENComm_Write32(handle, controller_base_address + FPGA_ACCESS_OFFSET, 1);
	return 0;
}
