/***********************************************************************************************
* MODULE:     N957 Demo application
* PURPOSE:    
* WRITTEN BY: NDA
* COPYRIGHT:  CAEN S.p.A. all rights reserved
* USAGE:      
************************************************************************************************/
////////////////////////////////////////////
// File includes
////////////////////////////////////////////
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/timeb.h>
#ifdef WIN32
#include <time.h>
#include <conio.h>
#include <process.h>
#include <sys/timeb.h>
#define popen  _popen   // redefine 'deprecated' popen as _popen
#define pclose _pclose  // redefine 'deprecated' pclose as _pclose
#else
#include <unistd.h>
#include <sys/time.h>
#define Sleep(t)  usleep((t)*1000)
#include <memory.h>
#pragma GCC diagnostic ignored "-Wformat-security"
#endif
#include "common_defs.h"
#include "./include/N957Lib.h"
#include "user_settings.h"
#include "keyb.h"


////////////////////////////////////////////
// File local defines
////////////////////////////////////////////
#define N957_SAMPLE_NUM_BITS	13
#define N957_MAX_HISTO_SAMPLES	(1<<N957_SAMPLE_NUM_BITS)

// Executable gnuplot. NOTE: use pgnuplot instead of wgnuplot in Windows, otherwise
// the pipe will not work.
#ifdef WIN32
#define GNUPLOT_COMMAND  "pgnuplot"
#else
#define GNUPLOT_COMMAND  "gnuplot"
#endif

////////////////////////////////////////////
// File local variables declaration
////////////////////////////////////////////
const char* HISTO_FILENAME= "histo.dat";

////////////////////////////////////////////
// Global visible variables declaration
////////////////////////////////////////////

////////////////////////////////////////////
// File local methods declaration
////////////////////////////////////////////
// ---------------------------------------------------------------------------
// get time in milliseconds since first procedure call
// ---------------------------------------------------------------------------
long get_time()
{
	static long startup= 0;
	long time_ms;

#ifdef WIN32
	struct _timeb timebuffer;

	_ftime( &timebuffer );
	if( !startup) {
		startup= (long)timebuffer.time;
	}
	time_ms = (long)( timebuffer.time- startup) * 1000 + (long)timebuffer.millitm;
#else
	struct timeval t1;
	struct timezone tz;

	gettimeofday(&t1, &tz);
	if( !startup) {
		startup= (long)t1.tv_sec;
	}
	time_ms = (t1.tv_sec- startup) * 1000 + t1.tv_usec / 1000;
#endif

	return time_ms;
}

/**************************************************
**************************************************/

/***********************************************************************************************
* METHOD:     main
* PURPOSE:    Main program
* PARAMETERS: <in> int argc: number of arguments
*             <in> void *argv[]: arguments' list 
* RETURN:     
* USAGE:      allowed command line input parameters:
************************************************************************************************/
int main(int argc, char **argv) 

{

	N957_UINT32 data32;
	int ret_val= 0;									// procedure exit value
	N957ErrorCodes N957_error_code;					// returned error code
	user_setting_data user_setting;					// user settings
	FILE* out_file= NULL;							// output file
	N957_UINT16 *data_buff= NULL;					// read data buffer
	unsigned long *histo_buff= NULL;				// Histogram data buffer
	int32_t tot_data_read= 0;
	int i;
	FILE* gnu_plot_pipe= NULL;						// gnu_plot pipe
	long next_refresh= 0;
	char aux_string[ 100];
	BOOL paused= FALSE;
	BOOL do_exit= FALSE;
	struct timeb start_time, curr_time;
	int diff;


	/////////////////////////////////////////
	// Demo application specific
	/////////////////////////////////////////

	memset( &user_setting, 0, sizeof( user_setting));

	//
	// print header
	printf( "\n");
	printf( "-------------------------------------------------------\n");
	printf( "-                   N 9 5 7   D E M O                 -\n");
	printf( "-                                                     -\n");
	printf( "-   www.caen.it                             rev.02.11 -\n");
	printf( "-------------------------------------------------------\n");
	printf( "\n");

	//
	// init user setting module
	if( !user_settings_open( &user_setting))
	{
		ret_val= -1;
		goto exit_point;
	}

	//
	// input parameter check
	if( !user_settings_parse_input_param( &user_setting, argc, (char**)argv))
	{
		ret_val= -2;
		goto exit_point;
	}

	/////////////////////////////////////////
	// Library specific
	/////////////////////////////////////////


	//
	// now board handle is valid and we can start calling boards API
	{
		// Get firmware revision
		char fw_rev[10];
		TRACE( "\nN957 FW Rev. ");
		if( ( N957_error_code= N957_GetFWRelease( user_setting.m_N957_handle, fw_rev, 10))!= N957Success)
		{
			TRACE( N957_DecodeError( N957_error_code));
			TRACE( "\n");
			ret_val= -3;
			goto exit_point;
		}
		TRACE( fw_rev);
	}


	TRACE( "\n");
	TRACE( "\nHit 's' to start acquisition mode , any other to quit ...\n");
	switch( getch()) {
		case 's':
		case 'S':
			break;
		default:
			ret_val= -4;
			goto exit_point;
	}

	// Scaler Reading
	TRACE("\nADC Conversions Number : ");
	if( ( N957_error_code=N957_GetScaler(user_setting.m_N957_handle, &data32)!= N957Success))
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -5;
		goto exit_point;
	}	
	TRACE1("%d", data32);

	// Timer Reading
	TRACE("\nTime (ms) : ");
	if( ( N957_error_code=N957_GetTimer(user_setting.m_N957_handle, &data32)!= N957Success))
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -6;
		goto exit_point;
	}	
	TRACE1("%d", data32);

	// Live Time Reading
	TRACE("\nLive Time (ms) : ");
	if( ( N957_error_code=N957_GetLiveTime(user_setting.m_N957_handle, &data32)!= N957Success))
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -7;
		goto exit_point;
	}	
	TRACE1("%d",data32);

	// Start acquisition with specific mode
	TRACE1( "\nN957 Start Acquisition mode %i : ", user_setting.m_mode);
	if( ( N957_error_code= N957_StartAcquire( user_setting.m_N957_handle, user_setting.m_mode))!= N957Success)
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -8;
		goto exit_point;
	}
	TRACE( "Ok");

	if( user_setting.m_debug) {
		//
		// If debugging features enable, convertion well be simulated by software
		//
		// Set software convertion
		TRACE( "\nN957 Debug Software convertion enable : ");
		if( ( N957_error_code= N957_SetSwConvFlag( user_setting.m_N957_handle, N957_TRUE))!= N957Success)
		{
			TRACE( N957_DecodeError( N957_error_code));
			TRACE( "\n");
			ret_val= -9;
			goto exit_point;
		}
		TRACE( "Ok");
	}

	// Allocate buffer storage
	data_buff= malloc( user_setting.m_bldim* sizeof( N957_UINT16));
	if( data_buff== NULL)
	{
		// Insufficient memory
		TRACE1( "N957 Insufficient memory to allocate data buffer (%li bytes)", (long)( user_setting.m_bldim* sizeof( N957_UINT16)));
		ret_val= -10;
		goto exit_point;
	}
	// Allocate histogram buffer storage
	histo_buff= malloc( N957_MAX_HISTO_SAMPLES* sizeof( *histo_buff));
	if( histo_buff== NULL)
	{
		// Insufficient memory
		TRACE1( "N957 Insufficient memory to allocate histogram data buffer (%li bytes)", (long)N957_MAX_HISTO_SAMPLES* sizeof( *histo_buff));
		ret_val= -11;
		goto exit_point;
	}
	memset( histo_buff, 0, N957_MAX_HISTO_SAMPLES* sizeof( *histo_buff));

	if( user_setting.m_log_to_file) {
		// Create output file
		if( ( out_file= fopen( user_setting.m_log_filename, "wt"))== NULL)
		{
			TRACE1( "N957 output file creation failure '%s'", user_setting.m_log_filename);
			ret_val= -12;
			goto exit_point;
		}
	}

	// open gnuplot for the first time
	{
		char *gnu_plot_filename= (char*)malloc( MAX_FILENAME_LENGHT);
		sprintf( gnu_plot_filename, "%s%s", user_setting.m_gnu_plot_path, GNUPLOT_COMMAND);
		// open the pipe
		if(( gnu_plot_pipe = popen( gnu_plot_filename, "w"))== NULL){
			TRACE( "N957 GNU Plot pipe creation failure");
			ret_val= -13;
			goto exit_point;
		}
		//// set some variables in gnuplot
		//fprintf( gnu_plot_pipe, "ch = %d; mod = %d; unit = %d; sens = %f\n", ChSel, ModSel, Unit, Sensitivity);
		free( gnu_plot_filename);
	}

	// gnu plot commands
	strcpy( aux_string, "load 'plot_histo.cfg'\n");
	if( fwrite( aux_string, 1, strlen( aux_string), gnu_plot_pipe)!= strlen( aux_string)) {
		// out file write error
		TRACE( "\n Pipe write failure\n");
		ret_val= -14;
		goto exit_point;
	}

	TRACE( "\nAcquisition running ...\n");
	if (user_setting.m_runtime == 0) {
		TRACE( "\nAcquisition duration -> Manual\n");
	}
	else {
		printf( "\nAcquisition duration -> %d sec.\n",user_setting.m_runtime);
	}
	TRACE( "\nHit 'r' to reset histogram, any other to quit ...");
	TRACE( "\nHit 'p' to pause/resume, any other to quit ...");
	TRACE( "\nAny other to quit ...");
	// 
	// Main loop
	do_exit= FALSE;
	ftime(&start_time); // get the start time in millisecond
	while( !do_exit) {
		N957_UINT16 data_read;
		// Check for user commands
		while( paused){
			if( kbhit()) {
				switch( getch()) {
	case 'r':
	case 'R':
		// reset the histogram here
		memset( histo_buff, 0, N957_MAX_HISTO_SAMPLES* sizeof( *histo_buff));
		break;
	case 'p':
	case 'P':
		paused= !paused;
		break;
	default:
		ret_val= 0;
		goto exit_point;
				}
			}
			Sleep( 1);
		}

		data_read= user_setting.m_bldim;
		if( user_setting.m_max_num_samples> 0){
			// Check for maximun numsber of samples
			if( tot_data_read>= user_setting.m_max_num_samples) {
				ret_val= 0;
				goto exit_point;
			}

			if( user_setting.m_max_num_samples- tot_data_read< user_setting.m_bldim) {
				data_read= ( N957_UINT16)(int32_t)(user_setting.m_max_num_samples - tot_data_read);
			}
		}
		//
		// ReadData
		if( ( N957_error_code= N957_ReadData( user_setting.m_N957_handle, data_buff, &data_read))!= N957Success) {
			TRACE( N957_DecodeError( N957_error_code));
			TRACE( "\n");
			ret_val= -15;
			goto exit_point;
		}
		tot_data_read+= data_read;

		for( i= 0; i< data_read; i++) {
			// Check for user commands
			if( kbhit()) {
				switch( getch()) {
	case 'r':
	case 'R':
		// reset the histogram here
		memset( histo_buff, 0, N957_MAX_HISTO_SAMPLES* sizeof( *histo_buff));
		break;
	case 'p':
	case 'P':
		paused= !paused;
		break;
	default:
		do_exit= TRUE;
		break;
				}
			}

			// Calculate the histogram
			++histo_buff[ data_buff[ i]& ( N957_MAX_HISTO_SAMPLES- 1)];
			// Log to file (if enabled)
			if( out_file) {
				sprintf( aux_string, "%04x\n", data_buff[ i]);
				if( fwrite( aux_string, 1, strlen( aux_string), out_file)!= strlen( aux_string)) {
					// out file write error
					TRACE( "\n output file write failure\n");
					ret_val= -16;
					goto exit_point;
				}
			}
		}

		if( get_time()>= next_refresh) {
			FILE* histo_file= NULL;
			next_refresh= get_time()+ user_setting.m_gnu_plot_refresh;

			if(( histo_file= fopen( HISTO_FILENAME, "wt"))== NULL){
				// out file write error
				TRACE1( "N957 output file creation failure '%s'", HISTO_FILENAME);
				ret_val= -17;
				goto exit_point;
			}

			// Save data to histogram file
			for( i= 0; i< N957_MAX_HISTO_SAMPLES; i++) {
				// Check for user commands
				if( kbhit()) {
					switch( getch()) {
	case 'r':
	case 'R':
		// reset the histogram here
		memset( histo_buff, 0, N957_MAX_HISTO_SAMPLES* sizeof( *histo_buff));
		break;
	case 'p':
	case 'P':
		paused= !paused;
		break;
	default:
		do_exit= TRUE;
		break;
					}
				}

				sprintf( aux_string, "%ld\n", histo_buff[ i]);
				//sprintf( aux_string, "%f\t%ld\n", (float)i* user_setting.m_gnu_plot_x_scale, histo_buff[ i]);
				if( fwrite( aux_string, 1, strlen( aux_string), histo_file)!= strlen( aux_string)){
					// out file write error
					TRACE( "\n Histogram file write failure\n");
					ret_val= -18;
					goto exit_point;
				}
			}
			fclose( histo_file);

			// write to gnuplot
			strcpy( aux_string, "plot 'histo.dat' with histogram \n");
			// strcpy( aux_string, "set style histogram rows\n plot 'histo.dat' using 2, '' using 4, '' using 6:xticlabels(1)\n");
			if( fwrite( aux_string, 1, strlen( aux_string), gnu_plot_pipe)!= strlen( aux_string)) 
			{
				// out file write error
				TRACE( "\n Pipe write failure\n");
				ret_val= -19;
				goto exit_point;
			}			
			fflush( gnu_plot_pipe);
			Sleep( 1);
		}
		ftime(&curr_time);
		diff = (int) (1000.0 * (curr_time.time - start_time.time) + (curr_time.millitm - start_time.millitm));
		if ((diff > (user_setting.m_runtime * 1000)) && (user_setting.m_runtime != 0)){
			break;
		}
	}

	if( user_setting.m_debug)
	{
		//
		// clear software convertion
		TRACE( "\nN957 Debug Software convertion disable : ");
		if( ( N957_error_code= N957_SetSwConvFlag( user_setting.m_N957_handle, N957_FALSE))!= N957Success)
		{
			TRACE( N957_DecodeError( N957_error_code));
			TRACE( "\n");
			ret_val= -20;
			goto exit_point;
		}
		TRACE( "Ok");
	}

	//Scaler Reading
	TRACE("\nADC Conversions Number : ");
	if( ( N957_error_code=N957_GetScaler(user_setting.m_N957_handle, &data32)!= N957Success))
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -21;
		goto exit_point;
	}	
	TRACE1("%d", data32);

	//Timer Reading
	TRACE("\nTime (ms) : ");
	if( ( N957_error_code=N957_GetTimer(user_setting.m_N957_handle, &data32)!= N957Success))
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -22;
		goto exit_point;
	}	
	TRACE1("%d", data32);

	//Live Time Reading
	TRACE("\nLive Time (ms) : ");
	if( ( N957_error_code=N957_GetLiveTime(user_setting.m_N957_handle, &data32)!= N957Success))
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -23;
		goto exit_point;
	}	
	TRACE1("%d",data32);

exit_point:	
	/////////////////////////////////////////
	// Library specific
	/////////////////////////////////////////

	/////////////////////////////////////////
	// Demo application specific
	/////////////////////////////////////////

	TRACE( "\n\nHit any key to exit...");
	getch();


	// Stop acquisition
	TRACE( "\nN957 Stop Acquisition : ");
	if( ( N957_error_code= N957_StopAcquire( user_setting.m_N957_handle))!= N957Success)
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -24;
	}
	TRACE( "Ok");

	if( out_file!= NULL)
	{
		fclose( out_file);
	}
	if( gnu_plot_pipe!= NULL)
	{
		fprintf( gnu_plot_pipe, "exit\n");
		pclose( gnu_plot_pipe); 
	}
	if( data_buff!= NULL)
	{
		free( data_buff);
	}

	// SW Clear
	TRACE( "\nSW Clear : ");
	if( ( N957_error_code= N957_SwClear(user_setting.m_N957_handle)!= N957Success))
	{
		TRACE( N957_DecodeError( N957_error_code));
		TRACE( "\n");
		ret_val= -25;
	}	
	TRACE( "Ok");

	// close modules
	user_settings_close( &user_setting);

	return ret_val;
}

