////////////////////////////////////////////////////////////////////////////////////////////////
/*! \file    N1568.h
*   \brief   N1568 board definitions
*   \author  NDA
*   \version 1.0
*   \date    09/2008
*            
*            Provides methods, properties and defines to handle N1568 boards
*/
////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef __N1568_DEF_H
#define __N1568_DEF_H

////////////////////////////////////////////////////////////////////////////////////////////////
// File includes
////////////////////////////////////////////////////////////////////////////////////////////////
#include "../common/common_defs.h"
#include "../common/board_commons.h"

////////////////////////////////////////////////////////////////////////////////////////////////
// Global defines
////////////////////////////////////////////////////////////////////////////////////////////////

#define ASW_BUFF_SIZE	2048	/* !< The N1568_data::m_p_asw_buff size */

typedef enum {
	N1568A,
	N1568B,
} N1568_BOARD_TYPE;

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \struct  N1568_data
*   \brief   N1568 board data structure
*            
*            This structure stores any board data, common and specific ones. 
*            It must be passed to each N1568's API 
*/
////////////////////////////////////////////////////////////////////////////////////////////////
typedef struct
{
	board_data	m_common_data;					/*!< Boards common data structure: always let at offset 0 */
	//
	// Board specific data
	char* m_p_asw_buff;
	int m_asw_buff_len;
	char* m_p_tmp_asw_buff;
	N1568_BOARD_TYPE m_board_type;
} N1568_data;

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \struct  N1568_channel_status
*   \brief   N1568 channel status structure
*            
*            This structure stores the relevant channel's status parameters
*/
////////////////////////////////////////////////////////////////////////////////////////////////
typedef struct
{
	int m_channel_id;						/*!< \brief The channel index */
	BOOL m_is_polarity_negative;			/*!< \brief Input polarity */
	UINT8 m_shape_setting;					/*!< \brief Shape setting */
	UINT8 m_pole_zero_adj;					/*!< \brief Pole Zero Adjust */
	UINT8 m_fine_gain;						/*!< \brief Fine Gain setting */
	UINT8 m_coarse_gain;					/*!< \brief Coarse Gain setting */
	UINT8 m_discr_thr;						/*!< \brief Discriminator threshold */
	UINT8 m_gain_timing;					/*!< \brief Gain of Timing section */
	union {
		BOOL m_is_strecher_enabled;			/*!< \brief Stretcher enable status */
		BOOL m_is_output_polarity_negative;	/*!< \brief Input polarity */
	};
	UINT8 m_cfd_out_width;					/*!< \brief CFD30% Out Width */
} N1568_channel_status;

////////////////////////////////////////////////////////////////////////////////////////////////
// Commands index
////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////
/*! \enum    _N1568A_COMMANDS
*   \brief   The board commands
*            
*            Provides an entry for supported command: keep synched with \ref cmd_table N1568_CMD_TABLE[ ]
*/
////////////////////////////////////////////////////////////////////////////////////////////////
typedef enum _N1568A_COMMANDS {
	N1568A_CMD_MOD= 0,						/*!< \brief Set module command */
	N1568A_CMD_CH,							/*!< \brief Set channel command */
	N1568A_CMD_CF_TH,						/*!< \brief Discriminator threshold command */
	N1568A_CMD_CFW,							/*!< \brief CFD30% Out Width command */
	N1568A_CMD_PZ,							/*!< \brief Pole Zero Adjust command */
	N1568A_CMD_FG,							/*!< \brief Fine Gain setting command */
	N1568A_CMD_CG,							/*!< \brief Coarse Gain setting command */
	N1568A_CMD_SH,							/*!< \brief Shape setting command */
	N1568A_CMD_CGT,							/*!< \brief Gain of Timing section command */
	N1568A_CMD_IN,							/*!< \brief Input polarity setting command */
	N1568A_CMD_OUT,							/*!< \brief Output polarity setting command */
	N1568A_CMD_EXP,							/*!< \brief Export command */
} N1568A_COMMANDS;

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \enum    _N1568B_COMMANDS
*   \brief   The board commands
*            
*            Provides an entry for supported command: keep synched with \ref cmd_table N1568_CMD_TABLE[ ]
*/
////////////////////////////////////////////////////////////////////////////////////////////////
typedef enum _N1568B_COMMANDS {
	N1568B_CMD_MOD= 0,						/*!< \brief Set module command */
	N1568B_CMD_CH,							/*!< \brief Set channel command */
	N1568B_CMD_CF_TH,						/*!< \brief Discriminator threshold command */
	N1568B_CMD_CFW,							/*!< \brief CFD30% Out Width command */
	N1568B_CMD_PZ,							/*!< \brief Pole Zero Adjust command */
	N1568B_CMD_FG,							/*!< \brief Fine Gain setting command */
	N1568B_CMD_CG,							/*!< \brief Coarse Gain setting command */
	N1568B_CMD_SH,							/*!< \brief Shape setting command */
	N1568B_CMD_CGT,							/*!< \brief Gain of Timing section command */
	N1568B_CMD_IN,							/*!< \brief Input polarity setting command */
	N1568B_CMD_STR,							/*!< \brief Stretcher enable command */
	N1568B_CMD_EXP,							/*!< \brief Export command */
} N1568B_COMMANDS;

////////////////////////////////////////////////////////////////////////////////////////////////
// Global variables declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// Global methods declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     B O A R D S   H A N D L I N G
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_open( N1568_data* p_data, const char *com_str, N1568_BOARD_TYPE type);
*   \brief   N1568 boards data initialization.
*            
*            Provides specific handling for N1568 boards opening.
*   \param   p_data Pointer to board data
*   \param   com_str The serial communication device string to open 
 *                    e.g COM1.. COM# for Windows or /dev/ttyS0.. /dev/ttyS# or /dev/ttyUSB0.. /dev/ttyUSB# for Linux
*   \param   type The board type
*   \return  TRUE: board successfully opened
*   \note    Must be called before any other board specific API.
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_open( N1568_data* p_data, const char *com_str, N1568_BOARD_TYPE type);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_close( N1568_data* p_data);
*   \brief   N1568 boards closing and resource free
*            
*            Provides specific handling for N1568 boards closing.
*   \param   p_data Pointer to board data
*   \return  TRUE: board successfully closed
*   \note    Must be called when done with any other board specific API.
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_close( N1568_data* p_data);

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   0   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      const char* N1568_get_last_answer( N1568_data* p_data);
*   \brief   Get the answer to the last successfull sent command
*            
*   \param   p_data Pointer to board data
*   \return  The answer buffer
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API const char* N1568_get_last_answer( N1568_data* p_data);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568A_send_cmd( N1568_data* p_data, N1568A_COMMANDS cmd, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Sends the command to the current channel and gets the channel status
*            
*   \param   p_data Pointer to board data
*   \param   cmd The command id
*   \param   value The command value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568A_send_cmd( N1568_data* p_data, N1568A_COMMANDS cmd, UINT8 value, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568B_send_cmd( N1568_data* p_data, N1568B_COMMANDS cmd, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Sends the command to the current channel and gets the channel status
*            
*   \param   p_data Pointer to board data
*   \param   cmd The command id
*   \param   value The command value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568B_send_cmd( N1568_data* p_data, N1568B_COMMANDS cmd, UINT8 value, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   1   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_module( N1568_data* p_data, UINT8 value);
*   \brief   Set the current module
*            
*            Any other command will direct to this module
*   \param   p_data Pointer to board data
*   \param   value The module id: valid values are [0..16]. 16 is the broadcast module.
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_module( N1568_data* p_data, UINT8 value);
	
////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_channel( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Set the current channel id
*            
*            Any other command will direct to this channel
*   \param   p_data Pointer to board data
*   \param   value The channel id: valid values are [0..15]
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_channel( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
	
////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_input_polarity( N1568_data* p_data, BOOL negative_polarity, N1568_channel_status* channel_status);
*   \brief   Set the input polarity for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   negative_polarity The desired input polarity: Set to TRUE for negative input polarity
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_input_polarity( N1568_data* p_data, BOOL negative_polarity, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_shape( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Set the shaping time for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   value The desired shaping time
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_shape( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_pole_zero_adj( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Set the pole zero adjust value for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   value The desired pole zero adjust value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_pole_zero_adj( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_fine_gain( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Set the fine gain value for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   value The desired fine gain value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_fine_gain( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_coarse_gain( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Set the coarse gain value for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   value The desired coarse gain value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_coarse_gain( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_discr_thr( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Set the discriminator threshold value for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   value The desired discriminator threshold value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_discr_thr( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_gain_of_timing( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Set the gain of the timing section value for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   value The desired gain of the timing section value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_gain_of_timing( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
	
////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_stretcher( N1568_data* p_data, BOOL enable, N1568_channel_status* channel_status);
*   \brief   Enable/Disable the stretcher for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   enable Set to TRUE to enable the stretcher
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_stretcher( N1568_data* p_data, BOOL enable, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_cfd_out_width( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Set the CFD30% Out Width value for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   value The desired CFD30% Out Width value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_cfd_out_width( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_export( N1568_data* p_data);
*   \brief   This command allows to copy settings from Ch0 to all other channels
*            
*   \param   p_data Pointer to board data
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_export( N1568_data* p_data);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_set_output_polarity( N1568_data* p_data, BOOL negative_polarity, N1568_channel_status* channel_status);
*   \brief   Set the output polarity for the current channel
*            
*   \param   p_data Pointer to board data
*   \param   negative_polarity The desired output polarity: Set to TRUE for negative output polarity
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568_set_output_polarity( N1568_data* p_data, BOOL negative_polarity, N1568_channel_status* channel_status);

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   2   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     M I S C E L L A N E O U S   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      const char* N1568_SW_rev( void);
*   \brief   Returns the SDK software release
*            
*   \return  const char*: The SDK revision string
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API const char* N1568_SW_rev( void);

#endif  // __CLT_N1568_DEF_H
