////////////////////////////////////////////////////////////////////////////////////////////////
/*! \file    N1568.c
*   \brief   N1568 NIM board implementation
*   \author  NDA
*   \version 1.0
*   \date    09/2008
*            
*            Provides methods, properties and defines to handle N1568 NIM boards
*/
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// File includes
////////////////////////////////////////////////////////////////////////////////////////////////
#include <string.h>
#include <memory.h>
#include <stdlib.h>
#include "N1568.h"

////////////////////////////////////////////////////////////////////////////////////////////////
// File local defines
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// Static variables declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \var     static const cmd_table N1568A_CMD_TABLE[]
*   \brief   The board commands table
*            
*            Provides an entry for each supported command: keep synched with \ref N1568_COMMANDS
*/
////////////////////////////////////////////////////////////////////////////////////////////////
static const cmd_table N1568A_CMD_TABLE[ ]=
{
	{ "M",		0,		16},			/*!< \brief N1568_CMD_MOD: Set module command */
	{ "N",		0,		15},			/*!< \brief N1568_CMD_CH: Set channel command */
	{ "C",		0,		255},			/*!< \brief N1568_CMD_CF_TH: Discriminator threshold command */
	{ "W",		0,		255},			/*!< \brief N1568_CMD_CFW: CFD30% Out Width command */
	{ "P",		0,		255},			/*!< \brief N1568_CMD_PZ: Pole Zero Adjust command */
	{ "g",		0,		191},			/*!< \brief N1568_CMD_FG: Fine Gain setting command */
	{ "G",		0,		3},				/*!< \brief N1568_CMD_CG: Coarse Gain setting command */
	{ "S",		0,		3},				/*!< \brief N1568_CMD_SH: Shape setting command */
	{ "T",		0,		3},				/*!< \brief N1568_CMD_CGT: Gain of Timing section command */
	{ "I",		0,		1},				/*!< \brief N1568_CMD_IN: Input polarity setting command */
	{ "O",		0,		1},				/*!< \brief N1568_CMD_OUT: Output polarity setting command */
	{ "X",		1,		1},				/*!< \brief N1568_CMD_EXP: Export command */	
};

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \var     static const cmd_table N1568B_CMD_TABLE[]
*   \brief   The board commands table
*            
*            Provides an entry for each supported command: keep synched with \ref N1568_COMMANDS
*/
////////////////////////////////////////////////////////////////////////////////////////////////
static const cmd_table N1568B_CMD_TABLE[ ]=
{
	{ "M",		0,		16},			/*!< \brief N1568_CMD_MOD: Set module command */
	{ "N",		0,		15},			/*!< \brief N1568_CMD_CH: Set channel command */
	{ "C",		0,		255},			/*!< \brief N1568_CMD_CF_TH: Discriminator threshold command */
	{ "W",		0,		255},			/*!< \brief N1568_CMD_CFW: CFD30% Out Width command */
	{ "P",		0,		255},			/*!< \brief N1568_CMD_PZ: Pole Zero Adjust command */
	{ "g",		0,		127},			/*!< \brief N1568_CMD_FG: Fine Gain setting command */
	{ "G",		0,		3},				/*!< \brief N1568_CMD_CG: Coarse Gain setting command */
	{ "S",		0,		3},				/*!< \brief N1568_CMD_SH: Shape setting command */
	{ "T",		0,		3},				/*!< \brief N1568_CMD_CGT: Gain of Timing section command */
	{ "I",		0,		1},				/*!< \brief N1568_CMD_IN: Input polarity setting command */
	{ "E",		0,		1},				/*!< \brief N1568_CMD_STR: Stretcher enable command */
	{ "X",		1,		1},				/*!< \brief N1568_CMD_EXP: Export command */	
};

////////////////////////////////////////////////////////////////////////////////////////////////
// Static methods declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL parse_channel_status_A( N1568_data* p_data, N1568_channel_status* channel_status)
*   \brief   Parses the answer buffer and tries to retrive the channel status
*            
*   \param   p_data Pointer to the board data
*   \param   last_cmd_id The last command string sent to the board 
*   \param   channel_status The parsed channel status
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
static BOOL parse_channel_status_A( N1568_data* p_data, N1568_channel_status* channel_status)
{
	const char* ASW_FORMAT_STRING= "CH%2d CFth=%3d CFWdt=%3d P/Z=%3d FG=%3d SH=%1d CG=%1d CGT=%1d %1sIn Out%1s";
	int CH, CFth, CFWdt, P_Z, FG, SH, CG, CGT;
	char In[ 20], Out[ 20];
	char *buff= ( char *)N1568_get_last_answer( p_data);
	// Strip away command echo (if any)
	if( !strncmp( buff, p_data->m_common_data.m_p_last_sent_cmd, strlen( p_data->m_common_data.m_p_last_sent_cmd)))
	{
		buff+= strlen( p_data->m_common_data.m_p_last_sent_cmd);
	}
	if( *buff== '\n')
		++buff;
	if( !strnicmp( buff, "Error", sizeof( "Error")))
	{
		return FALSE;
	}
	if( sscanf( buff, ASW_FORMAT_STRING, &CH, &CFth, &CFWdt, &P_Z, &FG, &SH, &CG, &CGT, In, Out)!= 10)
	{
		return FALSE;
	}

	channel_status->m_channel_id= CH;
	channel_status->m_cfd_out_width= CFWdt;
	channel_status->m_coarse_gain= CG;
	channel_status->m_discr_thr= CFth;
	channel_status->m_fine_gain= FG;
	channel_status->m_gain_timing= CGT;
	channel_status->m_is_polarity_negative= !strnicmp( In, "N", 1);
	channel_status->m_is_output_polarity_negative= !strnicmp( Out, "N", 1);
	channel_status->m_pole_zero_adj= P_Z;
	channel_status->m_shape_setting= SH;
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL parse_channel_status_A( N1568_data* p_data, N1568_channel_status* channel_status)
*   \brief   Parses the answer buffer and tries to retrive the channel status
*            
*   \param   p_data Pointer to the board data
*   \param   last_cmd_id The last command string sent to the board 
*   \param   channel_status The parsed channel status
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
static BOOL parse_channel_status_B( N1568_data* p_data, N1568_channel_status* channel_status)
{
	const char* ASW_FORMAT_STRING= "CH%2d CFth=%3d CFWdt=%3d P/Z=%3d FG=%3d SH=%1d CG=%1d CGT=%1d %1sIn STR%3s";
	int CH, CFth, CFWdt, P_Z, FG, SH, CG, CGT;
	char In[ 20], STR[ 20];
	char *buff= ( char *)N1568_get_last_answer( p_data);
	// Strip away command echo (if any)
	if( !strncmp( buff, p_data->m_common_data.m_p_last_sent_cmd, strlen( p_data->m_common_data.m_p_last_sent_cmd)))
	{
		buff+= strlen( p_data->m_common_data.m_p_last_sent_cmd);
	}
	if( *buff== '\n')
		++buff;
	if( !strnicmp( buff, "Error", sizeof( "Error")))
	{
		return FALSE;
	}
	if( sscanf( buff, ASW_FORMAT_STRING, &CH, &CFth, &CFWdt, &P_Z, &FG, &SH, &CG, &CGT, In, STR)!= 10)
	{
		return FALSE;
	}

	channel_status->m_channel_id= CH;
	channel_status->m_cfd_out_width= CFWdt;
	channel_status->m_coarse_gain= CG;
	channel_status->m_discr_thr= CFth;
	channel_status->m_fine_gain= FG;
	channel_status->m_gain_timing= CGT;
	channel_status->m_is_polarity_negative= !strnicmp( In, "N", 1);
	channel_status->m_is_strecher_enabled= !strnicmp( STR, "en", 2);
	channel_status->m_pole_zero_adj= P_Z;
	channel_status->m_shape_setting= SH;
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL parse_channel_status( N1568_data* p_data, N1568_channel_status* channel_status)
*   \brief   Parses the answer buffer and tries to retrive the channel status
*            
*   \param   p_data Pointer to the board data
*   \param   last_cmd_id The last command string sent to the board 
*   \param   channel_status The parsed channel status
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
static BOOL parse_channel_status( N1568_data* p_data, N1568_channel_status* channel_status) {
	switch( p_data->m_board_type) {
		case N1568A:
			return parse_channel_status_A( p_data, channel_status);
		case N1568B:
			return parse_channel_status_B( p_data, channel_status);
		default:
			return FALSE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////
// Global visible variables declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     B O A R D S   H A N D L I N G
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// 
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_open( N1568_data* p_data, const char *com_str, N1568_BOARD_TYPE type)
{
	const cmd_table* p_table;
	CPLConfig cpl_config;
	memset( p_data, 0, sizeof( N1568_data));

	switch( type) {
		case N1568A:
			p_table= N1568A_CMD_TABLE;
			break;
		case N1568B:
			p_table= N1568B_CMD_TABLE;
			break;
		default:
			return FALSE;
	}

	p_data->m_p_asw_buff= malloc( ASW_BUFF_SIZE);
	p_data->m_p_tmp_asw_buff= malloc( ASW_BUFF_SIZE);
	*p_data->m_p_asw_buff= '\0';
	*p_data->m_p_tmp_asw_buff= '\0';
	p_data->m_board_type= type;

	// Communication device setup
	// COM:9600,N,8,1 Hw flow control
	memset( &cpl_config, 0, sizeof( CPLConfig));
	cpl_config.m_device= CPL_COM;
	strncpy( cpl_config.m_device_config.m_com_device.m_com_str, com_str, sizeof( cpl_config.m_device_config.m_com_device.m_com_str)- 1);
	cpl_config.m_device_config.m_com_device.m_com_str[ sizeof( cpl_config.m_device_config.m_com_device.m_com_str)- 1]= '\0';
	cpl_config.m_device_config.m_com_device.m_baudrate= SERBAUD_9600;
	cpl_config.m_device_config.m_com_device.m_dataBit= DATA_EIGHT;
	cpl_config.m_device_config.m_com_device.m_parity= SERPARITY_NONE;
	cpl_config.m_device_config.m_com_device.m_DTRFlow= DTR_DISABLE;
	cpl_config.m_device_config.m_com_device.m_RTSFlow= RTS_HANDSHAKE;
	cpl_config.m_device_config.m_com_device.m_stopBit= STOP_ONE;

	// basic data iN1568ToolBoxnitialization
	if( !board_open( &p_data->m_common_data, &cpl_config, p_table)) {
		free( p_data->m_p_asw_buff);
		p_data->m_p_asw_buff= (char*)0;
		free( p_data->m_p_tmp_asw_buff);
		p_data->m_p_tmp_asw_buff= (char*)0;
		return FALSE;
	}
	//
	// board specific data initialization
	p_data->m_asw_buff_len= 0;
	memset( p_data->m_p_asw_buff, 0, ASW_BUFF_SIZE);

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////
// 
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_close( N1568_data* p_data)
{
	BOOL ret_val= TRUE;
	if( !board_close( &p_data->m_common_data)) {
		ret_val= FALSE;
	}
	if( p_data->m_p_asw_buff){

		free( p_data->m_p_asw_buff);
		p_data->m_p_asw_buff= (char*)0;
	}
	if( p_data->m_p_tmp_asw_buff){

		free( p_data->m_p_tmp_asw_buff);
		p_data->m_p_tmp_asw_buff= (char*)0;
	}
	return ret_val;
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   0   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
const char* N1568_get_last_answer( N1568_data* p_data) 
{
	int buff_len= p_data->m_asw_buff_len;
	char *src_buff= p_data->m_p_asw_buff;
	char *dest_buff= p_data->m_p_tmp_asw_buff;
	// strip away null chars from the last read buffer
	while( buff_len--)
	{
		if( *src_buff)
		{
			*dest_buff++= *src_buff;
		}
		++src_buff;
	}
	*dest_buff++= '\0';
	return p_data->m_p_tmp_asw_buff;
}
////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL N1568_send_cmd( N1568_data* p_data, N1568B_COMMANDS cmd, UINT8 value, N1568_channel_status* channel_status);
*   \brief   Sends the command to the current channel and gets the channel status
*            
*   \param   p_data Pointer to board data
*   \param   cmd The command id
*   \param   value The command value
*   \param   channel_status The channels status after the command execution. This parameter may be NULL 
*                           (no channel status returned)
*   \return  TRUE: operation successfully completed
*/
////////////////////////////////////////////////////////////////////////////////////////////////
static BOOL N1568_send_cmd( N1568_data* p_data, int cmd, UINT8 value, N1568_channel_status* channel_status)
{
	UINT16 asw_buff_size= ASW_BUFF_SIZE- 1;

	// 
	// Send command
	if( !send_cmd( &p_data->m_common_data, cmd, value))
	{
		return FALSE;
	}
	//
	// Receive answer 
	if( !receive_buffer( &p_data->m_common_data, (UINT8*)p_data->m_p_asw_buff, &asw_buff_size, 40)) 
	{
		return FALSE;
	}
	p_data->m_asw_buff_len= asw_buff_size;
	//
	// Parse the output to get actual channel status
	if( channel_status)
	{
		if( !parse_channel_status( p_data, channel_status))
		{
			return FALSE;
		}
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568A_send_cmd( N1568_data* p_data, N1568A_COMMANDS cmd, UINT8 value, N1568_channel_status* channel_status) {
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}
////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL N1568B_send_cmd( N1568_data* p_data, N1568B_COMMANDS cmd, UINT8 value, N1568_channel_status* channel_status) {
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   1   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_module( N1568_data* p_data, UINT8 value)
{
	UINT16 asw_buff_size= ASW_BUFF_SIZE- 1;
	int cmd;

	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_MOD;
			break;
		case N1568B:
			cmd= N1568B_CMD_MOD;
			break;
		default:
			return FALSE;
	}

	// 
	// Send command
	if( !send_cmd( &p_data->m_common_data, cmd, value))
	{
		return FALSE;
	}
	//
	// Receive answer : just to update the board answer buffer
	if( !receive_buffer( &p_data->m_common_data, (UINT8*)p_data->m_p_asw_buff, &asw_buff_size, 3000)) 
	{
		return FALSE;
	}
	p_data->m_asw_buff_len= asw_buff_size;
	return TRUE;
}
	
////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_channel( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_CH;
			break;
		case N1568B:
			cmd= N1568B_CMD_CH;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}
	
////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_input_polarity( N1568_data* p_data, BOOL negative_polarity, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_IN;
			break;
		case N1568B:
			cmd= N1568B_CMD_IN;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, negative_polarity? 1: 0, channel_status);
}
	
////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_output_polarity( N1568_data* p_data, BOOL negative_polarity, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_OUT;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, negative_polarity? 1: 0, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_shape( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_SH;
			break;
		case N1568B:
			cmd= N1568B_CMD_SH;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_pole_zero_adj( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_PZ;
			break;
		case N1568B:
			cmd= N1568B_CMD_PZ;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_fine_gain( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_FG;
			break;
		case N1568B:
			cmd= N1568B_CMD_FG;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_coarse_gain( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_CG;
			break;
		case N1568B:
			cmd= N1568B_CMD_CG;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_discr_thr( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_CF_TH;
			break;
		case N1568B:
			cmd= N1568B_CMD_CF_TH;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_gain_of_timing( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_CGT;
			break;
		case N1568B:
			cmd= N1568B_CMD_CGT;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_stretcher( N1568_data* p_data, BOOL enable, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568B:
			cmd= N1568B_CMD_STR;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, enable? 1: 0, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_set_cfd_out_width( N1568_data* p_data, UINT8 value, N1568_channel_status* channel_status)
{
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_CFW;
			break;
		case N1568B:
			cmd= N1568B_CMD_CFW;
			break;
		default:
			return FALSE;
	}
	return N1568_send_cmd( p_data, cmd, value, channel_status);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL N1568_export( N1568_data* p_data)
{
	UINT16 asw_buff_size= ASW_BUFF_SIZE- 1;
	int cmd;
	switch( p_data->m_board_type) {
		case N1568A:
			cmd= N1568A_CMD_EXP;
			break;
		case N1568B:
			cmd= N1568B_CMD_EXP;
			break;
		default:
			return FALSE;
	}

	// 
	// Send command
	if( !send_cmd( &p_data->m_common_data, cmd, 1))
	{
		return FALSE;
	}
	//
	// Receive answer : just to update the board answer buffer
	if( !receive_buffer( &p_data->m_common_data, (UINT8*)p_data->m_p_asw_buff, &asw_buff_size, 3000)) 
	{
		return FALSE;
	}
	p_data->m_asw_buff_len= asw_buff_size;
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   2   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     M I S C E L L A N E O U S   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
const char* N1568_SW_rev( void)
{
	return "N1568SDK Rev. 1.1";
}
