////////////////////////////////////////////////////////////////////////////////////////////////
/*! \file    board_commons.c
*   \brief   Board common features implementation
*   \author  NDA
*   \version 1.0
*   \date    03/2006
*            
*            boards common APIs: These APIs provide features common to every board.
*/
////////////////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////////////////////
// File includes
////////////////////////////////////////////////////////////////////////////////////////////////
#include <memory.h>
#include <string.h>
#include <stdlib.h>

#ifdef LINUX
	#include <unistd.h>
    #include <sys/time.h>
#endif

#ifdef WIN32
    #include <time.h>
    #include <sys/timeb.h>
#endif

#include "com_phy_layer.h"
#include "board_commons.h"

////////////////////////////////////////////////////////////////////////////////////////////////
// File local defines
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// Static variables declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// Static methods declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
UINT8 ror_bits( UINT8 data, int num_bits)
{
	while( num_bits--)
	{
		data= ( data& 0x01)? (data>> 1)|0x80: (data>> 1);
	}
	return data;
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
UINT8 swap_bits( UINT8 data)
{
	UINT8 tmp= 0x00;
	UINT8 msk_l, msk_r;
	for( msk_l= 0x80, msk_r= 0x01; msk_l; msk_l>>= 1, msk_r<<= 1)
	{
		tmp|= (data& msk_l)? msk_r: 0x00;
	}
	return tmp;
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
void delay(int msec)
{
#if defined (LINUX)
	usleep( msec* 1000); 
#elif defined (WIN32)
	Sleep( msec);
#else
	???
#endif
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
UINT32 get_time()
{
	static UINT32 startup= 0;
	UINT32 time_ms;
#if defined (LINUX)
	struct timeval t1;
	struct timezone tz;

	gettimeofday(&t1, &tz);
	if( !startup) {
		startup= (UINT32 )t1.tv_sec;
	}
	time_ms = (t1.tv_sec- startup) * 1000 + t1.tv_usec / 1000;

#elif defined (WIN32)
	struct _timeb timebuffer;

	_ftime( &timebuffer );
	if( !startup) {
		startup= (long)timebuffer.time;
	}
	time_ms = (UINT32 )( timebuffer.time- startup) * 1000 + (UINT32 )timebuffer.millitm;
#else
	???
#endif

	return time_ms;
}

////////////////////////////////////////////////////////////////////////////////////////////////
// Global visible variables declaration
////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef WIN32
/*
        -----------------------------------------------------------------------------

        DllMain
        Entry point della libreria (inizializzare qui i dati globali)

        -----------------------------------------------------------------------------
*/
BOOL WINAPI DllMain( HANDLE hModule, DWORD ul_reason_for_call, LPVOID lpReserved )
{
    static int m_num_instances= 0;

        switch (ul_reason_for_call) {
        case DLL_PROCESS_ATTACH:
                {
					if( !m_num_instances)
					{
					}

					++m_num_instances;
                }
                break;
        case DLL_PROCESS_DETACH:
				--m_num_instances;
				break;
        case DLL_THREAD_ATTACH:
        case DLL_THREAD_DETACH:
                break;
        }
        return TRUE;
}

#else   //Linux
/*
        -----------------------------------------------------------------------------

        _init
        Library entry point

        -----------------------------------------------------------------------------
*/
void _init( void )
{
}

/*
        -----------------------------------------------------------------------------

        _fini
        Library exit point

        -----------------------------------------------------------------------------
*/
void _fini( void )
{
}

#endif // WIN32

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     B O A R D S   H A N D L I N G
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// 
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL board_open( board_data* p_data, const CPLConfig *p_com_config, const cmd_table* p_cmd_table)
{
	memset( p_data, 0, sizeof( board_data));

	// init fields
	p_data->m_p_cmd_table= p_cmd_table;
	p_data->m_p_last_sent_cmd= malloc( LAST_SEND_CMD_SIZE);
	*p_data->m_p_last_sent_cmd= '\0';

	p_data->m_p_cpl_object= malloc( sizeof( CPL_OBJ));

	CPL_BiosInit( p_data->m_p_cpl_object);
	if( !CPL_Open( p_data->m_p_cpl_object, p_com_config))
	{
		free( p_data->m_p_cpl_object);
		p_data->m_p_cpl_object= (CPL_OBJ*)0;

		free( p_data->m_p_last_sent_cmd);
		p_data->m_p_last_sent_cmd= (char*)0;

		return FALSE;
	}
	return TRUE;
}
////////////////////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL board_close( board_data* p_data)
{
	if( p_data->m_p_cpl_object){
		if( !CPL_Close( p_data->m_p_cpl_object))
		{
			return FALSE;
		}
		free( p_data->m_p_cpl_object);
		p_data->m_p_cpl_object= (CPL_OBJ*)0;
	}
	if( p_data->m_p_last_sent_cmd) {
		free( p_data->m_p_last_sent_cmd);
		p_data->m_p_last_sent_cmd= (char*)0;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   0   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// 
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL send_buffer( board_data* p_data, const UINT8* buff, UINT16 buff_size)
{
	CPL_Flush( p_data->m_p_cpl_object, BOTH_BUFFER );
	// Don't send the whole data buffer but let a small delay to avoid N1568 data overrun
//	return CPL_direct_TX( p_data->m_p_cpl_object, buff, buff_size);
	if( !buff_size)
		return TRUE;
	while( buff_size--)
	{
		if( !CPL_direct_TX( p_data->m_p_cpl_object, buff, 1))
			return FALSE;
		buff++;
		Sleep( 50);
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////
// 
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL receive_buffer( board_data* p_data, UINT8* buff, UINT16 *buff_size, int timeout_msec)
{
	UINT16 bytes_to_read= *buff_size;
	int count_empty= 200;		// First byte timeout (10 mesc)

	timeout_msec/= 10;
	if( !timeout_msec) 
	{
		timeout_msec= 10;
	}

	*buff_size= 0;
	while( (*buff_size< bytes_to_read)&& ( count_empty--))
	{
		UINT16 bytes_read= bytes_to_read- *buff_size;

		if( !CPL_direct_RX( p_data->m_p_cpl_object, buff+ *buff_size, &bytes_read))
		{
			return FALSE;
		}
		if( bytes_read)
		{
			*buff_size+= bytes_read;
			count_empty= timeout_msec;
		}
		else
		{
			Sleep( 10);
		}
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   1   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// 
////////////////////////////////////////////////////////////////////////////////////////////////
BOOL send_cmd( board_data* p_data, int cmd_index, int cmd_value)
{
	if( ( cmd_value< p_data->m_p_cmd_table[ cmd_index].m_min_value)||
		( cmd_value> p_data->m_p_cmd_table[ cmd_index].m_max_value))
	{
		return FALSE;
	}
	sprintf( p_data->m_p_last_sent_cmd, "%s%d\r", p_data->m_p_cmd_table[ cmd_index].m_cmd_string, cmd_value);
	return send_buffer( p_data, (UINT8*)p_data->m_p_last_sent_cmd, (UINT16)strlen( p_data->m_p_last_sent_cmd));
}

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     G L O B A L   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     M I S C E L L A N E O U S   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////
