////////////////////////////////////////////////////////////////////////////////////////////////
/*! \file    board_commons.h
*   \brief   Board common features definitions
*   \author  NDA
*   \version 1.0
*   \date    07/2008
*            
*            Boards common APIs: typically level 0 APIs
*            These APIs provide features common to every board.
*/
////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef __BOARD_COMMONS_H
#define __BOARD_COMMONS_H

////////////////////////////////////////////////////////////////////////////////////////////////
// File includes
////////////////////////////////////////////////////////////////////////////////////////////////
#include <stdio.h>

#include "common_defs.h"
#include "com_phy_layer.h"

////////////////////////////////////////////////////////////////////////////////////////////////
// Global defines
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \struct  cmd_table
*   \brief   Board commands table entry
*            
*            This is a struct holding boards specific command table entries
*/
////////////////////////////////////////////////////////////////////////////////////////////////
typedef struct
{
	const char* m_cmd_string;	//!< The command string
	int m_min_value;			//!< The minimum parameter value
	int m_max_value;			//!< The maximum parameter value
} cmd_table;

#define LAST_SEND_CMD_SIZE 256	/* !< The board_data::m_p_last_sent_cmd buffer size */

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \struct  board_data
*   \brief   Data common to every board
*            
*            This is the basic structure for every board specific data. 
*            This struct MUST be the first data of any other specific board data.
*/
////////////////////////////////////////////////////////////////////////////////////////////////
typedef struct
{
	CPL_OBJ* m_p_cpl_object;					//!< The Communication Physical Layer object
	const cmd_table* m_p_cmd_table;				//!< Pointer to board specif command table
	char* m_p_last_sent_cmd;					//!< The last sent command string
} board_data;

////////////////////////////////////////////////////////////////////////////////////////////////
// Global variables declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
// Global methods declaration
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      UINT8 ror_bits( UINT8 data, int num_bits)
*   \brief   Perfoms a num_bits bitwise right rotation of data byte 
*            
*   \param   data The source byte to rotate
*   \param   num_bits The number of bits to rotate
*   \return  Converted data
*/
////////////////////////////////////////////////////////////////////////////////////////////////
UINT8 ror_bits( UINT8 data, int num_bits);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      UINT8 swap_bits( UINT8 data)
*   \brief   Swaps the bits of the source byte
*            
*   \param   data The source byte to swap
*   \return  Converted data
*/
////////////////////////////////////////////////////////////////////////////////////////////////
UINT8 swap_bits( UINT8 data);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      void delay(int msec)
*   \brief   Waits for specified time
*            
*   \param   msec The time delay to wait (msec)
*/
////////////////////////////////////////////////////////////////////////////////////////////////
void delay(int msec);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      UINT32 get_time
*   \brief   Gets the time in milliseconds since first procedure call
*            
*   \return  The time in milliseconds since first procedure call
*/
////////////////////////////////////////////////////////////////////////////////////////////////
UINT32 get_time( );

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     B O A R D S   H A N D L I N G
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL board_open( board_data* p_data, const CPLConfig *p_com_config, const cmd_table* p_cmd_table);
*   \brief   Basic board open
*            
*            Provides basic handling for board opening, common to all the boards.
*            It tries to open the communication channel, too.
*   \param   p_data Pointer to board data
*   \param   p_com_config The communication physical layer device settings
*   \param   p_cmd_table The board specific command table pointer
*   \return  TRUE: board successfully opened
*   \note    This API if typically called by derived boards during open procedure.
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL board_open( board_data* p_data, const CPLConfig *p_com_config, const cmd_table* p_cmd_table);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL board_close( board_data* p_data);
*   \brief   Basic board close
*            
*            Provides basic handling for board closing, common to all the boards.
*            It tries to close the communication channel, too.
*   \param   p_data Pointer to board data
*   \return  TRUE: board successfully closed
*   \note    This API if typically called by derived boards during close procedure.
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL board_close( board_data* p_data);

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   0   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL send_buffer( board_data* p_data, const UINT8* buff, UINT16 buff_size);
*   \brief   Sends a data buffer to board
*            
*   \param   p_data Pointer to board data
*   \param   buff The data buffer to send
*   \param   buff_size The data buffer size (BYTEs)
*   \return  TRUE  send procedure ok
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL send_buffer( board_data* p_data, const UINT8* buff, UINT16 buff_size);

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL receive_buffer( board_data* p_data, UINT8* buff, UINT16 *buff_size, int timeout_msec);
*   \brief   Read up to *buff_size BYTEs from board
*            
*   \param   p_data Pointer to board data
*   \param   buff The user allocated buffer 
*   \param   buff_size The user allocated buffer size (on enter) and the number of BYTEs read (on exit)
*   \param   timeout_msec The timeout between bytes receive (msec) 
*   \return  TRUE read procedure ok
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL receive_buffer( board_data* p_data, UINT8* buff, UINT16 *buff_size, int timeout_msec);

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     L E V E L   1   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
/*! \fn      BOOL send_cmd( board_data* p_data, int cmd_index, int cmd_value);
*   \brief   Sends a command to the board
*            
*   \param   p_data Pointer to board data
*   \param   cmd_index The index of the command (as from the board specific COMMANDS enumerator)
*   \param   cmd_value The value of the command
*   \return  TRUE  send procedure ok
*/
////////////////////////////////////////////////////////////////////////////////////////////////
DLL_API BOOL send_cmd( board_data* p_data, int cmd_index, int cmd_value);

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     G L O B A L   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////
//
//     M I S C E L L A N E O U S   A P I s
//
////////////////////////////////////////////////////////////////////////////////////////////////

#endif			// __BOARD_COMMONS_H
