/******************************************************************************
*
* Copyright (C) 2017 CAEN SpA - www.caen.it - support.computing@caen.it
*
***************************************************************************//**
* \note TERMS OF USE:
* This file is subject to the terms and conditions defined in file
* 'CAEN_License_Agreement.txt', which is part of this source code package.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. The user relies on the
* software, documentation and results solely at his own risk.
******************************************************************************/

#include "WaveDemo.h"
#include "WDWaveformProcess.h"

// --------------------------------------------------------------------------------------------------------- 
// Global Variables
// --------------------------------------------------------------------------------------------------------- 

float *WPsmooth;		// smoothed waveform generated by the WaveformProcessor
float *WPdiscr;			// discriminator waveform generated by the WaveformProcessor
int WPmaxNs;			// Max num of samples in the WaveformProcessor
int TrgShift;			// num of sample to shift for trigger jitter correction
uint64_t CoarseTimeStampRef;

static inline int roundUp(int numToRound, int multiple) {
	if (multiple == 0)
		return numToRound;

	int remainder = abs(numToRound) % multiple;
	if (remainder == 0)
		return numToRound;

	if (numToRound < 0)
		return -(abs(numToRound) - remainder);
	else
		return numToRound + multiple - remainder;
}

// --------------------------------------------------------------------------------------------------------- 
// Description: trigger jitter correction, shift waveforms and fill initial/final gaps by copying first/last sample.
// Inputs:		shift = number of samples to shift
// Outputs:		waves = waveforms shifted
// Return:		-
// --------------------------------------------------------------------------------------------------------- 
static void WaveformShift(Waveform_t *waves, int shift) {
	int i;
	int ns = waves->Ns;

	if (shift == 0 || abs(shift) >= ns)
		return;

	if (shift > 0) {
		//shift waveforms to the rigth
		for (int a = 0; a < NUM_ATRACE; a++)
			for (i = 0; i < ns - shift; i++)
				waves->AnalogTrace[a][ns - 1 - i] = waves->AnalogTrace[a][(ns - 1 - i) - shift];
		for (i = 0; i < ns - shift; i++)
			waves->DigitalTraces[ns - 1 - i] = waves->DigitalTraces[(ns - 1 - i) - shift];
		//fill initial gaps by copying first sample
		for (int a = 0; a < NUM_ATRACE; a++)
			for (i = 0; i < shift; i++)
				waves->AnalogTrace[a][i] = waves->AnalogTrace[a][shift];
		for (i = 0; i < shift; i++)
			waves->DigitalTraces[i] = waves->DigitalTraces[shift];
	}
	else {
		shift = -shift;
		//shift waveforms to the left
		for (int a = 0; a < NUM_ATRACE; a++)
			for (i = 0; i < ns - shift; i++)
				waves->AnalogTrace[a][i] = waves->AnalogTrace[a][i + shift];
		for (i = 0; i < ns - shift; i++)
			waves->DigitalTraces[i] = waves->DigitalTraces[i + shift];
		//fill final gaps by copying last sample
		for (int a = 0; a < NUM_ATRACE; a++)
			for (i = 0; i < shift; i++)
				waves->AnalogTrace[a][ns - 1 - i] = waves->AnalogTrace[a][ns - 1 - shift];
		for (i = 0; i < shift; i++)
			waves->DigitalTraces[ns - 1 - i] = waves->DigitalTraces[ns - 1 - shift];
	}
}

// --------------------------------------------------------------------------------------------------------- 
// Description: Off-line implementation of the discriminator (LED or CFD), time interpolation and energy.
//				Also performs trigger jitter correction.
// Inputs:		b = Board Number
//				c = channel number
//				ns = number of samples
//				Wavein = input waveform to process
//				CoarseTimeStamp = time stamp provided by the board in the event (in counts)
// Outputs:		Wavesout = waveforms generated by the processor (e.g. CFD) and digital probes (e.g. Trigger)
//				Baseline = result of the baseline calculation (in ADC counts)
//				TimeStamp = result of the time interpolation in nanoseconds (=0 if it is not found)
//				Energy = integration of the input signal into the energy gate (in ADC counts)
// Return:		0=OK, -1=error
// --------------------------------------------------------------------------------------------------------- 
static int SW_WaveformProcessor(int b, int ch, int ns, const float *Wavein, uint64_t CoarseTimeStamp, Waveform_t *Wavesout, float *Baseline, float *TimeStamp, float* Energy) {
	// get pointer to substructure
	WaveDemoBoardHandle_t *WDh = &WDcfg.handles[b];
	WaveDemoBoard_t *WDb = &WDcfg.boards[b];
	WaveDemoChannel_t *WDc = &WDb->channels[ch];

	int nsbl = 0, ncross = 0, armed = 0;
	const int wpns = (ns <= WPmaxNs) ? ns : WPmaxNs;
	float Q = 0;
	float baseline = 0;
	float ZCneg = 0, ZCpos = 0;

	memset(Wavesout->DigitalTraces, 0, wpns * sizeof(uint8_t));

	int ns_baseline;
	// default ns_baseline is 10
	ns_baseline = (WDc->NsBaseline < 1) ? 1 : WDc->NsBaseline; // coerce min to 1
	ns_baseline = (WDc->NsBaseline > WPmaxNs) ? WPmaxNs : WDc->NsBaseline; // coerce max

	const int Gwidth = (int)(WDc->GateWidth / WDh->Ts);
	const int PreGate = (int)(WDc->PreGate / WDh->Ts);
	const int CFDdelay = (int)(WDc->CFDdelay / WDh->Ts);

	const float atten = WDc->CFDatten;

	// discriminator is intended for negative pulses; if positive, invert it
	const int sign = (WDc->PulsePolarity == CAEN_DGTZ_PulsePolarityPositive) ? -1 : 1;

	if ((WPdiscr == NULL) || (WPsmooth == NULL))
		return -1;

	// Calculate baseline and apply smoothing
	float smean = 0;
	uint32_t smn = 0;
	float bslacc = 0;
	int bslstop = 0;

	for (int i = 0; i < wpns; i++) {
		Wavesout->DigitalTraces[i] |= bslstop ? DTRACE_BASELINE : 0;
		if (!bslstop) {
			if (nsbl < ns_baseline) {
				bslacc += Wavein[i];
				nsbl++;
			}
			else {
				bslacc += Wavein[i] - Wavein[i - nsbl];
			}
		}
		baseline = bslacc / nsbl;
		if ((i + 5 < wpns) && (fabsf(Wavein[i + 5] - baseline) > fabsf(WDc->TriggerThreshold_adc)))
			bslstop = 1;

		if (smn < (uint32_t)(1 << WDc->TTFsmoothing)) { // default TTFsmoothing is 0
			smean += Wavein[i];
			smn++;
		}
		else {
			smean += Wavein[i] - Wavein[i - smn];
		}
		WPsmooth[i] = smean / smn;  // smoothed input signal
	}
	*Baseline = baseline;

	// calculate discriminator waveform (either LED or CFD)
	for (int i = 0; i < wpns; i++) {
		if (WDc->DiscrMode == 1) {  // CFD
			// use alternative threshold if defined or use TriggerThreshold * atten
			float CFDThreshold = (WDc->CFDThreshold >= 0) ? WDc->CFDThreshold : WDc->TriggerThreshold_adc * atten;
			CFDThreshold *= sign;
			if (i >= CFDdelay)
				WPdiscr[i] = (float)(sign * (atten * (WPsmooth[i] - baseline) - (WPsmooth[i - CFDdelay] - baseline)));
			else
				WPdiscr[i] = sign * (WPsmooth[i] - baseline);
			if (!armed && (WPdiscr[i] < CFDThreshold))
				armed = 1;
			if (armed && ncross == 0 && WPdiscr[i] >= 0) {
				ncross = i;
				Wavesout->DigitalTraces[i] |= DTRACE_TRIGGER; // trigger
				ZCneg = WPdiscr[i - 1];
				ZCpos = WPdiscr[i];
			}
		}
		else {  // LED
			WPdiscr[i] = sign * (WPsmooth[i] - baseline);
			if ((i > 0) && (ncross == 0) && (WPdiscr[i] < sign * WDc->TriggerThreshold_adc)) {
				ncross = i;
				Wavesout->DigitalTraces[i] |= DTRACE_TRIGGER; // trigger
				ZCneg = sign * WDc->TriggerThreshold_adc - WPdiscr[i - 1];
				ZCpos = sign * WDc->TriggerThreshold_adc - WPdiscr[i];
			}
		}

		for (int a = 0; a < NUM_ATRACE; a++) {
			if (Wavesout->AnalogTrace[a] != NULL) {
				switch (a) {
				case 1:
					Wavesout->AnalogTrace[a][i] = WPdiscr[i] /*+ (1 << WDh->Nbit) / 2*/;
					break;
				case 2:
					Wavesout->AnalogTrace[a][i] = WPsmooth[i];
					break;
				case 3:
					Wavesout->AnalogTrace[a][i] = (ncross != 0) ? 0 : WDc->TriggerThreshold_adc;
					break;
				case 0:
				default:
					Wavesout->AnalogTrace[a][i] = Wavein[i];
					break;
				}
			}
		}

		//skip samples in pre-gate, check to avoid buffer overflow
		if (i - PreGate < 0)
			continue;

		if ((ncross > 0) && (i <= (ncross + Gwidth - PreGate))) {
			Q += sign * (baseline - Wavein[i - PreGate]);
			Wavesout->DigitalTraces[i - PreGate] |= DTRACE_ENERGY; // Energy gate
		}
	}

	// timing interpolation
	*TimeStamp = 0;
	if (WDcfg.WaveformProcessor & 0x01) {
		if (ncross > 0 && ZCneg < 0 && ZCpos >= 0) {
			*TimeStamp = (ncross + ((-ZCneg) / (ZCpos - ZCneg))) * WDh->Ts; // fine time stamp is expressed in ns
		}
	}

	// energy
	*Energy = 0;
	if (WDcfg.WaveformProcessor & 0x02) {
		*Energy = Q;
	}

	// trigger jitter correction
	if (WDcfg.WaveformProcessor & 0x04) {
		if (b == WDcfg.TOFstartBoard && ch == WDcfg.TOFstartChannel && ncross > 0) {
			// determine the waveform shift to remove trigger jitter (TOFstart channel is taken as a reference)
			TrgShift = (wpns * WDcfg.TriggerFix / 100) - ncross;
			CoarseTimeStampRef = CoarseTimeStamp;
		}
		//int samplesDiffTDC = roundUp((int)(CoarseTimeStamp - CoarseTimeStampRef), 5) * (5 / WDh->Ts);
		if (ncross > 0) {
			WaveformShift(Wavesout, TrgShift /*+ samplesDiffTDC*/);
		}
	}

	return 0;
}

// --------------------------------------------------------------------------------------------------------- 
// Description: Init Global Variables and allocate memory for the waveforms
// Inputs:		-
// Outputs:		-
// Return:		0=OK, -1=error
// --------------------------------------------------------------------------------------------------------- 
int InitWaveProcess() {
	int ret = 0;

	if (WDcfg.WaveformProcessor) {
		// allocate processed waveforms
		if (WPdiscr == NULL)
			WPdiscr = (float *)malloc(WDcfg.GlobalRecordLength * sizeof(float));
		if (WPsmooth == NULL)
			WPsmooth = (float*)malloc(WDcfg.GlobalRecordLength * sizeof(float));
		// reset arrays
		if (WPdiscr)
			memset(WPdiscr, 0, WDcfg.GlobalRecordLength * sizeof(float));
		if (WPsmooth)
			memset(WPsmooth, 0, WDcfg.GlobalRecordLength * sizeof(float));

		WPmaxNs = WDcfg.GlobalRecordLength; // to prevent longer waveform to make a memory overflow
	}

	for (int b = 0; b < WDcfg.NumBoards; b++) {
		// get pointer to substructure
		WaveDemoBoard_t* WDb = &WDcfg.boards[b];
		WaveDemoBoardHandle_t* WDh = &WDcfg.handles[b];
		for (int c = 0; c < MAX_CH; c++) {
			// get pointer to substructure
			WaveDemoChannel_t* WDc = &WDb->channels[c];

			int offset = (WDb->CorrectionLevel == 0 || WDb->CorrectionLevel == 2) ? 2048 : 0;
			float trg_val = offset + (1485 * WDc->TriggerThreshold_V);
			WDc->TriggerThreshold_adc = trg_val;
		}
	}

	return ret;
}


// --------------------------------------------------------------------------------------------------------- 
// Description:	Free memory buffer
// Return:		0=OK, -1=error
// --------------------------------------------------------------------------------------------------------- 
int CloseWaveProcess() {
	free(WPdiscr);
	WPdiscr = NULL;
	free(WPsmooth);
	WPsmooth = NULL;
	return 0;
}

// --------------------------------------------------------------------------------------------------------- 
// Description: 
// Inputs:		int b = Board Number
//				int ch = Channel Number
// In/Out:		event = data source/destination
// Outputs:		Wfm = waveform data
// Return: 0=OK, -1=error
// --------------------------------------------------------------------------------------------------------- 
int WaveformProcess(int b, int ch, WaveDemoEvent_t *event) {
	int groupIndex = ch / 2;
	int channelIndex = ch % 2;
	WaveDemo_EVENT_plus_t *EventPlus = &event->EventPlus[groupIndex][channelIndex];

	int ns = event->Event->DataGroup[groupIndex].ChSize;
	if (ns > WDcfg.GlobalRecordLength)
		ns = WDcfg.GlobalRecordLength;

	Waveform_t *Wfm = EventPlus->Waveforms;

	float *Wavein = event->Event->DataGroup[groupIndex].DataChannel[channelIndex];
	//float CoarseTimeStamp = ((float)event->Event->DataGroup[ch / 2].TDC * 5) / WDcfg.handles[b].Ts; // changes TDC at 200 MHz (5 ns) to time of sampling rate
	uint64_t CoarseTimeStamp = event->Event->DataGroup[ch / 2].TDC * 5;
	float Baseline = 0, TimeStamp = 0, Energy = 0;
	if (WDcfg.WaveformProcessor)
		SW_WaveformProcessor(b, ch, ns, Wavein, CoarseTimeStamp, Wfm, &Baseline, &TimeStamp, &Energy);
	EventPlus->Baseline = Baseline;
	EventPlus->Energy = Energy;
	if (TimeStamp != 0)
		EventPlus->FineTimeStamp = TimeStamp;

	return 0;
}

int MultiWaveformProcess(WaveDemoEvent_t *events[], int n) {
	int ret = 0;
	TrgShift = 0;
	for (int bd = 0; bd < n; bd++) {
		WaveDemoBoard_t *WDb = &WDcfg.boards[bd];
		WaveDemoBoardHandle_t *WDh = &WDcfg.handles[bd];
		for (int ch = 0; ch < WDh->Nch; ch++) {
			WaveDemoChannel_t *WDc = &WDb->channels[ch];
			if (!events[bd]->Event->GrPresent[ch / 2] || !WDc->ChannelEnable)
				continue;
			ret |= WaveformProcess(bd, ch, events[bd]);
		}
	}
	return ret;
}