/******************************************************************************
*
* CAEN SpA - Front End Division
* Via Vetraia, 11 - 55049 - Viareggio ITALY
* +390594388398 - www.caen.it
*
******************************************************************************/

#include <inttypes.h>

#include "dpp_qdc.h"
#ifdef LINUX
#include <sys/time.h> /* struct timeval, select() */
#include <sys/stat.h>
#endif




// ------------------------------------------------------------------------------
void usage() {
    printf("Syntax : readout_demo <acquisition_runtime (seconds)>\n");
    printf("acquisition_runtime must be an integer > 0 and < %ld\n", LONG_MAX);
}


// ------------------------------------------------------------------------------ 
// Checks if a key has been pressed and sets global variables accordingly.
// Returns 0 on success; -1 in case of any error detected or when exit key is pressed.
// ------------------------------------------------------------------------------ 
int check_user_input() {
    int c;

    /* check if any key has been pressed */
    if (kbhit()) {
        c = getch();
        switch (c) {
        case 'q':
            return -1;
            break;
        case 's':
            if (gAcqrun == 0) {
                // open gnuplot in a pipe and the data file//

#ifdef LINUX      
                gHistPlotFile = popen("gnuplot", "w");
                fprintf(gHistPlotFile, "set xlabel 'ADC channels'\n");
                fprintf(gHistPlotFile, "set ylabel 'Counts'\n");
#else

                gHistPlotFile = popen("pgnuplot.exe", "w");
                fprintf(gHistPlotFile, "set xlabel 'ADC channels'\n");
                fprintf(gHistPlotFile, "set xrange [0:%d]\n", HISTO_NBIN);
                fprintf(gHistPlotFile, "set ylabel 'Counts'\n");
#endif
                if (gHistPlotFile == NULL) {
                    printf("Can't open gnuplot\n");
                    return -1;
                }
                if (gParams.AcqMode == CAEN_DGTZ_DPP_ACQ_MODE_Mixed) {

#ifdef LINUX      
                    gWavePlotFile = popen("gnuplot", "w");
#else
                    gWavePlotFile = popen("pgnuplot.exe", "w");
#endif        
                    fprintf(gWavePlotFile, "set yrange [0:4096]\n");
                    fprintf(gWavePlotFile, "set xlabel 'sampling units (x16 ns)'\n");
                    fprintf(gWavePlotFile, "set ylabel 'ADC counts'\n");
                }
                CAEN_DGTZ_SWStartAcquisition(gHandle);
                printf("Acquisition started\n");
                Sleep(500);
                gAcqrun = 1;
            }
            break;
        case 'S':

            if (gHistPlotFile != NULL) {
                pclose(gHistPlotFile);
                gHistPlotFile = NULL;
            }
            if (gWavePlotFile != NULL) {
                pclose(gWavePlotFile);
                gWavePlotFile = NULL;
            }
            gAcqrun = 0;
            gStopped = 1;
            gRestart = 1;
            break;
        case 't':
            printf("Single SW trigger generated\n");
            gSWTrigger = 1;
            break;
        case '0':
            gAnalogTrace = 0;
            gToggleTrace = 1;
            break;
        case '1':
            gAnalogTrace = 1;
            gToggleTrace = 1;
            break;
        case '2':
            gAnalogTrace = 2;
            gToggleTrace = 1;
            break;
        case 'l':
            gParams.SaveList = 1;
            break;
        case 'T':
            gSWTrigger = gSWTrigger ^ 2;
            if (gSWTrigger == 0)
                printf("Continuous SW trigger disabled\n");
            else if (gSWTrigger == 2)
                printf("Continuous SW trigger enabled\n");
            break;
        case 'r':
            ResetStatistics();
            printf("All histograms have been reset\n");
            break;
        case 'R':
            if (gHistPlotFile != NULL) {
                pclose(gHistPlotFile);
                gHistPlotFile = NULL;
            }
            if (gWavePlotFile != NULL) {
                pclose(gWavePlotFile);
                gWavePlotFile = NULL;
            }
            gRestart = 1;
            gAcqrun = 0;
            break;
        case 'c':
            printf("Channel = ");
            scanf("%d", &gActiveChannel);
            if (gActiveChannel >= gEquippedChannels) {
                printf("Board has only %d channels\n", (int32_t)gEquippedChannels);
                gActiveChannel = gEquippedChannels - 1;
                Sleep(600);
            }
            break;
        case 'g':
            printf("Group = ");
            scanf("%d", &grp4stats);
            if (grp4stats >= (int)(gEquippedChannels) / 8) {
                printf("Board has only %d groups of channels\n", (int32_t)gEquippedChannels / 8);
                grp4stats = gEquippedChannels / 8 - 1;
                Sleep(600);
            }
            break;
        case ' ':
            PrintMenu();
            getch();
            break;
        default:
            break;

        }

    } /*  if (kbhit()) */

    return 0;
}


// ------------------------------------------------------------------------------ 
// Print menu to screen (keyboard commands)
/* --------------------------------------------------------------------------------------------------------- */
void PrintMenu() {
    printf("\ns ) Start acquisition\n");
    printf("S ) Stop acquisition\n");
    printf("R ) Reload configuration file and restart\n");
    printf("t ) Send a software trigger\n");
    printf("T ) Enable/Disable continuous software trigger\n");
    printf("c ) Change active channel\n");
    printf("g ) Change active group index\n");
    printf("0 ) Set Analog Waveform 1 = Input\n");
    printf("1 ) Set Analog Waveform 1 = Smoothed Input\n");
    printf("2 ) Set Analog Waveform 1 = Baseline\n");
    printf("l ) Save Charge and Timetag to file\n");
    printf("r ) Reset Histograms\n");
    printf("q ) Quit\n\n\n");
    printf("[SPACE] This help\n");

}

// ------------------------------------------------------------------------------ 
int ResetStatistics()
{
    int i;

	gBytesRead=0;
	gTotEvCnt=0;
	gPrevTotEvCnt=0;

	memset(gChEvCnt, 0, MAX_CHANNELS * sizeof(gChEvCnt[0]));
	memset(gPrevChEvCnt, 0, MAX_CHANNELS * sizeof(gPrevChEvCnt[0]));

    for (i = 0; i < (int)gEquippedChannels; ++i)
        memset(gHisto[i], 0, HISTO_NBIN * sizeof(uint32_t));
    gLoops = 0;
    return 0;
}

// ------------------------------------------------------------------------------ 
/*
** Read data from board and decode events.
** The function has a number of side effects since it relies
** on global variables.
** Actions performed by this function are:
**  - send a software trigger (if enabled)
**  - read available data from board
**  - get events fromm acquisition buffer
**  - updates energy histogram
**  - plot histogram every second
**  - saves list files (if enabled)
**  - gets waveforms from events and plots them
**  - saves data to disk
*/
int run_acquisition(FILE *csvFile) {
    int ret;
    unsigned int i;
    unsigned int j;
    uint32_t     bin;
    uint32_t     bsize;
    uint32_t     NumEvents[MAX_CHANNELS];
    static uint32_t     EventID = 0;
    FILE *       gPlotDataFile1 = NULL;
    FILE *       gPlotDataFile2 = NULL;
    CAEN_DGTZ_EventInfo_t* eventInfo;
    char *EventPtr = NULL;

    /* Send a SW Trigger if requested by user */
    if (gSWTrigger) {
        if (gSWTrigger == 1)    gSWTrigger = 0;
        ret = CAEN_DGTZ_SendSWtrigger(gHandle);
        if (ret != CAEN_DGTZ_Success) {
            printf("Error sending software trigger (ret = %d): exiting ....\n", ret);
            exit(-1);
        }
    }

    memset(NumEvents, 0, MAX_CHANNELS * sizeof(NumEvents[0]));
    /* Read a block of data from the digitizer */
    ret = CAEN_DGTZ_ReadData(gHandle, CAEN_DGTZ_SLAVE_TERMINATED_READOUT_MBLT, gAcqBuffer, &bsize); /* Read the buffer from the digitizer */
    gLoops += 1;
    if (ret) {
        printf("Readout Error\n");
        return ret;
    }
    if (bsize == 0)
        return 0;


    /* Get event number from global header info */
    /*
    eventInfo = (CAEN_DGTZ_EventInfo_t *)malloc(sizeof(CAEN_DGTZ_EventInfo_t));


    ret = CAENDGTZ_API CAEN_DGTZ_GetEventInfo(gHandle, gAcqBuffer, bsize, 0, eventInfo, &EventPtr);
    if (ret) {
        printf("Readout Error\n");
        return ret;
    }

    EventID = eventInfo->EventCounter;
    */

   

    /* Decode and analyze Events */
    if ((ret = CAEN_DGTZ_GetDPPEvents(gHandle, gAcqBuffer, bsize, (void **)gEvent, NumEvents)) != CAEN_DGTZ_Success) {
        printf("Error during _CAEN_DGTZ_GetDPPEvents() call (ret = %d): exiting ....\n", ret);
        exit(-1);
    }



    /* Save data to CSV file */
    if (gSaveCSVFile) {
        for (j = 0; j < NumEvents[0]; ++j) {
            int timestamp = gEvent[0][j].TimeTag;
            fprintf(csvFile, "%d", timestamp);
            fprintf(csvFile, ",%d", EventID);
            EventID++;
            for (i = 0; i < gEquippedChannels; ++i) {
                if ((gParams.ChannelTriggerMask >> i) & 0x1) {
                    fprintf(csvFile, ",%d", gEvent[i][j].Charge);
                }
                else {
                    fprintf(csvFile, ",%d", 0);
                }
            }
            fprintf(csvFile, "\n");
        }
    }


    /* Loop over all channels and events */
    for (i = 0; i < gEquippedChannels; ++i) {
        for (j = 0; j < NumEvents[i]; ++j) {
            uint32_t Charge;

			Charge = (gEvent[i][j].Charge & 0xFFFF);/// * HISTO_NBIN / 65536;  /* rebin charge to HISTO_NBIN */
            gTotEvCnt ++;


            /* Update energy histogram */
            if ((Charge < HISTO_NBIN) && (Charge >= CHARGE_LLD_CUT) && (Charge <= CHARGE_ULD_CUT) && (gEvent[i][j].Overrange == 0)) {
                gHisto[i][Charge]++;        
            }

            /* Plot Histogram */
            if ((gCurrTime - gPrevHPlotTime) > 1000) {
                gPlotDataFile1 = fopen(plotdata_file, "w+"); //"PlotData.txt"
                for (bin = 0; bin < HISTO_NBIN; bin++)
                    fprintf(gPlotDataFile1, "%d\n", gHisto[gActiveChannel][bin]);
                fclose(gPlotDataFile1);
                fprintf(gHistPlotFile, "plot '%s' title 'Channel %d' with step\n", plotdata_file, gActiveChannel);
                fflush(gHistPlotFile);
                gPrevHPlotTime = gCurrTime;
            }
            
            gCurrTimeTag[i] = gEvent[i][j].TimeTag;            

            /* Check roll over of time tag */
            if (gEvent[i][j].TimeTag < gPrevTimeTag[i])
                gETT[i]++;
            gExtendedTimeTag[i] = (gETT[i] << 32) + (uint64_t)(gEvent[i][j].TimeTag);

            /* Save event to output file */
            if (gParams.SaveList) {
                fprintf(gListFiles[i], "%-16"PRIu64"%-8d\n", gExtendedTimeTag[i], gEvent[i][j].Charge);
            }

            /* Plot Waveforms (if enabled) */
            if ((i == gActiveChannel) && (gParams.AcqMode == CAEN_DGTZ_DPP_ACQ_MODE_Mixed) && ((gCurrTime - gPrevWPlotTime) > 300) && (Charge >= CHARGE_LLD_CUT) && (Charge <= CHARGE_ULD_CUT)) {
                CAEN_DGTZ_DecodeDPPWaveforms(gHandle, &gEvent[i][j], gWaveforms);
                gPlotDataFile2 = fopen(plotwave_file, "w+"); //"PlotWave.txt"
                int k = 0;
                for (k = 0; k < (int)gWaveforms->Ns; k++) {
                    fprintf(gPlotDataFile2, "%d ", gWaveforms->Trace1[k]);                 /* samples */
                    fprintf(gPlotDataFile2, "%d ", 2000 + 200 * gWaveforms->DTrace1[k]);  /* gate    */
                    fprintf(gPlotDataFile2, "%d ", 1000 + 200 * gWaveforms->DTrace2[k]);  /* trigger */
                    fprintf(gPlotDataFile2, "%d ", 500 + 200 * gWaveforms->DTrace3[k]);   /* trg hold off */
                    fprintf(gPlotDataFile2, "%d\n", 100 + 200 * gWaveforms->DTrace4[k]);  /* overthreshold */
                }
                fclose(gPlotDataFile2);


                switch (gAnalogTrace) {
                case 0:
                    fprintf(gWavePlotFile, "plot '%s' u 1 t 'Input' w step, '%s' u 2 t 'Gate' w step, '%s' u 3 t 'Trigger' w step, '%s' u 4 t 'TrgHoldOff' w step, '%s' u 5 t 'OverThr' w step\n", plotwave_file, plotwave_file, plotwave_file, plotwave_file, plotwave_file);
                    break;
                case 1:
                    fprintf(gWavePlotFile, "plot '%s' u 1 t 'Smooth' w step, '%s' u 2 t 'Gate' w step, '%s' u 3 t 'Trigger' w step, '%s' u 4 t 'TrgHoldOff' w step, '%s' u 5 t 'OverThr' w step\n", plotwave_file, plotwave_file, plotwave_file, plotwave_file, plotwave_file);
                    break;
                case 2:
                    fprintf(gWavePlotFile, "plot '%s' u 1 t 'Baseline' w step, '%s' u 2 t 'Gate' w step, '%s' u 3 t 'Trigger' w step, '%s' u 4 t 'TrgHoldOff' w step, '%s' u 5 t 'OverThr' w step\n", plotwave_file, plotwave_file, plotwave_file, plotwave_file, plotwave_file);
                    break;
                default:
                    fprintf(gWavePlotFile, "plot '%s' u 1 t 'Input' w step, '%s' u 2 t 'Gate' w step, '%s' u 3 t 'Trigger' w step, '%s' u 4 t 'TrgHoldOff' w step, '%s' u 5 t 'OverThr' w step\n", plotwave_file, plotwave_file, plotwave_file, plotwave_file, plotwave_file);
                    break;
                }

                fflush(gWavePlotFile);
                gPrevWPlotTime = gCurrTime;
            }


            gPrevTimeTag[i] = gCurrTimeTag[i];
        }

		gChEvCnt[i] += NumEvents[i];

        if (gToggleTrace) {
            int probe;

            switch (gAnalogTrace) {

            case 0:
                probe = CAEN_DGTZ_DPP_VIRTUALPROBE_Input;
                break;
            case 1:
                probe = CAEN_DGTZ_DPP_VIRTUALPROBE_SmoothedInput;
                break;
            case 2:
                probe = CAEN_DGTZ_DPP_VIRTUALPROBE_Baseline;
                break;
            }

            if ((ret = CAEN_DGTZ_SetDPP_VirtualProbe(gHandle, ANALOG_TRACE_1, probe)) != CAEN_DGTZ_Success) {
                printf("Error during CAEN_DGTZ_SetDPPVirtualprobe() call (ret = %d): exiting ....\n", ret);
                exit(-1);
            }

            gToggleTrace = 0;
        }

    }
    gBytesRead += bsize;

    return 0;

}


/*
** Prints statistics to console.
*/
void print_statistics(uint32_t refresh_time) {
    unsigned int i;
	long         elapsed;
	uint64_t     PrevEvCnt = 0;

	gCurrTime = get_time();
	elapsed = (gCurrTime - gPrevTime);
	gRunElapsedTime = gCurrTime - gRunStartTime;

	if (elapsed > refresh_time) {
		uint64_t diff;
		gTotEvCnt = 0;

		clear_screen();

		for (i = 0; i < gEquippedChannels; ++i) {
			gTotEvCnt += gChEvCnt[i];
		}

		printf("*********** Global statistics ***************\n");
		printf("=============================================\n\n");
		printf("Run ID               = %s\n", gRunID);
		printf("Elapsed time         = %ld s\n", gRunElapsedTime / 1000); /* Seconds */
		printf("Readout Loops        = %d\n", gLoops);
		printf("Events               = %lld\n", (long long int)gTotEvCnt);
		printf("Readout Rate         = %.2f MB/s\n\n", (float)gBytesRead / 1024 / (elapsed));

		printf("******** Group  %d statistics **************\n\n", grp4stats);
		printf("Ch  Rate(KHz)   EvCounts\n\n");

		for (i = 0; i < 8; ++i) {
			diff = gChEvCnt[grp4stats * 8 + i] - gPrevChEvCnt[grp4stats * 8 + i];
			if (gChEvCnt[grp4stats * 8 + i] > 0) {
				printf("%2d %10.3f %10lld\n", grp4stats * 8 + i, (double)(diff) / elapsed, gChEvCnt[grp4stats * 8 + i]);
			}
			else {
				printf("%2d   -\n", grp4stats * 8 + i);
			}
			gPrevChEvCnt[grp4stats * 8 + i] = gChEvCnt[grp4stats * 8 + i];
		}
		gBytesRead = 0;
		gPrevTime = gCurrTime;
		PrevEvCnt = gTotEvCnt;
	}
}


/* ============================================================================== */
/* main                                                                           */
/* ============================================================================== */
int main(int argc, char* argv[])
{
    int ret;
    int i;
    int acquisition_runtime = 0; /* 0 = infinite run (exit with 'q') */
    int max_acquisition_events = 0;

    int ConnType;
    char* IpAddr = (char*)malloc(150);
    char* VmeBaseAddress = (char*)malloc(150);
    char* SerialNumber = (char*)malloc(150);
    char* ConfigFileName = (char*)malloc(150);
    char* CSVFilename = (char*)malloc(150);

    FILE* csvFile;

    gSWTrigger = 0;
    running = 0;
    grp4stats = 0; /* Select group for statistics update on screen */
    gLoops = 0;
    gToggleTrace = 0;
    gSaveCSVFile = 0;
    gAnalogTrace = 0;
    gRestart = 0;
    gStopped = 0;
    gAcqrun = 0;
    gAcquisitionBufferAllocated = 0;

    gAcqBuffer = NULL;
    gEventPtr = NULL;
    gHistPlotFile = NULL;
    gWavePlotFile = NULL;

    gBytesRead = 0;                                /* Number of bytes read                   */
    gTotEvCnt = 0;                                 /* Total number of events                 */
    gPrevTotEvCnt = 0;                             /* Previous Total number of events        */
    gChEvCnt[MAX_CHANNELS];                    /* Channel event counter                  */
    gPrevChEvCnt[MAX_CHANNELS];                /* Previous Channel event counter         */

    /* Default run ID*/
    gRunID = (char*)malloc(100);
    strcpy(gRunID, "run");

    ///get path to user directory and define file names//
#ifdef _WIN32
    sprintf(home_directory, "%s", getenv("USERPROFILE"));

    config_file = malloc(strlen(home_directory) + 1 + strlen(CONFIG_FILE_NAME) + strlen(WORKING_DIR));
    strcpy(config_file, home_directory);
    strcat(config_file, WORKING_DIR);
    strcat(config_file, CONFIG_FILE_NAME);
#else
    snprintf(home_directory, MAX_PATH, "%s", getenv("HOME"));
    config_file = malloc(100 + strlen(CONFIG_FILE_NAME));
    strcpy(config_file, CONFIG_FILE_NAME);

    w_dir = malloc(1 + strlen(home_directory) + strlen(WORKING_DIR));
    strcpy(w_dir, home_directory);
    strcat(w_dir, WORKING_DIR);

    struct stat st = { 0 };
    if (stat(w_dir, &st) == -1) {
        mkdir(w_dir, 0700);
    }
#endif 

    plotdata_file = malloc(strlen(home_directory) + 1 + strlen(PLOT_FILE) + strlen(WORKING_DIR));
    strcpy(plotdata_file, home_directory);
    strcat(plotdata_file, WORKING_DIR);
    strcat(plotdata_file, PLOT_FILE);


    plotwave_file = malloc(strlen(home_directory) + 1 + strlen(PLOT_WAVE) + strlen(WORKING_DIR));
    strcpy(plotwave_file, home_directory);
    strcat(plotwave_file, WORKING_DIR);
    strcat(plotwave_file, PLOT_WAVE);


    /* Command line processing */
    for (i = 1; i < argc; i++)  /* Skip argv[0] (program name). */
    {
        if (strcmp(argv[i], "-d") == 0)  /* Process optional arguments. */
        {

            /*
             * The last argument is argv[argc-1].  Make sure there are
             * enough arguments.
             */

            if (i + 1 <= argc - 1)  /* There are enough arguments in argv. */
            {
                i++;
                acquisition_runtime = atoi(argv[i]) * 1000;
                if (acquisition_runtime < 0) {
                    usage();
                    printf("Press any key to exit!\n");
                    getch();
                }
                if (errno == ERANGE) {
                    perror("acquisition_runtime parameter too big:\n");
                    printf("Press any key to exit!\n");
                    getch();
                }

            }
            else
            {
                /* Print usage statement and exit (see below). */
                usage();
                printf("Press any key to exit!\n");
                getch();
            }
        }
        else if (strcmp(argv[i], "-n") == 0)  /* Process optional arguments. */
        {

            /*
             * The last argument is argv[argc-1].  Make sure there are
             * enough arguments.
             */

            if (i + 1 <= argc - 1)  /* There are enough arguments in argv. */
            {
                i++;
                max_acquisition_events = atoi(argv[i]) * 1000;
                if (max_acquisition_events < 0) {
                    usage();
                    printf("Press any key to exit!\n");
                    getch();
                }
                if (errno == ERANGE) {
                    perror("acquisition_runtime parameter too big:\n");
                    printf("Press any key to exit!\n");
                    getch();
                }

            }
            else
            {
                /* Print usage statement and exit (see below). */
                usage();
                printf("Press any key to exit!\n");
                getch();
            }
        }
        else if (strcmp(argv[i], "-id") == 0)  /* Process optional arguments. */
        {

            /*
             * The last argument is argv[argc-1].  Make sure there are
             * enough arguments.
             */

            if (i + 1 <= argc - 1)  /* There are enough arguments in argv. */
            {
                i++;
                strcpy(gRunID, argv[i]);
            }
            else
            {
                /* Print usage statement and exit (see below). */
                usage();
                printf("Press any key to exit!\n");
                getch();
            }
        }
        else
        {
            /* Process non-optional arguments here. */
            config_file = argv[i];
        }
    }


    printf("Using configuration file %s\n", config_file);

    /* ---------------------------------------------------------------------------------------
    ** Set Parameters (default values or read from config file)
    ** ---------------------------------------------------------------------------------------
    */
    ret = setup_parameters(&gParams, config_file);///CONFIG_FILE_NAME
    if (ret < 0) {
        printf("Error in setting parameters\n");
        printf("Press any key to exit!\n");
        getch();
    }

    /* Board configuration prior to run */
    ret = setup_acquisition();
    if (ret) {
        printf("Error during acquisition setup (ret = %d) .... Exiting\n", ret);
        running = 0;
    }
    PrintMenu();

    /* Open CSV file*/
    strcpy(CSVFilename, home_directory);
    strcat(CSVFilename, WORKING_DIR);
    strcat(CSVFilename, "output.csv");
    csvFile = fopen(CSVFilename, "w");
    if (csvFile == NULL) {
        printf("Cannot open CSV file!\n");
        printf("Press any key to exit!\n");
        getch();
    }

    /* Init statistics */
    ResetStatistics();
    running = 1;

    /* Init time global variables */
    gCurrTime = get_time();
    gRunStartTime = gCurrTime;
    gPrevTime = gCurrTime;
    gPrevHPlotTime = gCurrTime;
    gPrevWPlotTime = gCurrTime;
    gPrevKeyTime = gCurrTime;

    /* Readout Loop */
    while (running) {

        gCurrTime = get_time();
        /* Periodic keyboard input -... could be a separate thread .... */
        if ((gCurrTime - gPrevKeyTime) > 200) {  // check every 200 ms
            ret = check_user_input();
            gPrevKeyTime = gCurrTime;
        }
        if (ret) running = 0;

        if (gRestart) {
            gRestart = 0;

            if (load_configuration_from_file(config_file, &gParams)) {
                printf("Error during configuration reload: file %s reload failed! .... Exiting\n", config_file); //CONFIG_FILE_NAME
                break;
            }

            if (gAcquisitionBufferAllocated && !gStopped) {

                ret = CAEN_DGTZ_FreeDPPWaveforms(gHandle, &gWaveforms);
                ret = CAEN_DGTZ_FreeReadoutBuffer(&gAcqBuffer);


                for (i = 0; i < MAX_CHANNELS; ++i) {
                    if (gListFiles[i] != NULL)
                        fclose(gListFiles[i]);
                    gListFiles[i] = NULL;
                }

                ret = CAEN_DGTZ_FreeDPPEvents(gHandle, (void**)&gEvent);///check this free

                for (i = 0; i < (int)gEquippedChannels; i++) {
                    if (gHisto[i] != NULL)
                        free(gHisto[i]);
                    gHisto[i] = NULL;
                }

                gAcquisitionBufferAllocated = 0;
            }

            /* ---------------------------------------------------------------------------------------
            ** Set Parameters (default values or read from config file)
            ** ---------------------------------------------------------------------------------------
            */
            ret = setup_parameters(&gParams, config_file);///CONFIG_FILE_NAME
            if (ret < 0) {
                printf("Error in setting parameters\n");
                printf("Press any key to exit!\n");
                getch();
            }

            if (setup_acquisition()) {
                printf("Error during acquisition restart: acquisition setup failed! .... Exiting\n");
                break;
            }

            clear_screen();
            PrintMenu();

            if (!gStopped) {
                gBytesRead = 0;
                gLoops = 0;
            }
            else {
                Sleep(100);
                print_statistics(REFRESH_RATE);
                printf("Acquisition stopped\n");
                gStopped = 0;
            }
        }

        if (gAcqrun == 0)
            continue;

        /* Get data, update histograms, do plots*/
        ret = run_acquisition(csvFile);
        if (ret) {
            printf("Error during acquisition loop (ret = %d) .... Exiting\n", ret);
            break;
        }

        /* Periodic console update -... could be a separate thread .... */
        print_statistics(REFRESH_RATE);

        /*
        ** Check acquisition run time.
        ** exit from loop if programmed time is elapsed.
        */
        if (acquisition_runtime)
            if (gRunElapsedTime > acquisition_runtime)
                running = 0; /* exit from acquisition loop */

        /* Check maximum event count */
        if ((max_acquisition_events > 0) && (gTotEvCnt > max_acquisition_events))
            running = 0;     /* exit from acquisition loop */


    } /* end of readout loop */

    /* Final memory cleanup */
    cleanup_on_exit(csvFile);


    printf("Press any key to exit!\n");
    getch();


    return 0;
}