
/******************************************************************************
*
* CAEN SpA - Front End Division
* Via Vetraia, 11 - 55049 - Viareggio ITALY
* +390594388398 - www.caen.it
*
******************************************************************************/
#include "dpp_qdc.h"

/* Globals */
int	        gHandle;                                   /* CAEN library handle */

/* Variable declarations */
unsigned int gActiveChannel;                            /* Active channel for data analysis */
unsigned int gEquippedChannels;                         /* Number of equipped channels      */
unsigned int gEquippedGroups;                           /* Number of equipped groups        */


uint32_t** gHisto;                                    /* Histograms                       */

uint64_t     gExtendedTimeTag[MAX_CHANNELS];            /* Extended Time Tag                */
uint64_t     gETT[MAX_CHANNELS];                        /* Extended Time Tag                */
uint64_t     gCurrTimeTag[MAX_CHANNELS];                /* Current Time Tag                */
uint64_t     gPrevTimeTag[MAX_CHANNELS];                /* Previous Time Tag                */

BoardParameters   gParams;                              /* Board parameters structure       */
CAEN_DGTZ_DPP_QDC_Params_t* DPPParams;				   /*Board DP-QDC parameters			*/

char home_directory[MAX_PATH];						/* User home directory path           */
char* w_dir;
char* config_file;									/* To store path to config file in user directory */
char* plotdata_file;									/* To store path to PlotData.txt file in user directory */
char* plotwave_file;									/* To store path to PlotWave.txt file in user directory */

FILE* gListFiles[MAX_CHANNELS];                  /* Output file for list data        */

CAEN_DGTZ_BoardInfo_t             gBoardInfo;           /* Board Informations               */
CAEN_DGTZ_DPP_QDC_Event_t* gEvent[MAX_CHANNELS]; /* Events                           */

CAEN_DGTZ_DPP_QDC_Waveforms_t* gWaveforms;           /* Waveforms                        */

/* Variable definitions */
int          gSWTrigger;                                /* Signal for software trigger          */
int			running;                                   /* Readout is running			       */
int		    gAcqrun;								   /* Acquisition is on			       */
int          grp4stats;                                 /* Selected group for statistics        */
int          gLoops;                                    /* Number of acquisition loops          */
int          gToggleTrace;                              /* Signal for trace toggle from user    */
int          gSaveCSVFile;                              /* Save CSV file                        */
int          gAnalogTrace;                              /* Signal for analog trace toggle       */
int          gRestart;                              /* Signal acquisition restart from user */
int          gStopped;                              /* Signal acquisition stopped from user */
char* gAcqBuffer;                                /* Acquisition buffer                   */
char* gEventPtr;                                 /* Events buffer pointer                */
FILE* gHistPlotFile;                             /* Source file for histogram plot       */
FILE* gWavePlotFile;                             /* Source file for waveform plot        */
int          gAcquisitionBufferAllocated;               /* Buffers are allocated  			  */

/* Runtime statistics */
unsigned int gBytesRead;                                /* Number of bytes read                   */
uint64_t     gTotEvCnt;                                 /* Total number of events                 */
uint64_t     gPrevTotEvCnt;                             /* Previous Total number of events        */
uint64_t     gChEvCnt[MAX_CHANNELS];                    /* Channel event counter                  */
uint64_t     gPrevChEvCnt[MAX_CHANNELS];                /* Previous Channel event counter         */

long         gCurrTime;                                 /* Current time                     */
long         gPrevTime;                                 /* Previous saved time              */
long         gPrevWPlotTime;                            /* Previous time of Waveform plot   */
long         gPrevHPlotTime;                            /* Previous time of Histogram plot  */
long         gPrevKeyTime;                              /* Previous keyboard check time     */
long         gRunStartTime;                             /* Time of run start                */
long         gRunElapsedTime;                           /* Elapsed time since start         */
long         gUpdateElapsedTime;                        /* Elapsed time since last update   */
char* gRunID;                                    /* Run Identifier                   */


#ifdef LINUX
#include <sys/time.h> /* struct timeval, select() */
#include <termios.h> /* tcgetattr(), tcsetattr() */
#include <stdlib.h> /* atexit(), exit() */
#include <unistd.h> /* read() */
#include <stdio.h> /* printf() */
#include <string.h> /* memcpy() */
#include <sys/types.h>
#include <sys/stat.h>

#define CLEARSCR "clear"

static struct termios g_old_kbd_mode;

/*****************************************************************************/
static void cooked(void)
{
	tcsetattr(0, TCSANOW, &g_old_kbd_mode);
}

static void raw(void)
{
	static char init;
	/**/
	struct termios new_kbd_mode;

	if (init)
		return;
	/* put keyboard (stdin, actually) in raw, unbuffered mode */
	tcgetattr(0, &g_old_kbd_mode);
	memcpy(&new_kbd_mode, &g_old_kbd_mode, sizeof(struct termios));
	new_kbd_mode.c_lflag &= ~(ICANON | ECHO);
	new_kbd_mode.c_cc[VTIME] = 0;
	new_kbd_mode.c_cc[VMIN] = 1;
	tcsetattr(0, TCSANOW, &new_kbd_mode);
	/* when we exit, go back to normal, "cooked" mode */
	atexit(cooked);

	init = 1;
}

/*****************************************************************************/
/*  SLEEP  */
/*****************************************************************************/
void Sleep(int t) {
	usleep(t * 1000);
}

/*****************************************************************************/
/*  GETCH  */
/*****************************************************************************/
int getch(void)
{
	unsigned char temp;

	raw();
	/* stdin = fd 0 */
	if (read(0, &temp, 1) != 1)
		return 0;
	return temp;

}


/*****************************************************************************/
/*  KBHIT  */
/*****************************************************************************/
int kbhit()
{

	struct timeval timeout;
	fd_set read_handles;
	int status;

	raw();
	/* check stdin (fd 0) for activity */
	FD_ZERO(&read_handles);
	FD_SET(0, &read_handles);
	timeout.tv_sec = timeout.tv_usec = 0;
	status = select(0 + 1, &read_handles, NULL, NULL, &timeout);
	if (status < 0)
	{
		printf("select() failed in kbhit()\n");
		exit(1);
	}
	return (status);
}


#else  /* Windows */

#include <conio.h>
#define CLEARSCR "cls"

#endif

/* Get time in milliseconds from the computer internal clock */
long get_time()
{
	long time_ms;
#ifdef WIN32
	struct _timeb timebuffer;
	_ftime(&timebuffer);
	time_ms = (long)timebuffer.time * 1000 + (long)timebuffer.millitm;
#else
	struct timeval t1;
	struct timezone tz;
	gettimeofday(&t1, &tz);
	time_ms = (t1.tv_sec) * 1000 + t1.tv_usec / 1000;
#endif
	return time_ms;
}

void clear_screen()
{
	system(CLEARSCR);
}

/*
** Set default parameters for the acquisition
*/
void set_default_parameters(BoardParameters *params) {
	int i;

	params->RecordLength = 200;		   /* Number of samples in the acquisition window (waveform mode only)    */
	params->PreTrigger = 100;  		   /* PreTrigger is in number of samples                                  */
	params->ActiveChannel = 0;			   /* Channel used for the data analysis (plot, histograms, etc...)       */
	params->SaveList = 0;             /*  Set flag to save list events to file */
	params->SaveCSVFile = 0;          /*  Set flag to save list events to CSV file */
	params->NevAggr = 1;             /* Number of events per aggregate (buffer). 0=automatic */
	params->AcqMode = CAEN_DGTZ_DPP_ACQ_MODE_List;  /* Acquisition Mode (LIST or MIXED)                     */
	params->DefaultTriggerThr = 10;            /* Default threshold for trigger                        */
	params->FixedBaseline = 2100;		   /* fixed baseline (used when BaselineMode = 0)                         */
	params->PreGate = 20;			   /* Position of the gate respect to the trigger (num of samples before) */
	params->BaselineMode = 2;			   /* Baseline: 0=Fixed, 1=4samples, 2=16 samples, 3=64samples            */
	params->TrgHoldOff = 10;            /*Trigger HoldOff */
	params->TrgMode = 0;             /*Trigger Mode: 0 Normal, 1 Paired */
	params->DisSelfTrigger = 0;     /*Disable self trigger: if disabled it can be propagated to TRGOUT, but not used to acquire*/
	params->TrgSmoothing = 0;             /*Input smoothing factor: 0=disabled, 1=2samples, 2=4samples ... 5=64samples  */
	params->ChargeSensitivity = 2;             /* Charge sesnitivity (0=max, 7=min) */
	params->PulsePol = 1;             /* Pulse Polarity (1=negative, 0=positive)              */
	params->EnChargePed = 0;             /* Enable Fixed Charge Pedestal in firmware (0=off, 1=on (1024 pedestal)) */
	params->DisTrigHist = 0;             /* 0 = Trigger Histeresys on; 1 = Trigger Histeresys off */


	for (i = 0; i < MAX_X740_GROUP_SIZE; ++i) {
		params->DPPParams.GateWidth[i] = 40;	   /* Gate Width in samples        */
		params->DPPParams.DisTrigHist[i] = params->DisTrigHist;
		params->DPPParams.DisSelfTrigger[i] = params->DisSelfTrigger;
		params->DPPParams.BaselineMode[i] = params->BaselineMode;
		params->DPPParams.TrgMode[i] = params->TrgMode;
		params->DPPParams.ChargeSensitivity[i] = params->ChargeSensitivity;
		params->DPPParams.PulsePol[i] = params->PulsePol;
		params->DPPParams.EnChargePed[i] = params->EnChargePed;
		params->DPPParams.TestPulsesRate[i] = params->TestPulsesRate;
		params->DPPParams.EnTestPulses[i] = params->EnTestPulses;
		params->DPPParams.InputSmoothing[i] = params->TrgSmoothing;
		params->DPPParams.BaselineMode[i] = params->BaselineMode;
		params->DPPParams.InputSmoothing[i] = params->TrgSmoothing;
		params->DPPParams.trgho[i] = params->TrgHoldOff;
		params->DPPParams.FixedBaseline[i] = params->FixedBaseline;
	}

	for (i = 0; i < MAX_X740_GROUP_SIZE; ++i)
		params->DCoffset[i] = 0x8000;            /* DC offset adjust in DAC counts (0x8000 = mid scale)  */

	for (i = 0; i < MAX_DPP_QDC_CHANNEL_SIZE; ++i) {
		params->TriggerThreshold[i] = params->DefaultTriggerThr;
	}

}

/*
** Read config file and assign new values to the parameters
** Returns 0 on success; -1 in case of any error detected.
*/
int load_configuration_from_file(char * fname, BoardParameters *params) {
	FILE *parameters_file;

	params->ConnectionType = CONNECTION_TYPE_AUTO;

	parameters_file = fopen(fname, "r");
	if (parameters_file != NULL) {
		while (!feof(parameters_file)) {
			char str[100];
			fscanf(parameters_file, "%s", str);
			if (str[0] == '#') {
				fgets(str, 100, parameters_file);
				continue;
			}

			if (strcmp(str, "AcquisitionMode") == 0) {
				char str1[100];
				fscanf(parameters_file, "%s", str1);
				if (strcmp(str1, "MIXED") == 0)
					params->AcqMode = CAEN_DGTZ_DPP_ACQ_MODE_Mixed;
				if (strcmp(str1, "LIST") == 0)
					params->AcqMode = CAEN_DGTZ_DPP_ACQ_MODE_List;
			}

			if (strcmp(str, "ConnectionType") == 0) {
				char str1[100];
				fscanf(parameters_file, "%s", str1);
				if (strcmp(str1, "USB") == 0)
					params->ConnectionType = CONNECTION_TYPE_USB;
				if (strcmp(str1, "OPT") == 0)
					params->ConnectionType = CONNECTION_TYPE_OPT;
			}


			if (strcmp(str, "ConnectionLinkNum") == 0)
				fscanf(parameters_file, "%d", &params->ConnectionLinkNum);
			if (strcmp(str, "ConnectionConetNode") == 0)
				fscanf(parameters_file, "%d", &params->ConnectionConetNode);
			if (strcmp(str, "ConnectionVmeBaseAddress") == 0)
				fscanf(parameters_file, "%lx", (unsigned long *)&params->ConnectionVMEBaseAddress);

			if (strcmp(str, "EnableCSV") == 0) {
				fscanf(parameters_file, "%d", (unsigned long *)&params->SaveCSVFile);
				gSaveCSVFile = params->SaveCSVFile;
			}

			if (strcmp(str, "TriggerThreshold") == 0) {
				int ch;
				fscanf(parameters_file, "%d", &ch);
				fscanf(parameters_file, "%d", &params->TriggerThreshold[ch]);
			}

			if (strcmp(str, "RecordLength") == 0)
				fscanf(parameters_file, "%d", &params->RecordLength);
			if (strcmp(str, "PreTrigger") == 0)
				fscanf(parameters_file, "%d", &params->PreTrigger);
			if (strcmp(str, "ActiveChannel") == 0)
				fscanf(parameters_file, "%d", &params->ActiveChannel);
			if (strcmp(str, "BaselineMode") == 0)
				fscanf(parameters_file, "%d", &params->BaselineMode);
			if (strcmp(str, "TrgMode") == 0)
				fscanf(parameters_file, "%d", &params->TrgMode);
			if (strcmp(str, "TrgSmoothing") == 0)
				fscanf(parameters_file, "%d", &params->TrgSmoothing);
			if (strcmp(str, "TrgHoldOff") == 0)
				fscanf(parameters_file, "%d", &params->TrgHoldOff);
			if (strcmp(str, "FixedBaseline") == 0)
				fscanf(parameters_file, "%d", &params->FixedBaseline);
			if (strcmp(str, "PreGate") == 0) {
				int gr;
				fscanf(parameters_file, "%d", &gr);
				fscanf(parameters_file, "%d", &params->DPPParams.PreGate[gr]);
			}
			if (strcmp(str, "GateWidth") == 0) {
				int gr;
				fscanf(parameters_file, "%d", &gr);
				fscanf(parameters_file, "%d", &params->DPPParams.GateWidth[gr]);
			}
			if (strcmp(str, "DCoffset") == 0) {
				int ch;
				fscanf(parameters_file, "%d", &ch);
				fscanf(parameters_file, "%d", &params->DCoffset[ch]);
			}
			if (strcmp(str, "ChargeSensitivity") == 0)
				fscanf(parameters_file, "%d", &params->ChargeSensitivity);
			if (strcmp(str, "NevAggr") == 0)
				fscanf(parameters_file, "%d", &params->NevAggr);
			if (strcmp(str, "SaveList") == 0)
				fscanf(parameters_file, "%d", &params->SaveList);
			if (strcmp(str, "ChannelTriggerMask") == 0)
				fscanf(parameters_file, "%llx", &params->ChannelTriggerMask);
			if (strcmp(str, "PulsePolarity") == 0)
				fscanf(parameters_file, "%d", &params->PulsePol);
			if (strcmp(str, "EnableChargePedestal") == 0)
				fscanf(parameters_file, "%d", &params->EnChargePed);
			if (strcmp(str, "DisableTriggerHysteresis") == 0)
				fscanf(parameters_file, "%d", &params->DisTrigHist);
			if (strcmp(str, "DisableSelfTrigger") == 0)
				fscanf(parameters_file, "%d", &params->DisSelfTrigger);
			if (strcmp(str, "EnableTestPulses") == 0)
				fscanf(parameters_file, "%d", &params->EnTestPulses);
			if (strcmp(str, "TestPulsesRate") == 0)
				fscanf(parameters_file, "%d", &params->TestPulsesRate);
			if (strcmp(str, "DefaultTriggerThr") == 0)
				fscanf(parameters_file, "%d", &params->DefaultTriggerThr);
			if (strcmp(str, "EnableExtendedTimeStamp") == 0)
				fscanf(parameters_file, "%d", &params->DPPParams.EnableExtendedTimeStamp);
		}
		fclose(parameters_file);

		/* copy values into DPPParams */
		int g = 0;
		for (g = 0; g < MAX_X740_GROUP_SIZE; g++) {
			params->DPPParams.DisTrigHist[g] = params->DisTrigHist;
			params->DPPParams.DisSelfTrigger[g] = params->DisSelfTrigger;
			params->DPPParams.BaselineMode[g] = params->BaselineMode;
			params->DPPParams.TrgMode[g] = params->TrgMode;
			params->DPPParams.ChargeSensitivity[g] = params->ChargeSensitivity;
			params->DPPParams.PulsePol[g] = params->PulsePol;
			params->DPPParams.EnChargePed[g] = params->EnChargePed;
			params->DPPParams.TestPulsesRate[g] = params->TestPulsesRate;
			params->DPPParams.EnTestPulses[g] = params->EnTestPulses;
			params->DPPParams.InputSmoothing[g] = params->TrgSmoothing;
			params->DPPParams.BaselineMode[g] = params->BaselineMode;
			params->DPPParams.InputSmoothing[g] = params->TrgSmoothing;
			params->DPPParams.trgho[g] = params->TrgHoldOff;
			params->DPPParams.FixedBaseline[g] = params->FixedBaseline;
		}

		return 0;
	}
	return -1;
}


/* Set parameters for the acquisition */
int setup_parameters(BoardParameters *params, char *fname) {
	int ret;
	set_default_parameters(params);

	ret = load_configuration_from_file(fname, params);
	return ret;
}

int configure_digitizer(int handle, int EquippedGroups, BoardParameters *params) {

	int ret = 0;
	int i;
	//uint32_t DppCtrl1;
	uint32_t GroupMask = 0;

	/* Reset Digitizer */
	ret |= CAEN_DGTZ_Reset(handle);

	/* Build Group enablemask from channel trigger mask */
	for (i = 0; i < EquippedGroups; i++) {
		uint8_t mask = (params->ChannelTriggerMask >> (i * 8)) & 0xFF;
		ret |= CAEN_DGTZ_SetChannelGroupMask(handle, i, (uint32_t)mask);
		if (mask)
			GroupMask |= (1 << i);
	}
	/* Set Group enable mask */
	ret |= CAEN_DGTZ_SetGroupEnableMask(handle, GroupMask);


	/*
	** Set selfTrigger threshold
	** Check if the module has 64 (VME) or 32 channels available (Desktop NIM)
	*/
	if ((gBoardInfo.FormFactor == CAEN_DGTZ_VME64_FORM_FACTOR) || (gBoardInfo.FormFactor == CAEN_DGTZ_VME64X_FORM_FACTOR))
		for (i = 0; i < 64; i++)
			ret |= CAEN_DGTZ_SetChannelTriggerThreshold(handle, i, params->TriggerThreshold[i]);
	else
		for (i = 0; i < 32; i++)
			ret |= CAEN_DGTZ_SetChannelTriggerThreshold(handle, i, params->TriggerThreshold[i]);


	/* Disable Group self trigger for the acquisition (mask = 0) */
	ret |= CAEN_DGTZ_SetGroupSelfTrigger(handle, CAEN_DGTZ_TRGMODE_ACQ_ONLY, 0x00);
	/* Set the behaviour when a SW tirgger arrives */
	ret |= CAEN_DGTZ_SetSWTriggerMode(handle, CAEN_DGTZ_TRGMODE_ACQ_ONLY);
	/* Set the max number of events/aggregates to transfer in a sigle readout */
	ret |= CAEN_DGTZ_SetMaxNumAggregatesBLT(handle, MAX_AGGR_NUM_PER_BLOCK_TRANSFER);
	/* Set the start/stop acquisition control */
	ret |= CAEN_DGTZ_SetAcquisitionMode(handle, CAEN_DGTZ_SW_CONTROLLED);

	/*Set DPP Acquisition mode*/
	// CAEN_DGTZ_DPP_SAVE_PARAM_ChargeAndTime    Both charge and time are returned
	ret |= CAEN_DGTZ_SetDPPAcquisitionMode(handle, params->AcqMode, CAEN_DGTZ_DPP_SAVE_PARAM_ChargeAndTime);

	// CTIN: questa parte di codice non funziona
	if (params->TrgMode == 2) {
		uint32_t d32;
		params->DisSelfTrigger = 1;  // va asseganto params->DPPParams e non params
		ret |= CAEN_DGTZ_ReadRegister(handle, 0x8000, &d32);
		d32 = d32 | (0 & 0x3 << 20);  //CTIN: vanno messi i bit 20 e 21 a 0 => va fatto un AND e non un OR: d32 = d32 & ~(0x3 << 20);
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x8000, d32);
	}
	else
		params->DisSelfTrigger = 0;

	/* Set Pre Trigger (in samples) */
	ret |= CAEN_DGTZ_SetDPPPreTriggerSize(handle, -1, params->PreTrigger);

	/* Set DPP-PSD parameters */
	ret |= CAEN_DGTZ_SetDPPParameters(handle, GroupMask, &params->DPPParams);

	/* Set the waveform lenght (in samples) */
	ret |= CAEN_DGTZ_SetRecordLength(handle, params->RecordLength);

	/* Set DC offset */
	for (i = 0; i < EquippedGroups; i++) {
		ret |= CAEN_DGTZ_SetGroupDCOffset(handle, i, params->DCoffset[i]);
		if (ret != CAEN_DGTZ_Success) {
			printf("Errors setting dco %d.\n", i);
		}
	}

	/* Set number of events per memory buffer */
	ret |= CAEN_DGTZ_SetDPPEventAggregation(handle, params->NevAggr, 0);

	/* enable test pulses on TRGOUT/GPO */
	if (ENABLE_TEST_PULSE) {
		uint32_t d32;
		ret |= CAEN_DGTZ_ReadRegister(handle, 0x811C, &d32);
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x811C, d32 | (1 << 15));
		ret |= CAEN_DGTZ_WriteRegister(handle, 0x8168, 2);
	}

	/* Set TTL I/O electrical standard */
	ret |= CAEN_DGTZ_SetIOLevel(handle, CAEN_DGTZ_IOLevel_NIM);

	/* Set run synchronization mode */
	ret |= CAEN_DGTZ_SetRunSynchronizationMode(handle, CAEN_DGTZ_RUN_SYNC_Disabled);

	/* Check errors */
	if (ret != CAEN_DGTZ_Success) {
		printf("Errors during Digitizer Configuration.\n");
		return -1;
	}

	return 0;
}


/*
** Cleanup heap memory
** Returns 0 on success; -1 in case of any error detected.
*/
int cleanup_on_exit(FILE *csvFile) {

	int ret;
	int i;

	/* Free the buffers and close the digitizer */
	CAEN_DGTZ_SWStopAcquisition(gHandle);
	ret = CAEN_DGTZ_FreeReadoutBuffer(&gAcqBuffer);
	if (gHistPlotFile != NULL)  pclose(gHistPlotFile);
	if (gWavePlotFile != NULL)  pclose(gWavePlotFile);

	if (gWaveforms) //free(gWaveforms);
		ret = CAEN_DGTZ_FreeDPPWaveforms(gHandle, gWaveforms);

	for (i = 0; i < MAX_CHANNELS; ++i) {
		if (gListFiles[i] != NULL)
			fclose(gListFiles[i]);
	}

	    /* Open CSV file */
        fclose (csvFile);

	ret = CAEN_DGTZ_FreeDPPEvents(gHandle, (void**)&gEvent);


	ret = CAEN_DGTZ_CloseDigitizer(gHandle);

	return 0;

}

/*
** Setup acquisition.
**   - setup board according to configuration file
**   - open a connection with the module
**   - allocates memory for acquisition buffers and structures
** Returns 0 on success; -1 in case of any error detected.
*/

int setup_acquisition() {

	int ret;
	unsigned int i;
	//  uint32_t     size = 0;
	uint32_t AllocatedSize = 0;
	uint32_t rom_version;
	uint32_t rom_board_id;

	//clear_screen();


	/* ---------------------------------------------------------------------------------------
	// Open the digitizer and read board information
	// ---------------------------------------------------------------------------------------
	*/

	CAEN_DGTZ_CloseDigitizer(gHandle);

	/* Open a conection with module and gets library handle */
	if (gParams.ConnectionType == CONNECTION_TYPE_AUTO) {
		/* Try USB connection first */
		ret = CAEN_DGTZ_OpenDigitizer(CAEN_DGTZ_USB, gParams.ConnectionLinkNum, gParams.ConnectionConetNode, gParams.ConnectionVMEBaseAddress, &gHandle);
		if (ret != CAEN_DGTZ_Success) {
			/* .. if not successful try opticallink connection then ...*/
			ret = CAEN_DGTZ_OpenDigitizer(CAEN_DGTZ_OpticalLink, gParams.ConnectionLinkNum, gParams.ConnectionConetNode, gParams.ConnectionVMEBaseAddress, &gHandle);
			if (ret != CAEN_DGTZ_Success) {
				printf("Can't open digitizer\n");
				return -1;
			}
		}
	}
	else {
		if (gParams.ConnectionType == CONNECTION_TYPE_USB)
			ret = CAEN_DGTZ_OpenDigitizer(CAEN_DGTZ_USB, gParams.ConnectionLinkNum, gParams.ConnectionConetNode, gParams.ConnectionVMEBaseAddress, &gHandle);
		if (gParams.ConnectionType == CONNECTION_TYPE_OPT)
			ret = CAEN_DGTZ_OpenDigitizer(CAEN_DGTZ_OpticalLink, gParams.ConnectionLinkNum, gParams.ConnectionConetNode, gParams.ConnectionVMEBaseAddress, &gHandle);
		if (ret != CAEN_DGTZ_Success) {
			printf("Can't open digitizer\n");
			return -1;
		}
	}

	/* Check board type */
	ret = CAEN_DGTZ_GetInfo(gHandle, &gBoardInfo);

	/* Check board ModelName */
	CAEN_DGTZ_ReadRegister(gHandle, 0xF030, &rom_version);
	if ((rom_version & 0xFF) == 0x54) ///0x54 for bigger flash  ---> 0x50 small flash
	{
		CAEN_DGTZ_ReadRegister(gHandle, 0xF034, &rom_board_id);
		switch (rom_board_id)
		{
		case 0:
			sprintf(gBoardInfo.ModelName, "V1740D");
			break;
		case 1:
			sprintf(gBoardInfo.ModelName, "VX1740D");
			break;
		case 2:
			sprintf(gBoardInfo.ModelName, "DT5740D");
			break;
		case 3:
			sprintf(gBoardInfo.ModelName, "N6740D");
			break;
		default:
			break;
		}
	}
	else
	{
		printf("This software is not appropriate for module %s\n", gBoardInfo.ModelName);
		printf("Press any key to exit!\n");
		getch();
		exit(0);
	}


	printf("\nConnected to CAEN Digitizer Model %s\n", gBoardInfo.ModelName);
	printf("ROC FPGA Release is %s\n", gBoardInfo.ROC_FirmwareRel);
	printf("AMC FPGA Release is %s\n\n", gBoardInfo.AMC_FirmwareRel);

	gEquippedChannels = gBoardInfo.Channels * 8; //gBoardInfo is DigitizerTable.Info --> N.B: Channels means Groups in case of x740
	gEquippedGroups = gEquippedChannels / 8;

	gActiveChannel = gParams.ActiveChannel;
	if (gActiveChannel >= gEquippedChannels)
		gActiveChannel = gEquippedChannels - 1;

	ret = CAEN_DGTZ_SWStopAcquisition(gHandle);

	ret = configure_digitizer(gHandle, gEquippedGroups, &gParams);
	if (ret) {
		printf("Error during digitizer configuration\n");
		return -1;
	}

	/* ---------------------------------------------------------------------------------------
	// Memory allocation and Initialization of counters, histograms, etc...
	*/
	memset(gExtendedTimeTag, 0, MAX_CHANNELS * sizeof(uint64_t));
	memset(gETT, 0, MAX_CHANNELS * sizeof(uint64_t));
    memset(gCurrTimeTag, 0, MAX_CHANNELS*sizeof(uint64_t));
	memset(gPrevTimeTag, 0, MAX_CHANNELS * sizeof(uint64_t));
	memset(gPrevTimeTag, 0, MAX_CHANNELS * sizeof(uint64_t));

	if (!gAcquisitionBufferAllocated) {
		gAcquisitionBufferAllocated = 1;

		/* Malloc Readout Buffer.
		NOTE: The mallocs must be done AFTER digitizer's configuration! */
		uint32_t size1;
		ret = CAEN_DGTZ_MallocReadoutBuffer(gHandle, &gAcqBuffer, &size1);
		if (ret) { printf("Cannot allocate %d bytes for the acquisition buffer! Exiting...", size1); exit(-1); }
		printf("Allocated %d bytes for readout buffer \n", size1);

		/* Allocate memory for channel events */
		int allocated_size;
		ret = CAEN_DGTZ_MallocDPPEvents(gHandle, gEvent, &allocated_size);
		if (ret) { printf("Cannot allocate memory for waveforms\n. Exiting...."); exit(-1); };


		/* allocate memory buffer for the data readout */
		ret = CAEN_DGTZ_MallocDPPWaveforms(gHandle, &gWaveforms, &AllocatedSize);
		if (ret) { printf("Cannot allocate memory for waveforms \n. Exiting...."); exit(-1); };

		/* Allocate memory for the histograms */
		gHisto = (uint32_t **)malloc(gEquippedChannels * sizeof(uint32_t *));
		if (gHisto == NULL) { printf("Cannot allocate %d bytes for histogram! Exiting...", (int)(gEquippedChannels * sizeof(uint32_t *))); exit(-1); }
		for (i = 0; i < gEquippedChannels; ++i) {
			gHisto[i] = (uint32_t *)malloc(HISTO_NBIN * sizeof(uint32_t));
			if (gHisto[i] == NULL) { printf("Cannot allocate %d bytes for histogram! Exiting...", (int)(HISTO_NBIN * sizeof(uint32_t))); exit(-1); }
		}
	}

	if (!gStopped)
		for (i = 0; i < gEquippedChannels; ++i) {
			memset(gHisto[i], 0, HISTO_NBIN * sizeof(uint32_t));
		}
	else {
		int size1;
		ret = CAEN_DGTZ_FreeReadoutBuffer(&gAcqBuffer);
		ret = CAEN_DGTZ_MallocReadoutBuffer(gHandle, &gAcqBuffer, &size1);
	}

	/* Crete output list files (one per channel) */
	for (i = 0; i < gEquippedChannels; ++i) {
		char filename[200];
		char fname[200];
#ifdef _WIN32
		sprintf(fname, "\\list_ch%d.txt", i);
#else
		sprintf(fname, "/list_ch%d.txt", i);
#endif
		strcpy(filename, home_directory);
		strcat(filename, WORKING_DIR);
		strcat(filename, fname);

		if (gListFiles[i] == NULL) gListFiles[i] = fopen(filename, "w");
	}

	return 0;
}

